/* ok_lcmp.h */
/*
 * Copyright (c) 2004-2015 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#ifndef INCLUSION_GUARD_UUID_9307CCFC_CC00_4B29_BCD4_1D6A8532D73C
#define INCLUSION_GUARD_UUID_9307CCFC_CC00_4B29_BCD4_1D6A8532D73C

#include <aicrypto/ok_x509.h>
#include <aicrypto/ok_cmp.h>
#include <aicrypto/ok_ssl.h>
#include "ok_ca.h"

#ifdef  __cplusplus
extern "C" {
#endif

/*
 * LCMP Operations
 */
/* operation contorol */
typedef struct lcmp_ctrl LCMPCtrl;
struct lcmp_ctrl {
	LCMPCtrl *next;

	char *ctrlType;
	int crit;
	int vlen;
	unsigned char *value;
};

/* operation content */
#define COMMON_USE_IN_LCMPOP \
	int version; \
	int messageId; \
	int opId; 		/* operation ID */ \
	LCMPCtrl *ctrl;		/* operation Ctrl */ \
	int resultCode;		/* only used response */ \
	char *resultMsg;	/* only used response */ \
	unsigned char *der;	/* encoded DER message */

typedef struct lcmp_operation {
	COMMON_USE_IN_LCMPOP
} LCMPOp;

/*------------*/
typedef struct lcmp_op_bindreq {
	COMMON_USE_IN_LCMPOP

	int authMode;

	char *caname;
	char *username;
	char *passwd;
} LO_BindReq;

typedef struct lcmp_op_bindrsp {
	COMMON_USE_IN_LCMPOP

	CertList *list;
} LO_BindRsp;

/*------------*/
typedef struct lcmp_op_signreq {
	COMMON_USE_IN_LCMPOP

	char *profName;
	Req *p10;
	CertTemplate *tmpl;
	int serialNum;	/* 0 means OPTION */
} LO_SignReq;

typedef struct lcmp_op_signrsp {
	COMMON_USE_IN_LCMPOP

	Cert *cert;
} LO_SignRsp;

enum csrform { PKCS10CSR, CMPREQUEST };

/*------------*/
typedef struct lcmp_op_listreq {
	COMMON_USE_IN_LCMPOP

	char *profName;
	int serialNum;	/* 0 means OPTION */
	char *sbjQuery;
} LO_ListReq;

typedef struct lcmp_op_listrsp {
	COMMON_USE_IN_LCMPOP

	CertStat *stat;
} LO_ListRsp;

/*------------*/
/*
 * - policyFlag[16]
 *   [0] profile working policy: set CPF_WP0_XXX [include/ok_ca.h]
 *   [1] profile working policy: set CPF_WP1_XXX [include/ok_ca.h]
 *   [2] unused.
 *   [3] unused.
 *   [4] CSR subject matching policy: C, ST, L, O
 *   [5] CSR subject matching policy: OU, CN, UID, Em
 *   [6]..[15] unused.
 * 
 * See also: struct certificate_profile [include/ok_ca.h]
 *           CA_input_profpolicy() [aica/aica_pol.c]
 *           CA_check_profpol() [ca/ca_sign.c]
 *           CA_input_profdnpolicy() [aica/aica_pol.c]
 *           CA_dnexist_check() [ca/ca_tool.c]
 * 
 * - updSec
 *   updSec stored the period that can be renewal the certificate
 *   prior to expiration.
 * 
 *   See also: struct certificate_profile [include/ok_ca.h]
 * 
 */
typedef struct lcmp_proftemple {
	char *tmplName;
	int certSec;	/* -1 means OPTION */
	int certVer;	/* -1 means OPTION */
	int baseNum;	/* -1 means OPTION */
	int keyType;	/* -1 means OPTION */
	int keyLength;	/* -1 means OPTION */
	/* In fact, hashType is used to store a signature type(OBJ_SIG_XXX). */
	int hashType;	/* -1 means OPTION */
	CertDN sbjTmpl;	/* num==0 means OPTION */
	unsigned char policyFlag[16];	/* 0xff means OPTION */
	struct tm start;
	struct tm end;
	int updSec;	/* -1 means OPTION */
	int bgtype;	/* -1 means OPTION */
	int edtype;	/* -1 means OPTION */
	int updtype;	/* -1 means OPTION */
} ProfTmpl;

typedef struct lcmp_op_profreq {
	COMMON_USE_IN_LCMPOP

	int profOp;
	char *profName;
	char *profRename;
	ProfTmpl *prof;
	CertExt *ext;
} LO_ProfReq;

typedef struct lcmp_op_profrsp {
	COMMON_USE_IN_LCMPOP

	ProfTmpl *prof;
	CertExt *ext;

	int listnum;
	char **proflist;	/* profile name list */
} LO_ProfRsp;

/*------------*/
typedef struct lcmp_op_certreq {
	COMMON_USE_IN_LCMPOP

	int certOp;
	int serialNum;
	int revokeReason;	/* default 0 */
	char *passwd;
	Key *key;	/* imported key */
} LO_CertReq;

typedef struct lcmp_op_certrsp {
	COMMON_USE_IN_LCMPOP

	Cert *cert;	/* updated or exported cert */
	Key *key;	/* exported key */
} LO_CertRsp;

/*------------*/
typedef struct lcmp_op_csrreq {
	COMMON_USE_IN_LCMPOP

	int csrOp;
	int serialNum;
	int acceptID;
	int rejectReason;	/* default 0 */
	Req *csr;	/* posted CSR */
	char *profName;	/* issue profile */
	CertTemplate *tmpl;
} LO_CSRReq;

typedef struct lcmp_op_csrrsp {
	COMMON_USE_IN_LCMPOP

	Req *csr;	/* exported CSR */
	int serialNum;	/* issue operation returns serialNumber */
	int acceptID;	/* acceptID should be > 0 */
	CertTemplate *tmpl;	/* exported CSR */
} LO_CSRRsp;

/*------------*/
typedef struct lcmp_op_crlreq {
	COMMON_USE_IN_LCMPOP

	int crlOp;
	char *profName;
	int location;
	char *path;
	int crlNum;
} LO_CRLReq;

typedef struct lcmp_op_crlrsp {
	COMMON_USE_IN_LCMPOP

	CRLList *list;
} LO_CRLRsp;

/*------------*/
typedef struct lcmp_op_servopreq {
	COMMON_USE_IN_LCMPOP

	int svOp;
	AuthInfo *uinfo;
} LO_SVOpReq;

typedef struct lcmp_op_servoprsp {
	COMMON_USE_IN_LCMPOP

	int rspLen;
	unsigned char *opRsp;
} LO_SVOpRsp;

/*------------*/
typedef struct lcmp_op_extopreq {
	COMMON_USE_IN_LCMPOP

	char *opOID;
	int ifLen;
	unsigned char *opInfo;
} LO_ExtReq;

typedef struct lcmp_op_extoprsp {
	COMMON_USE_IN_LCMPOP

	char *opOID;
	int rspLen;
	unsigned char *opRsp;
} LO_ExtRsp;

/*------------*/
typedef struct lcmp_op_renewalreq {
	COMMON_USE_IN_LCMPOP

	int serialNum;
	Req *p10;
	CertTemplate *tmpl;
	int newSerialNum;	/* 0 means OPTION */
} LO_RenewalReq;

typedef struct lcmp_op_renewalrsp {
	COMMON_USE_IN_LCMPOP

	Cert *cert;
} LO_RenewalRsp;

/* 
 * LCMP Context
 */
typedef struct lcmp {
	int version;

	char *svname;
	char *caname;
	int port;

	int usessl;
	int vfycert;
	char *store;
	char *certid;
	char *clcert;
	char *clkeyf;
	char *clctpw;

	int sessionId;
	int transactionId;
	int messageId;

	CA *ca;		/* CA information */

	LCMPOp *op;	/* lcmp operation */

	SSL *sock;	/* row or SSL socket */

	int servStat;

	int authLevel;	/* RA: /CA:  */
	char *username;	/* if exist */
	int uid;
	int gid;
	unsigned long userGrant;
} LCMP;

/*
 * LCMP Definitions
 */

/* LCMP Operations */
#define LCMP_OP_BINDREQ		0
#define LCMP_OP_BINDRSP		1
#define LCMP_OP_UNBIND		2
#define LCMP_OP_SIGNREQ		3
#define LCMP_OP_SIGNRSP		4
#define LCMP_OP_LISTREQ		5
#define LCMP_OP_LISTRSP		6
#define LCMP_OP_PROFREQ		7
#define LCMP_OP_PROFRSP		8
#define LCMP_OP_CERTREQ		9
#define LCMP_OP_CERTRSP		10
#define LCMP_OP_CSRREQ		11
#define LCMP_OP_CSRRSP		12
#define LCMP_OP_CRLREQ		13
#define LCMP_OP_CRLRSP		14
#define LCMP_OP_SVOPREQ		15
#define LCMP_OP_SVOPRSP		16
#define LCMP_OP_EXTREQ		17
#define LCMP_OP_EXTRSP		18
#define LCMP_OP_RENEWALREQ	19
#define LCMP_OP_RENEWALRSP	20

/* LCMP Server State */
#define LCMP_SV_INIT		0
#define LCMP_SV_BINDCA		1
#define LCMP_SV_CLOSE		2
#define LCMP_SV_CRLOUT		3

/* LCMP Auth Level */
#define LCMP_AT_NONE		0x00000000
#define LCMP_AT_GUEST		0x00000001
#define LCMP_AT_SIMPLE		0x00000002
#define LCMP_AT_SSLCL		0x00000004

/* LCMP Auth Level */

/* LCMP Operator Grant */
#define LCMP_GR_BIND		0x00000001
#define LCMP_GR_SIGN		0x00000002
#define LCMP_GR_LIST		0x00000004
#define LCMP_GR_PFVIEW		0x00000008
#define LCMP_GR_PFADD		0x00000010
#define LCMP_GR_PFDEL		0x00000020
#define LCMP_GR_PFMOD		0x00000040
#define LCMP_GR_EXVIEW		0x00000080
#define LCMP_GR_EXADD		0x00000100
#define LCMP_GR_EXDEL		0x00000200
#define LCMP_GR_EXMOD		0x00000400
#define LCMP_GR_PFLIST		0x00000800
#define LCMP_GR_CTUPDT		0x00001000
#define LCMP_GR_CTREV		0x00002000
#define LCMP_GR_CTUNREV		0x00004000
#define LCMP_GR_CTEXP		0x00008000
#define LCMP_GR_KEYIMP		0x00010000
#define LCMP_GR_KEYEXP		0x00020000
#define LCMP_GR_KEYDEL		0x00040000
#define LCMP_GR_CSRPOST		0x00100000
#define LCMP_GR_CSREXP		0x00200000
#define LCMP_GR_CSRDEL		0x00400000
#define LCMP_GR_CSRJUDGE	0x00800000
#define LCMP_GR_CRLISS		0x01000000
#define LCMP_GR_CRLEXP		0x02000000
#define LCMP_GR_USEROP		0x10000000
#define LCMP_GR_CHANGEPW	0x20000000
#define LCMP_GR_EXTOP		0x40000000

/* LCMP return code */
#define LCMP_SUCCESS		0
#define LCMP_OPERATION_ERR	1
#define LCMP_PROTOCOL_ERR	2
#define LCMP_UNSUPPORTED_CTRL	3
#define LCMP_UNSUPPORTED_OP	4
#define LCMP_AUTH_ERR		10
#define LCMP_UNSUPPORTED_AUTH	11
#define LCMP_PERMISSION_DENY	12
#define LCMP_BAD_PROFILE	15
#define LCMP_BAD_SUBJECTNAME	16
#define LCMP_BAD_SERIALNUMBER	17
#define LCMP_BAD_EXTENSION	18
#define LCMP_BAD_USERNAME	19
#define LCMP_BAD_OLDPASSWD	20
#define LCMP_BAD_UPDTIMING	21
#define LCMP_BAD_REVOCATION	22
#define LCMP_BAD_PUBLICKEY	23
#define LCMP_NOSUCH_PROFILE	25
#define LCMP_NOSUCH_SUBJECT	26
#define LCMP_NOSUCH_SERIALNUM	27
#define LCMP_NOSUCH_EXTENSION	28
#define LCMP_NOSUCH_TEMPLATE	29
#define LCMP_NOSUCH_USERNAME	30
#define LCMP_REQEST_ACCEPTED	40
#define LCMP_REQEST_REJECTED	41
#define LCMP_REQEST_BADACCID	42
#define LCMP_REQEST_OPDONE	43
#define LCMP_REQEST_BADCSR	44
#define LCMP_KEY_UNMATCHED	50
#define LCMP_KEY_BADPASSWD	51
#define LCMP_KEY_NOTEXIST	52
#define LCMP_KEY_EXIST		53
#define LCMP_KEY_UNSUPPORTED	54
#define LCMP_LICENSE_EXCEEDED	60

/* LCMP auth mode -- for bindreq operation */
#define LCMP_OPBRQ_ATSIMPLE	0
#define LCMP_OPBRQ_ATSSLCL	1

/* LCMP Profile Operation */
#define LCMP_OPPF_VIEWPF	0
#define LCMP_OPPF_ADDPF 	1
#define LCMP_OPPF_DELPF 	2
#define LCMP_OPPF_MODPF 	3
#define LCMP_OPPF_VIEWEXT	4
#define LCMP_OPPF_ADDEXT 	5
#define LCMP_OPPF_DELEXT 	6
#define LCMP_OPPF_UPDEXT 	7
#define LCMP_OPPF_PFLIST 	8
#define LCMP_OPPF_RENAMEPF 	9

/* LCMP Cert Operation */
#define LCMP_OPCT_UPDCERT	0
#define LCMP_OPCT_RVKCERT	1
#define LCMP_OPCT_UNRVKCERT	2
#define LCMP_OPCT_EXPCERT	3
#define LCMP_OPCT_IMPKEY	4
#define LCMP_OPCT_EXPKEY	5
#define LCMP_OPCT_DELKEY	6

#define LCMP_RR_unspecified		0
#define LCMP_RR_keyCompromise		1
#define LCMP_RR_cACompromise		2
#define LCMP_RR_affiliationChanged	3
#define LCMP_RR_superseded		4
#define LCMP_RR_cessationOfOperation	5
#define LCMP_RR_certificateHold		6
#define LCMP_RR_removeFromCRL		8
#define LCMP_RR_privilegeWithdrawn	9
#define LCMP_RR_aaCompromise		10

/* LCMP CSR Operation */
#define LCMP_OPCSR_POST		0
#define LCMP_OPCSR_EXP		1
#define LCMP_OPCSR_DEL		2
#define LCMP_OPCSR_ISSUE	3
#define LCMP_OPCSR_REJECT	4

/* LCMP CRL Operation */
#define LCMP_OPCRL_ISSNOW	0
#define LCMP_OPCRL_GETLTS	1

#define LCMP_LCCRL_TOCLIENT	0
#define LCMP_LCCRL_CAPATH	1
#define LCMP_LCCRL_SVPATH	2

/* LCMP Server Operation */
#define LCMP_OPSV_ADDUSER	0
#define LCMP_OPSV_DELUSER	1
#define LCMP_OPSV_MODUSER	2
#define LCMP_OPSV_CHGPWD	3

/*
 * LCMP functions
 */
/* lcmp.c */
LCMP *LCMP_new(void);
void LCMP_free(LCMP *lc);

LCMPOp *LCMP_op_new(int op);
void LCMP_op_free(LCMPOp *op);

ProfTmpl *ProfTmpl_new(void);
void ProfTmpl_free(ProfTmpl *pt);
ProfTmpl *ProfTmpl_dup(ProfTmpl *org);

LCMPCtrl *LCMP_ctrl_new(void);
void LCMP_ctrl_free(LCMPCtrl *ctrl);
void LCMP_ctrl_free_all(LCMPCtrl *top);

LO_BindReq *LCMP_bindreq_new(void);
LO_BindRsp *LCMP_bindrsp_new(void);
LCMPOp *LCMP_unbind_new(void);
LO_SignReq *LCMP_signreq_new(void);
LO_SignRsp *LCMP_signrsp_new(void);
LO_ListReq *LCMP_listreq_new(void);
LO_ListRsp *LCMP_listrsp_new(void);
LO_ProfReq *LCMP_profreq_new(void);
LO_ProfRsp *LCMP_profrsp_new(void);
LO_CertReq *LCMP_certreq_new(void);
LO_CertRsp *LCMP_certrsp_new(void);
LO_CSRReq *LCMP_csrreq_new(void);
LO_CSRRsp *LCMP_csrrsp_new(void);
LO_CRLReq *LCMP_crlreq_new(void);
LO_CRLRsp *LCMP_crlrsp_new(void);
LO_SVOpReq *LCMP_svopreq_new(void);
LO_SVOpRsp *LCMP_svoprsp_new(void);
LO_ExtReq *LCMP_extreq_new(void);
LO_ExtRsp *LCMP_extrsp_new(void);
LO_RenewalReq *LCMP_renewalreq_new(void);
LO_RenewalRsp *LCMP_renewalrsp_new(void);

void LCMP_bindreq_free(LO_BindReq *op);
void LCMP_bindrsp_free(LO_BindRsp *op);
void LCMP_signreq_free(LO_SignReq *op);
void LCMP_signrsp_free(LO_SignRsp *op);
void LCMP_listreq_free(LO_ListReq *op);
void LCMP_listrsp_free(LO_ListRsp *op);
void LCMP_profreq_free(LO_ProfReq *op);
void LCMP_profrsp_free(LO_ProfRsp *op);
void LCMP_certreq_free(LO_CertReq *op);
void LCMP_certrsp_free(LO_CertRsp *op);
void LCMP_csrreq_free(LO_CSRReq *op);
void LCMP_csrrsp_free(LO_CSRRsp *op);
void LCMP_crlreq_free(LO_CRLReq *op);
void LCMP_crlrsp_free(LO_CRLRsp *op);
void LCMP_svopreq_free(LO_SVOpReq *op);
void LCMP_svoprsp_free(LO_SVOpRsp *op);
void LCMP_extreq_free(LO_ExtReq *op);
void LCMP_extrsp_free(LO_ExtRsp *op);
void LCMP_renewalreq_free(LO_RenewalReq *op);
void LCMP_renewalrsp_free(LO_RenewalRsp *op);

#define LCMP_AUTH_ISSET(var, flag)  ( ((~(var))^(~(flag))) ==0)

/* lcmp_asn1d.c */
LCMPOp *ASN1_LCMP_msg(unsigned char *msg);
int asn1_lcmp_ctrl(LCMPOp *op, unsigned char *in);
int asn1_lcmp_result(LCMPOp *op, unsigned char *cp);

int asn1_lcmp_bindreq(LCMPOp *op, unsigned char *req);
int asn1_lcmp_bindrsp(LCMPOp *op, unsigned char *rsp);
int asn1_lcmp_signreq(LCMPOp *op, unsigned char *req);
int asn1_lcmp_signrsp(LCMPOp *op, unsigned char *rsp);
int asn1_lcmp_listreq(LCMPOp *op, unsigned char *req);
int asn1_lcmp_listrsp(LCMPOp *op, unsigned char *rsp);
int asn1_lcmp_profreq(LCMPOp *op, unsigned char *req);
int asn1_lcmp_profrsp(LCMPOp *op, unsigned char *rsp);
int asn1_lcmp_certreq(LCMPOp *op, unsigned char *req);
int asn1_lcmp_certrsp(LCMPOp *op, unsigned char *rsp);
int asn1_lcmp_csrreq(LCMPOp *op, unsigned char *req);
int asn1_lcmp_csrrsp(LCMPOp *op, unsigned char *rsp);
int asn1_lcmp_crlreq(LCMPOp *op, unsigned char *req);
int asn1_lcmp_crlrsp(LCMPOp *op, unsigned char *rsp);
int asn1_lcmp_svopreq(LCMPOp *op, unsigned char *req);
int asn1_lcmp_svoprsp(LCMPOp *op, unsigned char *rsp);
int asn1_lcmp_extreq(LCMPOp *op, unsigned char *req);
int asn1_lcmp_extrsp(LCMPOp *op, unsigned char *rsp);
int asn1_lcmp_renewalreq(LCMPOp *op, unsigned char *req);
int asn1_lcmp_renewalrsp(LCMPOp *op, unsigned char *rsp);

CertStat *asn1_lcmp_statlist(unsigned char *in);
Key *asn1_lcmp_keytype(unsigned char *in);
CertExt *asn1_lcmp_delext(unsigned char *in);
ProfTmpl *asn1_lcmp_prof(unsigned char *in);
AuthInfo *asn1_lcmp_authinfo(unsigned char *in);

/* lcmp_asn1e.c */
unsigned char *LCMP_op_toDER(LCMPOp *op, unsigned char *buf, int *ret_len);
int lcmp_DER_ctrl(LCMPCtrl *ct, unsigned char *buf, int *ret_len);
int lcmp_DER_result(LCMPOp *op, unsigned char *buf, int *ret_len);

int lcmp_DER_bindreq(LCMPOp *op, unsigned char *buf, int *ret_len);
int lcmp_DER_bindrsp(LCMPOp *op, unsigned char *buf, int *ret_len);
int lcmp_DER_signreq(LCMPOp *op, unsigned char *buf, int *ret_len);
int lcmp_DER_signrsp(LCMPOp *op, unsigned char *buf, int *ret_len);
int lcmp_DER_listreq(LCMPOp *op, unsigned char *buf, int *ret_len);
int lcmp_DER_listrsp(LCMPOp *op, unsigned char *buf, int *ret_len);
int lcmp_DER_profreq(LCMPOp *op, unsigned char *buf, int *ret_len);
int lcmp_DER_profrsp(LCMPOp *op, unsigned char *buf, int *ret_len);
int lcmp_DER_certreq(LCMPOp *op, unsigned char *buf, int *ret_len);
int lcmp_DER_certrsp(LCMPOp *op, unsigned char *buf, int *ret_len);
int lcmp_DER_csrreq(LCMPOp *op, unsigned char *buf, int *ret_len);
int lcmp_DER_csrrsp(LCMPOp *op, unsigned char *buf, int *ret_len);
int lcmp_DER_crlreq(LCMPOp *op, unsigned char *buf, int *ret_len);
int lcmp_DER_crlrsp(LCMPOp *op, unsigned char *buf, int *ret_len);
int lcmp_DER_svopreq(LCMPOp *op, unsigned char *buf, int *ret_len);
int lcmp_DER_svoprsp(LCMPOp *op, unsigned char *buf, int *ret_len);
int lcmp_DER_extreq(LCMPOp *op, unsigned char *buf, int *ret_len);
int lcmp_DER_extrsp(LCMPOp *op, unsigned char *buf, int *ret_len);
int lcmp_DER_renewalreq(LCMPOp *op, unsigned char *buf, int *ret_len);
int lcmp_DER_renewalrsp(LCMPOp *op, unsigned char *buf, int *ret_len);

int lcmp_DER_statlist(CertStat *top, unsigned char *ret, int *ret_len);
int lcmp_DER_tmpl(ProfTmpl *pf, unsigned char *buf, int *ret_len);
int lcmp_DER_delext(CertExt *ext, unsigned char *buf, int *ret_len);
int lcmp_DER_keytype(Key *key, unsigned char *buf, int *ret_len);
int lcmp_DER_userinfo(AuthInfo *ai, unsigned char *buf, int *ret_len);

int LCMPop_estimate_der_size(LCMPOp *op);
int lcmp_dsz_ctrl(LCMPCtrl *ct);
int lcmp_dsz_certlist(CertList *cl);
int lcmp_dsz_crllist(CRLList *cl);
int lcmp_dsz_ext(CertExt *ext);
int lcmp_dsz_prof(ProfTmpl *pf);
int lcmp_dsz_certstat(CertStat *st);
int lcmp_dsz_key(Key *key);
int lcmp_dsz_userinfo(AuthInfo *ai);

/* lcmp_net.c */
int LCMP_read(LCMP *lc);
int LCMP_write(LCMP *lc);
#define LCMP_set_sslsock(lc,s)	(lc->sock = (s))

char *LCMP_msg2str(int msg);

/* lcmp_tool.c */
LCMPOp *LCMP_get_response(int opid, int rsp, char *msg);

LCMPOp *LCMP_get_bindreq(char *ca, char *user, char *pwd, int auth);
LCMPOp *LCMP_get_bindrsp(PKCS12 *p12);
LCMPOp *LCMP_get_signreq(char *profname, enum csrform csrFormat,
			 void *req, int serialNum);
LCMPOp *LCMP_get_signrsp(Cert *ct);
LCMPOp *LCMP_get_listreq(char *profname, int serialNum);
LCMPOp *LCMP_get_listreq_(char *profname, int serialNum, char *sbjQuery);
LCMPOp *LCMP_get_listrsp(CertStat *cs, int serialNum);
LCMPOp *LCMP_get_profreq(int pfOp, char *profname, ProfTmpl *tmpl, CertExt *ext);
LCMPOp *LCMP_get_profreq_(int pfOp, char *profname, ProfTmpl *tmpl,
			  CertExt *ext, char *rename);
LCMPOp *LCMP_get_profrsp(LCMP *lc, ProfTmpl *tmpl, CertExt *ext,
			 CertProf *cpf);
LCMPOp *LCMP_get_certreq(int certOp, int snum, int reason, Key *key, char *kpwd);
LCMPOp *LCMP_get_certrsp(Cert *ct, Key *key);
LCMPOp *LCMP_get_csrreq(int csrOp, int accID, int snum, char *pfname,
			Req *csr, int reason, CertTemplate *tmpl);
LCMPOp *LCMP_get_csrrsp(int accID, int snum, Req *csr,	CertTemplate *tmpl);
LCMPOp *LCMP_get_crlreq(int crlOp, char *profname, int location, char *path);
LCMPOp *LCMP_get_crlrsp(CRLList *top);
LCMPOp *LCMP_get_svopreq(int svOp, AuthInfo *ai);
LCMPOp *LCMP_get_svoprsp(unsigned char *opRsp, int rspLen);
LCMPOp *LCMP_get_extreq(char *oid, unsigned char *bin, int blen);
LCMPOp *LCMP_get_extrsp(char *oid, unsigned char *bin, int blen);
LCMPOp *LCMP_get_renewalreq(int snum, enum csrform csrFormat, void *req,
			    int serialNum);
LCMPOp *LCMP_get_renewalrsp(Cert *ct);

#define LCMP_get_certupd(snum) LCMP_get_certreq(LCMP_OPCT_UPDCERT,(snum),0,NULL,NULL)
#define LCMP_get_certexp(snum) LCMP_get_certreq(LCMP_OPCT_EXPCERT,(snum),0,NULL,NULL)
#define LCMP_get_revoke(snum,reason) LCMP_get_certreq(LCMP_OPCT_RVKCERT,(snum),(reason),NULL,NULL)
#define LCMP_get_unrevoke(snum) LCMP_get_certreq(LCMP_OPCT_UNRVKCERT,(snum),0,NULL,NULL)
#define LCMP_get_keyimp(snum,key,pass) LCMP_get_certreq(LCMP_OPCT_IMPKEY,(snum),0,(key),(pass))
#define LCMP_get_keyexp(snum,pass) LCMP_get_certreq(LCMP_OPCT_EXPKEY,(snum),0,NULL,(pass))
#define LCMP_get_keydel(snum,pass) LCMP_get_certreq(LCMP_OPCT_DELKEY,(snum),0,NULL,(pass))

#define LCMP_get_csrpost(csr,snum) LCMP_get_csrreq(LCMP_OPCSR_POST,0,(snum),NULL,(csr),0,NULL)
#define LCMP_get_tmplpost(tmpl) LCMP_get_csrreq(LCMP_OPCSR_POST,0,(snum),NULL,NULL,0,(tmpl))
#define LCMP_get_csrexp(accID) LCMP_get_csrreq(LCMP_OPCSR_EXP,(accID),0,NULL,NULL,0,NULL)
#define LCMP_get_csrdel(accID) LCMP_get_csrreq(LCMP_OPCSR_DEL,(accID),0,NULL,NULL,0,NULL)
#define LCMP_get_csrissue(accID,pfname) LCMP_get_csrreq(LCMP_OPCSR_ISSUE,(accID),0,(pfname),NULL,0,NULL)
#define LCMP_get_csrreject(accID,reason) LCMP_get_csrreq(LCMP_OPCSR_ISSUE,(accID),0,NULL,NULL,(reason),NULL)

/* lcmp_api.c */
LCMP *LCMP_init(char *svname, int port, char *caname);
void LCMP_close(LCMP *lc);
int LCMP_bind_s(LCMP *lc, char *userid, char *passwd, int auth);
int LCMP_unbind_s(LCMP *lc);
int LCMP_sign_s(LCMP *lc, char *profile, int sn, void *req,
		enum csrform csrFormat);
int LCMP_list_s(LCMP *lc, char *profile, int sn);
int LCMP_listsbj_s(LCMP *lc, char *profile, char *query);
int LCMP_prof_s_(LCMP *lc, int pfOp, char *profile, ProfTmpl *tmpl,
		 CertExt *ext, char *rename);
int LCMP_prof_s(LCMP *lc, int pfOp, char *profile, ProfTmpl *tmpl, CertExt *ext);
int LCMP_cert_s(LCMP *lc, int certOp, int snum, int reason, Key *key, char *pwd);
int LCMP_csr_s(LCMP *lc, int csrOp, int accID, int snum, char *pfname,
	       Req *csr, int reason, CertTemplate *tmpl);
int LCMP_crl_s(LCMP *lc, int crlOp, char *profile);
int LCMP_svop_s(LCMP *lc, int svOp, void *opInfo);
int LCMP_extop_s(LCMP *lc, char *oid, unsigned char *bin, int blen);
int LCMP_getsock(LCMP *lc);

int LCMP_set_ssl(LCMP *lc, char *store_path, char *certid, char *pass,
		 int vfymode);

Cert *LCMP_get_signcert(LCMP *lc);
CertStat *LCMP_get_statptr(LCMP *lc);
char **LCMP_get_proflist(LCMP *lc, int *listnum);
void LCMP_proflist_free(char **list, int num);
ProfTmpl *LCMP_get_profinfo(LCMP *lc);
CertExt *LCMP_get_profext(LCMP *lc);
Cert *LCMP_get_expcert(LCMP *lc);
Key *LCMP_get_expkey(LCMP *lc);
Req *LCMP_get_expcsr(LCMP *lc);
CertTemplate *LCMP_get_exptmpl(LCMP *lc);
int LCMP_get_acceptid(LCMP *lc);
int LCMP_get_serialnum(LCMP *lc);
CRL *LCMP_get_signcrl(LCMP *lc);

#define LCMP_get_updcert(lc) LCMP_get_expcert((lc))

int LCMP_renewal_s(LCMP *lc, int sn, void *req, enum csrform csrFormat);
Cert *LCMP_get_renewalcert(LCMP *lc);

#ifdef  __cplusplus
}
#endif

#endif /* INCLUSION_GUARD_UUID_9307CCFC_CC00_4B29_BCD4_1D6A8532D73C */
