/* ok_ca.h */
/*
 * Copyright (c) 2004-2015 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#ifndef INCLUSION_GUARD_UUID_DE5BAAE3_AF07_4BA2_9B54_1A887976054E
#define INCLUSION_GUARD_UUID_DE5BAAE3_AF07_4BA2_9B54_1A887976054E

#include <stdio.h>
#include <time.h>

#include <aicrypto/ok_tool.h>	/* AILock, ... */
#include <aicrypto/ok_pkcs11.h>
#include <aicrypto/ok_cmp.h>	/* CertTemplate, ... */

#ifdef  __cplusplus
extern "C" {
#endif

/* certificate status structure */
/*
 * this structure has certificate information used for
 * CA operations. FILE pointer and its position indecates
 * *.cts (certificates) file can be used.
 */
typedef struct certificate_status CertStat;
struct certificate_status {
	int state;	/* misspell.. means status;; */
	int serialNum;
	char *subject;
	int acceptID;
	unsigned char *keyID;

	struct tm createdDate;
	struct tm notBefore;
	struct tm notAfter;
	struct tm revokedDate;
	CertStat *next;
	CertStat *prev;

	FILE *fp, *kfp;
	fpos_t pos, kpos;
};

/* this is for compatibility of old req file */
typedef struct request_state ReqStat;
struct request_state {
	char *subject;
	int serialNum;
	struct tm date;

	ReqStat *next;
	ReqStat *prev;

	FILE *rfp, *kfp;
	fpos_t rpos, kpos;
};

/* certificate request state structure */
/*
 * this structure has CSR information used for CA operations.
 * CSRStat is compatible with CertStat.
 */
typedef CertStat CSRStat;

typedef struct certificate_authority CA;
typedef struct certificate_profile CertProf;
typedef struct crl_profile CRLProf;
typedef struct csr_profile CSRProf;

/*
 * pol_flag[16]
 *   [0] profile working policy: set CPF_WP0_XXX [include/ok_ca.h]
 *   [1] profile working policy: set CPF_WP1_XXX [include/ok_ca.h]
 *   [2] unused.
 *   [3] unused.
 *   [4] CSR subject matching policy: C, ST, L, O
 *   [5] CSR subject matching policy: OU, CN, UID, Em
 *   [6]..[15] unused.
 * 
 * See also: struct lcmp_proftemple [include/ok_lcmp.h]
 *           CA_input_profpolicy() [aica/aica_pol.c]
 *           CA_check_profpol() [ca/ca_sign.c]
 *           CA_input_profdnpolicy() [aica/aica_pol.c]
 *           CA_dnexist_check() [ca/ca_tool.c]
 * 
 * 
 * - upd_sec
 *   upd_sec stored the period that can be renewal the certificate
 *   prior to expiration.
 * 
 *   See also: struct lcmp_proftemple [include/ok_lcmp.h]
 * 
 */
struct certificate_profile {
	int ver;
	int isCross;
	CA *pCA;	/* pointer */
	CertStat *stat;	/* allocated */
	char *ca_name;	/* pointer */
	char *name;	/* allocated */

	FILE *fp, *kfp;	/* allocated */
	fpos_t pos, kpos;

	CertExt *ext;	/* allocated */

	int serialNum;
	int cert_ver;
	int cert_sec;
	int upd_sec;
	int keylong;
	int sigtype;

	int bgtype, edtype, updtype;

	struct tm ct_begin;
	struct tm ct_end;

	CertDN sbj_tmpl;	/* allocated */

	unsigned char pol_flag[16];

	unsigned char *der;	/* allocated */

	/* system info */
	int gid;	/* group id */
	int msz;	/* last file size */
	time_t mtm;	/* last file update time */
	AILock lock;	/* mutex lock */

	CertProf *next;
};

struct crl_profile {
	int ver;
	CA *pCA;	/* pointer */
	char *ca_name;	/* pointer */
	char *name;	/* allocated */

	CertExt *ext;	/* allocated */

	int crlNum;
	int crl_sec;
	int crl_ver;
	int sigtype;

	int bgtype;

	struct tm cl_begin;

	unsigned char *der;	/* allocated */

	/* system info */
	int gid;	/* group id */
	int msz;	/* last file size */
	time_t mtm;	/* last file update time */
	AILock lock;	/* mutex lock */

	CRLProf *next;
};

struct csr_profile {
	int ver;
	CA *pCA;	/* pointer */
	char *ca_name;	/* pointer */

	int csrNum;

	CSRStat *stat;	/* allocated */

	FILE *fp;	/* allocated */
	fpos_t pos;

	unsigned char *der;	/* allocated */

	/* system info */
	int msz;	/* last file size */
	time_t mtm;	/* last file update time */
	AILock lock;	/* mutex lock */

	CSRProf *next;
};

struct certificate_authority {
	PKCS12 *p12;
	PKCS11 *p11;	/* used only with HW Key */
	P11Session *p11s;	/* keep p11 session handle */
	CA *next;

	char *subject;
	char *issuer;

	Cert *cert;
	Key *prvkey;
	CRL *crl;

	CertExt *ext;

	long serialNum;
	int cert_ver;
	int cert_days;
	int crl_ver;
	int crl_num;
	int crl_days;

	CertStat *stat;

	/* der */
	unsigned char *der;

	FILE *fp;
	fpos_t pos;

	unsigned char *policy;

	int ver;	/* CA version */
	int sigtype;
	int keylong;
	int sbj_ft;
	int plginsign;
	int ca_update;
	int use_p11;
	int p11_nofinal;	/* dont exec pkcs11 C_Finalize */
	int p11_mofn;	/* pkcs11 key m of n */
	char *ca_name;
	char *ca_path;
	char *lib_path;
	char *label;	/* label for PKCS#11 key */

	int sbj_num;
	char *m_tag;
	char *sbjList[16];

	/* system info */
	time_t ca_mtime;	/* CA file update time */
	time_t pw_mtime;	/* ca.passwd file update time */
	time_t gp_mtime;	/* ca.group file update time */
	AILock slock;	/* sign mutex lock */
	AILock clock;	/* ca.cai mutex lock */
	AILock ulock;	/* ca.passwd mutex lock */

	CertProf *cprof;	/* current active profile */
	CertProf *profList;
	CRLProf *lprof;	/* current active profile */
	CRLProf *crlpList;
	CSRProf *csrProf;	/* accepted CSR list */
};

typedef struct authinfo AuthInfo;
struct authinfo {
	char *name;
	char *group;
	char *passwd;
	char *oldpasswd;
	char *opt;

	int uid;
	int gid;
	unsigned long grant;

	AuthInfo *next;
};

#define	STAT_OK		0
#define STAT_CA		0x1
#define	STAT_EXPIRED	0x2
#define	STAT_REVOKED	0x4
#define STAT_XCERTFWD	0x10
#define STAT_XCERTREV	0x20
#define	STAT_HAVEKEY	0x100

#define STAT_RR_unspecified		0x00000
#define STAT_RR_keyCompromise		0x10000
#define STAT_RR_cACompromise		0x20000
#define STAT_RR_affiliationChanged	0x30000
#define STAT_RR_superseded		0x40000
#define STAT_RR_cessationOfOperation	0x50000
#define STAT_RR_certificateHold		0x60000
#define STAT_RR_removeFromCRL		0x80000
#define STAT_RR_privilegeWithdrawn	0x90000
#define STAT_RR_aaCompromise		0xa0000

#define	CSR_STAT_WAIT	0x1
#define	CSR_STAT_SIGNED	0x2
#define	CSR_STAT_REJECT	0x4

/*
 * profile working policy :
 * These are flags for the pol_flag member of CertProf.
 */
#define CPF_WP0_allowSameSbjDN		0x80
#define CPF_WP0_allowExpdSbjDN		0x40
#define CPF_WP0_allowRvkdSbjDN		0x20
#define CPF_WP0_allowInUpdSbjDN		0x10
#define CPF_WP0_allowSameKey		0x08
#define CPF_WP0_allowExpdKey		0x04
#define CPF_WP0_allowRvkdKey		0x02
#define CPF_WP0_allowInUpdKey		0x01
#define CPF_WP1_replaceWithTmplDN	0x80
#define CPF_WP1_replaceDNWithoutOU	0x40

/* ca.c */
CA *CA_new(void);
void CA_free(CA *ca);

CertStat *CertStat_new(void);
void CertStat_free(CertStat *cs);
void CertStat_free_all(CertStat *cs);

CertStat *CertStat_dup(CertStat *org);
CertStat *CertStat_dup_all(CertStat *top);

int CA_set_ca_path(CA *ca, char *path);

/* ca_asn1.c */
unsigned char *CA_toDER(CA *ca, unsigned char *buf, int *ret_len);
int CA_DER_extns(CA *ca, unsigned char *ret, int *ret_len);
int CA_DER_castat(CA *ca, unsigned char *ret, int *ret_len);
int CA_estimate_der_size(CA *ca);
char *CA_get_defsubstr(CA *ca);

CA *ASN1_CA_info(unsigned char *der);
int ASN1_CA_castat(CA *ca, unsigned char *in);
int ASN1_CA_certstat(CA *ca, unsigned char *in);

/* ca_file.c */
CA *CA_info_read(char *path);
int CA_info_write(CA *ca);
int CA_info_reload(CA *ca);
int CA_p12_read(CA *ca);
int CA_p11_read(CA *ca);
int CA_write_cert(CA *ca, Cert *cert, char *fname);
int CA_write_crl(CA *ca, char *fname);

/* ca_tool.c */
int CA_now_certnum(CA *ca);
int CA_can_serialNum_(CA *ca, int sn);
int CA_can_serialNum(CA *ca, Cert *req, int sn);
int CA_can_resign(CA *ca, Cert *ct);
int CA_inrenewal_check(CertStat *st, int upd_sec);
int CA_expire_check(CA *ca);
int CA_dnexist_check(CA *ca, CertProf *cpf, Cert *ct);
int CA_replace_subject(CertDN *tmpl, Req *req);
int CA_replace_subject_(CertDN *tmpl, Req *req, unsigned char *pol_flag);

int CA_key2sigalgo(int type);
int sighash2keyobj(int sig);
int keyhash2sigobj(int ktype, int hash, int oldsig);

int CA_add_cert(CA *ca, Cert *ct);
int CA_update_cert(CA *ca, Cert *ct);
int CA_add_certfile(CA *ca, Cert *ct);
void CA_delete_stat(CertStat *st);
CertStat *CA_find_stat(CA *ca, int serial);
CertStat *CA_find_statstr(CA *ca, char *subject);
CertStat *CA_find_statkeyID(CA *ca, unsigned char *kID);
CertStat *CA_find_sbjquery(CA *ca, char *query);
void CA_find_statstr_(CertProf **cur_pf, CertStat **cur_st, char *subject);
void CA_find_statkeyID_(CertProf **cur_pf, CertStat **cur_st,	unsigned char *kID);
CertProf *CA_find_prof_sn(CA *ca, int serial);
CertProf *CA_find_prof_sbj(CA *ca, char *subject);
CertProf *CA_find_prof_keyid(CA *ca, unsigned char *kID);

AILock CA_init_lock(char *ca, char *prof);
int CA_release_lock(AILock al);
int CA_lock(AILock al, int msec);
int CA_unlock(AILock *al);

/* ca_crl */
int CA_cert_revoke(CA *ca, int revoke, int reason);
int CA_cert_unrevoke(CA *ca, int unrev);
int CA_set_CRL(CA *ca);
int CA_crl_last_next(CA *ca, CRL *crl);
int CA_crl_set_revoked(CA *ca, CRL *crl, int *rev_num);
int CA_crl_set_ext(CA *ca, CRL *crl);

/* ca_err.c */
char *ca_get_err_location(int err);
char *ca_get_err_type(int err);
char *CA_get_errstr();

/* ca_sign.c */
int CA_sign(CA *ca, Req *req);
int CA_sign_validity(CA *ca, Cert *ct);
int CA_sign_ext(CA *ca, Cert *ct);
int CA_check_profpol(CA *ca, CertProf *cpf, Req *req, unsigned int flags);

/* CA_check_profpol flags */
#define CHKPOL_REUSE_SAMESBJDN		0x0001	/* reuse same subject DN */
#define CHKPOL_REUSE_EXPIREDSBJDN	0x0002	/* reuse expired subject DN */
#define CHKPOL_REUSE_REVOKEDSBJDN	0x0004	/* reuse revoked subject DN */
#define CHKPOL_REUSE_SBJDNINRENEWAL	0x0008	/* reuse subject DN in renewal period */

#define CHKPOL_REUSE_SAMEPUBKEY 	0x0010	/* reuse same public key */
#define CHKPOL_REUSE_EXPIREDPUBKEY	0x0020	/* reuse expired public key */
#define CHKPOL_REUSE_REVOKEDPUBKEY	0x0040	/* reuse revoked public key */
#define CHKPOL_REUSE_PUBKEYINRENEWAL	0x0080	/* reuse public key in renewal period */

/* ca_auth.c */
AuthInfo *CA_ai_new();
void CA_ai_free(AuthInfo *ai);
void CA_ai_free_all(AuthInfo *top);
AuthInfo *CA_ai_dup(AuthInfo *org);
AuthInfo *CA_ai_dup_all(AuthInfo *top);

AuthInfo *CA_read_authinfo(char *name);
int CA_write_authinfo(AuthInfo *ai, char *fname);

AuthInfo *CA_get_authinfo(char *name, char *raw_pwd, int uid, int gid,
			  int grant, char *opt);
AuthInfo *CA_lcmp_authinfo(char *name, char *pwd, char *oldpwd, int uid,
			   int gid, int grant, char *opt);
int CA_get_md5pwd(char *raw_pwd, char *ret, int max);
int CA_cmp_md5pwd(char *raw_pwd, char *md5_pwd);
int md5_hash_pwd(char *in, char *ret, int max);

AuthInfo *CA_ai_findbyname(AuthInfo *top, char *name);
AuthInfo *CA_ai_findbyuid(AuthInfo *top, int uid);

int CA_read_groupinfo(CA *ca);
int CA_write_groupinfo(CA *ca);

/* ca_csr.c */
int CA_init_csrProf(CA *ca);
int CA_is_csrProf_exist(CA *ca);

/* prof.c */
CertProf *Prof_cert_new(CA *ca, char *name);
void Prof_cert_free(CertProf *cpf);
void Prof_cert_free_all(CertProf *top);

CRLProf *Prof_crl_new(CA *ca, char *name);
void Prof_crl_free(CRLProf *cpf);
void Prof_crl_free_all(CRLProf *top);

CSRProf *Prof_csr_new(CA *ca);
void Prof_csr_free(CSRProf *rpf);
void Prof_csr_free_all(CSRProf *top);

/* prof_tmpl.c */
int Prof_cert_settmpl(CA *ca, char *name);
int Prof_crl_settmpl(CA *ca, char *name);
int get_tmpl_path(char *buf, int max);
int Prof_set_sbjtmpl(CertDN *org, CertDN *dst);

/* prof_asn1.c */
int ASN1_ctprof_info(CA *ca, unsigned char *in);
int ASN1_ctprof_stat(CA *ca, unsigned char *in);
int ASN1_ctprof_ctstat(CA *ca, unsigned char *in);
int ASN1_ctprof_ctfile(CA *ca, int sn, int mode);
int ASN1_ctprof_keyfile(CA *ca, int sn, int mode);
CertStat *asn1_ctprof_ctstat(unsigned char *in);

int ASN1_clprof_info(CA *ca, unsigned char *in);
int ASN1_clprof_stat(CA *ca, unsigned char *in);
int ASN1_prof_ext(CA *ca, unsigned char *in, unsigned char pf);

#define ASN1_ctprof_ext(ca,in)	ASN1_prof_ext((ca),(in),0xa3)
#define ASN1_clprof_ext(ca,in)	ASN1_prof_ext((ca),(in),0xa4)

int ASN1_csrprof_info(CA *ca, unsigned char *in);
int ASN1_csrprof_stat(CA *ca, unsigned char *in);
int ASN1_csrprof_ctstat(CA *ca, unsigned char *in);

int Prof_estimate_ctdersize(CertProf *cpf);
unsigned char *Prof_cert_toDER(CA *ca, unsigned char *buf, int *ret_len);
int Prof_DER_ctstat(CertProf *cpf, unsigned char *ret, int *ret_len);
int Prof_DER_ctctstat(CertProf *cpf, unsigned char *ret, int *ret_len);
int Prof_DER_ext(CA *ca, char exp, unsigned char *ret, int *ret_len);
int prof_der_ctctstat(CertStat *top, unsigned char *ret, int *ret_len);

#define Prof_DER_ctext(ca,ret,ret_len)	Prof_DER_ext((ca),3,(ret),(ret_len))
#define Prof_DER_clext(ca,ret,ret_len)	Prof_DER_ext((ca),4,(ret),(ret_len))

int Prof_estimate_cldersize(CRLProf *cpf);
unsigned char *Prof_crl_toDER(CA *ca, unsigned char *buf, int *ret_len);
int Prof_DER_clstat(CRLProf *cpf, unsigned char *ret, int *ret_len);

unsigned char *Prof_csr_toDER(CA *ca, unsigned char *buf, int *ret_len);
int Prof_DER_csrstat(CSRProf *rpf, unsigned char *ret, int *ret_len);
int Prof_DER_csrctstat(CSRProf *rpf, unsigned char *ret, int *ret_len);

/* prof_file.c */
/**
 * open a certificate profile in DER ASN.1 encoding format.
 */
int Prof_open_ctinfo(CA *ca);
int Prof_save_ctinfo(CA *ca);
int Prof_reload_ctinfo(CA *ca);
int Prof_open_ctfile(CA *ca);
int Prof_open_keyfile(CA *ca);

/**
 * open a CRL profile in DER ASN.1 encoding format.
 */
int Prof_open_clinfo(CA *ca);
int Prof_save_clinfo(CA *ca);
int Prof_reload_clinfo(CA *ca);

/**
 * open a CSR profile in DER ASN.1 encoding format.
 */
int Prof_open_csrinfo(CA *ca);
int Prof_save_csrinfo(CA *ca);
int Prof_reload_csrinfo(CA *ca);

int Prof_save_ctinfo_(CA *ca, char *path);
int Prof_reload_ctinfo_(CA *ca, char *path);
int Prof_open_ctfile_(CA *ca, char *path, int sn, int mode);
int Prof_open_keyfile_(CA *ca, char *path, int sn, int mode);

int Prof_save_clinfo_(CA *ca, char *path);
int Prof_reload_clinfo_(CA *ca, char *path);

int Prof_save_csrinfo_(CA *ca, char *path);
int Prof_reload_csrinfo_(CA *ca, char *path);

int prof_get_curstat(char *path, time_t *t, int *sz);
Cert *Prof_get_cert(CertStat *rs);
Key *Prof_get_key(CertStat *cs);
/**
 * get a CSR from <ca_path>/req/XXXXXXXX.req.
 */
Req *Prof_get_csr(CSRProf *rpf, CertStat *cs);
/**
 * get a CSR template.
 */
CertTemplate *Prof_get_csrtmpl(CSRProf *rpf, CertStat *cs);

/* prof_tool.c */
int Prof_now_certnum(CertProf *cpf);
int Prof_now_extnum(CertProf *cpf);
int Prof_expire_check(CertProf *cpf);
int prof_expire_check_(CertStat *stat);
int prof_stat_expire(CertStat *st, time_t now);
CertStat *Prof_find_stat(CertStat *top, int serial);
CertStat *Prof_find_statstr(CertStat *top, char *subject);
CertStat *Prof_find_stataid(CertStat *top, int acceptID);
CertStat *Prof_find_statkeyID(CertStat *top, unsigned char *keyID);
CertStat *Prof_find_sbjquery(CertStat *top, char *query);
CertProf *Prof_find(CA *ca, char *name);
CRLProf *Prof_crl_find(CA *ca, char *name);

int Prof_add_cert(CertProf *cpf, Cert *ct);
void Prof_delete_stat(CertStat *st);
int Prof_add_certfile(CertProf *cpf, Cert *ct);
int Prof_add_keyfile(CertProf *cpf, Key *key, int sn);
int Prof_del_keyfile(CertStat *cs);
int Prof_add_xcertfile(CertProf *cpf, CertPair *cp);

int Prof_add_csr_(CSRProf *rpf, char *subject, int sn);
int Prof_add_csr(CSRProf *rpf, Req *req, int sn);
int Prof_add_csrtmpl(CSRProf *rpf, CertTemplate *tmpl);
void Prof_del_csr(CSRProf *cpf, CertStat *cs);
int Prof_add_csrfile_(CSRProf *rpf, unsigned char *der, int accID);
int Prof_add_csrfile(CSRProf *cpf, Req *req, int accID);
int Prof_del_csrfile(CSRProf *cpf, CertStat *cs);

int Prof_add_ext(CertProf *cpf, CertExt *et);
int Prof_add_crlext(CRLProf *lpf, CertExt *et);
int Prof_delete_ext(CertProf *cpf, int id);
int Prof_delete_crlext(CRLProf *lpf, int id);

/* prof_conv.c */
int CA_reqkey2newkey(CA *ca);
int CA_is_oldreq_exist(CA *ca);

/* utils.c */
int set_path(char *path, const size_t sz_max, ...);

#ifdef  __cplusplus
}
#endif

#endif /* INCLUSION_GUARD_UUID_DE5BAAE3_AF07_4BA2_9B54_1A887976054E */
