/* gridmapgen.c */
/*
 * Copyright (c) 2004-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <unistd.h>

#include <aicrypto/ok_io.h>

#include "gridmapgen.h"

#ifndef GMAPCONF
#define GMAPCONF "gridmap.cnf"
#endif

char conf[256]    = GMAPCONF;

/* gridmap_conf.c */
extern char workdir[];

extern char usermap[];
extern char gridmap[];
extern char uudb_bin[];

extern int interval;

extern char refmap[REFMAX][128];
extern char refcerts[REFMAX][128];

int sesid = 0;
int end   = 0;

/* functions */
int GMap_do_operation();
void GMap_sig();

void options(int argc,char **argv);
void usage();

/* common/version.c */
void print_version(char *argv[]);

#define DEVNULL        "/dev/null"
#define OPENOPT        O_RDWR

/*-------------------------------------------------
  grid map generator main
-------------------------------------------------*/
int main(int argc,char **argv){
  int i,nullfd,err=-1;
  time_t wk1,wk2;
  pid_t pid;

  options(argc,argv);

  /* set signal handling */
  signal(SIGINT, GMap_sig);
  signal(SIGHUP, GMap_sig);
  signal(SIGTERM, GMap_sig);

  /* set configuration */
  if(GMap_read_config(conf)){
    printf("cannot read a config file : %s\n",conf);
    goto done;
  }

#ifndef _DEBUG

  /* fork and exit parent */
  if((pid=fork()) != 0){
    printf("start gridmapgen daemon process (%d)\n",pid);
    err=EXIT_SUCCESS;
    goto done;
  }

  /* set background process */
#ifdef SETPGRP_VOID 
  setpgrp();
#else
  setpgrp(0,0);
#endif


  /* close stdout and stderr */
  close(0);
  close(1);
  close(2);
  nullfd=open(DEVNULL,OPENOPT);
  dup2(nullfd,0);
  dup2(nullfd,1);
  dup2(nullfd,2);
#endif

  /* loop and interval work */
  for(;;){
    err = -1; sesid++;
    time(&wk1); /* get begining time */

    /* do operation */
    if(GMap_do_operation()) goto loop_done;

    err = 0;
loop_done:
    time(&wk2); wk1=wk2-wk1; /* get working time */

    /* check end time */
    if(end>0) break;

    /* sleep */
    i = interval-wk1;
    i = (i>0)?(i):(0);

    if(sleep(i)>0) break; /* may receive signal */
  }

  err=0;
done:
  return err;
}

/*------------------------------------------------*/
void GMap_sig(){
  end = 1;
  kill(0,SIGALRM);
}

/*-----------------------------------------
  usage and option check
-----------------------------------------*/
void options(int argc,char **argv){
  int i;

  /* check options */
  for(i=1;i<argc;i++){
    if(!strcmp("-help",argv[i])){
      usage();
      exit(EXIT_SUCCESS);
    }else if(!strcmp("-version",argv[i])){
      print_version(argv);
      exit(EXIT_SUCCESS);
    }else if(!strcmp("-conf",argv[i])){
      i++;
      if(i<argc) strncpy(conf,argv[i],254);
    }else if(!strcmp("-itv",argv[i])){
      i++;
      if(i<argc) interval = atoi(argv[i]);
    }else{
      goto usage;
    }
  }
  return;

usage:
  printf("option error!\n");
  printf("unknown option: `%s'\n", argv[i]);
  usage();
  exit(EXIT_FAILURE);
}

void usage(){
  printf("\
Usage: gridmapgen [OPTION...]\n\
\n\
Options:\n\
  -conf PATH	set the path for a gridmapgen configuration file\n\
  -itv TIME	set working interval (second)\n\
  -help		print this message\n\
  -version	print version information and exit\n\
");
}

/*------------------------------------------------
  do Grid map generator operation
------------------------------------------------*/
int GMap_do_operation(){
  GMapList *list=NULL,*old=NULL;
  int ret=-1;

  if(GMap_get_mapfiles(&list)){
    GMAPERRLOG("cannot construct user map information",NULL);
    goto done;
  }

  if(GMap_check_csvfile(&list)){
    GMAPERRLOG("cannot normalize user map information with csv file",NULL);
    goto done;
  }

  if(GMap_save_mapfile(list)){
    GMAPERRLOG("cannot generate new grid-mapfile",NULL);
    goto done;
  }

  /* UNICORE mapfile generation */
  if(*uudb_bin){
    old = GMap_load_oldmapfile(); /* maybe NULL */

    if(GMap_modify_uudb(list,old)){
      GMAPERRLOG("cannot modify UNICORE UUDB",NULL);
      goto done;
    }
  }
      
  ret=0;
done:
  GMapList_free_all(list);
  GMapList_free_all(old);
  return ret;
}

int GMap_get_mapfiles(GMapList **ret){
  char *buf=NULL;
  GMapList *tail=NULL,*list=NULL;
  int i,ok=-1;

  for(i=0; i<REFMAX; i++){
    if(refmap[i][0] == 0) break; /* no url specified */

    if((buf=GMap_get_urlfile(refmap[i]))==NULL){
      GMAPERRLOG("failed to get grid-mapfile",refmap[i]);
      goto done;
    }

    if(GMap_mapfile2list(buf,&list,i,0)){
      GMAPERRLOG("failed to analyze grid-mapfile",refmap[i]);
      goto done;
    }

    if(tail){
      tail->next = list;
    }else{
      *ret = list;
    }
    while(list){
      if(list->next==NULL) tail = list;
      list = list->next;
    }
    /* list should be NULL */
    free(buf); buf=NULL;

    GMAPACCLOG("success to get grid-mapfile",refmap[i]);
  }

  ok = 0;
done:
  GMapList_free_all(list); /* list should be NULL */
  if(buf) free(buf);
  return ok;
}

int GMap_check_csvfile(GMapList **list){
  GMapList *gm,*hd,*ret=NULL;
  char *cp,*t,*nx,*buf=NULL;
  char local[64],global[64];
  int i,j,in,ok=-1;
  off_t sz;

  if((buf=cp=get_file2buf(usermap,&sz))==NULL){
    GMAPERRLOG("cannot open user map file",usermap);
    goto done;
  }
  
  while(cp && *cp){
    if((nx=strchr(cp,'\n')) != NULL) {
      if(*(nx-1)=='\r') *(nx-1) = 0;
      *nx = 0; nx++;
    }

    memset(local,0,64);
    memset(global,0,64);
    i=j=in=0; t=local;

    if(*cp=='#'){ cp = nx; continue; } /* ignore comment line */

    while(*cp && ((*cp==' ')||(*cp=='\t'))) cp++;

    /* get user name & global name */
    while(*cp){
      switch(*cp){
      case '"':
        if(in){
          in=0;
        }else{
          in =1; i=0;
          t = (j)?(global):(local);
        }
        break;
      case ',':
        if(!in){
          i=0; j=1; t=global;
        }else{
          t[i] = *cp; i++;
        }
        break;
      case '\\':
        t[i] = *(cp+1); i++; cp++;
        break;
      default:
        t[i] = *cp; i++;
        break;
      }
      cp++;
      if(i>62) i--;
    }

    /* get find and marge grid user */
    if((gm=GMap_find_byglobal(*list,global)) != NULL) {
      if(gm->prev){
        gm->prev->next = gm->next;
        if(gm->next) gm->next->prev = gm->prev;
      }else{
        *list = gm->next;
        if(*list) (*list)->prev = NULL;
      }
      gm->next = NULL;
      strncpy(gm->localuser,local,62);

#ifdef _DEBUG
fprintf(stderr,"csv: local=%s,global=%s,subject=%s\n",local,global,gm->subject);
#endif
      if(ret){
        hd->next=gm; gm->prev=hd; hd=gm;
      }else{
        ret=hd=gm;
      }
    }

    /* set next pointer */
    cp = nx;
  }

  GMapList_free_all(*list);
  *list = ret;

  ok = 0;
done:
  if(buf) free(buf);
  return ok;
}

int GMap_save_mapfile(GMapList *list){
  char old[256],tmp[256],*buf=NULL;
  FILE *fp=NULL;
  int i,j,k,ok = -1;
  off_t sz;

  snprintf(tmp,254,"%s.tmp",gridmap);
  snprintf(old,254,"%s.old",gridmap);

  /* back up old grid-mapfile */
  if((buf=get_file2buf(gridmap,&sz)) != NULL) {
    if((fp=fopen(old,"w"))==NULL){
      GMAPERRLOG("cannot open grid-mapfile",old);
      goto done;
    }
    j=k=0;
    while(j<i){
      if((k=fwrite(&buf[j],sizeof(char),i-j,fp))<0){
	GMAPERRLOG("cannot write grid-mapfile",old);
	goto done;
      }
      j+=k;
    }
    fclose(fp); fp=NULL;
    free(buf); buf=NULL;
  }

  /* create new grid-mapfile */
  if((fp=fopen(tmp,"w"))==NULL){
    GMAPERRLOG("cannot open grid-mapfile",tmp);
    goto done;
  }

  while(list){
    if(fprintf(fp,"\"%s\" %s\n",list->subject,list->localuser)<0){
      GMAPERRLOG("cannot write grid-mapfile",gridmap);
      goto done;
    }
    list = list->next;
  }

  /* grid-mapfile.tmp -> grid-mapfile */
  rename(tmp,gridmap); 

  GMAPACCLOG("success to save grid-mapfile",gridmap);

  ok = 0;
done:
  if(buf) free(buf);
  if(fp) fclose(fp);
  return ok;
}

GMapList *GMap_load_oldmapfile(){
  GMapList *list=NULL;
  char name[256],*buf=NULL;
  off_t sz;

  snprintf(name,254,"%s.old",gridmap);

  if((buf=get_file2buf(name,&sz))==NULL){
    GMAPERRLOG("cannot open old map file",name);
    goto done;
  }
  if(GMap_mapfile2list(buf,&list,0,1)){
    GMAPERRLOG("failed to analyze grid-mapfile",name);
    goto done;
  }

done:
  if(buf) free(buf);
  return list;
}

int GMap_modify_uudb(GMapList *list, GMapList *old){
  GMapList *hd;
  char command[320],url[320],tmp[128],*pem;

  /***** delete certificates from UUDB *****/
  hd = old;
  while(hd){
    if(GMap_find_bylocal(list,hd->localuser)==NULL){
      /* cannot find a user. it should be deleted from UUDB */
      snprintf(command,318,"echo \"1\" | %s/delete %s",uudb_bin,hd->localuser);
      if(system(command)){
	GMAPERRLOG("failed to delete a user from UUDB",hd->localuser);
      }
    }
    hd = hd->next;
  }

  /***** add certificates to UUDB *****/
  hd = list;
  while(hd){
    if(GMap_find_bylocal(old,hd->localuser)==NULL){
      /* cannot find a user. it should be added to UUDB */

      snprintf(url,318,"%s/%s.pem",refcerts[hd->refnum],hd->globaluser);

      if((pem=GMap_get_urlfile(url))==NULL){
	GMAPERRLOG("failed to get a certificate",url);
      }else{
	FILE *fp;
	int fd;

	snprintf(tmp,126,"/tmp/%s.pem",hd->globaluser);
 	fd = open(tmp, O_WRONLY | O_CREAT | O_EXCL, S_IRUSR | S_IWUSR);
 	if (fd == -1) {
 	  fprintf(stderr, "ERROR: %s: %s\n", tmp, strerror(errno));
 	} else {
 	  fp = fdopen(fd, "w");
 	  if (fp != NULL) {
	    fwrite(pem,sizeof(char),strlen(pem),fp);
	    fclose(fp);
	  }

	  /* FIXME: can uudb_bin handle an invalid certificate? */
	  snprintf(command,318,"%s/add %s %s",uudb_bin,tmp,hd->localuser);
	  if(system(command)){
	    GMAPERRLOG("failed to add a user to UUDB",hd->localuser);
	  }

	  unlink(tmp);
	}

	free(pem); pem=NULL;
      }
    }
    hd = hd->next;
  }

  return 0;
}
