/* gridmap_log.c */
/*
 * Copyright (c) 2004-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must 
 * include this notice in the file.
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>

#ifdef HAVE_SYS_TIME_H
# include <sys/time.h>
#endif

#include "gridmapgen.h"

/* global values */
extern char acclog[];
extern char errlog[];

extern int acclog_size;
extern int errlog_size;
extern int sesid;

FILE *acc_fp=NULL;
FILE *err_fp=NULL;


/*-------------------------------------------------
	logging functions
-------------------------------------------------*/
int GMap_log_open(){
	int i=10;
	/* try 10 times to open */
	do{
		if(errlog[0]&&(err_fp==NULL)){
			if((err_fp=fopen(errlog,"a+"))==NULL){
				fprintf(stderr,"gridmap : cannot open error log\n");
			}
		}	
		if(acclog[0]&&(acc_fp==NULL)){
			if((acc_fp=fopen(acclog,"a+"))==NULL){
				fprintf(stderr,"gridmap : cannot open access log\n");
			}
		}
		if(err_fp && acc_fp) break;
		i--;

		{
			struct timeval tv;
			tv.tv_sec  = 0;
			tv.tv_usec = 100*1000; /* ms */
			select(0,NULL,NULL,NULL,&tv); /* wait 100 ms */
		}

	}while(i>0);
	
	return (i==0)?(-1):(0);
}

void GMap_log_close(){
	if(err_fp){ fclose(err_fp); err_fp=NULL; }
	if(acc_fp){ fclose(acc_fp); acc_fp=NULL; }
}

/*-------------------------------------------------
	output log
-------------------------------------------------*/
int GMap_log_out(int type, char *msg, char *opt){
	char tm[64],ss[32],buf[1024];
	time_t t;
	int i=0,j=0,ok=-1;

	if(GMap_log_open()) goto done;

	time(&t);
	strcpy(tm,"[");
	strcat(tm,ctime(&t)); tm[strlen(tm)-1] = 0; /* clear "\n" */

	sprintf(ss," #%.4d] ",sesid);

	/* get message */
	strncpy(buf,tm,64);
	strcat (buf,ss);
	strncat(buf,msg,128);
	if(opt && *opt){
	  strcat (buf," : ");
	  strncat(buf,opt,256);
	}
	strcat (buf,"\n");

	/* write to log file */
	if((type & GMAP_LOG_ERR)&&(err_fp)){
		if(fputs(buf,err_fp)<0) goto done;
		if(fflush(err_fp)<0) goto done;

		if((i=GMap_log_checksize(err_fp,errlog_size))<0) goto done;

		if((i>0) && GMap_log_rotate(errlog,&err_fp)) goto done;
	}
	
	if((type & GMAP_LOG_ACC)&&(acc_fp)){
		if(fputs(buf,acc_fp)<0) goto done;
		if(fflush(acc_fp)<0) goto done;

		if((j=GMap_log_checksize(acc_fp,acclog_size))<0) goto done;

		if((j>0) && GMap_log_rotate(acclog,&acc_fp)) goto done;
	}
	
	ok = 0;
done:
	if(ok) printf("cannot output log file \n%s\n",buf);
	GMap_log_close();

	return ok;
}

/*-------------------------------------------------
	check log size
-------------------------------------------------*/
int GMap_log_checksize(FILE *fp, int size_kb){
	struct stat st;

	if(size_kb<=0) return 0;
	if(fstat(fileno(fp),&st)) goto error;

	return (st.st_size >= (size_kb<<10))?(1):(0);
error:
	return -1;
}

/*-------------------------------------------------
	rotate logs
-------------------------------------------------*/
int GMap_log_rotate(char *fname,FILE **fp){
	time_t t;
	struct tm *ltm;
	char buf[256];

	time(&t);
	ltm = localtime(&t);
	snprintf(buf,254,"%s.%.2d%.2d%.2d%.2d%.2d%.2d",fname,
			ltm->tm_year+1900,ltm->tm_mon+1,ltm->tm_mday,
			ltm->tm_hour,ltm->tm_min,ltm->tm_sec);

	if(*fp){ fclose(*fp); *fp=NULL; }

	/* ignore error case, gridmapgen process should work alone */
	rename(fname,buf);

	if((*fp=fopen(fname,"a+"))==NULL){
		fprintf(stderr,"cannot open log file : %s\n",fname);
		goto error;
	}
	return 0;
error:
	return -1;
}
