/* gridmap_http.c */
/*
 * Copyright (c) 2004-2013 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must 
 * include this notice in the file.
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <netdb.h>

#include "gridmapgen.h"
#include "afi_utils.h"

extern char proxy_host[];
extern int proxy_port;

char http_hd[] ={ 
  "GET %s HTTP/1.1\r\n"
  "Accept: */*\r\n"
  "Host: %s\r\n"
  "%s: Keep-Alive\r\n\r\n"
};

/* gridmap_conf.c */
char *conf_get_line(char *bp, char **np, char *tmp);


/* get map file via http */
char *GMap_get_urlfile(char *url){
  SSL *sock=NULL;
  char svname[256],fname[128],*sv,*ret=NULL;
  int port=80,pt,ssl=0;

  /* analyze url */
  memset(svname,0,128);
  if(gmap_get_address(url,svname,&port,fname,&ssl)) goto done;

  /* get socket */
  sv = svname; pt = port;
  if(*proxy_host){
    sv = proxy_host;
    pt = proxy_port;
  }
  if((sock=gmap_get_sock(sv,pt,ssl))==NULL) goto done;

  /* read file from web server */
  if((ret=gmap_get_webfile(sock,svname,port,fname,ssl))==NULL) goto done;

done:
  SSL_close(sock);
  SSL_free(sock);
  return ret;
}

int gmap_get_address(char *url, char *svname, int *port, char *fname, int *ssl){
  if(parse_url(url,svname,256,port,fname,128,ssl)!=PARSE_URL_PARSE_SUCCESS){
    GMAPERRLOG("unsupported url type",url);
    return -1;
  }

  return 0;
}

SSL *gmap_get_sock(char *server, int port, int ssl){
  struct addrinfo hints;
  struct addrinfo *aihead = NULL, *ai;
  char *port_string = NULL;
  SSL *s = NULL;

  /* get client socket */
  if(asprintf(&port_string, "%d", port) < 0) {
    GMAPERRLOG("cannot get memory",server);
    goto error;
  }
  memset(&hints, 0, sizeof(hints));
  hints.ai_family = AF_UNSPEC;
  hints.ai_socktype = SOCK_STREAM;
  if(getaddrinfo(server, port_string, &hints, &aihead) != 0) {
    GMAPERRLOG("cannot get addrinfo",server);
    goto error;
  }

  for (ai = aihead; ai != NULL; ai = ai->ai_next) {
    if((s=SSL_socket(ai->ai_family, ai->ai_socktype, ai->ai_protocol))==NULL){
      GMAPERRLOG("cannot get socket",server);
      goto loop_warn;
    }

    /* setup SSL if necessary */
    if(ssl){
      /* currently not supported */

    }

    /* connect to the server */
    if(SSL_connect(s,ai->ai_addr,ai->ai_addrlen)){
      GMAPERRLOG("cannot connect to the server",server);
      goto loop_warn;
    }

    break;

loop_warn:
    if (s != NULL) {
      SSL_close(s);
      SSL_free(s);
      s = NULL;
    }
  }

  if (s == NULL) {
    GMAPERRLOG("no socket connection to", server);
    goto error;
  }

  /* do SSL handshake if necessary */
  if(ssl){
    if(SSL_handshake(s)){
      GMAPERRLOG("error in SSL handshake",server);
      goto error;
    }
  }

  free(port_string);
  freeaddrinfo(aihead);
  return s;
error:
  SSL_close(s);
  SSL_free(s);
  if(port_string != NULL) {
    free(port_string);
  }
  if(aihead != NULL) {
    freeaddrinfo(aihead);
  }
  return NULL;
}

char *gmap_get_webfile(SSL *sock, char *svname, int port, char *fname, int ssl){
  char *ret=NULL;
  int len;

  /* send http GET */
  if(gmap_send_httpget(sock,svname,port,fname,ssl)) goto done;

  /* recv response */
  if((len = gmap_recv_httpheader(sock))<0){
    GMAPERRLOG("cannot analyze http header",svname);
    goto done;
  }

  if((ret = gmap_recv_body(sock,len))==NULL){
    GMAPERRLOG("cannot recv http body",svname);
    goto done;
  }

done:
  return (char*)ret;
}

int gmap_send_httpget(SSL *sock, char *svname, int port, char *fname, int ssl){
  char buf[512],tmp[256],tm2[32],*con;
  char host_field[strlen(svname)+strlen("[]")+1];
  int i,j,k;

  /* make HTTP Host field */
  if(isipv6addr(svname)==ISIPV6ADDR_TRUE){
    char *svname_tmp, *cp;

    svname_tmp = strdup(svname);
    if ((cp = strchr(svname_tmp, '%')) != NULL) {
      *cp = '\0';
    }
    sprintf(host_field,"[%s]",svname_tmp);
    free(svname_tmp);
  } else {
    strcpy(host_field,svname);
  }

  /* set http header */
  if(*proxy_host){
    *tm2 = 0;
    if(port != 80) sprintf(tm2,":%d",port);
    snprintf(tmp,256,"http://%s%s%s",host_field,tm2,fname);
    con = "Proxy-Connection";
  }else{
    strcpy(tmp,fname);
    con = "Connection";
  }
  snprintf(buf,512,http_hd,tmp,host_field,con);

  /* write http request */
  i=strlen(buf); j=0;
  while(j<i){
    if((k=SSL_write(sock,&buf[j],i-j))<0){
      GMAPERRLOG("failed to send http request",svname);
      return -1;
    }
    j+=k;
  }

  return 0;
}

#if !defined(HAVE_STRNICMP) && !defined(strnicmp)
#ifdef HAVE_STRNCASECMP

int strnicmp(char *c1,char *c2,int len){ return strncasecmp(c1,c2,len); }

# else
/************ not ansi c **********/
int strnicmp(char *str1, char *str2, size_t count){
  int i,j,k=0;

  while(k < count){
    i = *str1;
    if(('a'<=i)&&(i<='z')) i-=0x20;
    j = *str2;
    if(('a'<=j)&&(j<='z')) j-=0x20;

    i-=j;
    if(i) return i;
    if(j==0) break;

    str1++; str2++; k++;
  }
  return 0;
}
# endif
#endif

int gmap_recv_httpheader(SSL *s){
  char buf[256],*cp,*t;
  int ret=-1;

  /* header */
  do{
    memset(buf,0,256);
    if(SSL_gets(s,buf,256)<0) goto done;

    if(strstr(buf,"HTTP")==NULL) continue;

    if(strstr(buf,"200")) break; /* 200 OK */
    else if(strstr(buf,"100")) continue; /* 100 CONTINUE */
    else {
      fprintf(stderr,"%s",buf); goto done; /* error */
    }
  }while(1);

  do{
    memset(buf,0,256);
    if(SSL_gets(s,buf,256)<0) goto done;

    t=buf;
    while(*t==' '){ t++; }

    if(!strncmp(t,"\r\n",2)) break;

    else if(!strnicmp(t,"Content-Length:",15)){
      if((cp=strchr(t,':'))==NULL) goto done;
      do{ ++cp; }while(*cp==' ');
      ret = atoi(cp);
    }
    else if(!strnicmp(t,"Content-Type:",13)){
      if((cp=strchr(t,':'))==NULL) goto done;
      do{ ++cp; }while(*cp==' ');
      if(strnicmp(cp,"text/plain",10) && strnicmp(cp,"text/html",9)) goto done;
    }
    else if(!strnicmp(t,"Transfer-Encoding:",17)){
      if((cp=strchr(t,':'))==NULL) goto done;
      do{ ++cp; }while(*cp==' ');
      if(!strnicmp(cp,"chunked",7)){ ret = 0; }
    }

  }while(1);

done:
  return ret;
}

char *gmap_recv_body(SSL *s, int len){
  char *ret=NULL,*tmp=NULL,buf[32];
  int i,j,k,chunk;

  chunk = (len==0);

  for(i=0;;){
    if(chunk){ /* chunked */
      if(SSL_gets(s,buf,32)<0) goto error;
      if((k = strtol(buf,NULL,16)) <= 0) break;
      len += k;
    }
    /* allocate memory */
    if((ret=(char*)malloc(len+2))==NULL) goto error;
    if(tmp){
      memcpy(ret,tmp,len-k);
      free(tmp); tmp=NULL;
    }
    tmp = ret; ret[len+1]=0;

    /* receive date */
    while(i<len){
      if((j=SSL_read(s,(char*)&ret[i],len-i))<0) goto error;
      i+=j;
    }

    if(chunk) SSL_gets(s,buf,32); /* clear chunk-data's CRLF */
    else break;
  }

  return ret;
error:
  if(ret) free(ret);
  return NULL;
}


/*//////////////////////////*/
#if 0
unsigned char *gmap_recv_httprsp(SSL *sock, int *ret_len){
  char buf[512],*t,*cp,*np,tmp;
  char *ret=NULL;
  int i,j,len;

  /* read http response */
  memset(buf,0,512);
  if((i=SSL_read(sock,buf,512))<0){
      GMAPERRLOG("failed to recv http response",NULL);
      goto error;
  }

  /* header */
  conf_get_line(buf,&np,&tmp);
  if(strstr(buf,"200")==NULL){
      GMAPERRLOG("received error response",buf);
      goto error;
  }
  cp = np;

  do{
    if(conf_get_line(cp,&np,&tmp)==NULL) goto error;
    t = cp;

    while(*t==' '){ t++; }

    if(!memcmp(t,"\0\n",2)) break;

    else if(!strnicmp(t,"Content-Length:",15)){
      if((cp=strchr(t,':'))==NULL) goto error;
      *ret_len = len = atoi(++cp);
    }

    cp = np;
    if((cp - buf)>512){
      GMAPERRLOG("received http header is too big",NULL);
      goto error;
    }

  }while(1);

  j = np - buf;
  if((ret=malloc(len+2))==NULL) goto error;
  memset(ret,0,len+2);

  if(i < j){
    GMAPERRLOG("http bad length",NULL);
    goto error;
  }

  i = i-j;
  memcpy(ret,np,i);

  if(len > i){    
    if((j=SSL_read(sock,&ret[i],len-i))<=0){
      GMAPERRLOG("failed to recv http response",NULL);
      goto error;
    }
    i+=j;
  }

  return ret;
error:
  if(ret) free(ret);
  return NULL;
}
#endif
