/* cgi_util.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>

#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_base64.h>
#include <aicrypto/ok_pem.h>

#include "cgi_common.h"

/*-----------------------------------------------
  read URL query string
-------------------------------------------------*/
int cgi_read_query(AiQC *info){
  char *buf,*cp,*t,*p;
  int i;

  if((buf = getenv("QUERY_STRING")) == NULL) goto done;

  cp = buf; 
  for(i=0; cp && *cp && i<QUERY_MAX; i++){
    /* find '&' to separate quary */
    if((t = strchr(cp,'&')) != NULL) { *t = 0; t++; }

    /* unescape value */
    if(ai_unescape_url(cp)) goto done;

    /* find '=' to separete value */
    if((p = strchr(cp,'=')) != NULL) { *p = 0; p++; }
    strncpy(info->urlqs[i].key,cp,126);
    if(p==NULL) p="";
    strncpy(info->urlqs[i].value,p,254);

    cp = t;
  }

done:
  return 0;
}

/*-----------------------------------------------
  read POST query string
-------------------------------------------------*/
int cgi_read_post(AiQC *info){
  char *ct,*cl,*p,*t,*buf=NULL;
  int i,len,ok=0;

  ct = getenv("CONTENT_TYPE");
  cl = getenv("CONTENT_LENGTH");

  if((ct==NULL)||(cl==NULL)) goto done;
  if(strcmp(ct,"application/x-www-form-urlencoded")) goto done;
  if((len=atoi(cl))<=0) goto done;

  /* read content */
  if((buf=(char*)malloc(len+2))==NULL){ ok = -1; goto done;}
  if(fread(buf,len,1,stdin) != 1){ ok = -1; goto done;}
  buf[len] = 0;

  /* parse content */
  ct = buf;
  for(i=0; ct && *ct && i<QUERY_MAX; i++){
    /* find '&' to separate quary */
    if((t = strchr(ct,'&')) != NULL) { *t = 0; t++; }

    /* unescape value */
    if(ai_unescape_url(ct)) goto done;

    /* find '=' to separete value */
    if((p = strchr(ct,'=')) != NULL) { *p = 0; p++; }
    strncpy(info->postqs[i].key,ct,126);
    if(p && *p){
      if((info->postqs[i].value=(char*)malloc(strlen(p)+2))==NULL){ ok = -1; goto done;}
      strcpy(info->postqs[i].value,p);
    }
    ct = t;
  }

done:
  if(ok){
    printf("Content-type: text/plain\n\n");
    printf("POST query read error.<br>\n");
  }
  if(buf) free(buf);
  return ok;
}

/*-----------------------------------------------
  read HTTP cookie string
-------------------------------------------------*/
int cgi_read_cookie(AiQC *info){
  char *buf,*cp,*t,*p;
  int i;

  if((buf = getenv("HTTP_COOKIE")) == NULL) goto done;

  cp = buf; 
  for(i=0; cp && *cp && i<QUERY_MAX; i++){
    /* ignore space in separate field */
    while(*cp == ' '){cp++;}

    /* find '&' to separate quary */
    if((t = strchr(cp,';')) != NULL) { *t = 0; t++; }

    /* unescape value */
    if(ai_unescape_url(cp)) goto done;

    /* find '=' to separete value */
    if((p = strchr(cp,'=')) != NULL) { *p = 0; p++; }
    strncpy(info->cooks[i].key,cp,126);
    if(p==NULL) p="";
    strncpy(info->cooks[i].value,p,254);

    cp = t;
  }

done:
  return 0;
}

/*-----------------------------------------------
  query string utils
-------------------------------------------------*/
int ai_htoc(char *in){
  int ret;

  ret = ((in[0]>='A')?(((in[0] & 0xdf)-'A') + 10):(in[0] - '0')) << 4;
  ret|= ((in[1]>='A')?(((in[1] & 0xdf)-'A') + 10):(in[1] - '0'));
  return ret;
}

int ai_ctoh(int in, char *ret){
  char t;
  ret[0] = '%';
  t = ( in>>4 ) & 0x0f;
  if( t >= 10 ){
    ret[1] = 'A' + ( t - 10 );
  }else{
    ret[1] = '0' + t;
  }
  t = in & 0x0f;
  if( t >= 10 ){
    ret[2] = 'A' + ( t - 10 );
  }else{
    ret[2] = '0' + t;
  }
  return 0;
}

int ai_unescape_url(char *buf){
  char *cp,*hd;

  cp = hd = buf;
  while(hd && *hd){
    switch(*hd){
    case '%': *cp = ai_htoc(&hd[1]); hd+=2; break;
    case '+': *cp = ' '; break;
    default:  *cp = *hd; break;
    }
    hd++; cp++;
  }
  if(cp) *cp = 0;

  return 0;
}

char *ai_escape_url(char *in,int len){
  char *hd,*ret=NULL;
  int i,j;

  if((ret=(char*)malloc(len*3+1))==NULL) goto done;
  memset(ret,0,len*3+1);
  
  hd = ret;
  for(i=0; i<len; i++){
    j = in[i];
    if(((j>='0')&&(j<='9')) ||
       ((j>='A')&&(j<='Z')) ||
       ((j>='a')&&(j<='z'))){
      
      *hd = in[i]; hd++;
    }else{
      switch(j){
      case '-':
      case '@':
      case '.':
      case '*':
      case '_':
        *hd = in[i]; hd++; break;
      default:
        /* modified by nagano */
        ai_ctoh(j,hd); hd+=3; break;
      }
    }
  }

done:
  return ret;
}

void ai_symbol_esc_sp(char *cp){
  while(cp && *cp){
    switch(*cp){
    case '>':
      *cp = ' ';
      break;
    case '<':
      *cp = ' ';
      if(cp[1]=='!') cp[1] = ' ';
      if(cp[1]=='/') cp[1] = ' ';
      break;
    }
    cp++;
  }
}

/*-----------------------------------------------
  CSR decoder
-------------------------------------------------*/
Req *cgi_csr_decode(char *in){
  unsigned char *der = NULL;
  Req *ret = NULL;
  int i;

  /* input has been unescaped */
  if((der=Base64_decode(in,&i))==NULL) goto done;
  ret = ASN1_read_req(der);

done:
  if((ret==NULL)&& der) free(der);
  return ret;
}

/*-----------------------------------------------
  SPKAC decoder
-------------------------------------------------*/
SPKAC *cgi_spkac_decode(char *in){
  unsigned char *der = NULL;
  SPKAC *ret = NULL;
  int i;

  /* input has been unescaped */
  if((der=Base64_decode(in,&i))==NULL) goto done;
  ret = ASN1_read_spkac(der);

done:
  if((ret==NULL)&& der) free(der);
  return ret;
}

/*-----------------------------------------------
  subject line to DN
-------------------------------------------------*/
int get_dn_kind(char *dn){
  int i=-1;

  switch(*dn){
  case 'C': i=(dn[1]=='N')?(OBJ_DIR_CN):(OBJ_DIR_C); break;
  case 'c': i=(dn[1]=='n')?(OBJ_DIR_CN):(OBJ_DIR_C); break;
  case 'D': i=(dn[1]=='C')?(OBJ_DIR_DC):(-1); break;
  case 'S': i=(dn[1]=='T')?(OBJ_DIR_ST):(-1); break;
  case 's': i=(dn[1]=='t')?(OBJ_DIR_ST):(-1); break;
  case 'l':
  case 'L': i=OBJ_DIR_L; break;
  case 'O': i=(dn[1]=='U')?(OBJ_DIR_OU):(OBJ_DIR_O); break;
  case 'o': i=(dn[1]=='u')?(OBJ_DIR_OU):(OBJ_DIR_O); break;
  case 'E':
  case 'e': i=OBJ_DIR_EMAIL; break;
  case 'U': i=((dn[1]=='I')&&(dn[2]=='D'))?(OBJ_DIR_UID):(-1); break;
  default:
    i = -1;
    break;
  }
  return i;
}

/*-----------------------------------------
  return file format
  1...binary, 2...pem, 3...text(base64?)
  4...IE7 type (pem & bad header)
  -1...error
-----------------------------------------*/
int cgi_get_fmt(char *in){
  int i=strlen(in),j,ret=-1;

  for(j=0;j<i;j++)
    if(in[j] & 0x80){ ret=1; goto done;}

  if(strstr(in,"-----")){
	ret=2; /* set pem */
    if(strstr(in,"BEGIN CERTIFICATE-----")) ret=4; /* IE7 send this type */
	goto done;
  }

  ret = 3;
done:
  return ret;
}


/*-----------------------------------------------
  query string utils
-------------------------------------------------*/
char *cgi_find_urlquery(AiQC *info,char *key,int *index){
  char *ret = NULL;
  int i;

  for(i=*index; i<QUERY_MAX; i++){
    if(!strncmp(key,info->urlqs[i].key,126)){
      ret = info->urlqs[i].value; *index = i; return ret;
    }
  }
  return NULL;
}

char *cgi_find_postquery(AiQC *info,char *key,int *index){
  char *ret = NULL;
  int i;

  for(i=*index; i<QUERY_MAX; i++){
    if(!strncmp(key,info->postqs[i].key,126)){
      ret = info->postqs[i].value; *index = i; return ret;
    }
  }
  return NULL;
}

char *cgi_find_query(AiQC *info,char *key){
  char *ret = NULL;
  int i;

  i=0;
  ret = cgi_find_urlquery(info,key,&i);
  if(ret) goto done; /* find */

  i=0;
  ret = cgi_find_postquery(info,key,&i);

done:
  if(ret==NULL) ret="";
  return ret;
}

char *cgi_find_cookie(AiQC *info,char *key){
  char *ret = NULL;
  int i;

  for(i=0; i<QUERY_MAX; i++){
    if(!strncmp(key,info->cooks[i].key,126)){
      ret = info->cooks[i].value; break;
    }
  }

  if(ret==NULL) ret="";
  return ret;
}


/*-----------------------------------------------
  client certificate utils
-------------------------------------------------*/
int cgi_dec_snum(char *p){
  int i,ret = 0;

  while(*p){
    i = *p;
    if(('0'<=i)&&(i<='9')){
      ret<<=4; ret|=i-'0';
    }else if(('A'<=i)&&(i<='F')){
      ret<<=4; ret|=i+10-'A';
    }else if(('a'<=i)&&(i<='f')){
      ret<<=4; ret|=i+10-'a';
    }
    p++;
  }
  return ret;
}

Cert *cgi_get_clientcert(){
  Cert *ret=NULL;
  char *buf;

  if((buf=getenv("SSL_CLIENT_CERT")) != NULL) { /* apache Environment variables */
    return PEM_read_cert_buf(buf);
  }
  if((buf=getenv("CLIENT_CERT_ENCODED")) != NULL) { /* IIS 4.0 or NS Environment variables */
    return PEM_read_cert_buf(buf);
  }
  if((buf=getenv("CERT_SERIALNUMBER")) != NULL) { /* IIS 5.0 or later */
    // 2008/10/6 buf is not NULL, but brank. then, empty certificate is generated.
    //           this cause SSL client authentication bug...
	int sn;
    if(*buf==0) goto error;
	if((sn=cgi_dec_snum(buf))==0) goto error;

    if((ret=Cert_new())==NULL) goto error;
    ret->serialNumber = sn;

    if((buf=getenv("CERT_ISSUER")) != NULL) {
	    if ((ret->issuer = strdup(buf)) == NULL) goto error;
    }
    if((buf=getenv("CERT_SUBJECT")) != NULL) {
	    if ((ret->subject = strdup(buf)) == NULL) goto error;
    }
    if((buf=getenv("CERT_COOKIE")) != NULL) {
	    if ((ret->signature = strdup(buf)) == NULL) goto error;
	}	
    return ret;
  }
error:
  Cert_free(ret);
  return NULL;
}

char *cgi_get_clientcertid(){
  return (char*)getenv("CERT_COOKIE");
}

/*-----------------------------------------------
  CGI Print content
-------------------------------------------------*/
void cgi_print_content(char *buf, AiVList *v){
  char *cp,*t,*p;
  int j;

  cp = buf;
  do{
    if ((t = strstr(cp,"${")) == NULL){ printf("%s", cp); break; }
    *t=0; t++; p = "$";

    for(j=0; v[j].tag; j++){
      if(!memcmp(t,v[j].tag,strlen(v[j].tag))){
        p = v[j].val[0]; t+=strlen(v[j].tag); break;
      }
    }
    printf("%s", cp);
    if(p) printf("%s", p);
    cp = t;
  }while(cp);
}

/*-----------------------------------------------
  session file base functions.
-------------------------------------------------*/
int cgi_load_session(char *fname,AccList **top){
  AccList *acc,*hd;
  FILE *fp=NULL;
  fpos_t tpos;
  unsigned char buf[512];
  int ok=-1;

  /* open RA session file */
  if((fp=fopen(fname,"rb")) != NULL) {
    do{
      if(fgetpos(fp,&tpos)) goto done;
      if(fread(buf,sizeof(char),WEB_SSBLOCK,fp)<WEB_SSBLOCK) break;
      if((acc = AccList_new())==NULL) goto done;
      memcpy(acc->name,buf,64); /* 64 */
      memcpy(acc->cn,&buf[64],64); /* 128 */
      memcpy(acc->pwd,&buf[128],64); /* 192 */
      memcpy(acc->group,&buf[192],64); /* 256 */
      memcpy(acc->email,&buf[256],64); /* 320 */
      memcpy(&acc->mode,&buf[320],4); /* 324 */
      memcpy(&acc->acceptID,&buf[324],4); /* 328 */
      memcpy(&acc->serialNum,&buf[328],4); /* 332 */
      memcpy(&acc->containerTime,&buf[332],4); /* 336 */
      memcpy(acc->sessionID,&buf[336],8); /* 344 */
      memcpy(acc->keyID,&buf[344],20); /* 364 */
      memcpy(&acc->isstype,&buf[364],4); /* 368 */
      memcpy(&acc->notAfter,&buf[368],4); /* 372 */
      acc->pos = tpos;

      if(*top){
	hd->next=acc; hd=acc;
      }else{
	*top=hd=acc;
      }
    }while(1);
  }

  ok = 0;
done:
  if(fp) fclose(fp);
  return ok;
}

int cgi_reload_session(char *fname,AccList *acc){
  unsigned char buf[512];
  FILE *fp=NULL;
  int ok = -1;

  /* open RA session file */
  if((fp=fopen(fname,"rb+"))==NULL) goto done;
  if(fsetpos(fp,&acc->pos)) goto done;
  if(fread(buf,sizeof(char),WEB_SSBLOCK,fp)<WEB_SSBLOCK) goto done;

  memcpy(acc->name,buf,64); /* 64 */
  memcpy(acc->cn, &buf[64],64); /* 128 */
  memcpy(acc->pwd,&buf[128],64); /* 192 */
  memcpy(acc->group,&buf[192],64); /* 256 */
  memcpy(acc->email,&buf[256],64); /* 320 */
  memcpy(&acc->mode,&buf[320],4); /* 324 */
  memcpy(&acc->acceptID,&buf[324],4); /* 328 */
  memcpy(&acc->serialNum,&buf[328],4); /* 332 */
  memcpy(&acc->containerTime,&buf[332],4); /* 336 */
  memcpy(acc->sessionID,&buf[336],8); /* 344 */
  memcpy(acc->keyID,&buf[344],20); /* 364 */
  memcpy(&acc->isstype,&buf[364],4); /* 368 */
  memcpy(&acc->notAfter,&buf[368],4); /* 372 */
  ok = 0;
done:
  if(fp) fclose(fp);
  return ok;
}

int cgi_update_session(char *fname,AccList *acc){
  FILE *fp=NULL;
  unsigned char buf[WEB_SSBLOCK];
  int ok = -1;

  /* open RA session file */
  if((fp=fopen(fname,"rb+"))==NULL) goto done;
  if(fsetpos(fp,&acc->pos)) goto done;

  memset(buf,0,WEB_SSBLOCK);
  memcpy(buf,acc->name,64);
  memcpy(&buf[64],acc->cn,64); /* 128 */
  memcpy(&buf[128],acc->pwd,64); /* 192 */
  memcpy(&buf[192],acc->group,64); /* 256 */
  memcpy(&buf[256],acc->email,64); /* 320 */
  memcpy(&buf[320],&acc->mode,4); /* 324 */
  memcpy(&buf[324],&acc->acceptID,4); /* 328 */
  memcpy(&buf[328],&acc->serialNum,4); /* 332 */
  memcpy(&buf[332],&acc->containerTime,4); /* 336 */
  memcpy(&buf[336],acc->sessionID,8); /* 344 */
  memcpy(&buf[344],acc->keyID,20); /* 364 */
  memcpy(&buf[364],&acc->isstype,4); /* 368 */
  memcpy(&buf[368],&acc->notAfter,4); /* 372 */

  if(fwrite(buf,sizeof(char),WEB_SSBLOCK,fp)<WEB_SSBLOCK) goto done;

  ok = 0;
done:
  if(fp) fclose(fp);
  return ok;
}

/*-----------------------------------------------
  session file utility
-------------------------------------------------*/
int cgi_set_session(char *fname, AccList **top, AiUserInfo *uinfo, unsigned char *sessionID){
  return cgi_set_sessiongrp(fname,top,uinfo,sessionID,NULL);
}

int cgi_add_session(char *fname, AccList **top, AccList *acc){
  AccList *al=NULL;
  FILE *fp=NULL;
  char buf[WEB_SSBLOCK];
  int ok = -1;

  memset(buf,0,WEB_SSBLOCK); /* use "buf" to set "0" on reserved location */
  memcpy(buf,acc->name,64); /* include cn & pwd */

  if((fp=fopen(fname,"ab+"))==NULL) goto done;
  if(fwrite(buf,sizeof(char),WEB_SSBLOCK,fp)<WEB_SSBLOCK) goto done;
  if(fseek(fp,-WEB_SSBLOCK,SEEK_CUR)) goto done;
  if(fgetpos(fp,&acc->pos)) goto done;
  fclose(fp); fp=NULL;

  if(*top){
    al = *top;
    while(al->next){ al=al->next; }
    al->next=acc;
  }else{
    *top = acc;
  }

  ok = 0;
done:
  if(fp) fclose(fp);
  return ok;
}

int cgi_set_sessiongrp(char *fname, AccList **top, AiUserInfo *uinfo, unsigned char *sessionID, char *group){
  AccList *acc;
  int ok = -1;

/*{FILE *fp; if(fp=fopen("session_dbg.txt","a+")){fprintf(fp,"%s:%s\n",uinfo->name,group); fclose(fp);}}*/

  if((acc=cgi_findbynamegrp(*top,uinfo->name,group))==NULL){
    /* add a new session */
    if((acc = AccList_new())==NULL) goto done;
    strncpy(acc->name,uinfo->name,62);
    strncpy(acc->cn,uinfo->cn,62);
    strncpy(acc->email,uinfo->mail,62);
    if(group){
      strncpy(acc->group,group,62);
    }
    if(cgi_add_session(fname,top,acc)) goto done;
  }
  strncpy(acc->caname,uinfo->caname,62);
  memcpy(acc->sessionID,sessionID,8);

  /* update a session */
  if(cgi_update_session(fname,acc)) goto done;

  ok = 0;
done:
  return ok;
}

int cgi_check_session(AccList *top, char *name, char *sessionID){
  return cgi_check_sessiongrp(top,name,sessionID,NULL);
}

int cgi_check_sessiongrp(AccList *top, char *name, char *sessionID, char *group){
  AccList *acc = NULL;
  unsigned char sb[16];
  int j,k;

  memset(sb,0,16);
  for(j=k=0; j<8; j++,k+=2){
    sb[j] = ai_htoc(&sessionID[k]);
  }

  if((acc=cgi_findbynamegrp(top,name,group))!=NULL){
    if(!memcmp(sb,acc->sessionID,8)) return 0;
  }

  return -1;
}

int cgi_get_mode(AccList *top,AiUserInfo *uinfo){
  return cgi_get_modegrp(top,uinfo,NULL);
}

int cgi_get_modegrp(AccList *top,AiUserInfo *uinfo,char *group){
  AccList *acc;

  if((acc=cgi_findbynamegrp(top,uinfo->name,group)) != NULL) {
    uinfo->mode = acc->mode;
  }else{
    uinfo->mode = AI_ENL_NEWISS;
  }
  return 0;
}

/*-----------------------------------------------
  User access info utility
-------------------------------------------------*/
AccList *cgi_findbyname(AccList *top, char *name){
  return cgi_findbynamegrp(top,name,NULL);
}

AccList *cgi_findbynamegrp(AccList *top, char *name, char *group){
  while(top){
    if(!strcmp(name,top->name)){
      if(group==NULL)
	return top;
      else if(!strcmp(group,top->group))
	return top;
    }
    top = top->next;
  }
  return NULL;
}

AccList *cgi_findbyaccid(AccList *top, int id){
  while(top){
    if(id == top->acceptID) return top;
    top = top->next;
  }
  return NULL;
}

AccList *cgi_findbysnum(AccList *top, int snum){
  while(top){
    if(snum == top->serialNum) return top;
    top = top->next;
  }
  return NULL;
}

AccList *cgi_findbykeyid(AccList *top, unsigned char *keyid){
  while(top){
    if(!memcmp(keyid,top->keyID,20)) return top;
    top = top->next;
  }
  return NULL;
}


/*-----------------------------------------------
  get license hash
-------------------------------------------------*/
/* hash should be big enough */
int cgi_gen_hash(char *license, char *hash){
  char num[20] = "0123456789ABCDEF";
  unsigned char buf[32],*cp;
  int i;

  OK_SHA1(strlen(license),license,buf);
  memcpy(hash,"HASH{",5); hash+=5;

  for(cp=buf,i=0; i<16; i++,cp++){
    *hash = num[(*cp>>4)&0xf]; hash++;
    *hash = num[*cp&0xf]; hash++;
  }
  *hash = '}'; hash++;
  *hash = 0;
  return 0;
}

/*-----------------------------------------------
  parse csv file (buffer)
-------------------------------------------------*/
int cgi_parse_csv(char *in, char *out[], int out_max){
  char *item;
  int i,j,inq=0;

  for(i=0; *in && (i<out_max); i++){
    item = out[i]; item[0] = 0;

    for(j=inq=0; *in && (j<256); j++, in++){
      switch(*in){
      case '\r': /* ignore */
	break;
      case '\n': /* end */
	item[j] = 0; j=256;
	break;
      case ',':
	if(*(in-1)!='\\'){ /* end */
	  if(inq){ item[j] = *in; }
	  else{ item[j] = 0; j=256;}
	}
	break;
      case '"':
	if(*(in-1)!='\\'){
	  if(inq){ item[j] = 0; inq = 0; } /* end string */
	  else{ j=-1; inq = 1; } /* start string */
	}
	break;
      case '\\': /* escape sequence */
	if(in[1]=='"'||in[1]==','||in[1]=='\\'){
	  in++; item[j] = *in;
	  break;
	}
        /* not break */
      default:
	item[j] = *in;
	break;
      }
    }
  }
  return 0;
}

/*-----------------------------------------------
  new & free AiEntryInfo content
-------------------------------------------------*/
AiEntryInfo *cgi_entryinfo_new(){
  AiEntryInfo *ret;
  if((ret = malloc(sizeof(AiEntryInfo)))==NULL) return NULL;
  memset(ret,0,sizeof(AiEntryInfo));
  return ret;
}

void cgi_entryinfo_free(AiEntryInfo *e){
  int i;
  if(e==NULL) return;
  if(e->dn) free(e->dn);
  if(e->cn) free(e->cn);
  if(e->firstName) free(e->firstName);
  if(e->lastName) free(e->lastName);
  if(e->pin) free(e->pin);
  if(e->group) free(e->group);
  if(e->employeeNumber) free(e->employeeNumber);
  if(e->unit) free(e->unit);
  if(e->title) free(e->title);
  if(e->tel1) free(e->tel1);
  if(e->tel2) free(e->tel2);
  if(e->fax1) free(e->fax1);
  if(e->fax2) free(e->fax2);
  if(e->postalCode) free(e->postalCode);
  if(e->postalAddress) free(e->postalAddress);
  if(e->labeledUri) free(e->labeledUri);
  if(e->description) free(e->description);
  if(e->o) free(e->o);
  if(e->ou) free(e->ou);
  if(e->st) free(e->st);
  if(e->street) free(e->street);
  if(e->l) free(e->l);
  if(e->queryattr) free(e->queryattr);
  if(e->ucert) free(e->ucert);
  for(i=0; i<MAXGROUP; i++)
    if(e->glist[i]) free(e->glist[i]);
  free(e);
}

/*-----------------------------------------------
  Cert_print() in aicrypt.dll does not work well
  with IIS (windows) environment...
-------------------------------------------------*/
#ifdef __WINDOWS__
void cgi_LN_print2(LNm *a,int space);
void cgi_print_dn(CertDN *dn);
void cgi_print_sig_algo(int algo);
void cgi_print_serialnum(Cert *ct);
void cgi_print_validity(Cert *ct);
void cgi_print_pubkey(Key *key);
void cgi_print_pubkey_algo(Key *key);
void cgi_print_v3_extensions(CertExt *top,int cf);
void cgi_print_signature(unsigned char *sig, int max, int algo);

char *cgi_dir_t[16]={
    "C","ST","L","O","OU","CN","SIR","SNUM",
	"GIVEN","TITLE","","", "","","",""
};


void cgi_cert_print(Cert *ct){
	if(ct==NULL) return;

	if(ct->issuer){
		printf("issuer: ");
		cgi_print_dn(&(ct->issuer_dn));
		printf("subject: ");
		cgi_print_dn(&(ct->subject_dn));
		printf("serial: %d\n",ct->serialNumber);

		printf("Certificate:\n");
	}else
		printf("Certificate Request:\n");

	printf("  Data:\n");
	printf("    Version: %d (0x%x)\n",ct->version+1,ct->version);

	if(ct->issuer){
		cgi_print_serialnum(ct);

		printf("    Signature Algorithm: ");
		cgi_print_sig_algo(ct->signature_algo);

		printf("    Issuer: %s\n",ct->issuer);
		cgi_print_validity(ct);
	}
	printf("    Subject: %s\n",ct->subject);
	cgi_print_pubkey(ct->pubkey);

	if(ct->ext) cgi_print_v3_extensions(ct->ext,(ct->issuer)?(0):(1));
	cgi_print_signature(ct->signature,ct->siglen,ct->signature_algo);
}

void cgi_LN_print2(LNm *a,int space){
	uint32_t *n;
	char	sp[16];
	int	i,j;
  
	memset(sp,' ',space);
	sp[space]=0;

	if(a==NULL){
		printf("%sNULL\n",sp);
	}else{
		n=a->num;
		for(j=0,i=LN_MAX-a->top;i<LN_MAX;i++,j++){
			if(!j) printf("%s",sp);
			printf("%.2x:%.2x:%.2x:%.2x:",(unsigned char)(n[i]>>24),(unsigned char)(n[i]>>16),
				(unsigned char)(n[i]>>8),(unsigned char)n[i]);

			if(j==4){ j=-1; printf("\n");}
		}
		if(j!=0) printf("\n");
	}
}

void cgi_print_serialnum(Cert *ct){
	int i;
	if(ct->long_sn){
		printf("    Serial Number: ");
		for(i=0;i<ct->long_sn[1];i++) printf("%.2x:",ct->long_sn[i+2]);
		printf("\n");
	}else{
		printf("    Serial Number: %d (0x%x)\n",ct->serialNumber,ct->serialNumber);	
	}
}

void cgi_print_dn(CertDN *dn){
	int i,j;
	for(i=0;i<dn->num;i++){
		j = dn->rdn[i].tagoid;
		if((OBJ_DIR_C<=j)&&(j<=OBJ_DIR_TITLE)){
			printf("/");
			printf(cgi_dir_t[j-OBJ_DIR_C]);
			printf("=");
			printf(dn->rdn[i].tag);
		}else if(j==OBJ_DIR_UID){
			printf("/UID=");
			printf(dn->rdn[i].tag);
		}else if(j==OBJ_DIR_DC){
			printf("/DC=");
			printf(dn->rdn[i].tag);
		}else if(j==OBJ_DIR_EMAIL){
			printf("/EMAIL=");
			printf(dn->rdn[i].tag);
		}else{
			printf("/\?\?=");
			printf(dn->rdn[i].tag);
		}
	}
	printf("\n");
}

void cgi_print_sig_algo(int algo){
	switch(algo){
	case OBJ_SIG_MD2RSA:
	case OBJ_SIGOIW_MD2RSA:
		printf("md2WithRSAEncryption\n");
		break;
	case OBJ_SIG_MD5RSA:
	case OBJ_SIGOIW_MD5RSA:
		printf("md5WithRSAEncryption\n");
		break;
	case OBJ_SIG_SHA1RSA:
	case OBJ_SIGOIW_SHA1RSA:
		printf("sha1WithRSAEncryption\n");
		break;
	case OBJ_SIG_SHA1DSA:
		printf("id-dsa-with-sha1\n");
		break;
	case OBJ_SIG_SHA1ECDSA:
		printf("ecdsa-with-SHA1\n");
		break;
	default:
		printf("\n");
		break;
	}
}

void cgi_print_validity(Cert *ct){
	char *cp;

	printf("    Validity\n");
	if((cp=stm2str(&ct->time.notBefore,0))==NULL) return;
	printf("      Not Before: %s\n",cp);
	if((cp=stm2str(&ct->time.notAfter,0))==NULL) return;
	printf("      Not After : %s\n",cp);
}

void cgi_print_pubkey(Key *pubkey){
	Pubkey_ECDSA *ek;
	char num[32];
	int bit = pubkey->size*8;

	printf("    Subject Public Key Info:\n");
	cgi_print_pubkey_algo(pubkey);
	switch(pubkey->key_type){
	case KEY_RSA_PUB:
		printf("      RSA Public Key: (%d bit)\n",bit);
		printf("        Modulus (%d bit):\n",bit);
		cgi_LN_print2(((Pubkey_RSA*)pubkey)->n,10);
		printf("        Exponent:\n");
		cgi_LN_print2(((Pubkey_RSA*)pubkey)->e,10);
		break;
	case KEY_DSA_PUB:
		printf("      DSA Public Key: \n");
		printf("        Parameters (%d bit):\n",bit);
		printf("        P :\n");
		cgi_LN_print2(((Pubkey_DSA*)pubkey)->pm->p,10);
		printf("        Q :\n");
		cgi_LN_print2(((Pubkey_DSA*)pubkey)->pm->q,10);
		printf("        G :\n");
		cgi_LN_print2(((Pubkey_DSA*)pubkey)->pm->g,10);
		printf("        Public key w:\n");
		cgi_LN_print2(((Pubkey_DSA*)pubkey)->w,10);
		break;
	case KEY_ECDSA_PUB:
		ek =(Pubkey_ECDSA*)pubkey;
		printf("        Elliptic Curve Parameters: \n");

		if((ek->E->curve_type != ECP_ORG_primeParam) &&
		   (ek->E->curve_type != ECP_ORG_char2Param)){
			switch_str(ek->E->curve_type,num);
			printf("        prime-field (Prime-p):\n");
			printf("          %s\n",num);
		}else{
			printf("        FieldID : ");
			switch(ek->E->type){
			case OBJ_X962_FT_PRIME: 
			  printf(" prime-field (Prime-p):\n");
			  cgi_LN_print2(ek->E->p,10);
			  break;
			case OBJ_X962_FT_CHR2:
			  printf(" characteristic-two-field\n");
			  break;
			}
			/* curve */
			printf("        Curve :\n");
			printf("        a :\n");
			cgi_LN_print2(ek->E->a,10);
			printf("        b :\n");
			cgi_LN_print2(ek->E->b,10);

			/* base */
			printf("        Base point G :\n");
			printf("        G.x :\n");
			cgi_LN_print2(ek->E->G->x,10);
			printf("        G.y :\n");
			cgi_LN_print2(ek->E->G->y,10);

			/* order */
			printf("        order of base point (n):\n");
			cgi_LN_print2(ek->E->n,10);

			/* cofactor */
			if(ek->E->h->top){
			  printf("        cofactor ( h = #E(F)/n ) :\n");
			  cgi_LN_print2(ek->E->h,10);
			}
		}
		/* curve */
		printf("        ECDSA Public Key W:\n");
		printf("        W.x :\n");
		cgi_LN_print2(ek->W->x,10);
		printf("        W.y :\n");
		cgi_LN_print2(ek->W->y,10);
		break;
	}
}

void cgi_print_pubkey_algo(Key *pubkey){
	printf("      Public Key Algorithm: ");
	switch(pubkey->key_type){
	case KEY_RSA_PUB: printf("rsaEncryption\n"); break;
	case KEY_DSA_PUB: printf("dsaEncryption\n"); break;
	case KEY_ECDSA_PUB: printf("ecdsaEncryption\n"); break;
	}
}

void cgi_print_v3_extensions(CertExt *top, int cf){
	unsigned char *cp;
	CertExt *ext,*e2;
	char  *cs,*str,buf[512];
	int	i;

	switch(cf){
	case 1:  printf("    PKCS#10 Attributes:\n");break;
	case 2:  printf("    X509 CRL extensions:\n");break;
	case 3:  printf("        crlEntryExtensions:\n");break;
	case 4:  break;
	default: printf("    X509v3 extensions:\n");break;
	}

	for(ext=top;ext!=NULL;ext=ext->next){
		cp = ext->der;
		cs = (ext->critical)?("[critical]"):("");
		switch(ext->extnID){
		case OBJ_X509v3_BASIC:
			Ext_basiccons_str((CE_BasicCons*)ext,buf,510);
			printf("      x509 Basic Constraints:%s\n        %s",cs,buf);
			break;
		case OBJ_X509v3_NameConst:
			Ext_namecons_str((CE_NameCons*)ext,buf,510);
			printf("      x509 Name Constraints:%s\n%s",cs,buf);
			break;
		case OBJ_X509v3_PolicyConst:
			Ext_polcons_str((CE_PolCons*)ext,buf,510);
			printf("      x509 Policy Constraints:%s\n%s",cs,buf);
			break;

		case OBJ_X509v3_SbjKeyIdt:
			printf("      x509 Subject Key Identifier:%s\n        ",cs);
			for(i=2;i<cp[1]+2;i++)
				printf("%.2x:",cp[i]);
			printf("\n");
			break;
		case OBJ_X509v3_AuthKeyIdt:
			Ext_authkey_str((CE_AuthKID*)ext,buf,510);
			printf("      x509 Authority Key Identifier:%s\n%s",cs,buf);
			break;
		case OBJ_X509v3_KEY_Usage:
			Ext_keyusage_str((CE_KUsage*)ext,buf,510);
			printf("      x509 Key Usage:%s\n        %s",cs,buf);
			break;
		case OBJ_X509v3_ExtKeyUsage:
			Ext_extkeyusage_str((CE_ExtKUsage*)ext,buf,510);
			printf("      x509 Ext Key Usage:%s\n%s",cs,buf);
			break;
		case OBJ_X509v3_PrvKeyUsgPrd:
			Ext_prvkey_period_str((CE_PKUsagePrd*)ext,buf,510);
			printf("      x509 PrivateKey Usage Period:%s\n%s",cs,buf);
			break;
		case OBJ_X509v3_CERT_Pol:
			Ext_certpol_str((CE_CertPol*)ext,buf,510);
			printf("      x509 Certificate Policies:%s\n%s",cs,buf);
			break;
		case OBJ_X509v3_CertPolMap:
			Ext_certpolmap_str((CE_PolMap*)ext,buf,510);
			printf("      x509 Policy Mappings:%s\n%s",cs,buf);
			break;
		case OBJ_X509v3_SbjAltName:
			Ext_altname_str((CE_SbjAltName*)ext,buf,510);
			printf("      x509 Subject Alt Name:%s\n%s",cs,buf);
			break;
		case OBJ_X509v3_IssAltName:
			Ext_altname_str((CE_IssAltName*)ext,buf,510);
			printf("      x509 Issuer Alt Name:%s\n%s",cs,buf);
			break;
		case OBJ_X509v3_CRL_Point:
			Ext_crlpoint_str((CE_CRLDistPt*)ext,buf,510);
			printf("      x509 CRL Distribution Points:%s\n%s",cs,buf);
			break;

		case OBJ_X509v3_SubDirAtt:
			printf("      x509 Subject Directory Attribute:%s {\n",cs);
			cgi_print_v3_extensions(((CE_SbjDirAttr*)ext)->attr,4);
			printf("      }\n");
			break;

		/***** CRL Extensions *****/
		case OBJ_X509v3_CRLNumber:
			Ext_crlnum_str((CE_CRLNum*)ext,buf,510);
			printf("      x509 CRL Number:%s\n%s",cs,buf);
			break;
		case OBJ_X509v3_IssDistPoint:
			Ext_issdistpt_str((CE_IssDistPt*)ext,buf,510);
			printf("      x509 Issuer Distribution Points:%s\n%s",cs,buf);
			break;
		case OBJ_X509v3_CRLReason: /* CRL entryExtension */
			Ext_reasoncode_str((CE_Reason*)ext,buf,510);
			printf("        reasonCode: %s",buf);
			break;

		/***** pkix *****/
		case OBJ_PKIX_IDPE_AIA:
			Ext_pkixaia_str((CE_AIA*)ext,buf,510);
			printf("      pkix-idpe-AuthInfoAccess: %s\n%s",cs,buf);
			break;
		case OBJ_PKIX_IDPE_IPADDR:
			Ext_ipaddr_str((CE_IPAddrDlg*)ext,buf,510);
			printf("      pkix IP Address Delegation: %s\n%s",cs,buf);
			break;
		case OBJ_PKIX_IDPE_ASID:
			Ext_asid_str((CE_ASId*)ext,buf,510);
			printf("      pkix ASId: %s\n%s",cs,buf);
			break;

		/***** pkix ocsp *****/
		case OBJ_PKIX_OCSP_NOCHECK:
			Ext_ocspnochk_str(ext,buf,510);
			printf("      pkix-ocsp-nocheck: %s\n%s",cs,buf);
			break;

		/***** moj extensions *****/
		case OBJ_MOJ_JCertPol:
			Ext_certpol_str((CE_JCertPol*)ext,buf,510);
			printf("      MOJ JCertificate Policies:%s\n%s",cs,buf);
			break;
		case OBJ_MOJ_Registrar:
			Ext_comment_str((CE_Com*)ext,buf,510);
			printf("      MOJ Registrar:%s\n        %s\n",cs,buf);
			break;
		case OBJ_MOJ_RegCoInfo:
			Ext_mojcorpinfo_str((CE_MOJCoInfo*)ext,buf,510);
			printf("      MOJ RegistCorpInfo:%s\n%s\n",cs,buf);
			break;

		/***** Netscape Extensions *****/
		case OBJ_NS_CERT_RVKURL:
			Ext_comment_str((CE_Com*)ext,buf,510);
			printf("      Netscape Revocation Url:%s\n        %s",cs,buf);
			break;
		case OBJ_NS_CERT_CRLURL:
			Ext_comment_str((CE_Com*)ext,buf,510);
			printf("      Netscape CA Revocation Url:%s\n        %s",cs,buf);
			break;
		case OBJ_NS_CERT_RENEW:
			Ext_comment_str((CE_Com*)ext,buf,510);
			printf("      Netscape Renewal Url:%s\n        %s",cs,buf);
			break;
		case OBJ_NS_CERT_COMMENT:
			Ext_comment_str((CE_Com*)ext,buf,510);
			printf("      Netscape Comment:%s\n        %s",cs,buf);
			break;
		case OBJ_NS_CERT_TYPE:
			Ext_nscerttype_str((CE_NSType*)ext,buf,510);
			printf("      Netscape Cert Type:%s\n        %s",cs,buf);
			break;

		/***** CSR (PKCS#10) Attribute *****/
		case OBJ_P9_CHALL_PWD:
			str=asn1_get_str(cp,&i);
			printf("      PKCS#9 Challenge Password:\n        %s\n",str);
			free(str);
			break;
		case OBJ_P9_UNST_NAME:
			str=asn1_get_str(cp,&i);
			printf("      PKCS#9 Unstructured Name:\n        %s\n",str);
			free(str);
			break;

		case OBJ_P9_EXT_REQ:
			printf("      PKCS#9 X.509v3 Extension request {\n");
			if((e2=asn1_get_exts(cp,&i))==NULL){
				printf("        decode error!\n");
			}else{
				cgi_print_v3_extensions(e2, 4);
			}
			printf("      }\n");
			break;

		/* attribute */
		case OBJ_ISO17090_HCACT:
			Attr_hcrole_str((AT_HcActorData*)ext,buf,510);
			printf("      hcRole Attribute:%s\n%s",cs,buf);
			break;

		/***** Unknown or Unsupported Extensions *****/
		case 0:
		default:
			{
				char buf[64];

				if(ext->extnID==0)
					objid2str(ext->objid,buf,62);
				else if((ext->extnID>3000)&&(ext->extnID<10000))
					switch_str(ext->extnID,buf);
				else
					break;
				printf("      %s:%s\n        ",buf,cs);

				for(i=0;i<ext->dlen;i++){
					printf("%.2x:",cp[i]);
					if((i%16)==15)printf("\n        ");
				}
				printf("\n");
				break;
			}
	    }
	}
}

void cgi_print_signature(unsigned char *sig, int max, int algo){
	int	i,j;

	if(algo){
		printf("  Signature Algorithm: ");
		cgi_print_sig_algo(algo);
	}else{
		printf("  Signature:\n");
	}

	for(i=0;i<max;i++){
		if(!(j=i%18)) printf("    ");
		printf("%.2x:",sig[i]);
		if(j==17) printf("\n");
	}
	printf("\n");
}

#endif /* __WINDOWS__ */

