/* cgi_mail.c */
/*
 * Copyright (c) 2004-2015 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <netdb.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_io.h>
#include <aicrypto/ok_uconv.h>

#include "aienr_cgi.h"

/* cgi_mail.c */
SSL *cgi_init_mail(char *smtphost, int smtpport);
int cgi_send_mail(SSL *s, char *from, char *recipient, char *body);

/* definitions */
char hostname[128] = "localhost";

/*-----------------------------------------------------
  AiEnroll CGI mail body
-----------------------------------------------------*/
char *aienr_get_mailbody(AiMailInfo *ml, AccList *acc, char *svmail, char *cn, char *fname){
  char path[256],tmp[512],accid[32],sn[32];
  char *cp,*t,*buf=NULL,*body=NULL;
  off_t sz;
  int i,j,k,len,cv;

#ifdef __WINDOWS__
  if(get_install_path(path,210)) goto done;
  strcat(path,"\\templates\\");
  strcat(path,ml->lang);
  strcat(path,"\\");
  strcat(path,fname);
#else
  if (strcmp("jp", ml->lang) == 0) {
	  /* XXX: Japanese locale is 'jp' for htdocs while 'ja' for mail. */
	  strcpy(ml->lang, "ja");
  }
  snprintf(path,254,"%s/templates/mail/%s/%s",DATADIR,ml->lang,fname);
#endif

  if((buf=(char*)get_file2buf(path,&sz))==NULL) goto done;

  i = strlen(buf) + 1024;	/* XXX: size_t... */
  if((body=(char*)malloc(i))==NULL) goto done;
  memset(body,0,i);

  sprintf(accid,"%.7d",acc->acceptID);
  sprintf(sn,"%.7d",acc->serialNum);

  cp = buf;
  do{
    AiVList v[] = {
      {"{ACCEPTID}",{accid,NULL}},
      {"{SERIALNUM}",{sn,NULL}},      
      {"{LOGINNAME}",{acc->name,NULL}},
      {"{GROUPNAME}",{acc->group,NULL}},
      {"{USERNAME}",{cn,NULL}},
      {"{EMAIL}",{acc->email,NULL}},
      {"{ADMEMAIL}",{ml->admemail,NULL}},
      {"{OPEMAIL}",{ml->opemail,NULL}},
      {"{SYSEMAIL}",{svmail,NULL}},
      {"{CANAME}",{ml->caname,NULL}},
      {"{RANAME}",{ml->raname,NULL}},
      {NULL,{NULL}}
    };

    if((t=strstr(cp,"${"))==NULL){
      strcat(body,cp);
      break;
    }
    *t=0; t++; strcpy(tmp,"$");

    for(j=0; v[j].tag; j++){
      if(!memcmp(t,v[j].tag,strlen(v[j].tag))){
	/* check ascii or not */
        len = strlen(v[j].val[0]);
        for(k=cv=0; k<len; k++) cv |= 0x80 & (v[j].val[0])[k];

        if(cv){
	  /*
	   * maybe JP 
	   * "cn","acc->group","ml->caname" and "ml->raname" should be UTF8.
	   */
          char jis_end[8] = { 0x1b, '(', 'B', 0 };
          UC_conv(UC_CODE_UTF8,UC_CODE_JIS,v[j].val[0],strlen(v[j].val[0]),tmp,512);
          strcat(tmp,jis_end);
        }else{ /* ascii */
          strncpy(tmp,v[j].val[0],512);
        }
	t+=strlen(v[j].tag); break;
      }
    }
    strcat(body,cp);
    strcat(body,tmp);
    cp = t;
  }while(cp);

done:
  if(buf) free(buf);
  return body;
}

int aienr_sendmail_sys2adm(AiMailInfo *ml, AccList *acc, char *to, char *cn, char *fname){
        char svmail[256];
        char *body=NULL;
        SSL *ssl=NULL;
        int err=-1;

        /* initialize socket */
        if((ssl = cgi_init_mail(ml->smtphost,ml->smtpport)) == NULL) goto done;

        strcpy(svmail,"airad-system@");
        strncat(svmail,hostname,128);

        /* get mail body */
        if((body=aienr_get_mailbody(ml,acc,svmail,cn,fname))==NULL) goto done;

        /* send mail */
        if(cgi_send_mail(ssl,svmail,to,body)) goto done;

        err = 0;
done:
        SSL_free(ssl);
        if(body) free(body);
        return err;
}

/*------------------------------------------------------------------------------*/

SSL *cgi_init_mail(char *smtphost, int smtpport){
  struct addrinfo hints;
  struct addrinfo *aihead = NULL, *ai;
  SSL *ret = NULL;
  char *smtpport_string = NULL;

  if(gethostname(hostname,128)){
    OK_set_error(ERR_ST_BADSTATE,ERR_LC_WEBE,ERR_PT_WEBMAIL+1,NULL);
    goto error;
  }

  /* initialize socket */
  if(asprintf(&smtpport_string, "%d", smtpport) < 0) {
    OK_set_error(ERR_ST_BADPARAM,ERR_LC_WEBE,ERR_PT_WEBMAIL+1,NULL);
    goto error;
  }
  memset(&hints, 0, sizeof(hints));
  hints.ai_family = AF_UNSPEC;
  hints.ai_socktype = SOCK_STREAM;
  if(getaddrinfo(smtphost, smtpport_string, &hints, &aihead) != 0) {
    OK_set_error(ERR_ST_BADPARAM,ERR_LC_WEBE,ERR_PT_WEBMAIL+1,NULL);
    goto error;
  }
  for (ai = aihead; ai != NULL; ai = ai->ai_next) {
    if ((ret = SSL_socket(ai->ai_family,ai->ai_socktype,ai->ai_protocol)) == NULL) {
      OK_set_error(ERR_ST_SOCKOPEN,ERR_LC_WEBE,ERR_PT_WEBMAIL+1,NULL);
      goto loop_warn;
    }

    if(SSL_connect(ret,ai->ai_addr,ai->ai_addrlen)) {
      OK_set_error(ERR_ST_CONNECT,ERR_LC_WEBE,ERR_PT_WEBMAIL+1,NULL);
      goto loop_warn;
    }

    break;

loop_warn:
    if (ret != NULL) {
      SSL_close(ret);
      SSL_free(ret);
      ret = NULL;
    }
  }

  if (ret == NULL) {
    OK_set_error(ERR_ST_BADPARAM,ERR_LC_WEBE,ERR_PT_WEBMAIL+1,NULL);
    goto error;
  }

  free(smtpport_string);
  freeaddrinfo(aihead);
  return ret;
error:
  SSL_free(ret);
  if(smtpport_string != NULL) {
    free(smtpport_string);
  }
  if(aihead != NULL) {
    freeaddrinfo(aihead);
  }
  return NULL;
}

int cgi_send_mail(SSL *s, char *from, char *recipient, char *body){
  char tmp[512];
  int err=-1;

  /* read ok message */
  while(1){
    if((SSL_gets(s,tmp,512))<=0){ err = 2; goto done; }
    if(!strncmp(tmp,"220",3)){
      if(tmp[3]==' ') break;
    }else{
      OK_set_error(ERR_ST_SMTPSTART,ERR_LC_WEBE,ERR_PT_WEBMAIL+2,(void*)tmp);
      goto done;
    }
  }

  /* send hello */
  strcpy(tmp,MHELO);
  strncat(tmp,hostname,256);
  strncat(tmp,"\r\n",4);
  if((SSL_write(s,tmp,strlen(tmp)))<=0){ err = 1; goto done; }
  while(1){
    if((SSL_gets(s,tmp,512))<=0){ break; }
    if(!strncmp(tmp,"250",3)){
      if(tmp[3]==' ') break;
    }else{
      OK_set_error(ERR_ST_SMTPHELO,ERR_LC_RAD,ERR_PT_RADMAIL+2,(void*)tmp);
      goto done;
    }
  }

  /* send message from */
  strcpy(tmp,MFROM);
  strncat(tmp,from,256);
  strncat(tmp,"\r\n",4);
  if((SSL_write(s,tmp,strlen(tmp)))<=0){ err = 1; goto done; }
  if((SSL_gets(s,tmp,512))<=0){ err = 2; goto done; }
  if(strncmp(tmp,"250",3)){
    OK_set_error(ERR_ST_SMTPFROM,ERR_LC_WEBE,ERR_PT_WEBMAIL+2,(void*)tmp);
    goto done;
  }

  /* send message recipient */
  strcpy(tmp,MRCPT);
  strncat(tmp,recipient,256);
  strncat(tmp,"\r\n",4);
  if((SSL_write(s,tmp,strlen(tmp)))<=0){ err = 1; goto done; }
  if((SSL_gets(s,tmp,512))<=0){ err = 2; goto done; }
  if(strncmp(tmp,"250",3)&&strncmp(tmp,"251",3)){
    OK_set_error(ERR_ST_SMTPRCPT,ERR_LC_WEBE,ERR_PT_WEBMAIL+2,(void*)tmp);
    goto done;
  }

  /* send data command */
  if((SSL_write(s,MDATA,strlen(MDATA)))<=0){ err = 1; goto done; }
  if((SSL_gets(s,tmp,512))<=0){ err = 2; goto done; }
  if(strncmp(tmp,"354",3)){
    OK_set_error(ERR_ST_SMTPDATA,ERR_LC_WEBE,ERR_PT_WEBMAIL+2,(void*)tmp);
    goto done;
  }

  if((SSL_write(s,body,strlen(body)))<=0){ err = 1; goto done; }
  if((SSL_gets(s,tmp,512))<=0){ err = 2; goto done; }
  if(strncmp(tmp,"250",3)){
    OK_set_error(ERR_ST_SMTPMBODY,ERR_LC_WEBE,ERR_PT_WEBMAIL+2,(void*)tmp);
    goto done;
  }

  /* send quit */
  if((SSL_write(s,MQUIT,strlen(MQUIT)))<=0){ err = 1; goto done; }
  if((SSL_gets(s,tmp,512))<=0){ err = 2; goto done; }

  err = 0;
done:
  switch(err){
  case 1: OK_set_error(ERR_ST_SOCKWRITE,ERR_LC_WEBE,ERR_PT_WEBMAIL+2,NULL); break;
  case 2: OK_set_error(ERR_ST_SOCKREAD,ERR_LC_WEBE,ERR_PT_WEBMAIL+2,NULL); break;
  }
  SSL_close(s);

  return err;
}

