/* cgi_log.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifndef __WINDOWS__
# include <sys/time.h>
#endif

#include <aicrypto/ok_err.h>

#include "cgi_common.h"

/* log information */
char peer[128] = "";

/* global values */
FILE *acc_fp=NULL;
FILE *iss_fp=NULL;
FILE *err_fp=NULL;


/*-------------------------------------------------
	logging functions
-------------------------------------------------*/
int cgi_log_open(AiLogInfo *info){
	int i=10;
	/* try 10 times to open */
	do{
		if(info->acclog[0]&&(acc_fp==NULL)){
			if((acc_fp=fopen(info->acclog,"a+"))==NULL){
				fprintf(stderr,"enroll : cannot open access log : %s\n",info->acclog);
			}
		}
		if(info->isslog[0]&&(iss_fp==NULL)){
			if((iss_fp=fopen(info->isslog,"a+"))==NULL){
				fprintf(stderr,"enroll : cannot open issue log : %s\n",info->isslog);
			}
		}
		if(info->errlog[0]&&(err_fp==NULL)){
			if((err_fp=fopen(info->errlog,"a+"))==NULL){
				fprintf(stderr,"enroll : cannot open error log : %s\n",info->errlog);
			}
		}	
		if(err_fp && iss_fp && acc_fp) break;
		i--;
#ifdef __WINDOWS__
		Sleep(100);
#else
		{
			struct timeval tv;
			tv.tv_sec  = 0;
			tv.tv_usec = 100*1000; /* ms */
			select(0,NULL,NULL,NULL,&tv); /* wait 100 ms */
		}
#endif
	}while(i>0);

	return (i==0)?(-1):(0);
}

void cgi_log_close(){
	if(acc_fp){ fclose(acc_fp); acc_fp=NULL; }
	if(err_fp){ fclose(err_fp); err_fp=NULL; }
	if(iss_fp){ fclose(iss_fp); iss_fp=NULL; }
}

/*-------------------------------------------------
	output log
-------------------------------------------------*/
int cgi_log_out(AiLogInfo *info,int type, char *caname, char *msg, int opt){
	char tm[64],buf[1024];
	time_t t;
	int i=0,j=0,ok=-1;

	if(cgi_log_open(info)) goto done;

	time(&t);
	strcpy(tm,"[");
	strcat(tm,ctime(&t)); tm[strlen(tm)-1] = 0; /* clear "\n" */

	/* get message */
	strncpy(buf,tm,64);
	strcat (buf,"] ");

	cgi_set_peer();

	if(*peer){
	  strncat(buf,peer,128);
	  strcat (buf," => ");
	}
	strncat(buf,caname,256);
	if(msg){
	  strcat (buf," => ");
	  strncat(buf,msg,128);
	}
	if((opt)&&(type & AICA_LOG_ERR)&&(OK_get_error())){
	  strcat (buf," << ");
	  strncat(buf,CA_get_errstr(),128);
	}
	strcat(buf,"\n");

	/* write to log file */
	if((type & AICA_LOG_ACC)&&(acc_fp)){
		if(fputs(buf,acc_fp)<0) goto done;
		if(fflush(acc_fp)<0) goto done;

		if((i=cgi_log_checksize(acc_fp,info->acclog_size))<0) goto done;

		if((i>0) && cgi_log_rotate(info->acclog,&acc_fp)) goto done;
	}

	if((type & AICA_LOG_ERR)&&(err_fp)){
//fprintf(stderr,buf);

		if(fputs(buf,err_fp)<0) goto done;
		if(fflush(err_fp)<0) goto done;

		if((i=cgi_log_checksize(err_fp,info->errlog_size))<0) goto done;

		if((i>0) && cgi_log_rotate(info->errlog,&err_fp)) goto done;
	}
	
	if((type & AICA_LOG_ISS)&&(iss_fp)){
		if(fputs(buf,iss_fp)<0) goto done;
		if(fflush(iss_fp)<0) goto done;

		if((j=cgi_log_checksize(iss_fp,info->isslog_size))<0) goto done;

		if((j>0) && cgi_log_rotate(info->isslog,&iss_fp)) goto done;
	}
	
	ok = 0;
done:
	cgi_log_close();

	return ok;
}

/*-------------------------------------------------
	check log size
-------------------------------------------------*/
int cgi_log_checksize(FILE *fp, int size_kb){
	struct stat st;

	if(size_kb<=0) return 0;
	if(fstat(fileno(fp),&st)) goto error;

	return (st.st_size >= (size_kb<<10))?(1):(0);
error:
	return -1;
}

/*-------------------------------------------------
	rotate logs
-------------------------------------------------*/
int cgi_log_rotate(char *fname,FILE **fp){
	time_t t;
	struct tm *ltm;
	char buf[256];

	time(&t);
	ltm = localtime(&t);
	snprintf(buf,254,"%s.%.2d%.2d%.2d%.2d%.2d%.2d",fname,
			ltm->tm_year+1900,ltm->tm_mon+1,ltm->tm_mday,
			ltm->tm_hour,ltm->tm_min,ltm->tm_sec);

	if(*fp){ fclose(*fp); *fp=NULL; }

	/* ignore error case, because other crlpublisher keeps
	 * file handler and rename cannot be accomplished.
	 */
	rename(fname,buf);

	if((*fp=fopen(fname,"a+"))==NULL){
		fprintf(stderr,"cannot open log file : %s\n",fname);
		goto error;
	}
	return 0;
error:
	return -1;
}

/*---------------------------------------------------
  set peer information
---------------------------------------------------*/
void cgi_set_peer(){
  char *buf;

  *peer = 0;
  if((buf=getenv("REMOTE_HOST")) != NULL) {
    strncpy(peer,buf,126);
  }else if((buf=getenv("REMOTE_ADDR")) != NULL) {
    strncpy(peer,buf,126);
  }
}
