/* cgi_ldap.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>

#include "cgi_common.h"

#ifdef __WINDOWS__
# define STRCASECMP stricmp
#else
# define STRCASECMP strcasecmp
#endif

#ifdef HAVE_LDAP_H

#if HAVE_SASL_H
#include <sasl/sasl.h>

char *userdn = "";
char *userpw = "";

/* callback function for CRAM-MD5 and DIGEST-MD5 */
static int cb_md5(LDAP *ld, unsigned flags, void *defaults, void *in)
{
	sasl_interact_t *interact = in;

	while (interact->id != SASL_CB_LIST_END) {
		switch (interact->id) {
		case SASL_CB_USER:     /* authz id */
		case SASL_CB_AUTHNAME: /* authc id */
			interact->result = strdup(userdn);
			interact->len = strlen(userdn);
			break;
		case SASL_CB_PASS:     /* password */
			interact->result = strdup(userpw);
			interact->len = strlen(userpw);
			break;
		case SASL_CB_GETREALM: /* realm */
		default:
			interact->result = strdup("");
			interact->len = 0;
			break;

		}
		interact++;
	}

	return LDAP_SUCCESS;
}
#endif /* HAVE_SASL_H */

/*--------------------------------------------------------
	LDAP bind (simple, cram-md5, and digest-md5)
---------------------------------------------------------*/
int cgi_ldap_bind(LDAP *ld, char *dn, char *passwd, int method){
	struct berval cred;
	int version = LDAP_VERSION3;
	int ret = -1;

	memset(&cred,0,sizeof(struct berval));
	if(passwd){
		cred.bv_len = strlen( passwd );
		cred.bv_val = passwd;
	}

	switch(method){
#ifndef WITHAD
	case 1: /* cram md5 */
#if __WINDOWS__
		ret = ldap_sasl_md5_bind_s(ld,dn,&cred,NULL,NULL);
#else
# if HAVE_SASL_H
		if((ret=ldap_set_option(ld,LDAP_OPT_PROTOCOL_VERSION,&version))!=LDAP_SUCCESS)
		  break;

		userdn = (dn)?(dn):("");
		userpw = (passwd)?(passwd):("");
		ret = ldap_sasl_interactive_bind_s(ld,NULL,"CRAM-MD5",NULL,NULL,
							LDAP_SASL_QUIET,cb_md5,0);
# endif
#endif
		break;
	case 2: /* digest md5 */
#if __WINDOWS__
		ret = ldap_sasl_digest_md5_bind_s(ld,dn,cred.bv_val,NULL,NULL);
#else
# if HAVE_SASL_H
		if((ret=ldap_set_option(ld,LDAP_OPT_PROTOCOL_VERSION,&version))!=LDAP_SUCCESS)
		  break;

		userdn = (dn)?(dn):("");
		userpw = (passwd)?(passwd):("");
		ret = ldap_sasl_interactive_bind_s(ld,NULL,"DIGEST-MD5",NULL,NULL,
							LDAP_SASL_QUIET,cb_md5,0);

		if(ret == LDAP_MORE_RESULTS_TO_RETURN){
		  /* maybe we should parse this.., but here we ignore this message
		     and return LDAP_SUCCESS to CGI */
		  ret = LDAP_SUCCESS;
		}
# endif
#endif
		break;
#endif /* not WITHAD */
	default:
		if((ret=ldap_set_option(ld,LDAP_OPT_PROTOCOL_VERSION,&version))!=LDAP_SUCCESS)
		  break;
		ret = ldap_simple_bind_s(ld,dn,passwd);
		break;
	}
	return ret;
}

/*--------------------------------------------------------
	LDAP modify attribute 
---------------------------------------------------------*/
int cgi_mod_attr(LDAP *ld, char *dn, int op, char *attr, char *val){
  LDAPMod *mod[2];
  int i,ok=-1;

  memset(mod,0,sizeof(LDAPMod*)*2);

  if((mod[0]=get_ldapmov(op,attr,val))==NULL) goto done;

  ok = ldap_modify_s(ld,dn,mod);

done:
  for(i=0;mod[i];i++) ldapmod_free(mod[i]);
  return ok;
}

int cgi_mod_attr_bin(LDAP *ld, char *dn, int op, char *attr, unsigned char *bin, int blen){
  LDAPMod *mod[2];
  int i,ok=-1;

  memset(mod,0,sizeof(LDAPMod*)*2);

  if((mod[0]=get_ldapmov_bin(op,attr,bin,blen))==NULL) goto done;

  ok = ldap_modify_s(ld,dn,mod);

done:
  for(i=0;mod[i];i++) ldapmod_bin_free(mod[i]);
  return ok;
}


LDAPMod *get_ldapmov(int op,char *attr,char *val){
  LDAPMod *ret = NULL;

  if((ret=(LDAPMod*)malloc(sizeof(LDAPMod)))==NULL) goto error;
  memset(ret,0,sizeof(LDAPMod));

  if((ret->mod_values=(char**)malloc(sizeof(char*)*2))==NULL) goto error;
  memset(ret->mod_values,0,sizeof(char*)*2);
  if(val)
  if((ret->mod_values[0]=strdup(val))==NULL) goto error;

  if((ret->mod_type=strdup(attr))==NULL) goto error;
  ret->mod_op = op;

  return ret;
error:
  ldapmod_bin_free(ret);
  return NULL;
}

LDAPMod *get_ldapmov_bin(int op,char *atname,unsigned char *bin,int blen){
  LDAPMod *ret = NULL;

  if((ret=(LDAPMod*)malloc(sizeof(LDAPMod)))==NULL) goto error;
  memset(ret,0,sizeof(LDAPMod));

  if((ret->mod_bvalues=(struct berval**)malloc(sizeof(struct berval*)*2))==NULL) goto error;
  memset(ret->mod_bvalues,0,sizeof(struct berval*)*2);

  if((ret->mod_bvalues[0]=(struct berval*)malloc(sizeof(struct berval)))==NULL) goto error;
  if((ret->mod_bvalues[0]->bv_val=(char*)malloc(blen))==NULL) goto error;
  memcpy(ret->mod_bvalues[0]->bv_val,bin,blen);
  ret->mod_bvalues[0]->bv_len = blen;

  if((ret->mod_type=strdup(atname))==NULL) goto error;
  ret->mod_op = op;

  return ret;
error:
  ldapmod_bin_free(ret);
  return NULL;
}

void ldapmod_free(LDAPMod *mod){
  int i;

  if(mod==NULL) return;

  for(i=0;mod->mod_values[i];i++)
    free(mod->mod_values[i]);

  free(mod->mod_values);
  free(mod->mod_type);
  free(mod);
}

void ldapmod_bin_free(LDAPMod *mod){
  int i;

  if(mod==NULL) return;

  for(i=0;mod->mod_bvalues[i];i++){
    if(mod->mod_bvalues[i]->bv_val)
      free(mod->mod_bvalues[i]->bv_val); 
    free(mod->mod_values[i]);
  }
  free(mod->mod_bvalues);
  free(mod->mod_type);
  free(mod);
}

int cgi_ldap_addcert(AiLDAPInfo *info, int op, char *moddn, char *attr, Cert *ct){
  LDAP *ld=NULL;
  int len, err=-1;

  ASN1_skip_(ct->der,&len);

  if((ld=ldap_init(info->ldaphost,info->ldapport))==NULL){
    OK_set_error(ERR_ST_LDAPINIT,ERR_LC_WEBE,ERR_PT_WEBLDAP+4,NULL);
    goto done;
  }

  if(cgi_ldap_bind(ld,info->ldapadmin,info->ldapadminpwd,info->ldapbind) != LDAP_SUCCESS){
    OK_set_error(ERR_ST_LDAPBIND,ERR_LC_WEBE,ERR_PT_WEBLDAP+4,NULL);
    goto done;
  }
  
  if(cgi_mod_attr_bin(ld, moddn, op, attr, ct->der, len) != LDAP_SUCCESS){
    OK_set_error(ERR_ST_LDAPMODIFY,ERR_LC_WEBE,ERR_PT_WEBLDAP+4,NULL);
    goto done;
  }

  err = 0;
done:
  if(ld) ldap_unbind(ld);
  return err;
}


/*----------------------------------------------------------
  LDAP utilities
----------------------------------------------------------*/
int ldapdn2certdn(char *ldn, CertDN *cdn){
  char *c,*v,buf[256];
  int more_rdn=0, in_quo=0, last_c=0;
  int i=0, j=0, err=-1;
  int end=0;

  /* split elements */
  while(ldn && *ldn &&(i<RDN_MAX)){
    /* ignore space until charactor */
    if(*ldn == ' '){ ldn++; continue; }

    /* get type char */
    if((v = strchr(ldn,'='))==NULL){
      OK_set_error(ERR_ST_BADPARAM,ERR_LC_WEBE,ERR_PT_WEBLDAP+6,NULL);
      goto done;
    }
    *v=0; v++;

    /* ignore space until charactor -- LDAPv2 compatible */
    while(*v == ' '){ v++; }
    c=v;

    /* get string value */
    more_rdn = in_quo = last_c = j = 0;
    memset(buf,0,256);

    while(*c &&(j<254)){
      switch(*c){
      case '"':
        if(!in_quo)
          in_quo = 1;
        else{
          in_quo = 0; c++;
          while(*c == ' '){ c++; }
          continue;
        }
        break;
      case '\\': /* escape 1 char */
        c++; buf[j] = *c; j++;
        break;
      case ',':
      case ';':
        if(!in_quo){
          more_rdn = 1;
          last_c = 1; /* last char */
        }else{
          buf[j] = *c; j++;
        }
        break;
      default:
        buf[j] = *c; j++;
        break;
      }

      if(last_c) break;
      c++;
    }
    
    if(!last_c&&*c=='\0') end=1;
    if(last_c) *c=0;

    if ((cdn->rdn[i].tag = strdup(buf)) == NULL){
      OK_set_error(ERR_ST_STRDUP,ERR_LC_WEBE,ERR_PT_WEBLDAP+6,NULL);
      goto done;
    }
    if((cdn->rdn[i].tagoid = get_dn_kind(ldn))<0){
      OK_set_error(ERR_ST_UNSUPPORTED_PARAM,ERR_LC_WEBE,ERR_PT_WEBLDAP+6,NULL);
      goto done;
    }
    cdn->rdn[i].derform = asn1_str_type(v);

    i++;
    if(end==1){
      ldn=c;
    }else{
      ldn=c+1;
    }
  }
  cdn->num = i;

  /* check if last "," exists */
  if(more_rdn){
    OK_set_error(ERR_ST_BADPARAM,ERR_LC_WEBE,ERR_PT_WEBLDAP+6,NULL);
    goto done;
  }

  err = 0;
done:
  return err;
}

int certdn2ldapdn(CertDN *dn, char *ret, int ret_max){
  char *dir_t[16]={
    "c","st","l","o","ou","cn","sir","snum",
    "given","title","","", "","","",""
  };
  unsigned char dcnt[16];
  char tmp[256];
  int i,j,k,l,swap,dnum;

  memset(dcnt,0,16);

  dnum = dn->num;
  swap = (dn->rdn[0].tagoid == OBJ_DIR_DC)||
    ((dn->rdn[0].tagoid < dn->rdn[dnum-1].tagoid)&&(dn->rdn[dnum-1].tagoid != OBJ_DIR_DC));

  *ret = 0;

  /* need attribute tag counter. if same tag like "ou=..., ou=..." exists,
   * last one should be double-quoted. this is compativility for OpenLDAP.
   */
  i = (swap)?(dnum-1):(0);
  for(l=0;l<dnum;l++){ 
    j = dn->rdn[i].tagoid; 
    if((OBJ_DIR_C<=j)&&(j<=OBJ_DIR_TITLE)){ (dcnt[j-OBJ_DIR_C])++; }
    i += (swap)?(-1):(1);
  }

  i = (swap)?(dnum-1):(0);
  for(l=0;l<dnum;l++){
    j = dn->rdn[i].tagoid;

    *tmp = 0;
    if((OBJ_DIR_C<=j)&&(j<=OBJ_DIR_TITLE)){
      strcpy (tmp,dir_t[j-OBJ_DIR_C]);
      strcat (tmp,"=");

      if(dcnt[j-OBJ_DIR_C] > 1) strcat (tmp,"\"");
      strncat(tmp,dn->rdn[i].tag,240);
      if(dcnt[j-OBJ_DIR_C] > 1) strcat (tmp,"\"");
      (dcnt[j-OBJ_DIR_C])--;

    }else if(j==OBJ_DIR_UID){
      strcpy (tmp,"uid=");
      strncat(tmp,dn->rdn[i].tag,240);
    }else if(j==OBJ_DIR_DC){
      strcpy (tmp,"dc=");
      strncat(tmp,dn->rdn[i].tag,240);
    }

    if((l<dnum-1)&&(j!=OBJ_DIR_EMAIL)) strcat(tmp,", ");

    k = strlen(tmp);
    k = (ret_max>k)?(k):(ret_max);
    strncat(ret,tmp,k);
    ret_max -= k;

    i += (swap)?(-1):(1);
  }

  return 0;
}

int swap_dn(CertDN *dn){
  char *tc;
  int i,j,tk,sk,max = dn->num >> 1;

  for(i=0;i<max;i++){
    j = dn->num-i-1;
    tk = dn->rdn[i].tagoid;
    dn->rdn[i].tagoid = dn->rdn[j].tagoid;
    dn->rdn[j].tagoid = tk;

    sk = dn->rdn[i].derform;
    dn->rdn[i].derform = dn->rdn[j].derform;
    dn->rdn[j].derform = sk;

    tc = dn->rdn[i].tag;
    dn->rdn[i].tag = dn->rdn[j].tag;
    dn->rdn[j].tag = tc;
  }
  return 0;
}

/*--------------------------------------------------------
  certificate registration
---------------------------------------------------------*/
int cgi_ldap_checkent( LDAP * ld, char * filter, char * base, char * license, char ** dn, Cert ** ret ){
  char * attrs[] = { "uid", "userSMIMECertificate", NULL };
  char **val = NULL, *dntmp, *der=NULL;
  LDAPMessage *res=NULL, *ent;
  struct berval **vals;
  int err = -1;

  *license=0; *dn=NULL; *ret=NULL;

  if(ldap_search_s( ld, base, LDAP_SCOPE_SUBTREE, filter, attrs, 0, &res ) != LDAP_SUCCESS){
    OK_set_error(ERR_ST_LDAPSEARCH,ERR_LC_WEBE,ERR_PT_WEBLDAP+8,NULL);
    goto done;
  }
  if(ldap_count_entries( ld, res ) < 1){
    OK_set_error(ERR_ST_LDAPNOENT,ERR_LC_WEBE,ERR_PT_WEBLDAP+8,NULL);
    goto done;
  }
  if((ent = ldap_first_entry(ld, res))==NULL) goto done;
  
  if((dntmp = ldap_get_dn(ld, ent))==NULL) goto done;
  if((*dn=(char*)malloc(strlen(dntmp)+1))==NULL) goto done;

  strcpy( *dn, dntmp );
  ldap_memfree(dntmp);

  if((val = ldap_get_values( ld, ent, "uid" ))!=NULL && *val!=NULL ){
    strcpy( license, val[0] );  
  }
  if(val) ldap_value_free(val);

  if((vals = ldap_get_values_len( ld, ent, "userSMIMECertificate" ))!=NULL && *vals!=NULL ){
    if((der=(char*)malloc((*vals)->bv_len+2))==NULL) goto done;

    memcpy(der,(*vals)->bv_val, (*vals)->bv_len);
    if((*ret = ASN1_read_cert(der))==NULL) goto done;
  }
  if(vals) ldap_value_free_len(vals);

  err = 0;
done:
  if(err){
    if(*ret){ Cert_free(*ret); *ret=NULL;}
    if(*dn){ free(*dn); *dn=NULL;}
  }
  if(res) ldap_msgfree(res);
  return err;
}

int cgi_ldap_addperson( LDAP * ld, char * name, char * base, char * email, char * o ){
  LDAPMod *attrs[6];
  char * dn=NULL;
  int i=0, rc=0, err=-1;

  memset(attrs,0,sizeof(LDAPMod*)*6);
  if((attrs[0]=(LDAPMod*)malloc(sizeof(LDAPMod)))==NULL) goto done;
  if((attrs[0]->mod_vals.modv_strvals = (char**)calloc(5,sizeof(char*)))==NULL) goto done;
  attrs[0]->mod_op=0;
  attrs[0]->mod_type = "objectClass";
  attrs[0]->mod_vals.modv_strvals[0] = "top";
  attrs[0]->mod_vals.modv_strvals[1] = "person";
  attrs[0]->mod_vals.modv_strvals[2] = "organizationalPerson";
  attrs[0]->mod_vals.modv_strvals[3] = "inetOrgPerson";

  if((attrs[1]=(LDAPMod*)malloc(sizeof(LDAPMod)))==NULL) goto done;
  if((attrs[1]->mod_vals.modv_strvals = (char**)calloc(2,sizeof(char*)))==NULL) goto done;
  attrs[1]->mod_op=0;
  attrs[1]->mod_type = "cn";
  attrs[1]->mod_vals.modv_strvals[0] = name;

  if((attrs[2]=(LDAPMod*)malloc(sizeof(LDAPMod)))==NULL) goto done;
  if((attrs[2]->mod_vals.modv_strvals = (char**)calloc(2,sizeof(char*)))==NULL) goto done;
  attrs[2]->mod_op=0;
  attrs[2]->mod_type = "sn";
  attrs[2]->mod_vals.modv_strvals[0] = name;

  if((attrs[3]=(LDAPMod*)malloc(sizeof(LDAPMod)))==NULL) goto done;
  if((attrs[3]->mod_vals.modv_strvals = (char**)calloc(2,sizeof(char*)))==NULL) goto done;
  attrs[3]->mod_op=0;
  attrs[3]->mod_type = "mail";
  attrs[3]->mod_vals.modv_strvals[0] = email;

  if((attrs[4]=(LDAPMod*)malloc(sizeof(LDAPMod)))==NULL) goto done;
  if((attrs[4]->mod_vals.modv_strvals = (char**)calloc(2,sizeof(char*)))==NULL) goto done;
  attrs[4]->mod_op=0;
  attrs[4]->mod_type = "o";
  attrs[4]->mod_vals.modv_strvals[0] = o;
  
  if((dn = (char*)malloc(strlen(name)+strlen(base)+12))==NULL) goto done;
  sprintf(dn,"cn=\"%s\" , %s",name,base);

  if((rc=ldap_add_s( ld, dn, attrs ))!=LDAP_SUCCESS){
    OK_set_error(ERR_ST_LDAPMODIFY,ERR_LC_WEBE,ERR_PT_WEBLDAP+9,NULL);
    goto done;
  }

  err = 0;
done:
  if(dn) free(dn);
  for(i=0;attrs[i];++i){
    if(attrs[i]->mod_vals.modv_strvals){
      free(attrs[i]->mod_vals.modv_strvals);
    }
    free(attrs[i]);
  }
  return err;
}

int cgi_ldap_addorg( LDAP * ld, char * org, char * base, char * flag ){
  LDAPMod *attrs[3];
  char * dn=NULL;
  int i=0, rc=0, err=-1;

  memset(attrs,0,sizeof(LDAPMod*)*3);
  if((attrs[0]=(LDAPMod*)malloc(sizeof(LDAPMod)))==NULL) goto done;
  if((attrs[0]->mod_vals.modv_strvals = (char**)calloc(3,sizeof(char*)))==NULL) goto done;
  attrs[0]->mod_op=0;
  attrs[0]->mod_type = "objectClass";
  attrs[0]->mod_vals.modv_strvals[0] = "top";
  if(STRCASECMP(flag,"ou")==0){
    attrs[0]->mod_vals.modv_strvals[1] = "organizationalUnit";
  }else{
    attrs[0]->mod_vals.modv_strvals[1] = "organization";
  }

  if((attrs[1]=(LDAPMod*)malloc(sizeof(LDAPMod)))==NULL) goto done;
  if((attrs[1]->mod_vals.modv_strvals = (char**)calloc(2,sizeof(char*)))==NULL) goto done;
  attrs[1]->mod_op=0;
  if(STRCASECMP(flag,"ou")==0){
    attrs[1]->mod_type = "ou";
  }else{
    attrs[1]->mod_type = "o";
  }
  attrs[1]->mod_vals.modv_strvals[0] = org;
  
  if((dn = (char*)malloc(strlen(org)+strlen(base)+12))==NULL) goto done;
  if(STRCASECMP(flag,"ou")==0){
    sprintf(dn,"ou=\"%s\" , %s",org,base);
  }else{
    sprintf(dn,"o=\"%s\" , %s",org,base);
  }    

  if((rc=ldap_add_s( ld, dn, attrs ))!=LDAP_SUCCESS && (rc!=LDAP_ALREADY_EXISTS)){
    OK_set_error(ERR_ST_LDAPMODIFY,ERR_LC_WEBE,ERR_PT_WEBLDAP+10,NULL);
    goto done;
  }

  err = 0;
done:
  if(dn) free(dn);
  for(i=0;attrs[i];++i){
    if(attrs[i]->mod_vals.modv_strvals){
      free(attrs[i]->mod_vals.modv_strvals);
    }
    free(attrs[i]);
  }
  return err;
}

int cgi_ldap_replace_attr( LDAP * ld, char * dn, char * attr, char **vals){
  LDAPMod *attrs[2];
  int cnt,i=0, err = -1;

  for(cnt=0; vals[cnt]&&*(vals[cnt]); cnt++); /* get vals count */

  memset(attrs,0,sizeof(LDAPMod*)*2);
  if((attrs[0]=(LDAPMod*)malloc(sizeof(LDAPMod)))==NULL) goto done;

#ifdef __WINDOWS__ /* eds libldap.dll */
  if(cnt==0){
	  attrs[0]->mod_vals.modv_strvals = NULL;
  }else{
	  if((attrs[0]->mod_vals.modv_strvals = (char**)calloc(cnt+1,sizeof(char*)))==NULL) goto done;
  }
#else /* linux default ldap library (openldap) */
  if((attrs[0]->mod_vals.modv_strvals = (char**)calloc(cnt+1,sizeof(char*)))==NULL) goto done;
#endif
  attrs[0]->mod_op=LDAP_MOD_REPLACE;
  attrs[0]->mod_type = attr;
  for(i=0; i<cnt; i++){ attrs[0]->mod_vals.modv_strvals[i] = vals[i]; }
  if(ldap_modify_s( ld, dn, attrs )!=LDAP_SUCCESS){
    OK_set_error(ERR_ST_LDAPMODIFY,ERR_LC_WEBE,ERR_PT_WEBLDAP+11,NULL);
    goto done;
  }

  err = 0;
done:
  for(i=0;attrs[i];++i){
    if(attrs[i]->mod_vals.modv_strvals){
      free(attrs[i]->mod_vals.modv_strvals);
    }
    free(attrs[i]);
  }
  return err;
}

int cgi_ldap_getmail( LDAP * ld, char * dn, char * name, char * mail ){
  char *attrs[] = { "cn", "mail", NULL };
  char **val = NULL;
  LDAPMessage *res=NULL, *ent;
  int err = -1;

  if(ldap_search_s(ld,dn,LDAP_SCOPE_BASE,NULL,attrs,0,&res)!=LDAP_SUCCESS){
    OK_set_error(ERR_ST_LDAPSEARCH,ERR_LC_WEBE,ERR_PT_WEBLDAP+12,NULL);
    goto done;
  }
  if(ldap_count_entries( ld, res ) < 1 ){
    OK_set_error(ERR_ST_LDAPNOENT,ERR_LC_WEBE,ERR_PT_WEBLDAP+12,NULL);
    goto done;
  }
  if((ent = ldap_first_entry( ld, res )) == NULL) goto done;

  if((val = ldap_get_values( ld, ent, "mail" ))!=NULL && *val!=NULL ){
    strncpy( mail, val[0], 254);
  }
  if(val) ldap_value_free(val);

  if((val = ldap_get_values( ld, ent, "cn" ))!=NULL && *val!=NULL ){
    strncpy( name, val[0], 126);
  }
  if(val) ldap_value_free(val);

  err = 0;
done:
  if(res) ldap_msgfree(res);
  return err;
}

int cgi_ldap_entcount( LDAP * ld, char * filter, char * base, int * ret ){
  LDAPMessage * res=NULL;
  int rc,err=-1;

  rc = ldap_search_s( ld, base, LDAP_SCOPE_SUBTREE,filter,NULL,0,&res);
  
  if(rc==LDAP_NO_SUCH_OBJECT){
    *ret = 0;
  }else if(rc==LDAP_SUCCESS){
    *ret=ldap_count_entries( ld, res );
  }else{
    OK_set_error(ERR_ST_LDAPSEARCH,ERR_LC_WEBE,ERR_PT_WEBLDAP+13,NULL);
    goto done;
  }
  err = 0;
done:
  if(res) ldap_msgfree(res);
  return err;
}

int cgi_ldap_attrvals(LDAP *ld, LDAPMessage *ent, AiLAttr *v){
  char **vals=NULL,*at=NULL;
  void *ptr=NULL; /* BerElement */
  struct berval **bvals=NULL;
  int i,j,ok = -1;

  for(i=0; v[i].tag; i++){
    if((at=ldap_first_attribute(ld,ent,(BERELEMENT**)&ptr))==NULL) goto done;

    while(at){
      if(at && !igcase_strcmp(at,v[i].tag)){

	if(strstr(v[i].tag,";binary")){
	  /* binary */
	  bvals = ldap_get_values_len(ld,ent,at);

	  for(j=0; j<MAXGROUP && bvals[j]; j++){
	    if(v[i].val[j]){
	      if((*v[i].val[j]=malloc(bvals[j]->bv_len+2))==NULL) goto done;
	      memcpy(*v[i].val[j],bvals[j]->bv_val,bvals[j]->bv_len);
	    }
	  }
	}else{
	  /* text */
	  vals = ldap_get_values(ld,ent,at);

	  for(j=0; j<MAXGROUP && vals[j]; j++){
	    if(v[i].val[j]){
	      /* code should be UTF8 */
		    if ((*v[i].val[j] = strdup(vals[j])) == NULL) goto done;
	    }
	  }
	}
	break;
      }

      if(at){ ldap_memfree(at); at = NULL; }
      if(vals){ ldap_value_free(vals); vals = NULL; }
      if(bvals){ ldap_value_free_len(bvals); bvals = NULL; }

      at = ldap_next_attribute(ld,ent,ptr);
    }
  }

  ok = 0;
done:
  if(ptr) LDAP_BER_FREE(ptr,0);
  return ok;
}


#endif /* HAVE_LDAP_H */
