/* cgi_http.c */
/*
 * Copyright (c) 2004-2012 National Institute of Informatics in Japan, 
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must 
 * include this notice in the file.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "cgi_common.h"
#include "ok_ssl.h"
#include "afi_utils.h"

/* http header */
char http_get[] ={ 
  "GET %s HTTP/1.1\r\n"
  "Host: %s\r\n"
  "Connection: Keep-Alive\r\n\r\n"
};

char http_post[] ={ 
  "POST %s HTTP/1.1\r\n"
  "Host: %s\r\n"
  "Content-Type: application/x-www-form-urlencoded\r\n"
  "Content-Length: %d\r\n"
  "Connection: Keep-Alive\r\n\r\n"
  "%s\r\n"
};


/*-------------------------------------------------
  get url body
-------------------------------------------------*/
char* cgi_url_get(AiLogInfo *log, char * url, int *ret_len){
  char host[256],path[256],buf[512];
  char *ret = NULL;
  SSL *s = NULL;
  int len;

  *ret_len = 0;
  if((s = cgi_connect_http(url,host,path))==NULL){
    cgi_log_out(log,AICA_LOG_ERR,"cannot connect http server",url,1);
    goto error;
  }

  len = snprintf(buf,510,http_get,path,host);
  if(cgi_send(s,buf,len)<0){
    cgi_log_out(log,AICA_LOG_ERR,"cannot send GET operation",url,1);
    goto error;  
  }

  if((len = cgi_recv_httpheader(s))<0){
    cgi_log_out(log,AICA_LOG_ERR,"cannot analyze http header",url,1);
    goto error;
  }

  if((ret = cgi_recv_body(s,len))==NULL){
    cgi_log_out(log,AICA_LOG_ERR,"cannot recv http body",url,1);
    goto error;
  }
  *ret_len = len;

  SSL_close(s);
  SSL_free(s);

  return ret;
error:
  if(s){ SSL_close(s); SSL_free(s); }
  if(ret) AIFREE(ret);
  return NULL;
}

/*-------------------------------------------------
  post url body
-------------------------------------------------*/
char* cgi_url_post(AiLogInfo *log, char * url, char *post, int *ret_len){
  char host[256],path[256],buf[2048];
  char *cnt = NULL, *ret = NULL;
  SSL *s = NULL;
  int len;

  *ret_len = 0;
  if((s = cgi_connect_http(url,host,path))==NULL){
    cgi_log_out(log,AICA_LOG_ERR,"cannot connect http server",url,1);
    goto error;
  }

  len = snprintf(buf,2046,http_post,path,host,strlen(post),post);
  if(cgi_send(s,buf,len)<0){
    cgi_log_out(log,AICA_LOG_ERR,"cannot send POST operation",url,1);
    goto error;
  }

  if((len = cgi_recv_httpheader(s))<0){
    cgi_log_out(log,AICA_LOG_ERR,"cannot analyze http header",url,1);
    goto error;
  }

  if((ret = cgi_recv_body(s,len))==NULL){
    cgi_log_out(log,AICA_LOG_ERR,"cannot recv http body",url,1);
    goto error;
  }
  *ret_len = len;

  SSL_close(s);
  SSL_free(s);

  return ret;
error:
  if(s){ SSL_close(s); SSL_free(s); }
  if(ret) AIFREE(ret);
  return NULL;
}


/*-------------------------------------------------
  utility functions
-------------------------------------------------*/
SSL *cgi_connect_http(char *url, char *host, char *path){
  struct addrinfo hints;
  struct addrinfo *aihead = NULL, *ai;
  SSL *ret = NULL;
  int port=80,ssl=0;
  char *port_string = NULL;
  char hst[strlen(url)+1], *cp;

  /* host & path length should be less than 256 byte */
  if(parse_url(url,hst,sizeof(hst),&port,path,256,&ssl)!=PARSE_URL_PARSE_SUCCESS) goto error;

  /* get client socket */
  if(asprintf(&port_string, "%d", port)<0) goto error;
  memset(&hints, 0, sizeof(hints));
  hints.ai_family = AF_UNSPEC;
  hints.ai_socktype = SOCK_STREAM;
  if(getaddrinfo(hst, port_string, &hints, &aihead)!=0) goto error;
  for (ai = aihead; ai != NULL; ai = ai->ai_next) {
    if((ret=SSL_socket(ai->ai_family, ai->ai_socktype, ai->ai_protocol))==NULL) goto loop_warn;

    /* set client certificate */
    if(ssl){
      //if(SSL_set_store(lc->sock,lc->store)) goto loop_warn;
      SSL_set_vfytype(ret,DONT_VERIFY);
    }

    /* connect to the server */
    if(SSL_connect(ret,ai->ai_addr,ai->ai_addrlen)) goto loop_warn;

    break;

loop_warn:
    if (ret != NULL) {
      SSL_close(ret);
      SSL_free(ret);
      ret = NULL;
    }
  }

  if (ret == NULL) {
    goto error;
  }

  /* do SSL handshake if necessary */
  if(ssl){
    if(SSL_handshake(ret)) goto error;
  }

  /* make HTTP Host field */
  if(isipv6addr(hst)==ISIPV6ADDR_TRUE){
    if ((cp = strchr(hst, '%')) != NULL) {
      *cp = '\0';
    }
    sprintf(host,"[%s]",hst);
  } else {
    strcpy(host,hst);
  }

  free(port_string);
  freeaddrinfo(aihead);
  return ret;
error:
  if(ret){ SSL_close(ret); SSL_free(ret); }
  if(port_string != NULL) { free(port_string); }
  if(aihead != NULL) { freeaddrinfo(aihead); }
  return NULL;
}

int cgi_send(SSL *s, char *buf, int len){
  int i,ret=0;

  while(ret<len){
    if((i=SSL_write(s,&buf[ret],len-ret))<=0) return i;
    ret+=i;
  }
  return ret;
}

int cgi_gets(SSL *s, char *buf,int max){
  unsigned char tmp[4];
  char lst = 0;
  int i;

  for(i=0; i<max; i++, buf++){
    if(SSL_read(s,(char*)tmp,1)<0) return -1;
    *buf = *tmp;
    if((lst=='\r')&&(*tmp=='\n')) return i+1;
    lst = *buf;
  }
  return -1;
}

#if !defined(HAVE_STRNICMP) && !defined(strnicmp)
#ifdef HAVE_STRNCASECMP

int strnicmp(char *c1,char *c2,int len){ return strncasecmp(c1,c2,len); }

# else
/************ not ansi c **********/
int strnicmp(char *str1, char *str2, size_t count){
  int i,j,k=0;

  while(k < count){
    i = *str1;
    if(('a'<=i)&&(i<='z')) i-=0x20;
    j = *str2;
    if(('a'<=j)&&(j<='z')) j-=0x20;

    i-=j;
    if(i) return i;
    if(j==0) break;

    str1++; str2++; k++;
  }
  return 0;
}
# endif
#endif

int cgi_recv_httpheader(SSL *s){
  char buf[256],*cp,*t;
  int ret=-1;

  /* header */
  do{
    memset(buf,0,256);
    if(cgi_gets(s,buf,256)<0) goto done;

    if(strncmp(buf,"HTTP",4)) continue;

    if(strstr(buf,"200")) break; /* 200 OK */
    else if(strstr(buf,"100")) continue; /* 100 CONTINUE */
    else {
      fprintf(stderr,buf); goto done; /* error */
    }
  }while(1);

  do{
    memset(buf,0,256);
    if(cgi_gets(s,buf,256)<0) goto done;

    t=buf;
    while(*t==' '){ t++; }

    if(!strncmp(t,"\r\n",2)) break;

    else if(!strnicmp(t,"Content-Length:",15)){
      if((cp=strchr(t,':'))==NULL) goto done;
      do{ ++cp; }while(*cp==' ');
      ret = atoi(cp);
    }
    else if(!strnicmp(t,"Content-Type:",13)){
      if((cp=strchr(t,':'))==NULL) goto done;
      do{ ++cp; }while(*cp==' ');
      if(strnicmp(cp,"text/plain",10)) goto done;
    }
    else if(!strnicmp(t,"Transfer-Encoding:",17)){
      if((cp=strchr(t,':'))==NULL) goto done;
      do{ ++cp; }while(*cp==' ');
      if(!strnicmp(cp,"chunked",7)) goto done;
    }

  }while(1);

done:
  return ret;
}

char *cgi_recv_body(SSL *s, int len){
  char *ret,buf[32];
  int i=0,j;

  if(len==0){ /* chunked */
    if(cgi_gets(s,buf,32)<0) goto error;
    if((len = strtol(buf,NULL,16)) <= 0)
      len = 8192; /* len is undefined */
  }

  /* allocate memory */
  if((ret=(char*)AIMALLOC(len+2))==NULL) goto error;
  memset(ret,0,len+2);

  /* receive date */
  while(i<len){
    if((j=SSL_read(s,(char*)&ret[i],len-i))<0) goto error;
    i+=j;

    if(len==8192) break;
  }

  /* fprintf(stderr,"len=%d\nbody=%s",len,ret); */

  return ret;
error:
  if(ret) AIFREE(ret);
  return NULL;
}
