/* aireg_util.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_asn1.h>

#include "aienr_cgi.h"

#ifndef __WITH_LANG
#define __WITH_LANG	0
#endif /* FIXME */

#define SCHMAX 8
#define LDAP_CAENTRY_CLASS "certificationAuthority"
#define LDAP_USERENTRY_CLASS "inetOrgPerson"

/*------------------------------------------------------
  check LDAP tree environment and create CA entry
------------------------------------------------------*/
int aira_add_caaux(LDAP *ld, CertDN *dn, char *attrs[]){
  char *caclass = LDAP_CAENTRY_CLASS; 
  char buf[512];
  LDAPMod *mod[SCHMAX*2+2];
  int i,j,ok=-1;

  memset(mod,0,sizeof(LDAPMod*)*(SCHMAX*2+2));

  /* get entry string */
  if(certdn2ldapdn(dn,buf,512)) goto done;

  /* add objectclass */
  if((mod[0]=get_ldapmov(LDAP_MOD_ADD,"objectClass",caclass))==NULL) goto done;

  /* attribute -- add dummy value */
  for(j=0; attrs[j]; j++){
    if((mod[1+j]=get_ldapmov(LDAP_MOD_ADD,attrs[j],"."))==NULL) goto done;
  }

  ok = ldap_modify_s(ld,buf,mod);
done:
  for(i=0;mod[i];i++) ldapmod_free(mod[i]);
  return ok;
}

int aira_add_entry(LDAP *ld, CertDN *dn, char *mch){
  char *objs[]={"country","locality","locality","organization","organizationalunit"};
  char *attrs[]={"c","st","l","o","ou"};
  char *uclass[]={LDAP_USERENTRY_CLASS, NULL };
  char *uattr[]={"cn", "sn", NULL };
  char buf[512];
  LDAPMod *mod[SCHMAX*2+2];
  CertDN mdn;
  int swap,dnum,i,j,ok=-1;

  dnum = dn->num;
  swap = (dn->rdn[0].tagoid == OBJ_DIR_DC)||
    ((dn->rdn[0].tagoid < dn->rdn[dnum-1].tagoid)&&(dn->rdn[dnum-1].tagoid != OBJ_DIR_DC));

  memset(mod,0,sizeof(LDAPMod*)*(SCHMAX*2+2));
  cert_dn_init(&mdn);

  /* make CertDN data from ldap dn string
   * *caution* this function breaks "mch" char
   */
  if(ldapdn2certdn(mch, &mdn)) goto done;

  /* find out which entry should be created */
  i = (swap)?(dnum-2):(1);
  do{
    if(dn->rdn[i].tagoid == mdn.rdn[0].tagoid){
      if((dn->rdn[i].derform==ASN1_UTF8STRING)||(mdn.rdn[0].derform==ASN1_UTF8STRING))
	j = strcmp(dn->rdn[i].tag,mdn.rdn[0].tag);
      else
	j = igcase_strcmp(dn->rdn[i].tag,mdn.rdn[0].tag);

      if(!j){
	i += (swap)?(1):(-1);
	/* move mdn rdns first */
	for(j=mdn.num-1; j>=0; j--)
	  memcpy(&mdn.rdn[j+1],&mdn.rdn[j],sizeof(struct cert_rdn));
	mdn.rdn[0].derform = dn->rdn[i].derform;
	mdn.rdn[0].tagoid = dn->rdn[i].tagoid;
	if ((mdn.rdn[0].tag = strdup(dn->rdn[i].tag)) == NULL) goto done;
	mdn.num++;
	break;
      }
    }

    if(swap){
      i--;
      if(i < 0) break;
    }else{
      i++;
      if(i >= dn->num) break;
    }
  }while(1);

  /* get entry string */
  if(certdn2ldapdn(&mdn,buf,512)) goto done;

  /* add entry */
  switch(j=mdn.rdn[0].tagoid){
  case OBJ_DIR_CN:
    /* objectClass */
    for(i=0;uclass[i];i++){
      if((mod[i]=get_ldapmov(LDAP_MOD_ADD,"objectClass",uclass[i]))==NULL) goto done;
    }
    /* attribute */
    for(j=0;uattr[j];j++){
      if((mod[i+j]=get_ldapmov(LDAP_MOD_ADD,uattr[j],mdn.rdn[0].tag))==NULL) goto done;
    }
    break;
  default:
    if((mod[0]=get_ldapmov(LDAP_MOD_ADD,"objectClass",objs[j-OBJ_DIR_C]))==NULL) goto done;
    if((mod[1]=get_ldapmov(LDAP_MOD_ADD,attrs[j-OBJ_DIR_C],mdn.rdn[0].tag))==NULL) goto done;
    break;
  }

  ok = ldap_add_s(ld,buf,mod);
done:
  cert_dn_free(&mdn);
  for(i=0;mod[i];i++) ldapmod_free(mod[i]);
  return ok;
}

int aira_create_ca_entry(LDAP *ld, Cert *ca, char *cadn){
  LDAPMessage *res = NULL;
  char *attrs[] = { "cACertificate;binary", "authorityrevocationlist;binary",
                    "certificaterevocationlist;binary", NULL };
  char filter[128],*mdn=NULL;
  int i,ok=-1;

  /* search entry */
  do{
    /* check "cACertificate", ARL and CRL */

    if((i=ldap_search_s(ld,cadn,LDAP_SCOPE_BASE,NULL,attrs,1,&res)) == LDAP_SUCCESS){
      /* entry exists */
      strcpy(filter,"(objectClass=");
      strcat(filter,LDAP_CAENTRY_CLASS);
      strcat(filter,")");
      ldap_msgfree(res); res=NULL;

      if((i=ldap_search_s(ld,cadn,LDAP_SCOPE_BASE,filter,attrs,0,&res)) != LDAP_SUCCESS)
	goto done;

      if(ldap_count_entries(ld,res) <= 0){
	/* entry & objectClass does not exist */
	/* add CA (auxiliary) */
	if(aira_add_caaux(ld,&ca->subject_dn,attrs)) goto done;
      }else{
	/* entry & objectClass exists */
	break;
      }

    }else if(i==LDAP_NO_SUCH_OBJECT){
      /* entry does not exist */
      if(ldap_parse_result(ld,res,&i,&mdn,NULL,NULL,NULL,0) != LDAP_SUCCESS) goto done;

      if(mdn==NULL) goto done;

      /* add new entry */
      if(aira_add_entry(ld,&ca->subject_dn,mdn)) goto done;
      ldap_memfree(mdn); mdn=NULL;

    }else{
      goto done;
    }
    if(res){ ldap_msgfree(res); res=NULL; }

  }while(1);

  ok = 0;
done:
  if(mdn) ldap_memfree(mdn);
  if(res) ldap_msgfree(res);
  return ok;
}

/* rabase should be bigger than 32 byte */
int aira_check_ldapenv(AiEnrollInfo *info,LDAP *ld, char *rabase, int max){
  LDAPMessage *res = NULL;
  Cert *ca = NULL;
  char path[256],cadn[512];
  int ok=-1;

  /* get ca certificate */
  snprintf(path,256,"..%s%s.cer","/",info->ca.caname);
  if((ca = Cert_read_file(path))==NULL) goto done;

  /* get entry string */
  if(certdn2ldapdn(&ca->subject_dn,cadn,512)) goto done;

  memset (rabase,0,max);
  strncpy(rabase,"ou=\"RARegistration\", ",30);
  strncat(rabase,cadn,max-30);

  /* check RA Operator base */
  if(ldap_search_s(ld,rabase,LDAP_SCOPE_BASE,NULL,NULL,1,&res) != LDAP_SUCCESS){
    if(ldap_search_s(ld,cadn,LDAP_SCOPE_BASE,NULL,NULL,1,&res) != LDAP_SUCCESS){
      if(aira_create_ca_entry(ld,ca,cadn)) goto done;
    }
    /* add RA base entry */
    if(cgi_ldap_addorg(ld,"RARegistration",cadn,"ou")) goto done;
  }

  ok = 0;
done:
  if(res) ldap_msgfree(res);
  Cert_free(ca);
  return ok;
}

/*------------------------------------------------------
  check registration session id
------------------------------------------------------*/
int aira_check_regsess(AiEnrollInfo *info,LDAP *ld, char *rabase, char *ssnum){
  LDAPMessage *res = NULL;
  char filter[128];
  int ok=-1;

  /* search entry */
  snprintf(filter,126,"(%s=%s)",info->ld.lidattr,ssnum);

  if(ldap_search_s(ld,rabase,LDAP_SCOPE_SUBTREE,filter,NULL,0,&res) != LDAP_SUCCESS) goto done; 

  /* if entry found, session already accepted */
  if(ldap_count_entries(ld,res) > 0) goto done;

  ok = 0;
done:
  if(res) ldap_msgfree(res);
  return ok;
}

/* base should be bigger than 32 byte */
int aira_check_regsess2(AiEnrollInfo *info,LDAP *ld,char *gbase,char *base,int max, char *ssnum)
{
  LDAPMessage *res = NULL;
  char filter[128];
  int ok=-1;

  if(gbase==NULL || base==NULL || ssnum==NULL) goto done;

  /* search base */
  memset (base,0,max);
  strncpy(base,"ou=\"Registration\", ",30);
  strncat(base,gbase,max-30);

  /* search entry */
  snprintf(filter,126,"(%s=%s)",info->ld.lidattr,ssnum);

  /* check group base */
  if(ldap_search_s(ld,base,LDAP_SCOPE_SUBTREE,filter,NULL,0,&res) != LDAP_SUCCESS){
    /* add RA base entry */
    if(cgi_ldap_addorg(ld,"Registration",gbase,"ou")) goto done;

    /* search again */
    if(ldap_search_s(ld,base,LDAP_SCOPE_SUBTREE,filter,NULL,0,&res) != LDAP_SUCCESS) goto done;
  }

  /* if entry found, session already accepted */
  if(ldap_count_entries(ld,res) > 0) goto done;

  ok = 0;
done:
  if(res) ldap_msgfree(res);
  return ok;
}

/*------------------------------------------------------
  cn & email address check
------------------------------------------------------*/
int aira_check_regcn(AiEnrollInfo *info, LDAP *ld, char *gbase, AiEntryInfo *e){
  LDAPMessage *res = NULL;
  char filter[128];
  char *attrs[2] = { "cn", NULL };
  int ok=-1;

  /* search entry */
  if(__WITH_LANG == 1){ /* JP */
    /* code should be UTF8 */
    snprintf(filter,126,"(cn=%s %s)",e->lastName,e->firstName);
  }else{
    snprintf(filter,126,"(cn=%s %s)",e->firstName,e->lastName);
  }

  if(ldap_search_s(ld,gbase,LDAP_SCOPE_SUBTREE,filter,attrs,1,&res) != LDAP_SUCCESS) goto done;
  if(ldap_count_entries(ld,res) > 0) goto done;

  ok = 0;
done:
  if(res) ldap_msgfree(res);
  return ok;
}

int aira_check_regquery(AiEnrollInfo *info, LDAP *ld, char *gbase, char *val){
  LDAPMessage *res = NULL;
  char filter[128];
  char *attrs[2] = { "mail", NULL };
  int ok=-1;

  /* search entry */
  snprintf(filter,126,"(%s=%s)",info->ld.uquattr,val);

  if(ldap_search_s(ld,gbase,LDAP_SCOPE_SUBTREE,filter,attrs,1,&res) != LDAP_SUCCESS) goto done;
  if(ldap_count_entries(ld,res) > 0) goto done;

  ok = 0;
done:
  if(res) ldap_msgfree(res);
  return ok;
}
