/* airareg_op.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_io.h>
#include <aicrypto/ok_rand.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_pem.h>

#include "ok_conf.h"
#include "aienr_cgi.h"

/* HTML Template files */
#define AI_DEFAULT_HTM  "ai_default.html"
#define AI_REGFORM_HTM  "ai_register.html"
#define AI_REGCONF_HTM  "ai_regiconf.html"
#define AI_REGDONE_HTM  "ai_regidone.html"
#define AI_LOGIN_HTM    "ai_login.html"
#define AI_LOGIN2_HTM   "ai_login2.html"
#define AI_RVKCONF_HTM  "ai_rvkconf.html"
#define AI_RVKDONE_HTM  "ai_rvkdone.html"
#define AI_VIEWCACT_HTM "ai_viewcacert.html"
#define AI_UPDPIN_HTM   "ai_updpin.html"
#define AI_UPDPINDONE_HTM "ai_updpindone.html"
#define AI_ERROR_HTM    "ai_error.html"

/* common value */
char path[256], *errmsg = "";


/*----------------------------------------------------------------------
  default : 
----------------------------------------------------------------------*/
char *get_html_name(AiEnrollInfo *info,char *file){
  return cgi_html_path(info,"raregister",file);
}

void aira_print_err(AiEnrollInfo *info,char *err1,char *err2,LCMP *lc){
  char *buf,tmp[256]="";
  off_t sz;

  if((buf=(char*)get_file2buf(get_html_name(info,AI_ERROR_HTM),&sz)) != NULL) {
    AiVList v[] = {
      {"{ERROR1}",(err1)?(err1):("")},
      {"{ERROR2}",(err2)?(err2):("")},
      {"{OKERR}",(OK_get_error())?(OK_get_errstr()):("")},
      {"{LCMPERR}",tmp},
      {NULL,NULL}
    };
    if(lc && lc->op && lc->op->resultCode != LCMP_SUCCESS){
      snprintf(tmp,254,"%s (%d) => %s\n",LCMP_msg2str(lc->op->resultCode),lc->op->resultCode,
               lc->op->resultMsg);
    }
    cgi_print_content(buf,v);

    free(buf);
  }
}

int aira_default(AiEnrollInfo *info){
  char *buf=NULL;
  int ret=-1;
  off_t sz;

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_DEFAULT_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  printf("%s",buf);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_default : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aira_regiform(AiEnrollInfo *info){
  char *buf=NULL;
  int ret=-1;
  off_t sz;

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_REGFORM_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  printf("%s",buf);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_regiform : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

int aira_regiconfirm(AiEnrollInfo *info){
  char *buf=NULL;
  char tmp[64],ss[128];
  char *cknum,*dec=NULL,*imgNum,*cry=NULL;
  AiEntryInfo e;
  int ret=-1;
  off_t sz;

  memset(&e,0,sizeof(AiEntryInfo));

  /* get user input */
  e.firstName= cgi_find_query(&info->qc,"firstName");
  e.lastName = cgi_find_query(&info->qc,"lastName");
  e.mail     = cgi_find_query(&info->qc,"email");
  e.pin      = cgi_find_query(&info->qc,"pin");
  imgNum     = cgi_find_query(&info->qc,"imgNum");

  if(*e.firstName== 0){ errmsg = "post message not found"; goto done; }
  if(*e.lastName == 0){ errmsg = "post message not found"; goto done; }
  if(*e.mail     == 0){ errmsg = "post message not found"; goto done; }
  if(*e.pin      == 0){ errmsg = "post message not found"; goto done; }
  if(*imgNum     == 0){ errmsg = "post message not found"; goto done; }

  e.unit  = cgi_find_query(&info->qc,"unit");
  e.title = cgi_find_query(&info->qc,"title");
  e.tel1  = cgi_find_query(&info->qc,"tel1");
  e.tel2  = cgi_find_query(&info->qc,"tel2");
  e.fax1  = cgi_find_query(&info->qc,"fax1");
  e.fax2  = cgi_find_query(&info->qc,"fax2");
  e.employeeNumber = cgi_find_query(&info->qc,"employeeNumber");
  e.postalCode     = cgi_find_query(&info->qc,"postalCode");
  e.postalAddress  = cgi_find_query(&info->qc,"postalAddress");
  e.labeledUri     = cgi_find_query(&info->qc,"labeledUri");
  e.description    = cgi_find_query(&info->qc,"description");

  /* replace html symbol tags '<' or '>' to space char */
  ai_symbol_esc_sp(e.firstName);
  ai_symbol_esc_sp(e.lastName);
  ai_symbol_esc_sp(e.mail);
  ai_symbol_esc_sp(e.unit);
  ai_symbol_esc_sp(e.title);
  ai_symbol_esc_sp(e.tel1);
  ai_symbol_esc_sp(e.tel2);
  ai_symbol_esc_sp(e.fax1);
  ai_symbol_esc_sp(e.fax2);
  ai_symbol_esc_sp(e.employeeNumber);
  ai_symbol_esc_sp(e.postalCode);
  ai_symbol_esc_sp(e.postalAddress);
  ai_symbol_esc_sp(e.labeledUri);
  ai_symbol_esc_sp(e.description);
  
  /* PNG image number (cookie encrypted) */
  cknum = cgi_find_cookie(&info->qc,"AIMGGENUM");
  if((dec = aica_decry_passwd(cknum))==NULL) goto done;

  /* check number */
  if(strncmp(dec,imgNum,5)||strlen(dec)<15){
    errmsg = "image number authentication faild. Please input again"; goto done;
  }

  /* create session information */
  memset (ss,0,128);
  strncpy(ss,dec,15);
  strncat(ss,e.pin,64);
  if((cry = aica_encry_passwd(ss))==NULL) goto done;

  memset(tmp,'*',64); tmp[strlen(e.pin)] = 0;

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_REGCONF_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{firstName}",e.firstName},
      {"{lastName}",e.lastName},
      {"{email}",e.mail},
      {"{pin}",tmp},
      {"{unit}",e.unit},
      {"{title}",e.title},
      {"{tel1}",e.tel1},
      {"{tel2}",e.tel2},
      {"{fax1}",e.fax1},
      {"{fax2}",e.fax2},
      {"{employeeNumber}",e.employeeNumber},
      {"{postalCode}",e.postalCode},
      {"{postalAddress}",e.postalAddress},
      {"{labeledUri}",e.labeledUri},
      {"{description}",e.description},
      {"{session}",cry},
      {"{UPD}","0"},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_regiconfirm : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  if(dec) free(dec);
  if(cry) free(cry);
  return ret;
}

int aira_regidone(AiEnrollInfo *info){
  LDAP *ld = NULL;
  time_t ti;
  struct tm *stm;
  char *buf=NULL;
  char tmp[128],cnt[32],*tmstr;
  char *cknum,*dec1=NULL,*dec2=NULL,*ss,*cn;
  char rabase[512],hash[64],lang[8];
  char *m1,*m2;
  AiEntryInfo e;
  int i,j,upd,ret=-1;
  off_t sz;

  memset(&e,0,sizeof(AiEntryInfo));
  
  /* get user input */
  e.firstName= cgi_find_query(&info->qc,"firstName");
  e.lastName = cgi_find_query(&info->qc,"lastName");
  e.mail     = cgi_find_query(&info->qc,"email");

  if(*e.firstName== 0){ errmsg = "post message not found"; goto done; }
  if(*e.lastName == 0){ errmsg = "post message not found"; goto done; }
  if(*e.mail     == 0){ errmsg = "post message not found"; goto done; }

  cn  = cgi_find_query(&info->qc,"CN");
  upd = atoi(cgi_find_query(&info->qc,"UPD"));

  e.unit  = cgi_find_query(&info->qc,"unit");
  e.title = cgi_find_query(&info->qc,"title");
  e.tel1  = cgi_find_query(&info->qc,"tel1");
  e.tel2  = cgi_find_query(&info->qc,"tel2");
  e.fax1  = cgi_find_query(&info->qc,"fax1");
  e.fax2  = cgi_find_query(&info->qc,"fax2");
  e.employeeNumber = cgi_find_query(&info->qc,"employeeNumber");
  e.postalCode     = cgi_find_query(&info->qc,"postalCode");
  e.postalAddress  = cgi_find_query(&info->qc,"postalAddress");
  e.labeledUri     = cgi_find_query(&info->qc,"labeledUri");
  e.description    = cgi_find_query(&info->qc,"description");
  ss  = cgi_find_query(&info->qc,"session");
  cgi_get_lang(info,lang);
  e.plang = lang;

  /* replace html symbol tags '<' or '>' to space char */
  ai_symbol_esc_sp(e.firstName);
  ai_symbol_esc_sp(e.lastName);
  ai_symbol_esc_sp(e.mail);
  ai_symbol_esc_sp(e.unit);
  ai_symbol_esc_sp(e.title);
  ai_symbol_esc_sp(e.tel1);
  ai_symbol_esc_sp(e.tel2);
  ai_symbol_esc_sp(e.fax1);
  ai_symbol_esc_sp(e.fax2);
  ai_symbol_esc_sp(e.employeeNumber);
  ai_symbol_esc_sp(e.postalCode);
  ai_symbol_esc_sp(e.postalAddress);
  ai_symbol_esc_sp(e.labeledUri);
  ai_symbol_esc_sp(e.description);

  /* PNG image number (cookie encrypted) */
  cknum = cgi_find_cookie(&info->qc,"AIMGGENUM");
  if((dec1 = aica_decry_passwd(cknum))==NULL) goto done;
  if((dec2 = aica_decry_passwd(ss))==NULL) goto done;
  cgi_gen_hash(&dec2[15],hash);
  e.pin = hash;

  /* check number */
  i = strlen(dec1);
  j = strlen(dec2);
  if(!strncmp(cknum,ss,15)||strncmp(dec1,dec2,15)||(i==j)||(i<15)){
    errmsg = "session check faild. Please input again"; goto done;
  }

  /* get acccept date */
  time(&ti);
  stm=(struct tm*)gmtime(&ti);
  if((tmstr=stm2str(stm,3))==NULL){
    errmsg = "can not get accept date."; goto done;
  }

  if(upd==1){
    /* set WAIT_UPDATE flag */
    if(aira_change_entryst(info,cn,"WAIT_UPDATE",NULL,1,0)){
      errmsg = "cannot update RA operator entry status"; goto done;
    }
    strncpy(cnt,cn,30);
    m1 = "rareg_accept_updop1.txt";
    m2 = "rareg_accept_updop2.txt";
  }else{
    /*---------- do registration -----------*/
    /* check LDAP environment & search session id */
    if((ld=ldap_init(info->ld.ldaphost,info->ld.ldapport))==NULL){
      errmsg = "ldap init error"; goto done;
    }
    if(cgi_ldap_bind(ld,info->ld.ldapadmin,info->ld.ldapadminpwd,info->ld.ldapbind) 
       != LDAP_SUCCESS){
      errmsg = "ldap bind error"; goto done;
    }
    if(aira_check_ldapenv(info,ld,rabase,512)){
      errmsg = "ldap environment check error. can not create CA entry."; goto done;
    }
    if(aira_check_regsess(info,ld,rabase,cknum)){
      errmsg = "bad session number. registration already done."; goto done;
    }
    
    /* get accept ID */
    if((i = aira_get_count(info,"rareg"))<0){
      errmsg = "can not get accept number. check counter file."; goto done;
    }
    sprintf(cnt,"RAREG%.4d",i);

    if(aira_add_regentry(info,ld,rabase,&e,cnt,tmstr,cknum)){
      errmsg = "can not create registration entry on the LDAP server."; goto done;
    }
    m1 = "rareg_accept_op1.txt";
    m2 = "rareg_accept_op2.txt";
  }

  if(*info->ml.smtphost && *info->ml.admemail){
    /* send to admin */
    cgi_get_deflang(info->ml.lang);
    if(aira_sendmail_sys2adm(&info->ml,&e,info->ml.admemail,cnt,m1)){
      sprintf(tmp,"cannot send a email to admin (accID=%s) << %s",cnt,CA_get_errstr());
      ENCERRLOG(info->ca.caname,tmp);
    }
    /* send to ra operator */
    cgi_get_lang(info,info->ml.lang);
    if(aira_sendmail_adm2usr(&info->ml,&e,info->ml.admemail,cnt,m2)){
      sprintf(tmp,"cannot send a email to ra operator (accID=%s) << %s",cnt,CA_get_errstr());
      ENCERRLOG(info->ca.caname,tmp);
    }
  }

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_REGDONE_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{firstName}",e.firstName},
      {"{lastName}",e.lastName},
      {"{email}",e.mail},
      {"{unit}",e.unit},
      {"{title}",e.title},
      {"{tel1}",e.tel1},
      {"{tel2}",e.tel2},
      {"{fax1}",e.fax1},
      {"{fax2}",e.fax2},
      {"{employeeNumber}",e.employeeNumber},
      {"{postalCode}",e.postalCode},
      {"{postalAddress}",e.postalAddress},
      {"{labeledUri}",e.labeledUri},
      {"{description}",e.description},
      {"{acceptnum}",cnt},
      {"{acceptdate}",tmstr},
      {NULL,NULL}
    };

    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_regidone : System error!!",errmsg,NULL);
  }
  if(ld) ldap_unbind(ld);
  if(buf) free(buf);
  if(dec1) free(dec1);
  if(dec2) free(dec2);
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aira_login(AiEnrollInfo *info){
  char *buf=NULL;
  char *nx,*fm,*cn;
  int ret=-1;
  off_t sz;

  cn = cgi_find_query(&info->qc,"CN");
  fm = cgi_find_query(&info->qc,"Fm");
  nx = cgi_find_query(&info->qc,"Nx");

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_LOGIN_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{CN}",cn},
      {"{FROM}",fm},
      {"{NEXT}",nx},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }
  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_login : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

int aira_login2(AiEnrollInfo *info){
  char *nx,*fm,*buf=NULL;
  int ret=-1;
  off_t sz;

  fm = cgi_find_query(&info->qc,"Fm");
  nx = cgi_find_query(&info->qc,"Nx");

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_LOGIN2_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{FROM}",fm},
      {"{NEXT}",nx},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }
  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_login2 : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aira_certmanage(AiEnrollInfo *info){
  char *nx = cgi_find_query(&info->qc,"Nx");

  if(!strncmp(nx,"Rvk",16)){
    if(aira_rvkconfirm(info)) return -1;
  }else{
    if(aira_updconfirm(info)) return -1;
  }
  return 0;
}

int aira_updconfirm(AiEnrollInfo *info){
  AiEntryInfo *e = NULL;
  char *cn,*fm,*pin,*buf=NULL;
  char tmp[64],ss[128],phash[64];
  char *bs=NULL,*cry=NULL;
  unsigned char rnd[16];
  int i,ret=-1;
  off_t sz;

  cn = cgi_find_query(&info->qc,"CN");
  fm = cgi_find_query(&info->qc,"Fm");
  pin= cgi_find_query(&info->qc,"ENPin");

  /* get RAOP entry */
  if((e=cgi_entryinfo_new())==NULL){
    errmsg = "entryinfo memory allocate error"; goto done;
  }
  if(aira_get_raentry(info,e,cn,1,0)){
    errmsg = "can not get request info from LDAP server"; goto done;
  }

  /* check challenge pin */
  cgi_gen_hash(pin,phash);
  if(strcmp(phash,e->pin)){
    errmsg = "invalid challenge PIN. Please input again."; goto done;
  }
  if(strcmp("ACTIVE",e->st)&&strcmp("REVOKED",e->st)){
    errmsg = "bad status. cannot request update."; goto done;
  }


  /* generate random number text and set cookie (dummy) */
  RAND_bytes(rnd,16);
  i = ((rnd[0]<<16)|(rnd[1]<<8)|(rnd[2])) % 100000;
  sprintf(tmp,"%.5d%.2x%.2x%.2x%.2x%.2x",i,rnd[3],rnd[6],rnd[7],rnd[8],rnd[9]);
  if((cry = aica_encry_passwd(tmp))==NULL) goto done;
  if((bs = (char*)ai_escape_url(cry,strlen(cry)))==NULL) goto done;
  free(cry); cry=NULL;

  printf("Set-Cookie: AIMGGENUM=%s\n", bs);

  /* create session information */
  memset (ss,0,128);
  strncpy(ss,tmp,15);
  strncat(ss,pin,64);
  if((cry = aica_encry_passwd(ss))==NULL) goto done;

  memset(tmp,'*',64); tmp[strlen(pin)] = 0;

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_REGCONF_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{firstName}",{e->firstName}},
      {"{lastName}",{e->lastName}},
      {"{email}",{e->mail}},
      {"{pin}",{tmp}},
      {"{unit}",{e->unit}},
      {"{title}",{e->title}},
      {"{tel1}",{e->tel1}},
      {"{tel2}",{e->tel2}},
      {"{fax1}",{e->fax1}},
      {"{fax2}",{e->fax2}},
      {"{employeeNumber}",{e->employeeNumber}},
      {"{postalCode}",{e->postalCode}},
      {"{postalAddress}",{e->postalAddress}},
      {"{labeledUri}",{e->labeledUri}},
      {"{description}",{e->description}},
      {"{session}",{cry}},
      {"{UPD}",{"1"}},
      {"{CN}",{cn}},
      {NULL,{NULL}}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_updconfirm : System error!!",errmsg,NULL);
  }
  cgi_entryinfo_free(e);
  if(buf) free(buf);
  if(cry) free(cry);
  if(bs) free(bs);
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aira_rvkconfirm(AiEnrollInfo *info){
  Cert *ct = NULL;
  AiEntryInfo *e = NULL;
  char *p,*cn,*fm,*pin,*buf=NULL,*cry=NULL,*csbj=NULL;
  char sn[64],start[32],end[32];
  char phash[64];
  int ret=-1;
  off_t sz;

  cn = cgi_find_query(&info->qc,"CN");
  fm = cgi_find_query(&info->qc,"Fm");
  pin= cgi_find_query(&info->qc,"ENPin");

  /* get RAOP entry */
  if((e=cgi_entryinfo_new())==NULL){
    errmsg = "entryinfo memory allocate error"; goto done;
  }
  if(aira_get_raentry(info,e,cn,1,0)){
    errmsg = "can not get request info from LDAP server"; goto done;
  }

  /* check challenge pin */
  cgi_gen_hash(pin,phash);
  if(strcmp(phash,e->pin)){
    errmsg = "invalid challenge PIN. Please input again."; goto done;
  }
  if(strcmp("ACTIVE",e->st)){
    errmsg = "bad status. cannot request revocation."; goto done;
  }

  /* get ra oprator certificate */
  if((ct = ASN1_read_cert(e->ucert))==NULL){
    errmsg = "cannot get RA operator certificate"; goto done;
  }
  e->ucert = NULL; csbj = ct->subject;
  snprintf(sn,14,"%d",ct->serialNumber);
  if((p = stm2str(&ct->time.notBefore,3)) != NULL) strncpy(start,p,30);
  if((p = stm2str(&ct->time.notAfter,3)) != NULL) strncpy(end,p,30);

  /* pin session information */
  if((cry = aica_encry_passwd(pin))==NULL) goto done;

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_RVKCONF_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{sbj}",csbj},
      {"{SERIAL}",sn},
      {"{notbefore}",start},
      {"{notafter}",end},
      {"{FROM}",fm},
      {"{CN}",cn},
      {"{ENPin}",cry},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_rvkconfirm : System error!!",errmsg,NULL);
  }
  cgi_entryinfo_free(e);
  Cert_free(ct);
  if(buf) free(buf);
  if(cry) free(cry);
  return ret;
}

int aira_rvkdone(AiEnrollInfo *info){
  Cert *ct = NULL;
  AiEntryInfo *e = NULL;
  time_t ti;
  struct tm *stm;
  char *p,*tmstr;
  char *rs,*cn,*fm,*pin,*buf=NULL,*dec=NULL,*csbj=NULL;
  char sn[64],start[32],end[32],tmp[256];
  int ret=-1;
  off_t sz;

  cn = cgi_find_query(&info->qc,"CN");
  fm = cgi_find_query(&info->qc,"Fm");
  rs = cgi_find_query(&info->qc,"Reason");
  pin= cgi_find_query(&info->qc,"ENPin");

  if((dec = aica_decry_passwd(pin))==NULL) goto done;

  /* get RAOP entry */
  if((e=cgi_entryinfo_new())==NULL){
    errmsg = "entryinfo memory allocate error"; goto done;
  }
  if(aira_get_raentry(info,e,cn,1,0)){
    errmsg = "can not get request info from LDAP server"; goto done;
  }

  /* check challenge pin */
  cgi_gen_hash(dec,tmp);
  if(strcmp(tmp,e->pin)){
    errmsg = "invalid challenge PIN. Please input again."; goto done;
  }

  /* get ra oprator certificate */
  if((ct = ASN1_read_cert(e->ucert))==NULL){
    errmsg = "cannot get RA operator certificate"; goto done;
  }
  e->ucert = NULL; csbj = ct->subject;
  snprintf(sn,14,"%d",ct->serialNumber);
  if((p = stm2str(&ct->time.notBefore,3)) != NULL) strncpy(start,p,30);
  if((p = stm2str(&ct->time.notAfter,3)) != NULL) strncpy(end,p,30);

  /* get acccept date */
  time(&ti);
  stm=(struct tm*)gmtime(&ti);
  if((tmstr=stm2str(stm,3))==NULL){
    errmsg = "can not get accept date."; goto done;
  }

  /* set WAIT_REVOKE flag */
  if(aira_change_entryst(info,cn,"WAIT_REVOKE",NULL,1,0)){
    errmsg = "cannot update RA operator entry status"; goto done;
  }

  /* send email */
  if(*info->ml.smtphost && *info->ml.admemail){
    /* send to admin */
    cgi_get_deflang(info->ml.lang);
    if(aira_sendmail_sys2adm(&info->ml,e,info->ml.admemail,cn,"rareg_accept_rvkop1.txt")){
      sprintf(tmp,"cannot send a email to admin (accID=%s) << %s",cn,CA_get_errstr());
      ENCERRLOG(info->ca.caname,tmp);
    }
    /* send to ra operator */
    cgi_get_lang(info,info->ml.lang);
    if(aira_sendmail_adm2usr(&info->ml,e,info->ml.admemail,cn,"rareg_accept_rvkop2.txt")){
      sprintf(tmp,"cannot send a email to ra operator (accID=%s) << %s",cn,CA_get_errstr());
      ENCERRLOG(info->ca.caname,tmp);
    }
  }

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_RVKDONE_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{sbj}",csbj},
      {"{SERIAL}",sn},
      {"{notbefore}",start},
      {"{notafter}",end},
      {"{acceptdate}",tmstr},
      {"{Reason}",rs},
      {"{FROM}",fm},
      {"{CN}",cn},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_rvkdone : System error!!",errmsg,NULL);
  }
  cgi_entryinfo_free(e);
  if(buf) free(buf);
  if(dec) free(dec);
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aira_updatepin(AiEnrollInfo *info){
  char *buf=NULL;
  int ret=-1;
  off_t sz;

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_UPDPIN_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  printf("%s",buf);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_updatepin : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

int aira_updpindone(AiEnrollInfo *info){
  char *cn,*oldpin,*newpin,*buf=NULL;
  int i,ret=-1;
  off_t sz;

  cn = cgi_find_query(&info->qc,"CN");
  oldpin = cgi_find_query(&info->qc,"ENPin");
  newpin = cgi_find_query(&info->qc,"ENNewPin");

  /* update user entry */
  if((i=aira_change_pin(info,cn,oldpin,newpin,1,0))<0){
    errmsg = "system error.cannot change pin."; goto done;
  }
  if(i>0){
    errmsg = "bad old password. please input again"; goto done;
  }

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_UPDPINDONE_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  printf("%s",buf);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_updatepin : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aira_viewcacert(AiEnrollInfo *info){
  Cert *ca = NULL;
  char *cp,*p,*t;
  char *buf=NULL;
  int ret=-1;
  off_t sz;

  /* get ca certificate */
  snprintf(path,256,"..%s%s.cer","/",info->ca.caname);
  if((ca = Cert_read_file(path))==NULL) goto done;

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_VIEWCACT_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  cp = buf;
  do{
    if((t=strstr(cp,"${"))==NULL){ printf("%s",cp); break; }
    *t=0; t++;
    if(!memcmp(t,"{CERTTEXT}",10)){
      printf("%s",cp); CERT_PRINT(ca); cp=t+10; continue;
    }else{
      p = "$";
    }
    printf("%s",cp);
    printf("%s",p);
    cp = t;
  }while(cp);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_viewcacert : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  Cert_free(ca);
  return ret;
}

int aira_getcacert(AiEnrollInfo *info){
  Cert *ca=NULL;
  char *p1=NULL;
  int ret=-1;

  /* get ca certificate */
  snprintf(path,256,"..%s%s.cer","/",info->ca.caname);
  if((ca = Cert_read_file(path))==NULL) goto done;

  /*---------- print out text -------------*/
  if((p1 = PEM_write_cert_buf(ca))==NULL) goto done;

  printf("Content-type: application/x-x509-ca-cert\n");
  printf("Content-Disposition: attachment; filename=\"%s.cer\"\n",info->ca.caname);
  printf("Content-length: %zu\n\n",strlen(p1));

  printf("%s",p1);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_getcacert : System error!!",errmsg,NULL);
  }
  Cert_free(ca);
  if(p1) free(p1);
  return ret;
}

int aira_getcrl(AiEnrollInfo *info){
  CRL *crl=NULL;
  char *p1=NULL;
  char path[256];
  int ret=-1;

  /* get ca certificate */
#ifdef __WINDOWS__
  strncpy(path,"out-CRL-All.crl",256);
#else
  strncpy(path,"../out/out-CRL-All.crl",256);
#endif
  if((crl = CRL_read_file(path))==NULL){
    errmsg = "cannot open CRL file. please start CRL Publisher"; goto done;
  }

  /*---------- print out text -------------*/
  if((p1 = PEM_write_crl_buf(crl))==NULL) goto done;

  printf("Content-type: application/x-x509-crl\n");
  printf("Content-Disposition: attachment; filename=\"%s.crl\"\n",info->ca.caname);
  printf("Content-length: %zu\n\n",strlen(p1));

  printf("%s",p1);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_getcrl : System error!!",errmsg,NULL);
  }
  CRL_free(crl);
  if(p1) free(p1);
  return ret;
}
