/* airae_op.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_io.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_base64.h>
#include <aicrypto/ok_pem.h>

#include "aienr_cgi.h"

/* HTML Template files */
#define AI_ERROR_HTM            "ai_error.html"
#define AI_SENDCSR_FORM_HTM     "ai_sendcsr_form.html"
#define AI_SENDCSR_FORM_IE7_HTM "ai_sendcsr_form_ie7.html"
#define AI_SENDCSR_FORM_MZ_HTM  "ai_sendcsr_form_mz.html"
#define AI_ACCEPT_FORM_HTM      "ai_accept_form.html"
#define AI_ACCEPT_FORM_IE7_HTM  "ai_accept_form_ie7.html"
#define AI_DONE_FORM_HTM        "ai_done_form.html"
#define AI_LOGIN_HTM            "ai_login.html"


/*----------------------------------------------------------------------
  login : user login operation
----------------------------------------------------------------------*/
char *get_html_name(AiEnrollInfo *info,char *file){
  return cgi_html_path(info,"raenroll",file);
}

int airae_check_license(AiEnrollInfo *info, char *license, char *pin, AiEntryInfo *e){
  LDAPMessage *res=NULL,*ent=NULL;
  LDAP *ld = NULL;
  char *attrs[] = {"cn","mail",NULL};
  char cadn[512],hash[64],phash[64],filter[256];
  char **vals=NULL,*at=NULL;
  void *ptr=NULL; /* BerElement */
  int i,j,ret = -1;

  cgi_gen_hash(license,hash);
  if(pin){
    cgi_gen_hash(pin,phash);
    sprintf(filter,"(&(%s=%s)(%s=%s))",info->ld.lidattr,hash,info->ld.pinattr,phash);
  }else{
    sprintf(filter,"(%s=%s)",info->ld.lidattr,hash);
  }
  /* get CA DN */
  if(aira_get_cadn(info,cadn,512)) goto done;

  /* connect ldap server */
  if((ld = ldap_init(info->ld.ldaphost,info->ld.ldapport))==NULL) goto done;

  if(cgi_ldap_bind(ld,info->ld.ldapadmin,info->ld.ldapadminpwd,info->ld.ldapbind) != LDAP_SUCCESS)
    goto done;

  if(ldap_search_s(ld,cadn,LDAP_SCOPE_ONELEVEL,filter,attrs,0,&res) != LDAP_SUCCESS) goto done;

  if((ent = ldap_first_entry(ld, res)) == NULL) goto done;

  if((at=ldap_first_attribute(ld,ent,(BERELEMENT**)&ptr))==NULL) goto done;
  while(at){
    struct str_dups{
      char *cmp;
      char **dst[2];
    } v[] = {
      {"cn",&e->dn,&e->cn},
      {"mail",&e->mail,NULL},
      {"ou",&e->unit,NULL},
      {NULL,NULL,NULL}
    };

    vals = ldap_get_values(ld,ent,at);
    for(i=0; v[i].cmp; i++){
      if(at && !igcase_strcmp(at,v[i].cmp)){
        for(j=0; j<2; j++){
          if(v[i].dst[j] && vals[j]){
		  if ((*v[i].dst[j] = strdup(vals[j])) == NULL) goto done; /* code should be UTF8 */
          }
        }
        break;
      }
    }

    if(at){ ldap_memfree(at); at = NULL; }
    if(vals){ ldap_value_free(vals); vals = NULL; }

    at = ldap_next_attribute(ld,ent,ptr);
  }

  ret = 0;
done:
  if(ld) ldap_unbind(ld);
  if(res) ldap_msgfree(res);
  if(ptr) LDAP_BER_FREE(ptr,0);
  return ret;
}

int airae_login(AiEnrollInfo *info){
  AiEntryInfo *e = NULL;
  char *html,*buf=NULL,*err="";
  char *lid,*keyid,*sid,*fm,*pin,*apv;
  char path[256];
  int i,ie7=0,ret=-1;
  off_t sz;

  lid   = cgi_find_query(&info->qc,"ENLoginName");
  pin   = cgi_find_query(&info->qc,"ENPin");
  fm    = cgi_find_query(&info->qc,"Fm");
  apv   = cgi_find_query(&info->qc,"appVersion");
  keyid = cgi_find_cookie(&info->qc,"RAKEYID");
  sid   = cgi_find_cookie(&info->qc,"RASessionID");

  if(*lid == 0){ err = "license ID is empty"; goto done; }
  if(pin == NULL){ pin = ""; }
  if(strstr(apv,"MSIE 7.0")){
	if(atof(strstr(apv,"Windows NT")+10) >= 6.0){
		ie7 = 1; /* IE7.0 and VISTA */
	}
  } 

  if((e=cgi_entryinfo_new())==NULL){
    err = "entryinfo memory allocate error"; goto done;
  }
  html = (ie7)?(AI_SENDCSR_FORM_IE7_HTM):(AI_SENDCSR_FORM_HTM);

  switch(info->authmode){
  case AE_AUTH_LICE:
    if(airae_check_license(info,lid,NULL,e)){
      err = "check_license error"; goto done;
    }
    break;
  case AE_AUTH_LCHA:
    /* load session file */
    if(info->list==NULL && cgi_load_session(info->sespath,&info->list)){
      err = "cannot load session file"; goto done;
    }
    /* check session info */
    i = cgi_check_sessiongrp(info->list,keyid,sid,"$$RAADMIN$$");

    if(i && *pin==0){
      /* operator side & first time access. show pin input page */
      html = AI_LOGIN_HTM;
    }else{
      if(airae_check_license(info,lid,(i==0)?(NULL):(pin),e)){
	err = "check_license & pin error"; goto done;
      }
    }
    break;
  default:
    err = "auth mode error"; goto done;
  }

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,html),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{GROUPNAMES}","Operators"},
      {"{CN0}",e->dn},
      {"{CN1}",e->cn},
      {"{EMAIL}",e->mail},
      {"{LID}",lid},
      {"{FROM}",fm},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  /* success - output log */
  snprintf(path,254,"%s : success to login",lid);
  ENCACCLOG(info->ca.caname,path);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    airae_print_err(info,"airae_login : System error!!",err,NULL);
  }
  cgi_entryinfo_free(e);
  if(buf) free(buf);
  return ret;
}

void airae_print_err(AiEnrollInfo *info,char *err1,char *err2,LCMP *lc){
  char *buf,tmp[256]="";
  off_t sz;

  if((buf=(char*)get_file2buf(get_html_name(info,AI_ERROR_HTM),&sz)) != NULL) {
    AiVList v[] = {
      {"{ERROR1}",(err1)?(err1):("")},
      {"{ERROR2}",(err2)?(err2):("")},
      {"{OKERR}",(OK_get_error())?(OK_get_errstr()):("")},
      {"{LCMPERR}",tmp},
      {NULL,NULL}
    };
    if(lc && lc->op && lc->op->resultCode != LCMP_SUCCESS){
      snprintf(tmp,254,"%s (%d) => %s\n",LCMP_msg2str(lc->op->resultCode),lc->op->resultCode,
               lc->op->resultMsg);
    }
    cgi_print_content(buf,v);

    free(buf);
  }
}

/*----------------------------------------------------------------------
        send csr : send certificate request operation
----------------------------------------------------------------------*/
int airae_ldap_getsubject(AiEnrollInfo *info, LDAP *ld, char *lid, CertTemplate *ctt,
			  char *cn /*out*/ , char *email /*out*/){
  LDAPMessage *res=NULL, *ent;
  char *dn=NULL;
  char *attrs[] = {"cn","mail",NULL};
  char cadn[512],hash[64],filter[256];
  char *t1=NULL,*t2=NULL;
  int ret = -1;

  cgi_gen_hash(lid,hash);
  sprintf(filter,"(%s=%s)", info->ld.lidattr, hash);

  /* get CA DN */
  if(aira_get_cadn(info,cadn,512)) goto done;

  /* search ldap */
  if(ldap_search_s(ld,cadn,LDAP_SCOPE_ONELEVEL,filter,attrs,0,&res) != LDAP_SUCCESS) goto done;

  if((ent = ldap_first_entry(ld, res))==NULL) goto done;

  if((dn = ldap_get_dn(ld, ent))==NULL) goto done;

  if(ldapdn2certdn(dn, &ctt->subject)) goto done;

  /* return cn & email value */
  {
    AiLAttr v[] = {
      {"cn",&t1,NULL},
      {"mail",&t2,NULL},
      {NULL,NULL,NULL}
    };
    if(cgi_ldap_attrvals(ld,ent,v)) goto done;
    strncpy(cn,t1,62);
    strncpy(email,t2,62);
  }

  ret = 0;
done:
  if(t1) free(t1);
  if(t2) free(t2);
  if(dn) ldap_memfree(dn);
  if(res) ldap_msgfree(res);
  return ret;
}

int airae_ldap_addcert(AiEnrollInfo *info, LDAP *ld, char *lid, Cert *ct){
  char dn[512],hash[64],filter[256];
  char *st="ACTIVE",*targetdn=NULL;
  LDAPMessage *res=NULL, *ent=NULL;
  int len,ret = -1;

  ASN1_skip_(ct->der,&len);
  cgi_gen_hash(lid,hash);

  snprintf(filter,256,"(%s=%s)",info->ld.lidattr,hash);

  /* get CA DN */
  if(aira_get_cadn(info,dn,512)) goto done;

  /* get entry string */
  {
    if(ldap_search_s(ld,dn,LDAP_SCOPE_SUBTREE,filter,NULL,0,&res) != LDAP_SUCCESS) goto done;

    if(ldap_count_entries(ld,res)!=1) goto done;
    if((ent=ldap_first_entry(ld,res))==NULL) goto done;

    targetdn=ldap_get_dn(ld,ent);
    strncpy(dn,targetdn,512);
  }


  /* modify add a certificate (keep all issued certificate) 
   * userSMIMECertificate : OCTETSTRING (no ;binary option)
   */
  /*if(cgi_mod_attr_bin(ld,dn,LDAP_MOD_ADD|LDAP_MOD_BVALUES,"userSMIMECertificate",
		      ct->der,len) != LDAP_SUCCESS)
    goto done;
  */
  /* modify replace a certificate (keep latest certificate) 
   * userCertificate : Certificate (need ;binary option)
   */
  if(cgi_mod_attr_bin(ld,dn,LDAP_MOD_REPLACE|LDAP_MOD_BVALUES,"userCertificate;binary",
		      ct->der,len) != LDAP_SUCCESS){
    /* replace failed. retry to modify_add */
    if(cgi_mod_attr_bin(ld,dn,LDAP_MOD_ADD|LDAP_MOD_BVALUES,"userCertificate;binary",
			ct->der,len) != LDAP_SUCCESS){
      goto done;
    }
  }

  /* remove license ID */
  if(cgi_mod_attr(ld,dn,LDAP_MOD_DELETE,info->ld.lidattr,hash) != LDAP_SUCCESS)
    goto done;

  /* replace 'st' operator status */
  if(cgi_mod_attr(ld,dn,LDAP_MOD_REPLACE,"st",st)!= LDAP_SUCCESS)
    goto done;

  ret = 0;
done:
  if(res) ldap_msgfree(res);
  if(targetdn) ldap_memfree(targetdn);
  return ret;
}

int airae_send_csr(AiEnrollInfo *info){
  AiUserInfo uinfo;
  CertTemplate *ctt = NULL;
  Req *csr = NULL;
  Cert *ct = NULL;
  SPKAC *spk = NULL;
  PKCS12 *p12 = NULL;
  LDAP *ld = NULL;
  LCMP *lcmp = NULL;

  char *buf=NULL,cs[256],*err="";
  char *lid,*keyid,*id,*fm,*cp,*sp,*apv;
  char *t,*p1,*p2,*p7c=NULL;
  unsigned char *der=NULL;
  int i,mode,fmt,ie7=0,ret=-1;
  off_t sz;

#ifdef __WINDOWS__
  WSADATA wsaData;

  if(WSAStartup(MAKEWORD(1, 1), &wsaData) != 0) {
    err = "failed to initialize winsock"; goto done;
  }
#endif
  memset(&uinfo,0,sizeof(AiUserInfo));

  /* receive query & cookie */
  lid   = cgi_find_query(&info->qc,"ENLoginName");
  fm    = cgi_find_query(&info->qc,"Fm");
  apv   = cgi_find_query(&info->qc,"appVersion");
  keyid = cgi_find_cookie(&info->qc,"RAKEYID");
  id    = cgi_find_cookie(&info->qc,"RASessionID");

  if(strstr(apv,"MSIE 7.0")){
	if(atof(strstr(apv,"Windows NT")+10) >= 6.0){
		ie7 = 1; /* IE7.0 and VISTA */
	}
  } 
  if(*lid == 0){ err = "license ID is empty"; goto done; }

  /* received CSR */
  cp = cgi_find_query(&info->qc,"SendCSR");
  sp = cgi_find_query(&info->qc,"SPKAC");

//{FILE *fp; if(fp=fopen("recv_req.txt","a+")){fprintf(fp,cp); fclose(fp);}}

  if(*sp){
    if((spk=cgi_spkac_decode(sp))==NULL){
      err = "cannot decode form data to SPKAC"; goto done;
    }
  }
  else if(*cp){
    fmt = cgi_get_fmt(cp);
    switch(fmt){
    case 2: csr=PEM_read_req_buf(cp); break;
    case 3: csr=cgi_csr_decode(cp); break;
    case 4: csr=cgi_csr_decode(cp+27); break;
    }
    if(csr==NULL){
      err = "cannot decode form data to CSR"; goto done;
    }
  }
  else {
    err = "cannot decode form data to CSR"; goto done;
  }

  /* get certificate template for signing request */
  if((ctt=CMP_certtmpl_new())==NULL) goto done;

  if(csr){
    ctt->publicKey = csr->pubkey; csr->pubkey = NULL;
  }else{
    ctt->publicKey = spk->pubkey; spk->pubkey = NULL;
  }

  /* bind LDAP server & get certificate subject */
  if((ld = ldap_init(info->ld.ldaphost,info->ld.ldapport))==NULL){
    err = "ldap_init error"; goto done;
  }
  if(cgi_ldap_bind(ld,info->ld.ldapadmin,info->ld.ldapadminpwd,info->ld.ldapbind) != LDAP_SUCCESS){
    err = "ldap bind error"; goto done;
  }
  if(airae_ldap_getsubject(info,ld,lid,ctt,uinfo.cn,uinfo.mail)){
    err = "cannot get cert subject"; goto done;
  }

  /* connect CA and send CSR */
  if((lcmp=LCMP_init(info->ca.svname,info->ca.caport,info->ca.caname))==NULL){
    err = "cannot initialize CA connection"; goto done;
  }
  if(info->ca.usessl){
    if(LCMP_set_ssl(lcmp,info->ca.store,info->ca.certid,info->ca.clctpw,info->ca.vfycert)){
      err = "cannot set SSL certificate"; goto done;
    }
    mode = LCMP_OPBRQ_ATSSLCL;
  }
  if(LCMP_bind_s(lcmp,info->ca.userid,info->ca.pwd,mode)){
    err = "cannot bind CA server"; goto done;
  }

  /* get PKCS12 (ca certificates) */
  if((p12=P12_dup(lcmp->ca->p12))==NULL) goto done;

  if((i=LCMP_sign_s(lcmp,"Operators",0,ctt,CMPREQUEST))<0){
    err = "cannot issue a certificate."; goto done;
  }

  if((ct=LCMP_get_signcert(lcmp))==NULL){
    err = "cannot get issued certificate."; goto done;
  }

  /* output a certificate to the ldap & remove licenseID */
  if(airae_ldap_addcert(info,ld,lid,ct)){
    err = "cannot output a certificate & remove licenseID"; goto done;
  }

  /* update session file */
  {
    AccList *al = NULL;
    unsigned char kid[32];

    cs_get_keyhash(ctt->publicKey,kid,&i);
    for(i=0,*uinfo.name=0; i<20; i++){ sprintf(cs,"%.2x",kid[i]); strcat(uinfo.name,cs); }

    if(cgi_load_session(info->sespath,&info->list)){
      err = "cannot load session file"; goto done;
    }
    /* sessionID is just dummy */
    if(cgi_set_sessiongrp(info->sespath,&info->list,&uinfo,kid,"$$RAOP$$")){
      err = "cannot set session : session save error."; goto done;
    }
    if((al=cgi_findbynamegrp(info->list,uinfo.name,"$$RAOP$$"))==NULL){
      err = "cannot find namegrp"; goto done;
    }
    al->serialNum = ct->serialNumber;
    al->notAfter = (unsigned long)timegm(&ct->time.notAfter);
    al->mode = AI_ENL_ISSDONE;
    memcpy(al->keyID,kid,32); 

    /* update a session */
    if(cgi_update_session(info->sespath,al)) goto done;
  }

  /* get return pkcs7 data */
  if(P12_add_cert(p12,ct,NULL,0xff)) goto done;
  ct=NULL;

  if(P12_check_chain(p12,0)) goto done;
  if((der=P7_signed_toDER((PKCS7*)p12,NULL,&i))==NULL) goto done;

  if((p7c=Base64_encode(i,der,32))==NULL) goto done;

  /*---------- print out HTML text -------------*/
  cp = (ie7)?(AI_ACCEPT_FORM_IE7_HTM):(AI_ACCEPT_FORM_HTM);
  if((buf=(char*)get_file2buf(get_html_name(info,cp),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");

  cp = buf;
  do{
    if((t=strstr(cp,"${"))==NULL){ printf("%s",cp); break; }
    *t=0; t++;

    printf("%s",cp);
    if(!memcmp(t,"{RETURNPKCS7}",13)){
      p1=p2=p7c;
      while(p1 && *p1){
	if((p2=strchr(p1,'\n')) != NULL) { *p2=0; p2++; }
	printf("pkcs7+=\"%s\";\n",p1);
	p1 = p2;
      }
      t+=13;
    }else if(!memcmp(t,"{FROM}",6)){
      printf("%s",fm); t+=6;
    }else{
      printf("$");
    }
    cp = t;
  }while(cp);

  /* output log */
  sprintf(cs,"%s : success to issue a operator cert",lid);
  ENCACCLOG(info->ca.caname,cs);
  ENCISSLOG(info->ca.caname,cs);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    airae_print_err(info,"send csr : System error!!",err,lcmp);
  }
  if(ld) ldap_unbind(ld);
  if(lcmp){ LCMP_unbind_s(lcmp); LCMP_free(lcmp); }
  Req_free(csr);
  Cert_free(ct);
  SPKAC_free(spk);
  CMP_certtmpl_free(ctt);
  if(p12) P12_free(p12);
  if(der) free(der);
  if(p7c) free(p7c);
  if(buf) free(buf);
#ifdef __WINDOWS__
  WSACleanup();
#endif
  return ret;
}

/*----------------------------------------------------------------------
  get cert done : confirm certificate installation
----------------------------------------------------------------------*/
int airae_get_certdone(AiEnrollInfo *info){
  char *buf=NULL,cs[256],*err="";
  char *fm;
  int ret=-1;
  off_t sz;

  fm = cgi_find_query(&info->qc,"Fm");

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_DONE_FORM_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{FROM}",fm},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  /* output log */
  sprintf(cs,"success to import a user certificate");
  ENCACCLOG(info->ca.caname,cs);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    airae_print_err(info,"done_form : System error!!",err,NULL);
  }
  if(buf) free(buf);
  return ret;
}
