/* aira_util.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_io.h>
#include <aicrypto/ok_rand.h>
#include <aicrypto/ok_asn1.h>

#include "aienr_cgi.h"

#ifndef __WITH_LANG
#define __WITH_LANG	0
#endif /* FIXME */

#define LDAP_USERENTRY_CLASS "inetOrgPerson"

/*------------------------------------------------------
  get unique and sequence number
------------------------------------------------------*/
int aira_get_count(AiEnrollInfo *info,char *name){
  AILock lock = info->plock;
  FILE *fp = NULL;
  char path[256],*buf=NULL;
  int ret = -1;
  off_t sz;

  /*** start critical section ***/
  if(OK_lock(lock,10000)){lock=NULL; goto done;}

  sprintf(path,"..%s%scount.0","/",name);
  if((buf=(char*)get_file2buf(path,&sz))==NULL){
    ret = 0;
  }else{
    ret = atoi(buf);
  }
  if((fp=fopen(path,"w"))==NULL) goto done;
  fprintf(fp,"%d",ret+1);

  /*** end critical section ***/

done:
  if(fp){ fclose(fp); }
  if(lock){ OK_unlock(lock); lock=NULL;}
  if(buf) free(buf);
  return ret;
}

/*------------------------------------------------------
  get CA DN string
------------------------------------------------------*/
int aira_get_cadn(AiEnrollInfo *info, char *ret, int max){
  Cert *ca = NULL;
  char path[256];
  int ok = -1;

  /* get ca certificate */
  snprintf(path,256,"..%s%s.cer","/",info->ca.caname);
  if((ca = Cert_read_file(path))==NULL) goto done;

  /* get entry string */
  if(certdn2ldapdn(&ca->subject_dn,ret,max)) goto done;

  ok = 0;
done:
  Cert_free(ca);
  return ok;
}

/*------------------------------------------------------
  add new raop/registration entry
------------------------------------------------------*/
int aira_add_regentry(AiEnrollInfo *info,LDAP *ld, char *rabase, AiEntryInfo *ei, 
		      char *id, char *tmstr, char *ss)
{
  LDAPMod *attrs[MAXGROUP+32];
  char dn[512],cn[128];
  int i,j,k,ok = -1;
  AiVList v[] = {
    {"cn",{id,NULL,NULL}}, /* set common name */
    {"sn",{ei->lastName,NULL}},
    {info->ld.lidattr,{ss,NULL}}, /* session secret & licenseID */
    {"mail",{ei->mail,NULL}},
    {"ou",{ei->unit,NULL}},
    {"title",{ei->title,NULL}},
    {"employeeNumber",{ei->employeeNumber,NULL}},
    {"telephoneNumber",{ei->tel1,ei->tel2,NULL}},
    {"facsimileTelephoneNumber",{ei->fax1,ei->fax2,NULL}},
    {"postalCode",{ei->postalCode,NULL}},
    {"postalAddress",{ei->postalAddress,NULL}},
    {"labeledUri",{ei->labeledUri,NULL}},
    {"description",{ei->description,NULL}},
    {"preferredLanguage",{ei->plang,NULL}},
    {"st",{ei->st,NULL}},
    {info->ld.pinattr,{ei->pin,NULL}}, /* challege PIN */
    {"street",{ei->street,NULL}},
    {"l",{tmstr,NULL}},
    {info->ld.uquattr,{ei->queryattr,NULL}}, /* unique query key */
    {NULL,{NULL}}
  };

  if(__WITH_LANG == 1){ /* JP */
    snprintf(cn,128,"%s %s",ei->lastName,ei->firstName);
  }else{
    snprintf(cn,128,"%s %s",ei->firstName,ei->lastName);
  }
  v[0].val[(id==NULL)?(0):(1)] = cn;

  /* create registration entry */
  strcpy (dn,"cn=");
  if(id==NULL){
    strncat(dn,cn,64); /* code should be UTF8 */
  }else{
    strncat(dn,id,64);
  }
  strcat (dn,", ");
  strncat(dn,rabase,440);

  /* set attribute */
  memset(attrs,0,sizeof(LDAPMod*)*(MAXGROUP+32));
  if((attrs[0]=(LDAPMod*)calloc(1,sizeof(LDAPMod)))==NULL) goto done;
  if((attrs[0]->mod_vals.modv_strvals = (char**)calloc(5,sizeof(char*)))==NULL) goto done;
  attrs[0]->mod_op=0;
  attrs[0]->mod_type = "objectClass";
  attrs[0]->mod_vals.modv_strvals[0] = "top";
  attrs[0]->mod_vals.modv_strvals[1] = "person";
  attrs[0]->mod_vals.modv_strvals[2] = "organizationalPerson";
  attrs[0]->mod_vals.modv_strvals[3] = LDAP_USERENTRY_CLASS;

  for(i=j=1; v[i-1].tag && (i<31); i++){
    if(v[i-1].val[0] && *(v[i-1].val[0])){
      for(k=0;v[i-1].val[k]&&*(v[i-1].val[k]);k++); /* count k */

      if((attrs[j]=(LDAPMod*)calloc(1,sizeof(LDAPMod)))==NULL) goto done;
      if((attrs[j]->mod_vals.modv_strvals = (char**)calloc(k+1,sizeof(char*)))==NULL) goto done;
      attrs[j]->mod_op=0;
      attrs[j]->mod_type = v[i-1].tag;

      for(k=0;v[i-1].val[k]&&*(v[i-1].val[k]); k++){
	attrs[j]->mod_vals.modv_strvals[k] = v[i-1].val[k]; /* code should be UTF8 */
      }
      j++;
    }
  }

  /* set managing group names */
  for(k=0; ei->glist[k]&&(k<(MAXGROUP+32)); k++); /* count k */
  if(k > 0){
    if((attrs[j]=(LDAPMod*)calloc(1,sizeof(LDAPMod)))==NULL) goto done;
    if((attrs[j]->mod_vals.modv_strvals = (char**)calloc(k+1,sizeof(char*)))==NULL) goto done;
    attrs[j]->mod_op=0;
    attrs[j]->mod_type = "o";
  
    for(k=0; ei->glist[k]&&(k<(MAXGROUP+32)); k++){
      attrs[j]->mod_vals.modv_strvals[k] = ei->glist[k];
    }
  }
  /* add entry */
  if(ldap_add_s( ld, dn, attrs ) != LDAP_SUCCESS) goto done;

  ok = 0;
done:
  for(i=0;attrs[i];++i){
    if(attrs[i]->mod_vals.modv_strvals){
      free(attrs[i]->mod_vals.modv_strvals);
    }
    free(attrs[i]);
  }
  return ok;
}

int gen_license_id2(char *email, char *ret){
  char *lstr = "ZYX345WVUT012SRQPONM89LKJIHGFE76DCBA";
  unsigned char rnd[32];
  char buf[32];
  time_t t;
  int i,j,max = strlen(lstr);

  time(&t);
  memset(ret,0,32);
  memset(buf,lstr[0],8);

  /* first 7 char */
  for(i=6; t &&(i>=0) ;i--){
    j = (int)t % max;
    t = (int)t / max;
    buf[i] = lstr[j];
  }

  /* other 17 char (random) */
  RAND_add(email,strlen(email),8.0);
  RAND_bytes(rnd,32);

  for(i=7; i<24; i++)
    buf[i] = lstr[rnd[i-7]%max];

  /* set return char */
  strncpy(ret,buf,6);
  strcat (ret,"-");
  strncat(ret,&buf[6],6);
  strcat (ret,"-");
  strncat(ret,&buf[12],6);
  strcat (ret,"-");
  strncat(ret,&buf[18],6);

  return 0;
}

int aira_lid2querykey(char *lid, char *ret){
  memset(ret,0,24);
  memcpy(&ret[6],lid,6);
  memcpy(&ret[0],&lid[7],6);
  memcpy(&ret[12],&lid[14],6);
  return 0;
}

/*------------------------------------------------------
  get raop/registration entry
------------------------------------------------------*/
int aira_get_raentry(AiEnrollInfo *info,AiEntryInfo *e,char *cn,int raop,int gnum){
  LDAPMessage *res = NULL, *ent;
  LDAP *ld = NULL;
  char cadn[512],radn[512];
  char *tmp,*t1=NULL,*t2=NULL,*p,*filter=NULL;
  int i,ok=-1,scope=LDAP_SCOPE_BASE;

  if(raop<2){
    /* get CA DN */
    if(aira_get_cadn(info,cadn,512)) goto done;
    memset (radn,0,512);
    
    if(raop) snprintf(radn,510,"cn=%s, %s",cn,cadn);
    else     snprintf(radn,510,"cn=%s, ou=\"RARegistration\", %s",cn,cadn);

  }else if(raop==2){
    snprintf(radn,510,"cn=%s, ou=\"Registration\", %s",cn,info->grpbase[gnum]);
  }else if(raop==3){
    strncpy(radn,info->grpbase[gnum],510);
    snprintf(cadn,510,"(&(%s=%s)(!(cn=REG*)))",info->ld.uquattr,cn);
    filter = cadn;
    scope = LDAP_SCOPE_SUBTREE;
  }else if(raop==4){
    strncpy(radn,info->grpbase[gnum],510);
    snprintf(cadn,510,"(&(mail=%s)(%s=%s))",cn,info->ld.pinattr,e->pin);
    e->pin = NULL;
    filter = cadn;
    scope = LDAP_SCOPE_SUBTREE;
  }

  if((ld = ldap_init(info->ld.ldaphost,info->ld.ldapport))==NULL) goto done;

  if(cgi_ldap_bind(ld,info->ld.ldapadmin,info->ld.ldapadminpwd,info->ld.ldapbind) != LDAP_SUCCESS)
    goto done;

  if(ldap_search_s(ld,radn,scope,filter,NULL,0,&res) != LDAP_SUCCESS) goto done;

  if((ent = ldap_first_entry(ld, res)) == NULL) goto done;

  {
    AiLAttr v[] = {
      {"cn",&t1,&t2,NULL},
      {"mail",&e->mail,NULL},
      {"ou",&e->unit,NULL},
      {"o",NULL}, /* set group list */
      {"title",&e->title,NULL},
      {"telephoneNumber",&e->tel1,&e->tel2,NULL},
      {"facsimileTelephoneNumber",&e->fax1,&e->fax2,NULL},
      {"employeeNumber",&e->employeeNumber,NULL},
      {"postalCode",&e->postalCode,NULL},
      {"postalAddress",&e->postalAddress,NULL},
      {"labeledUri",&e->labeledUri,NULL},
      {"description",&e->description,NULL},
      {"preferredLanguage",&e->plang,NULL},
      {"l",&e->l,NULL},
      {"st",&e->st,NULL},
      {info->ld.pinattr,&e->pin,NULL},
      {info->ld.uquattr,&e->queryattr,NULL},
      {"userCertificate;binary",(char**)&e->ucert,NULL},
      {NULL,NULL,NULL}
    };
    /* set group pointer */
    for(i=0; i<MAXGROUP; i++) v[3].val[i] = &e->glist[i];

    if(cgi_ldap_attrvals(ld,ent,v)) goto done;
  }

  /* get lastName & firstName */
  tmp = ((raop==3)||(raop==4))?(t1):(t2);
  p = strchr(tmp,' ');
  if(p){ *p=0; p++; }

  if(__WITH_LANG == 1){ /* jp */
	  if ((e->lastName = strdup(tmp)) == NULL) goto done;
	  if(p)
		  if ((e->firstName = strdup(p)) == NULL) goto done;
  }else{
	  if ((e->firstName = strdup(tmp)) == NULL) goto done;
	  if(p)
		  if ((e->lastName = strdup(p)) == NULL) goto done;
  }

  ok = 0;
done:
  if(t1) free(t1);
  if(t2) free(t2);
  if(ld) ldap_unbind(ld);
  if(res) ldap_msgfree(res);
  return ok;
}

int aira_get_gridLid(AiEnrollInfo *info,char *attr,char *schID,char **lid,char **stuID){
  LDAPMessage *res = NULL, *ent;
  LDAP *ld = NULL;
  char filter[256];
  int ok=-1,scope=LDAP_SCOPE_SUBTREE;

  /* set filter */
  snprintf(filter,254,"(%s=%s)",attr,schID);

  /* get user entry */
  if((ld = ldap_init(info->ld.ldaphost,info->ld.ldapport))==NULL) goto done;

  if(cgi_ldap_bind(ld,info->ld.ldapadmin,info->ld.ldapadminpwd,info->ld.ldapbind) != LDAP_SUCCESS)
    goto done;

  if(ldap_search_s(ld,info->ld.ldapbase,scope,filter,NULL,0,&res) != LDAP_SUCCESS) goto done;

  if((ent = ldap_first_entry(ld, res)) == NULL) goto done;

  {
    AiLAttr v[] = {
      {info->ld.lidattr,lid,NULL},
      {info->ld.stuattr,stuID,NULL},
      {NULL,NULL,NULL}
    };
    if(cgi_ldap_attrvals(ld,ent,v)) goto done;
  }

  ok = 0;
done:
  if(ld) ldap_unbind(ld);
  if(res) ldap_msgfree(res);
  return ok;
}

int aira_get_ctentry(AiEnrollInfo *info,AiEntryInfo *e,Cert *ct,char *st){
  LDAPMessage *res = NULL, *ent;
  LDAP *ld = NULL;
  char dn[256],fbuf[256];
  char *p=NULL,*t1=NULL,*filter=NULL;
  int i,ok=-1,scope=LDAP_SCOPE_BASE;

  /* get entry string */
  if(ct->pubkey == NULL){ /* IIS */
	  if(aienr_line2dn(ct->subject,&ct->subject_dn,&p)) goto done;
  }
  if(certdn2ldapdn(&ct->subject_dn,dn,256)) goto done;

  /* set filter */
  if(st){ snprintf(fbuf,254,"(st=%s)",st); filter = fbuf; }

  /* get user entry */
  if((ld = ldap_init(info->ld.ldaphost,info->ld.ldapport))==NULL) goto done;

  if(cgi_ldap_bind(ld,info->ld.ldapadmin,info->ld.ldapadminpwd,info->ld.ldapbind) != LDAP_SUCCESS)
    goto done;

  if(ldap_search_s(ld,dn,scope,filter,NULL,0,&res) != LDAP_SUCCESS) goto done;

  if((ent = ldap_first_entry(ld, res)) == NULL) goto done;

  {
    AiLAttr v[] = {
      {"cn",&t1,NULL},
      {"mail",&e->mail,NULL},
      {"ou",&e->unit,NULL},
      {"o",NULL}, /* set group list */
      {"title",&e->title,NULL},
      {"telephoneNumber",&e->tel1,&e->tel2,NULL},
      {"facsimileTelephoneNumber",&e->fax1,&e->fax2,NULL},
      {"employeeNumber",&e->employeeNumber,NULL},
      {"postalCode",&e->postalCode,NULL},
      {"postalAddress",&e->postalAddress,NULL},
      {"labeledUri",&e->labeledUri,NULL},
      {"description",&e->description,NULL},
      {"preferredLanguage",&e->plang,NULL},
      {"l",&e->l,NULL},
      {"st",&e->st,NULL},
      {info->ld.pinattr,&e->pin,NULL},
      {"userCertificate;binary",(char**)&e->ucert,NULL},
      {NULL,NULL,NULL}
    };
    /* set group pointer */
    for(i=0; i<MAXGROUP; i++) v[3].val[i] = &e->glist[i];

    if(cgi_ldap_attrvals(ld,ent,v)) goto done;
  }

  p = strchr(t1,' ');
  if(p){ *p=0; p++; }

  if(__WITH_LANG == 1){ /* jp */
	  if ((e->lastName = strdup(t1)) == NULL) goto done;
	  if(p)
		  if ((e->firstName = strdup(p)) == NULL) goto done;
  }else{
	  if ((e->firstName = strdup(t1)) == NULL) goto done;
	  if(p)
		  if ((e->lastName = strdup(p)) == NULL) goto done;
  }

  ok = 0;
done:
  if(t1) free(t1);
  if(ld) ldap_unbind(ld);
  if(res) ldap_msgfree(res);
  return ok;
}

Cert *aira_get_entrycert_(AiEnrollInfo *info, char *cn, int raop, int gnum, char *retem, char *lang)
{
  LDAP *ld = NULL;
  LDAPMessage *res = NULL, *ent = NULL;
  Cert *ret=NULL;
  char *attrs[] = {"userCertificate;binary","mail","preferredLanguage",NULL};
  char cadn[512],radn[512],**val = NULL;
  char *filter = "(objectClass=inetOrgPerson)";
  unsigned char *der = NULL;
  int scope=LDAP_SCOPE_BASE;

  if(raop==3){
    /* search user entry */
    strncpy(radn,info->grpbase[gnum],510);
    snprintf(cadn,510,"(&(%s=%s)(!(cn=REG*)))",info->ld.uquattr,cn);
    filter = cadn;
    scope = LDAP_SCOPE_SUBTREE;
  }else{
    /* search ra operator entry */
    /* get CA DN */
    if(aira_get_cadn(info,cadn,512)) goto done;

    memset (radn,0,512);
    snprintf(radn,510,"cn=%s, %s",cn,cadn);
  }

  if((ld = ldap_init(info->ld.ldaphost,info->ld.ldapport))==NULL) goto done;

  if(cgi_ldap_bind(ld,info->ld.ldapadmin,info->ld.ldapadminpwd,info->ld.ldapbind) != LDAP_SUCCESS)
    goto done;

  if(ldap_search_s(ld,radn,scope,filter,attrs,0,&res) != LDAP_SUCCESS) goto done;

  if(ldap_count_entries(ld,res) < 1) goto done;

  if((ent = ldap_first_entry(ld,res)) == NULL) goto done;

  if((val = ldap_get_values(ld,ent,attrs[0]))==NULL || *val==NULL ) goto done;

  /* cert struct has der pointer. need to duplicate. */
  if((der = ASN1_dup(val[0]))==NULL) goto done;
  if((ret = ASN1_read_cert(der))==NULL) goto done;
  der = NULL;

  if(retem){
    if((val = ldap_get_values(ld,ent,attrs[1])) && *val){
      strncpy(retem,val[0],126);
    }
  }
  if(lang){
    if((val = ldap_get_values(ld,ent,attrs[2])) && *val){
      strncpy(lang,val[0],4);
    }
  }

done:
  if(der) free(der);
  if(res) ldap_msgfree(res);
  if(val) ldap_value_free(val);
  if(ld) ldap_unbind(ld);
  return ret;
}

Cert *aira_get_entrycert(AiEnrollInfo *info, char *cn, int raop, int gnum){
  return aira_get_entrycert_(info,cn,raop,gnum,NULL,NULL);
}

/*------------------------------------------------------
  modify raop/registration status
------------------------------------------------------*/
int aira_change_entryst_(AiEnrollInfo *info,char *cn,char *status,char *license,
			 int raop,int gnum,int stop,char *que)
{
  LDAP *ld = NULL;
  LDAPMessage *res = NULL, *ent = NULL;
  char cadn[512],radn[512];
  char *attrs[2] = {"mail", NULL};
  char *dn=NULL;
  int ret=-1;

  if(raop==3){
    strncpy(radn,info->grpbase[gnum],510);
    snprintf(cadn,510,"(&(%s=%s)(!(cn=REG*)))",info->ld.uquattr,cn);
  }else{
    /* get CA DN */
    if(aira_get_cadn(info,cadn,512)) goto done;

    memset (radn,0,512);
    snprintf(radn,510,"cn=%s, %s",cn,cadn);
  }

  if((ld = ldap_init(info->ld.ldaphost,info->ld.ldapport))==NULL) goto done;

  if(cgi_ldap_bind(ld,info->ld.ldapadmin,info->ld.ldapadminpwd,info->ld.ldapbind) != LDAP_SUCCESS)
    goto done;

  if(raop==3){
    if(ldap_search_s(ld,info->grpbase[gnum],LDAP_SCOPE_SUBTREE,cadn,attrs,1,&res) != LDAP_SUCCESS)
      goto done;

    if(ldap_count_entries(ld,res) <= 0) goto done;

    /* find current active user entry */
    ent = ldap_first_entry(ld,res);
    if((dn = ldap_get_dn(ld, ent))==NULL) goto done;
    strncpy(radn,dn,510);
  }

  if(status)
    if(cgi_mod_attr(ld,radn,stop,"st",status)!= LDAP_SUCCESS) goto done;

  if(license)
    if(cgi_mod_attr(ld,radn,LDAP_MOD_ADD,info->ld.lidattr,license)!= LDAP_SUCCESS) goto done;

  if(que)
    if(cgi_mod_attr(ld,radn,LDAP_MOD_ADD,info->ld.uquattr,que)!= LDAP_SUCCESS) goto done;

  ret = 0;
done:
  if(dn) ldap_memfree(dn);
  if(res) ldap_msgfree(res);
  if(ld) ldap_unbind(ld);
  return ret;
}

int aira_change_entryst(AiEnrollInfo *info,char *cn,char *status,char *license,int raop,int gnum){
  return aira_change_entryst_(info,cn,status,license,raop,gnum,LDAP_MOD_REPLACE,NULL);
}

/*------------------------------------------------------
  delete raop/registration entry
------------------------------------------------------*/
int aira_delete_raentry(AiEnrollInfo *info, char *cn, int raop, int gnum){
  LDAP *ld = NULL;
  LDAPMessage *res = NULL, *ent = NULL;
  char cadn[512],radn[512];
  char *attrs[2] = {"mail", NULL};
  char *dn=NULL;
  int ret=-1;

  if(raop<2){
    /* get CA DN */
    if(aira_get_cadn(info,cadn,512)) goto done;

    memset (radn,0,512);
    if(raop) snprintf(radn,510,"cn=%s, %s",cn,cadn);
    else     snprintf(radn,510,"cn=%s, ou=\"RARegistration\", %s",cn,cadn);
  }else if(raop==2){
    snprintf(radn,510,"cn=%s, ou=\"Registration\", %s",cn,info->grpbase[gnum]);
  }else if(raop==3){
    strncpy(radn,info->grpbase[gnum],510);
    snprintf(cadn,510,"(&(%s=%s)(!(cn=REG*)))",info->ld.uquattr,cn);
  }

  if((ld = ldap_init(info->ld.ldaphost,info->ld.ldapport))==NULL) goto done;

  if(cgi_ldap_bind(ld,info->ld.ldapadmin,info->ld.ldapadminpwd,info->ld.ldapbind) != LDAP_SUCCESS)
    goto done;

  if(raop==3){
    if(ldap_search_s(ld,info->grpbase[gnum],LDAP_SCOPE_SUBTREE,cadn,attrs,1,&res) != LDAP_SUCCESS)
      goto done;

    if(ldap_count_entries(ld,res) <= 0) goto done;

    /* find current active user entry */
    ent = ldap_first_entry(ld,res);
    if((dn = ldap_get_dn(ld, ent))==NULL) goto done;
    strncpy(radn,dn,510);
  }

  if(ldap_delete_s(ld, radn) != LDAP_SUCCESS) goto done;

  ret = 0;
done:
  if(dn) ldap_memfree(dn);
  if(res) ldap_msgfree(res);
  if(ld) ldap_unbind(ld);
  return ret;
}

/*------------------------------------------------------
  change raop / user challenge PIN
------------------------------------------------------*/
int aira_change_pin(AiEnrollInfo *info, char *cn, char *old, char *new, int raop, int gnum){
  LDAP *ld = NULL;
  LDAPMessage *res = NULL, *ent = NULL;
  char cadn[512],radn[512];
  char ohash[64],nhash[64];
  char *attrs[2] = {"mail", NULL};
  char *dn=NULL;
  int ret=-1;

  cgi_gen_hash(old,ohash);
  cgi_gen_hash(new,nhash);

  if(raop==1){
    /* get CA DN */
    if(aira_get_cadn(info,cadn,512)) goto done;

    memset (radn,0,512);
    if(raop) snprintf(radn,510,"cn=%s, %s",cn,cadn);
    else     snprintf(radn,510,"cn=%s, ou=\"RARegistration\", %s",cn,cadn);
  }else if(raop==3){
    strncpy(radn,info->grpbase[gnum],510);
    snprintf(cadn,510,"(&(%s=%s)(!(cn=REG*)))",info->ld.uquattr,cn);
  }else if(raop==4){
    strncpy(radn,info->grpbase[gnum],510);
    snprintf(cadn,510,"(&(mail=%s)(%s=%s))",cn,info->ld.pinattr,ohash);
  }

  if((ld = ldap_init(info->ld.ldaphost,info->ld.ldapport))==NULL) goto done;

  if(cgi_ldap_bind(ld,info->ld.ldapadmin,info->ld.ldapadminpwd,info->ld.ldapbind) != LDAP_SUCCESS)
    goto done;

  if((raop==3)||(raop==4)){
    if(ldap_search_s(ld,info->grpbase[gnum],LDAP_SCOPE_SUBTREE,cadn,attrs,1,&res) != LDAP_SUCCESS)
      goto done;

    if(ldap_count_entries(ld,res) <= 0) goto done;

    /* find current active user entry */
    ent = ldap_first_entry(ld,res);
    if((dn = ldap_get_dn(ld, ent))==NULL) goto done;
    strncpy(radn,dn,510);
  }

  if(cgi_mod_attr_bin(ld,radn,LDAP_MOD_DELETE|LDAP_MOD_BVALUES,info->ld.pinattr,
		      ohash,strlen(ohash))!=LDAP_SUCCESS){
    ret = 1; goto done;
  }
  if(cgi_mod_attr_bin(ld,radn,LDAP_MOD_ADD|LDAP_MOD_BVALUES,info->ld.pinattr,
		      nhash,strlen(nhash))!= LDAP_SUCCESS) goto done;

  ret = 0;
done:
  if(dn) ldap_memfree(dn);
  if(res) ldap_msgfree(res);
  if(ld) ldap_unbind(ld);
  return ret;
}

/*---------------------------------------------------------
  print raop & user list
  !! caution !! windows libldap (old dll) will crash
  when sort function returns 0.
---------------------------------------------------------*/
int strcmp_nor(const char *c1,const char *c2){
  int i;
  if(c1==NULL) return -1;
  if(c2==NULL) return 1;
  i = strcmp(c1,c2);
  return (i==0)?(1):(i);
}

int strcmp_rev(const char *c1,const char *c2){
  int i;
  if(c2==NULL) return -1;
  if(c1==NULL) return 1;
  i = strcmp(c2,c1);
  return (i==0)?(-1):(i);
}

int print_list_req(AiEnrollInfo *info,char *filter,char *from,int opf,int lmax,
		   int start,int gnum,int sort){
  LDAPMessage *res = NULL, *ent;
  LDAP *ld = NULL;
  char cadn[512],rabase[512];
  char *attrs[] = { "cn", "mail", "l", "st", info->ld.uquattr, NULL };
  char *color1[2] = {"#fafafa","#eeeeee"};
  char *color2[2] = {"#ffffff","#f2f2f2"};
  char *base,*cgi,*op;
  char *skey="cn"; /* sort key */
  char *p,*q,*dn=NULL,*prn[8];
  int i,j,k,l,ok=-1,lv=LDAP_SCOPE_ONELEVEL;
  int cnt,areg=0;
  int (*sfunc)(const char*,const char*);

  if(opf < 2){
    /* get CA DN */
    if(aira_get_cadn(info,cadn,512)) goto done;

    memset (rabase,0,512);
    strncpy(rabase,"ou=\"RARegistration\", ",30);
    strncat(rabase,cadn,480);

    base = (opf==1)?(cadn):(rabase); /* set search base */
#ifdef __WINDOWS__
    cgi  = "raadmcgi.exe";
#else
    cgi  = "airaadmin";
#endif
  }else{    
    memset (rabase,0,512);
    strncpy(rabase,"ou=\"Registration\", ",30);
    strncat(rabase,info->grpbase[gnum],480);

    if(opf==3){
      base = info->grpbase[gnum];  /* set search base */
      lv   = LDAP_SCOPE_SUBTREE;
    }else{
      base = rabase; /* set search base */
    }
#ifdef __WINDOWS__
    cgi  = "raopcgi.exe";
#else
    cgi  = "airaop";
#endif
  }

  switch(sort){
  case 1:
  case 2:
  case 3:
  case 4: skey = "cn"; break;
  case 5:
  case 6: skey = "mail"; break;
  case 7:
  case 8: skey = "l"; break;
  case 9:
  case 10: skey = "st"; break;
  }
  if(sort&0x1) sfunc = strcmp_nor;
  else         sfunc = strcmp_rev;

  if((ld = ldap_init(info->ld.ldaphost,info->ld.ldapport))==NULL) goto done;

  if(cgi_ldap_bind(ld,info->ld.ldapadmin,info->ld.ldapadminpwd,info->ld.ldapbind) != LDAP_SUCCESS)
    goto done;

  if(ldap_search_s(ld,base,lv,filter,attrs,0,&res) != LDAP_SUCCESS){ ok=0; goto done; }
  cnt = ldap_count_entries(ld,res);

  /* sort entries (client) */
  if(ldap_sort_entries(ld,&res,skey,sfunc) != LDAP_SUCCESS) goto done;

  ent = ldap_first_entry(ld,res);
  lmax += start;
  i = k = 0;

  while(ent && i<lmax){
    if((dn = ldap_get_dn(ld, ent))==NULL) goto done;
    memset(prn,0,sizeof(char*)*8);

    if(i<start) goto ldap_done;

    {
      AiLAttr v[] = {
        {"cn",&prn[0],&prn[1],NULL},
        {"mail",&prn[2],NULL},
        {"l",&prn[3],NULL},
        {"st",&prn[4],&prn[5],NULL},
	{info->ld.uquattr,&prn[6],NULL},
        {NULL,NULL,NULL}
      };
      if(cgi_ldap_attrvals(ld,ent,v)) goto done;
    }

    j = -1; areg = 0;
    if(prn[4]){
      if(!strcmp(prn[4],"ACTIVE")) j=0;
      else if(!strcmp(prn[4],"EXPIRED")) j=10;
      else if(!strcmp(prn[4],"REVOKED")) j=20;
      else if(!strcmp(prn[4],"WAIT_REVOKE")) j=30;
      else if(!strcmp(prn[4],"WAIT_UPDATE")) j=40;
      else if(!strcmp(prn[4],"WAIT_ISSUE"))  j=50;
      else if(!strcmp(prn[4],"AUTOREG")){ prn[5]=prn[4]; prn[4]=NULL; }
    }
    if(prn[5] && !strcmp(prn[5],"AUTOREG")){ 
      areg = 1;
    }

    if(opf & 0x1){
      printf("<tr>");
      if(opf==3){ /* only user list */
	printf("<td bgcolor=\"%s\">",color1[i&0x1]);
	printf("<input type=checkbox name=\"UL\" value=\"%s\"></td>",prn[6]);
      }

      /* print RA operater / user list */
      printf("<td bgcolor=\"%s\">",color1[i&0x1]);
      switch(j){
      case 0:  p="i_ctok.gif"; break;
      case 10:
      case 20: p="i_ctng.gif"; break;
      case 30:
      case 40:
      case 50: p="i_csrwait.gif"; break;
      default: p="i_space.gif"; break;
      }
      printf("<img src=\"img/%s\"> ",p);

      if(opf==1){
	/* RA operator list */
	printf("<a href=\"%s?Op=ShowOP&CN=%s&Fm=%s\">%s</a></td>",cgi,prn[0],from,prn[0]);
	printf("<td bgcolor=\"%s\">%s</td>",color2[i&0x1],prn[1]);
	printf("<td bgcolor=\"%s\">%s</td>",color1[i&0x1],prn[2]);
	printf("<td bgcolor=\"%s\">%s</td>",color2[i&0x1],prn[4]);
	printf("<td bgcolor=\"%s\"><font size=-1>",color1[i&0x1]);
	op = "OP"; q="CN"; l=0;
      }else{
	/* user & operator list */
	printf("<a href=\"%s?Op=ShowUsr&Qu=%s&Fm=%s&Gp=%d\">%s</a></td>",cgi,prn[6],from,gnum,prn[0]);
	printf("<td bgcolor=\"%s\">%s</td>",color2[i&0x1],prn[2]);
	printf("<td bgcolor=\"%s\">%s</td>",color1[i&0x1],prn[4]);
	if(opf==3){ /* only user list */
	  printf("<td bgcolor=\"%s\" align=center><a href=\"javascript:autoReg('AutoReg',%d,'%s');\">",
		 color1[i&0x1],(areg)?(0):(1),prn[6]);
	  printf("<img src=\"img/%s\" border=0></a></td>",(areg)?("i_check.gif"):("i_nocheck.gif"));
	}
	printf("<td bgcolor=\"%s\"><font size=-1>",color2[i&0x1]);
	op = "Usr"; q="Qu"; l=6;
      }
      if(j>=0&&(j<30||j==40))
	printf("[<a href=\"%s?Op=Upd%s&%s=%s&Fm=%s&Gp=%d\">update</a>]",cgi,op,q,prn[l],from,gnum);
      if(j>=0&&(j<10||j==30))
	printf("[<a href=\"%s?Op=Rvk%s&%s=%s&Fm=%s&Gp=%d\">revoke</a>]",cgi,op,q,prn[l],from,gnum);
      printf("<br>");
      printf("[<a href=\"%s?Op=Del%s&%s=%s&Fm=%s&Gp=%d\">delete</a>]",cgi,op,q,prn[l],from,gnum);
      if(j>=0&&j<50){
	printf("[<a href=\"%s\" onClick=\"window.open('%s?Op=View%sCt&",cgi,cgi,op);
	printf("%s=%s&Fm=%s&Gp=%d','viewcert',",q,prn[l],from,gnum);
	printf("'width=770,height=650,scrollbars=1,resizable=1');return false;\">view</a>]");
      }
      printf("</font></td>");
      printf("</tr>\n");

    }else{
      /* print RA requester list */
      printf("<tr><td bgcolor=\"%s\" valign=top>",color1[i&0x1]);
      printf("<img src=\"img/i_csrwait.gif\"> ");
      printf("<a href=\"%s?Op=ConfirmReq&CN=%s&Fm=%s&Gp=%d\">%s</a></td>",
	     cgi,prn[0],from,gnum,prn[0]);
      printf("<td bgcolor=\"%s\">%s</td>",color2[i&0x1],prn[1]);
      printf("<td bgcolor=\"%s\">%s</td>",color1[i&0x1],prn[2]);
      printf("<td bgcolor=\"%s\">%s</td>",color2[i&0x1],prn[3]);
      printf("</tr>\n");
    }

ldap_done:
    ent = ldap_next_entry(ld,ent);
    i++;

    if(dn){ ldap_memfree(dn); dn=NULL; }
    for(j=0;j<8;j++) if(prn[j]) free(prn[j]);
  }

  ok = cnt;
done:
  if(ld) ldap_unbind(ld);
  if(res) ldap_msgfree(res);
  return ok;
}

void print_page_req(char *cgi,char *op,int lmax,int start,int sort,int count,
		    char *query,int qtype,int gnum)
{
  char *done = (!strncmp(op,"Search",6))?("Done"):("");
  int i,j;

  printf("<b>Page</b> ");
  for(i=j=0; i<count && j<10 ; i+=lmax,j++){
    if((i<=start)&&(start<i+lmax))
      printf("<b>%d</b>&nbsp; ",j+1);
    else{
      printf("<a href=\"%s?Op=%s%s&Lm=%d&St=%d&So=%d&Gp=%d",cgi,op,done,lmax,i,sort,gnum);
      if(query){
        printf("&Q=%s&QT=%d",query,qtype);
      }
      printf("\">%d</a>&nbsp; ",j+1);
    }
  }
  printf(" (Return : %d)",count);
}

/*-----------------------------------------------
  check language and set cookie
-------------------------------------------------*/
void cgi_get_lang(AiEnrollInfo *info,char *ret){
  char *lg;

  lg = cgi_find_query(&info->qc,"Lang");
  if(lg==NULL || *lg==0){
    lg = cgi_find_cookie(&info->qc,"AILANG");
    if(lg==NULL || *lg==0){
      switch(__WITH_LANG){
      case 1:  lg = "jp"; break;
      default: lg = "en"; break;
      }
    }
  }
  strncpy(ret,lg,2); ret[2]=0;
}

void cgi_get_deflang(char *ret){
  char *lg;
  switch(__WITH_LANG){
  case 1:  lg = "jp"; break;
  default: lg = "en"; break;
  }
  strncpy(ret,lg,4);
}

void cgi_get_acclang(AccList *al,char *ret){
  char *lg;
  switch(al->isstype&0xffff0000){
  case AI_LNG_JP: lg = "jp"; break;
  case AI_LNG_EN: lg = "en"; break;
  default: cgi_get_deflang(ret); return;
  }
  strncpy(ret,lg,4);
}

char *cgi_html_path(AiEnrollInfo *info,char *base,char *file){
  static char html[128];
  char lang[8];

  cgi_get_lang(info,lang);
  snprintf(html,126,"..%s%s%s%s%s%s","/",lang,"/",base,"/",file);

  return html;
}

int cgi_get_langnum(AiEnrollInfo *info){
  char lang[8];
  int ret=AI_LNG_EN;

  cgi_get_lang(info,lang);
  if(!strcmp(lang,"jp")) ret=AI_LNG_JP;
  return ret;
}

void cgi_check_lang(AiEnrollInfo *info){
  char *lg,lang[8];

  lg = cgi_find_query(&info->qc,"Lang");
  if(lg && *lg){
    strncpy(lang,lg,2); lang[2]=0;
    printf("Set-Cookie: AILANG=%s; expires=Thu, 31-Dec-2099 00:00:00 GMT;\n", lang);
  }
}
