/* aira_op.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_io.h>
#include <aicrypto/ok_rand.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_pem.h>

#include "aienr_cgi.h"

#ifndef __WITH_LANG
#define __WITH_LANG	0
#endif

/* HTML Template files */
#define AI_LOGIN_HTM      "ai_login.html"
#define AI_LOGOUT_HTM     "ai_logout.html"
#define AI_ERROR_HTM      "ai_error.html"
#define AI_VIEWCERT_HTM   "ai_viewcert.html"
#define AI_VIEWCONF_HTM   "ai_viewconf.html"
#define AI_LISTREQ_HTM    "ai_listreq.html"
#define AI_SEARCHREQ_HTM  "ai_searchreq.html"
#define AI_SEARCHREQDONE_HTM "ai_searchreqdone.html"
#define AI_CONFIRMREQ_HTM "ai_confirmreq.html"
#define AI_LISTOP_HTM     "ai_listop.html"
#define AI_SHOWOP_HTM     "ai_showop.html"
#define AI_DELOP_HTM      "ai_delop.html"
#define AI_VIEWOPCT_HTM   "ai_viewopcert.html"
#define AI_VIEWCACT_HTM   "ai_viewcacert.html"
#define AI_RVKOP_HTM      "ai_rvkop.html"
#define AI_UPDOP_HTM      "ai_updop.html"
#define AI_SEARCHOP_HTM   "ai_searchop.html"
#define AI_SEARCHOPDONE_HTM "ai_searchopdone.html"
#define AI_ENRJUMP_HTM    "ai_enrjump.html"
#define AI__HTM           "ai_error.html"


/*-- aira_op.c local value --*/
char *sbj,*keyid,*sid,path[256];
char gpl[2048]="";
char *errmsg = "";

/*----------------------------------------------------------------------
  default : login operation. check ra admin certificate
----------------------------------------------------------------------*/
char *get_html_name(AiEnrollInfo *info,char *file){
  return cgi_html_path(info,"admin",file);
}

int aira_default(AiEnrollInfo *info){
  Cert *ct = NULL;
  CertStat *st = NULL; /* no longer used ? */
  LCMP *lcmp = NULL;
  AiUserInfo uinfo;
  unsigned char sb[16],tmp2[32];
  char keyid[256],tmp[256],cn[32];
  char *buf=NULL,*cp,*sid,*sbj,*kid;
  int i=-1,ret=-1;
  off_t sz;

#ifdef __WINDOWS__
  WSADATA wsaData;

  if(WSAStartup(MAKEWORD(1, 1), &wsaData) != 0) {
    err = "failed to initialize winsock"; goto done;
  }
#endif

  /* get user info */
  if((ct = cgi_get_clientcert())==NULL){
    errmsg = "cannot get client certificate"; goto done;
  }
  sbj = cgi_find_cookie(&info->qc,"RASUBJECT");
  kid = cgi_find_cookie(&info->qc,"RAKEYID");
  sid = cgi_find_cookie(&info->qc,"RASessionID");

  /*ENCACCLOG(info->ca.caname,ct->signature);*/

  if(*kid && *sid){
    /* load session file */
    if(cgi_load_session(info->sespath,&info->list)){
      errmsg = "cannot load session file"; goto done;
    }
    /* check session info */
    i = cgi_check_sessiongrp(info->list,kid,sid,"$$RAADMIN$$");
  }

  if(i != 0){
    /* cannot find valid session. check SSL client certificate */

    /* connect CA and check client certificate */
    if((lcmp=LCMP_init(info->ca.svname,info->ca.caport,info->ca.caname))==NULL){
      errmsg = "cannot initialize CA connection"; goto done;
    }

    /* check login */
    if((i=aira_check_admin(info,ct,lcmp,&st))<0){
      errmsg = "cannot check client certificate"; goto done;
    }
    if(i==0){
      errmsg = "client certificate is not RA admin certificate"; goto done;
    }

    /* set cookie (one time session ID) */
    if((cp=ai_escape_url(ct->subject,strlen(ct->subject)))==NULL){
      errmsg = "cannot escape subject"; goto done;
    }
    printf("Set-Cookie: RASUBJECT=%s\n", cp);
    free(cp); cp=NULL;
    sbj = ct->subject;

	if(ct->pubkey){ /* apache */
	    if(cs_get_keyhash(ct->pubkey,tmp2,&i)) goto done;
	}else{ /* IIS */
		memcpy(tmp2,ct->signature,20);
	}
    for(i=0,*keyid=0; i<20; i++){ sprintf(tmp,"%.2x",tmp2[i]); strcat(keyid,tmp); }
    printf("Set-Cookie: RAKEYID=%s\n", keyid);
    kid = keyid;

    if(RAND_bytes(sb,8)) goto done;
    printf("Set-Cookie: RASessionID=%.2x%.2x%.2x%.2x%.2x%.2x%.2x%.2x\n",
	   sb[0],sb[1],sb[2],sb[3],sb[4],sb[5],sb[6],sb[7]);

    /* set new user session */
    if(cgi_load_session(info->sespath,&info->list)){
      errmsg = "cannot load session file"; goto done;
    }
    
    memset(&uinfo,0,sizeof(AiUserInfo));
    strncpy(uinfo.name,keyid,62);
    if(cgi_set_sessiongrp(info->sespath,&info->list,&uinfo,sb,"$$RAADMIN$$")){
      errmsg = "cannot set session : session save error."; goto done;
    }

    /* success - output log */
    sbj = strstr(sbj,"CN=RAAdm"); /* to be shorter */
    snprintf(tmp,254,"%s : success to login",sbj);
    ENCACCLOG(info->ca.caname,tmp);
  }
  sbj = strstr(sbj,"CN=RAAdm"); /* to be shorter */

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_LOGIN_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");

  {
    AiVList v[] = {
      {"{RASUBJECT}",sbj},
      {"{RAKEYID}",kid},
      {"{CN}",cn},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_default : System error!!",errmsg,lcmp);
  }
  if(lcmp){ LCMP_unbind_s(lcmp); LCMP_free(lcmp); }
  if(buf) free(buf);
  Cert_free(ct);
  CertStat_free_all(st);
#ifdef __WINDOWS__
  WSACleanup();
#endif
  return ret;
}

void aira_print_err(AiEnrollInfo *info,char *err1,char *err2,LCMP *lc){
  char *buf,tmp[256]="";
  off_t sz;

  if((buf=(char*)get_file2buf(get_html_name(info,AI_ERROR_HTM),&sz)) != NULL) {
    AiVList v[] = {
      {"{ERROR1}",(err1)?(err1):("")},
      {"{ERROR2}",(err2)?(err2):("")},
      {"{OKERR}",(OK_get_error())?(OK_get_errstr()):("")},
      {"{LCMPERR}",tmp},
      {NULL,NULL}
    };
    if(lc && lc->op && lc->op->resultCode != LCMP_SUCCESS){
      snprintf(tmp,254,"%s (%d) => %s\n",LCMP_msg2str(lc->op->resultCode),lc->op->resultCode,
	       lc->op->resultMsg);
    }
    cgi_print_content(buf,v);

    free(buf);
  }
}

int aira_check_session(AiEnrollInfo *info){
  int ret=-1;

  /* get user info */
  sbj   = cgi_find_cookie(&info->qc,"RASUBJECT");
  keyid = cgi_find_cookie(&info->qc,"RAKEYID");
  sid   = cgi_find_cookie(&info->qc,"RASessionID");
  if(*sbj==0) { errmsg = "cannot get cookie (subject)"; goto done; }
  if(*keyid==0){ errmsg = "cannot get cookie (keyid)"; goto done; }
  if(*sid==0) { errmsg = "cannot get cookie (sessionID)"; goto done; }

  sbj = strstr(sbj,"CN=RAAdm"); /* to be shorter */

  /* load session file */
  if(info->list==NULL && cgi_load_session(info->sespath,&info->list)){
    errmsg = "cannot load session file"; goto done;
  }
  /* check session info */
  if(cgi_check_sessiongrp(info->list,keyid,sid,"$$RAADMIN$$")){
    snprintf(path,254,"check session error : keyid:%s, sid:%s",keyid,sid);
    ENCERRLOG(info->ca.caname,path);
    errmsg = "check session error"; goto done;
  }

  ret = 0;
done:
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aira_logout(AiEnrollInfo *info){
  char tmp[256],*buf=NULL;
  int ret=-1;
  off_t sz;

  sbj = cgi_find_cookie(&info->qc,"RASUBJECT");
  sbj = strstr(sbj,"CN=RAAdm"); /* to be shorter */

  printf("Set-Cookie: RAKEYID=00; expires=Thu, 01-Jan-1970 00:00:00 GMT;\n");
  printf("Set-Cookie: RASessionID=00; expires=Thu, 01-Jan-1970 00:00:00 GMT;\n");

  /* success - output log */
  snprintf(tmp,254,"%s : success to logout",sbj);
  ENCACCLOG(info->ca.caname,tmp);

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_LOGOUT_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_logout : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aira_updateop(AiEnrollInfo *info){
  AiEntryInfo *e = NULL;
  Cert *ct = NULL;
  char *fm,*cn,sn[16],*buf=NULL;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aira_check_session(info)) goto done;

  cn = cgi_find_query(&info->qc,"CN");
  fm = cgi_find_query(&info->qc,"Fm");

  if((e=cgi_entryinfo_new())==NULL){
    errmsg = "entryinfo memory allocate error"; goto done;
  }
  if(aira_get_raentry(info,e,cn,1,0)){
    errmsg = "can not get request info from LDAP server"; goto done;
  }
  if((ct = ASN1_read_cert(e->ucert))==NULL){
    errmsg = "RA operator cert decode error"; goto done;
  }
  e->ucert = NULL;
  snprintf(sn,14,"%d",ct->serialNumber);

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_UPDOP_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{RASUBJECT}",sbj},
      {"{RAKEYID}",keyid},
      {"{firstName}",e->firstName},
      {"{lastName}",e->lastName},
      {"{email}",e->mail},
      {"{unit}",e->unit},
      {"{title}",e->title},
      {"{tel1}",e->tel1},
      {"{tel2}",e->tel2},
      {"{fax1}",e->fax1},
      {"{fax2}",e->fax2},
      {"{employeeNumber}",e->employeeNumber},
      {"{postalCode}",e->postalCode},
      {"{postalAddress}",e->postalAddress},
      {"{labeledUri}",e->labeledUri},
      {"{description}",e->description},
      {"{plang}",e->plang},
      {"{st}",e->st},
      {"{CN}",cn},
      {"{SERIAL}",sn},
      {"{FROM}",fm},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_updateop : System error!!",errmsg,NULL);
  }
  cgi_entryinfo_free(e);
  Cert_free(ct);
  if(buf) free(buf);
  return ret;
}

int aira_updopdone(AiEnrollInfo *info){
  AiEntryInfo e;
  LCMP *lcmp = NULL;
  CertStat *st = NULL;
  char *ml,*fm,*cn,*sn,*buf = NULL;
  char lid[256],lhash[64];
  int i,cl,inf,mode,ret = -1;
  off_t sz;

#ifdef __WINDOWS__
  WSADATA wsaData;

  if(WSAStartup(MAKEWORD(1, 1), &wsaData) != 0) {
    err = "failed to initialize winsock"; goto done;
  }
#endif
  memset(&e,0,sizeof(AiEntryInfo));

  /* check session */
  if(aira_check_session(info)) goto done;

  cn = cgi_find_query(&info->qc,"CN");
  fm = cgi_find_query(&info->qc,"Fm");
  sn = cgi_find_query(&info->qc,"SERIAL");
  cl = atoi(cgi_find_query(&info->qc,"Cancel"));
  inf = atoi(cgi_find_query(&info->qc,"inform"));
  e.mail = cgi_find_query(&info->qc,"email");
  e.plang = cgi_find_query(&info->qc,"plang");

  if(cn==NULL){ errmsg = "cannot get CN"; goto done; }
  if(sn==NULL){ errmsg = "cannot get serial number"; goto done; }
  if(fm==NULL){ errmsg = "cannot get fm"; goto done; }

  if(cl==1){
    /* cancel WAIT_UPDATE */
    if(aira_change_entryst(info,cn,"ACTIVE",NULL,1,0)){
      errmsg = "cannot update ldap entry"; goto done;
    }
    ml = "raadm_cancel_upd.txt"; inf = cl;
  }else{
    /* connect CA and revoke Cert */
    if(!info->offlineca){
      if((lcmp=LCMP_init(info->ca.svname,info->ca.caport,info->ca.caname))==NULL){
	errmsg = "cannot initialize CA connection"; goto done;
      }
      if(info->ca.usessl){
	if(LCMP_set_ssl(lcmp,info->ca.store,info->ca.certid,info->ca.clctpw,info->ca.vfycert)){
	  errmsg = "cannot set SSL certificate"; goto done;
	}
	mode = LCMP_OPBRQ_ATSSLCL;
      }
      if(LCMP_bind_s(lcmp,info->ca.userid,info->ca.pwd,mode)) goto done;
    
      if(LCMP_list_s(lcmp,"Operators",atoi(sn))) goto done;
      if((st=LCMP_get_statptr(lcmp))==NULL) goto done;
    
      /* if old certificate is active, revoke it */
      if((st->state & (STAT_EXPIRED|STAT_REVOKED)) == 0){
	if((i=LCMP_cert_s(lcmp,LCMP_OPCT_RVKCERT,atoi(sn),0,NULL,NULL))<0) goto done;
	if(i){
	  sprintf(lid,"CA returns error code : %.8x",i);
	  errmsg = lid; goto done;
	}
      }
    }

    /* generate licenseID & hashed value */
    if(gen_license_id2(cn,lid)){
      errmsg = "can not generate license ID"; goto done;
    }
    if(cgi_gen_hash(lid,lhash)){
      errmsg = "can not get hashed license ID"; goto done;
    }

    /* update ldap status */
    if(aira_change_entryst(info,cn,"WAIT_ISSUE",lhash,1,0)){
      errmsg = "cannot update ldap entry"; goto done;
    }
    ml = "raadm_accept_op.txt";
  }

  /* send email to user */
  if(inf){
    if(*e.plang) strncpy(info->ml.lang,e.plang,4);
    else cgi_get_deflang(info->ml.lang);

    if(aira_sendmail_adm2usr(&info->ml,&e,info->ml.admemail,lid,ml)){
      snprintf(lid,254,"cannot send a email to ra operator (cn=%s) << %s",cn,info->ml.lang);
      ENCERRLOG(info->ca.caname,lid);
    }

    /* show list or search page */
    if(!strcmp(fm,"SearchOP")) aira_searchop(info);
    else if(!strcmp(fm,"ListOP")) aira_listop(info);
    else if(!strcmp(fm,"ListRvkReq")) aira_listrvkreq(info);
    else if(!strcmp(fm,"ListUpdReq")) aira_listupdreq(info);

  }else{
    /* jump to airaenroll */
    if((buf=(char*)get_file2buf(get_html_name(info,AI_ENRJUMP_HTM),&sz))==NULL) goto done;

    cgi_check_lang(info);
    printf("Content-type: text/html\n\n");
    {
      AiVList v[] = {
        {"{LID}",lid},
        {"{FROM}",fm},
        {NULL,NULL}
      };
      cgi_print_content(buf,v);
    }
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_updopdone : System error!!",errmsg,lcmp);
  }
  CertStat_free_all(st);
  if(lcmp){ LCMP_unbind_s(lcmp); LCMP_free(lcmp); }
  if(buf) free(buf);
#ifdef __WINDOWS__
  WSACleanup();
#endif
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aira_revokeop(AiEnrollInfo *info){
  AiEntryInfo *e=NULL;
  Cert *ct=NULL;
  char *csbj=NULL,*p,sn[16],start[32],end[32];
  char *cn,*fm,*buf=NULL;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aira_check_session(info)) goto done;

  cn = cgi_find_query(&info->qc,"CN");
  fm = cgi_find_query(&info->qc,"Fm");

  /* get user entry info */
  if((e=cgi_entryinfo_new())==NULL){
    errmsg = "entryinfo memory allocate error"; goto done;
  }
  if(aira_get_raentry(info,e,cn,1,0)){
    errmsg = "can not get operator entry from LDAP server"; goto done;
  }

  /* get ra oprator certificate */
  if((ct = ASN1_read_cert(e->ucert))==NULL){
    errmsg = "cannot get oprator certificate"; goto done;
  }
  e->ucert = NULL; csbj = ct->subject;
  snprintf(sn,14,"%d",ct->serialNumber);
  if((p = stm2str(&ct->time.notBefore,3)) != NULL) strncpy(start,p,30);
  if((p = stm2str(&ct->time.notAfter,3)) != NULL) strncpy(end,p,30);


  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_RVKOP_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{RASUBJECT}",sbj},
      {"{RAKEYID}",keyid},
      {"{sbj}",csbj},
      {"{SERIAL}",sn},
      {"{notbefore}",start},
      {"{notafter}",end},
      {"{email}",e->mail},
      {"{plang}",e->plang},
      {"{FROM}",fm},
      {"{CN}",cn},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_revokeop : System error!!",errmsg,NULL);
  }
  Cert_free(ct);
  cgi_entryinfo_free(e);
  if(buf) free(buf);
  return ret;
}

int aira_rvkopdone(AiEnrollInfo *info){
  LCMP *lcmp = NULL;
  char *ml,*em,*fm,*cn,*sn,*rs,*lg,cs[256];
  int cl,code,mode,ret = -1;

#ifdef __WINDOWS__
  WSADATA wsaData;

  if(WSAStartup(MAKEWORD(1, 1), &wsaData) != 0) {
    err = "failed to initialize winsock"; goto done;
  }
#endif

  /* check session */
  if(aira_check_session(info)) goto done;

  cn = cgi_find_query(&info->qc,"CN");
  sn = cgi_find_query(&info->qc,"SERIAL");
  rs = cgi_find_query(&info->qc,"Reason");
  fm = cgi_find_query(&info->qc,"Fm");
  em = cgi_find_query(&info->qc,"email");
  lg = cgi_find_query(&info->qc,"plang");
  cl = atoi(cgi_find_query(&info->qc,"Cancel"));
  if(sn==NULL){ errmsg = "cannot get serial number"; goto done; }
  if(rs==NULL){ errmsg = "cannot get revoke reason"; goto done; }
  if(fm==NULL){ errmsg = "cannot get fm"; goto done; }
  if(em==NULL){ errmsg = "cannot get em"; goto done; }

  if(cl==1){
    /* cansel WAIT_REVOKE */
    if(aira_change_entryst(info,cn,"ACTIVE",NULL,1,0)){
      errmsg = "cannot update ldap entry"; goto done;
    }
    ml = "raadm_cancel_rvk.txt";
  }else{
    /* connect CA and revoke Cert */
    if(!info->offlineca){
      if((lcmp=LCMP_init(info->ca.svname,info->ca.caport,info->ca.caname))==NULL){
	errmsg = "cannot initialize CA connection"; goto done;
      }
      if(info->ca.usessl){
	if(LCMP_set_ssl(lcmp,info->ca.store,info->ca.certid,info->ca.clctpw,info->ca.vfycert)){
	  errmsg = "cannot set SSL certificate"; goto done;
	}
	mode = LCMP_OPBRQ_ATSSLCL;
      }
      if(LCMP_bind_s(lcmp,info->ca.userid,info->ca.pwd,mode)) goto done;
    
      if((code=LCMP_cert_s(lcmp,LCMP_OPCT_RVKCERT,atoi(sn),atoi(rs),NULL,NULL))<0) goto done;
    
      if(code){
	sprintf(cs,"CA returns error code : %.8x",code);
	errmsg = cs; goto done;
      }
    }

    /* update ldap status */
    if(aira_change_entryst(info,cn,"REVOKED",NULL,1,0)){
      errmsg = "cannot update ldap entry"; goto done;
    }
    ml = "raadm_revoke_op.txt";
  }

  if(*info->ml.smtphost && *info->ml.admemail){
    AiEntryInfo e;
    e.mail = em;

    if(*lg) strncpy(info->ml.lang,lg,4);
    else cgi_get_lang(info,info->ml.lang);

    if(aira_sendmail_adm2usr(&info->ml,&e,info->ml.admemail,cn,ml)){
      snprintf(cs,254,"cannot send a email to ra operator (cn=%s) << %s",cn,info->ml.lang);
      ENCERRLOG(info->ca.caname,cs);
    }
  }

  /* show list or search page */
  if(!strcmp(fm,"SearchOP")) aira_searchop(info);
  else if(!strcmp(fm,"ListOP")) aira_listop(info);
  else if(!strcmp(fm,"ListRvkReq")) aira_listrvkreq(info);
  else if(!strcmp(fm,"ListUpdReq")) aira_listupdreq(info);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_rvkopdone : System error!!",errmsg,lcmp);
  }
  if(lcmp){ LCMP_unbind_s(lcmp); LCMP_free(lcmp); }
#ifdef __WINDOWS__
  WSACleanup();
#endif
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aira_delop(AiEnrollInfo *info){
  AiEntryInfo *e = NULL;
  char *fm,*cn,*buf=NULL;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aira_check_session(info)) goto done;

  cn = cgi_find_query(&info->qc,"CN");
  fm = cgi_find_query(&info->qc,"Fm");

  if((e=cgi_entryinfo_new())==NULL){
    errmsg = "entryinfo memory allocate error"; goto done;
  }
  if(aira_get_raentry(info,e,cn,1,0)){
    errmsg = "can not get request info from LDAP server"; goto done;
  }

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_DELOP_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{RASUBJECT}",sbj},
      {"{RAKEYID}",keyid},
      {"{firstName}",e->firstName},
      {"{lastName}",e->lastName},
      {"{email}",e->mail},
      {"{unit}",e->unit},
      {"{title}",e->title},
      {"{tel1}",e->tel1},
      {"{tel2}",e->tel2},
      {"{fax1}",e->fax1},
      {"{fax2}",e->fax2},
      {"{employeeNumber}",e->employeeNumber},
      {"{postalCode}",e->postalCode},
      {"{postalAddress}",e->postalAddress},
      {"{labeledUri}",e->labeledUri},
      {"{description}",e->description},
      {"{st}",e->st},
      {"{CN}",cn},
      {"{FROM}",fm},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_delop : System error!!",errmsg,NULL);
  }
  cgi_entryinfo_free(e);
  if(buf) free(buf);
  return ret;
}

int aira_delopdone(AiEnrollInfo *info){
  char *fm,*cn;
  int ret = -1;

  /* check session */
  if(aira_check_session(info)) goto done;

  cn = cgi_find_query(&info->qc,"CN");
  fm = cgi_find_query(&info->qc,"Fm");
  if(fm==NULL){ errmsg = "cannot get fm"; goto done; }

  /* delete operator entry */
  if(aira_delete_raentry(info,cn,1,0)){
    errmsg = "cannot delete ra operator entry"; goto done;
  }
  /* show list or search page */
  if(!strcmp(fm,"SearchOP")) aira_searchop(info);
  else if(!strcmp(fm,"ListOP")) aira_listop(info);
  else if(!strcmp(fm,"ListRvkReq")) aira_listrvkreq(info);
  else if(!strcmp(fm,"ListUpdReq")) aira_listupdreq(info);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_delopdone : System error!!",errmsg,NULL);
  }
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aira_viewopcert(AiEnrollInfo *info){
  Cert *ct = NULL;
  char *cp,*p,*t;
  char *fm,*cn,*buf=NULL;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aira_check_session(info)) goto done;

  cn = cgi_find_query(&info->qc,"CN");
  fm = cgi_find_query(&info->qc,"Fm");

  /* get user RA operator certificate */
  if((ct = aira_get_entrycert(info,cn,1,0))==NULL){
    errmsg = "cannot get operator certificate"; goto done;
  }

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_VIEWOPCT_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  cp = buf;
  do{
    if((t=strstr(cp,"${"))==NULL){ printf("%s",cp); break; }
    *t=0; t++;

    if(!memcmp(t,"{RASUBJECT}",11)){
      p = sbj; t+=11;
    }else if(!memcmp(t,"{RAKEYID}",9)){
      p = keyid; t+=9;
    }else if(!memcmp(t,"{CERTTEXT}",10)){
      printf("%s",cp); CERT_PRINT(ct); cp=t+10; continue;
    }else{
      p = "$";
    }
    printf("%s",cp);
    printf("%s",p);
    cp = t;
  }while(cp);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_viewopcert : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  Cert_free(ct);
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aira_viewraadm(AiEnrollInfo *info){
  Cert *ct = NULL;
  char *cp,*p,*t;
  char *buf=NULL;
  int ret=-1;
  off_t sz;

  /* get user info */
  if((ct = cgi_get_clientcert())==NULL){
    errmsg = "cannot get client certificate"; goto done;
  }
  /* check session */
  if(aira_check_session(info)) goto done;

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_VIEWCERT_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  cp = buf;
  do{
    if((t=strstr(cp,"${"))==NULL){ printf("%s",cp); break; }
    *t=0; t++;

    if(!memcmp(t,"{RASUBJECT}",11)){
      p = sbj; t+=11;
    }else if(!memcmp(t,"{RAKEYID}",9)){
      p = keyid; t+=9;
    }else if(!memcmp(t,"{CERTTEXT}",10)){
      printf("%s",cp); CERT_PRINT(ct); cp=t+10; continue;
    }else{
      p = "$";
    }
    printf("%s",cp);
    printf("%s",p);
    cp = t;
  }while(cp);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_viewraadm : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  Cert_free(ct);
  return ret;
}

void print_cgi_conf(AiEnrollInfo *info){
  struct cgi_conf_data{
    char *tagname;
    int  valtype; /* 0: (char *) 1: (int) */
    void *value;
  } dat[] = {
    {"RA path",0,info->rapath},
    {"CA server",0,info->ca.svname},
    {"CA port",1,&info->ca.caport},
    {"CA name",0,info->ca.caname},
    {"CA : certid",0,info->ca.certid},
    {"CA : use SSL",1,&info->ca.usessl},
    {"CA : verify level",1,&info->ca.vfycert},
    {"RA authmode",1,&info->authmode},
    {"RA postmode",1,&info->postmode},
    {"RA ldaplogin",1,&info->ldaplogin},
    {"Access log",0,info->log.acclog},
    {"Issue log",0,info->log.isslog},
    {"Error log",0,info->log.errlog},
    {"SMTP server",0,info->ml.smtphost},
    {"SMTP port",1,&info->ml.smtpport},
    {"Mail : admin",0,info->ml.admemail},
    {"Mail : system",0,info->ml.sysemail},
    {"Mail : base URL",0,info->ml.enrollurl},
    {"LDAP server",0,info->ld.ldaphost},
    {"LDAP port",1,&info->ld.ldapport},
    {"LDAP : base",0,info->ld.ldapbase},
    {"LDAP : admin",0,info->ld.ldapadmin},
    {"LDAP : bind type",1,&info->ld.ldapbind},
    {"LDAP : userID attr",0,info->ld.uidattr},
    {"LDAP : licenseID attr",0,info->ld.lidattr},
    {"LDAP : challenge PIN attr",0,info->ld.pinattr},
    {NULL,0,NULL}
  };
  char *color1[2] = {"#fafafa","#eeeeee"};
  char *color2[2] = {"#ffffff","#f2f2f2"};
  int i = 0;

  while(dat[i].tagname){
    printf("<tr><td bgcolor=\"%s\"><b>&nbsp;%s</b></td>",color1[i&0x1],dat[i].tagname);
    printf("<td bgcolor=\"%s\">",color2[i&0x1]);
    switch(dat[i].valtype){
    case 1:  printf("%d</td></tr>\n",*(int *)dat[i].value); break;
    default: printf("%s</td></tr>\n",(char *)dat[i].value); break;
    }
    i++;
  }
}

int aira_viewcgiconf(AiEnrollInfo *info){
  char *cp,*p,*t,*buf=NULL;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aira_check_session(info)) goto done;

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_VIEWCONF_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  cp = buf;
  do{
    if((t=strstr(cp,"${"))==NULL){ printf("%s",cp); break; }
    *t=0; t++;

    if(!memcmp(t,"{RASUBJECT}",11)){
      p = sbj; t+=11;
    }else if(!memcmp(t,"{RAKEYID}",9)){
      p = keyid; t+=9;
    }else if(!memcmp(t,"{VIEWCONF}",10)){
      printf("%s",cp); print_cgi_conf(info); cp=t+10; continue;
    }else{
      p = "$";
    }
    printf("%s",cp);
    printf("%s",p);
    cp = t;
  }while(cp);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_viewcgiconf : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int print_search_list(AiEnrollInfo *info, char *buf, char *filter, char *from, int raop,
		      int lmax, int start, int sort, char *query, int qtype)
{
  char cbg[16],clm[16],qt[16],so[16];
  char *cgi,*cp,*p,*t,*tmp=NULL;
  int i,ret=-1;

  sprintf(clm,"%d",lmax);
  sprintf(cbg,"%d",start);
  sprintf(qt,"%d",qtype);
  sprintf(so,"%d",sort);

#ifdef __WINDOWS__
  cgi = "raadmcgi.exe";
#else
  cgi = "airaadmin";
#endif

  cp = buf;
  do{
    if((t=strstr(cp,"${"))==NULL){ printf("%s",cp); break; }
    *t=0; t++;

    if(!memcmp(t,"{RASUBJECT}",11)){
      p = sbj; t+=11;
    }else if(!memcmp(t,"{RAKEYID}",9)){
      p = keyid; t+=9;
    }else if(!memcmp(t,"{SEARCHLIST}",12)){
      printf("%s",cp);
      if((i=print_list_req(info,filter,from,raop,lmax,start,0,sort))<0){
	errmsg = "can not search LDAP server"; goto done;
      }
      cp=t+12; continue;
    }else if(!memcmp(t,"{PAGELIST}",10)){
      printf("%s",cp);
      print_page_req(cgi,from,lmax,start,sort,i,query,qtype,0);
      cp=t+10; continue;
    }else if(!memcmp(t,"{LMAX}",6)){
      p = clm; t+=6;
    }else if(!memcmp(t,"{SORT}",6)){
      p = so; t+=6;
    }else if(!memcmp(t,"{LST}",5)){
      p = cbg; t+=5;
    }else if(!memcmp(t,"{Q}",3)){
      if(query){
	if((tmp=ai_escape_url(query,strlen(query)))==NULL) goto done;
	p = tmp;
      }else{
	p = "";
      }
      t+=3;
    }else if(!memcmp(t,"{Op}",4)){
      p = from; t+=4;
    }else if(!memcmp(t,"{QT}",4)){
      p = qt; t+=4;
    }else{
      p = "$";
    }
    fputs(cp,stdout);
    fputs(p,stdout);
    cp = t;
  }while(cp);

  ret = 0;
done:
  if(tmp) free(tmp);
  return ret;
}

int aira_listreq(AiEnrollInfo *info){
  char *buf=NULL;
  int bg,lm,so;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aira_check_session(info)) goto done;

  lm = atoi(cgi_find_query(&info->qc,"Lm"));
  bg = atoi(cgi_find_query(&info->qc,"St"));
  so = atoi(cgi_find_query(&info->qc,"So"));
  if(lm <= 0) lm = 20;
  if(bg < 0) bg = 0;
  if(so <= 0) so = 2;

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_LISTREQ_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  if(print_search_list(info,buf,"(cn=RAREG*)","ListReq",0,lm,bg,so,NULL,0)) goto done;

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_listreq : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

int aira_searchreq(AiEnrollInfo *info){
  char *buf=NULL;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aira_check_session(info)) goto done;

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_SEARCHREQ_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{RASUBJECT}",sbj},
      {"{RAKEYID}",keyid},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_searchreq : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

int aira_searchreqdone(AiEnrollInfo *info){
  char *buf=NULL,*text;
  char filter[256];
  int bg,lm,so;
  int st,ret=-1;
  off_t sz;

  /* check session */
  if(aira_check_session(info)) goto done;

  text = cgi_find_query(&info->qc,"Q");
  st   = atoi(cgi_find_query(&info->qc,"QT"));
  lm   = atoi(cgi_find_query(&info->qc,"Lm"));
  bg   = atoi(cgi_find_query(&info->qc,"St"));
  so   = atoi(cgi_find_query(&info->qc,"So"));
  if(*text==0){ errmsg = "can not search LDAP server (NULL input)."; goto done; }

  if(lm <= 0) lm = 20;
  if(bg < 0) bg = 0;
  if(so <= 0) so = 2;

  memset(filter,0,256);

  switch(st){
  case 1: snprintf(filter,254,"(cn=*%s*)",text); break;
  case 2: snprintf(filter,254,"(mail=*%s*)",text); break;
  case 3: snprintf(filter,254,"(cn=%s)",text); break;
  case 4: snprintf(filter,254,"(ou=*%s*)",text); break;
  case 5: snprintf(filter,254,"(telephoneNumber=*%s*)",text); break;
  }

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_SEARCHREQDONE_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  if(print_search_list(info,buf,filter,"SearchReq",0,lm,bg,so,text,st)) goto done;

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_searchreqdone : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

/*//////////////////////////////////////////////////////////////////////////*/

int aira_listupdreq(AiEnrollInfo *info){
  char *buf=NULL;
  int bg,lm,so;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aira_check_session(info)) goto done;

  lm = atoi(cgi_find_query(&info->qc,"Lm"));
  bg = atoi(cgi_find_query(&info->qc,"St"));
  so = atoi(cgi_find_query(&info->qc,"So"));

  if(lm <= 0) lm = 20;
  if(bg < 0) bg = 0;
  if(so <= 0) so = 2;

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_LISTOP_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  if(print_search_list(info,buf,"(st=WAIT_UPDATE)","ListUpdReq",1,lm,bg,so,NULL,0)) goto done;

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_listupdreq : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

int aira_listrvkreq(AiEnrollInfo *info){
  char *buf=NULL;
  int bg,lm,so;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aira_check_session(info)) goto done;

  lm = atoi(cgi_find_query(&info->qc,"Lm"));
  bg = atoi(cgi_find_query(&info->qc,"St"));
  so = atoi(cgi_find_query(&info->qc,"So"));

  if(lm <= 0) lm = 20;
  if(bg < 0) bg = 0;
  if(so <= 0) so = 2;

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_LISTOP_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  if(print_search_list(info,buf,"(st=WAIT_REVOKE)","ListRvkReq",1,lm,bg,so,NULL,0)) goto done;

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_listrvkreq : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

/*//////////////////////////////////////////////////////////////////////////*/

void aira_gcklist_buf(AiEnrollInfo *info,AiEntryInfo *e,char *buf,int max){
  char line[256];
  char *ck;
  int i,j,k;

  /* get all group name */
  for(i=j=0; i<MAXGROUP; i++){
    if(info->grpname[i]==NULL) break;
    ck = "";

    for(k=0; k<MAXGROUP && e->glist[k]; k++)
      if(!strcmp(info->grpname[i],e->glist[k])){ ck=" checked"; break; }

    snprintf(line,254,"<input type=checkbox name=\"Gp%d\" value=\"%d\"%s>%s&nbsp;&nbsp;\n",
	     i,i,ck,info->grpname[i]);
    j+=strlen(line);

    if(j>max) break; /* overflow check */
    strcat(buf,line);
  }
}

int aira_confirmreq(AiEnrollInfo *info){
  AiEntryInfo *e = NULL;
  char *buf=NULL;
  char *anum,*fm;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aira_check_session(info)) goto done;

  anum = cgi_find_query(&info->qc,"CN");
  fm = cgi_find_query(&info->qc,"Fm");

  if((e=cgi_entryinfo_new())==NULL){
    errmsg = "entryinfo memory allocate error"; goto done;
  }
  if(aira_get_raentry(info,e,anum,0,0)){
    errmsg = "can not get request info from LDAP server"; goto done;
  }

  /* get active group names */
  aira_gcklist_buf(info,e,gpl,2048);

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_CONFIRMREQ_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{RASUBJECT}",sbj},
      {"{RAKEYID}",keyid},
      {"{firstName}",e->firstName},
      {"{lastName}",e->lastName},
      {"{email}",e->mail},
      {"{unit}",e->unit},
      {"{title}",e->title},
      {"{tel1}",e->tel1},
      {"{tel2}",e->tel2},
      {"{fax1}",e->fax1},
      {"{fax2}",e->fax2},
      {"{employeeNumber}",e->employeeNumber},
      {"{postalCode}",e->postalCode},
      {"{postalAddress}",e->postalAddress},
      {"{labeledUri}",e->labeledUri},
      {"{description}",e->description},
      {"{plang}",e->plang},
      {"{chlg}",e->pin},
      {"{acceptnum}",anum},
      {"{acceptdate}",e->l},
      {"{GROUPNAMES}",gpl},
      {"{FROM}",fm},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_confirmreq : System error!!",errmsg,NULL);
  }
  cgi_entryinfo_free(e);
  if(buf) free(buf);
  return ret;
}

int aira_rejectdone(AiEnrollInfo *info){
  AiEntryInfo e;
  char tmp[256];
  char *fm,*anum;
  int inf,ret=-1;

  memset(&e,0,sizeof(AiEntryInfo));

  /* check session */
  if(aira_check_session(info)) goto done;

  /* get query */
  anum = cgi_find_query(&info->qc,"acceptnum");
  inf  = atoi(cgi_find_query(&info->qc,"inform"));
  fm   = cgi_find_query(&info->qc,"FROM");
  e.mail = cgi_find_query(&info->qc,"email");
  e.plang = cgi_find_query(&info->qc,"plang");
  e.firstName= cgi_find_query(&info->qc,"firstName");
  e.lastName = cgi_find_query(&info->qc,"lastName");

  /* delete entry */
  if(aira_delete_raentry(info,anum,0,0)){
    errmsg = "can not delete ra request entry"; goto done;
  }
  /* send email to user */
  strncpy(info->ml.lang,e.plang,4); /* set user preferred lungage */
  if(inf && aira_sendmail_adm2usr(&info->ml,&e,info->ml.admemail,anum,"raadm_reject_op.txt")){
    snprintf(tmp,254,"cannot send a email to ra operator (cn=%s) << %s",anum,info->ml.lang);
    ENCERRLOG(info->ca.caname,tmp);
  }

  /* show list or search page */
  if(!strcmp(fm,"SearchReq")) aira_searchreq(info);
  else if(!strcmp(fm,"ListReq")) aira_listreq(info);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_rejectdone : System error!!",errmsg,NULL);
  }
  return ret;  
}

int aira_add_raopentry(AiEnrollInfo *info, AiEntryInfo *e, char *opid, char *regid, char *lid){
  LDAP *ld = NULL;
  char cadn[512],radn[512];
  int ok=-1;

  /* get CA DN */
  if(aira_get_cadn(info,cadn,512)) goto done;

  memset (radn,0,512);
  snprintf(radn,510,"cn=%s, ou=\"RARegistration\", %s",regid,cadn);

  if((ld = ldap_init(info->ld.ldaphost,info->ld.ldapport))==NULL) goto done;

  if(cgi_ldap_bind(ld,info->ld.ldapadmin,info->ld.ldapadminpwd,info->ld.ldapbind) != LDAP_SUCCESS)
    goto done;

  if(aira_add_regentry(info,ld,cadn,e,opid,"",lid)) goto done;

  if(ldap_delete_s(ld, radn) != LDAP_SUCCESS) goto done;

  ok = 0;
done:
  if(ld) ldap_unbind(ld);
  return ok;
}

void aira_set_glist(AiEnrollInfo *info, AiEntryInfo *e){
  char tag[16],*cp;
  int i,j;

  /* get all group name */
  for(i=j=0; i<MAXGROUP; i++){
    if(info->grpname[i]==NULL) break;

    snprintf(tag,16,"Gp%d",i);
    cp = cgi_find_query(&info->qc,tag);

    if(cp && *cp){
      e->glist[j] = info->grpname[i]; j++;
    }
  }
}

int aira_acceptdone(AiEnrollInfo *info){
  AiEntryInfo e;
  char tmp[256],raid[32];
  char lid[64],lhash[64];
  char *fm,*anum,*buf=NULL;
  int i,inf,ret=-1;
  off_t sz;

  memset(&e,0,sizeof(AiEntryInfo));

  /* check session */
  if(aira_check_session(info)) goto done;

  /* get query */
  anum = cgi_find_query(&info->qc,"acceptnum");
  inf  = atoi(cgi_find_query(&info->qc,"inform"));
  fm   = cgi_find_query(&info->qc,"FROM");
  e.firstName= cgi_find_query(&info->qc,"firstName");
  e.lastName = cgi_find_query(&info->qc,"lastName");
  e.mail     = cgi_find_query(&info->qc,"email");

  if(*e.firstName== 0){ errmsg = "post message not found"; goto done; }
  if(*e.lastName == 0){ errmsg = "post message not found"; goto done; }
  if(*e.mail     == 0){ errmsg = "post message not found"; goto done; }

  e.unit  = cgi_find_query(&info->qc,"unit");
  e.title = cgi_find_query(&info->qc,"title");
  e.tel1  = cgi_find_query(&info->qc,"tel1");
  e.tel2  = cgi_find_query(&info->qc,"tel2");
  e.fax1  = cgi_find_query(&info->qc,"fax1");
  e.fax2  = cgi_find_query(&info->qc,"fax2");
  e.employeeNumber = cgi_find_query(&info->qc,"employeeNumber");
  e.postalCode     = cgi_find_query(&info->qc,"postalCode");
  e.postalAddress  = cgi_find_query(&info->qc,"postalAddress");
  e.labeledUri     = cgi_find_query(&info->qc,"labeledUri");
  e.description    = cgi_find_query(&info->qc,"description");
  e.plang = cgi_find_query(&info->qc,"plang");
  e.pin   = cgi_find_query(&info->qc,"chlg");
  e.st    = "WAIT_ISSUE";

  /* set managing group list */
  aira_set_glist(info,&e);

  /* replace html symbol tags '<' or '>' to space char */
  ai_symbol_esc_sp(e.firstName);
  ai_symbol_esc_sp(e.lastName);
  ai_symbol_esc_sp(e.mail);
  ai_symbol_esc_sp(e.unit);
  ai_symbol_esc_sp(e.title);
  ai_symbol_esc_sp(e.tel1);
  ai_symbol_esc_sp(e.tel2);
  ai_symbol_esc_sp(e.fax1);
  ai_symbol_esc_sp(e.fax2);
  ai_symbol_esc_sp(e.employeeNumber);
  ai_symbol_esc_sp(e.postalCode);
  ai_symbol_esc_sp(e.postalAddress);
  ai_symbol_esc_sp(e.labeledUri);
  ai_symbol_esc_sp(e.description);

  /* get accept ID */
  if((i = aira_get_count(info,"raop"))<0){
    errmsg = "can not get raop number. check counter file"; goto done;
  }
  sprintf(raid,"RAOP%.4d",i);

  /* generate licenseID & hashed value */
  if(gen_license_id2(e.mail,lid)){
    errmsg = "can not generate license ID"; goto done;
  }
  if(cgi_gen_hash(lid,lhash)){
    errmsg = "can not get hashed license ID"; goto done;
  }

  /* create new raop entry & delete request entry */
  if(aira_add_raopentry(info,&e,raid,anum,lhash)){
    errmsg = "can not create ra operator entry / delete reg entry"; goto done;
  }

  /* send email to user */
  if(inf){
    strncpy(info->ml.lang,e.plang,4); /* set user preferred lungage */
    if(aira_sendmail_adm2usr(&info->ml,&e,info->ml.admemail,lid,"raadm_accept_op.txt")){
      snprintf(tmp,254,"cannot send a email to ra operator (accID=%s) << %s",anum,CA_get_errstr());
      ENCERRLOG(info->ca.caname,tmp);
    }

    /* show list or search page */
    if(!strcmp(fm,"SearchReq")) aira_searchreq(info);
    else if(!strcmp(fm,"ListReq")) aira_listreq(info);
  }else{
    /* jump to airaenroll */
    if((buf=(char*)get_file2buf(get_html_name(info,AI_ENRJUMP_HTM),&sz))==NULL) goto done;

    cgi_check_lang(info);
    printf("Content-type: text/html\n\n");
    {
      AiVList v[] = {
	{"{LID}",lid},
	{"{FROM}",fm},
	{NULL,NULL}
      };
      cgi_print_content(buf,v);
    }
  }
  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_acceptdone : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}


/*//////////////////////////////////////////////////////////////////////////*/

int aira_listop(AiEnrollInfo *info){
  char *buf=NULL;
  int bg,lm,so;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aira_check_session(info)) goto done;

  lm   = atoi(cgi_find_query(&info->qc,"Lm"));
  bg   = atoi(cgi_find_query(&info->qc,"St"));
  so = atoi(cgi_find_query(&info->qc,"So"));

  if(lm <= 0) lm = 20;
  if(bg < 0) bg = 0;
  if(so <= 0) so = 2;

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_LISTOP_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  if(print_search_list(info,buf,"(cn=RAOP*)","ListOP",1,lm,bg,so,NULL,0)) goto done;

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_listop : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

int aira_showop(AiEnrollInfo *info){
  Cert *ct=NULL;
  char *cn,*fm,*buf=NULL;
  char *csbj=NULL,*p,sn[16]="",start[32]="",end[32]="";
  AiEntryInfo *e=NULL;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aira_check_session(info)) goto done;

  /* get user info */
  cn = cgi_find_query(&info->qc,"CN");
  fm = cgi_find_query(&info->qc,"Fm");
  if(*cn==0) { errmsg = "cannot get CN"; goto done; }

  /* get entry info from LDAP */
  if((e=cgi_entryinfo_new())==NULL){
    errmsg = "entryinfo memory allocate error"; goto done;
  }
  if(aira_get_raentry(info,e,cn,1,0)){
    errmsg = "cannot get RA operator entry info"; goto done;
  }
  if(e->ucert){
    if((ct = ASN1_read_cert(e->ucert))==NULL){
      errmsg = "certificate encode error"; goto done;
    }
    csbj = ct->subject;
    snprintf(sn,14,"%d",ct->serialNumber);
    if((p = stm2str(&ct->time.notBefore,3)) != NULL) strncpy(start,p,30);
    if((p = stm2str(&ct->time.notAfter,3)) != NULL) strncpy(end,p,30);
    e->ucert = NULL;
  }

  /* get active group names */
  aira_gcklist_buf(info,e,gpl,2048);

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_SHOWOP_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{RASUBJECT}",sbj},
      {"{RAKEYID}",keyid},
      {"{firstName}",e->firstName},
      {"{lastName}",e->lastName},
      {"{email}",e->mail},
      {"{unit}",e->unit},
      {"{title}",e->title},
      {"{tel1}",e->tel1},
      {"{tel2}",e->tel2},
      {"{fax1}",e->fax1},
      {"{fax2}",e->fax2},
      {"{employeeNumber}",e->employeeNumber},
      {"{postalCode}",e->postalCode},
      {"{postalAddress}",e->postalAddress},
      {"{labeledUri}",e->labeledUri},
      {"{description}",e->description},
      {"{st}",e->st},
      {"{SBJ}",csbj},
      {"{SERIAL}",sn},
      {"{NOTBEFORE}",start},
      {"{NOTAFTER}",end},
      {"{GROUPNAMES}",gpl},
      {"{FROM}",fm},
      {"{CN}",cn},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_showop : System error!!",errmsg,NULL);
  }
  cgi_entryinfo_free(e);
  Cert_free(ct);
  if(buf) free(buf);
  return ret;
}

int aira_modifyop(AiEnrollInfo *info){
  LDAP *ld = NULL;
  char path[MAXGROUP*64],cadn[512],radn[512];
  char *attr,*cn,*cp,num[16];
  char *tmp[4],*atval[MAXGROUP];
  int i,j,ret=-1;

  /* check session */
  if(aira_check_session(info)) goto done;

  /* get CA DN */
  if(aira_get_cadn(info,cadn,512)) goto done;

  memset(tmp,0,sizeof(char*)*4);
  memset(atval,0,sizeof(char*)*4);
  memset(path,0,MAXGROUP*64);
  memset(radn,0,512);

  /* get query */
  attr   = cgi_find_query(&info->qc,"attrName");
  tmp[0] = cgi_find_query(&info->qc,"updAttr1");
  tmp[1] = cgi_find_query(&info->qc,"updAttr2");
  cn     = cgi_find_query(&info->qc,"CN");

  snprintf(radn,510,"cn=%s, %s",cn,cadn);

  if(!strcmp(attr,"o")){
    /* update group list */
    for(i=j=0; i<MAXGROUP && info->grpname[i]; i++){
      atval[i] = &path[64*i];

      snprintf(num,16,"Gp%d",i);
      cp = cgi_find_query(&info->qc,num);
      if(cp && *cp){
	strncpy(atval[j],info->grpname[i],62); j++;
      }
    }
  }else{
    for(i=0; i<2; i++){
      atval[i] = &path[i*256];
      strncpy(atval[i],tmp[i],254); /* input code should be UTF8 */
    }

    if(!strcmp(attr,"cn")){ /* check firstName & lastName */
      i = cgi_get_langnum(info);
      if(((i==AI_LNG_JP)&&(__WITH_LANG != 1))||
	 ((i!=AI_LNG_JP)&&(__WITH_LANG == 1))){ /* invert text */
	if((cp = strchr(tmp[1],' ')) != NULL) {
	  *cp = 0; cp++;
	  snprintf(atval[1],254,"%s %s",cp,tmp[1]);
	}
      }
    }
  }

  if((ld = ldap_init(info->ld.ldaphost,info->ld.ldapport))==NULL) goto done;

  if(cgi_ldap_bind(ld,info->ld.ldapadmin,info->ld.ldapadminpwd,info->ld.ldapbind) != LDAP_SUCCESS)
    goto done;

  if(cgi_ldap_replace_attr(ld,radn,attr,atval)) goto done;

  aira_showop(info); /* ignore error */

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_modifyop : System error!!",errmsg,NULL);
  }
  return ret;
}

int aira_searchop(AiEnrollInfo *info){
  char *buf=NULL;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aira_check_session(info)) goto done;

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_SEARCHOP_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{RASUBJECT}",sbj},
      {"{RAKEYID}",keyid},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_searchop : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

int aira_searchopdone(AiEnrollInfo *info){
  char *buf=NULL,*text;
  char filter[256];
  int lm,bg,so;
  int st,ret=-1;
  off_t sz;

  /* check session */
  if(aira_check_session(info)) goto done;

  text = cgi_find_query(&info->qc,"Q");
  st   = atoi(cgi_find_query(&info->qc,"QT"));
  lm   = atoi(cgi_find_query(&info->qc,"Lm"));
  bg   = atoi(cgi_find_query(&info->qc,"St"));
  so   = atoi(cgi_find_query(&info->qc,"So"));
  if(*text==0){ errmsg = "can not search LDAP server (NULL input)."; goto done; }

  if(lm <= 0) lm = 20;
  if(bg < 0) bg = 0;
  if(so <= 0) so = 2;

  memset(filter,0,256);

  switch(st){
  case 1: snprintf(filter,254,"(cn=*%s*)",text); break;
  case 2: snprintf(filter,254,"(mail=*%s*)",text); break;
  case 3: snprintf(filter,254,"(cn=%s)",text); break;
  case 4: snprintf(filter,254,"(ou=*%s*)",text); break;
  case 5: snprintf(filter,254,"(telephoneNumber=*%s*)",text); break;
  case 6: snprintf(filter,254,"(st=%s)",text); break;
  }

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_SEARCHOPDONE_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  if(print_search_list(info,buf,filter,"SearchOP",1,lm,bg,so,text,st)) goto done;

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_searchopdone : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aira_viewcacert(AiEnrollInfo *info){
  Cert *ca = NULL;
  char *cp,*p,*t;
  char *buf=NULL;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aira_check_session(info)) goto done;

  /* get ca certificate */
  snprintf(path,256,"..%s%s.cer","/",info->ca.caname);
  if((ca = Cert_read_file(path))==NULL) goto done;

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_VIEWCACT_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  cp = buf;

  do{
    if((t=strstr(cp,"${"))==NULL){ printf("%s",cp); break; }
    *t=0; t++;

    if(!memcmp(t,"{CERTTEXT}",10)){
      printf("%s",cp); CERT_PRINT(ca); cp=t+10; continue;
    }else{
      p = "$";
    }
    printf("%s",cp);
    printf("%s",p);
    cp = t;
  }while(cp);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_viewcacert : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  Cert_free(ca);
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aira_getadmcert(AiEnrollInfo *info){
  Cert *ct=NULL;
  char *p1=NULL;
  int ret=-1;

  /* get user info */
  if((ct = cgi_get_clientcert())==NULL){
    errmsg = "cannot get client certificate"; goto done;
  }
  /* check session */
  if(aira_check_session(info)) goto done;

  /*---------- print out text -------------*/
  if((p1 = PEM_write_cert_buf(ct))==NULL) goto done;

  sbj[13]=0;
  printf("Content-type: application/x-x509-user-cert\n");
  printf("Content-Disposition: attachment; filename=\"%s.cer\"\n",&sbj[3]);
  printf("Content-length: %zu\n\n",strlen(p1));

  printf("%s",p1);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_getopcert : System error!!",errmsg,NULL);
  }
  Cert_free(ct);
  if(p1) free(p1);
  return ret;
}

int aira_getcacert(AiEnrollInfo *info){
  Cert *ca=NULL;
  char *p1=NULL;
  char path[256];
  int ret=-1;

  /* check session */
  if(aira_check_session(info)) goto done;

  /* get ca certificate */
  snprintf(path,256,"..%s%s.cer","/",info->ca.caname);
  if((ca = Cert_read_file(path))==NULL) goto done;

  /*---------- print out text -------------*/
  if((p1 = PEM_write_cert_buf(ca))==NULL) goto done;

  printf("Content-type: application/x-x509-ca-cert\n");
  printf("Content-Disposition: attachment; filename=\"%s.cer\"\n",info->ca.caname);
  printf("Content-length: %zu\n\n",strlen(p1));

  printf("%s",p1);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_getcacert : System error!!",errmsg,NULL);
  }
  Cert_free(ca);
  if(p1) free(p1);
  return ret;
}

int aira_getcrl(AiEnrollInfo *info){
  CRL *crl=NULL;
  char *p1=NULL;
  char path[256];
  int ret=-1;

  /* check session */
  if(aira_check_session(info)) goto done;

  /* get ca certificate */
#ifdef __WINDOWS__
  strncpy(path,"out-CRL-All.crl",256);
#else
  strncpy(path,"../out/out-CRL-All.crl",256);
#endif
  if((crl = CRL_read_file(path))==NULL){
    errmsg = "cannot open CRL file. please start CRL Publisher"; goto done;
  }

  /*---------- print out text -------------*/
  if((p1 = PEM_write_crl_buf(crl))==NULL) goto done;

  printf("Content-type: application/x-x509-crl\n");
  printf("Content-Disposition: attachment; filename=\"%s.crl\"\n",info->ca.caname);
  printf("Content-length: %zu\n\n",strlen(p1));

  printf("%s",p1);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aira_print_err(info,"aira_getcrl : System error!!",errmsg,NULL);
  }
  CRL_free(crl);
  if(p1) free(p1);
  return ret;
}
