/* aiop_op.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_io.h>
#include <aicrypto/ok_rand.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_pem.h>

#include "aienr_cgi.h"

#ifndef __WITH_LANG
#define __WITH_LANG	0
#endif /* FIXME */

/* HTML Template files */
#define AI_LOGIN_HTM      "ai_login.html"
#define AI_LOGOUT_HTM     "ai_logout.html"
#define AI_LISTREQ_HTM    "ai_listreq.html"
#define AI_LISTREQ_HTM    "ai_listreq.html"
#define AI_ERROR_HTM      "ai_error.html"
#define AI_VIEWOPCT_HTM   "ai_viewopcert.html"
#define AI_VIEWCACT_HTM   "ai_viewcacert.html"
#define AI_SHOWOP_HTM     "ai_showop.html"
#define AI_SEARCHREQ_HTM  "ai_searchreq.html"
#define AI_SEARCHREQDONE_HTM  "ai_searchreqdone.html"
#define AI_CONFIRMREQ_HTM "ai_confirmreq.html"
#define AI_LISTUSR_HTM    "ai_listuser.html"
#define AI_SHOWUSR_HTM    "ai_showuser.html"
#define AI_DELUSR_HTM     "ai_deluser.html"
#define AI_VIEWUSRCT_HTM  "ai_viewucert.html"
#define AI_RVKUSR_HTM     "ai_rvkuser.html"
#define AI_UPDUSR_HTM     "ai_upduser.html"
#define AI_UPDPIN_HTM     "ai_updpin.html"
#define AI_UPDPINDONE_HTM "ai_updpindone.html"
#define AI_SEARCHUSR_HTM  "ai_searchuser.html"
#define AI_SEARCHUSRDONE_HTM  "ai_searchuserdone.html"
#define AI_ENRJUMP_HTM    "ai_enrjump.html"

/*-- aiop_op.c local value --*/
char *sbj,*keyid,*sid,path[256];
char gpl[2048]="";
char *errmsg = "";

/*----------------------------------------------------------------------
  default : login operation. check ra operator certificate
----------------------------------------------------------------------*/
char *get_html_name(AiEnrollInfo *info,char *file){
  return cgi_html_path(info,"raop",file);
}

void aiop_get_glist(AiEnrollInfo *info,char *name,char *buf,int max,int gnum){
  AccList *al=NULL;
  char line[256];
  int i=0,j,k;

  if((al=cgi_findbynamegrp(info->list,name,"$$RAOP$$")) != NULL) {

    for(i=j=0; (unsigned char)(al->cn[i]^0xff)&&(i<MAXGROUP); i++){
      k = al->cn[i];
      snprintf(line,254,"<option value=\"%d\"%s>%s\n",k,
	       (k==gnum)?(" selected"):(""),info->grpname[k]);

      j+=strlen(line);

      if(j>max) break; /* overflow check */
      strcat(buf,line);
    }
  }
  if(i==0) strcpy(buf,"<option value=\"\"> ----- \n");
}

int aiop_check_gnum(AiEnrollInfo *info,char *name,int gnum){
  AccList *al=NULL;
  int i,k,ret = -1;

  if(gnum<0) goto done;
  if(gnum>=MAXGROUP) goto done;

  if((al=cgi_findbynamegrp(info->list,name,"$$RAOP$$"))==NULL) goto done;

  for(i=0; (unsigned char)(al->cn[i]^0xff)&&(i<MAXGROUP); i++){
    k = al->cn[i];
    if(k==gnum){ ret = 0; break; }
  }

  /* set current operator mail address */
  if(ret==0) strncpy(info->ml.opemail,info->grpemail[gnum],62);

done:
  return ret;
}

void aiop_set_grp2acc(AiEnrollInfo *info, AiEntryInfo *e, AccList *al){
  int i,j,k;
  /* get all group name */
  for(i=j=0; i<MAXGROUP && info->grpname[i]; i++){
    for(k=0; k<MAXGROUP && e->glist[k]; k++){
      if(!strcmp(info->grpname[i],e->glist[k])){
	al->cn[j] = i; j++; break;
      }
    }
  }
  al->cn[j] = (char)0xff; /* END */
}

int aiop_default(AiEnrollInfo *info){
  Cert *ct = NULL;
  CertStat *st = NULL; /* no longer used */
  LCMP *lcmp = NULL;
  AccList *al=NULL;
  AiUserInfo uinfo;
  AiEntryInfo *e=NULL;
  unsigned char sb[16],tmp2[32];
  char keyid[256],tmp[256],cn[32];
  char *buf=NULL,*gp,*cp,*sid,*sbj,*kid;
  int i=-1,ret=-1;
  off_t sz;

#ifdef __WINDOWS__
  WSADATA wsaData;

  if(WSAStartup(MAKEWORD(1, 1), &wsaData) != 0) {
    err = "failed to initialize winsock"; goto done;
  }
#endif

  /* get user info */
  if((ct = cgi_get_clientcert())==NULL){
    errmsg = "cannot get client certificate"; goto done;
  }
  sbj = cgi_find_cookie(&info->qc,"RASUBJECT");
  kid = cgi_find_cookie(&info->qc,"RAKEYID");
  sid = cgi_find_cookie(&info->qc,"RASessionID");

  if(*kid && *sid){
    /* load session file */
    if(cgi_load_session(info->sespath,&info->list)){
      errmsg = "cannot load session file"; goto done;
    }
    /* check session info */
    i = cgi_check_sessiongrp(info->list,kid,sid,"$$RAOP$$");
  }

  if(i != 0){
    /* cannot find valid session. check SSL client certificate */

    /* connect CA and check client certificate */
    if((lcmp=LCMP_init(info->ca.svname,info->ca.caport,info->ca.caname))==NULL){
      errmsg = "cannot initialize CA connection"; goto done;
    }

    /* check login */
    if((i=aiop_check_operator(info,ct,lcmp,&st))<0){
      errmsg = "cannot check client certificate"; goto done;
    }
    if(i==0){
      errmsg = "client certificate is not RA operator certificate"; goto done;
    }

    /* set cookie (one time session ID) */
    if((cp=ai_escape_url(ct->subject,strlen(ct->subject)))==NULL){
      errmsg = "cannot escape subject"; goto done;
    }
    printf("Set-Cookie: RASUBJECT=%s\n", cp);
    free(cp); cp=NULL;
    sbj = ct->subject;

	if(ct->pubkey){ /* apache */
      if(cs_get_keyhash(ct->pubkey,tmp2,&i)) goto done;
	}else{ /* IIS */
		memcpy(tmp2,ct->signature,20);
	}
    for(i=0,*keyid=0; i<20; i++){ sprintf(tmp,"%.2x",tmp2[i]); strcat(keyid,tmp); }
    printf("Set-Cookie: RAKEYID=%s\n", keyid);
    kid = keyid;

    if(RAND_bytes(sb,8)) goto done;
    printf("Set-Cookie: RASessionID=%.2x%.2x%.2x%.2x%.2x%.2x%.2x%.2x\n",
           sb[0],sb[1],sb[2],sb[3],sb[4],sb[5],sb[6],sb[7]);

    /* set new user session */
    if(cgi_load_session(info->sespath,&info->list)){
      errmsg = "cannot load session file"; goto done;
    }

    memset(&uinfo,0,sizeof(AiUserInfo));
    strncpy(uinfo.name,keyid,62);
    if(cgi_set_sessiongrp(info->sespath,&info->list,&uinfo,sb,"$$RAOP$$")){
      errmsg = "cannot set session : session save error."; goto done;
    }

    /********/
    /* get operator entry information */
    if((cp = strstr(sbj,"CN=RAOP"))==NULL){
      errmsg = "cannot get cn from certificate"; goto done;
    }
    strncpy(cn,&cp[3],30); cn[8]=0;

    /* get entry info from LDAP */
    if((e=cgi_entryinfo_new())==NULL){
      errmsg = "entryinfo memory allocate error"; goto done;
    }
    if(aira_get_raentry(info,e,cn,1,0)){
      errmsg = "cannot get RA operator entry info"; goto done;
    }

    /* update session with group info */
    if((al=cgi_findbynamegrp(info->list,uinfo.name,"$$RAOP$$"))==NULL){
      errmsg = "cannot find current session acclist"; goto done;
    }
    /* set group number to acc */
    aiop_set_grp2acc(info,e,al);
    strncpy(al->email,e->mail,62);

    if(cgi_update_session(info->sespath,al)){ errmsg = "cannot update session file"; goto done; }

    /* success - output log */
    sbj = strstr(sbj,"CN=RAOP"); sbj[11]=0; /* to be shorter */
    snprintf(tmp,254,"%s : success to login",sbj);
    ENCACCLOG(info->ca.caname,tmp);
  }
  sbj = strstr(sbj,"CN=RAOP"); sbj[11]=0; /* to be shorter */

  /* get group list */
  gp = cgi_find_query(&info->qc,"Gp");
  aiop_get_glist(info,kid,gpl,2048,atoi(gp));

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_LOGIN_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{RASUBJECT}",sbj},
      {"{RAKEYID}",kid},
      {"{GROUPNAMES}",gpl},
      {"{CN}",&sbj[3]},
      {"{Gp}",gp},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aira_default : System error!!",errmsg,lcmp);
  }
  if(lcmp){ LCMP_unbind_s(lcmp); LCMP_free(lcmp); }
  if(buf) free(buf);
  Cert_free(ct);
  CertStat_free_all(st);
  cgi_entryinfo_free(e);
#ifdef __WINDOWS__
  WSACleanup();
#endif
  return ret;
}

void aiop_print_err(AiEnrollInfo *info,char *err1,char *err2,LCMP *lc){
  char *buf,tmp[256]="";
  off_t sz;

  if((buf=(char*)get_file2buf(get_html_name(info,AI_ERROR_HTM),&sz)) != NULL) {
    AiVList v[] = {
      {"{ERROR1}",(err1)?(err1):("")},
      {"{ERROR2}",(err2)?(err2):("")},
      {"{OKERR}",(OK_get_error())?(OK_get_errstr()):("")},
      {"{LCMPERR}",tmp},
      {NULL,NULL}
    };
    if(lc && lc->op && lc->op->resultCode != LCMP_SUCCESS){
      snprintf(tmp,254,"%s (%d) => %s\n",LCMP_msg2str(lc->op->resultCode),lc->op->resultCode,
               lc->op->resultMsg);
    }
    cgi_print_content(buf,v);

    free(buf);
  }
}

int aiop_check_session(AiEnrollInfo *info){
  int ret=-1;

  /* get user info */
  sbj   = cgi_find_cookie(&info->qc,"RASUBJECT");
  keyid = cgi_find_cookie(&info->qc,"RAKEYID");
  sid   = cgi_find_cookie(&info->qc,"RASessionID");
  if(*sbj==0) { errmsg = "cannot get cookie (subject)"; goto done; }
  if(*keyid==0){ errmsg = "cannot get cookie (keyid)"; goto done; }
  if(*sid==0) { errmsg = "cannot get cookie (sessionID)"; goto done; }

  sbj = strstr(sbj,"CN=RAOP"); /* to be shorter */

  /* load session file */
  if(info->list==NULL && cgi_load_session(info->sespath,&info->list)){
    errmsg = "cannot load session file"; goto done;
  }
  /* check session info */
  if(cgi_check_sessiongrp(info->list,keyid,sid,"$$RAOP$$")){
    snprintf(path,254,"check session error : keyid:%s, sid:%s",keyid,sid);
    ENCERRLOG(info->ca.caname,path);
    errmsg = "check session error"; goto done;
  }

  ret = 0;
done:
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aiop_logout(AiEnrollInfo *info){
  char tmp[256],*buf=NULL;
  int ret=-1;
  off_t sz;

  sbj = cgi_find_cookie(&info->qc,"RASUBJECT");
  sbj = strstr(sbj,"CN=RAOP"); /* to be shorter */

  printf("Set-Cookie: RAKEYID=00; expires=Thu, 01-Jan-1970 00:00:00 GMT;\n");
  printf("Set-Cookie: RASessionID=00; expires=Thu, 01-Jan-1970 00:00:00 GMT;\n");

  /* success - output log */
  snprintf(tmp,254,"%s : success to logout",sbj);
  ENCACCLOG(info->ca.caname,tmp);

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_LOGOUT_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_logout : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int print_search_list(AiEnrollInfo *info, char *buf, char *filter, char *from, int raop,
                      int lmax, int start, int sort, char *query, int qtype, int gnum)
{
  char cbg[16],clm[16],qt[16],so[16],gp[8];
  char *cgi,*cp,*p,*t,*tmp=NULL;
  int i,ret=-1;

  sprintf(clm,"%d",lmax);
  sprintf(cbg,"%d",start);
  sprintf(qt,"%d",qtype);
  sprintf(so,"%d",sort);
  sprintf(gp,"%d",gnum);

#ifdef __WINDOWS__
  cgi  = "raopcgi.exe";
#else
  cgi  = "airaop";
#endif

  cp = buf;
  do{
    if((t=strstr(cp,"${"))==NULL){ printf("%s",cp); break; }
    *t=0; t++;

    if(!memcmp(t,"{RASUBJECT}",11)){
      p = sbj; t+=11;
    }else if(!memcmp(t,"{RAKEYID}",9)){
      p = keyid; t+=9;
    }else if(!memcmp(t,"{SEARCHLIST}",12)){
      printf("%s",cp);
      if((i=print_list_req(info,filter,from,raop,lmax,start,gnum,sort))<0){
        errmsg = "can not search LDAP server"; goto done;
      }
      cp=t+12; continue;
    }else if(!memcmp(t,"{PAGELIST}",10)){
      printf("%s",cp);
      print_page_req(cgi,from,lmax,start,sort,i,query,qtype,gnum);
      cp=t+10; continue;
    }else if(!memcmp(t,"{LMAX}",6)){
      p = clm; t+=6;
    }else if(!memcmp(t,"{SORT}",6)){
      p = so; t+=6;
    }else if(!memcmp(t,"{LST}",5)){
      p = cbg; t+=5;
    }else if(!memcmp(t,"{GROUPNAMES}",12)){
      p = gpl; t+=12;
    }else if(!memcmp(t,"{Gp}",4)){
      p = gp; t+=4;
    }else if(!memcmp(t,"{Op}",4)){
      p = from; t+=4;
    }else if(!memcmp(t,"{Q}",3)){
      if(query){
        if((tmp=ai_escape_url(query,strlen(query)))==NULL) goto done;
        p = tmp;
      }else{
        p = "";
      }
      t+=3;
    }else if(!memcmp(t,"{QT}",4)){
      p = qt; t+=4;
    }else{
      p = "$";
    }
    fputs(cp,stdout);
    fputs(p,stdout);
    cp = t;
  }while(cp);

  ret = 0;
 done:
  if(tmp) free(tmp);
  return ret;
}

int aiop_listreq(AiEnrollInfo *info){
  char *buf=NULL;
  int bg,lm,so,gn;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aiop_check_session(info)) goto done;

  lm = atoi(cgi_find_query(&info->qc,"Lm"));
  bg = atoi(cgi_find_query(&info->qc,"St"));
  gn = atoi(cgi_find_query(&info->qc,"Gp"));
  so = atoi(cgi_find_query(&info->qc,"So"));
  if(so <= 0) so = 2;
  if(lm <= 0) lm = 20;
  if(bg < 0) bg = 0;

  if(aiop_check_gnum(info,keyid,gn)){ errmsg = "bad group number"; goto done; }

  /* get group list */
  aiop_get_glist(info,keyid,gpl,2048,gn);

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_LISTREQ_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  if(print_search_list(info,buf,"(cn=REG*)","ListReq",2,lm,bg,so,NULL,0,gn)) goto done;

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aira_listreq : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aiop_confirmreq(AiEnrollInfo *info){
  AiEntryInfo *e = NULL;
  char *buf=NULL;
  char *anum,*fm,*gp;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aiop_check_session(info)) goto done;

  anum= cgi_find_query(&info->qc,"CN");
  fm  = cgi_find_query(&info->qc,"Fm");
  gp  = cgi_find_query(&info->qc,"Gp");

  if(aiop_check_gnum(info,keyid,atoi(gp))){ errmsg = "bad group number"; goto done; }

  if((e=cgi_entryinfo_new())==NULL){
    errmsg = "entryinfo memory allocate error"; goto done;
  }
  if(aira_get_raentry(info,e,anum,2,atoi(gp))){
    errmsg = "can not get request info from LDAP server"; goto done;
  }

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_CONFIRMREQ_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{RASUBJECT}",sbj},
      {"{RAKEYID}",keyid},
      {"{firstName}",e->firstName},
      {"{lastName}",e->lastName},
      {"{email}",e->mail},
      {"{unit}",e->unit},
      {"{title}",e->title},
      {"{tel1}",e->tel1},
      {"{tel2}",e->tel2},
      {"{fax1}",e->fax1},
      {"{fax2}",e->fax2},
      {"{employeeNumber}",e->employeeNumber},
      {"{postalCode}",e->postalCode},
      {"{postalAddress}",e->postalAddress},
      {"{labeledUri}",e->labeledUri},
      {"{description}",e->description},
      {"{plang}",e->plang},
      {"{chlg}",e->pin},
      {"{acceptnum}",anum},
      {"{acceptdate}",e->l},
      {"{Gp}",gp},
      {"{FROM}",fm},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_confirmreq : System error!!",errmsg,NULL);
  }
  cgi_entryinfo_free(e);
  if(buf) free(buf);
  return ret;
}


/*////////////////////////////////////////////////////////////////////////*/

int aiop_rejectdone(AiEnrollInfo *info){
  AiEntryInfo e;
  char tmp[256];
  char *fm,*anum;
  int inf,gn,ret=-1;

  memset(&e,0,sizeof(AiEntryInfo));

  /* check session */
  if(aiop_check_session(info)) goto done;

  /* get query */
  anum= cgi_find_query(&info->qc,"acceptnum");
  inf = atoi(cgi_find_query(&info->qc,"inform"));
  gn  = atoi(cgi_find_query(&info->qc,"Gp"));
  fm  = cgi_find_query(&info->qc,"FROM");
  e.mail = cgi_find_query(&info->qc,"email");
  e.plang = cgi_find_query(&info->qc,"plang");
  e.firstName= cgi_find_query(&info->qc,"firstName");
  e.lastName = cgi_find_query(&info->qc,"lastName");

  if(aiop_check_gnum(info,keyid,gn)){ errmsg = "bad group number"; goto done; }

  /* delete entry */
  if(aira_delete_raentry(info,anum,2,gn)){
    errmsg = "can not delete ra request entry"; goto done;
  }
  /* send email to user */
  strncpy(info->ml.lang,e.plang,4); /* set user preferred lungage */
  if(inf && aira_sendmail_adm2usr(&info->ml,&e,info->ml.opemail,anum,"raop_reject_user.txt")){
    snprintf(tmp,254,"cannot send an email to user (cn=%s) << %s",anum,info->ml.lang);
    ENCERRLOG(info->ca.caname,tmp);
  }

  /* show list or search page */
  if(!strcmp(fm,"SearchReq")) aiop_searchreq(info);
  else if(!strcmp(fm,"ListReq")) aiop_listreq(info);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_rejectdone : System error!!",errmsg,NULL);
  }
  return ret;
}

int aiop_add_userentry(AiEnrollInfo *info, AiEntryInfo *e, char *regid, char *lid, int gnum){
  LDAP *ld = NULL;
  LDAPMessage *res=NULL;
  void *ptr=NULL; /* BerElement */
  char **vals=NULL,*at=NULL;
  char filter[256],radn[512];
  /* char *attrs[4] = {info->ld.pinattr, info->ld.uquattr, NULL}; *//* XXX:0 */
  char *dn=NULL;
  int ok=-1;

  memset (radn,0,512);
  snprintf(radn,510,"cn=%s, ou=\"Registration\", %s",regid,info->grpbase[gnum]);
  snprintf(filter,254,"(&(%s=%s)(!(cn=REG*)))",info->ld.uquattr,e->queryattr);

  /* e->queryattr should be unique (v2.1.1)
   * so, same entry should not found.
   */
  if((ld = ldap_init(info->ld.ldaphost,info->ld.ldapport))==NULL) goto done;

  if(cgi_ldap_bind(ld,info->ld.ldapadmin,info->ld.ldapadminpwd,info->ld.ldapbind) != LDAP_SUCCESS)
    goto done;

#if 0
  if(ldap_search_s(ld,info->grpbase[gnum],LDAP_SCOPE_SUBTREE,filter,attrs,1,&res) != LDAP_SUCCESS)
    goto done;

  if(ldap_count_entries(ld,res) > 0){
    /* find current active user entry */
    ent = ldap_first_entry(ld,res);
    if((dn = ldap_get_dn(ld, ent))==NULL) goto done;
    if(cgi_mod_attr(ld,dn,LDAP_MOD_REPLACE,"st",e->st)!= LDAP_SUCCESS) goto done;
    if(cgi_mod_attr(ld,dn,LDAP_MOD_ADD,info->ld.lidattr,lid)!= LDAP_SUCCESS) goto done;

    /* check challenge pin is already set or not */
    if((at=ldap_first_attribute(ld,ent,(BERELEMENT**)&ptr))==NULL) goto done;
    while(at){
      if(!igcase_strcmp(at,info->ld.pinattr)){
	vals = ldap_get_values(ld,ent,at);
	if(vals==NULL||vals[0]==NULL)
	  if(cgi_mod_attr(ld,dn,LDAP_MOD_REPLACE,info->ld.pinattr,e->pin)!= LDAP_SUCCESS) goto done;

      }
      if(at){ ldap_memfree(at); at = NULL; }
      if(vals){ ldap_value_free(vals); vals = NULL; }
      at = ldap_next_attribute(ld,ent,ptr);
    }

  }else{
#endif
    /* create new user entry */
    if(aira_add_regentry(info,ld,info->grpbase[gnum],e,NULL,"",lid)) goto done;
#if 0
  }
#endif

  if(ldap_delete_s(ld, radn) != LDAP_SUCCESS) goto done;

  ok = 0;
done:
  if(at){ ldap_memfree(at); at = NULL; }
  if(vals){ ldap_value_free(vals); vals = NULL; }
  if(ptr) LDAP_BER_FREE(ptr,0);

  if(dn) ldap_memfree(dn);
  if(res) ldap_msgfree(res);
  if(ld) ldap_unbind(ld);
  return ok;
}

int aiop_acceptdone(AiEnrollInfo *info){
  AiEntryInfo e;
  char tmp[256];
  char lid[64],lhash[64],uq[32];
  char *fm,*cn,*buf=NULL;
  int inf,ret=-1;
  off_t sz;

  memset(&e,0,sizeof(AiEntryInfo));

  /* check session */
  if(aiop_check_session(info)) goto done;

  /* get query */
  cn  = cgi_find_query(&info->qc,"acceptnum");
  inf = atoi(cgi_find_query(&info->qc,"inform"));
  fm  = cgi_find_query(&info->qc,"FROM");
  e.firstName= cgi_find_query(&info->qc,"firstName");
  e.lastName = cgi_find_query(&info->qc,"lastName");
  e.mail     = cgi_find_query(&info->qc,"email");
  e.group    = cgi_find_query(&info->qc,"Gp");

  if(*e.firstName== 0){ errmsg = "post message not found"; goto done; }
  if(*e.lastName == 0){ errmsg = "post message not found"; goto done; }
  /*if(*e.mail     == 0){ errmsg = "post message not found"; goto done; }*/

  if(aiop_check_gnum(info,keyid,atoi(e.group))){ errmsg = "bad group number"; goto done; }

  e.unit  = cgi_find_query(&info->qc,"unit");
  e.title = cgi_find_query(&info->qc,"title");
  e.tel1  = cgi_find_query(&info->qc,"tel1");
  e.tel2  = cgi_find_query(&info->qc,"tel2");
  e.fax1  = cgi_find_query(&info->qc,"fax1");
  e.fax2  = cgi_find_query(&info->qc,"fax2");
  e.employeeNumber = cgi_find_query(&info->qc,"employeeNumber");
  e.postalCode     = cgi_find_query(&info->qc,"postalCode");
  e.postalAddress  = cgi_find_query(&info->qc,"postalAddress");
  e.labeledUri     = cgi_find_query(&info->qc,"labeledUri");
  e.description    = cgi_find_query(&info->qc,"description");
  e.plang = cgi_find_query(&info->qc,"plang");
  e.pin   = cgi_find_query(&info->qc,"chlg");
  e.st    = "WAIT_ISSUE";

  /* replace html symbol tags '<' or '>' to space char */
  ai_symbol_esc_sp(e.firstName);
  ai_symbol_esc_sp(e.lastName);
  ai_symbol_esc_sp(e.mail);
  ai_symbol_esc_sp(e.group);
  ai_symbol_esc_sp(e.unit);
  ai_symbol_esc_sp(e.title);
  ai_symbol_esc_sp(e.tel1);
  ai_symbol_esc_sp(e.tel2);
  ai_symbol_esc_sp(e.fax1);
  ai_symbol_esc_sp(e.fax2);
  ai_symbol_esc_sp(e.employeeNumber);
  ai_symbol_esc_sp(e.postalCode);
  ai_symbol_esc_sp(e.postalAddress);
  ai_symbol_esc_sp(e.labeledUri);
  ai_symbol_esc_sp(e.description);

  /* generate licenseID & hashed value */
  if(gen_license_id2(e.mail,lid)){
    errmsg = "can not generate license ID"; goto done;
  }
  if(cgi_gen_hash(lid,lhash)){
    errmsg = "can not get hashed license ID"; goto done;
  }
  /* get unique search query */
  aira_lid2querykey(lid,uq);
  e.queryattr = uq;

  /* create new raop entry & delete request entry */
  if(aiop_add_userentry(info,&e,cn,lhash,atoi(e.group))){
    errmsg = "can not create user entry / delete reg entry"; goto done;
  }

  /* send email to user */
  if(inf){
    strncpy(info->ml.lang,e.plang,4); /* set user preferred lungage */
    if(aira_sendmail_adm2usr(&info->ml,&e,info->ml.opemail,lid,"raop_accept_user.txt")){
      snprintf(tmp,254,"cannot send an email to user (cn=%s) << %s",cn,info->ml.lang);
      ENCERRLOG(info->ca.caname,tmp);
    }

    /* show list or search page */
    if(!strcmp(fm,"SearchReq")) aiop_searchreq(info);
    else if(!strcmp(fm,"ListReq")) aiop_listreq(info);
  }else{
    /* jump to airaenroll */
    if((buf=(char*)get_file2buf(get_html_name(info,AI_ENRJUMP_HTM),&sz))==NULL) goto done;

    cgi_check_lang(info);
    printf("Content-type: text/html\n\n");
    {
      AiVList v[] = {
        {"{LID}",lid},
        {"{FROM}",fm},
        {"{Gp}",e.group},
        {NULL,NULL}
      };
      cgi_print_content(buf,v);
    }
  }
  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_acceptdone : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aiop_viewopcert(AiEnrollInfo *info){
  Cert *ct = NULL;
  char *cp,*p,*t;
  char *buf=NULL;
  int ret=-1;
  off_t sz;

  /* get user info */
  if((ct = cgi_get_clientcert())==NULL){
    errmsg = "cannot get client certificate"; goto done;
  }
  /* check session */
  if(aiop_check_session(info)) goto done;

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_VIEWOPCT_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  cp = buf;
  do{
    if((t=strstr(cp,"${"))==NULL){ printf("%s",cp); break; }
    *t=0; t++;

    if(!memcmp(t,"{CERTTEXT}",10)){
      printf("%s",cp); CERT_PRINT(ct); cp=t+10; continue;
    }else{
      p = "$";
    }
    printf("%s",cp);
    printf("%s",p);
    cp = t;
  }while(cp);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_viewopcert : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  Cert_free(ct);
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aiop_showop(AiEnrollInfo *info){
  Cert *ct=NULL;
  char cn[32],*buf=NULL;
  char *csbj=NULL,*p,sn[16]="",start[32]="",end[32]="";
  AiEntryInfo *e=NULL;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aiop_check_session(info)) goto done;

  /* get user info */
  if((ct = cgi_get_clientcert())==NULL){
    errmsg = "cannot get client certificate"; goto done;
  }
  if((p = strstr(ct->subject,"CN=RAOP"))==NULL){
    errmsg = "cannot get cn from certificate"; goto done;
  }
  strncpy(cn,&p[3],30); cn[8]=0;

  /* get entry info from LDAP */
  if((e=cgi_entryinfo_new())==NULL){
    errmsg = "entryinfo memory allocate error"; goto done;
  }
  if(aira_get_raentry(info,e,cn,1,0)){
    errmsg = "cannot get RA operator entry info"; goto done;
  }
  csbj = ct->subject;
  snprintf(sn,14,"%d",ct->serialNumber);
  if((p = stm2str(&ct->time.notBefore,3)) != NULL) strncpy(start,p,30);
  if((p = stm2str(&ct->time.notAfter,3)) != NULL) strncpy(end,p,30);


  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_SHOWOP_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{RASUBJECT}",sbj},
      {"{RAKEYID}",keyid},
      {"{firstName}",e->firstName},
      {"{lastName}",e->lastName},
      {"{email}",e->mail},
      {"{unit}",e->unit},
      {"{title}",e->title},
      {"{tel1}",e->tel1},
      {"{tel2}",e->tel2},
      {"{fax1}",e->fax1},
      {"{fax2}",e->fax2},
      {"{employeeNumber}",e->employeeNumber},
      {"{postalCode}",e->postalCode},
      {"{postalAddress}",e->postalAddress},
      {"{labeledUri}",e->labeledUri},
      {"{description}",e->description},
      {"{st}",e->st},
      {"{SBJ}",csbj},
      {"{SERIAL}",sn},
      {"{NOTBEFORE}",start},
      {"{NOTAFTER}",end},
      {"{CN}",cn},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_showop : System error!!",errmsg,NULL);
  }
  cgi_entryinfo_free(e);
  Cert_free(ct);
  if(buf) free(buf);
  return ret;
}

int aiop_modifyop(AiEnrollInfo *info){
  LDAP *ld = NULL;
  char path[1024],cadn[512],radn[512];
  char *attr,*cn,*cp;
  char *tmp[4],*atval[4];
  int i,ret=-1;

  memset(tmp,0,sizeof(char*)*4);
  memset(atval,0,sizeof(char*)*4);
  memset(path,0,sizeof(char)*1024);
  memset(radn,0,512);

  /* check session */
  if(aiop_check_session(info)) goto done;

  /* get query */
  attr   = cgi_find_query(&info->qc,"attrName");
  tmp[0] = cgi_find_query(&info->qc,"updAttr1");
  tmp[1] = cgi_find_query(&info->qc,"updAttr2");
  cn     = cgi_find_query(&info->qc,"CN");

  /* get CA DN */
  if(aira_get_cadn(info,cadn,512)) goto done;

  memset (radn,0,512);
  snprintf(radn,510,"cn=%s, %s",cn,cadn);

  for(i=0; i<2; i++){
    atval[i] = &path[i*256];
    strncpy(atval[i],tmp[i],254); /* input code should be UTF8 */
  }
  if(!strcmp(attr,"cn")){ /* check firstName & lastName */
    i = cgi_get_langnum(info);
    if(((i==AI_LNG_JP)&&(__WITH_LANG != 1))||
       ((i!=AI_LNG_JP)&&(__WITH_LANG == 1))){ /* invert text */
      if((cp = strchr(tmp[1],' ')) != NULL) {
	*cp = 0; cp++;
	snprintf(atval[1],254,"%s %s",cp,tmp[1]);
      }
    }
  }

  if((ld = ldap_init(info->ld.ldaphost,info->ld.ldapport))==NULL) goto done;

  if(cgi_ldap_bind(ld,info->ld.ldapadmin,info->ld.ldapadminpwd,info->ld.ldapbind) != LDAP_SUCCESS)
    goto done;

  if(cgi_ldap_replace_attr(ld,radn,attr,atval)) goto done;

  aiop_showop(info); /* ignore error */

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_modifyop : System error!!",errmsg,NULL);
  }
  return ret;
}

int aiop_modifyuser(AiEnrollInfo *info){
  LDAP *ld = NULL;
  LDAPMessage *res=NULL, *ent;
  char path[1024],filter[256];
  char *attrs[2] = {"mail", NULL};
  char *qu,*attr,*dn=NULL,*tmp[4],*atval[4];
  int i,gn,ret=-1;

  memset(tmp,0,sizeof(char*)*4);
  memset(atval,0,sizeof(char*)*4);
  memset(path,0,sizeof(char)*1024);

  /* check session */
  if(aiop_check_session(info)) goto done;

  /* get query */
  attr   = cgi_find_query(&info->qc,"attrName");
  tmp[0] = cgi_find_query(&info->qc,"updAttr1");
  tmp[1] = cgi_find_query(&info->qc,"updAttr2");
  qu     = cgi_find_query(&info->qc,"Qu");
  gn     = atoi(cgi_find_query(&info->qc,"Gp"));

  if(aiop_check_gnum(info,keyid,gn)){ errmsg = "bad group number"; goto done; }

  /* filter */
  snprintf(filter,254,"(&(%s=%s)(!(cn=REG*)))",info->ld.uquattr,qu);

  for(i=0; i<2; i++){
    atval[i] = &path[i*256];
    strncpy(atval[i],tmp[i],254); /* input code should be UTF8 */
  }
  /* if attr is same as query value (ex. mail), need to replace query value */
  if(!strcmp(attr,info->ld.uquattr)){
    for(i=0; i<QUERY_MAX; i++){
      if(!strncmp("Qu",info->qc.postqs[i].key,126)){
	info->qc.postqs[i].value = atval[0]; break;
      }
    }
  }

  /* update ldap entry */
  if((ld = ldap_init(info->ld.ldaphost,info->ld.ldapport))==NULL) goto done;

  if(cgi_ldap_bind(ld,info->ld.ldapadmin,info->ld.ldapadminpwd,info->ld.ldapbind) != LDAP_SUCCESS)
    goto done;

  if(ldap_search_s(ld,info->grpbase[gn],LDAP_SCOPE_SUBTREE,filter,attrs,1,&res) != LDAP_SUCCESS)
    goto done;

  if(ldap_count_entries(ld,res) <= 0){ errmsg = "cannot search user entry"; goto done; }

  /* find current active user entry */
  ent = ldap_first_entry(ld,res);
  if((dn = ldap_get_dn(ld, ent))==NULL) goto done;
  
  if(cgi_ldap_replace_attr(ld,dn,attr,atval)) goto done;

  aiop_showuser(info); /* ignore error */

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_modifyuser : System error!!",errmsg,NULL);
  }
  if(dn) ldap_memfree(dn);
  if(res) ldap_msgfree(res);
  if(ld) ldap_unbind(ld);
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aiop_searchreq(AiEnrollInfo *info){
  char *gp,*buf=NULL;
  int gn,ret=-1;
  off_t sz;

  /* check session */
  if(aiop_check_session(info)) goto done;

  gp = cgi_find_query(&info->qc,"Gp");
  gn = atoi(gp);

  if(aiop_check_gnum(info,keyid,gn)){ errmsg = "bad group number"; goto done; }

  /* get group list */
  aiop_get_glist(info,keyid,gpl,2048,gn);

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_SEARCHREQ_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{RASUBJECT}",sbj},
      {"{RAKEYID}",keyid},
      {"{GROUPNAMES}",gpl},
      {"{Gp}",gp},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_searchreq : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

int aiop_searchreqdone(AiEnrollInfo *info){
  char *buf=NULL,*text;
  char filter[256];
  int bg,lm,so,gn;
  int st,ret=-1;
  off_t sz;

  /* check session */
  if(aiop_check_session(info)) goto done;

  text = cgi_find_query(&info->qc,"Q");
  st   = atoi(cgi_find_query(&info->qc,"QT"));
  lm   = atoi(cgi_find_query(&info->qc,"Lm"));
  bg   = atoi(cgi_find_query(&info->qc,"St"));
  gn   = atoi(cgi_find_query(&info->qc,"Gp"));
  so   = atoi(cgi_find_query(&info->qc,"So"));

  if(*text==0){ errmsg = "can not search LDAP server (NULL input)."; goto done; }

  if(so <= 0) so = 2;
  if(lm <= 0) lm = 20;
  if(bg < 0) bg = 0;

  if(aiop_check_gnum(info,keyid,gn)){ errmsg = "bad group number"; goto done; }

  /* get group list */
  aiop_get_glist(info,keyid,gpl,2048,gn);

  memset(filter,0,256);

  switch(st){
  case 1: snprintf(filter,254,"(cn=*%s*)",text); break;
  case 2: snprintf(filter,254,"(mail=*%s*)",text); break;
  case 3: snprintf(filter,254,"(cn=%s)",text); break;
  case 4: snprintf(filter,254,"(ou=*%s*)",text); break;
  case 5: snprintf(filter,254,"(telephoneNumber=*%s*)",text); break;
  }

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_SEARCHREQDONE_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  if(print_search_list(info,buf,filter,"SearchReq",2,lm,bg,so,text,st,gn)) goto done;

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_searchreqdone : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aiop_listuser(AiEnrollInfo *info){
  char *buf=NULL;
  char *filter = "(&(objectClass=inetOrgPerson)(!(cn=REG*)))";
  int bg,lm,gn,so;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aiop_check_session(info)) goto done;

  lm = atoi(cgi_find_query(&info->qc,"Lm"));
  bg = atoi(cgi_find_query(&info->qc,"St"));
  gn = atoi(cgi_find_query(&info->qc,"Gp"));
  so = atoi(cgi_find_query(&info->qc,"So"));
  if(so <= 0) so = 1;
  if(lm <= 0) lm = 20;
  if(bg < 0) bg = 0;

  if(aiop_check_gnum(info,keyid,gn)){ errmsg = "bad group number"; goto done; }

  /* get group list */
  aiop_get_glist(info,keyid,gpl,2048,gn);

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_LISTUSR_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  if(print_search_list(info,buf,filter,"ListUsr",3,lm,bg,so,NULL,0,gn)) goto done;

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_listuser : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

/*//////////////////////////////////////////////////////////////////////////*/

int aiop_listupdreq(AiEnrollInfo *info){
  char *buf=NULL;
  int bg,lm,gn,so;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aiop_check_session(info)) goto done;

  lm = atoi(cgi_find_query(&info->qc,"Lm"));
  bg = atoi(cgi_find_query(&info->qc,"St"));
  gn = atoi(cgi_find_query(&info->qc,"Gp"));
  so = atoi(cgi_find_query(&info->qc,"So"));
  if(so <= 0) so = 1;
  if(lm <= 0) lm = 20;
  if(bg < 0) bg = 0;

  if(aiop_check_gnum(info,keyid,gn)){ errmsg = "bad group number"; goto done; }

  /* get group list */
  aiop_get_glist(info,keyid,gpl,2048,gn);

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_LISTUSR_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  if(print_search_list(info,buf,"(st=WAIT_UPDATE)","ListUpdReq",3,lm,bg,so,NULL,0,gn)) goto done;

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_listupdreq : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

int aiop_listrvkreq(AiEnrollInfo *info){
  char *buf=NULL;
  int bg,lm,gn,so;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aiop_check_session(info)) goto done;

  lm = atoi(cgi_find_query(&info->qc,"Lm"));
  bg = atoi(cgi_find_query(&info->qc,"St"));
  gn = atoi(cgi_find_query(&info->qc,"Gp"));
  so = atoi(cgi_find_query(&info->qc,"So"));
  if(so <= 0) so = 1;
  if(lm <= 0) lm = 20;
  if(bg < 0) bg = 0;

  if(aiop_check_gnum(info,keyid,gn)){ errmsg = "bad group number"; goto done; }

  /* get group list */
  aiop_get_glist(info,keyid,gpl,2048,gn);

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_LISTUSR_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  if(print_search_list(info,buf,"(st=WAIT_REVOKE)","ListRvkReq",3,lm,bg,so,NULL,0,gn)) goto done;

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_listrvkreq : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aiop_showuser(AiEnrollInfo *info){
  Cert *ct=NULL;
  char *fm,*qu,*gp,*buf=NULL;
  char *csbj=NULL,*p,sn[16]="",start[32]="",end[32]="";
  AiEntryInfo *e=NULL;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aiop_check_session(info)) goto done;

  fm = cgi_find_query(&info->qc,"Fm");
  qu = cgi_find_query(&info->qc,"Qu");
  gp = cgi_find_query(&info->qc,"Gp");
  if(*qu==0) { errmsg = "cannot get query info"; goto done; }

  if(aiop_check_gnum(info,keyid,atoi(gp))){ errmsg = "bad group number"; goto done; }

  /* get entry info from LDAP */
  if((e=cgi_entryinfo_new())==NULL){
    errmsg = "entryinfo memory allocate error"; goto done;
  }
  if(aira_get_raentry(info,e,qu,3,atoi(gp))){
    errmsg = "cannot get user entry info"; goto done;
  }
  if(e->ucert){
    if((ct = ASN1_read_cert(e->ucert))==NULL){
      errmsg = "certificate encode error"; goto done;
    }
    csbj = ct->subject;
    snprintf(sn,14,"%d",ct->serialNumber);
    if((p = stm2str(&ct->time.notBefore,3)) != NULL) strncpy(start,p,30);
    if((p = stm2str(&ct->time.notAfter,3)) != NULL) strncpy(end,p,30);
    e->ucert = NULL;
  }

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_SHOWUSR_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{RASUBJECT}",sbj},
      {"{RAKEYID}",keyid},
      {"{firstName}",e->firstName},
      {"{lastName}",e->lastName},
      {"{email}",e->mail},
      {"{unit}",e->unit},
      {"{title}",e->title},
      {"{tel1}",e->tel1},
      {"{tel2}",e->tel2},
      {"{fax1}",e->fax1},
      {"{fax2}",e->fax2},
      {"{employeeNumber}",e->employeeNumber},
      {"{postalCode}",e->postalCode},
      {"{postalAddress}",e->postalAddress},
      {"{labeledUri}",e->labeledUri},
      {"{description}",e->description},
      {"{st}",e->st},
      {"{SBJ}",csbj},
      {"{SERIAL}",sn},
      {"{NOTBEFORE}",start},
      {"{NOTAFTER}",end},
      {"{Gp}",gp},
      {"{Qu}",qu},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_showuser : System error!!",errmsg,NULL);
  }
  cgi_entryinfo_free(e);
  Cert_free(ct);
  if(buf) free(buf);
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aiop_viewusercert(AiEnrollInfo *info){
  Cert *ct = NULL;
  char *cp,*p,*t;
  char *fm,*qu,*buf=NULL;
  int gn,ret=-1;
  off_t sz;

  /* check session */
  if(aiop_check_session(info)) goto done;

  fm = cgi_find_query(&info->qc,"Fm");
  qu = cgi_find_query(&info->qc,"Qu");
  gn = atoi(cgi_find_query(&info->qc,"Gp"));
  if(*qu==0) { errmsg = "cannot get query info"; goto done; }

  if(aiop_check_gnum(info,keyid,gn)){ errmsg = "bad group number"; goto done; }

  /* get user certificate */
  if((ct=aira_get_entrycert(info,qu,3,gn))==NULL){
    errmsg = "cannot get user certificate"; goto done;
  }

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_VIEWUSRCT_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  cp = buf;
  do{
    if((t=strstr(cp,"${"))==NULL){ printf("%s",cp); break; }
    *t=0; t++;

    if(!memcmp(t,"{RASUBJECT}",11)){
      p = sbj; t+=11;
    }else if(!memcmp(t,"{RAKEYID}",9)){
      p = keyid; t+=9;
    }else if(!memcmp(t,"{CERTTEXT}",10)){
      printf("%s",cp); CERT_PRINT(ct); cp=t+10; continue;
    }else{
      p = "$";
    }
    printf("%s",cp);
    printf("%s",p);
    cp = t;
  }while(cp);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_viewusercert : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  Cert_free(ct);
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aiop_revokeuser(AiEnrollInfo *info){
  Cert *ct=NULL;
  char *csbj=NULL,*p,sn[16],start[32],end[32],em[128],lg[8];
  char *qu,*fm,*gp,*buf=NULL;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aiop_check_session(info)) goto done;

  qu = cgi_find_query(&info->qc,"Qu");
  fm = cgi_find_query(&info->qc,"Fm");
  gp = cgi_find_query(&info->qc,"Gp");

  if(aiop_check_gnum(info,keyid,atoi(gp))){ errmsg = "bad group number"; goto done; }

  /* get end user certificate */
  if((ct=aira_get_entrycert_(info,qu,3,atoi(gp),em,lg))==NULL){
    errmsg = "cannot get oprator certificate"; goto done;
  }
  csbj = ct->subject;
  snprintf(sn,14,"%d",ct->serialNumber);
  if((p = stm2str(&ct->time.notBefore,3)) != NULL) strncpy(start,p,30);
  if((p = stm2str(&ct->time.notAfter,3)) != NULL) strncpy(end,p,30);

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_RVKUSR_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{RASUBJECT}",sbj},
      {"{RAKEYID}",keyid},
      {"{sbj}",csbj},
      {"{email}",em},
      {"{SERIAL}",sn},
      {"{notbefore}",start},
      {"{notafter}",end},
      {"{plang}",lg},
      {"{Qu}",qu},
      {"{Gp}",gp},
      {"{FROM}",fm},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_revokeuser : System error!!",errmsg,NULL);
  }
  Cert_free(ct);
  if(buf) free(buf);
  return ret;
}

int aiop_rvkuserdone(AiEnrollInfo *info){
  LCMP *lcmp = NULL;
  char *ml,*fm,*qu,*em,*sn,*rs,*lg,cs[256];
  int cl,code,gn,mode,ret = -1;

#ifdef __WINDOWS__
  WSADATA wsaData;

  if(WSAStartup(MAKEWORD(1, 1), &wsaData) != 0) {
    err = "failed to initialize winsock"; goto done;
  }
#endif

  /* check session */
  if(aiop_check_session(info)) goto done;

  qu = cgi_find_query(&info->qc,"Qu");
  em = cgi_find_query(&info->qc,"email");
  lg = cgi_find_query(&info->qc,"plang");
  sn = cgi_find_query(&info->qc,"SERIAL");
  rs = cgi_find_query(&info->qc,"Reason");
  fm = cgi_find_query(&info->qc,"Fm");
  gn = atoi(cgi_find_query(&info->qc,"Gp"));
  cl = atoi(cgi_find_query(&info->qc,"Cancel"));
  if(sn==NULL){ errmsg = "cannot get serial number"; goto done; }
  if(rs==NULL){ errmsg = "cannot get revoke reason"; goto done; }
  if(fm==NULL){ errmsg = "cannot get fm"; goto done; }

  if(aiop_check_gnum(info,keyid,gn)){ errmsg = "bad group number"; goto done; }

  if(cl==1){
    /* cansel WAIT_REVOKE */
    if(aira_change_entryst(info,qu,"ACTIVE",NULL,3,gn)){
      errmsg = "cannot update ldap entry"; goto done;
    }
    ml = "raop_cancel_rvk.txt";
  }else{
    /* connect CA and revoke Cert */
    if(!info->offlineca){
      if((lcmp=LCMP_init(info->ca.svname,info->ca.caport,info->ca.caname))==NULL){
	errmsg = "cannot initialize CA connection"; goto done;
      }
      if(info->ca.usessl){
	if(LCMP_set_ssl(lcmp,info->ca.store,info->ca.certid,info->ca.clctpw,info->ca.vfycert)){
	  errmsg = "cannot set SSL certificate"; goto done;
	}
	mode = LCMP_OPBRQ_ATSSLCL;
      }
      if(LCMP_bind_s(lcmp,info->ca.userid,info->ca.pwd,mode)) goto done;
    
      if((code=LCMP_cert_s(lcmp,LCMP_OPCT_RVKCERT,atoi(sn),atoi(rs),NULL,NULL))<0) goto done;
      
      if(code){
	sprintf(cs,"CA returns error code : %.8x",code);
	errmsg = cs; goto done;
      }
    }

    /* update ldap status */
    if(aira_change_entryst(info,qu,"REVOKED",NULL,3,gn)){
      errmsg = "cannot update ldap entry"; goto done;
    }
    ml = "raop_revoke_user.txt";
  }
  
  /* send email to user */
  if(*info->ml.smtphost && *info->ml.admemail){
    AiEntryInfo e;
    e.mail = em;

    if(*lg) strncpy(info->ml.lang,lg,4);
    else cgi_get_deflang(info->ml.lang); /* version compatible */
    
    if(aira_sendmail_adm2usr(&info->ml,&e,info->ml.opemail,"--",ml)){
      snprintf(cs,254,"cannot send a email to user (em=%s) << %s",em,info->ml.lang);
      ENCERRLOG(info->ca.caname,cs);
    }
  }

  /* show list or search page */
  if(!strcmp(fm,"SearchUsr")) aiop_searchuser(info);
  else if(!strcmp(fm,"ListUsr")) aiop_listuser(info);
  else if(!strcmp(fm,"ListRvkReq")) aiop_listrvkreq(info);
  else if(!strcmp(fm,"ListUpdReq")) aiop_listupdreq(info);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aira_rvkuserdone : System error!!",errmsg,lcmp);
  }
  if(lcmp){ LCMP_unbind_s(lcmp); LCMP_free(lcmp); }
#ifdef __WINDOWS__
  WSACleanup();
#endif
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aiop_updateuser(AiEnrollInfo *info){
  AiEntryInfo *e = NULL;
  Cert *ct = NULL;
  char *fm,*qu,*gp,sn[16],*buf=NULL;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aiop_check_session(info)) goto done;

  qu = cgi_find_query(&info->qc,"Qu");
  fm = cgi_find_query(&info->qc,"Fm");
  gp = cgi_find_query(&info->qc,"Gp");

  if(aiop_check_gnum(info,keyid,atoi(gp))){ errmsg = "bad group number"; goto done; }

  if((e=cgi_entryinfo_new())==NULL){
    errmsg = "entryinfo memory allocate error"; goto done;
  }
  if(aira_get_raentry(info,e,qu,3,atoi(gp))){
    errmsg = "can not get request info from LDAP server"; goto done;
  }
  if((ct = ASN1_read_cert(e->ucert))==NULL){
    errmsg = "RA operator cert decode error"; goto done;
  }
  e->ucert = NULL;
  snprintf(sn,14,"%d",ct->serialNumber);

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_UPDUSR_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{RASUBJECT}",sbj},
      {"{RAKEYID}",keyid},
      {"{firstName}",e->firstName},
      {"{lastName}",e->lastName},
      {"{email}",e->mail},
      {"{unit}",e->unit},
      {"{title}",e->title},
      {"{tel1}",e->tel1},
      {"{tel2}",e->tel2},
      {"{fax1}",e->fax1},
      {"{fax2}",e->fax2},
      {"{employeeNumber}",e->employeeNumber},
      {"{postalCode}",e->postalCode},
      {"{postalAddress}",e->postalAddress},
      {"{labeledUri}",e->labeledUri},
      {"{description}",e->description},
      {"{plang}",e->plang},
      {"{st}",e->st},
      {"{Gp}",gp},
      {"{Qu}",qu},
      {"{SERIAL}",sn},
      {"{FROM}",fm},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_updateuser : System error!!",errmsg,NULL);
  }
  cgi_entryinfo_free(e);
  Cert_free(ct);
  if(buf) free(buf);
  return ret;
}

int aiop_upduserdone(AiEnrollInfo *info){
  AiEntryInfo e;
  LCMP *lcmp = NULL;
  CertStat *st = NULL;
  char *ml,*fm,*qu,*gp,*sn,*buf = NULL;
  char lid[256],lhash[64],que[32];
  int i,cl,inf,mode,ret = -1;
  off_t sz;

#ifdef __WINDOWS__
  WSADATA wsaData;

  if(WSAStartup(MAKEWORD(1, 1), &wsaData) != 0) {
    err = "failed to initialize winsock"; goto done;
  }
#endif
  memset(&e,0,sizeof(AiEntryInfo));

  /* check session */
  if(aiop_check_session(info)) goto done;

  qu = cgi_find_query(&info->qc,"Qu");
  fm = cgi_find_query(&info->qc,"Fm");
  sn = cgi_find_query(&info->qc,"SERIAL");
  gp = cgi_find_query(&info->qc,"Gp");
  cl = atoi(cgi_find_query(&info->qc,"Cancel"));
  inf = atoi(cgi_find_query(&info->qc,"inform"));
  e.mail = cgi_find_query(&info->qc,"email");
  e.plang = cgi_find_query(&info->qc,"plang");
  e.group = gp;
  e.queryattr = qu;

  if(qu==NULL){ errmsg = "cannot get query info"; goto done; }
  if(sn==NULL){ errmsg = "cannot get serial number"; goto done; }
  if(fm==NULL){ errmsg = "cannot get fm"; goto done; }

  if(aiop_check_gnum(info,keyid,atoi(gp))){ errmsg = "bad group number"; goto done; }

  if(cl==1){
    /* cansel WAIT_UPDATE */
    if(aira_change_entryst(info,qu,"ACTIVE",NULL,3,atoi(gp))){
      errmsg = "cannot update ldap entry"; goto done;
    }
    ml = "raop_cancel_upd.txt"; inf = cl;
  }else{
    /* connect CA and revoke Cert */
    if(!info->offlineca){
      if((lcmp=LCMP_init(info->ca.svname,info->ca.caport,info->ca.caname))==NULL){
	errmsg = "cannot initialize CA connection"; goto done;
      }
      if(info->ca.usessl){
	if(LCMP_set_ssl(lcmp,info->ca.store,info->ca.certid,info->ca.clctpw,info->ca.vfycert)){
	  errmsg = "cannot set SSL certificate"; goto done;
	}
	mode = LCMP_OPBRQ_ATSSLCL;
      }
      if(LCMP_bind_s(lcmp,info->ca.userid,info->ca.pwd,mode)) goto done;
      
      if(LCMP_list_s(lcmp,"*",atoi(sn))) goto done;
      if((st=LCMP_get_statptr(lcmp))==NULL) goto done;
      
      /* if old certificate is active, revoke it */
      if((st->state & (STAT_EXPIRED|STAT_REVOKED)) == 0){
	if((i=LCMP_cert_s(lcmp,LCMP_OPCT_RVKCERT,atoi(sn),0,NULL,NULL))<0) goto done;
	if(i){
	  sprintf(lid,"CA returns error code : %.8x",i);
	  errmsg = lid; goto done;
	}
      }
    }

    /* generate licenseID & hashed value */
    if(gen_license_id2(qu,lid)){
      errmsg = "can not generate license ID"; goto done;
    }
    if(cgi_gen_hash(lid,lhash)){
      errmsg = "can not get hashed license ID"; goto done;
    }
    aira_lid2querykey(lid,que);

    /* update ldap status */
    if(aira_change_entryst_(info,qu,"WAIT_ISSUE",lhash,3,atoi(gp),LDAP_MOD_REPLACE,que)){
      errmsg = "cannot update ldap entry"; goto done;
    }
    ml = "raop_accept_user.txt";
  }

  /* send email to user */
  if(inf){
    if(*e.plang) strncpy(info->ml.lang,e.plang,4);
    else cgi_get_deflang(info->ml.lang);

    if(aira_sendmail_adm2usr(&info->ml,&e,info->ml.opemail,lid,ml)){
      snprintf(lid,254,"cannot send an email to user (email=%s) << %s",e.mail,info->ml.lang);
      ENCERRLOG(info->ca.caname,lid);
    }

    /* show list or search page */
    if(!strcmp(fm,"SearchUsr")) aiop_searchuser(info);
    else if(!strcmp(fm,"ListUsr")) aiop_listuser(info);
    else if(!strcmp(fm,"ListRvkReq")) aiop_listrvkreq(info);
    else if(!strcmp(fm,"ListUpdReq")) aiop_listupdreq(info);

  }else{
    /* jump to airaenroll */
    if((buf=(char*)get_file2buf(get_html_name(info,AI_ENRJUMP_HTM),&sz))==NULL) goto done;

    cgi_check_lang(info);
    printf("Content-type: text/html\n\n");
    {
      AiVList v[] = {
        {"{LID}",lid},
        {"{FROM}",fm},
        {"{Gp}",gp},
        {NULL,NULL}
      };
      cgi_print_content(buf,v);
    }
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_upduserdone : System error!!",errmsg,lcmp);
  }
  CertStat_free_all(st);
  if(lcmp){ LCMP_unbind_s(lcmp); LCMP_free(lcmp); }
  if(buf) free(buf);
#ifdef __WINDOWS__
  WSACleanup();
#endif
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aiop_deluser(AiEnrollInfo *info){
  AiEntryInfo *e = NULL;
  char *fm,*qu,*gp,*buf=NULL;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aiop_check_session(info)) goto done;

  qu = cgi_find_query(&info->qc,"Qu");
  fm = cgi_find_query(&info->qc,"Fm");
  gp = cgi_find_query(&info->qc,"Gp");

  if(aiop_check_gnum(info,keyid,atoi(gp))){ errmsg = "bad group number"; goto done; }

  /* allocate entryinfo */
  if((e=cgi_entryinfo_new())==NULL){
    errmsg = "entryinfo memory allocate error"; goto done;
  }
  if(aira_get_raentry(info,e,qu,3,atoi(gp))){
    errmsg = "can not get request info from LDAP server"; goto done;
  }

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_DELUSR_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{RASUBJECT}",sbj},
      {"{RAKEYID}",keyid},
      {"{firstName}",e->firstName},
      {"{lastName}",e->lastName},
      {"{email}",e->mail},
      {"{unit}",e->unit},
      {"{title}",e->title},
      {"{tel1}",e->tel1},
      {"{tel2}",e->tel2},
      {"{fax1}",e->fax1},
      {"{fax2}",e->fax2},
      {"{employeeNumber}",e->employeeNumber},
      {"{postalCode}",e->postalCode},
      {"{postalAddress}",e->postalAddress},
      {"{labeledUri}",e->labeledUri},
      {"{description}",e->description},
      {"{st}",e->st},
      {"{FROM}",fm},
      {"{Gp}",gp},
      {"{Qu}",qu},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_deluser : System error!!",errmsg,NULL);
  }
  cgi_entryinfo_free(e);
  if(buf) free(buf);
  return ret;
}

int aiop_deluserdone(AiEnrollInfo *info){
  char *fm,*qu;
  int gn,ret = -1;

  /* check session */
  if(aiop_check_session(info)) goto done;

  qu = cgi_find_query(&info->qc,"Qu");
  fm = cgi_find_query(&info->qc,"Fm");
  gn = atoi(cgi_find_query(&info->qc,"Gp"));

  if(fm==NULL){ errmsg = "cannot get fm"; goto done; }
  if(aiop_check_gnum(info,keyid,gn)){ errmsg = "bad group number"; goto done; }

  /* delete operator entry */
  if(aira_delete_raentry(info,qu,3,gn)){
    errmsg = "cannot delete ra operator entry"; goto done;
  }
  /* show list or search page */
  if(!strcmp(fm,"SearchUsr")) aiop_searchuser(info);
  else if(!strcmp(fm,"ListUsr")) aiop_listuser(info);
  else if(!strcmp(fm,"ListRvkReq")) aiop_listrvkreq(info);
  else if(!strcmp(fm,"ListUpdReq")) aiop_listupdreq(info);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_deluserdone : System error!!",errmsg,NULL);
  }
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aiop_autoreg(AiEnrollInfo *info){
  char *fm,*qu;
  int gn,sw,ret = -1;

  /* check session */
  if(aiop_check_session(info)) goto done;

  qu = cgi_find_query(&info->qc,"Qu");
  fm = cgi_find_query(&info->qc,"Fm");
  gn = atoi(cgi_find_query(&info->qc,"Gp"));
  sw = atoi(cgi_find_query(&info->qc,"S"));

  if(fm==NULL){ errmsg = "cannot get fm"; goto done; }
  if(aiop_check_gnum(info,keyid,gn)){ errmsg = "bad group number"; goto done; }

  /* add or delete auto register flag */
  if(aira_change_entryst_(info,qu,"AUTOREG",NULL,3,gn,
			 (sw)?(LDAP_MOD_ADD):(LDAP_MOD_DELETE),NULL)){
    errmsg = "cannot modify auto register flag"; goto done;
  }

  /* show list or search page */
  if(!strcmp(fm,"SearchUsr")) aiop_searchuserdone(info);
  else if(!strcmp(fm,"ListUsr")) aiop_listuser(info);
  else if(!strcmp(fm,"ListRvkReq")) aiop_listrvkreq(info);
  else if(!strcmp(fm,"ListUpdReq")) aiop_listupdreq(info);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_autoreg : System error!!",errmsg,NULL);
  }
  return ret;
}

int aiop_multiautoreg(AiEnrollInfo *info){
  char *fm,*ul;
  int i,gn,sw,ret = -1;

  /* check session */
  if(aiop_check_session(info)) goto done;

  fm = cgi_find_query(&info->qc,"Fm");
  gn = atoi(cgi_find_query(&info->qc,"Gp"));
  sw = atoi(cgi_find_query(&info->qc,"S"));

  if(fm==NULL){ errmsg = "cannot get fm"; goto done; }
  if(aiop_check_gnum(info,keyid,gn)){ errmsg = "bad group number"; goto done; }

  /* add or delete auto register flag */
  for(i=0; i<QUERY_MAX; i++){
    ul = cgi_find_postquery(&info->qc,"UL",&i);
    if(ul && *ul){
      /* ignore ldap error */
      aira_change_entryst_(info,ul,"AUTOREG",NULL,3,gn,(sw)?(LDAP_MOD_ADD):(LDAP_MOD_DELETE),NULL);
    }
  }

  /* show list or search page */
  if(!strcmp(fm,"SearchUsr")) aiop_searchuserdone(info);
  else if(!strcmp(fm,"ListUsr")) aiop_listuser(info);
  else if(!strcmp(fm,"ListRvkReq")) aiop_listrvkreq(info);
  else if(!strcmp(fm,"ListUpdReq")) aiop_listupdreq(info);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_multiautoreg : System error!!",errmsg,NULL);
  }
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aiop_updatepin(AiEnrollInfo *info){
  char *fm,*buf=NULL;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aiop_check_session(info)) goto done;

  fm = cgi_find_query(&info->qc,"Fm");

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_UPDPIN_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{RASUBJECT}",sbj},
      {"{RAKEYID}",keyid},
      {"{FROM}",fm},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_updatepin : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

int aiop_updpindone(AiEnrollInfo *info){
  char *fm,*oldpin,*newpin,*buf=NULL;
  int i,ret=-1;
  off_t sz;

  /* check session */
  if(aiop_check_session(info)) goto done;

  fm = cgi_find_query(&info->qc,"Fm");
  oldpin = cgi_find_query(&info->qc,"ENPin");
  newpin = cgi_find_query(&info->qc,"ENNewPin");

  /* update RA entry */
  sbj[11]=0;
  if((i=aira_change_pin(info,&sbj[3],oldpin,newpin,1,0))<0){
    errmsg = "system error.cannot change pin."; goto done;
  }
  if(i>0){
    errmsg = "bad old password. please input again"; goto done;
  }

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_UPDPINDONE_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{FROM}",fm},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_updatepin : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aiop_searchuser(AiEnrollInfo *info){
  char *gp,*buf=NULL;
  int gn,ret=-1;
  off_t sz;

  /* check session */
  if(aiop_check_session(info)) goto done;

  gp = cgi_find_query(&info->qc,"Gp");
  gn = atoi(gp);

  if(aiop_check_gnum(info,keyid,gn)){ errmsg = "bad group number"; goto done; }

  /* get group list */
  aiop_get_glist(info,keyid,gpl,2048,gn);

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_SEARCHUSR_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  {
    AiVList v[] = {
      {"{RASUBJECT}",sbj},
      {"{RAKEYID}",keyid},
      {"{GROUPNAMES}",gpl},
      {"{Gp}",gp},
      {NULL,NULL}
    };
    cgi_print_content(buf,v);
  }

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_searchop : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

int aiop_searchuserdone(AiEnrollInfo *info){
  char *buf=NULL,*text;
  char filter[256];
  int bg,lm,gn,so;
  int st,ret=-1;
  off_t sz;

  /* check session */
  if(aiop_check_session(info)) goto done;

  text = cgi_find_query(&info->qc,"Q");
  st   = atoi(cgi_find_query(&info->qc,"QT"));
  lm   = atoi(cgi_find_query(&info->qc,"Lm"));
  bg   = atoi(cgi_find_query(&info->qc,"St"));
  gn   = atoi(cgi_find_query(&info->qc,"Gp"));
  so   = atoi(cgi_find_query(&info->qc,"So"));

  if(*text==0){ errmsg = "can not search LDAP server (NULL input)."; goto done; }

  if(so <= 0) so = 1;
  if(lm <= 0) lm = 20;
  if(bg < 0) bg = 0;

  if(aiop_check_gnum(info,keyid,gn)){ errmsg = "bad group number"; goto done; }

  /* get group list */
  aiop_get_glist(info,keyid,gpl,2048,gn);

  memset(filter,0,256);

  switch(st){
  case 1: snprintf(filter,230,"(&(cn=*%s*)",text); break;
  case 2: snprintf(filter,230,"(&(mail=*%s*)",text); break;
  case 3: snprintf(filter,230,"(&(ou=*%s*)",text); break;
  case 4: snprintf(filter,230,"(&(telephoneNumber=*%s*)",text); break;
  case 5: snprintf(filter,254,"(&(st=%s)",text); break;
  }
  strcat(filter,"(!(cn=REG*)))");

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_SEARCHUSRDONE_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  if(print_search_list(info,buf,filter,"SearchUsr",3,lm,bg,so,text,st,gn)) goto done;

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_searchuserdone : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aiop_viewcacert(AiEnrollInfo *info){
  Cert *ca = NULL;
  char *cp,*p,*t;
  char *buf=NULL;
  int ret=-1;
  off_t sz;

  /* check session */
  if(aiop_check_session(info)) goto done;

  /* get ca certificate */
  snprintf(path,256,"..%s%s.cer","/",info->ca.caname);
  if((ca = Cert_read_file(path))==NULL) goto done;

  /*---------- print out HTML text -------------*/
  if((buf=(char*)get_file2buf(get_html_name(info,AI_VIEWCACT_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  cp = buf;
  do{
    if((t=strstr(cp,"${"))==NULL){ printf("%s",cp); break; }
    *t=0; t++;

    if(!memcmp(t,"{CERTTEXT}",10)){
      printf("%s",cp); CERT_PRINT(ca); cp=t+10; continue;
    }else{
      p = "$";
    }
    printf("%s",cp);
    printf("%s",p);
    cp = t;
  }while(cp);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_viewcacert : System error!!",errmsg,NULL);
  }
  if(buf) free(buf);
  Cert_free(ca);
  return ret;
}

/*////////////////////////////////////////////////////////////////////////*/

int aiop_getopcert(AiEnrollInfo *info){
  Cert *ct=NULL;
  char *p1=NULL;
  int ret=-1;

  /* get user info */
  if((ct = cgi_get_clientcert())==NULL){
    errmsg = "cannot get client certificate"; goto done;
  }
  /* check session */
  if(aiop_check_session(info)) goto done;

  /*---------- print out text -------------*/
  if((p1 = PEM_write_cert_buf(ct))==NULL) goto done;

  sbj[11]=0;
  printf("Content-type: application/x-x509-user-cert\n");
  printf("Content-Disposition: attachment; filename=\"%s.cer\"\n",&sbj[3]);
  printf("Content-length: %zu\n\n",strlen(p1));

  printf("%s",p1);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_getopcert : System error!!",errmsg,NULL);
  }
  Cert_free(ct);
  if(p1) free(p1);
  return ret;
}

int aiop_getcacert(AiEnrollInfo *info){
  Cert *ca=NULL;
  char *p1=NULL;
  char path[256];
  int ret=-1;

  /* check session */
  if(aiop_check_session(info)) goto done;

  /* get ca certificate */
  snprintf(path,256,"..%s%s.cer","/",info->ca.caname);
  if((ca = Cert_read_file(path))==NULL) goto done;

  /*---------- print out text -------------*/
  if((p1 = PEM_write_cert_buf(ca))==NULL) goto done;

  printf("Content-type: application/x-x509-ca-cert\n");
  printf("Content-Disposition: attachment; filename=\"%s.cer\"\n",info->ca.caname);
  printf("Content-length: %zu\n\n",strlen(p1));

  printf("%s",p1);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_getcacert : System error!!",errmsg,NULL);
  }
  Cert_free(ca);
  if(p1) free(p1);
  return ret;
}

int aiop_getcrl(AiEnrollInfo *info){
  CRL *crl=NULL;
  char *p1=NULL;
  char path[256];
  int ret=-1;

  /* check session */
  if(aiop_check_session(info)) goto done;

  /* get ca certificate */
#ifdef __WINDOWS__
  strncpy(path,"out-CRL-All.crl",256);
#else
  strncpy(path,"../out/out-CRL-All.crl",256);
#endif
  if((crl = CRL_read_file(path))==NULL){
    errmsg = "cannot open CRL file. please start CRL Publisher"; goto done;
  }

  /*---------- print out text -------------*/
  if((p1 = PEM_write_crl_buf(crl))==NULL) goto done;

  printf("Content-type: application/x-x509-crl\n");
  printf("Content-Disposition: attachment; filename=\"%s.crl\"\n",info->ca.caname);
  printf("Content-length: %zu\n\n",strlen(p1));

  printf("%s",p1);

  ret = 0;
done:
  if(ret){
    printf("Content-type: text/html\n\n");
    aiop_print_err(info,"aiop_getcrl : System error!!",errmsg,NULL);
  }
  CRL_free(crl);
  if(p1) free(p1);
  return ret;
}
