/* aienr_util.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_io.h>
#include <aicrypto/ok_asn1.h>

#include "aienr_cgi.h"

/*-----------------------------------------------
  init & free
-------------------------------------------------*/
int aienr_init_eninfo(AiEnrollInfo *info){
  char buf[128];

  memset(info,0,sizeof(AiEnrollInfo));

  /* init lock */
#ifdef __WINDOWS__
  SNPRINTF(buf,126,"/aica/RAdPwd%d",info->section);
#else
  snprintf(buf,126,"%s/RAdPwd%d",LOCKDIR,info->section);
#endif
  if((info->plock=OK_init_lock(buf))==NULL) goto error;

  return 0;
error:
  printf("Content-type: text/plain\n\n");
  printf("WEB Enroll info init error (lock failed)<br>\n");
  return -1;
}

void aienr_clean_eninfo(AiEnrollInfo *info){
  int i;

  if(info->plock) OK_release_lock(info->plock);
  AccList_free_all(info->list);

  for(i=0; i<MAXGROUP; i++){
    if(info->grpname[i]){ free(info->grpname[i]); }
    if(info->grpprof[i]){ free(info->grpprof[i]); }
    if(info->grpemail[i]){ free(info->grpemail[i]); }
    if(info->grpbase[i]){ free(info->grpbase[i]); }
    if(info->grphost[i]){ free(info->grphost[i]); }
    if(info->grpbind[i]){ free(info->grpbind[i]); }
    if(info->grpbindpwd[i]){ free(info->grpbindpwd[i]); }
  }
  
  for(i=0; i<QUERY_MAX; i++)
    if(info->qc.postqs[i].value) free(info->qc.postqs[i].value);

  memset(info,0,sizeof(AiEnrollInfo));
}

int aienr_line2dn(char *in, CertDN *cdn, char **err){
  char *c,*v,buf[256];
  int more_rdn=0, in_quo=0, last_c=0;
  int i=0, j=0, ok=-1;
  int end=0;

  /* split elements */
  while(in && *in &&(i<RDN_MAX)){
    /* ignore space until charactor */
    if(*in == ' '){ in++; continue; }

    /* get type char */
    if((v = strchr(in,'='))==NULL){
      *err = "line2dn : '=' not found."; goto done;
    }
    *v=0; v++;

    /* ignore space until charactor -- LDAPv2 compatible */
    while(*v == ' '){ v++; }
    c=v;

    /* get string value */
    more_rdn = in_quo = last_c = j = 0;
    memset(buf,0,256);

    while(*c &&(j<254)){
      switch(*c){
      case '"':
        if(!in_quo)
          in_quo = 1;
        else{
          in_quo = 0; c++;
          while(*c == ' '){ c++; }
          continue;
        }
        break;
      case '\\': /* escape 1 char */
        c++; buf[j] = *c; j++;
        break;
      case ',':
      case ';':
        if(!in_quo){
          more_rdn = 1;
          last_c = 1; /* last char */
        }else{
          buf[j] = *c; j++;
        }
        break;
      default:
        buf[j] = *c; j++;
        break;
      }

      if(last_c) break;
      c++;
    }

    if(!last_c&&*c=='\0') end=1;
    if(last_c) *c=0;

    if ((cdn->rdn[i].tag = strdup(buf)) == NULL){
      *err = "line2dn : strdup error"; goto done;
    }
    if((cdn->rdn[i].tagoid = get_dn_kind(in))<0){
      *err = "line2dn : cannot get DN kind"; goto done;
    }
    cdn->rdn[i].derform = asn1_str_type(v);

    i++;
    if(end==1){
      in=c;
    }else{
      in=c+1;
    }
  }
  cdn->num = i;

  /* check if last "," exists */
  if(more_rdn){
    *err = "line2dn : line DN parse error"; goto done;
  }

  ok = 0;
done:
  return ok;
}

/*-----------------------------------------------
  query string utils
-------------------------------------------------*/
char *aienr_get_profname(AiEnrollInfo *info, char *group){
  int i;
  if((i=aienr_get_grpnum(info,group))>=0) return info->grpprof[i];
  return NULL;
}

char *aienr_get_basedn(AiEnrollInfo *info, char *group){
  int i;
  if((i=aienr_get_grpnum(info,group))>=0) return info->grpbase[i];
  return NULL;
}

char *aienr_get_host(AiEnrollInfo *info, char *group){
  int i;
  if((i=aienr_get_grpnum(info,group))>=0) return info->grphost[i];
  return NULL;
}

int aienr_get_grpnum(AiEnrollInfo *info, char *group){
  int i,ret=-1;

  for(i=0; i<MAXGROUP; i++){
    if(info->grpname[i]==NULL) break;
    if(!strcmp(info->grpname[i],group)){ ret=i; break; }
  }
  return ret;
}

/*-----------------------------------------------
  http_referer check
------------------------------------------------*/
int aienr_check_referer(AiEnrollInfo *info, char *page){
	char *ref;

	if((ref = getenv("HTTP_REFERER")) == NULL) ref = "";
	if(!strcmp(page,"newcert")){
		if(*info->ref_newcert && strcmp(ref,info->ref_newcert)) goto error;
	}

	return 0;
error:
	return -1;
}

/*------------------------------------------------------
  get unique and sequence number
------------------------------------------------------*/
int aienr_get_count(AiEnrollInfo *info,char *name){
  AILock lock = info->plock;
  FILE *fp = NULL;
  char path[256],*buf=NULL;
  off_t sz;
  int ret = -1;

  /*** start critical section ***/
  if(OK_lock(lock,10000)){lock=NULL; goto done;}

  sprintf(path,"..%s%scount.0","/",name);
  if((buf=(char*)get_file2buf(path,&sz))==NULL){
    ret = 0;
  }else{
    ret = atoi(buf);
  }
  if((fp=fopen(path,"w"))==NULL) goto done;
  fprintf(fp,"%d",ret+1);

  /*** end critical section ***/

done:
  if(fp){ fclose(fp); }
  if(lock){ OK_unlock(lock); lock=NULL;}
  if(buf) free(buf);
  return ret;
}

