/* aienr_user.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *	  this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <aicrypto/ok_io.h>
#include <aicrypto/ok_rand.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_base64.h>

#include "aienr_cgi.h"

/* login / password check */
int aienr_check_login(AiEnrollInfo *info, AiUserInfo *uinfo){
	char *name="",*passwd="";

	name   = cgi_find_query(&info->qc,"ENLoginName");
	passwd = cgi_find_query(&info->qc,"ENLoginPasswd");

	return aienr_check_login_(info,name,passwd,uinfo);
}

int aienr_check_login_(AiEnrollInfo *info, char *name, char *pwd, AiUserInfo *uinfo){
	char *buf=NULL,*cp,*t0,*t1,**val=NULL,*base=NULL,*host=NULL,*err="",cs[256];
	int auth, ok=-1;

	strncpy(uinfo->name,name,62);
	strncpy(uinfo->passwd,pwd,PWD_BUFLEN);
	uinfo->mode = AI_ENL_AUTHERR; /* set authentication error */

	if(info->ldaplogin){
#ifdef HAVE_LDAP_H
		LDAP *ret=NULL;
		LDAPMessage *res=NULL, *res_tmp=NULL, *ent=NULL;

		if((host=info->grphost[info->gnum])==NULL){
			if((host=info->ld.ldaphost)==NULL){
				err="cannot get hostname"; goto done_ldap;
			}
		}
		if((base=info->grpbase[info->gnum])==NULL){
			if((base=info->ld.ldapbase )==NULL){
				err="cannot get LDAP base"; goto done_ldap;
			}
		}
#ifdef WITHAD
		/* init ldap connection */
		if((ret=ldap_open(host, info->ld.ldapport))==NULL){
			err="ldap_open error"; goto done_ldap;
		}
#else
		if((ret=ldap_init(host,info->ld.ldapport))==NULL){
			err="ldap_init error"; goto done_ldap;
		}
#endif

		/* check ldap bind */
#ifdef WITHAD
		{
			SEC_WINNT_AUTH_IDENTITY wincred;
			char filter[256+17]; /* strlen("(sAMAccountName=)") => 17 */
			char *attrs[] = {"cn","mail","userPrincipalName",NULL};
			wincred.Flags = SEC_WINNT_AUTH_IDENTITY_ANSI;
			wincred.User = (unsigned char *)name;
			wincred.UserLength = strlen( name );
			wincred.Password = (unsigned char *)pwd;
			wincred.PasswordLength = strlen( pwd );
			wincred.Domain = (unsigned char *)( info->grpname[info->gnum] );
			wincred.DomainLength = strlen( wincred.Domain );
			if((i=ldap_bind_s( ret, NULL, (char*)&wincred, LDAP_AUTH_NEGOTIATE )) != LDAP_SUCCESS){
#if 0 //FOR DEBUG
				{	char cs[512];
					snprintf(cs,512,"DEBUG) ldap_bind_s returns 0x%.4x",i); ENCERRLOG(info->ca.caname,cs);
					snprintf(cs,512,"DEBUG) wincred.User=%s, wincred.Password=%s, wincred.Domain=%s",
						wincred.User, wincred.Password, wincred.Domain); ENCERRLOG(info->ca.caname,cs);
				}
#endif
				err="ldap_bind error"; goto done_ldap;
			}
#if 0 //FOR DEBUG
			snprintf(cs,512,"DEBUG) ldap_bind_s OK : wincred.User=%s, wincred.Password=%s, wincred.Domain=%s",
				wincred.User, wincred.Password, wincred.Domain); ENCERRLOG(info->ca.caname,cs);
#endif
			if((i=ldap_search_s(ret,base,LDAP_SCOPE_BASE,NULL,NULL,0,&res_tmp)) != LDAP_SUCCESS){
				err="ldap_search(BASE) error"; goto done_ldap;
			}
			uinfo->mode = 0; /* set authentication ok */
			sprintf(filter,"(sAMAccountName=%s)",name);

			if((i=ldap_search_s(ret,base,LDAP_SCOPE_SUBTREE,filter,attrs,0,&res)) != LDAP_SUCCESS){
#if 0 //FOR DEBUG
				{	char cs[512];
					snprintf(cs,512,"DEBUG) ldap_search_s returns 0x%.4x",i); ENCERRLOG(info->ca.caname,cs);
					snprintf(cs,512,"DEBUG) base=%s, filter=%s", base, filter); ENCERRLOG(info->ca.caname,cs);
				}
#endif
				err="ldap_search(SUBTREE) error"; goto done_ldap;
			}
#if 0 //FOR DEBUG
			snprintf(cs,512,"DEBUG) ldap_search_s OK : base=%s, filter=%s", base, filter); ENCERRLOG(info->ca.caname,cs);
#endif

			if((i=ldap_count_entries(ret,res))!=1){
#if 0 //FOR DEBUG
				{	char cs[512];
					snprintf(cs,512,"DEBUG) ldap_count_entries returns %d",i); ENCERRLOG(info->ca.caname,cs);
				}
#endif
				err="ldap_count_entries error"; goto done_ldap;
			}
			if((ent=ldap_first_entry(ret,res))==NULL){
				err="ldap_first_entry error"; goto done_ldap;
			}
			if((val=ldap_get_values(ret,ent,"mail"))!=NULL && *val!=NULL){
				strncpy(uinfo->mail,*val,254);
			}else{
				*(uinfo->mail)=0;
			}
			if(val) ldap_value_free(val);
			if((val=ldap_get_values(ret,ent,"userPrincipalName"))!=NULL && *val!=NULL){
				strncpy(uinfo->prname,*val,254);
			}else{
				err="cannot get userPrincipalName from LDAP"; goto done_ldap;
			}
			if(val) ldap_value_free(val);
			if((val=ldap_get_values(ret,ent,"cn"))!=NULL && *val!=NULL){
				/* 2007/9/23 AD returns SJIS code. convert it to UTF8 */
				int j, k;
				i = strlen(*val);
				for(k=j=0; k<i; k++) j |= 0x80 & (*val)[k];

				if(j){ /* maybe JP */
					UC_conv(UC_CODE_SJIS,UC_CODE_UTF8,*val,strlen(*val),uinfo->cn,254);
				}else{
					strncpy(uinfo->cn,*val,254);
				}
			}else{
				err="cannot get cn from LDAP"; goto done_ldap;
			}
			if(val) ldap_value_free(val);
		}
#else /* ! WITHAD */
		{
			char binddn[512];
			char *attrs[] = {"mail",NULL};
			if(!(strstr(name,"C=")||strstr(name,"c=")||strstr(name,"O=")||strstr(name,"o="))){
				memset (binddn,0,512);
				strncpy(binddn,info->ld.uidattr,64);
				strcat (binddn,"=");
				strncat(binddn,name,128);
				strcat (binddn,",");
				strncat(binddn,base,256);
			}else{
				strncpy(binddn,name,256);
			}
			if(cgi_ldap_bind(ret,binddn,(char*)pwd,info->ld.ldapbind) != LDAP_SUCCESS){
				err="ldap_bind error"; goto done_ldap;
			}
			uinfo->mode = 0; /* set authentication ok */

			if(ldap_search_s(ret,binddn,LDAP_SCOPE_BASE,NULL,attrs,0,&res) != LDAP_SUCCESS){
				err="ldap_search(BASE) error"; goto done_ldap;
			}
			if((ent=ldap_first_entry(ret,res))==NULL){
				err="ldap_first_entry error"; goto done_ldap;
			}
			if((val=ldap_get_values(ret,ent,"mail"))!=NULL && *val!=NULL){
				strncpy(uinfo->mail,*val,254);
			}else{
				*(uinfo->mail)=0;
			}
			if(val) ldap_value_free(val);
			strncpy( uinfo->cn,name,254);
			strncpy( uinfo->prname,name,254);
		}
#endif	/* WITHAD */
		ok = 0;
done_ldap:
		if(res) ldap_msgfree(res);
		if(res_tmp) ldap_msgfree(res_tmp);
		if(ret) ldap_unbind(ret);
		goto done;
#endif /* HAVE_LDAP_H */

	}else{
		off_t sz;
		/* check local passwd file */
		if((buf=(char*)get_file2buf(info->pwdpath,&sz))==NULL) goto done;
		cp = buf;

		while(cp && *cp){
			t0 = cp;
			if((cp=strchr(cp,'\n'))!=NULL){
				if(*(cp-1)=='\r') *(cp-1) = 0;
				*cp = 0; cp++;
			}
			if((*t0 == 0)||(*t0 == '#')) continue; /* name error */

			if((t1=strchr(t0,':'))==NULL) continue; /* format error */
			*t1=0; t1++;

			/* check name & passwd */
			if(strcmp(uinfo->name,t0)) continue; /* name error */

			auth = aienr_check_passwd(info,t1,uinfo->passwd);

			/* authentication ok and return */
			if(auth == 0) uinfo->mode = 0;
			break;
		}
		ok = 0;
	}

done:
	if(buf) free(buf);
	if(*err){
		sprintf(cs,"%s : login error : %s",name, err);
		ENCERRLOG(info->ca.caname,cs);
	}
	return ok;
}

/* license (one time password) check */
int aienr_check_license(AiEnrollInfo *info, AiUserInfo *uinfo){
	char *license="",*pin="";

	license = cgi_find_query(&info->qc,"ENLoginName");
	pin = cgi_find_query(&info->qc,"ENPin");

	return aienr_check_license_(info,license,uinfo,pin);
}

int aienr_check_license_(AiEnrollInfo *info, char *license, AiUserInfo *uinfo, char *pin){
	time_t tim;
	struct tm *ctm,stm;
	char hash[64],phash[64],tmp[32],tbuf[1024];
	char *buf=NULL,*cp,*t0,*t[4];
	int i,j, ok=-1;

	strncpy(uinfo->name,license,62);
	cgi_gen_hash(license,hash);
	cgi_gen_hash(pin,phash);
	uinfo->mode = AI_ENL_AUTHERR; /* set authentication error */

	if((info->authmode==AE_AUTH_LCHA)&&(pin==NULL||*pin==0)){
		uinfo->mode = AI_ENL_NOPIN; ok = 0; goto done;
	}

	if(info->ldaplogin){
#ifdef HAVE_LDAP_H
		LDAP *ret=NULL;
		LDAPMessage *res=NULL, *res_tmp=NULL, *ent=NULL;
		char **val=NULL, *host=NULL, *base=NULL, *err="", *admindn=NULL, *adminpwd=NULL;
		char *targetdn=NULL, **rdns=NULL, *pt=NULL;
		char filter[256], cs[256];
		int i;

		if(*pin)
		  sprintf(filter,"(&(%s=%s)(%s=%s))",info->ld.lidattr,hash,info->ld.pinattr,phash);
		else
		  sprintf(filter,"(%s=%s)",info->ld.lidattr,(info->rawlid)?(license):(hash));

		for(i=0; i<MAXGROUP && info->grpname[i]!=NULL; i++){

			if((host=info->grphost[i])==NULL){
				if((host=info->ld.ldaphost)==NULL){
					err="cannot get hostname"; goto done_ldap;
				}
			}
			if((base=info->grpbase[i])==NULL){
				if((base=info->ld.ldapbase )==NULL){
					err="cannot get LDAP base"; goto done_ldap;
				}
			}
			if((admindn=info->grpbind[i])==NULL){
				if((admindn=info->ld.ldapadmin )==NULL){
					err="cannot get LDAP base"; goto done_ldap;
				}
			}
			if((adminpwd=info->grpbindpwd[i])==NULL){
				if((adminpwd=info->ld.ldapadminpwd )==NULL){
					err="cannot get LDAP base"; goto done_ldap;
				}
			}
#ifdef WITHAD
			/* init ldap connection */
			if((ret=ldap_open(host, info->ld.ldapport))==NULL){
				err="ldap_open error"; goto done_ldap;
			}
#else
			if((ret=ldap_init(host,info->ld.ldapport))==NULL){
				err="ldap_init error"; goto done_ldap;
			}
#endif

			/* check ldap bind */
#ifdef WITHAD
			{
				SEC_WINNT_AUTH_IDENTITY wincred;
				char *attrs[] = {"cn","mail","userPrincipalName",NULL};
				wincred.Flags = SEC_WINNT_AUTH_IDENTITY_ANSI;
				wincred.User = (unsigned char *)admindn;
				wincred.UserLength = strlen( admindn );
				wincred.Password = (unsigned char *)adminpwd;
				wincred.PasswordLength = strlen( adminpwd );
				wincred.Domain = (unsigned char *)( info->grpname[i] );
				wincred.DomainLength = strlen( wincred.Domain );
				if(ldap_bind_s( ret, NULL, (char*)&wincred, LDAP_AUTH_NEGOTIATE ) != LDAP_SUCCESS){
					err="ldap_bind error"; goto done_ldap;
				}
				if(ldap_search_s(ret,base,LDAP_SCOPE_BASE,NULL,NULL,0,&res_tmp) != LDAP_SUCCESS){
					err="ldap_search(BASE) error"; goto done_ldap;
				}
				if(res_tmp){ ldap_msgfree(res_tmp); res_tmp = NULL; }

				if(ldap_search_s(ret,base,LDAP_SCOPE_SUBTREE,filter,attrs,0,&res) != LDAP_SUCCESS){
					err="ldap_search(SUBTREE) error"; goto done_ldap;
				}
				if(ldap_count_entries(ret,res)!=1){
					err="ldap_count_entries error"; goto done_ldap;
				}
				if((ent=ldap_first_entry(ret,res))==NULL){
					err="ldap_first_entry error"; goto done_ldap;
				}
				if((val=ldap_get_values(ret,ent,"mail"))!=NULL && *val!=NULL){
					strncpy(uinfo->mail,*val,254);
				}else{
					*(uinfo->mail)=0;
				}
				if(val) ldap_value_free(val);

				if((val=ldap_get_values(ret,ent,"userPrincipalName"))!=NULL && *val!=NULL){
					strncpy(uinfo->prname,*val,254);
				}else{
					err="cannot get userPrincipalName from LDAP"; goto done_ldap;
				}
				if(val) ldap_value_free(val);

				if((val=ldap_get_values(ret,ent,"cn"))!=NULL && *val!=NULL){
					/* 2007/9/23 AD returns SJIS code. convert it to UTF8 */
					/* 2008/3/8 *BUG* i= strlen(...) "i" already used for group!! */
					int k;
					int vlen = strlen(*val);
					for(k=j=0; k<vlen; k++) j |= 0x80 & (*val)[k];

					if(j){ /* maybe JP */
						UC_conv(UC_CODE_SJIS,UC_CODE_UTF8,*val,strlen(*val),uinfo->cn,254);
					}else{
						strncpy(uinfo->cn,*val,254);
					}
				}else{
					err="cannot get cn from LDAP"; goto done_ldap;
				}
				if(val) ldap_value_free(val);

				uinfo->mode = 0; /* set authentication ok */
				info->gnum = i;

				ok = 0;
			}
#else
			{
				char *attrs[] = {"cn","mail",NULL};
				if(cgi_ldap_bind(ret,admindn,adminpwd,info->ld.ldapbind) != LDAP_SUCCESS){
					err="ldap_bind error"; goto done_ldap;
				}

				if(ldap_search_s(ret,base,LDAP_SCOPE_SUBTREE,filter,attrs,0,&res) != LDAP_SUCCESS){
					err="ldap_search(SUBTREE) error"; goto done_ldap;
				}
				if(ldap_count_entries(ret,res)!=1){
					err=""; goto done_ldap;
				}
				if((ent=ldap_first_entry(ret,res))==NULL){
					err="ldap_first_entry error"; goto done_ldap;
				}

				if((val=ldap_get_values(ret,ent,"mail"))!=NULL && *val!=NULL){
					strncpy(uinfo->mail,*val,254);
				}else{
					*(uinfo->mail)=0;
				}
				if(val) ldap_value_free(val);

				if((val=ldap_get_values(ret,ent,"cn"))!=NULL && *val!=NULL){
					strncpy(uinfo->cn,*val,254);
				}else{
					err="cannot get cn from LDAP"; goto done_ldap;
				}
				if(val) ldap_value_free(val);

				// setting UPN? no more necessary ??
				targetdn=ldap_get_dn(ret,ent);
				rdns=ldap_explode_dn(targetdn,0);
				if(rdns!=NULL && *rdns!=NULL){
					pt=strchr(*rdns,'=');
					pt++;
				}else{
					goto done_ldap;
				}
				while(*pt==' '){ pt++; }
				if(pt){
					char *upn = uinfo->prname;
					int j = strlen(pt);

					if(*pt=='"'){
						strncpy(upn,pt+1,254); j--;
						while(upn[j-1]==' '){ upn[j-1]=0; j--; }
						if(upn[j-1]=='"'){ upn[j-1]=0; }
					}else{
						strncpy(upn,pt,254);
						while(upn[j-1]==' '){ upn[j-1]=0; j--; }
					}
				}

				uinfo->mode = 0; /* set authentication ok */
				info->gnum = i;

				ok = 0;
			}
#endif
done_ldap:
			if(rdns){ ldap_value_free(rdns); rdns=NULL; }
			if(targetdn){ ldap_memfree(targetdn); targetdn=NULL; }
			if(res){ ldap_msgfree(res); res=NULL; }
			if(res_tmp){ ldap_msgfree(res_tmp); res_tmp=NULL; }
			if(ret){ ldap_unbind(ret); ret=NULL; }

			if(ok==0) break;
			if(*err){
			  sprintf(cs,"%s : ldap error : %s",license, err);
			  ENCERRLOG(info->ca.caname,cs);
			}
		}
		if(ok && *err==0){
		  sprintf(cs,"%s : ldap error : license not found.",license);
		  ENCERRLOG(info->ca.caname,cs);
		}

#endif /* HAVE_LDAP_H */

	}else
	/* no matter what, license file is used */
	{
		off_t sz;
		/* check local license file */
		if((buf=(char*)get_file2buf(info->lidpath,&sz))==NULL) goto done;
		cp = buf;

		for(i=0; i<4; i++) t[i] = &tbuf[i*256]; /* set csv buffer */

		while(cp && *cp){
			t0 = cp;
			if((cp=strchr(cp,'\n'))!=NULL){
				if(*(cp-1)=='\r') *(cp-1) = 0;
				*cp = 0; cp++;
			}
			memset(tbuf,0,1024);
			if(cgi_parse_csv(t0,t,4)) goto done;

			/* check license */
			if((*t[0])&&(*t[0]!='#')&&(!strcmp(uinfo->name,t[0])||!strcmp(hash,t[0]))){
				/* license found. check time */
				j = strlen(t[1]);

				tmp[0] = ASN1_GENERALIZEDTIME; tmp[1] = j;
				strncpy(&tmp[2],t[1],30);
				strncpy(uinfo->cn,t[2],62);
				strncpy(uinfo->mail,t[3],62);

				if(*t[1]){
					memset(&stm,0,sizeof(struct tm));
					if(UTC2stm(tmp,&stm)) goto done;

					/* get current time */
					time(&tim);
					ctm=(struct tm*)gmtime(&tim);

					/* check set time is expired or not */
					if(stmcmp(&stm,ctm)>0) continue;
				}
				uinfo->mode = 0; /* set authentication ok */
				break;
			}
		}
	}

	ok = 0;
done:
	if(buf) free(buf);
	/* if(ret) ldap_unbind(ret); */
	return ok;
}

int aienr_remove_licensefile(AiEnrollInfo *info, char *license, AccList *acc){
	FILE *fp=NULL;
	char buf[256],hash[64];
	int i,j,ok = -1;

	memset(hash,0,64);
	cgi_gen_hash(license,hash);
	j = strlen(hash);

	if((fp=fopen(info->lidpath,"rb+"))==NULL) goto done;
	i = strlen(license);

	while(fgets(buf,256,fp)){
		/* check license */
		if((*buf != 0)&&(*buf != '#')&&(!strncmp(license,buf,i)||!strncmp(hash,buf,j))){
			if(fseek(fp,0 - strlen(buf),SEEK_CUR)) goto done;
			if(fwrite("#",sizeof(char),1,fp)<1) goto done;
			break;
		}		
	}

	ok = 0;
done:
	if(fp) fclose(fp);
	return ok;
}

/* password check */
int aienr_get_passwd(AiEnrollInfo *info,char *pwd,char *salt,char *ret){
	char *p1=NULL,*p2=NULL;
	unsigned char buf[32],tmp[8];
	SHA1_CTX ctx;
	int i, ok = -1;

	if(salt==NULL){
		RAND_bytes(tmp,4);
		if((p1=Base64_encode(4,tmp,8))==NULL) goto done;
		salt = p1;
	}
	OK_SHA1(strlen(pwd),(unsigned char*)pwd,buf);

	for(i=0; i<100; i++){
		SHA1init(&ctx);
		SHA1update(&ctx,(unsigned char*)salt,4);
		SHA1update(&ctx,buf,20);
		SHA1final(buf,&ctx);
	}

	*ret = 0;
	if((p2=Base64_encode(20,buf,8))==NULL) goto done;
	strncpy(ret,salt,4);
	strncpy(&ret[4],p2,48);

	ok = 0;
done:
	if(p1) free(p1);
	if(p2) free(p2);
	return ok;
}

/* equal == 0, not equal == -1 */
int aienr_check_passwd(AiEnrollInfo *info,char *encry, char *pwd){
	char buf[64];

	if(*encry == 0) return 0;
	if(aienr_get_passwd(info,pwd,encry,buf)) return -1;
	if(memcmp(encry,buf,strlen(buf))) return -1;
	return 0;
}

/* return 0..success, 1..auth err, -1..system err */
int aienr_change_passwd(AiEnrollInfo *info,char *group,char *name, char *oldpwd, char *newpwd){
	char *buf=NULL,*sv=NULL,*hd,cry[64],*cp,*t0,*t1;
	char path[256];
	AILock lock = info->plock;
	LDAP *ld = NULL;
	FILE *fp = NULL;
	int i,j,ok = -1;

	if(info->ldaplogin){
#if defined(HAVE_LDAP_H) && !defined(WITHAD)
		char binddn[512];
		lock = NULL;
		/* init ldap connection */
		if((ld = ldap_init(info->ld.ldaphost,info->ld.ldapport))==NULL) goto done;

		/* check ldap bind */
		if(!(strstr(name,"C=")||strstr(name,"c=")||strstr(name,"O=")||strstr(name,"o="))){
			memset (binddn,0,512);
			strncpy(binddn,info->ld.uidattr,64);
			strcat (binddn,"=");
			strncat(binddn,name,128);
			if((cp = aienr_get_basedn(info,group)) != NULL) {
				strcat (binddn,",");
				strncat(binddn,cp,256);
			}
		}else{
			strncpy(binddn,name,256);
		}
		if(cgi_ldap_bind(ld,binddn,(char*)oldpwd,info->ld.ldapbind) != LDAP_SUCCESS){ ok=1; goto done;}

		/* change password */
		if(cgi_mod_attr_bin(ld,binddn,LDAP_MOD_REPLACE|LDAP_MOD_BVALUES,"userPassword",
					(unsigned char*)newpwd,strlen(newpwd)))
			goto done;
#endif /* HAVE_LDAP_H */

	}else{
		off_t sz;
		/*** start critical section ***/
		if(OK_lock(lock,10000)){lock=NULL; goto done;}
		
		if((buf=(char*)get_file2buf(info->pwdpath,&sz))==NULL) goto done;
		if((sv=(char*)malloc(i+64))==NULL) goto done;
		memset(sv,0,i+64);
		memset(cry,0,64);
		cp = buf; hd = sv;

		while(cp && *cp){
			t0 = cp;
			if((t1=strchr(cp,':'))==NULL) break;
			*t1=0; t1++;
			if((cp=strchr(t1,'\n'))!=NULL){
				if(*(cp-1)=='\r') *(cp-1) = 0;
				*cp = 0; cp++;
			}

			/* check name & passwd */
			if(strcmp(name,t0)){
				/* not change user */
				strcpy(hd,t0);
				strcat(hd,":");
				strcat(hd,t1);
				strcat(hd,"\n");
				hd+=strlen(hd);
			}else{
				/* password change user */
				if(aienr_check_passwd(info,t1,(char*)oldpwd)){ ok=1; goto done;}
				if(aienr_get_passwd(info,(char*)newpwd,NULL,cry)) goto done;

				strcpy(hd,t0);
				strcat(hd,":");
				strcat(hd,cry);
				strcat(hd,"\n");
				hd+=strlen(hd);
				break;
			}
		}
		if(cp && *cp) strcpy(hd,cp);

		/* save backup */
		strncpy(path,info->pwdpath,240);
		strcat (path,".bak");
		unlink(path);
		rename(info->pwdpath, path);

		j = strlen(sv);
		if((fp=fopen(info->pwdpath,"wb"))==NULL) goto done;
		if((fwrite(sv,sizeof(char),j,fp))<(unsigned)j) goto done;
		fclose(fp);
		/*** end critical section ***/
	}

	ok = 0;
done:
	if(lock){ OK_unlock(lock); lock=NULL;}
	if(ld) ldap_unbind(ld);
	if(buf) free(buf);
	if(sv) free(sv);
	return ok;
}

/*=====================================================================*/

int aienr_get_name(AiEnrollInfo *info,Cert *ct,AiUserInfo *uinfo,char **group){
	AccList *acc;
	Cert *ca = NULL;
	unsigned char hash[32];
	char path[256];
	int i,ok=-1;

	if(ct->pubkey){
		if(cs_get_keyhash(ct->pubkey,hash,&i)) goto done;
		if((acc=cgi_findbykeyid(info->list,hash))==NULL) goto done;
	}else{
		strncpy(path,info->rapath,220);
		strcat (path,"/");
		strcat (path,info->ca.caname);
		strcat (path,".cer");

		if((ca=Cert_read_file(path))==NULL) goto done;

/*{FILE *fp;if(fp=fopen("d:\\out.txt","a+")){fprintf(fp,ct->issuer);fclose(fp);}}*/

		if(strncmp(ct->issuer,ca->subject,strlen(ct->issuer))) goto done;
		if((acc=cgi_findbysnum(info->list,ct->serialNumber))==NULL) goto done;
	}

	strncpy(uinfo->name,acc->name,64);
	uinfo->mode = acc->mode;
	*group = acc->group;

	ok = 0;
done:
	Cert_free(ca);
	return ok;
}

