/* aienr_op.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <unistd.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_io.h>
#include <aicrypto/ok_rand.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_base64.h>
#include <aicrypto/ok_pem.h>
#include <aicrypto/ok_uconv.h>

#include "ok_conf.h"
#include "cgi_common.h"
#include "aienr_cgi.h"

#ifndef __WITH_LANG
#define __WITH_LANG	0
#endif

/* HTML Template files */
#define AI_LOGIN_HTM      "ai_login.html"
#define AI_LOGINONE_HTM   "ai_login_one.html"
#define AI_LOGINANON_HTM  "ai_login_anon.html"
#define AI_LOGINNEW_HTM   "ai_login_new.html"
#define AI_LOGINWAIT_HTM  "ai_login_wait.html"
#define AI_LOGINOP_HTM    "ai_login_op.html"
#define AI_LOGINPIN_HTM   "ai_login_pin.html"
#define AI_LOGINERR2_HTM  "ai_login_err2.html"
#define AI_LOGINERR_HTM   "ai_login_err.html"
#define AI_ERROR_HTM      "ai_error.html"
#define AI_SENDCSR_FORM4_HTM "ai_sendcsr_form4.html"
#define AI_SENDCSR_FORM3_HTM "ai_sendcsr_form3.html"
#define AI_SENDCSR_FORM2_HTM "ai_sendcsr_form2.html"
#define AI_SENDCSR_FORM_HTM  "ai_sendcsr_form.html"
#define AI_SENDCSR_ANON2_HTM "ai_sendcsr_anon2.html"
#define AI_SENDCSR_ANON_HTM  "ai_sendcsr_anon.html"
#define AI_SENDCSR_FORM4_MZ_HTM "ai_sendcsr_form4_mz.html"
#define AI_SENDCSR_FORM2_MZ_HTM "ai_sendcsr_form2_mz.html"
#define AI_SENDCSR_FORM_MZ_HTM  "ai_sendcsr_form_mz.html"
#define AI_SENDCSR_ANON_MZ_HTM  "ai_sendcsr_anon_mz.html"
#define AI_SENDCSR_FORM4_IE7_HTM "ai_sendcsr_form4_ie7.html"
#define AI_SENDCSR_FORM2_IE7_HTM "ai_sendcsr_form2_ie7.html"
#define AI_SENDCSR_FORM_IE7_HTM  "ai_sendcsr_form_ie7.html"
#define AI_SENDCSR_ANON_IE7_HTM "ai_sendcsr_anon_ie7.html"
#define AI_SUBMITDONE_HTM   "ai_submit_done.html"
#define AI_SUBMITERR_HTM    "ai_submit_err.html"
#define AI_ACCEPT_FORM2_HTM "ai_accept_form2.html"
#define AI_ACCEPT_FORM_HTM  "ai_accept_form.html"
#define AI_ACCEPT_FORM_IE7_HTM  "ai_accept_form_ie7.html"
#define AI_ACCEPT_FORM_MZ_HTM   "ai_accept_form_mz.html"
#define AI_ACCEPT_FORM_SF_HTM   "ai_accept_form_sf.html"
#define AI_REVOKEOP_HTM     "ai_revoke_op.html"
#define AI_UPDATEOP_HTM     "ai_update_op.html"
#define AI_UPDATEOP_IE7_HTM "ai_update_op_ie7.html"
#define AI_REVOKEDONE_HTM   "ai_revoke_done.html"
#define AI_DONE_FORM_HTM    "ai_done_form.html"
#define AI_PWD_CHANGE_HTM   "ai_pwd_change.html"
#define AI_PWD_DONE_HTM     "ai_pwd_done.html"


#define AI_SENDCSR_DL_ANON_HTM  "ai_sendcsr_dl_anon.html"
#define AI_SENDCSR_DL_ANON_MZ_HTM  "ai_sendcsr_dl_anon_mz.html"
#define AI_SENDCSR_DL_FORM_HTM  "ai_sendcsr_dl_form.html"
#define AI_SENDCSR_DL_FORM_MZ_HTM  "ai_sendcsr_dl_form_mz.html"
#define AI_SENDCSR_DL_FORM2_HTM  "ai_sendcsr_dl_form2.html"
#define AI_SENDCSR_DL_FORM2_MZ_HTM  "ai_sendcsr_dl_form2_mz.html"
#define AI_SENDCSR_DL_FORM4_HTM  "ai_sendcsr_dl_form4.html"
#define AI_SENDCSR_DL_FORM4_MZ_HTM  "ai_sendcsr_dl_form4_mz.html"
#define AI_ACCEPT_DL_FORM_HTM   "ai_accept_dl_form.html"
#define AI_DONE_DL_FORM_HTM    "ai_done_dl_form.html"
// 2008/7/19 actually, template is almost same. so it was marged.
//#define AI_LOGINANON_DL_HTM  "ai_login_dl_anon.html"
//#define AI_LOGINNEW_DL_HTM   "ai_login_dl_new.html"
//#define AI_LOGINOP_DL_HTM    "ai_login_dl_op.html"

/*------------------------------------------------------
  check language and set cookie : same as aira_util.c
------------------------------------------------------*/
void cgi_get_lang(AiEnrollInfo *info,char *ret){
  char *lg;

  lg = cgi_find_query(&info->qc,"Lang");
  if(lg==NULL || *lg==0){
    lg = cgi_find_cookie(&info->qc,"AILANG");
    if(lg==NULL || *lg==0){
      switch(__WITH_LANG){
      case 1: lg = "jp"; break;
      default: lg = "en"; break;
      }
    }
  }
  strncpy(ret,lg,2); ret[2]=0;
}

void cgi_get_deflang(char *ret){
  char *lg;
  switch(__WITH_LANG){
  case 1: lg = "jp"; break;
  default: lg = "en"; break;
  }
  strncpy(ret,lg,4);
}

char *cgi_html_path(AiEnrollInfo *info,char *base,char *file){
  static char html[128];
  char lang[8];

  cgi_get_lang(info,lang);
  snprintf(html,126,"..%s%s%s%s%s%s","/",lang,"/",base,"/",file);

  return html;
}

int cgi_get_langnum(AiEnrollInfo *info){
  char lang[8];
  int ret=AI_LNG_EN;

  cgi_get_lang(info,lang);
  if(!strcmp(lang,"jp")) ret=AI_LNG_JP;
  return ret;
}

void cgi_check_lang(AiEnrollInfo *info){
  char *lg,lang[8];

  lg = cgi_find_query(&info->qc,"Lang");
  if(lg && *lg){
    strncpy(lang,lg,2); lang[2]=0;
    printf("Set-Cookie: AILANG=%s; expires=Thu, 31-Dec-2099 00:00:00 GMT;\n", lang);
  }
}

char *get_html_name(AiEnrollInfo *info,char *file){
  return cgi_html_path(info,"enroll",file);
}

/*----------------------------------------------------------------------
  default : no operation
----------------------------------------------------------------------*/
int aienr_default(AiEnrollInfo *info){
	char *buf=NULL,*cp,*t,*p,path[256],grp[2048];
	off_t sz;
	int i,j,ret=-1;

	cgi_check_lang(info);
	printf("Content-type: text/html\n\n");

	switch(info->authmode){
	case AE_AUTH_WEB:	
	case AE_AUTH_IDPW: strcpy(path,AI_LOGIN_HTM); break;
	case AE_AUTH_LCHA: 
	case AE_AUTH_LICE: strcpy(path,AI_LOGINONE_HTM); break;
	default:
// 2008/7/19 actually, template is almost same. so it was marged.
//		if (!info->postmode && !info->offlineca) {
//			strcpy(path,AI_LOGINANON_DL_HTM);
//		} else {
			strcpy(path,AI_LOGINANON_HTM);
//		}
		break;
	}
	if((buf=(char*)get_file2buf(get_html_name(info,path),&sz))==NULL) goto done;

	if(info->authmode == AE_AUTH_IDPW || info->authmode == AE_AUTH_WEB){

		/* set appropriate message */
		cp = buf;
		do{
			if((t=strstr(cp,"${"))==NULL){ printf("%s",cp); break; }
			*grp=0; *t=0; t++;

			if(!memcmp(t,"{GROUPNAMES}",12)){

				for(i=j=0; i<MAXGROUP; i++){
					if(info->grpname[i]==NULL) break;

					snprintf(path,254,"<option value=\"%s\">%s\n",info->grpname[i],info->grpname[i]);
					j+=strlen(path);

					if(j>2048) break; /* overflow check */
					strcat(grp,path);
				}
				if(i==0) strcpy(grp,"<option value=\"\"> ----- \n");

				t+=12;
				p = grp;
			}else{
				p = "$";
			}
			printf("%s",cp);
			printf("%s",p);
			cp = t;
		}while(cp);

	}else{
		printf("%s",buf);
	}
	ret = 0;
done:
	if(ret){
		printf("aienr_default : System error!!<br>\n");
		printf("%s",(OK_get_error())?(OK_get_errstr()):(""));
		printf("<br>\n%s",path);
	}
	if(buf) free(buf);
	return ret;
}

/*----------------------------------------------------------------------
	login : user login operation
----------------------------------------------------------------------*/
int aienr_login(AiEnrollInfo *info){
	AccList *al = NULL;
	AiUserInfo uinfo;
	Cert *ct = NULL;
	char *buf=NULL,*err="",*cp,*ln,*gp,*pw,*cry;
	char path[256],*fm,*gn;
	unsigned char sb[16];
	off_t sz;
	int ssl=0,ret=-1;

	memset(&uinfo,0,sizeof(AiUserInfo));
	
	/* get SSL client cert if it exists */
	ct = cgi_get_clientcert();
/*	p = cgi_get_clientcertid();
	{FILE *fp;if(fp=fopen("d:\\out.txt","wb")){fprintf(fp,p);fclose(fp);}}*/
	ln = cgi_find_query(&info->qc,"ENLoginName");
	gp = cgi_find_query(&info->qc,"ENGroupName");
	pw = cgi_find_query(&info->qc,"ENLoginPasswd");
	fm = cgi_find_query(&info->qc,"Fm");
	gn = cgi_find_query(&info->qc,"Gp");
	cp = ln;

	/* check login & password */
	if(!ct || *cp){
		switch(info->authmode){
		case AE_AUTH_WEB:
		case AE_AUTH_IDPW:
			if((info->gnum=aienr_get_grpnum(info,gp))<0){
				err = "bad group name"; goto done;
			}
			if(aienr_check_login(info,&uinfo)){
				err = "check_login error"; goto done;
			}
			break;
		case AE_AUTH_LCHA:
		case AE_AUTH_LICE:
			if(aienr_check_license(info,&uinfo)){
				err = "check_license error"; goto done;
			}
			if(info->ldaplogin){
				gp = info->grpname[info->gnum];
			}else{
				gp = NULL;
			}
			break;
		default:
			err = "auth mode error"; goto done;
		}
	}
	/* do not login with certificate at license & pin mode */
	if((info->authmode==AE_AUTH_LCHA)&&(*cp==0)){ 
		err = "auth mode error"; goto done;
	}

	if((uinfo.mode != AI_ENL_AUTHERR)&&(uinfo.mode != AI_ENL_NOPIN)){
		if(cgi_load_session(info->sespath,&info->list)){
			err = "cannot load session file"; goto done;
		}

		if(cgi_get_modegrp(info->list,&uinfo,gp)){
			err = "cannot get current user mode"; goto done;
		}

		if(ct &&(*cp==0)){
			if(aienr_get_name(info,ct,&uinfo,&gp)){
				err = "cannot get license info from client certificate"; goto done;
			}
			ssl = 1; /* set ssl client authentication flag */
		}

		/* set cookie (one time session ID) */
		if((cp=ai_escape_url(uinfo.name,strlen(uinfo.name)))==NULL){
			err = "cannot escape user name"; goto done;
		}
		printf("Set-Cookie: ENLoginName=%s\n", cp);
		printf("Set-Cookie: ENFm=%s\n", fm);
		printf("Set-Cookie: ENGn=%s\n", gn);
		free(cp); cp=NULL;

		if(gp){
			if((cp=ai_escape_url(gp,strlen(gp)))==NULL){
				err = "cannot escape group name"; goto done;
			}
			printf("Set-Cookie: ENGroupName=%s\n",cp);
			free(cp); cp=NULL;
		}

		if(RAND_bytes(sb,8)) goto done;
		printf("Set-Cookie: ENSessionID=%.2x%.2x%.2x%.2x%.2x%.2x%.2x%.2x\n",
				sb[0],sb[1],sb[2],sb[3],sb[4],sb[5],sb[6],sb[7]);

		if(cgi_set_sessiongrp(info->sespath,&info->list,&uinfo,sb,gp)){
			err = "cannot set session : session save error."; goto done;
		}
		/* encrypt password */
		if(info->authmode && info->ldaplogin){
			/* clear cookie */
			printf( "Set-Cookie: ENLoginPasswd=\n");
			printf( "Set-Cookie: ENMail=\n");
			printf( "Set-Cookie: ENPRName=\n");

			if(info->authmode==AE_AUTH_IDPW){
				if((cry=aica_encry_passwd(pw))==NULL){
					err = "cannot encrypt password"; goto done;
				}
				if((pw=ai_escape_url(cry,strlen(cry)))==NULL){
					err = "cannot escape password"; goto done;
				}
				printf( "Set-Cookie: ENLoginPasswd=%s\n", pw );
				if(cry) free(cry);
				if(pw) free(pw);
			}
			if(*uinfo.mail){
				char * ml;
				if((ml=ai_escape_url(uinfo.mail,strlen(uinfo.mail)))==NULL){
					err = "cannot escape e-mail address"; goto done;
				}
				printf( "Set-Cookie: ENMail=%s\n", uinfo.mail );
				if(ml) free(ml);
			}
			if(*uinfo.prname){
				char * pr;
				if((pr=ai_escape_url(uinfo.prname,strlen(uinfo.prname)))==NULL){
					err = "cannot escape principal name"; goto done;
				}
				printf( "Set-Cookie: ENPRName=%s\n", pr );
				if(pr) free(pr);
			}
		}

		/* get current list */
		if((al=cgi_findbynamegrp(info->list,uinfo.name,gp))==NULL){
			err = "cannot find access list"; goto done;
		}
		if(al->mode==AI_ENL_WAITISS && !info->postmode && !info->offlineca){
			/* AI_ENL_WAITISS should not be used except postmode & offlineca mode */
			al->mode = AI_ENL_NEWISS;
			if(cgi_update_session(info->sespath,al)){ err="cannot update session file"; goto done; }
		}
		uinfo.mode = al->mode;

		/* check one time pwd or not */
		if(ssl){
			if(uinfo.mode == AI_ENL_REVOKED)
				uinfo.mode = AI_ENL_AUTHERR;
		}else if((info->authmode==AE_AUTH_LICE)&&(al->serialNum)){
			/* set error */
			uinfo.mode = AI_ENL_AUTHERR;
		}else{
			/* success - output log */
			snprintf(path,254,"%s : success to login",uinfo.name);
			ENCACCLOG(info->ca.caname,path);
		}
	}

	/* send next page */
	switch(uinfo.mode){
	case AI_ENL_WAITGET:
	case AI_ENL_WAITISS:
		strcpy(path,AI_LOGINWAIT_HTM);
		break;
	case AI_ENL_ISSDONE:
//		if (al && (al->isstype & AI_IS_DLP12)) {
//			strcpy(path,AI_LOGINOP_DL_HTM);
//		} else {
			strcpy(path,AI_LOGINOP_HTM);
//		}
		break;
	case AI_ENL_AUTHERR:
		switch(info->authmode){
		case AE_AUTH_LCHA:
		case AE_AUTH_LICE:
		  strcpy(path,AI_LOGINERR2_HTM); break;
		default: 
		  strcpy(path,AI_LOGINERR_HTM); break;
		}
		break;
	case AI_ENL_NOPIN:
		strcpy(path,AI_LOGINPIN_HTM);
		break;	  
	case AI_ENL_REJECTED:
	case AI_ENL_REVOKED:
	case AI_ENL_NEWISS:
	default:
		switch(info->authmode){
		case AE_AUTH_LCHA:
		case AE_AUTH_LICE:
//			if (!info->postmode && !info->offlineca) {
//				strcpy(path,AI_LOGINANON_DL_HTM);
//			} else {
				strcpy(path,AI_LOGINANON_HTM);
//			}
			break;
		default:
//			if (!info->postmode && !info->offlineca) {
//				strcpy(path,AI_LOGINNEW_DL_HTM);
//			} else {
				strcpy(path,AI_LOGINNEW_HTM);
//			}
			break;
		}
		break;
	}
	if((buf=(char*)get_file2buf(get_html_name(info,path),&sz))==NULL) goto done;

	cgi_check_lang(info);
	printf("Content-type: text/html\n\n");
	{
		char aid[8]="",pwmode[8],itype[8]="";
		AiVList v[] = {
		  {"{ACCEPTID}",aid},
		  {"{PWMODE}",pwmode},
		  {"{ISSTYPE}",itype},
		  {"{LID}",ln},
		  {"{FROM}",fm},
		  {"{Gp}",gn},
		  {NULL,NULL}
		};
		if(al) sprintf(aid,"%.7d",al->acceptID);
		if(al) sprintf(itype,"%.7d",al->isstype);
		sprintf(pwmode,"%d",(info->authmode==1)?(1):(0));
		cgi_print_content(buf,v);
	}

	ret = 0;
done:
	if(ret){
		printf("Content-type: text/html\n\n");
		aienr_print_err(info,"aienr_login : System error!!",err,NULL);
	}
	if(buf) free(buf);
	Cert_free(ct);
	return ret;
}

void aienr_print_err(AiEnrollInfo *info,char *err1,char *err2,LCMP *lc){
  char *buf,tmp[256]="";
  off_t sz;

  if((buf=(char*)get_file2buf(get_html_name(info,AI_ERROR_HTM),&sz)) != NULL) {
    AiVList v[] = {
      {"{ERROR1}",(err1)?(err1):("")},
      {"{ERROR2}",(err2)?(err2):("")},
      {"{OKERR}",(OK_get_error())?(OK_get_errstr()):("")},
      {"{LCMPERR}",tmp},
      {NULL,NULL}
    };
    if(lc && lc->op && lc->op->resultCode != LCMP_SUCCESS){
      snprintf(tmp,254,"%s (%d) => %s\n",LCMP_msg2str(lc->op->resultCode),lc->op->resultCode,
               lc->op->resultMsg);
    }
    cgi_print_content(buf,v);

    free(buf);
  }
}

/*----------------------------------------------------------------------
	user_newcert : send user cert form
----------------------------------------------------------------------*/
int aienr_user_newcert(AiEnrollInfo *info){
	AccList *al = NULL;
	Cert *ct = NULL;
	char *buf=NULL,*cp,*t,*p,*op,*name,*cgp,*id,*err="",*mail="",*prname="";
	char *apn,*apv,path[256],caDN[256],grp[2048]="";
	off_t sz;
	int i,j,k,bk=0,ie7=0,ret=-1;

	cgi_check_lang(info);
	printf("Content-type: text/html\n\n");

	/* check browser */
	apn = cgi_find_query(&info->qc,"appName");
	apv = cgi_find_query(&info->qc,"appVersion");
	op  = cgi_find_query(&info->qc,"CertOP");
	if(!strcmp(apn,"Microsoft Internet Explorer")){
		bk = 0; /* IE user */
		if(strstr(apv,"MSIE 7.0")){
			if(atof(strstr(apv,"Windows NT")+10) >= 6.0){
				ie7 = 1; /* IE7.0 and VISTA */
			}
		} 
	}else if(!strcmp(apn,"Netscape")){
		bk = 1; /* Netscape / FireFox / Safari */
	}else{
		err = "unknown browser"; goto done;
	}
	/* printf("appVersion:%s\n",apv); */

	/* get user name for anonymous mode */
	name = cgi_find_query(&info->qc,"UName");

	/* referer check */
	if(aienr_check_referer(info,"newcert")){
		err = "http_referer check error"; goto done;
	}

	/* check user session */
	if(info->authmode){
		if(cgi_load_session(info->sespath,&info->list)){
			err = "cannot load session file"; goto done;
		}
		name = cgi_find_cookie(&info->qc,"ENLoginName");
		cgp  = cgi_find_cookie(&info->qc,"ENGroupName");

		if(info->authmode && info->ldaplogin){
			mail = cgi_find_cookie(&info->qc,"ENMail");
			prname = cgi_find_cookie(&info->qc,"ENPRName");
		}
		if(*cgp==0) cgp = NULL;

		id = cgi_find_cookie(&info->qc,"ENSessionID");
		if(*name==0){ err="cannot get cookie (login)"; goto done; }
		if(*id	==0){ err="cannot get cookie (sessionID)"; goto done; }

		mail = cgi_find_cookie(&info->qc,"ENMail");

		if(cgi_check_sessiongrp(info->list,name,id,cgp)){
			snprintf(path,254,"check session error : name:%s, id:%s, cgp:%s",name,id,cgp);
			ENCERRLOG(info->ca.caname,path);
			err = "check session error"; goto done;
		}

		/* open CA certificate & get CA DN */
		if((al=cgi_findbynamegrp(info->list,name,cgp))==NULL){
			err = "cannot find access list"; goto done;
		}
	}

	memset(caDN,0,256);

	/* get CA certificate */
	strcpy (path,"..");
	strcat (path,"/");
	strncat(path,info->ca.caname,32);
	strcat (path,".cer");

	if((ct=Cert_read_file(path)) != NULL) {

		for(i=j=0; i<ct->subject_dn.num; i++){
			switch(ct->subject_dn.rdn[i].tagoid){
			case OBJ_DIR_C:	cp="C="; j=1; break;
			case OBJ_DIR_ST: cp="ST="; j=1; break;
			case OBJ_DIR_L:	cp="L="; j=1; break;
			case OBJ_DIR_O:	cp="O="; j=1; break;
			case OBJ_DIR_OU: cp="OU="; j=1; break;
			default: cp=""; j=0; break;
			}

			if(j){
				k = 254 - strlen(caDN);
				if(k<254) strncat(caDN,",",k); k--;
				strncat(caDN,cp,k); k-=strlen(caDN);
				strncat(caDN,ct->subject_dn.rdn[i].tag,k);
			}
		}
	}

	/* get source file */
	switch(info->authmode){
	case AE_AUTH_WEB:
	case AE_AUTH_IDPW:
		if(!strcmp(op,"textreq")){
			strcpy(path,AI_SENDCSR_FORM3_HTM);
		}else if(!strcmp(op,"certdl")){
			if(info->ldaplogin){
				if(bk)       strcpy(path,AI_SENDCSR_DL_FORM2_MZ_HTM);
				else         strcpy(path,AI_SENDCSR_DL_FORM2_HTM);
			}else{
				if(bk)       strcpy(path,AI_SENDCSR_DL_FORM_MZ_HTM);
				else         strcpy(path,AI_SENDCSR_DL_FORM_HTM);
			}
		}else{
			if(info->ldaplogin){
				if(bk)       strcpy(path,AI_SENDCSR_FORM2_MZ_HTM);
				else if(ie7) strcpy(path,AI_SENDCSR_FORM2_IE7_HTM);
				else         strcpy(path,AI_SENDCSR_FORM2_HTM);
			}else{
				if(bk)       strcpy(path,AI_SENDCSR_FORM_MZ_HTM);
				else if(ie7) strcpy(path,AI_SENDCSR_FORM_IE7_HTM);
				else         strcpy(path,AI_SENDCSR_FORM_HTM);
			}
		}
		break;
	case AE_AUTH_LICE:
	case AE_AUTH_LCHA:
	default:
		if(!strcmp(op,"textreq")){
			if((info->authmode==1)||info->ldaplogin)
			     strcpy(path,AI_SENDCSR_FORM3_HTM);
			else strcpy(path,AI_SENDCSR_ANON2_HTM);
		}else if(!strcmp(op,"certdl")){
			if(info->ldaplogin){
				if(bk)       strcpy(path,AI_SENDCSR_DL_FORM2_MZ_HTM);
				else         strcpy(path,AI_SENDCSR_DL_FORM2_HTM);
			}else if(al && *(al->cn)){
				if(bk)       strcpy(path,AI_SENDCSR_DL_FORM4_MZ_HTM);
				else         strcpy(path,AI_SENDCSR_DL_FORM4_HTM);
			}else{
				if(bk)       strcpy(path,AI_SENDCSR_DL_ANON_MZ_HTM);
				else         strcpy(path,AI_SENDCSR_DL_ANON_HTM);
			}
		}else{
			if(info->ldaplogin){
				if(bk)       strcpy(path,AI_SENDCSR_FORM2_MZ_HTM);
				else if(ie7) strcpy(path,AI_SENDCSR_FORM2_IE7_HTM);
				else         strcpy(path,AI_SENDCSR_FORM2_HTM);
			}else if(al && *(al->cn)){
				if(bk)       strcpy(path,AI_SENDCSR_FORM4_MZ_HTM);
				else if(ie7) strcpy(path,AI_SENDCSR_FORM4_IE7_HTM);
				else         strcpy(path,AI_SENDCSR_FORM4_HTM);
			}else{
				if(bk)       strcpy(path,AI_SENDCSR_ANON_MZ_HTM);
				else if(ie7) strcpy(path,AI_SENDCSR_ANON_IE7_HTM);
				else         strcpy(path,AI_SENDCSR_ANON_HTM);
			}
		}
		break;
	}
	if((buf=(char*)get_file2buf(get_html_name(info,path),&sz))==NULL) goto done;

	/* set appropriate message */
	cp = buf;
	do{
		if((t=strstr(cp,"${"))==NULL){ printf("%s",cp); break; }
		*t=0; t++;

		if(!memcmp(t,"{CASBJDN}",9)){
			p = caDN; t+=9;

		}else if(!memcmp(t,"{EMAIL}",7)){
			p = (al && *(al->email))?(al->email):(mail); t+=7;

		}else if(!memcmp(t,"{USERNAME}",10)){
			/* name code should be UTF8 */
			p = (al && *(al->cn))?(al->cn):((*prname)?(prname):(name));
			t+=10;
		}else if(!memcmp(t,"{GROUPNAMES}",12)){
			if((info->authmode==1)||info->ldaplogin){
				p = cgp;

			}else{
				for(i=j=0; i<MAXGROUP; i++){
					if(info->grpname[i]==NULL) break;

					snprintf(path,254,"<option value=\"%s\">%s\n",info->grpname[i],info->grpname[i]);
					j+=strlen(path);

					if(j>2048) break; /* overflow check */
					strcat(grp,path);
				}
				if(i==0) strcpy(grp,"<option value=\"\"> ----- \n");

				p = grp;
			}
			t+=12;

		}else if(!memcmp(t,"{PWMODE}",8)){
			sprintf(path,"%d",(info->authmode==1)?(1):(0)); t+=8;
			p = path;
		}else{
			p = "$";
		}
		printf("%s",cp);
		printf("%s",p);
		cp = t;
	}while(cp);

	ret = 0;
done:
	if(ret){
		aienr_print_err(info,"aienr_user_newcert : System error!!",err,NULL);
	}
	if(buf) free(buf);
	if(ct) Cert_free(ct);
	return ret;
}

/*----------------------------------------------------------------------
	send csr : send certificate request operation
----------------------------------------------------------------------*/
int aienr_send_csr(AiEnrollInfo *info){
	CertTemplate *ctt = NULL;
	AccList sal,*dmy=NULL,*al=NULL;
	Req *csr = NULL;
	SPKAC *spk = NULL;
	Cert *ct = NULL;
	PKCS12 *p12 = NULL;
	LCMP *lcmp = NULL;
	char path[256],cs[256],*name,*tm,*em,*cn,*sp;
	char *apv,*buf=NULL,*p7c=NULL,*p1,*p2,*cp,*gp,*t,*err="",*pwd, *group;
	unsigned char *der=NULL;
	off_t sz;
	int i,k,cv,len,code=0,accID=0,op,sf=0,ie7=0,mode=0,fmt=0,ok=-1;
#ifdef WITHAD
	char *prname;
#endif

#ifdef __WINDOWS__
	WSADATA wsaData;

	if(WSAStartup(MAKEWORD(1, 1), &wsaData) != 0) {
		err = "failed to initialize winsock"; goto done;
	}
#endif

	/* get user info */
	tm = cgi_find_query(&info->qc,"ContName");
	if(*tm == 0){
		err = "cannot get ContName form value"; goto done;
	}
	em = cgi_find_query(&info->qc,"Email"); /* maybe empty */
	cn = cgi_find_query(&info->qc,"CN");
	if(*cn == 0){
		err = "cannot get CN form value"; goto done;
	}
	group = cgi_find_query(&info->qc,"ENGroupName");
	if(*group == 0){
		err = "cannot get group name form value"; goto done;
	}
	/* check appVersion */
	apv = cgi_find_query(&info->qc,"appVersion");
	if(strstr(apv,"Safari")) sf = 1;
	if(strstr(apv,"MSIE 7.0")){
		if(atof(strstr(apv,"Windows NT")+10) >= 6.0){
			ie7 = 1; /* IE7.0 and VISTA */
		}
	}
	/* check user session */
	if(info->authmode){
		if(cgi_load_session(info->sespath,&info->list)){
			err = "cannot load session file"; goto done;
		}
		name = cgi_find_cookie(&info->qc,"ENLoginName");
		p1   = cgi_find_cookie(&info->qc,"ENSessionID");
		if(*name==0){ err="cannot get cookie (login)"; goto done; }
		if(*p1  ==0){ err="cannot get cookie (sessionID)"; goto done; }

#ifdef WITHAD
		prname = cgi_find_cookie(&info->qc,"ENPRName");
		if(*prname==0){ err="cannot get cookie (prname)"; goto done; }
#endif

		if(info->authmode && info->ldaplogin){
			pwd = cgi_find_cookie(&info->qc,"ENLoginPasswd");
		}

		/* find enroll user name */
		gp = (info->authmode==AE_AUTH_LICE && !info->ldaplogin)?(NULL):(group);

		if(cgi_check_sessiongrp(info->list,name,p1,gp)){
			snprintf(path,254,"check session error : name:%s, id:%s, cgp:%s",name,p1,gp);
			ENCERRLOG(info->ca.caname,path);
			err = "check session error"; goto done;			  
		}
		if((al=cgi_findbynamegrp(info->list,name,gp))==NULL){
			err = "cannot find access list"; goto done;
		}
		/* user state check */
		if(al->mode==AI_ENL_WAITISS){
			err = "certificate request is already accepted"; goto done;
		}
	}else{
		/* anonymous mode */
		memset(&sal,0,sizeof(AccList)); al = &sal;
                strncpy(al->name,cn,64);
	}

	/* received CSR */
	cp = cgi_find_query(&info->qc,"SendCSR");
	sp = cgi_find_query(&info->qc,"SPKAC");

//{FILE *fp; if(fp=fopen("recv_req.txt","a+")){fprintf(fp,cp); fclose(fp);}}

	if(*sp){
		if((spk=cgi_spkac_decode(sp))==NULL){
			err = "cannot decode form data to SPKAC"; goto done;
		}
	}
	else if(*cp){
		fmt = cgi_get_fmt(cp);
		switch(fmt){
		case 2: csr=PEM_read_req_buf(cp); break;
		case 3: csr=cgi_csr_decode(cp); break;
		case 4: csr=cgi_csr_decode(cp+27); break;
		}
		if(csr==NULL){
			err = "cannot decode form data to CSR"; goto done;
		}
	}
	else {
		err = "cannot decode form data to CSR"; goto done;
	}

	/* get certificate template for signing request */
	if((ctt=CMP_certtmpl_new())==NULL) goto done;

	if(csr){
		ctt->publicKey = csr->pubkey; csr->pubkey = NULL;
		Cert_dncopy(&csr->subject_dn,&ctt->subject);
	}else{
		ctt->publicKey = spk->pubkey; spk->pubkey = NULL;
		cp = cgi_find_query(&info->qc,"CaDN");
		if(aienr_line2dn(cp,&ctt->subject,&err)) goto done;
	}
#ifdef WITHAD
	{
		ExtGenNames *egn=NULL;
		CertExt *et=NULL;
		if((egn=ExtGN_set_upnname(prname))==NULL) goto done;
		if((et=Extnew_altname(OBJ_X509v3_SbjAltName,egn))==NULL) goto done;
		if((ctt->ext=Extnew_extreq(et))==NULL) goto done;
	}

	/* set prname as cn */
	//strncpy(al->cn,prname,62);
#endif

	/* replace subject cn and email from user data source */
	if(al && *al->cn && al->mode!=AI_ENL_REJECTED){
		cn = al->cn;
		if(Cert_find_dn(&ctt->subject,OBJ_DIR_CN,&i,0)){
			len = strlen(cn);
			for(k=cv=0; k<len; k++) cv |= 0x80 & cn[k];

			free(ctt->subject.rdn[i].tag);
			if(cv){ /* maybe JP */
			  UC_conv(UC_CODE_UTF8,UC_LOCAL_CODESET,cn,strlen(cn),cs,254);
			  cn = cs;
			}
			if ((ctt->subject.rdn[i].tag = strdup(cn)) == NULL) goto done;
			ctt->subject.rdn[i].derform = asn1_str_type(cn);
			cn = al->cn;
		}
	}
	if(al && *al->email && al->mode!=AI_ENL_REJECTED){
		em = al->email;
		if(Cert_find_dn(&ctt->subject,OBJ_DIR_EMAIL,&i,0)){
			free(ctt->subject.rdn[i].tag);
			if ((ctt->subject.rdn[i].tag = strdup(em)) == NULL) goto done;
			ctt->subject.rdn[i].derform = asn1_str_type(em);
		}
	}

	/* email subject filter check */
	switch(info->emsbjfilter){
	case 1: /* in cert extension. subjectAltName */
	case 2: /* not in cert. remove from subject DN */
		/* remove email from subject DN */
		if(Cert_find_dn(&ctt->subject,OBJ_DIR_EMAIL,&i,0)){
		  free(ctt->subject.rdn[i].tag); 
		  ctt->subject.rdn[i].tag = NULL;
		  ctt->subject.rdn[i].tagoid = 0;
		  ctt->subject.num--;
		}
		/* add to subjectAltName */
		if(info->emsbjfilter==1 && em && *em){
		  ExtGenNames *egn=NULL;
		  CertExt *et=NULL;
		  
		  if((egn=ExtGN_set_email(em))==NULL) goto done;
		  if((et=CertExt_find(ctt->ext,OBJ_X509v3_SbjAltName))==NULL){
		    if((et=Extnew_altname(OBJ_X509v3_SbjAltName,egn))==NULL) goto done;
		  }else{
		    egn->next = ((CE_SbjAltName*)et)->egn;
		    ((CE_SbjAltName*)et)->egn = egn;
		  }
		  if((ctt->ext=Extnew_extreq(et))==NULL) goto done;
		}
		break;
	default: /* noting to do */
		break;
	}

	/* connect CA and send CSR (!offline ca mode) */
	if(!info->offlineca){
		mode = LCMP_OPBRQ_ATSSLCL;
		if((lcmp=LCMP_init(info->ca.svname,info->ca.caport,info->ca.caname))==NULL){
			err = "cannot initialize CA connection"; goto done;
		}
		if(info->ca.usessl){
			if(LCMP_set_ssl(lcmp,info->ca.store,info->ca.certid,info->ca.clctpw,info->ca.vfycert)){
			  err = "cannot set SSL certificate"; goto done;
			}
			mode = LCMP_OPBRQ_ATSSLCL;
		}
		if(LCMP_bind_s(lcmp,info->ca.userid,info->ca.pwd,mode)){
			err = "cannot bind CA server"; goto done;
		}
	}

	/* (online) posting a CSR or issuing a certificate / (offline) posting */
	if(info->postmode || info->offlineca){
		if(info->postmode){ /* CSR post mode */
			if((code=LCMP_csr_s(lcmp,LCMP_OPCSR_POST,0,0,NULL,NULL,0,ctt))<0){
				err = "cannot post a CSR to CA server"; goto done;
			}
			accID = LCMP_get_acceptid(lcmp);

		}else{ /* offline ca mode */
			if((accID = aienr_get_count(info,"reg"))<0){
				err = "cannot get reg counter"; goto done;
			}
			/* output pkcs10 file (only need csr->der) */
#ifdef __WINDOWS__
			sprintf(path,"..\\req\\u%.7d.csr",accID);
#else
			sprintf(path,"../req/u%.7d.csr",accID);
#endif
			if((der=CMPtmpl_toDER(ctt,&i))==NULL){
				err = "cannot get certTmpl DER"; goto done;
			}
			if(ASN1_write_der(der,path)){
				err = "cannot output certTmpl file"; goto done;
			}
			/*
			if(PEM_write_req(csr,path)){ err = "cannot output pkcs10 file"; goto done; }
			*/
		}
	  
		/* show result page */
		if(code==0){ 
			/* no error - save session info */
			al->isstype &= ~AI_IS_DLP12;
			if(fmt==2){
				if((cn=Cert_find_dn(&ctt->subject,OBJ_DIR_CN,&i,0))==NULL) cn = name;
				al->isstype |= AI_IS_COPYP10;
			}else{
				al->isstype &= ~AI_IS_COPYP10;
			}
			if(spk){
				al->isstype |= AI_IS_MOZILLA;
				if(sf) al->isstype |= AI_IS_SAFARI; /* set safari flag */
			}else{
				al->isstype &= ~AI_IS_MOZILLA;
				if(ie7) al->isstype |= AI_IS_IE7; /* set ie7 flag */
			}
			al->isstype       |= cgi_get_langnum(info);
			al->mode          = AI_ENL_WAITISS;
			al->acceptID      = accID;
			al->containerTime = atol(tm);
			strncpy(al->cn,cn,62);
			strncpy(al->email,em,62);
			strncpy(al->group,group,62);
			/* clear session ID ... aienroll checks sessionID [0]==0 && [1]==0.
			 *  If they are true, it returns grid-certreq accept mail. 
			 *  so, in web enroll, we need sessionID to fulfill with 0xff.
			 */
			memset(al->sessionID,0xff,32);

			if((info->authmode==AE_AUTH_IDPW || info->authmode==AE_AUTH_WEB)&& info->ldaplogin){
				strncpy(al->pwd,pwd,62);
			}
			if(cs_get_keyhash(ctt->publicKey,al->keyID,&i)){ err="cannot get key id"; goto done; }

			/* if anonymous mode, add new session block */
			if(!info->authmode){
				/* set acceptID as user name */
				snprintf(al->name,62,"anon-ac-%.8d",accID);
				if(cgi_add_session(info->sespath,&dmy,al)){
					err="cannot add new session block"; goto done;
				}
			}
			if(cgi_update_session(info->sespath,al)){ 
				err="cannot update session file"; goto done;
			}
			strcpy(path,AI_SUBMITDONE_HTM);
			sprintf(cs,"%.7d",accID);
		}else{
			strcpy(path,AI_SUBMITERR_HTM);
			sprintf(cs,"%.8x",code);
		}
		
		if((buf=(char*)get_file2buf(get_html_name(info,path),&sz))==NULL) goto done;
		
		cgi_check_lang(info);
		printf("Content-type: text/html\n\n");

		/* set appropriate message */
		cp = buf;
		do{
			if((t=strstr(cp,"${"))==NULL){ printf("%s",cp); break; }
			*t=0; t++;
			
			if(!memcmp(t,"{ACCEPTID}",10)){
				p1 = cs; t+=10;
			}else if(!memcmp(t,"{PWMODE}",8)){
				sprintf(path,"%d",(info->authmode==1)?(1):(0)); t+=8;
				p1 = path;
			}else{
				p1 = "$";
			}
			printf("%s",cp);
			printf("%s",p1);
			cp = t;
		}while(cp);
		
		/* send email to admin */
		if(*info->ml.smtphost && *info->ml.admemail){
			char *ml = "enroll_accept_csr.txt";
			cgi_get_deflang(info->ml.lang);
			if(aienr_sendmail_sys2adm(&info->ml,al,info->ml.admemail,cn,ml)){
				sprintf(cs,"%s : cannot send a email to admin (accID=%.7d) << %s",
						al->name,al->acceptID,CA_get_errstr());
				ENCERRLOG(info->ca.caname,cs);
			}
		}

		/* output log */
		sprintf(cs,"%s : success to accept a CSR (accID=%.7d)",al->name,al->acceptID);
		ENCACCLOG(info->ca.caname,cs);

	}else{
		/* get PKCS12 (ca certificates) */
		if((p12=P12_dup(lcmp->ca->p12))==NULL) goto done;

		/* Sign direct mode */
		gp = aienr_get_profname(info,group);

		if((code=LCMP_sign_s(lcmp,gp,0,ctt,CMPREQUEST))<0){
			err = "cannot issue a certificate."; goto done;
		}

		if((ct=LCMP_get_signcert(lcmp))==NULL){
			err = "cannot get issued certificate."; goto done;
		}

		if(code==0){
			/* no error - save session info */
			al->isstype &= ~AI_IS_DLP12;
			if(fmt==2){
				if((cn=Cert_find_dn(&ctt->subject,OBJ_DIR_CN,&i,0))==NULL) cn = name;
				al->mode    = AI_ENL_ISSDONE;
				al->isstype = AI_IS_COPYP10;
			}else{
				al->mode     = AI_ENL_WAITISS;
				al->isstype &= ~AI_IS_COPYP10;
			}
			if(spk){
				al->isstype |= AI_IS_MOZILLA;
				if(sf) al->isstype |= AI_IS_SAFARI; /* set safari flag */
			}else{
				al->isstype &= ~AI_IS_MOZILLA;
				if(ie7) al->isstype |= AI_IS_IE7; /* set ie7 flag */
			}
			al->isstype       |= cgi_get_langnum(info);
			al->containerTime = atol(tm);
			al->serialNum     = ct->serialNumber;
			al->notAfter      = (unsigned long)timegm(&ct->time.notAfter);
			strncpy(al->cn,cn,62);
			strncpy(al->email,em,62);
			strncpy(al->group,group,62);
			memset(al->sessionID,0xff,32); /* clear session ID */

			if(cs_get_keyhash(ctt->publicKey,al->keyID,&i)){ err="cannot get key id"; goto done; }
			/* if anonymous mode, add new session block */
			if(!info->authmode){
				/* set acceptID as user name */
				snprintf(al->name,62,"anon-sn-%.8d",al->serialNum);
				if(cgi_add_session(info->sespath,&dmy,al)){
					err="cannot add new session block"; goto done;
				}
				printf("Set-Cookie: ENLoginName=%s\n", al->name);
			}
			if(cgi_update_session(info->sespath,al)){ 
				err="cannot update session file"; goto done;
			}
		}

		/* output a certificate file for UNICORE mapping */
		if(info->gridcert[0]){
			char path[320];
			snprintf(path,318,"%s/%s.pem",info->gridcert,al->name);
			if(PEM_write_cert(ct,path)){
				sprintf(cs,"%s : cannot output a certificate to file (sn=%.7d)",
						al->name,ct->serialNumber);
				ENCERRLOG(info->ca.caname,cs);
			}
		}

		/* out to ldap server */
#ifdef HAVE_LDAP_H
		if(info->ldaplogin){
			/* add certificate & remove license ID */
			op = (info->authmode==AE_AUTH_LCHA)?(LDAP_MOD_REPLACE):(LDAP_MOD_ADD);
			if(aienr_ldap_addcert(info,op|LDAP_MOD_BVALUES,ct,NULL,NULL,NULL,name)){
				sprintf(cs,"%s : cannot output a certificate to LDAP (sn=%.7d)",
						al->name,ct->serialNumber);
				ENCERRLOG(info->ca.caname,cs);
			}
		}else
#endif
		if(info->authmode==AE_AUTH_LICE){
			/* remove license ID from en.license file */
			if(aienr_remove_licensefile(info,al->name,al)){ err="cannot remove license id"; goto done; }
		}

		/* send email to admin */
		if(*info->ml.smtphost && *info->ml.admemail){
			char *ml = "enroll_accept_csr2.txt";
			cgi_get_deflang(info->ml.lang);

			if(info->authmode==AE_AUTH_LCHA){
				for(i=0; i<MAXGROUP; i++){
				  if(!strcmp(group,info->grpname[i])){ 
				    strncpy(info->ml.admemail,info->grpemail[i],126); break;
				  }
				}
			}
			if(aienr_sendmail_sys2adm(&info->ml,al,info->ml.admemail,cn,ml)){
				sprintf(cs,"%s : cannot send a email to admin (sn=%.7d) << %s",
						al->name,ct->serialNumber,CA_get_errstr());
				ENCERRLOG(info->ca.caname,cs);
			}
		}

		/* set cookie */
		printf("Set-Cookie: ENSerialNum=%.7d\n", ct->serialNumber);
		printf("Set-Cookie: ENGroupName=%s\n", group);

		if(P12_add_cert(p12,ct,NULL,0xff)) goto done;
		accID=ct->serialNumber; ct=NULL;

		if(P12_check_chain(p12,0)) goto done;
		if((der=P7_signed_toDER((PKCS7*)p12,NULL,&i))==NULL) goto done;

		if((p7c=Base64_encode(i,der,32))==NULL) goto done;

		/* get source file */
		cp = (csr)
			?((fmt==2)?(AI_ACCEPT_FORM2_HTM):((ie7)?(AI_ACCEPT_FORM_IE7_HTM):(AI_ACCEPT_FORM_HTM)))
			:((sf)?(AI_ACCEPT_FORM_SF_HTM):(AI_ACCEPT_FORM_MZ_HTM));
		if((buf=(char*)get_file2buf(get_html_name(info,cp),&sz))==NULL) goto done;

		/* set appropriate message */
		cgi_check_lang(info);
		printf("Content-type: text/html\n\n");

		cp = buf;
		ct = P12_get_usercert(p12);
		do{
			if((t=strstr(cp,"${"))==NULL){ printf("%s",cp); break; }
			*t=0; t++;

			printf("%s",cp);
			if(!memcmp(t,"{PRINTCERT}",11)){
				CERT_PRINT(ct); t+=11;
				fflush(stdout);
			}else if(!memcmp(t,"{RETURNCERT}",12)){
				p1 = PEM_write_cert_buf(ct); t+=12;
				printf("%s",p1); free(p1);
			}else if(!memcmp(t,"{RETURNPKCS7}",13)){
				p1=p2=p7c;
				while(p1 && *p1){
					if((p2=strchr(p1,'\n')) != NULL) { *p2=0; p2++; }
					printf("pkcs7+=\"%s\";\n",p1);
					p1 = p2;
				}
				t+=13;
			}else if(!memcmp(t,"{DOWNLOADCERT}",14)){
				/* for Safari mode */
#ifdef __WINDOWS__
				sprintf(cs,"sf%.7d.dl.pem",accID);
				if(PEM_write_cert(ct,cs)){ ct=NULL; goto done; }
#else
				sprintf(cs,"out/sf%.7d.dl.pem",accID);
				sprintf(path,"../out/sf%.7d.dl.pem",accID);
				if(PEM_write_cert(ct,path)){ ct=NULL; goto done; }
#endif
				printf("%s",cs); t+=14;
			}else if(!memcmp(t,"{DOWNLOADCACERT}",16)){
				/* for Safari mode */
#ifdef __WINDOWS__
				SNPRINTF(cs,254,"%s.p7b",lcmp->caname);
				if(P7b_write_file((PKCS7*)lcmp->ca->p12,cs)){ ct=NULL; goto done; }
#else
				snprintf(cs,254,"out/%s.p7b",lcmp->caname);
				snprintf(path,254,"../out/%s.p7b",lcmp->caname);
				if(P7b_write_file((PKCS7*)lcmp->ca->p12,path)){ ct=NULL; goto done; }
#endif
				printf("%s",cs); t+=16;
			}else if(!memcmp(t,"{CLEARCT}",9)){
				/* for Safari mode */
#ifdef __WINDOWS__
				sprintf(cs,"sf%.7d",accID);
#else
				sprintf(cs,"../out/sf%.7d",accID);
#endif
				printf("%s",cs); t+=9;
			}else{
				printf("$");
			}
			cp = t;
		}while(cp);

		ct = NULL;

		/* output log */
		sprintf(cs,"%s : success to issue a Cert (sn=%.7d)",
			((info->authmode)?(name):("anonymous")),accID);
		ENCACCLOG(info->ca.caname,cs);
		ENCISSLOG(info->ca.caname,cs);
	}

	ok = 0;
done:
	if(ok){
		printf("Content-type: text/html\n\n");
		aienr_print_err(info,"send csr : System error!!",err,lcmp);
	}
	if(lcmp){ LCMP_unbind_s(lcmp); LCMP_free(lcmp); }
	Req_free(csr);
	Cert_free(ct);
	SPKAC_free(spk);
	CMP_certtmpl_free(ctt);
	if(p12) P12_free(p12);
	if(der) free(der);
	if(p7c) free(p7c);
	if(buf) free(buf);
#ifdef __WINDOWS__
	WSACleanup();
#endif
	return ok;
}

/*----------------------------------------------------------------------
  user certop : user certificate operation
----------------------------------------------------------------------*/
int aienr_user_certop(AiEnrollInfo *info){
  AccList *al = NULL;
  Cert *ct = NULL;
  char path[256],cs[256],caDN[256],*name,*op;
  char *apn,*apv,*buf=NULL,*err="",*cp,*t,*p,*gp;
  off_t sz;
  int i,j,k,bk=0,ie7=0,ok=-1;

  /* get user info */
  op = cgi_find_query(&info->qc,"CertOP");

  /* check browser */
  apn = cgi_find_query(&info->qc,"appName");
  apv = cgi_find_query(&info->qc,"appVersion");
  if(!strcmp(apn,"Microsoft Internet Explorer")){
	if(strstr(apv,"MSIE 7.0")){
		if(atof(strstr(apv,"Windows NT")+10) >= 6.0){
			ie7 = 1; /* IE7.0 and VISTA */
		}
	} 
    bk = 0; /* IE user */
  }else if(!strcmp(apn,"Netscape")){
    bk = 1; /* Netscape / FireFox */
  }else{
    err = "unknown browser"; goto done;
  }

  /* check user session */
  if(cgi_load_session(info->sespath,&info->list)){
    err = "cannot load session file"; goto done;
  }  
  name = cgi_find_cookie(&info->qc,"ENLoginName");
  cp = cgi_find_cookie(&info->qc,"ENSessionID");
  gp = cgi_find_cookie(&info->qc,"ENGroupName");
  if(*name==0){ err="cannot get cookie (login)"; goto done; }
  if(*gp  ==0){ err="cannot get cookie (group)"; goto done; }
  if(*cp  ==0){ err="cannot get cookie (sessionID)"; goto done; }
  if(cgi_check_sessiongrp(info->list,name,cp,gp)){
    err = "check session error"; goto done;
  }

  /* find enroll user name */
  if((al=cgi_findbynamegrp(info->list,name,gp))==NULL){
    err = "cannot find access list"; goto done;
  }

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");

  /* get source file name */
  if(!strcmp(op,"revoke")){
    strcpy(path,AI_REVOKEOP_HTM);
  }else if(!strcmp(op,"update")){
    if(ie7) strcpy(path,AI_UPDATEOP_IE7_HTM);
	else    strcpy(path,AI_UPDATEOP_HTM);
  }else if(!strcmp(op,"newcsr") || !strcmp(op,"textreq") || !strcmp(op,"certdl")){

    memset(caDN,0,256);

    /* get CA certificate */
    strcpy (path,"..");
    strcat (path,"/");
    strncat(path,info->ca.caname,32);
    strcat (path,".cer");

    if((ct=Cert_read_file(path)) != NULL) {

      for(i=j=0; i<ct->subject_dn.num; i++){
	switch(ct->subject_dn.rdn[i].tagoid){
	case OBJ_DIR_C:  cp="C="; j=1; break;
	case OBJ_DIR_ST: cp="ST="; j=1; break;
	case OBJ_DIR_L:  cp="L="; j=1; break;
	case OBJ_DIR_O:  cp="O="; j=1; break;
	case OBJ_DIR_OU: cp="OU="; j=1; break;
	default: cp=""; j=0; break;
	}

	if(j){
	  k = 254 - strlen(caDN);
	  if(k<254) strncat(caDN,",",k); k--;
	  strncat(caDN,cp,k); k-=strlen(caDN);
	  strncat(caDN,ct->subject_dn.rdn[i].tag,k);
	}
      }
    }

    /* get html template */
    if(!strcmp(op,"textreq")){
      strcpy(path,AI_SENDCSR_FORM3_HTM);
    }else if (!strcmp(op,"certdl")){
      if(bk)       strcpy(path,AI_SENDCSR_DL_FORM2_MZ_HTM);
      else         strcpy(path,AI_SENDCSR_DL_FORM2_HTM);
    }else{
      if(bk)       strcpy(path,AI_SENDCSR_FORM2_MZ_HTM);
      else if(ie7) strcpy(path,AI_SENDCSR_FORM2_IE7_HTM);
      else         strcpy(path,AI_SENDCSR_FORM2_HTM);
    }
  }else{
    err = "unknown operation"; goto done;
  }

  /* send next page */
  if((buf=(char*)get_file2buf(get_html_name(info,path),&sz))==NULL) goto done;

  cp = buf;
  do{
    if((t=strstr(cp,"${"))==NULL){ printf("%s",cp); break; }
    *t=0; t++;

    if(!memcmp(t,"{CONTAINTIME}",13)){
      sprintf(cs,"%lu",al->containerTime);
      p = cs; t+=13;
    }else if(!memcmp(t,"{CASBJDN}",9)){
      p = caDN; t+=9;
    }else if(!memcmp(t,"{EMAIL}",7)){
      p = al->email; t+=7;
    }else if(!memcmp(t,"{USERNAME}",10)){
      p = (*(al->cn))?(al->cn):(name); t+=10;
    }else if(!memcmp(t,"{GROUPNAMES}",12)){
      p = gp; t+=12;
    }else if(!memcmp(t,"{PWMODE}",8)){
      sprintf(cs,"%d",(info->authmode==1)?(1):(0)); t+=8;
      p = cs;
    }else{
      p = "$";
    }
    printf("%s",cp);
    printf("%s",p);
    cp = t;
  }while(cp);

  ok = 0;
done:
  if(ok){
    printf("Content-type: text/html\n\n");
    aienr_print_err(info,"cert op : System error!!",err,NULL);
  }
  if(buf) free(buf);
  if(ct) Cert_free(ct);
  return ok;
}

/*----------------------------------------------------------------------
  revoke cert : revoke user certificate operation
----------------------------------------------------------------------*/
int aienr_revoke_cert(AiEnrollInfo *info){
  AccList *al = NULL;
  LCMP *lcmp = NULL;
  char cs[256],*name,*rs;
  char *buf=NULL,*err="",*cp,*gp;
  off_t sz;
  int i,mode=0,code=0,ok=-1,prevmode;

#ifdef __WINDOWS__
  WSADATA wsaData;

  if(WSAStartup(MAKEWORD(1, 1), &wsaData) != 0) {
	err = "failed to initialize winsock"; goto done;
  }
#endif
  /* get user info */
  rs = cgi_find_query(&info->qc,"Reason");

  /* check user session */
  if(cgi_load_session(info->sespath,&info->list)){
    err = "cannot load session file"; goto done;
  }
  name = cgi_find_cookie(&info->qc,"ENLoginName");
  gp   = cgi_find_cookie(&info->qc,"ENGroupName");
  cp = cgi_find_cookie(&info->qc,"ENSessionID");
  if(*name==0){ err="cannot get cookie (login)"; goto done; }
  if(*gp  ==0){ err="cannot get cookie (group)"; goto done; }
  if(*cp  ==0){ err="cannot get cookie (sessionID)"; goto done; }
  if(cgi_check_sessiongrp(info->list,name,cp,gp)){
    err = "check session error"; goto done;
  }

  /* find enroll user name */
  if((al=cgi_findbynamegrp(info->list,name,gp))==NULL){
    err = "cannot find access list"; goto done;
  }

  /* revoked check */
  if(al->mode==AI_ENL_REVOKED){
    err = "certificate is already revoked"; goto done;
  }
  prevmode = al->mode;

  /* connect CA and revoke Cert */
  if(!info->offlineca){
    if((lcmp=LCMP_init(info->ca.svname,info->ca.caport,info->ca.caname))==NULL){
      err = "cannot initialize CA connection"; goto done;
    }
    if(info->ca.usessl){
      if(LCMP_set_ssl(lcmp,info->ca.store,info->ca.certid,info->ca.clctpw,info->ca.vfycert)){
	err = "cannot set SSL certificate"; goto done;
      }
      mode = LCMP_OPBRQ_ATSSLCL;
    }
    if(LCMP_bind_s(lcmp,info->ca.userid,info->ca.pwd,mode)){
      err = "cannot bind CA server"; goto done;
    }
    if((code=LCMP_cert_s(lcmp,LCMP_OPCT_RVKCERT,al->serialNum,atoi(rs),NULL,NULL))<0){
      err = "cannot revoke a certificate on the CA server"; goto done;
    }
  }

  /* show result page */
  if(code){
    sprintf(cs,"CA returns error code : %.8x",code);
    err = cs; goto done;
  }

  /* save session info */
  al->mode = AI_ENL_REVOKED;
  if(cgi_update_session(info->sespath,al)){
    err="cannot update session file"; goto done;
  }

  /* send email to admin */
  if(*info->ml.smtphost && *info->ml.admemail && prevmode!=AI_ENL_WAITUPD){
    char *ml = (info->offlineca)?("enroll_revoke_user2.txt"):("enroll_revoke_user.txt");

    if(info->authmode==AE_AUTH_LCHA){
      for(i=0; i<MAXGROUP; i++){
	if(!strcmp(al->group,info->grpname[i])){
	  strncpy(info->ml.admemail,info->grpemail[i],126); break;
	}
      }
    }
    cgi_get_deflang(info->ml.lang);
    if(aienr_sendmail_sys2adm(&info->ml,al,info->ml.admemail,al->cn,ml)){
      sprintf(cs,"%s : cannot send a revoke email to admin (sn=%.7d) << %s",
	      al->name,al->serialNum,CA_get_errstr());
      ENCERRLOG(info->ca.caname,cs);
    }
  }

  /* print page */
  if((buf=(char*)get_file2buf(get_html_name(info,AI_REVOKEDONE_HTM),&sz))==NULL) goto done;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  printf("%s",buf);

  /* output log */
  sprintf(cs,"%s : success to revoke a certificate (sn=%.7d,accID=%.7d,updperiod=%d)",
             al->name,al->serialNum,al->acceptID,(prevmode==AI_ENL_WAITUPD)?(1):(0));
  ENCACCLOG(info->ca.caname,cs);

  ok = 0;
done:
  if(ok){
    printf("Content-type: text/html\n\n");
    aienr_print_err(info,"revoke op : System error!!",err,lcmp);
  }
  if(lcmp){ LCMP_unbind_s(lcmp); LCMP_free(lcmp); }
  if(buf) free(buf);
#ifdef __WINDOWS__
  WSACleanup();
#endif
  return ok;
}

/*----------------------------------------------------------------------
  update cert : update user certificate operation
----------------------------------------------------------------------*/
int aienr_update_cert(AiEnrollInfo *info){
  AccList *al = NULL;
  LCMP *lcmp = NULL;
  Cert *ct = NULL;
  PKCS7 *p7 = NULL;
  char cs[256],*name;
  char *apv,*buf=NULL,*p7c=NULL,*p1,*p2,*cp,*err="",*gp;
  unsigned char *der=NULL;
  off_t sz;
  int i,op,code,ie7=0,mode=0,ok=-1;

#ifdef __WINDOWS__
  WSADATA wsaData;

  if(WSAStartup(MAKEWORD(1, 1), &wsaData) != 0) {
	err = "failed to initialize winsock"; goto done;
  }
#endif
  /* check user session */
  if(cgi_load_session(info->sespath,&info->list)){
    err = "cannot load session file"; goto done;
  }
  name = cgi_find_cookie(&info->qc,"ENLoginName");
  gp   = cgi_find_cookie(&info->qc,"ENGroupName");
  cp   = cgi_find_cookie(&info->qc,"ENSessionID");
  if(*name==0){ err="cannot get cookie (login)"; goto done; }
  if(*gp  ==0){ err="cannot get cookie (group)"; goto done; }
  if(*cp  ==0){ err="cannot get cookie (sessionID)"; goto done; }
  if(cgi_check_sessiongrp(info->list,name,cp,gp)){
    err = "check session error"; goto done;
  }
  /* check ie7 */
  apv = cgi_find_query(&info->qc,"appVersion");
  if(strstr(apv,"MSIE 7.0")){
	if(atof(strstr(apv,"Windows NT")+10) >= 6.0){
		ie7 = 1; /* IE7.0 and VISTA */
	}
  } 

  /* find enroll user name */
  if((al=cgi_findbynamegrp(info->list,name,gp))==NULL){
    err = "cannot find access list"; goto done;
  }

  /* connect CA and update Cert */
  if((lcmp=LCMP_init(info->ca.svname,info->ca.caport,info->ca.caname))==NULL){
    err = "cannot initialize CA connection"; goto done;
  }
  if(info->ca.usessl){
    if(LCMP_set_ssl(lcmp,info->ca.store,info->ca.certid,info->ca.clctpw,info->ca.vfycert)){
      err = "cannot set SSL certificate"; goto done;
    }
    mode = LCMP_OPBRQ_ATSSLCL;
  }
  if(LCMP_bind_s(lcmp,info->ca.userid,info->ca.pwd,mode)){
    err = "cannot bind CA server"; goto done;
  }
  if((code=LCMP_cert_s(lcmp,LCMP_OPCT_UPDCERT,al->serialNum,0,NULL,NULL))<0){
    err = "cannot update a certificate on the CA server"; goto done;
  }
  if((ct=LCMP_get_updcert(lcmp))==NULL){
    err = "cannot get updated certificate."; goto done;
  }

  /* show result error code */
  if(code){
    sprintf(cs,"CA returns error code : %.8x",code);
    err = cs; goto done;
  }
  
  /* set cookie */
  printf("Set-Cookie: ENAcceptID=%.7d\n", al->acceptID);
  printf("Set-Cookie: ENSerialNum=%.7d\n", al->serialNum);

  /* get source file */
  cp = (ie7)?(AI_ACCEPT_FORM_IE7_HTM):(AI_ACCEPT_FORM_HTM);
  if((buf=(char*)get_file2buf(get_html_name(info,cp),&sz))==NULL) goto done;

  /* out to ldap server */
#ifdef HAVE_LDAP_H
  if(info->ldaplogin){
    op = (info->authmode==AE_AUTH_LCHA)?(LDAP_MOD_REPLACE):(LDAP_MOD_ADD);
    if(aienr_ldap_addcert(info,op|LDAP_MOD_BVALUES,ct,NULL,NULL,NULL,name)){
      sprintf(cs,"%s : cannot output certificate to LDAP (sn=%.7d)",
	      al->name,ct->serialNumber);
      ENCERRLOG(info->ca.caname,cs);
    }
  }
#endif

  /* get pkcs7 */
  if((p7 = P7_new(OBJ_P7_SIGNED))==NULL) goto done;
  if(P12_add_cert((PKCS12*)p7,ct,NULL,0xff)) goto done;
  ct = NULL;

  if((der=P7_signed_toDER(p7,NULL,&i))==NULL) goto done;
  if((p7c=Base64_encode(i,der,32))==NULL) goto done;

  /* set return PKCS7 */
  if((cp=strstr(buf,"${RETURNPKCS7}"))==NULL){
    err="cannot find PKCS7 tag"; goto done;
  }
  *cp = 0;

  cgi_check_lang(info);
  printf("Content-type: text/html\n\n");
  printf("%s",buf);

  p1=p2=p7c;
  while(p1 && *p1){
    if((p2=strchr(p1,'\n')) != NULL) { *p2=0; p2++; }
    printf("pkcs7+=\"");
    printf("%s",p1);
    printf("\";\n");
    p1 = p2;
  }
  printf("%s", cp+14);

  /* output log */
  sprintf(cs,"%s : success to update a certificate (sn=%.7d,accID=%.7d)",
             al->name,al->serialNum,al->acceptID);
  ENCACCLOG(info->ca.caname,cs);

  ok = 0;
done:
  if(ok){
    printf("Content-type: text/html\n\n");
    aienr_print_err(info,"update op : System error!!",err,lcmp);
  }
  if(lcmp){ LCMP_unbind_s(lcmp); LCMP_free(lcmp); }
  Cert_free(ct);
  if(p7) P7_free(p7);
  if(der) free(der);
  if(p7c) free(p7c);
  if(buf) free(buf);
#ifdef __WINDOWS__
  WSACleanup();
#endif
  return ok;
}

/*----------------------------------------------------------------------
  get cert : get user certificate operation
----------------------------------------------------------------------*/
int aienr_get_cert(AiEnrollInfo *info){
	AccList *al = NULL;
	PKCS7 *cap7=NULL, *p7=NULL;
	Cert *ct = NULL;
	char path[256],cs[256];
	char *buf=NULL,*p7c=NULL,*p1,*p2,*cp,*t,*err="",*pwd=NULL;
	unsigned char *der=NULL;
	off_t sz;
	int i,op,accID=0,ok=-1;

	/* get accept ID */
	cp = cgi_find_query(&info->qc,"acceptID");

	accID = atoi(cp); 
	printf("Set-Cookie: ENAcceptID=%.7d\n", accID);

	/* check user name */
	if(cgi_load_session(info->sespath,&info->list)){
		err = "cannot load session file"; goto done;
	}

	/* set session mode */
	if((al = cgi_findbyaccid(info->list,accID))==NULL){
		err = "cannot get session info"; goto done;
	}
	if((pwd=aica_decry_passwd(al->pwd))==NULL){
		err="cannot decrypt session password"; goto done;
	}

	/* get source file */
	if(al->isstype & AI_IS_COPYP10){
		cp = AI_ACCEPT_FORM2_HTM;	
	}else if(al->isstype & AI_IS_MOZILLA){
		cp = (al->isstype & AI_IS_SAFARI)?(AI_ACCEPT_FORM_SF_HTM):(AI_ACCEPT_FORM_MZ_HTM);
	}else{
		cp = (al->isstype & AI_IS_IE7)?(AI_ACCEPT_FORM_IE7_HTM):(AI_ACCEPT_FORM_HTM);
	}
	if((buf=(char*)get_file2buf(get_html_name(info,cp),&sz))==NULL) goto done;

	/* get pkcs7 */
	snprintf(path,254,"%s%sout%su%.7d.p7b",info->rapath,"/","/",accID);

	if((p7 = P7b_read_file(path))==NULL) goto done;
	if(P12_check_chain((PKCS12*)p7,0)) goto done;
	if((der=P7_signed_toDER(p7,NULL,&i))==NULL) goto done;

	if((p7c=Base64_encode(i,der,32))==NULL) goto done;
	if((ct=P12_get_usercert((PKCS12*)p7))==NULL) goto done;

	/* set cookie */
	printf("Set-Cookie: ENSerialNum=%.7d\n", ct->serialNumber);

#ifdef HAVE_LDAP_H
	if(info->ldaplogin){
		/* add certificate & remove license ID */
		op = (info->authmode==AE_AUTH_LCHA)?(LDAP_MOD_REPLACE):(LDAP_MOD_ADD);
		if(aienr_ldap_addcert(info,op|LDAP_MOD_BVALUES,ct,al->name,al->group,pwd,al->name)){
			char cs[256];
			sprintf(cs,"%s : cannot output certificate to LDAP (sn=%.7d)",
				al->name,ct->serialNumber);
			ENCERRLOG(info->ca.caname,cs);
		}
	}else
#endif
	if(info->authmode==AE_AUTH_LICE){
		/* remove license ID from en.license file */
		if(aienr_remove_licensefile(info,al->name,al)){ err="cannot remove license id"; goto done; }
	}

	/* clear user password in session file */
	memset(al->pwd,0,64);
	al->serialNum = ct->serialNumber;
	al->notAfter  = (unsigned long)timegm(&ct->time.notAfter);
	if(al->isstype & AI_IS_COPYP10){ al->mode = AI_ENL_ISSDONE; }

	if(cgi_update_session(info->sespath,al)){
		err="cannot update session file"; goto done;
	}

	/* set return certificate */
	cgi_check_lang(info);
	printf("Content-type: text/html\n\n");

	cp = buf;
	do{
		if((t=strstr(cp,"${"))==NULL){ printf("%s",cp); break; }
		*t=0; t++;

		printf("%s",cp);
		if(!memcmp(t,"{PRINTCERT}",11)){
			CERT_PRINT(ct); t+=11;
		}else if(!memcmp(t,"{RETURNCERT}",12)){
			p1 = PEM_write_cert_buf(ct); t+=12;
			printf("%s",p1); free(p1);
		}else if(!memcmp(t,"{RETURNPKCS7}",13)){
			p1=p2=p7c;
			while(p1 && *p1){
				if((p2=strchr(p1,'\n')) != NULL) { *p2=0; p2++; }
				printf("pkcs7+=\"%s\";\n",p1);
				p1 = p2;
			}
			t+=13;
		}else if(!memcmp(t,"{DOWNLOADCERT}",14)){
			/* for Safari mode */
#ifdef __WINDOWS__
			sprintf(cs,"sf%.7d.dl.pem",ct->serialNumber);
			if(PEM_write_cert(ct,cs)) goto done;
#else
			sprintf(cs,"out/sf%.7d.dl.pem",ct->serialNumber);
			sprintf(path,"../out/sf%.7d.dl.pem",ct->serialNumber);
			if(PEM_write_cert(ct,path)) goto done;
#endif
			printf("%s",cs); t+=14;
		}else if(!memcmp(t,"{DOWNLOADCACERT}",16)){
			/* for Safari mode */
			snprintf(path,254,"%s%s%s.p7b",info->rapath,"/",info->ca.caname);
			if((cap7 = P7b_read_file(path))==NULL) goto done;

#ifdef __WINDOWS__
			SNPRINTF (cs,254,"%s.p7b",info->ca.caname);
			if(P7b_write_file(cap7,cs)) goto done;
#else
			snprintf(cs,254,"out/%s.p7b",info->ca.caname);
			snprintf(path,254,"../out/%s.p7b",info->ca.caname);
			if(P7b_write_file(cap7,path)) goto done;
#endif
			printf("%s",cs); t+=16;
		}else if(!memcmp(t,"{CLEARCT}",9)){
			/* for Safari mode */
#ifdef __WINDOWS__
			sprintf(cs,"sf%.7d",ct->serialNumber);
#else
			sprintf(cs,"../out/sf%.7d",ct->serialNumber);
#endif
			printf("%s",cs); t+=9;
		}else{
			printf("$");
		}
		cp = t;
	}while(cp);

	ct = NULL;

	ok = 0;
done:
	if(ok){
		printf("Content-type: text/html\n\n");
		aienr_print_err(info,"get cert : System error!!",err,NULL);
	}
	if(p7) P7_free(p7);
	if(cap7) P7_free(cap7);
	if(der) free(der);
	if(p7c) free(p7c);
	if(pwd) free(pwd);
	if(buf) free(buf);
	return ok;
}

/*----------------------------------------------------------------------
	get cert done : confirm certificate installation
----------------------------------------------------------------------*/
int aienr_get_certdone(AiEnrollInfo *info){
	AccList *al = NULL;
	char cs[256],*name,*sn;
	char *buf=NULL,*cer="",*err="",*gp,*fm,*gn;
	off_t sz;
	int accID,snum,ok=-1;

	fm = cgi_find_cookie(&info->qc,"ENFm");
	gn = cgi_find_cookie(&info->qc,"ENGn");

	/* set content */
	cgi_check_lang(info);
	printf("Content-type: text/html\n\n");

	/* delete temporary cert file (for Safari) */
	cer = cgi_find_query(&info->qc,"ClearCT");
	if(*cer){
		sprintf(cs,"%s.dl.pem",cer);
		unlink(cs);
	}

	/* check user name */
	if(cgi_load_session(info->sespath,&info->list)){
		err = "cannot load session file"; goto done;
	}
	if(info->postmode || info->offlineca){
		sn = cgi_find_cookie(&info->qc,"ENAcceptID");
		if(*sn == 0){ err="cannot get cookie (acceptID)"; goto done; }
		accID = atoi(sn);
	}else{
		name = cgi_find_cookie(&info->qc,"ENLoginName");
		if(*name == 0){ err="cannot get cookie (login)"; goto done; }
		gp = cgi_find_cookie(&info->qc,"ENGroupName");
		if(*gp == 0){ err="cannot get cookie (group)"; goto done; }
	}
	sn = cgi_find_cookie(&info->qc,"ENSerialNum");
	if(*sn == 0){ err="cannot get cookie (ENSerialNum)"; goto done; }
	snum = atoi(sn);

	/* set session mode */
	if(info->postmode || info->offlineca){
		al = cgi_findbyaccid(info->list,accID);
	}else{
		al = cgi_findbynamegrp(info->list,name,gp);
	}
	if(al){
		al->serialNum = snum;
		al->mode = AI_ENL_ISSDONE;
		memset(al->sessionID,0,32);
		if(cgi_update_session(info->sespath,al)){ err="cannot update session file"; goto done; }
	}

	/* get source file */
	if((buf=(char*)get_file2buf(get_html_name(info,AI_DONE_FORM_HTM),&sz))==NULL) goto done;
	{
	  AiVList v[] = {
	    {"{FROM}",fm},
	    {"{Gp}",gn},
	    {NULL,NULL}
	  };
	  cgi_print_content(buf,v);
	}

	/* output log */
	sprintf(cs,"%s : success to import a user certificate (sn=%.7d,accID=%.7d)",
		al->name,al->serialNum,al->acceptID);
	ENCACCLOG(info->ca.caname,cs);

	ok = 0;
done:
	if(ok){
		aienr_print_err(info,"get cert done : System error!!",err,NULL);
	}
	if(buf) free(buf);
	return ok;
}

/*----------------------------------------------------------------------
	for mozzila, certificate installation
----------------------------------------------------------------------*/
int aienr_moz_certinst(AiEnrollInfo *info){
	char *p7c=NULL,*err="";
	int i,ok=-1;

	char *bg = "-----BEGIN CERTIFICATE-----\n";
	char *ed = "\n-----END CERTIFICATE-----\n";

	/* get user info */
	if((p7c = cgi_find_query(&info->qc,"PKCS7"))==NULL){
		err = "cannot get PKCS7 query"; goto done;
	}

	i = strlen(p7c) + strlen(bg) + strlen(ed);

	printf("Content-type: application/x-x509-user-cert\n");
	printf("Content-length: %d\n\n",i);

	printf("%s",bg);
	printf("%s",p7c);
	printf("%s",ed);

	/* output log */
	ENCACCLOG(info->ca.caname,"x-x509-user-cert output");

	ok = 0;
done:
	if(ok){
		printf("Content-type: text/html\n\n");
		aienr_print_err(info,"aienr_moz_certinst : System error!!",err,NULL);
	}
	return ok;
}


/*----------------------------------------------------------------------
	change user password form
----------------------------------------------------------------------*/
int aienr_change_pwd(AiEnrollInfo *info){
	char *buf=NULL,*err="";
	char *name,*cp,*gp;
	off_t sz;
	int ok=-1;

	cgi_check_lang(info);
	printf("Content-type: text/html\n\n");

	/* check user session */
	if(cgi_load_session(info->sespath,&info->list)){
		err = "cannot load session file"; goto done;
	}
	name = cgi_find_cookie(&info->qc,"ENLoginName");
	gp   = cgi_find_cookie(&info->qc,"ENGroupName");
	cp = cgi_find_cookie(&info->qc,"ENSessionID");
	if(*name==0){ err="cannot get cookie (login)"; goto done; }
	if(*cp  ==0){ err="cannot get cookie (sessionID)"; goto done; }
	if(cgi_check_sessiongrp(info->list,name,cp,gp)){
		err = "check session error"; goto done;
	}

	/* get source file */
	if((buf=(char*)get_file2buf(get_html_name(info,AI_PWD_CHANGE_HTM),&sz))==NULL) goto done;

	printf("%s",buf);

	ok = 0;
done:
	if(ok){
		aienr_print_err(info,"change password : System error!!",err,NULL);
	}
	if(buf) free(buf);
	return ok;
}

/*----------------------------------------------------------------------
	update user password
----------------------------------------------------------------------*/
int aienr_update_pwd(AiEnrollInfo *info){
	char path[256],*cp,*cgp,*name,*oldpw,*newpw;
	char *buf=NULL,*err="";
	off_t sz;
	int i,ok=-1;

	cgi_check_lang(info);
	printf("Content-type: text/html\n\n");

	/* parse posted values */
	oldpw = cgi_find_query(&info->qc,"ENLoginPasswd");
	newpw = cgi_find_query(&info->qc,"ENNewPwd");
	if(*oldpw == 0){
		err = "cannot get OldPasswd form value"; goto done;
	}
	if(*newpw == 0){
		err = "cannot get NewPasswd form value"; goto done;
	}

	/* check user session */
	if(cgi_load_session(info->sespath,&info->list)){
		err = "cannot load session file"; goto done;
	}
	name = cgi_find_cookie(&info->qc,"ENLoginName");
	cgp = cgi_find_cookie(&info->qc,"ENGroupName");
	cp = cgi_find_cookie(&info->qc,"ENSessionID");
	if(*name==0){ err="cannot get cookie (login)"; goto done; }
	if(*cgp ==0){ err="cannot get cookie (group)"; goto done; }
	if(*cp  ==0){ err="cannot get cookie (sessionID)"; goto done; }
	if(cgi_check_sessiongrp(info->list,name,cp,cgp)){
		err = "check session error"; goto done;
	}

	/* check & change user passwd */
	if((i=aienr_change_passwd(info,cgp,name,oldpw,newpw))<0){
		err = "cannot set new password."; goto done;
	}

	if(i > 0){
		strcpy(path,AI_LOGINERR_HTM);
	}else{
		strcpy(path,AI_PWD_DONE_HTM);
	}
	if((buf=(char*)get_file2buf(get_html_name(info,path),&sz))==NULL) goto done;

	printf("%s",buf);

	ok = 0;
done:
	if(ok){
		aienr_print_err(info,"update password : System error!!",err,NULL);
	}
	if(buf) free(buf);
	return ok;
}

/*----------------------------------------------------------------------
	send csr dl : send certificate request operation
----------------------------------------------------------------------*/
int aienr_send_csr_dl(AiEnrollInfo *info){
	CertTemplate *ctt = NULL;
	AccList sal,*dmy=NULL,*al=NULL;
	Key *prv = NULL;
	Key *pub = NULL;
	Cert *ct = NULL;
	PKCS12 *p12 = NULL;
	LCMP *lcmp = NULL;
	char path[256],cs[256],*name,*tm,*em,*cn;
	char *apv,*buf=NULL,*p1,*cp,*gp,*t,*err="",*pwd, *group;
	char *passwd, *keylen;
	unsigned char *der=NULL;
	off_t sz;
	int i,k,cv,len,code=0,accID=0,op,sf=0,ie7=0,mode=0,ok=-1;
#ifdef WITHAD
	char *prname;
#endif

#ifdef __WINDOWS__
	WSADATA wsaData;

	if(WSAStartup(MAKEWORD(1, 1), &wsaData) != 0) {
		err = "failed to initialize winsock"; goto done;
	}
#endif

	/* get user info */
	tm = cgi_find_query(&info->qc,"ContName");
	if(*tm == 0){
		err = "cannot get ContName form value"; goto done;
	}
	em = cgi_find_query(&info->qc,"Email"); /* maybe empty */
	cn = cgi_find_query(&info->qc,"CN");
	if(*cn == 0){
		err = "cannot get CN form value"; goto done;
	}
	group = cgi_find_query(&info->qc,"ENGroupName");
	if(*group == 0){
		err = "cannot get group name form value"; goto done;
	}
	/* check appVersion */
	apv = cgi_find_query(&info->qc,"appVersion");
	if(strstr(apv,"Safari")) sf = 1;
	if(strstr(apv,"MSIE 7.0")){
		if(atof(strstr(apv,"Windows NT")+10) >= 6.0){
			ie7 = 1; /* IE7.0 and VISTA */
		}
	} 

	/* check user session */
	if(info->authmode){
		if(cgi_load_session(info->sespath,&info->list)){
			err = "cannot load session file"; goto done;
		}
		name = cgi_find_cookie(&info->qc,"ENLoginName");
		p1   = cgi_find_cookie(&info->qc,"ENSessionID");
		if(*name==0){ err="cannot get cookie (login)"; goto done; }
		if(*p1  ==0){ err="cannot get cookie (sessionID)"; goto done; }

#ifdef WITHAD
		prname = cgi_find_cookie(&info->qc,"ENPRName");
		if(*prname==0){ err="cannot get cookie (prname)"; goto done; }
#endif

		if(info->authmode && info->ldaplogin){
			pwd = cgi_find_cookie(&info->qc,"ENLoginPasswd");
		}

		/* find enroll user name */
		gp = (info->authmode==AE_AUTH_LICE && !info->ldaplogin)?(NULL):(group);

		if(cgi_check_sessiongrp(info->list,name,p1,gp)){
			snprintf(path,254,"check session error : name:%s, id:%s, cgp:%s",name,p1,gp);
			ENCERRLOG(info->ca.caname,path);
			err = "check session error"; goto done;			  
		}
		if((al=cgi_findbynamegrp(info->list,name,gp))==NULL){
			err = "cannot find access list"; goto done;
		}
		/* user state check */
		if(al->mode==AI_ENL_WAITISS){
			err = "certificate request is already accepted"; goto done;
		}
	}else{
		/* anonymous mode */
		memset(&sal,0,sizeof(AccList)); al = &sal;
                strncpy(al->name,cn,64);
	}

	passwd = cgi_find_query(&info->qc,"Passwd");
	if(*passwd == 0){
		err = "cannot get passwd form value"; goto done;
	}
	keylen = cgi_find_query(&info->qc,"KeyLen");
	if(*keylen == 0){
		err = "cannot get key length form value"; goto done;
	}
	len = atoi(keylen);
	if(len<512 || 2048<len) {
		err = "bad key size"; goto done;
	}

	/* get certificate template for signing request */
	if((ctt=CMP_certtmpl_new())==NULL) goto done;

	/* careate key pair */
	if((prv=(Key *)RSAprvkey_new())==NULL) goto done;
	if((pub=(Key*)RSApubkey_new())==NULL) goto done;
	{
		uint32_t p1s[LN_MAX],q1s[LN_MAX],phis[LN_MAX];
		LNm	p1,q1,phi,*tmp;
		int	i,err;

		p1.num=p1s; q1.num=q1s; phi.num=phis;
		p1.size=q1.size=phi.size=LN_MAX;

		/* generate p and q */
		if(LN_prime(len>>4,((Prvkey_RSA *)prv)->p,0)) {
			if(((Prvkey_RSA *)prv)->der) free(((Prvkey_RSA *)prv)->der);
			goto done;
		}
		if(LN_prime(len>>4,((Prvkey_RSA *)prv)->q,0)) {
			if(((Prvkey_RSA *)prv)->der) free(((Prvkey_RSA *)prv)->der);
			goto done;
		}

		/* because LN_mod_inverse() might be faster */
		if(LN_cmp(((Prvkey_RSA *)prv)->p,((Prvkey_RSA *)prv)->q)<0){ /* if p < q */
			tmp=((Prvkey_RSA *)prv)->p;
			((Prvkey_RSA *)prv)->p = ((Prvkey_RSA *)prv)->q;
			((Prvkey_RSA *)prv)->q = tmp;
		}

		/* set p1,q1,phi */
		LN_copy(((Prvkey_RSA *)prv)->p,&p1);
		LN_copy(((Prvkey_RSA *)prv)->q,&q1);

		/* p or q is prime, so last bit must have "1." */
		p1s[LN_MAX-1]&=0xfffffffe;	/* p1=p-1 */
		q1s[LN_MAX-1]&=0xfffffffe;	/* q1=q-1 */
		if(LN_multi(&p1,&q1,&phi)) {
			if(((Prvkey_RSA *)prv)->der) free(((Prvkey_RSA *)prv)->der);
			goto done;
		}

		LN_long_set(((Prvkey_RSA *)prv)->e,0x10001L);			/* e = 0x10001 */
		err = LN_multi(((Prvkey_RSA *)prv)->p,((Prvkey_RSA *)prv)->q,((Prvkey_RSA *)prv)->n);      	/* n =p*q */
		err|= LN_mod_inverse(((Prvkey_RSA *)prv)->e,&phi,((Prvkey_RSA *)prv)->d);	/* d = e^-1 mod phi */
		if(err) {
			if(((Prvkey_RSA *)prv)->der) free(((Prvkey_RSA *)prv)->der);
			goto done;
		}

		err = LN_div_mod(((Prvkey_RSA *)prv)->d,&p1,&phi,((Prvkey_RSA *)prv)->e1);	/* e1= d mod p-1 */
		err|= LN_div_mod(((Prvkey_RSA *)prv)->d,&q1,&phi,((Prvkey_RSA *)prv)->e2);	/* e2= d mod q-1 */
		err|= LN_mod_inverse(((Prvkey_RSA *)prv)->q,((Prvkey_RSA *)prv)->p,((Prvkey_RSA *)prv)->cof);	/* cof = q^-1 mod p */
		if(err) {
			if(((Prvkey_RSA *)prv)->der) free(((Prvkey_RSA *)prv)->der);
			goto done;
		}

		((Prvkey_RSA *)prv)->size=(len>>4)*2;

		if((((Prvkey_RSA *)prv)->der=RSAprv_toDER(((Prvkey_RSA *)prv),NULL,&i))==NULL) {
			if(((Prvkey_RSA *)prv)->der) free(((Prvkey_RSA *)prv)->der);
			goto done;
		}
	}

	RSAprv_2pub((Prvkey_RSA *)prv,(Pubkey_RSA*)pub);
	ctt->publicKey = pub; pub = NULL;

	cp = cgi_find_query(&info->qc,"CaDN");
	if(aienr_line2dn(cp,&ctt->subject,&err)) goto done;

#ifdef WITHAD
	{
		ExtGenNames *egn=NULL;
		CertExt *et=NULL;
		if((egn=ExtGN_set_upnname(prname))==NULL) goto done;
		if((et=Extnew_altname(OBJ_X509v3_SbjAltName,egn))==NULL) goto done;
		if((ctt->ext=Extnew_extreq(et))==NULL) goto done;
	}

	/* set prname as cn */
	//strncpy(al->cn,prname,62);
#endif

	/* replace subject cn and email from user data source */
	if(al && *al->cn && al->mode!=AI_ENL_REJECTED){
		cn = al->cn;
		if(Cert_find_dn(&ctt->subject,OBJ_DIR_CN,&i,0)){
			len = strlen(cn);
			for(k=cv=0; k<len; k++) cv |= 0x80 & cn[k];

			free(ctt->subject.rdn[i].tag);
			if(cv){ /* maybe JP */
			  UC_conv(UC_CODE_UTF8,UC_LOCAL_CODESET,cn,strlen(cn),cs,254);
			  cn = cs;
			}
			if ((ctt->subject.rdn[i].tag = strdup(cn)) == NULL) goto done;
			ctt->subject.rdn[i].derform = asn1_str_type(cn);
			cn = al->cn;
		}
	}
	if(al && *al->email && al->mode!=AI_ENL_REJECTED){
		em = al->email;
		if(Cert_find_dn(&ctt->subject,OBJ_DIR_EMAIL,&i,0)){
			free(ctt->subject.rdn[i].tag);
			if ((ctt->subject.rdn[i].tag = strdup(em)) == NULL) goto done;
			ctt->subject.rdn[i].derform = asn1_str_type(em);
		}
	}

	/* email subject filter check */
	switch(info->emsbjfilter){
	case 1: /* in cert extension. subjectAltName */
	case 2: /* not in cert. remove from subject DN */
		/* remove email from subject DN */
		if(Cert_find_dn(&ctt->subject,OBJ_DIR_EMAIL,&i,0)){
		  free(ctt->subject.rdn[i].tag); 
		  ctt->subject.rdn[i].tag = NULL;
		  ctt->subject.rdn[i].tagoid = 0;
		  ctt->subject.num--;
		}
		/* add to subjectAltName */
		if(info->emsbjfilter==1 && em && *em){
		  ExtGenNames *egn=NULL;
		  CertExt *et=NULL;
		  
		  if((egn=ExtGN_set_email(em))==NULL) goto done;
		  if((et=CertExt_find(ctt->ext,OBJ_X509v3_SbjAltName))==NULL){
		    if((et=Extnew_altname(OBJ_X509v3_SbjAltName,egn))==NULL) goto done;
		  }else{
		    egn->next = ((CE_SbjAltName*)et)->egn;
		    ((CE_SbjAltName*)et)->egn = egn;
		  }
		  if((ctt->ext=Extnew_extreq(et))==NULL) goto done;
		}
		break;
	default: /* noting to do */
		break;
	}

	/* connect CA and send CSR (!offline ca mode) */
	if(!info->offlineca){
		mode = LCMP_OPBRQ_ATSSLCL;
		if((lcmp=LCMP_init(info->ca.svname,info->ca.caport,info->ca.caname))==NULL){
			err = "cannot initialize CA connection"; goto done;
		}
		if(info->ca.usessl){
			if(LCMP_set_ssl(lcmp,info->ca.store,info->ca.certid,info->ca.clctpw,info->ca.vfycert)){
			  err = "cannot set SSL certificate"; goto done;
			}
			mode = LCMP_OPBRQ_ATSSLCL;
		}
		if(LCMP_bind_s(lcmp,info->ca.userid,info->ca.pwd,mode)){
			err = "cannot bind CA server"; goto done;
		}
	}

	/* (online) posting a CSR or issuing a certificate / (offline) posting */
	if(info->postmode || info->offlineca){
		if(info->postmode){ /* CSR post mode */
			if((code=LCMP_csr_s(lcmp,LCMP_OPCSR_POST,0,0,NULL,NULL,0,ctt))<0){
				err = "cannot post a CSR to CA server"; goto done;
			}
			accID = LCMP_get_acceptid(lcmp);

		}else{ /* offline ca mode */
			if((accID = aienr_get_count(info,"reg"))<0){
				err = "cannot get reg counter"; goto done;
			}
			/* output pkcs10 file (only need csr->der) */
#ifdef __WINDOWS__
			sprintf(path,"..\\req\\u%.7d.csr",accID);
#else
			sprintf(path,"../req/u%.7d.csr",accID);
#endif
			if((der=CMPtmpl_toDER(ctt,&i))==NULL){
				err = "cannot get certTmpl DER"; goto done;
			}
			if(ASN1_write_der(der,path)){
				err = "cannot output certTmpl file"; goto done;
			}
			/*
			if(PEM_write_req(csr,path)){ err = "cannot output pkcs10 file"; goto done; }
			*/
		}

		/* show result page */
		if(code==0){ 
			/* no error - save session info */
			al->isstype &= ~AI_IS_COPYP10;
			al->isstype |= AI_IS_DLP12;
			al->isstype |= AI_IS_MOZILLA;
			if(sf) al->isstype |= AI_IS_SAFARI; /* set safari flag */

			al->isstype       |= cgi_get_langnum(info);
			al->mode          = AI_ENL_WAITISS;
			al->acceptID      = accID;
			al->containerTime = atol(tm);
			strncpy(al->cn,cn,62);
			strncpy(al->email,em,62);
			strncpy(al->group,group,62);
			/* clear session ID ... aienroll checks sessionID [0]==0 && [1]==0.
			 *  If they are true, it returns grid-certreq accept mail. 
			 *  so, in web enroll, we need sessionID to fulfill with 0xff.
			 */
			memset(al->sessionID,0xff,32);

			if((info->authmode==AE_AUTH_IDPW || info->authmode==AE_AUTH_WEB)&& info->ldaplogin){
				strncpy(al->pwd,pwd,62);
			}
			if(cs_get_keyhash(ctt->publicKey,al->keyID,&i)){ err="cannot get key id"; goto done; }

			/* if anonymous mode, add new session block */
			if(!info->authmode){
				/* set acceptID as user name */
				snprintf(al->name,62,"anon-ac-%.8d",accID);
				if(cgi_add_session(info->sespath,&dmy,al)){
					err="cannot add new session block"; goto done;
				}
			}
			if(cgi_update_session(info->sespath,al)){ 
				err="cannot update session file"; goto done;
			}
			strcpy(path,AI_SUBMITDONE_HTM);
			sprintf(cs,"%.7d",accID);
		}else{
			strcpy(path,AI_SUBMITERR_HTM);
			sprintf(cs,"%.8x",code);
		}
		
		if((buf=(char*)get_file2buf(get_html_name(info,path),&sz))==NULL) goto done;
		
		cgi_check_lang(info);
		printf("Content-type: text/html\n\n");

		/* set appropriate message */
		cp = buf;
		do{
			if((t=strstr(cp,"${"))==NULL){ printf("%s",cp); break; }
			*t=0; t++;
			
			if(!memcmp(t,"{ACCEPTID}",10)){
				p1 = cs; t+=10;
			}else if(!memcmp(t,"{PWMODE}",8)){
				sprintf(path,"%d",(info->authmode==1)?(1):(0)); t+=8;
				p1 = path;
			}else{
				p1 = "$";
			}
			printf("%s",cp);
			printf("%s",p1);
			cp = t;
		}while(cp);
		
		/* send email to admin */
		if(*info->ml.smtphost && *info->ml.admemail){
			char *ml = "enroll_accept_csr.txt";
			cgi_get_deflang(info->ml.lang);
			if(aienr_sendmail_sys2adm(&info->ml,al,info->ml.admemail,cn,ml)){
				sprintf(cs,"%s : cannot send a email to admin (accID=%.7d) << %s",
						al->name,al->acceptID,CA_get_errstr());
				ENCERRLOG(info->ca.caname,cs);
			}
		}

		/* output log */
		sprintf(cs,"%s : success to accept a CSR (accID=%.7d)",al->name,al->acceptID);
		ENCACCLOG(info->ca.caname,cs);

	}else{
		/* get PKCS12 (ca certificates) */
		if((p12=P12_dup(lcmp->ca->p12))==NULL) goto done;

		/* Sign direct mode */
		gp = aienr_get_profname(info,group);

		if((code=LCMP_sign_s(lcmp,gp,0,ctt,CMPREQUEST))<0){
			err = "cannot issue a certificate."; goto done;
		}

		if((ct=LCMP_get_signcert(lcmp))==NULL){
			err = "cannot get issued certificate."; goto done;
		}

		if(code==0){
			/* no error - save session info */
			al->mode     = AI_ENL_WAITISS;
			al->isstype &= ~AI_IS_COPYP10;
			al->isstype |= AI_IS_DLP12;
			al->isstype |= AI_IS_MOZILLA;
			if(sf) al->isstype |= AI_IS_SAFARI; /* set safari flag */

			al->isstype       |= cgi_get_langnum(info);
			al->containerTime = atol(tm);
			al->serialNum     = ct->serialNumber;
			al->notAfter      = (unsigned long)timegm(&ct->time.notAfter);
			strncpy(al->cn,cn,62);
			strncpy(al->email,em,62);
			strncpy(al->group,group,62);
			memset(al->sessionID,0xff,32); /* clear session ID */

			if(cs_get_keyhash(ctt->publicKey,al->keyID,&i)){ err="cannot get key id"; goto done; }
			/* if anonymous mode, add new session block */
			if(!info->authmode){
				/* set acceptID as user name */
				snprintf(al->name,62,"anon-sn-%.8d",al->serialNum);
				if(cgi_add_session(info->sespath,&dmy,al)){
					err="cannot add new session block"; goto done;
				}
				printf("Set-Cookie: ENLoginName=%s\n", al->name);
			}
			if(cgi_update_session(info->sespath,al)){ 
				err="cannot update session file"; goto done;
			}
		}

		/* output a certificate file for UNICORE mapping */
		if(info->gridcert[0]){
			char path[320];
			snprintf(path,318,"%s/%s.pem",info->gridcert,al->name);
			if(PEM_write_cert(ct,path)){
				sprintf(cs,"%s : cannot output a certificate to file (sn=%.7d)",
						al->name,ct->serialNumber);
				ENCERRLOG(info->ca.caname,cs);
			}
		}

		/* out to ldap server */
#ifdef HAVE_LDAP_H
		if(info->ldaplogin){
			/* add certificate & remove license ID */
			op = (info->authmode==AE_AUTH_LCHA)?(LDAP_MOD_REPLACE):(LDAP_MOD_ADD);
			if(aienr_ldap_addcert(info,op|LDAP_MOD_BVALUES,ct,NULL,NULL,NULL,name)){
				sprintf(cs,"%s : cannot output a certificate to LDAP (sn=%.7d)",
						al->name,ct->serialNumber);
				ENCERRLOG(info->ca.caname,cs);
			}
		}else
#endif
		if(info->authmode==AE_AUTH_LICE){
			/* remove license ID from en.license file */
			if(aienr_remove_licensefile(info,al->name,al)){ err="cannot remove license id"; goto done; }
		}

		/* send email to admin */
		if(*info->ml.smtphost && *info->ml.admemail){
			char *ml = "enroll_accept_csr2.txt";
			cgi_get_deflang(info->ml.lang);

			if(info->authmode==AE_AUTH_LCHA){
				for(i=0; i<MAXGROUP; i++){
				  if(!strcmp(group,info->grpname[i])){ 
				    strncpy(info->ml.admemail,info->grpemail[i],126); break;
				  }
				}
			}
			if(aienr_sendmail_sys2adm(&info->ml,al,info->ml.admemail,cn,ml)){
				sprintf(cs,"%s : cannot send a email to admin (sn=%.7d) << %s",
						al->name,ct->serialNumber,CA_get_errstr());
				ENCERRLOG(info->ca.caname,cs);
			}
		}

		/* set cookie */
		printf("Set-Cookie: ENSerialNum=%.7d\n", ct->serialNumber);
		printf("Set-Cookie: ENGroupName=%s\n", group);

		if(*passwd) OK_set_passwd(passwd);
		if(P12_add_cert(p12,ct,NULL,0xff)) goto done;
		accID=ct->serialNumber; ct=NULL;
		if(P12_add_key(p12,prv,NULL,0xff)) goto done;
		prv = NULL;

		if(P12_check_chain(p12,0)) goto done;
		/* get source file */
		cp = AI_ACCEPT_DL_FORM_HTM;
		if((buf=(char*)get_file2buf(get_html_name(info,cp),&sz))==NULL) goto done;

		/* set appropriate message */
		cgi_check_lang(info);
		printf("Content-type: text/html\n\n");

		cp = buf;
		ct = P12_get_usercert(p12);
		do{
			if((t=strstr(cp,"${"))==NULL){ printf("%s",cp); break; }
			*t=0; t++;

			printf("%s",cp);
			if(!memcmp(t,"{DOWNLOADCERT}",14)){
				/* for Safari mode */
#ifdef __WINDOWS__
				sprintf(cs,"sf%.7d.dl.p12",accID);
				if(P12_write_file(p12,cs)){ ct=NULL; goto done; }
#else
				sprintf(cs,"out/sf%.7d.dl.p12",accID);
				sprintf(path,"../out/sf%.7d.dl.p12",accID);
				if(P12_write_file(p12,path)){ ct=NULL; goto done; }
#endif
				printf("%s",cs); t+=14;
			}else if(!memcmp(t,"{CLEARCT}",9)){
				/* for Safari mode */
#ifdef __WINDOWS__
				sprintf(cs,"sf%.7d",accID);
#else
				sprintf(cs,"../out/sf%.7d",accID);
#endif
				printf("%s",cs); t+=9;
			}else{
				printf("$");
			}
			cp = t;
		}while(cp);

		ct = NULL;

		/* output log */
		sprintf(cs,"%s : success to issue a Cert (sn=%.7d)",
			((info->authmode)?(name):("anonymous")),accID);
		ENCACCLOG(info->ca.caname,cs);
		ENCISSLOG(info->ca.caname,cs);
	}

	ok = 0;
done:
	if(ok){
		printf("Content-type: text/html\n\n");
		aienr_print_err(info,"send csr : System error!!",err,lcmp);
	}
	if(lcmp){ LCMP_unbind_s(lcmp); LCMP_free(lcmp); }
	Cert_free(ct);
	CMP_certtmpl_free(ctt);
	if(p12) P12_free(p12);
	if(der) free(der);
	if(buf) free(buf);
	if(prv) Key_free(prv);
	if(pub) Key_free(pub);
#ifdef __WINDOWS__
	WSACleanup();
#endif
	return ok;
}

/*----------------------------------------------------------------------
	get cert done dl : confirm certificate installation
----------------------------------------------------------------------*/
int aienr_get_certdone_dl(AiEnrollInfo *info){
	AccList *al = NULL;
	char cs[256],*name,*sn;
	char *buf=NULL,*cer="",*err="",*gp,*fm,*gn;
	off_t sz;
	int accID,snum,ok=-1;

	fm = cgi_find_cookie(&info->qc,"ENFm");
	gn = cgi_find_cookie(&info->qc,"ENGn");

	/* set content */
	cgi_check_lang(info);
	printf("Content-type: text/html\n\n");

	/* delete temporary cert file (for Safari) */
	cer = cgi_find_query(&info->qc,"ClearCT");
	if(*cer){
		sprintf(cs,"%s.dl.p12",cer);
		unlink(cs);
	}

	/* check user name */
	if(cgi_load_session(info->sespath,&info->list)){
		err = "cannot load session file"; goto done;
	}
	if(info->postmode || info->offlineca){
		sn = cgi_find_cookie(&info->qc,"ENAcceptID");
		if(*sn == 0){ err="cannot get cookie (acceptID)"; goto done; }
		accID = atoi(sn);
	}else{
		name = cgi_find_cookie(&info->qc,"ENLoginName");
		if(*name == 0){ err="cannot get cookie (login)"; goto done; }
		gp = cgi_find_cookie(&info->qc,"ENGroupName");
		if(*gp == 0){ err="cannot get cookie (group)"; goto done; }
	}
	sn = cgi_find_cookie(&info->qc,"ENSerialNum");
	if(*sn == 0){ err="cannot get cookie (ENSerialNum)"; goto done; }
	snum = atoi(sn);

	/* set session mode */
	if(info->postmode || info->offlineca){
		al = cgi_findbyaccid(info->list,accID);
	}else{
		al = cgi_findbynamegrp(info->list,name,gp);
	}
	if(al){
		al->serialNum = snum;
		al->mode = AI_ENL_ISSDONE;
		memset(al->sessionID,0,32);
		if(cgi_update_session(info->sespath,al)){ err="cannot update session file"; goto done; }
	}

	/* get source file */
	if((buf=(char*)get_file2buf(get_html_name(info,AI_DONE_DL_FORM_HTM),&sz))==NULL) goto done;
	{
	  AiVList v[] = {
	    {"{FROM}",fm},
	    {"{Gp}",gn},
	    {NULL,NULL}
	  };
	  cgi_print_content(buf,v);
	}

	/* output log */
	sprintf(cs,"%s : success to download a user certificate (sn=%.7d,accID=%.7d)",
		al->name,al->serialNum,al->acceptID);
	ENCACCLOG(info->ca.caname,cs);

	ok = 0;
done:
	if(ok){
		aienr_print_err(info,"get cert done : System error!!",err,NULL);
	}
	if(buf) free(buf);
	return ok;
}

