/* aienr_log.c */
/*
 * Copyrightc 2004-2006 National Institute of Informatics in Japan, 
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must 
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifndef __WINDOWS__
# include <sys/time.h>
#endif

#include "aienr_cgi.h"

/* global values */
FILE *acc_fp=NULL;
FILE *iss_fp=NULL;
FILE *err_fp=NULL;


/*-------------------------------------------------
	logging functions
-------------------------------------------------*/
int aienr_log_open(AiEnrollInfo *info){
	int i=10;
	/* try 10 times to open */
	do{
		if(info->acclog[0]&&(acc_fp==NULL)){
			if((acc_fp=fopen(info->acclog,"a+"))==NULL){
				fprintf(stderr,"enroll : cannot open access log : %s\n",info->acclog);
			}
		}
		if(info->isslog[0]&&(iss_fp==NULL)){
			if((iss_fp=fopen(info->isslog,"a+"))==NULL){
				fprintf(stderr,"enroll : cannot open issue log : %s\n",info->isslog);
			}
		}
		if(info->errlog[0]&&(err_fp==NULL)){
			if((err_fp=fopen(info->errlog,"a+"))==NULL){
				fprintf(stderr,"enroll : cannot open error log : %s\n",info->errlog);
			}
		}	
		if(err_fp && iss_fp && acc_fp) break;
		i--;
#ifdef __WINDOWS__
		Sleep(100);
#else
		{
			struct timeval tv;
			tv.tv_sec  = 0;
			tv.tv_usec = 100*1000; /* ms */
			select(0,NULL,NULL,NULL,&tv); /* wait 100 ms */
		}
#endif
	}while(i>0);

	return (i==0)?(-1):(0);
}

void aienr_log_close(){
	if(acc_fp){ fclose(acc_fp); acc_fp=NULL; }
	if(err_fp){ fclose(err_fp); err_fp=NULL; }
	if(iss_fp){ fclose(iss_fp); iss_fp=NULL; }
}

/*-------------------------------------------------
	output log
-------------------------------------------------*/
int aienr_log_out(AiEnrollInfo *info,int type, char *caname, char *msg){
	char tm[64],buf[1024];
	time_t t;
	int i=0,j=0,ok=-1;

	if(aienr_log_open(info)) goto done;

	time(&t);
	strcpy(tm,"[");
	strcat(tm,ctime(&t)); tm[strlen(tm)-1] = 0; /* clear "\n" */

	/* get message */
	strncpy(buf,tm,64);
	strcat (buf,"] WEB Enroll: ");
	strncat(buf,caname,128);
	strcat (buf," => ");
	strncat(buf,msg,256);
	strcat (buf,"\n");

	/* write to log file */
	if((type & AICA_LOG_ACC)&&(acc_fp)){
		if(fputs(buf,acc_fp)<0) goto done;
		if(fflush(acc_fp)<0) goto done;

		if((i=aienr_log_checksize(acc_fp,info->acclog_size))<0) goto done;

		if((i>0) && aienr_log_rotate(info->acclog,&acc_fp)) goto done;
	}

	if((type & AICA_LOG_ERR)&&(err_fp)){
		if(fputs(buf,err_fp)<0) goto done;
		if(fflush(err_fp)<0) goto done;

		if((i=aienr_log_checksize(err_fp,info->errlog_size))<0) goto done;

		if((i>0) && aienr_log_rotate(info->errlog,&err_fp)) goto done;
	}
	
	if((type & AICA_LOG_ISS)&&(iss_fp)){
		if(fputs(buf,iss_fp)<0) goto done;
		if(fflush(iss_fp)<0) goto done;

		if((j=aienr_log_checksize(iss_fp,info->isslog_size))<0) goto done;

		if((j>0) && aienr_log_rotate(info->isslog,&iss_fp)) goto done;
	}
	
	ok = 0;
done:
	aienr_log_close();

	return ok;
}

/*-------------------------------------------------
	check log size
-------------------------------------------------*/
int aienr_log_checksize(FILE *fp, int size_kb){
	struct stat st;

	if(size_kb<=0) return 0;
	if(fstat(fileno(fp),&st)) goto error;

	return (st.st_size >= (size_kb<<10))?(1):(0);
error:
	return -1;
}

/*-------------------------------------------------
	rotate logs
-------------------------------------------------*/
int aienr_log_rotate(char *fname,FILE **fp){
	time_t t;
	struct tm *ltm;
	char buf[256];

	time(&t);
	ltm = localtime(&t);
	SNPRINTF(buf,254,"%s.%.2d%.2d%.2d%.2d%.2d%.2d",fname,
			ltm->tm_year+1900,ltm->tm_mon+1,ltm->tm_mday,
			ltm->tm_hour,ltm->tm_min,ltm->tm_sec);

	if(*fp){ fclose(*fp); *fp=NULL; }

	/* ignore error case, because other crlpublisher keeps
	 * file handler and rename cannot be accomplished.
	 */
	rename(fname,buf);

	if((*fp=fopen(fname,"a+"))==NULL){
		fprintf(stderr,"cannot open log file : %s\n",fname);
		goto error;
	}
	return 0;
error:
	return -1;
}
