/* aiconf.c */
/*
 * Copyright (c) 2004-2015 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <aicrypto/ok_err.h>
#include <aicrypto/ok_io.h>
#include <aicrypto/ok_rc2.h>
#include <aicrypto/ok_sha1.h>
#include <aicrypto/ok_rand.h>
#include <aicrypto/ok_base64.h>

#include "ok_caerr.h"
#include "ok_conf.h"
#include "afi_utils.h"

/* encryption salt */
#include "airand.h"

/* pathname of configuration file */
#ifndef AICONFIG
#define AICONFIG	"aica.cnf"
#endif

/**
 * Get a section of configuration file to a buffer.
 */
char *conf_get_section(const char *fname, const char *begin, char *end)
{
	char *buf, *sp, *ep, *ret = NULL, *tmp = NULL;
	off_t sz;

	size_t blen = strlen(begin);
	if (blen >= CONF_SECT_NAME_MAX - 4) {	/* "end]" */
		fprintf(stderr, "section name too long");
		return NULL;
	}

	if (end == NULL) {
		tmp = (char *)malloc(sizeof(char) * blen + 4 + 1);
		memset(tmp, 0, blen + 4 + 1);
		strcpy(tmp, begin);
		tmp[blen - 1] = ' ';	/* ']' -> SPC */
		strcat(tmp, "end]");
		end = tmp;
	}

	if ((buf = (char *)get_file2buf(fname, &sz)) == NULL)
		return NULL;

	if ((sp = strstr(buf, begin)) == NULL) {
		OK_set_error(ERR_ST_CA_BADCONF, ERR_LC_CONF, ERR_PT_CONF, NULL);
		goto done;
	}
	if ((ep = strstr(sp, end)) == NULL) {
		OK_set_error(ERR_ST_BADFORMAT, ERR_LC_CONF, ERR_PT_CONF, NULL);
		goto done;
	}
	*ep = '\0';

	if ((ret = strdup(sp)) == NULL) {
		OK_set_error(ERR_ST_STRDUP, ERR_LC_CONF, ERR_PT_CONF, NULL);
		goto done;
	}

done:
	free(buf);
	if (tmp != NULL) free(tmp);
	return ret;
}

/**
 * Clear comment lines in a buffer.
 */
int conf_clear_comment(char *buf)
{
	char *line, *p;
	size_t i, j, k;

	i = strlen(buf);
	line = buf;
	j = 0;			/* subscript of buf[] */

	while (j < i) {
		if ((p = strchr(line, '\n')) != NULL)
			*p = '\0';
		k = strlen(line);

		if (line[0] == '#')
			memset(line, ' ', k);
		if (p) {
			*p = '\n';
			k++;
		}

		j += k;
		line += k;
	}
	return 0;
}

/**
 * Locate the first occurrence of the newline from a buffer.
 */
char *conf_get_line(const char *bp, char **np, char *tmp)
{
	char *p;		/* point of LF or CR(+LF) */

	if ((p = strchr(bp, '\n')) != NULL) {
		*np = p + 1;
		if (*(p - 1) == '\r')
			p--;
		*tmp = *p;
		*p = '\0';
	} else {
		*tmp = '\0';
		*np = NULL;
	}
	return p;
}

/**
 * Locate the occurrence of the string "item" in a buffer.
 */
char *conf_comp_item(char *bp, const char *item)
{
	char *p = bp;
	char i;
	size_t len = strlen(item);

	while ((*p == ' ') || (*p == '\t')) {
		p++;
	}

	i = p[len];
	if ((i == ' ') || (i == '\t') || (i == '='))
		if (memcmp(p, item, len) == 0)
			return p;

	return NULL;
}

/**
 * Get a character string that is the value of an item.
 */
int conf_getstr(char *buf, const char *name, char *ret, const int max)
{
	char *cp, *ve, *np, tmp;

	/*
	 * special handler for the item "smtp_from".
	 * however, this code is quite tricky...
	 */
	if (strcmp(name, "smtp_from") == 0)
		ret[0] = '\0';

	while (buf != NULL && *buf != '\0') {
		ve = conf_get_line(buf, &np, &tmp);
		if (*buf != '\0' && *buf != '#') {	/* maybe verbose... */
			if ((cp = conf_comp_item(buf, name)) != NULL) {
				if ((cp = strchr(cp, '=')) != NULL) {
					strncpy(ret, cp + 1, max);
				}
			}
		}
		if (ve)
			*ve = tmp;
		buf = np;
	}
	return 0;		/* XXX: always 0... */
}

/**
 * Get an integer that is the value of an item.
 */
int conf_getint(char *buf, const char *name, int *ret)
{
	char *cp, *ve, *np, tmp;

	while (buf && *buf) {
		ve = conf_get_line(buf, &np, &tmp);
		if (*buf && *buf != '#') {
			if ((cp = conf_comp_item(buf, name)) != NULL) {
				if ((cp = strchr(cp, '=')) != NULL) {
					*ret = atoi(cp + 1);
				}
			}
		}
		if (ve)
			*ve = tmp;
		buf = np;
	}
	return 0;
}

/**
 * Get true or false that is the value of an item.
 */
int conf_getbool(char *buf, const char *name, int *ret)
{
	char *cp, *ve, *np, tmp;

	while (buf && *buf) {
		ve = conf_get_line(buf, &np, &tmp);
		/* there may be "0xa" line */
		if (*buf && *buf != '#') {
			if ((cp = conf_comp_item(buf, name)) != NULL) {
				if ((cp = strchr(cp, '=')) != NULL) {
					cp++;

					*ret = 0;
					if (strstr(cp, "TRUE") != NULL)
						*ret = 1;
					else if (strstr(cp, "true") != NULL)
						*ret = 1;
					else if (strstr(cp, "yes") != NULL)
						*ret = 1;
				}
			}
		}
		if (ve)
			*ve = tmp;
		buf = np;
	}
	return 0;
}

/* svpath parser */
int conf_parse_svpath(const char *svpath,
		      char *svname, size_t svname_len, char *caname,
		      size_t caname_len)
{
	char *svpath_ = strdup(svpath);
	char *pos_svname = NULL;
	char *pos_caname = NULL;
	char *pos_first_lbracket = NULL;
	char *pos_last_rbracket = NULL;
	char *pos_sep = NULL;	/* separator is `:' */
	int ret = CONF_PARSE_SVPATH_PARSE_SUCCESS;

	if (!svpath_) {
		ret = CONF_PARSE_SVPATH_ENOMEM;
		goto done;
	}
	pos_first_lbracket = strchr(svpath_, '[');
	pos_last_rbracket = strrchr(svpath_, ']');

	/* get svname */
	if (pos_first_lbracket && pos_last_rbracket) {
		/* valid IPv6 address? */
		pos_sep = pos_last_rbracket + 1;
		if ((pos_first_lbracket != svpath_)
		    || (*pos_sep != ':')) {
			ret = CONF_PARSE_SVPATH_PARSE_FAILURE;
			goto done;
		}
		*pos_last_rbracket = '\0';
		pos_svname = pos_first_lbracket + 1;
		if (strlen(pos_svname) >= svname_len) {
			ret = CONF_PARSE_SVPATH_ERANGE_SVNAME;
			goto done;
		}
		if (isipv6addr(pos_svname) != ISIPV6ADDR_TRUE) {
			ret = CONF_PARSE_SVPATH_PARSE_FAILURE;
			goto done;
		}
	} else if (!pos_first_lbracket && !pos_last_rbracket) {
		/* IPv4 address, host name or FQDN */
		if ((pos_sep = strchr(svpath_, ':')) == NULL) {
			/* no colon */
			ret = CONF_PARSE_SVPATH_PARSE_FAILURE;
			goto done;
		}
		if (svpath_ == pos_sep) {
			/* no svname */
			ret = CONF_PARSE_SVPATH_PARSE_FAILURE;
			goto done;
		}
		*pos_sep = '\0';
		if (strlen(svpath_) >= svname_len) {
			ret = CONF_PARSE_SVPATH_ERANGE_SVNAME;
			goto done;
		}
		pos_svname = svpath_;
	} else {
		ret = CONF_PARSE_SVPATH_PARSE_FAILURE;
		goto done;
	}

	/* get caname */
	pos_caname = pos_sep + 1;
	if (*pos_caname == '\0') {
		/* no caname */
		ret = CONF_PARSE_SVPATH_PARSE_FAILURE;
		goto done;
	}
	if (strlen(pos_caname) >= caname_len) {
		ret = CONF_PARSE_SVPATH_ERANGE_CANAME;
		goto done;
	}

	/* copy back to arguments */
	strcpy(caname, pos_caname);
	strcpy(svname, pos_svname);
done:
	if (svpath_) {
		free(svpath_);
	}
	return ret;
}

/**
 * Get the salt value in "general info" section in configuration file.
 */
static int conf_getsalt(char *ret)
{
	char *buf = conf_get_section(AICONFIG, "[general info]", NULL);
	if (buf == NULL)
		goto done;
	if (conf_getstr(buf, "salt_val", ret, 64)) {
		/* XXX: need some error message */
		goto done;
	}

done:
	if (buf != NULL)
		free(buf);
	return 0;	/* XXX: return the value according to the result. */
}

/**
 * Encrypt a password.
 */
char *aica_encry_passwd(char *passwd)
{
	unsigned char key[128], out[128];
	char conf_salt[64];
	char *ret = NULL;
	Key_RC2 *rc2 = NULL;
	SHA1_CTX ctx;
	int plen = strlen(passwd);

	memset(conf_salt, 0, 64);
	conf_getsalt(conf_salt);

	if (RAND_bytes(key, 8))
		goto done;
	memcpy(out, key, 8);

	SHA1init(&ctx);
	SHA1update(&ctx, ai_rand_salt, 128);
	SHA1update(&ctx, (unsigned char *)conf_salt, 64);
	SHA1final(&key[8], &ctx);

	if ((rc2 = RC2key_new(28, key)) == NULL)
		goto done;
	RC2_ecb_encrypt(rc2, plen, (unsigned char *)passwd, &out[8]);

	if ((ret =
	     Base64_encode((((plen - 1) >> 3) + 2) << 3, out, 64)) == NULL)
		goto done;

done:
	if (rc2)
		RC2key_free(rc2);
	return ret;
}

/**
 * Decrypt a password.
 */
char *aica_decry_passwd(char *in)
{
	unsigned char key[128], *dec = NULL;
	char conf_salt[64];
	Key_RC2 *rc2 = NULL;
	SHA1_CTX ctx;
	int i;

	memset(conf_salt, 0, 64);
	conf_getsalt(conf_salt);

	if ((dec = Base64_decode(in, &i)) == NULL)
		goto done;
	memcpy(key, dec, 8);

	SHA1init(&ctx);
	SHA1update(&ctx, ai_rand_salt, 128);
	SHA1update(&ctx, (unsigned char *)conf_salt, 64);
	SHA1final(&key[8], &ctx);

	if (i > 8) {
		if ((rc2 = RC2key_new(28, key)) == NULL)
			goto done;
		RC2_ecb_decrypt(rc2, i - 8, &dec[8], dec);
		dec[i - 8] = 0;
	} else {
		memset(dec, 0, i);
	}
done:
	if (rc2)
		RC2key_free(rc2);
	return (char *)dec;
}
