/* afi_utils.c - utility for Address Family Indepedent */
/*
 * Copyright (c) 2012 National Institute of Informatics in Japan, 
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must 
 * include this notice in the file.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>

#include "afi_utils.h"

/* IPv6 address validator */
int isipv6addr(char *addr)
{
	struct addrinfo hints;
	struct addrinfo *ai = NULL;
	int ret = ISIPV6ADDR_FALSE;

	memset(&hints, 0, sizeof(hints));
	hints.ai_family = AF_INET6;
	hints.ai_socktype = SOCK_RAW;
	hints.ai_flags = AI_NUMERICHOST;
	if(getaddrinfo(addr, NULL, &hints, &ai) == 0) {
		ret = ISIPV6ADDR_TRUE;
		freeaddrinfo(ai);
	}
	return ret;
}

/* *_host normalizer */
int conf_normalize_host(char *host)
{
	int ret = CONF_NORMALIZE_HOST_PARSE_FAILURE;
	char *host_tmp;
	char *v6head;
	char *lbracket = strchr(host, '[');
	char *rbracket = strrchr(host, ']');

	if(lbracket && rbracket) {
		if(lbracket != host || *(rbracket+1) != '\0') {
			goto done;
		}
		/* remove brackets */
		v6head = lbracket+1;
		host_tmp = strdup(v6head);
		if(!host_tmp) {
			ret = CONF_NORMALIZE_HOST_ENOMEM;
			goto done;
		}
		host_tmp[rbracket-v6head] = '\0';
		/* valid IPv6 address? */
		if(isipv6addr(host_tmp) == ISIPV6ADDR_TRUE) {
			strcpy(host, host_tmp);
			ret = CONF_NORMALIZE_HOST_PARSE_SUCCESS;
		}
		free(host_tmp);
		goto done;
	} else if(!lbracket && !rbracket) {
		ret = CONF_NORMALIZE_HOST_PARSE_SUCCESS;
		goto done;
	} else {
		goto done;
	}

done:
	return ret;
}

/* URL parser (based on CApub_connect_http() in v2.3.4)*/
int parse_url(const char *url, char *host, size_t host_len, int *port,
              char *path, size_t path_len, int *ssl)
{
	int ret = PARSE_URL_PARSE_FAILURE;
	size_t url_len = strlen(url);
	char *url_tmp = (char*)malloc(url_len+strlen("/")+1);
	char *host_tmp = (char*)malloc(url_len+1);
	short int port_tmp = 80;  /* default is HTTP */
	char *path_tmp = (char*)malloc(url_len+1);
	int ssl_tmp = PARSE_URL_SSL_DISABLE;
	char *iter;

	if(!url_tmp || !host_tmp || !path_tmp) {
		ret = PARSE_URL_ENOMEM;
		goto done;
	}

	/* append '/' if no path (e.g. http://127.0.0.1) */
	strcpy(url_tmp, url);
	iter = strstr(url_tmp, "://");
	if(!iter) {
		goto done;
	}
	if(!strchr(iter+strlen("://"), '/')) {
		url_tmp[url_len] = '/';
		url_tmp[url_len+1] = '\0';
	}

	if(sscanf(url_tmp, "http://[%[^]]]:%hu%s", host_tmp, &port_tmp, path_tmp) == 3
	   && isipv6addr(host_tmp) == ISIPV6ADDR_TRUE) {
		/* matched http://[fe80::1]:80/index.html */
	}else if(sscanf(url_tmp, "http://[%[^]]]%s", host_tmp, path_tmp) == 2
		 && isipv6addr(host_tmp) == ISIPV6ADDR_TRUE) {
		/* matched http://[fe80::1]/index.html */
	}else if(sscanf(url_tmp, "https://[%[^]]]:%hu%s", host_tmp, &port_tmp, path_tmp) == 3
		 && isipv6addr(host_tmp) == ISIPV6ADDR_TRUE) {
		/* matched https://[fe80::1]:80/index.html */
		ssl_tmp = PARSE_URL_SSL_ENABLE;
	}else if(sscanf(url_tmp, "https://[%[^]]]%s", host_tmp, path_tmp) == 2
		 && isipv6addr(host_tmp) == ISIPV6ADDR_TRUE){
		/* matched https://[fe80::1]/index.html */
		port_tmp = 443; ssl_tmp = PARSE_URL_SSL_ENABLE;
	}else if(sscanf(url_tmp, "http://%[^][:]:%hu%s", host_tmp, &port_tmp, path_tmp) == 3) {
		/* matched http://127.0.0.1:80/index.html and http://www.example.com:80/index.html */
	}else if(sscanf(url_tmp, "http://%[^][/]%s", host_tmp, path_tmp) == 2) {
		/* matched http://127.0.0.1/index.html and http://www.example.com/index.html */
	}else if(sscanf(url_tmp, "https://%[^][:]:%hu%s", host_tmp, &port_tmp, path_tmp) == 3) {
		/* matched https://127.0.0.1:80/index.html and https://www.example.com:80/index.html */
		ssl_tmp = PARSE_URL_SSL_ENABLE;
	}else if(sscanf(url_tmp, "https://%[^][/]%s", host_tmp, path_tmp) == 2) {
		/* matched https://127.0.0.1/index.html and https://www.example.com/index.html */
		port_tmp = 443; ssl_tmp = PARSE_URL_SSL_ENABLE;
	}else{
		/* bad url format */
		goto done;
	}

	if(host_len < strlen(host_tmp)+1) {
		ret = PARSE_URL_ERANGE_HOST;
		goto done;
	} else if(path_len < strlen(path_tmp)+1) {
		ret = PARSE_URL_ERANGE_PATH;
		goto done;
	}

	/* copy back to arguments */
	strcpy(host, host_tmp);
	*port = port_tmp;
	strcpy(path, path_tmp);
	*ssl = ssl_tmp;
	ret = PARSE_URL_PARSE_SUCCESS;

done:
	if(url_tmp) {
		free(url_tmp);
	}
	if(host_tmp) {
		free(host_tmp);
	}
	if(path_tmp) {
		free(path_tmp);
	}
	return ret;
}
