/* prof_tool.c */
/*
 * Copyright (c) 2004-2015 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_rand.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_pem.h>
#include <aicrypto/ok_store.h>

#include "ok_caerr.h"
#include "ok_ca.h"

/*-----------------------------------------
  count issued certificates
-----------------------------------------*/
int Prof_now_certnum(CertProf *cpf)
{
	CertStat *st;
	int ret = 0;

	for (st = cpf->stat; st; st = st->next)
		ret++;
	return ret;
}

/*-----------------------------------------
  count extensions
-----------------------------------------*/
int Prof_now_extnum(CertProf *cpf)
{
	CertExt *et = cpf->ext;
	int i = 0;

	while (et) {
		i++;
		et = et->next;
	}
	return i;
}

/*-----------------------------------------
  check expiration of certificates
  output : 0 ... no expired cert
         : 1 ... some certs was expired
-----------------------------------------*/
int Prof_expire_check(CertProf *cpf)
{
	return prof_expire_check_(cpf->stat);
}

int prof_expire_check_(CertStat *stat)
{
	CertStat *st;
	time_t now;
	int ret = 0;

	time(&now);		/* get current time */

	for (st = stat; st; st = st->next) {
		ret |= prof_stat_expire(st, now);
	}
	return ret;
}

int prof_stat_expire(CertStat *st, time_t now)
{
	time_t t;
	int ret = 0;

	if (now == 0)
		time(&now);

	st->state &= ~STAT_EXPIRED;

	/* notAfter */
	t = timegm(&st->notAfter);
	if (now > t) {
		st->state |= STAT_EXPIRED;
		ret = 1;
	}

	/* notBefore */
	t = timegm(&st->notBefore);
	if (now < t) {
		st->state |= STAT_EXPIRED;
		ret = 1;
	}

	return ret;
}

/*-------------------------------------------
  CA utility : find CertStat by serial num
-------------------------------------------*/
CertStat *Prof_find_stat(CertStat *top, int serial)
{
	CertStat *st;
	for (st = top; st; st = st->next)
		if (st->serialNum == serial)
			return st;

	OK_set_error(ERR_ST_CA_NOCSTAT, ERR_LC_PROF, ERR_PT_PFTOOL + 2, NULL);
	return NULL;
}

CertStat *Prof_find_stataid(CertStat *top, int acceptID)
{
	CertStat *st;
	for (st = top; st; st = st->next)
		if (st->acceptID == acceptID)
			return st;

	OK_set_error(ERR_ST_CA_NOCSTAT, ERR_LC_PROF, ERR_PT_PFTOOL + 2, NULL);
	return NULL;
}

/*-----------------------------------------
  CA utility : find CertStat by subject
-----------------------------------------*/
CertStat *Prof_find_statstr(CertStat *top, char *subject)
{
	CertStat *st;
	for (st = top; st; st = st->next)
		if (!strcmp(st->subject, subject))
			return st;

	OK_set_error(ERR_ST_CA_NOCSTAT, ERR_LC_PROF, ERR_PT_PFTOOL + 3, NULL);
	return NULL;
}

/*-----------------------------------------
  CA utility : find CertStat by keyID
-----------------------------------------*/
CertStat *Prof_find_statkeyID(CertStat *top, unsigned char *keyID)
{
	CertStat *st;
	for (st = top; st; st = st->next)
		if (st->keyID && !memcmp(st->keyID, keyID, 20))
			return st;

	OK_set_error(ERR_ST_CA_NOCSTAT, ERR_LC_PROF, ERR_PT_PFTOOL + 3, NULL);
	return NULL;
}

/*-----------------------------------------------
  CA utility : find CertStat by part of subject
-----------------------------------------------*/
CertStat *Prof_find_sbjquery(CertStat *top, char *query)
{
	CertStat *ret = NULL, *dp, *st, *hd;
	for (st = top; st; st = st->next) {
		if (strstr(st->subject, query)) {
			if ((dp = CertStat_dup(st)) == NULL)
				goto done;
			dp->next = dp->prev = NULL;

			if (ret == NULL) {
				ret = hd = dp;
			} else {
				hd->next = dp;
				hd = dp;
			}
		}
	}
	if (ret == NULL)
		OK_set_error(ERR_ST_CA_NOCSTAT, ERR_LC_PROF, ERR_PT_PFTOOL + 3,
			     NULL);
done:
	return ret;
}

/*-------------------------------------------
  CA utility : find CertProf by profile name
-------------------------------------------*/
CertProf *Prof_find(CA *ca, char *name)
{
	CertProf *cpf;
	for (cpf = ca->profList; cpf; cpf = cpf->next)
		if (!strcmp(cpf->name, name))
			return cpf;

	OK_set_error(ERR_ST_CA_NOPROF, ERR_LC_PROF, ERR_PT_PFTOOL + 4, NULL);
	return NULL;
}

CRLProf *Prof_crl_find(CA *ca, char *name)
{
	CRLProf *cpf;
	for (cpf = ca->crlpList; cpf; cpf = cpf->next)
		if (!strcmp(cpf->name, name))
			return cpf;

	OK_set_error(ERR_ST_CA_NOPROF, ERR_LC_PROF, ERR_PT_PFTOOL + 4, NULL);
	return NULL;
}

/*---------------------------------------------------------------*/
/*-----------------------------------------
  CA add new certificate info
-----------------------------------------*/
int Prof_add_cert(CertProf *cpf, Cert *ct)
{
	CertStat *st, *tmp;
	int i;

	if ((st = CertStat_new()) == NULL)
		goto error;
	st->state = STAT_OK | ((Cert_is_CA(ct) > 0) ? (STAT_CA) : (0));
	st->serialNum = ct->serialNumber;

	if ((st->keyID = (unsigned char *)malloc(22)) == NULL) {
		OK_set_error(ERR_ST_MEMALLOC, ERR_LC_PROF, ERR_PT_PFTOOL + 5,
			     NULL);
		goto error;
	}
	if (cs_get_keyhash(ct->pubkey, st->keyID, &i))
		goto error;

	if ((st->subject = strdup(ct->subject)) == NULL) {
		OK_set_error(ERR_ST_MEMALLOC, ERR_LC_PROF, ERR_PT_PFTOOL + 5,
			     NULL);
		goto error;
	}
	memcpy(&st->notBefore, &ct->time.notBefore, sizeof(struct tm));
	memcpy(&st->notAfter, &ct->time.notAfter, sizeof(struct tm));
	memset(&st->revokedDate, 0, sizeof(struct tm));

	tmp = cpf->stat;
	st->next = tmp;
	st->prev = NULL;
	cpf->stat = st;
	if (tmp)
		tmp->prev = st;
	return 0;
error:
	CertStat_free(st);
	return -1;
}

/*-----------------------------------------
  CA delete certificate info
-----------------------------------------*/
void Prof_delete_stat(CertStat *st)
{
	/* we don't need "(cpf->stat=st->next);" */
	if (st == NULL)
		return;
	if (st->prev)
		st->prev->next = st->next;
	if (st->next)
		st->next->prev = st->prev;

	CertStat_free(st);
}

/*-----------------------------------------
  CA add new certificate file
-----------------------------------------*/
int Prof_add_certfile(CertProf *cpf, Cert *ct)
{
	CertStat *st;
	unsigned char buf[128];
	int i;

	if ((st = Prof_find_stat(cpf->stat, ct->serialNumber)) == NULL)
		return -1;

	ASN1_set_integer(ct->serialNumber, buf, &i);	/* set serialNum (for reqInfo) */
	if (fsetpos(cpf->fp, &(cpf->pos)))
		goto error;
	if (fwrite(buf, sizeof(char), i, cpf->fp) < (unsigned)i)
		goto error;

	st->fp = cpf->fp;
	if (fgetpos(cpf->fp, &(st->pos)))
		goto error;

	if (ASN1_skip_(ct->der, &i) == NULL)
		goto error;
	if (fwrite(ct->der, sizeof(char), i, cpf->fp) < (unsigned)i)
		goto error;
	if (fgetpos(cpf->fp, &(cpf->pos)))
		goto error;

	/* commit to disk */
	if (fflush(cpf->fp))
		goto error;

	return 0;
error:
	OK_set_error(ERR_ST_FILEWRITE, ERR_LC_PROF, ERR_PT_PFTOOL + 6, NULL);
	return -1;
}

/*-----------------------------------------
  CA add new key file
-----------------------------------------*/
int Prof_add_keyfile(CertProf *cpf, Key *key, int sn)
{
	unsigned char buf[128], *enc = NULL, *der = NULL, *kder = NULL;
	CertStat *st = NULL;
	int i, j, k, kt, ok = -1;

	if ((st = Prof_find_stat(cpf->stat, sn)) == NULL)
		return -1;

	kt = key->key_type;
	if ((kt != KEY_RSA_PRV) && (kt != KEY_DSA_PRV) && (kt != KEY_ECDSA_PRV))
		goto done;

	ASN1_set_integer(sn, buf, &i);	/* set serialNum (for certstat) */
	if (fseek(cpf->kfp, 0, SEEK_END))
		goto done;
	if (fwrite(buf, sizeof(char), i, cpf->kfp) < (unsigned)i)
		goto done;

	st->kfp = cpf->kfp;
	if (fgetpos(cpf->kfp, &(st->kpos)))
		goto done;

	if (RAND_bytes(buf, 8))
		goto done;

	switch (kt) {
	case KEY_RSA_PRV:
		if (ASN1_skip_(((Prvkey_RSA *) key)->der, &i) == NULL)
			goto done;
		kder = ((Prvkey_RSA *) key)->der;
		break;
	case KEY_DSA_PRV:
		if (ASN1_skip_(((Prvkey_DSA *) key)->der, &i) == NULL)
			goto done;
		kder = ((Prvkey_DSA *) key)->der;
		break;
	case KEY_ECDSA_PRV:
		if (ASN1_skip_(((Prvkey_ECDSA *) key)->der, &i) == NULL)
			goto done;
		kder = ((Prvkey_ECDSA *) key)->der;
		break;
	}

	if ((enc = PEM_msg_encrypt(kder, &i, buf, OBJ_CRYALGO_3DESCBC)) == NULL)
		goto done;
	if ((der = (unsigned char *)malloc(i + 64)) == NULL)
		goto done;
	ASN1_set_integer(kt, der, &k);
	ASN1_set_octetstring(8, buf, der + k, &j);
	ASN1_set_octetstring(i, enc, der + k + j, &i);
	i += k + j;

	ASN1_set_sequence(i, der, &i);

	if (fwrite(der, sizeof(char), i, st->kfp) < (unsigned)i)
		goto done;
	st->state |= STAT_HAVEKEY;

	/* commit to disk */
	if (fflush(st->kfp))
		goto done;

	ok = 0;
done:
	if (enc)
		free(enc);
	if (der)
		free(der);
	if (ok)
		OK_set_error(ERR_ST_FILEWRITE, ERR_LC_PROF, ERR_PT_PFTOOL + 7,
			     NULL);
	return ok;
}

/*-----------------------------------------
  CA delete key file
-----------------------------------------*/
int Prof_del_keyfile(CertStat *cs)
{
	unsigned char buf[128], *der = NULL;
	int i, j, ok = -1;
	FILE *kfp;

	if ((kfp = cs->kfp) == NULL)
		goto done;

	/* get length of encrypted private key */
	if (fsetpos(kfp, &(cs->kpos)))
		goto done;
	if (fread(buf, sizeof(char), 32, kfp) < 32)
		goto done;

	if ((ASN1_skip_(buf, &i)) == NULL)
		goto done;

	if ((der = (unsigned char *)malloc(i + 16)) == NULL)
		goto done;

	/* get key information & encrypted private key */
	if (fsetpos(kfp, &(cs->kpos)))
		goto done;
	if (fread(der, sizeof(char), i, kfp) < (unsigned)i)
		goto done;

	if ((j = ASN1_tlen(der)) < 0)
		goto done;

	der[0] = ASN1_OCTETSTRING;
	memset(&der[i - j], 0xcd, j);

	/* rewrite request data */
	if (fsetpos(kfp, &(cs->kpos)))
		goto done;
	if (fwrite(der, sizeof(char), i, kfp) < (unsigned)i)
		goto done;
	cs->state &= ~STAT_HAVEKEY;
	cs->kfp = NULL;
	memset(&cs->kpos, 0, sizeof(fpos_t));

	/* commit to disk */
	if (fflush(kfp))
		goto done;

	ok = 0;
done:
	if (der)
		free(der);
	if (ok)
		OK_set_error(ERR_ST_FILEWRITE, ERR_LC_PROF, ERR_PT_PFTOOL + 8,
			     NULL);
	return ok;
}

int Prof_add_xcertfile(CertProf *cpf, CertPair *cp)
{
	CertStat *st = NULL;
	unsigned char buf[128];
	char *sbj;
	int i;

	if (cp->issuedByThisCA) {
		sbj = cp->issuedByThisCA->subject;
		if ((st =
		     Prof_find_stat(cpf->stat,
				    cp->issuedByThisCA->serialNumber)) == NULL)
			goto error;
	} else if (cp->issuedToThisCA) {
		sbj = cp->issuedToThisCA->issuer;
		if ((st =
		     Prof_find_statstr(cpf->stat,
				       cp->issuedToThisCA->issuer)) == NULL)
			goto error;
	}

	ASN1_set_integer(st->serialNum, buf, &i);	/* set serialNum (for reqInfo) */
	if (fsetpos(cpf->fp, &(cpf->pos)))
		goto error;
	if (fwrite(buf, sizeof(char), i, cpf->fp) < (unsigned)i)
		goto error;

	st->fp = cpf->fp;
	if (fgetpos(cpf->fp, &(st->pos)))
		goto error;

	if (ASN1_skip_(cp->der, &i) == NULL)
		goto error;
	if (fwrite(cp->der, sizeof(char), i, cpf->fp) < (unsigned)i)
		goto error;
	if (fgetpos(cpf->fp, &(cpf->pos)))
		goto error;

	/* commit to disk */
	if (fflush(cpf->fp))
		goto error;

	return 0;
error:
	OK_set_error(ERR_ST_FILEWRITE, ERR_LC_PROF, ERR_PT_PFTOOL + 9, NULL);
	return -1;
}

/*-----------------------------------------
  CA add new certificate info
-----------------------------------------*/
int Prof_add_csr_(CSRProf *rpf, char *subject, int sn)
{
	CertStat *st, *tmp;
	time_t t;
	struct tm *stm;

	if ((st = CertStat_new()) == NULL)
		goto error;
	st->state = CSR_STAT_WAIT;
	st->acceptID = ++rpf->csrNum;
	st->serialNum = sn;

	if ((st->subject = strdup(subject)) == NULL) {
		OK_set_error(ERR_ST_MEMALLOC, ERR_LC_PROF, ERR_PT_PFTOOL + 10,
			     NULL);
		goto error;
	}

	time(&t);
	stm = (struct tm *)gmtime(&t);
	memcpy(&st->notBefore, stm, sizeof(struct tm));
	memset(&st->notAfter, 0, sizeof(struct tm));
	memset(&st->revokedDate, 0, sizeof(struct tm));

	tmp = rpf->stat;
	st->next = tmp;
	st->prev = NULL;
	rpf->stat = st;
	if (tmp)
		tmp->prev = st;
	return 0;
error:
	CertStat_free(st);
	return -1;
}

int Prof_add_csr(CSRProf *rpf, Req *req, int sn)
{
	return Prof_add_csr_(rpf, req->subject, sn);
}

int Prof_add_csrtmpl(CSRProf *rpf, CertTemplate *tmpl)
{
	char *sbj = NULL;
	int ret = -1;

	if ((sbj = Cert_subject_str(&(tmpl->subject))) == NULL)
		goto done;
	ret = Prof_add_csr_(rpf, sbj, tmpl->serialNumber);
done:
	if (sbj)
		free(sbj);
	return ret;
}

void Prof_del_csr(CSRProf *rpf, CertStat *cs)
{
	CertStat *next = cs->next;
	Prof_delete_stat(cs);
	if (rpf->stat == cs)
		rpf->stat = next;
}

int Prof_add_csrfile_(CSRProf *rpf, unsigned char *der, int accID)
{
	char buf[256], tmp[32];

	sprintf(tmp, "%.8d.req", accID);
	if (set_path(buf, 256,
		     rpf->pCA->ca_path, "/req/", tmp, NULL) == -1) {
		return -1;
	}

	if (ASN1_write_der(der, buf))
		return -1;

	return 0;
}

int Prof_add_csrfile(CSRProf *rpf, Req *req, int accID)
{
	return Prof_add_csrfile_(rpf, req->der, accID);
}

int Prof_del_csrfile(CSRProf *rpf, CertStat *cs)
{
	char buf[256], tmp[32];

	sprintf(tmp, "%.8d.req", cs->acceptID);
	if (set_path(buf, 256,
		     rpf->pCA->ca_path, "/req/", tmp, NULL) == -1) {
		return -1;
	}

	unlink(buf);

	return 0;
}

/*---------------------------------------------------------------*/
/*-----------------------------------------
  add CertExt to the Prof structure
-----------------------------------------*/
int Prof_add_ext(CertProf *cpf, CertExt *et)
{
	CertExt *hd = NULL;

	if (et == NULL) {
		OK_set_error(ERR_ST_NULLPOINTER, ERR_LC_PROF,
			     ERR_PT_PFTOOL + 11, NULL);
		return -1;
	}

	for (hd = cpf->ext; hd; hd = hd->next)
		if (hd->next == NULL)
			break;

	if (hd)
		hd->next = et;
	else
		cpf->ext = et;

	return 0;
}

int Prof_add_crlext(CRLProf *lpf, CertExt *et)
{
	CertExt *hd = NULL;

	if (et == NULL) {
		OK_set_error(ERR_ST_NULLPOINTER, ERR_LC_PROF,
			     ERR_PT_PFTOOL + 11, NULL);
		return -1;
	}

	for (hd = lpf->ext; hd; hd = hd->next)
		if (hd->next == NULL)
			break;

	if (hd)
		hd->next = et;
	else
		lpf->ext = et;

	return 0;
}

/*-----------------------------------------
  delete any CertExt in Prof structure
-----------------------------------------*/
int Prof_delete_ext(CertProf *cpf, int id)
{
	CertExt *et, *pv;

	for (et = cpf->ext; et; et = et->next) {
		if (et->extnID == id)
			break;
		pv = et;
	}

	if (et == NULL) {
		OK_set_error(ERR_ST_NULLPOINTER, ERR_LC_PROF,
			     ERR_PT_PFTOOL + 12, NULL);
		return -1;
	}

	if (et == cpf->ext) {
		cpf->ext = et->next;
	} else {
		pv->next = et->next;
	}
	CertExt_free(et);
	return 0;
}

int Prof_delete_crlext(CRLProf *lpf, int id)
{
	CertExt *et, *pv;

	for (et = lpf->ext; et; et = et->next) {
		if (et->extnID == id)
			break;
		pv = et;
	}

	if (et == NULL) {
		OK_set_error(ERR_ST_NULLPOINTER, ERR_LC_PROF,
			     ERR_PT_PFTOOL + 12, NULL);
		return -1;
	}

	if (et == lpf->ext) {
		lpf->ext = et->next;
	} else {
		pv->next = et->next;
	}
	CertExt_free(et);
	return 0;
}
