/* prof_file.c */
/*
 * Copyright (c) 2004-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_pem.h>

#include "ok_caerr.h"
#include "ok_ca.h"

/**
 * read certificate profile in DER ASN.1 encoding format.
 */
int Prof_open_ctinfo(CA *ca)
{
	CertProf *cpf = ca->cprof;
	AILock lock = NULL;
	char buf[272] = "";

	if (set_path(buf, 272,
		     ca->ca_path, "/cert/", cpf->name, ".cpi", NULL) == -1) {
		return -1;
	}

	if (cpf->der)
		free(cpf->der);	/* free */

	lock = cpf->lock;
	if (CA_lock(lock, 10000)) {
		lock = NULL;
		goto error;
	}			/* read lock */
	if ((cpf->der = ASN1_read_der(buf)) == NULL)
		goto error;
	if (CA_unlock(&lock))
		goto error;

	if (ASN1_ctprof_info(ca, cpf->der))
		goto error;

	if (prof_get_curstat(buf, &cpf->mtm, &cpf->msz))
		goto error;

	return 0;
error:
	if (lock)
		CA_unlock(&lock);
	if (cpf->der) {
		free(cpf->der);
		cpf->der = NULL;
	}
	return -1;
}

int prof_get_curstat(char *path, time_t *t, int *sz)
{
	struct stat sbuf;
	FILE *fp = NULL;
	int ok = -1;

	if ((fp = fopen(path, "rb")) == NULL) {
		OK_set_error(ERR_ST_FILEOPEN, ERR_LC_PROF, ERR_PT_PFFILE + 1,
			     NULL);
		goto done;
	}
	if (fstat(fileno(fp), &sbuf)) {
		OK_set_error(ERR_ST_FILEREAD, ERR_LC_PROF, ERR_PT_PFFILE + 1,
			     NULL);
		goto done;
	}
	*t = sbuf.st_mtime;
	*sz = sbuf.st_size;
	ok = 0;
done:
	if (fp)
		fclose(fp);
	return ok;
}

/*-----------------------------------------
  asn1 reload cert profile info
  XXX: should be simplified
-----------------------------------------*/
int Prof_reload_ctinfo(CA *ca)
{
	char path[272] = "";

	if (set_path(path, 272,
		     ca->ca_path, "/cert", NULL) == -1) {
		return -1;
	}
	return Prof_reload_ctinfo_(ca, path);
}

int Prof_reload_ctinfo_(CA *ca, char *path)
{
	CertProf *cpf = ca->cprof;
	char buf[272] = "";
	/* time_t cur; *//* XXX:currently unused. see below. */
	/* int sz; *//* ditto */

	if (set_path(buf, 272,
		     path, "/", cpf->name, ".cpi", NULL) == -1) {
		return -1;
	}

	/* problem that an updated "cpi" file is not reloaded.
	 * when reloaded, modified so that XXX always loads the "cpi" file.
	 */
//      if(prof_get_curstat(buf,&cur,&sz)) goto error;

//      if((cur != cpf->mtm)||(sz != cpf->msz)){
	if (cpf->der) {
		free(cpf->der);
		cpf->der = NULL;
	}
	cert_dn_free(&cpf->sbj_tmpl);
	cpf->sbj_tmpl.num = 0;
	CertExt_free_all(cpf->ext);
	cpf->ext = NULL;
	CertStat_free_all(cpf->stat);
	cpf->stat = NULL;
	if (Prof_open_ctinfo(ca))
		goto error;
//      }

	return 0;
error:
	if (cpf->der) {
		free(cpf->der);
		cpf->der = NULL;
	}
	return -1;
}

/*-----------------------------------------
  asn1 save cert profile info
  XXX: should be simplified
-----------------------------------------*/
int Prof_save_ctinfo(CA *ca)
{
	char path[272] = "";

	if (set_path(path, 272,
		     ca->ca_path, "/cert", NULL) == -1) {
		return -1;
	}
	return Prof_save_ctinfo_(ca, path);
}

int Prof_save_ctinfo_(CA *ca, char *path)
{
	CertProf *cpf = ca->cprof;
	char buf[272] = "", bak[272];
	int i;

	if (set_path(buf, 272,
		     path, "/", cpf->name, ".cpi", NULL) == -1) {
		return -1;
	}

	/* save backup */
	if (cpf->der) {
		struct stat st;
		strcpy(bak, buf);
		strcat(bak, ".bak");
		if (stat(buf, &st) == 0) {
			rename(buf, bak);	/* need an error handler? */
		}

		free(cpf->der);
		cpf->der = NULL;
	}

	/* save new one */
	if ((cpf->der = Prof_cert_toDER(ca, NULL, &i)) == NULL)
		goto error;

	if (ASN1_write_der(cpf->der, buf))
		goto error;

	if (prof_get_curstat(buf, &cpf->mtm, &cpf->msz))
		goto error;

	return 0;
error:
	if (cpf->der) {
		free(cpf->der);
		cpf->der = NULL;
	}
	return -1;
}

/*-----------------------------------------
  asn1 open certs file
  XXX: should be simplified
-----------------------------------------*/
/* this function must be executed after Prof_open_ctinfo */
int Prof_open_ctfile(CA *ca)
{
	char path[272] = "";

	if (set_path(path, 272,
		     ca->ca_path, "/cert", NULL) == -1) {
		return -1;
	}
	return Prof_open_ctfile_(ca, path, 0, 0x3);
}

/* mode flag
 *  ... 0x01 FOPEN
 *  ... 0x02 set all cert position
 *  ... 0x10 set first cert position with SN if found.
 *  ... 0x20 set last cert position with SN if found.
 */
int Prof_open_ctfile_(CA *ca, char *path, int sn, int mode)
{
	CertProf *cpf = ca->cprof;
	char buf[272] = "";

	if (set_path(buf, 272,
		     path, "/", cpf->name, ".cts", NULL) == -1) {
		return -1;
	}

	if ((mode & 0x01) && (cpf->fp == NULL)) {
		if ((cpf->fp = fopen(buf, "ab+")) == NULL) {
			OK_set_error(ERR_ST_FILEOPEN, ERR_LC_PROF,
				     ERR_PT_PFFILE + 3, NULL);
			goto error;
		}
		if (fgetpos(cpf->fp, &cpf->pos)) {
			OK_set_error(ERR_ST_FILEREAD, ERR_LC_PROF,
				     ERR_PT_PFFILE + 3, NULL);
			goto error;
		}
	}
	if (mode & 0x32)
		if (ASN1_ctprof_ctfile(ca, sn, mode))
			goto error;

	return 0;
error:
	if (cpf->fp) {
		fclose(cpf->fp);
		cpf->fp = NULL;
	}
	return -1;
}

/**
 * read CRL profile in DER ASN.1 encoding format.
 */
int Prof_open_clinfo(CA *ca)
{
	CRLProf *cpf = ca->lprof;
	AILock lock = NULL;
	char buf[272] = "";

	if (set_path(buf, 272,
		     ca->ca_path, "/cert/", cpf->name, ".lpi", NULL) == -1) {
		return -1;
	}

	if (cpf->der)
		free(cpf->der);

	lock = cpf->lock;
	if (CA_lock(lock, 10000)) {
		lock = NULL;
		goto error;
	}			/* read lock */
	if ((cpf->der = ASN1_read_der(buf)) == NULL)
		goto error;
	if (CA_unlock(&lock))
		goto error;

	if (ASN1_clprof_info(ca, cpf->der))
		goto error;

	if (prof_get_curstat(buf, &cpf->mtm, &cpf->msz))
		goto error;

	return 0;
error:
	if (lock)
		CA_unlock(&lock);
	if (cpf->der) {
		free(cpf->der);
		cpf->der = NULL;
	}
	return -1;
}

/*-----------------------------------------
  asn1 reload CRL profile info
  XXX: should be simplified
-----------------------------------------*/
int Prof_reload_clinfo(CA *ca)
{
	char path[272] = "";

	if (set_path(path, 272,
		     ca->ca_path, "/cert", NULL) == -1) {
		return -1;
	}
	return Prof_reload_clinfo_(ca, path);
}

int Prof_reload_clinfo_(CA *ca, char *path)
{
	CRLProf *lpf = ca->lprof;
	time_t cur;
	int sz;
	char buf[272] = "";

	if (set_path(buf, 272,
		     path, "/", lpf->name, ".lpi", NULL) == -1) {
		return -1;
	}

	if (prof_get_curstat(buf, &cur, &sz))
		goto error;

	if ((cur != lpf->mtm) || (sz != lpf->msz)) {
		if (lpf->der) {
			free(lpf->der);
			lpf->der = NULL;
		}
		CertExt_free_all(lpf->ext);
		lpf->ext = NULL;
		if (Prof_open_clinfo(ca))
			goto error;
	}
	return 0;
error:
	if (lpf->der) {
		free(lpf->der);
		lpf->der = NULL;
	}
	return -1;
}

/*-----------------------------------------
  asn1 save CRL profile info
  XXX: should be simplified
-----------------------------------------*/
int Prof_save_clinfo(CA *ca)
{
	char path[272] = "";

	if (set_path(path, 272,
		     ca->ca_path, "/cert", NULL) == -1) {
		return -1;
	}
	return Prof_save_clinfo_(ca, path);
}

int Prof_save_clinfo_(CA *ca, char *path)
{
	CRLProf *lpf = ca->lprof;
	char buf[272] = "", bak[272];
	int i;

	if (set_path(buf, 272,
		     path, "/", lpf->name, ".lpi", NULL) == -1) {
		return -1;
	}

	/* save backup */
	if (lpf->der) {
		struct stat st;
		strcpy(bak, buf);
		strcat(bak, ".bak");
		if (stat(buf, &st) == 0) {
			rename(buf, bak);	/* need an error handler? */
		}

		free(lpf->der);
		lpf->der = NULL;
	}

	/* save new data */
	if ((lpf->der = Prof_crl_toDER(ca, NULL, &i)) == NULL)
		goto error;

	if (ASN1_write_der(lpf->der, buf))
		goto error;

	if (prof_get_curstat(buf, &lpf->mtm, &lpf->msz))
		goto error;

	return 0;
error:
	if (lpf->der) {
		free(lpf->der);
		lpf->der = NULL;
	}
	return -1;
}

/*-----------------------------------------
  asn1 read key profile file
  XXX: should be simplified
-----------------------------------------*/
int Prof_open_keyfile(CA *ca)
{
	char path[272] = "";

	if (set_path(path, 272,
		     ca->ca_path, "/cert", NULL) == -1) {
		return -1;
	}
	return Prof_open_keyfile_(ca, path, 0, 0x3);
}

/* mode flag
 *  ... 0x01 FOPEN
 *  ... 0x02 set all key position
 *  ... 0x10 set key position with SN if found.
 */
int Prof_open_keyfile_(CA *ca, char *path, int sn, int mode)
{
	CertProf *cpf = ca->cprof;
	char buf[272] = "";

	if (set_path(buf, 272,
		     path, "/", cpf->name, ".kys", NULL) == -1) {
		return -1;
	}

	if ((mode & 0x01) && (cpf->kfp == NULL)) {
		if ((cpf->kfp = fopen(buf, "rb+")) == NULL)
			if ((cpf->kfp = fopen(buf, "wbc+")) == NULL) {
				OK_set_error(ERR_ST_FILEOPEN, ERR_LC_PROF,
					     ERR_PT_PFFILE + 6, NULL);
				goto error;
			}
		if (fgetpos(cpf->kfp, &cpf->kpos)) {
			OK_set_error(ERR_ST_FILEREAD, ERR_LC_PROF,
				     ERR_PT_PFFILE + 6, NULL);
			goto error;
		}
	}
	if (mode & 0x12)
		if (ASN1_ctprof_keyfile(ca, sn, mode))
			goto error;

	return 0;
error:
	if (cpf->kfp) {
		fclose(cpf->kfp);
		cpf->kfp = NULL;
	}
	return -1;
}

/**
 * read CSR profile in DER ASN.1 encoding format.
 */
int Prof_open_csrinfo(CA *ca)
{
	CSRProf *rpf = ca->csrProf;
	AILock lock = NULL;
	char buf[272] = "";

	if (set_path(buf, 272,
		     ca->ca_path, "/req/csr.rpi", NULL) == -1) {
		return -1;
	}

	if (rpf->der)
		free(rpf->der);

	lock = rpf->lock;
	if (CA_lock(lock, 10000)) {
		lock = NULL;
		goto error;
	}			/* read lock */
	if ((rpf->der = ASN1_read_der(buf)) == NULL)
		goto error;
	if (CA_unlock(&lock))
		goto error;

	if (ASN1_csrprof_info(ca, rpf->der))
		goto error;

	if (prof_get_curstat(buf, &rpf->mtm, &rpf->msz))
		goto error;

	return 0;
error:
	if (lock)
		CA_unlock(&lock);
	if (rpf->der) {
		free(rpf->der);
		rpf->der = NULL;
	}
	return -1;
}

/*-----------------------------------------
  asn1 reload CSR profile info
  XXX: should be simplified
-----------------------------------------*/
int Prof_reload_csrinfo(CA *ca)
{
	char path[272] = "";

	if (set_path(path, 272,
		     ca->ca_path, "/req", NULL) == -1) {
		return -1;
	}
	return Prof_reload_csrinfo_(ca, path);
}

int Prof_reload_csrinfo_(CA *ca, char *path)
{
	CSRProf *rpf = ca->csrProf;
	time_t cur;
	int sz;
	char buf[272] = "";

	if (set_path(buf, 272,
		     path, "/csr.rpi", NULL) == -1) {
		return -1;
	}

	if (prof_get_curstat(buf, &cur, &sz))
		goto error;

	if ((cur != rpf->mtm) || (sz != rpf->msz)) {
		if (rpf->der) {
			free(rpf->der);
			rpf->der = NULL;
		}
		if (Prof_open_csrinfo(ca))
			goto error;
	}
	return 0;
error:
	if (rpf->der) {
		free(rpf->der);
		rpf->der = NULL;
	}
	return -1;
}

/*-----------------------------------------
  asn1 save CSR profile info
  XXX: should be simplified
-----------------------------------------*/
int Prof_save_csrinfo(CA *ca)
{
	char path[272] = "";

	if (set_path(path, 272,
		     ca->ca_path, "/req", NULL) == -1) {
		return -1;
	}
	return Prof_save_csrinfo_(ca, path);
}

int Prof_save_csrinfo_(CA *ca, char *path)
{
	CSRProf *rpf = ca->csrProf;
	char buf[272] = "", bak[272];
	int i;

	if (set_path(buf, 272,
		     path, "/csr.rpi", NULL) == -1) {
		return -1;
	}

	/* save backup */
	if (rpf->der) {
		struct stat st;
		strcpy(bak, buf);
		strcat(bak, ".bak");
		if (stat(buf, &st) == 0) {
			rename(buf, bak);	/* need an error handler? */
		}

		free(rpf->der);
		rpf->der = NULL;
	}

	/* save new data */
	if ((rpf->der = Prof_csr_toDER(ca, NULL, &i)) == NULL)
		goto error;

	if (ASN1_write_der(rpf->der, buf))
		goto error;

	if (prof_get_curstat(buf, &rpf->mtm, &rpf->msz))
		goto error;

	return 0;
error:
	if (rpf->der) {
		free(rpf->der);
		rpf->der = NULL;
	}
	return -1;
}

/*-----------------------------------------
  get a certificate from ctfile
-----------------------------------------*/
Cert *Prof_get_cert(CertStat *rs)
{
	unsigned char buf[32], *der = NULL;
	Cert *ret = NULL;
	int i;

	if (rs->fp == NULL) {
		OK_set_error(ERR_ST_NULLPOINTER, ERR_LC_PROF, ERR_PT_PFFILE + 9,
			     NULL);
		return NULL;
	}
	if (fsetpos(rs->fp, &(rs->pos)))
		goto error;
	if (fread(buf, sizeof(char), 32, rs->fp) < 32)
		goto error;

	if ((ASN1_skip_(buf, &i)) == NULL) {
		OK_set_error(ERR_ST_NULLPOINTER, ERR_LC_PROF, ERR_PT_PFFILE + 9,
			     NULL);
		return NULL;
	}
	if ((der = (unsigned char *)malloc(i + 16)) == NULL) {
		OK_set_error(ERR_ST_MEMALLOC, ERR_LC_PROF, ERR_PT_PFFILE + 9,
			     NULL);
		return NULL;
	}

	if (fsetpos(rs->fp, &(rs->pos)))
		goto error;
	if (fread(der, sizeof(char), i, rs->fp) < (unsigned)i)
		goto error;

	ret = ASN1_read_cert(der);

	return ret;
error:
	if (der)
		free(der);
	OK_set_error(ERR_ST_FILEREAD, ERR_LC_PROF, ERR_PT_PFFILE + 9, NULL);
	return NULL;
}

/*-----------------------------------------
  get a key from keyfile
-----------------------------------------*/
Key *Prof_get_key(CertStat *cs)
{
	unsigned char buf[32], *der = NULL, *ivc = NULL, *enc = NULL, *dec =
	    NULL, *cp;
	Key *ret = NULL;
	int i, j, type;

	if (cs->kfp == NULL) {
		OK_set_error(ERR_ST_NULLPOINTER, ERR_LC_PROF,
			     ERR_PT_PFFILE + 10, NULL);
		return NULL;
	}
	/* get length of encrypted private key */
	if (fsetpos(cs->kfp, &(cs->kpos)))
		goto error;
	if (fread(buf, sizeof(char), 32, cs->kfp) < 32)
		goto error;

	if ((ASN1_skip_(buf, &i)) == NULL) {
		OK_set_error(ERR_ST_NULLPOINTER, ERR_LC_PROF,
			     ERR_PT_PFFILE + 10, NULL);
		return NULL;
	}

	if ((der = (unsigned char *)malloc(i + 16)) == NULL) {
		OK_set_error(ERR_ST_MEMALLOC, ERR_LC_PROF, ERR_PT_PFFILE + 10,
			     NULL);
		return NULL;
	}

	/* get key information & encrypted private key */
	if (fsetpos(cs->kfp, &(cs->kpos)))
		goto error;
	if (fread(der, sizeof(char), i, cs->kfp) < (unsigned)i)
		goto error;

	cp = ASN1_next(der);
	if ((type = ASN1_integer(cp, &i)) == 0)
		goto error;

	if ((type != KEY_RSA_PRV) && (type != KEY_DSA_PRV)
	    && (type != KEY_ECDSA_PRV))
		goto error;

	cp = ASN1_next(cp);
	if (ASN1_octetstring(cp, &i, &ivc, &j) < 0)
		goto error;
	cp = ASN1_next(cp);
	if (ASN1_octetstring(cp, &i, &enc, &j) < 0)
		goto error;

	/* get decrypted private key */
	if ((dec = PEM_msg_decrypt(enc, j, ivc, OBJ_CRYALGO_3DESCBC)) == NULL)
		goto error;

	switch (type) {
	case KEY_RSA_PRV:
		ret = (Key *) ASN1_read_rsaprv(dec);
		break;
	case KEY_DSA_PRV:
		ret = (Key *) ASN1_read_dsaprv(dec);
		break;
	case KEY_ECDSA_PRV:
		ret = (Key *) ASN1_read_ecdsaprv(dec);
		break;
	}

	if (ret == NULL)
		goto error;

	memset(buf, 0, 32);
	free(der);
	free(ivc);
	free(enc);

	return ret;
error:
	if (der)
		free(der);
	if (ivc)
		free(ivc);
	if (enc)
		free(enc);
	if (dec)
		free(dec);
	if (ret)
		Key_free(ret);
	OK_set_error(ERR_ST_FILEREAD, ERR_LC_PROF, ERR_PT_PFFILE + 10, NULL);
	return NULL;
}

/**
 * get a CSR from <ca_path>/req/XXXXXXXX.req
 */
Req *Prof_get_csr(CSRProf *rpf, CertStat *cs)
{
	Req *ret = NULL;
	unsigned char *der = NULL;
	char buf[272] = "", tmp[32];

	sprintf(tmp, "%.8d.req", cs->acceptID);
	if (set_path(buf, 272,
		     rpf->pCA->ca_path, "/req/", tmp, NULL) == -1) {
		return NULL;
	}

	if ((der = ASN1_read_der(buf)) == NULL)
		goto error;
	if ((ret = ASN1_read_req(der)) == NULL)
		goto error;

	return ret;
error:
	if (der)
		free(der);
	return NULL;
}

/**
 * read a CSR template.
 */
CertTemplate *Prof_get_csrtmpl(CSRProf *rpf, CertStat *cs)
{
	CertTemplate *ret = NULL;
	unsigned char *der = NULL;
	char buf[272] = "", tmp[32];
	int i;

	sprintf(tmp, "%.8d.req", cs->acceptID);
	if (set_path(buf, 272,
		     rpf->pCA->ca_path, "/req/", tmp, NULL) == -1) {
		return NULL;
	}

	if ((der = ASN1_read_der(buf)) == NULL)
		goto error;
	if ((ret = ASN1_cmp_certtmpl(der, &i)) == NULL)
		goto error;

	return ret;
error:
	if (der)
		free(der);
	return NULL;
}
