/* prof_conv.c */
/*
 * Copyright (c) 2004-2015 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>

#include "ok_caerr.h"
#include "ok_ca.h"

/* ReqStat functions */
ReqStat *ReqStat_new();
void ReqStat_free(ReqStat *st);
void ReqStat_free_all(ReqStat *top);
ReqStat *ReqStat_find(ReqStat *top, int sn);
Req *ReqStat_get_req(ReqStat *rs);

/* convert functions */
ReqStat *ca_read_reqstat(char *fname);
ReqStat *ca_asn1_reqstat(unsigned char *in);
int ca_read_reqfile(FILE *rfp, ReqStat *top);
int ca_read_keyfile(FILE *kfp, ReqStat *top);
int conv_ca_keydata(CA *ca, ReqStat *top);
int conv_add_keyfile(CertProf *cpf, CertStat *cs, ReqStat *rs);

/*-----------------------------------------
  ReqStat new & free
-----------------------------------------*/
ReqStat *ReqStat_new()
{
	ReqStat *ret;

	if ((ret = (ReqStat *) malloc(sizeof(ReqStat))) == NULL) {
		OK_set_error(ERR_ST_MEMALLOC, ERR_LC_PROF, ERR_PT_PFCONV, NULL);
		return NULL;
	}
	memset(ret, 0, sizeof(ReqStat));
	return ret;
}

void ReqStat_free(ReqStat *st)
{
	if (st == NULL)
		return;
	if (st->subject)
		free(st->subject);
	free(st);
}

void ReqStat_free_all(ReqStat *top)
{
	ReqStat *next;

	while (top) {
		next = top->next;
		ReqStat_free(top);
		top = next;
	}
}

ReqStat *ReqStat_find(ReqStat *top, int sn)
{
	while (top) {
		if (top->serialNum == sn)
			break;
		top = top->next;
	}
	return top;
}

Req *ReqStat_get_req(ReqStat *rs)
{
	unsigned char buf[32], *der = NULL;
	int i;
	Req *ret = NULL;

	if (rs->rfp == NULL)
		goto error;

	if (fsetpos(rs->rfp, &(rs->rpos)))
		goto error;
	if (fread(buf, sizeof(char), 32, rs->rfp) < 32)
		goto error;

	if ((ASN1_skip_(buf, &i)) == NULL)
		goto error;

	if ((der = (unsigned char *)malloc(i + 16)) == NULL) {
		OK_set_error(ERR_ST_MEMALLOC, ERR_LC_PROF, ERR_PT_PFCONV + 1,
			     NULL);
		return NULL;
	}

	if (fsetpos(rs->rfp, &(rs->rpos)))
		goto error;
	if (fread(der, sizeof(char), i, rs->rfp) < (unsigned)i)
		goto error;

	if ((ret = ASN1_read_req(der)) == NULL)
		goto error;

	return ret;
error:
	OK_set_error(ERR_ST_BADFORMAT, ERR_LC_PROF, ERR_PT_PFCONV + 1, NULL);
	if (der)
		free(der);
	return NULL;
}

/*-----------------------------------------
  old req & key file to v2 key file
-----------------------------------------*/
int CA_reqkey2newkey(CA *ca)
{
	char buf[256] = "";
	FILE *rfp = NULL, *kfp = NULL;
	ReqStat *stat = NULL;
	int ok = -1;

	/* get request stat */
	if (set_path(buf, 256,
		     ca->ca_path, "/req/req.rqi", NULL) == -1) {
		goto done;
	}
	if ((stat = ca_read_reqstat(buf)) != NULL) {
		/* get reqfile */
		if (set_path(buf, 256,
			     ca->ca_path, "/req/req.rqs", NULL) == -1) {
			goto done;
		}
		if ((rfp = fopen(buf, "rb")) == NULL)
			goto done;
		if (ca_read_reqfile(rfp, stat))
			goto done;

		/* get keyfile */
		if (set_path(buf, 256,
			     ca->ca_path, "/req/req.kys", NULL) == -1) {
			goto done;
		}
		if ((kfp = fopen(buf, "rb")) == NULL)
			goto done;
		if (ca_read_keyfile(kfp, stat))
			goto done;

		/* move key data to new format */
		if (conv_ca_keydata(ca, stat))
			goto done;

		if (rfp) {
			fclose(rfp);
			rfp = NULL;
		}
		if (kfp) {
			fclose(kfp);
			kfp = NULL;
		}
	}
#if 1
	/* remove old data */
	set_path(buf, 256, ca->ca_path, "/req/req.rqi", NULL);
	if (unlink(buf))
		goto done;

	/* get reqfile */
	set_path(buf, 256, ca->ca_path, "/req/req.rqs", NULL);
	if (unlink(buf))
		goto done;

	/* get keyfile */
	set_path(buf, 256, ca->ca_path, "/req/req.kys", NULL);
	if (unlink(buf))
		goto done;
#endif

	ok = 0;
done:
	if (rfp)
		fclose(rfp);
	if (kfp)
		fclose(kfp);
	if (stat)
		ReqStat_free_all(stat);
	return ok;
}

int CA_is_oldreq_exist(CA *ca)
{
	char buf[256];
	FILE *fp = NULL;
	int ret = 0;

	if (set_path(buf, 256,
		     ca->ca_path, "/req/req.rqi", NULL) == -1) {
		return -1;
	}
	if ((fp = fopen(buf, "rb")) != NULL) {
		ret = 1;
		fclose(fp);
	}
	return ret;
}

/*-----------------------------------------
  read files
-----------------------------------------*/
ReqStat *ca_read_reqstat(char *fname)
{
	unsigned char *cp, *der = NULL;
	ReqStat *ret = NULL;

	if ((der = ASN1_read_der(fname)) == NULL)
		goto done;

	cp = ASN1_next(der);
	if ((cp[0] != 0xa0) || (cp[1] != 0x03))
		goto done;

	/* version != 0 */
	cp = ASN1_next(cp);
	if (cp[2] != 0)
		goto done;

	/* get stat */
	cp = ASN1_step(cp, 2);
	ret = ca_asn1_reqstat(cp);

done:
	if (der)
		free(der);
	return ret;
}

ReqStat *ca_asn1_reqstat(unsigned char *in)
{
	unsigned char *cp, *sq;
	ReqStat *st, *ret = NULL;
	int len, i, j, k;

	if (*in != 0x30)
		goto error;
	len = ASN1_tlen(in);
	sq = ASN1_next(in);

	for (i = 0; i < len; i += j) {
		if (ret) {
			if ((st->next = ReqStat_new()) == NULL)
				goto error;
			st->next->prev = st;
			st = st->next;
			st->next = NULL;
		} else {
			if ((st = ret = ReqStat_new()) == NULL)
				goto error;
		}

		cp = (unsigned char *)ASN1_next(sq);
		st->serialNum = ASN1_integer(cp, &k);
		cp += k;

		if ((st->subject = asn1_get_str(cp, &k)) == NULL)
			goto error;
		cp += k;

		if (UTC2stm(cp, &st->date))
			goto error;
		st->rfp = st->kfp = NULL;
		memset(&st->rpos, 0, sizeof(fpos_t));
		memset(&st->kpos, 0, sizeof(fpos_t));

		if ((sq = ASN1_skip_(sq, &j)) == NULL)
			goto error;
	}
	return ret;
error:
	ReqStat_free_all(ret);
	OK_set_error(ERR_ST_BADFORMAT, ERR_LC_PROF, ERR_PT_PFCONV + 2, NULL);
	return NULL;
}

int ca_read_reqfile(FILE *rfp, ReqStat *top)
{
	unsigned char buf[64];
	int i, j;
	fpos_t rpos;
	ReqStat *st;

	rewind(rfp);
	if (fgetpos(rfp, &rpos))
		goto error;
	do {
		i = fread(buf, sizeof(char), 64, rfp);
		if (fsetpos(rfp, &rpos))
			goto error;

		if (i <= 1)
			break;
		if ((i = ASN1_integer(buf, &j)) == 0)
			goto error;

		st = ReqStat_find(top, i);

		if (fread(buf, sizeof(char), j, rfp) < (unsigned)j)
			goto error;
		if (fgetpos(rfp, &rpos))
			goto error;
		if (st) {
			st->rfp = rfp;
			st->rpos = rpos;
		}
		ASN1_skip_(ASN1_next(buf), &j);

		if (fseek(rfp, j, SEEK_CUR))
			goto error;
		if (fgetpos(rfp, &rpos))
			goto error;

	} while (1);

	return 0;
error:
	OK_set_error(ERR_ST_BADFORMAT, ERR_LC_PROF, ERR_PT_PFCONV + 3, NULL);
	return -1;
}

int ca_read_keyfile(FILE *kfp, ReqStat *top)
{
	unsigned char buf[64];
	int i, j;
	fpos_t kpos;
	ReqStat *st;

	rewind(kfp);
	if (fgetpos(kfp, &kpos))
		goto error;
	do {
		i = fread(buf, sizeof(char), 64, kfp);
		if (fsetpos(kfp, &kpos))
			goto error;

		if (i <= 1)
			break;
		if ((i = ASN1_integer(buf, &j)) == 0)
			goto error;

		st = ReqStat_find(top, i);

		if (fread(buf, sizeof(char), j, kfp) < (unsigned)j)
			goto error;
		if (fgetpos(kfp, &kpos))
			goto error;
		if (st) {
			st->kfp = kfp;
			st->kpos = kpos;
		}
		ASN1_skip_(ASN1_next(buf), &j);

		if (fseek(kfp, j, SEEK_CUR))
			goto error;
		if (fgetpos(kfp, &kpos))
			goto error;

	} while (1);

	return 0;
error:
	OK_set_error(ERR_ST_BADFORMAT, ERR_LC_PROF, ERR_PT_PFCONV + 3, NULL);
	return -1;
}

/*-----------------------------------------
  convert data
-----------------------------------------*/
int conv_ca_keydata(CA *ca, ReqStat *top)
{
	CertStat *cs;
	CertProf *cpf;
	Req *req = NULL;
	Cert *ct = NULL;
	int i, ok = -1;

	/* set CertStat */
	for (cpf = ca->profList; cpf; cpf = cpf->next) {
		ca->cprof = cpf;
		if (Prof_reload_ctinfo(ca))
			goto done;
		if (Prof_open_ctfile(ca))
			goto done;
		if (Prof_open_keyfile(ca))
			goto done;
		/* clear all key flag */
		for (cs = cpf->stat; cs; cs = cs->next)
			cs->state &= ~STAT_HAVEKEY;
	}

	/* convert key file */
	for (; top; top = top->next) {
		/* find appropriate CertStat */
		for (cpf = ca->profList; cpf; cpf = cpf->next) {
			for (i = 0, cs = cpf->stat; cs; cs = cs->next) {
				if ((cs =
				     Prof_find_statstr(cs,
						       top->subject)) == NULL)
					break;

				/* check key */
				req = ReqStat_get_req(top);
				ct = Prof_get_cert(cs);
				if (req && ct)
					if (!Key_cmp(req->pubkey, ct->pubkey))
						i = 1;
				Req_free(req);
				req = NULL;
				Cert_free(ct);
				ct = NULL;
				if (i)
					break;
			}
			if (cs)
				break;	/* find matched key pair */
		}

		if (cpf && top->kfp)
			if (conv_add_keyfile(cpf, cs, top))
				goto done;
	}

	/* save state changes */
	for (cpf = ca->profList; cpf; cpf = cpf->next) {
		ca->cprof = cpf;
		if (Prof_save_ctinfo(ca))
			goto done;
	}

	ok = 0;
done:
	Req_free(req);
	Cert_free(ct);
	return ok;
}

int conv_add_keyfile(CertProf *cpf, CertStat *cs, ReqStat *rs)
{
	unsigned char buf[128], *der = NULL;
	int i, ok = -1;

	/* set serialNum (for certstat) */
	ASN1_set_integer(cs->serialNum, buf, &i);
	if (fseek(cpf->kfp, 0, SEEK_END))
		goto done;
	if (fwrite(buf, sizeof(char), i, cpf->kfp) < (unsigned)i)
		goto done;

	/* set pointer */
	cs->kfp = cpf->kfp;
	if (fgetpos(cpf->kfp, &(cs->kpos)))
		goto done;

	/* move key data */
	if (fsetpos(rs->kfp, &(rs->kpos)))
		goto done;
	if (fread(buf, sizeof(char), 32, rs->kfp) < 32)
		goto done;

	if ((ASN1_skip_(buf, &i)) == NULL)
		goto done;
	if ((der = (unsigned char *)malloc(i + 16)) == NULL)
		goto done;

	/* get key information & encrypted private key */
	if (fsetpos(rs->kfp, &(rs->kpos)))
		goto done;
	if (fread(der, sizeof(char), i, rs->kfp) < (unsigned)i)
		goto done;

	/* write encrypted private key info to new file */
	if (fwrite(der, sizeof(char), i, cs->kfp) < (unsigned)i)
		goto done;
	cs->state |= STAT_HAVEKEY;

	/* commit to disk */
	if (fflush(cpf->kfp))
		goto done;

	ok = 0;
done:
	if (der)
		free(der);
	if (ok)
		OK_set_error(ERR_ST_FILEWRITE, ERR_LC_PROF, ERR_PT_PFCONV + 5,
			     NULL);
	return ok;
}
