/* prof_asn1.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>

#include "ok_caerr.h"
#include "ok_ca.h"

/*-----------------------------------------
  asn1 read cert profile info
-----------------------------------------*/
int ASN1_ctprof_info(CA *ca,unsigned char *in){
	unsigned char *cp;
	int i;

	cp = ASN1_next(in); /* skip sequence */

	if((cp[0]!=0xa0)||(cp[1]!=0x03)){
		OK_set_error(ERR_ST_BADFORMAT,ERR_LC_PROF,ERR_PT_PFASN1,NULL);
		goto error;
	}
	cp = ASN1_next(cp);

	/* version should be 0 or more */
	if((ca->cprof->ver = ASN1_integer(cp,&i)) < 0) goto error;
	cp = ASN1_next(cp);

	/* read ca stat */
	if(ASN1_ctprof_stat(ca,cp)) goto error;
	cp = ASN1_skip(cp);

	/* read cert stat */
	if(ASN1_ctprof_ctstat(ca,cp)) goto error;

	if(*cp==0xa2)
		if((cp = ASN1_skip(cp))==NULL) goto error;
	
	/* read cert extension info */
	if(ASN1_ctprof_ext(ca,cp)) goto error;

	return 0;
error:
	OK_set_error(ERR_ST_BADFORMAT,ERR_LC_PROF,ERR_PT_PFASN1,NULL);
	return -1;
}

/*-----------------------------------------
  asn1 read crl profile info
-----------------------------------------*/
int ASN1_clprof_info(CA *ca,unsigned char *in){
	unsigned char *cp;
	int i;

	cp = ASN1_next(in); /* skip sequence */

	if((cp[0]!=0xa0)||(cp[1]!=0x03)){
		OK_set_error(ERR_ST_BADFORMAT,ERR_LC_PROF,ERR_PT_PFASN1+1,NULL);
		goto error;
	}
	cp = ASN1_next(cp);

	/* version should be 0 or more */
	if((ca->lprof->ver = ASN1_integer(cp,&i)) < 0) goto error;
	cp = ASN1_next(cp);

	/* read ca stat */
	if(ASN1_clprof_stat(ca,cp)) goto error;
	cp = ASN1_skip(cp);

	/* read cert extension info */
	if(ASN1_clprof_ext(ca,cp)) goto error;

	return 0;
error:
	return -1;
}

/*-----------------------------------------
  asn1 read csr profile info
-----------------------------------------*/
int ASN1_csrprof_info(CA *ca,unsigned char *in){
	unsigned char *cp;
	int i;

	cp = ASN1_next(in); /* skip sequence */

	if((cp[0]!=0xa0)||(cp[1]!=0x03)){
		OK_set_error(ERR_ST_BADFORMAT,ERR_LC_PROF,ERR_PT_PFASN1+1,NULL);
		goto error;
	}
	cp = ASN1_next(cp);

	/* version should be 0 or more */
	if((ca->csrProf->ver = ASN1_integer(cp,&i)) < 0) goto error;
	cp = ASN1_next(cp);

	/* read ca stat */
	if(ASN1_csrprof_stat(ca,cp)) goto error;
	cp = ASN1_skip(cp);

	/* read cert stat */
	if(ASN1_csrprof_ctstat(ca,cp)) goto error;

	return 0;
error:
	return -1;
}

/*-----------------------------------------
  asn1 read cert profile status
-----------------------------------------*/
int ASN1_ctprof_stat(CA *ca, unsigned char *in){
	CertProf *cpf = ca->cprof;
	unsigned char *cp,*t,*tmp=NULL;
	int i,j;

	if(*in!=0xa1){
		OK_set_error(ERR_ST_BADFORMAT,ERR_LC_PROF,ERR_PT_PFASN1+2,NULL);
		goto error;
	}
	cp = ASN1_next(in);

	/* certificate serial number */
	if((cpf->serialNum = ASN1_integer(cp,&i))<0) goto error;
	cp = ASN1_next(cp);

	/* certificate valid days */
	if(*cp==ASN1_INTEGER){ /* old version */
		if((cpf->cert_sec = ASN1_integer(cp,&i))<0) goto error;
		cp = ASN1_next(cp);
	}else{
		t=ASN1_next(cp);
		if(*t==ASN1_NULL){
			cpf->bgtype = 0;
		}else if((*t==ASN1_GENERALIZEDTIME)||(*t==ASN1_UTCTIME)){
			cpf->bgtype = 1;
			if(UTC2stm(t,&cpf->ct_begin)) goto error;
		}else if(*t==ASN1_PRINTABLE_STRING){
			cpf->bgtype = 2;
			if(UTC2stm(t,&cpf->ct_begin)) goto error;
		}
		t=ASN1_next(t);

		if(*t==ASN1_INTEGER){
			cpf->edtype = 0;
			if((cpf->cert_sec = ASN1_integer(t,&i))<0) goto error;
		}else if((*t==ASN1_GENERALIZEDTIME)||(*t==ASN1_UTCTIME)){
			cpf->edtype = 1;
			if(UTC2stm(t,&cpf->ct_end)) goto error;
		}else if(*t==ASN1_PRINTABLE_STRING){
			cpf->edtype = 2;
			if(UTC2stm(t,&cpf->ct_end)) goto error;
		}
		t=ASN1_next(t);

		if(*t==ASN1_INTEGER){
			cpf->updtype = 0;
			if((cpf->upd_sec = ASN1_integer(t,&i))<0) goto error;
		}
		cp = ASN1_skip(cp);
	}
	if((cpf->ver<2)&&(cpf->edtype==0)){
		cpf->cert_sec  = cpf->cert_sec*3600*24;
	}

	/* crl days ... skip */
	cp = ASN1_next(cp);

	/* issuer DN ... skip */
	cp = ASN1_next(cp);

	/* subject DN ... skip */
	cp = ASN1_next(cp);

	/* get default tag point ... skip */
	cp = ASN1_next(cp);

	/* get default directory information ... skip */
	cp = ASN1_next(cp);

	/* get DN signature type */
	if((cpf->sigtype = ASN1_integer(cp,&i))<0) goto error; /* if file is old type, return... */

	if(cpf->sigtype<=1000) cpf->sigtype = DEFAULT_SIG_TYPE;
	cp = ASN1_next(cp);

	/* get default public key long */
	if((cpf->keylong = ASN1_integer(cp,&i))<0) goto error;
	cp = ASN1_next(cp);

	/* get default subject filter ... skip */
	cp = ASN1_next(cp);

	/* if version is 0.84 or later, CA policy is supported.. */
	/* this section becomes profile working policy since ver1.2 */
	if(*cp == ASN1_OCTETSTRING){
		if(cpf->ver > 2){ /* should be format ver 3 or later*/
			if(ASN1_octetstring(cp,&i,&tmp,&j)) goto error;
			memcpy(cpf->pol_flag,tmp,(j>8)?(8):(j));
			free(tmp); tmp=NULL;
		}else{
			cpf->pol_flag[0] |= CPF_WP0_allowSameSbjDN | CPF_WP0_allowSameKey;
		}
		cp = ASN1_next(cp);
	}
	/* if version is 0.88 or later, CA has crl number */
	if(*cp == ASN1_INTEGER) cp = ASN1_next(cp);

	/* if version is 0.88 or later, CA has cert version */
	if(*cp == ASN1_INTEGER){
		if((cpf->cert_ver = ASN1_integer(cp,&i))<0) goto error;
		cp = ASN1_next(cp);
	}
	/* if version is 0.88 or later, CA has CRL version */
	if(*cp == ASN1_INTEGER) cp = ASN1_next(cp);

	/* cross certificate flag. */
	if(*cp == ASN1_INTEGER){
		if((cpf->isCross = ASN1_integer(cp,&i))<0) goto error;
		cp = ASN1_next(cp);
	}
	/* if AiCA version is 1.2 or later, subject template is saved */
	if(*cp == 0xa0){
		if((tmp=ASN1_get_subject(cp,&cpf->sbj_tmpl))==NULL) goto error;
		free(tmp); tmp=NULL;
		if((cp=ASN1_skip(cp))==NULL) goto error;
	}

	return 0;
error:
	if(tmp) free(tmp);
	return -1;
}

/*-----------------------------------------
  asn1 read cert profile status
-----------------------------------------*/
int ASN1_clprof_stat(CA *ca, unsigned char *in){
	CRLProf *cpf = ca->lprof;
	unsigned char *cp,*t;
	int i;

	if(*in!=0xa1){
		OK_set_error(ERR_ST_BADFORMAT,ERR_LC_PROF,ERR_PT_PFASN1+3,NULL);
		goto error;
	}
	cp = ASN1_next(in);

	/* certificate serial number ... skip */
	cp = ASN1_next(cp);

	/* certificate valid days ... skip */
	cp = ASN1_next(cp);

	/* crl valid days */
	if(*cp==ASN1_INTEGER){ /* old version */
		if((cpf->crl_sec = ASN1_integer(cp,&i))<0) goto error;
		cp = ASN1_next(cp);
	}else{
		t=ASN1_next(cp);
		if(*t==ASN1_NULL){
			cpf->bgtype = 0;
		}else if((*t==ASN1_GENERALIZEDTIME)||(*t==ASN1_UTCTIME)){
			cpf->bgtype = 1;
			if(UTC2stm(t,&cpf->cl_begin)) goto error;
		}
		t=ASN1_next(t);

		if((cpf->crl_sec = ASN1_integer(t,&i))<0) goto error;
		cp = ASN1_skip(cp);
	}
	if(cpf->ver<2){
		cpf->crl_sec  = cpf->crl_sec*3600*24;
	}

	/* issuer DN ... skip */
	cp = ASN1_next(cp);

	/* subject DN ... skip */
	cp = ASN1_next(cp);

	/* get default tag point ... skip */
	cp = ASN1_next(cp);

	/* get default directory information ... skip */
	cp = ASN1_next(cp);

	/* get DN signature type */
	if((cpf->sigtype = ASN1_integer(cp,&i))<0) goto error; /* if file is old type, return... */

	if(cpf->sigtype<=1000) cpf->sigtype = DEFAULT_SIG_TYPE;
	cp = ASN1_next(cp);

	/* get default public key long ... skip */
	cp = ASN1_next(cp);

	/* get default subject filter ... skip */
	cp = ASN1_next(cp);

	/* if version is 0.84 or later, CA policy is supported ... skip */
	if(*cp == ASN1_OCTETSTRING) cp = ASN1_next(cp);

	/* if version is 0.88 or later, CA has crl number */
	if(*cp == ASN1_INTEGER){
		if((cpf->crlNum = ASN1_integer(cp,&i))<0) goto error;
		cp = ASN1_next(cp);
	}

	/* if version is 0.88 or later, CA has cert version ... skip */
	if(*cp == ASN1_INTEGER) cp = ASN1_next(cp);

	/* if version is 0.88 or later, CA has crl version */
	if(*cp == ASN1_INTEGER){
		if((cpf->crl_ver = ASN1_integer(cp,&i))<0) goto error;
	}

	return 0;
error:
	return -1;
}

/*-----------------------------------------
  asn1 read cert profile status
-----------------------------------------*/
int ASN1_csrprof_stat(CA *ca, unsigned char *in){
	CSRProf *rpf = ca->csrProf;
	unsigned char *cp;
	int i;

	if(*in!=0xa1){
		OK_set_error(ERR_ST_BADFORMAT,ERR_LC_PROF,ERR_PT_PFASN1+4,NULL);
		goto error;
	}
	cp = ASN1_next(in);

	/* current CSR number */
	if((rpf->csrNum=ASN1_integer(cp,&i))<0) goto error;

	return 0;
error:
	return -1;
}

/*-----------------------------------------
  asn1 read profile cert status
-----------------------------------------*/
int ASN1_ctprof_ctstat(CA *ca, unsigned char *in){

	if(*in!=0xa2) return 0;
	if((ca->cprof->stat=asn1_ctprof_ctstat(in))==NULL) return -1;

	return 0;
}
	
int ASN1_csrprof_ctstat(CA *ca, unsigned char *in){

	if(*in!=0xa2) return 0;
	if((ca->csrProf->stat=asn1_ctprof_ctstat(in))==NULL) return -1;

	return 0;
}

CertStat *asn1_ctprof_ctstat(unsigned char *in){
	unsigned char *cp,*sq;
	CertStat *st,*hd=NULL,*ret=NULL;
	int	len,i,j,k,l;

	len=ASN1_tlen(in);
	sq =ASN1_next(in);

	for(i=0;i<len;){
		if((st=CertStat_new())==NULL) goto error;
		if(ret){
			hd->next=st; st->prev=hd; hd=st;
		}else{
			ret=hd=st; 
		}
		cp = ASN1_next(sq);

		if((st->state = ASN1_integer(cp,&k))<0) goto error;
		cp=ASN1_next(cp);

		st->serialNum = ASN1_integer(cp,&k);
		cp=ASN1_next(cp);

		if((st->subject = asn1_get_str(cp,&k))==NULL) goto error;
		cp=ASN1_next(cp);

		if(UTC2stm(cp,&st->notBefore)) goto error;
		cp=ASN1_next(cp);

		if(UTC2stm(cp,&st->notAfter)) goto error;
		cp=ASN1_next(cp);

		if((*cp==ASN1_UTCTIME)||(*cp==ASN1_GENERALIZEDTIME)){
			if(UTC2stm(cp,&st->revokedDate)) goto error;
			cp=ASN1_next(cp);
		}
		if(*cp==0x80){
			if((st->acceptID = ASN1_integer_(cp,&k,1))<0) goto error;
			cp=ASN1_next(cp);
		}
		if(*cp==0x81){
			*cp = ASN1_GENERALIZEDTIME;
			if(UTC2stm(cp,&st->createdDate)) goto error;
			cp=ASN1_next(cp);
		}
		if(*cp==0x82){ /* keyID OPTIONAL */
			if(ASN1_octetstring_(cp,&k,&st->keyID,&l,1)) goto error;
			cp=ASN1_next(cp);
		}

		if((sq=ASN1_skip_(sq,&j))==NULL) goto error;
		i+=j;
	}
	return ret;
error:
	CertStat_free_all(ret);
	return NULL;
}

/*-----------------------------------------
  asn1 read profile extensions
-----------------------------------------*/
int ASN1_prof_ext(CA *ca,unsigned char *in,unsigned char pf){
	unsigned char *cp,*ecp,*oid,cr;
	int i,j,k,id,len;
	CertExt *et,*now,*top=NULL;

	if(*in!=pf) return 0;

	if(pf==0xa3){ CertExt_free_all(ca->cprof->ext); ca->cprof->ext=NULL; }
	else{ CertExt_free_all(ca->lprof->ext); ca->lprof->ext=NULL; }

	len= ASN1_tlen(in);
	in = ASN1_next(in);

	for(et=now=NULL,i=0;i<len;){
		cp = ASN1_next(in);
		cr = 0;
		oid= NULL;

		if(*cp==ASN1_INTEGER){
			if((id = ASN1_integer(cp,&k))<0) goto error;

		}else if(*cp==ASN1_OBJECT_IDENTIFIER){
			if((id = ASN1_object_2int(cp))<0) goto error;
			if((oid= ASN1_dup(cp))==NULL) goto error;
		}
		cp = ASN1_next(cp);

		/* get critical */
		if(*cp==ASN1_BOOLEAN){
			cr = cp[2];
			cp = ASN1_next(cp);
		}

		/* get octetstring */
		if(ASN1_octetstring(cp,&j,&ecp,&j)) goto error;
	
		if((now=ASN1_get_ext(id,ecp))==NULL) goto error;
		now->critical = cr;
		now->objid = oid;
		free(ecp);

		if(top){
			et->next = now;
			et = et->next;
		}else{
			top=et=now;
		}

		if((in = ASN1_skip_(in,&j))==NULL) goto error;
		i+= j;
	}

	if(pf==0xa3){ ca->cprof->ext = top; }
	else{ ca->lprof->ext = top; }

	return 0;
error:
	if(top) CertExt_free_all(top);
	return -1;
}

/*-----------------------------------------
  asn1 read certificates file
-----------------------------------------*/
int ASN1_ctprof_ctfile(CA *ca, int sn, int mode){
	unsigned char buf[64];
	CertProf *cpf = ca->cprof;
	CertStat *st;
	fpos_t	pos;
	int i,j,k;

	rewind(cpf->fp);
	if(fgetpos(cpf->fp,&pos)) goto error;
	do{
		i = fread(buf,sizeof(char),64,cpf->fp);
		if(fsetpos(cpf->fp,&pos)) goto error; /* set last position */

		if(i<=1) break;
		if((i = ASN1_integer(buf,&j))<0) goto error;
		
		if((mode&0x2)||((mode&0x30)&&(sn==i))){
			st = CA_find_stat(ca,i); k=0;

			if(fread(buf,sizeof(char),j,cpf->fp)<(unsigned)j) goto error;
			if(fgetpos(cpf->fp,&pos)) goto error;
			if(st){
				st->fp = cpf->fp;
				memcpy(&st->pos,&pos,sizeof(fpos_t));
			}
			if((mode&0x10)&&(sn==i)) break;
		}else{
			k=j;
		}
		ASN1_skip_(ASN1_next(buf),&j);

		if(fseek(cpf->fp,j+k,SEEK_CUR)) goto error;
		if(fgetpos(cpf->fp,&pos)) goto error;

	}while(1);

	return 0;
error:
	OK_set_error(ERR_ST_BADFORMAT,ERR_LC_PROF,ERR_PT_PFASN1+6,NULL);
	return -1;
}

/*-----------------------------------------
  asn1 read certificates file
-----------------------------------------*/
int ASN1_ctprof_keyfile(CA *ca, int sn, int mode){
	unsigned char buf[64],*cp;
	CertProf *cpf = ca->cprof;
	CertStat *st;
	fpos_t	pos;
	int i,j,k;

	rewind(cpf->kfp);
	if(fgetpos(cpf->kfp,&pos)) goto error;
	do{
		i = fread(buf,sizeof(char),64,cpf->kfp);
		if(fsetpos(cpf->kfp,&pos)) goto error; /* set last position */

		if(i<=1) break;
		if((i = ASN1_integer(buf,&j))==0) goto error;
		cp = ASN1_next(buf);
		
		if( ((mode&0x2)||((mode&0x10)&&(sn==i))) &&
					(cp[0] != ASN1_OCTETSTRING)){ /* should not deleted */
			st = CA_find_stat(ca,i); k=0;

			if(fread(buf,sizeof(char),j,cpf->kfp)<(unsigned)j) goto error;
			if(fgetpos(cpf->kfp,&pos)) goto error;
			if(st){
				st->kfp = cpf->kfp;
				memcpy(&st->kpos,&pos,sizeof(fpos_t));
				st->state |= STAT_HAVEKEY;

				if((mode&0x10)&&(sn==i)) break;
			}
		}else{
			k=j;
		}
		ASN1_skip_(ASN1_next(buf),&j);

		if(fseek(cpf->kfp,j+k,SEEK_CUR)) goto error;
		if(fgetpos(cpf->kfp,&pos)) goto error;

	}while(1);

	return 0;
error:
	OK_set_error(ERR_ST_BADFORMAT,ERR_LC_PROF,ERR_PT_PFASN1+7,NULL);
	return -1;
}

/*-----------------------------------------------------------------*/
/*-----------------------------------------
  Cert Profile structure to DER
-----------------------------------------*/
int Prof_estimate_ctdersize(CertProf *cpf){
	CertExt *et;
	CertStat *st;
	int	ret=132,i;

	for(i=0,st=cpf->stat; st ;st=st->next){
		ret += strlen(st->subject) + 80;
		if(st->keyID) ret += 24;
	}  
	for(i=0,et=cpf->ext; et ;et=et->next)
		ret += et->dlen + 16;

	if(cpf->sbj_tmpl.num > 0){
		for(i=0;i<cpf->sbj_tmpl.num;i++)
			if(cpf->sbj_tmpl.rdn[i].tag)
				ret+=strlen(cpf->sbj_tmpl.rdn[i].tag)+20;
	}

	return ret;
}

unsigned char *Prof_cert_toDER(CA *ca,unsigned char *buf,int *ret_len){
	CertProf *cpf = ca->cprof;
	unsigned char *cp,*ret;
	int	i,j;

	if(buf==NULL){
		if((i=Prof_estimate_ctdersize(cpf))<=0)
			return NULL;

		if((ret=(unsigned char*)malloc(i))==NULL){
			OK_set_error(ERR_ST_MEMALLOC,ERR_LC_PROF,ERR_PT_PFASN1+8,NULL);
			return NULL;
		}
		memset(ret,0,i);
	}else{
		ret=buf;
	}
	
	/*
	 * AiCA 0.81 or older ... version 1
	 * AiCA 0.9 to 1.1    ... version 2
	 * AiCA 1.2 or later  ... version 3
	 */
	ASN1_set_integer(3,ret,&i);
	ASN1_set_explicit(i,0,ret,&i);

	cp = ret+i;
	if(Prof_DER_ctstat(cpf,cp,&j)) goto error;
	cp+=j; i+=j;

	if(Prof_DER_ctctstat(cpf,cp,&j)) goto error;
	cp+=j; i+=j;

	if(Prof_DER_ctext(ca,cp,&j)) goto error;
	i+=j;

	ASN1_set_sequence(i,ret,ret_len);
	return ret;
error:
	if(ret!=buf){free(ret);}
	return NULL;
}

int Prof_DER_ctstat(CertProf *cpf,unsigned char *ret,int *ret_len){
	unsigned char *t,*cp;
	int	i,j,k;

	ASN1_set_integer(cpf->serialNum,ret,&i); /* serialNumber */
	cp = t = ret+i;

	/* !!caution!!
	 * old version (ctprof DER ver 1) set cert_days here.
	 * but ctprof DER ver 2 set cert_sec.
	 */
	k=0;
	if(cpf->bgtype==0){ 
		ASN1_set_null(t);k=2;
	}else if(cpf->bgtype==1){
		if(Cert_DER_time(&cpf->ct_begin,t,&k)) goto error;
	}else{ /* ver 3 */
		if(stm2UTC(&cpf->ct_begin,t,ASN1_PRINTABLE_STRING)==NULL) goto error;
		k=ASN1_tlen(t)+2;
	}
	t+=k; j=0;
	if(cpf->edtype==0){
		ASN1_set_integer(cpf->cert_sec,t,&j); /* ver 2 */
	}else if(cpf->edtype==1){
		if(Cert_DER_time(&cpf->ct_end,t,&j)) goto error;
	}else{ /* ver 3 */
		if(stm2UTC(&cpf->ct_end,t,ASN1_PRINTABLE_STRING)==NULL) goto error;
		j=ASN1_tlen(t)+2;
	}
	k+=j; t+=j; j=0;
	if(cpf->updtype==0){
		ASN1_set_integer(cpf->upd_sec,t,&j); /* ver 3 */
	}
	k+=j; t+=j;
	ASN1_set_explicit(k,0,cp,&j);
	cp+=j; i+=j;

	ASN1_set_integer(0,cp,&j); /* dummy */
	cp+=j; i+=j;
	if(ASN1_set_t61("dummy",cp,&j)) goto error; /* dummy */
	cp+=j; i+=j;
	if(ASN1_set_t61("dummy",cp,&j)) goto error; /* dummy */
	cp+=j; i+=j;

	/* default tag point */
	if(ASN1_set_ia5("CN",cp,&j)) goto error; /* dummy */
	cp+=j; i+=j;

	/* default subject list */
	if(ASN1_set_t61("$$",cp,&j)) goto error; /* dummy */
	cp+=j; i+=j;

	/* default key type and key long */
	ASN1_set_integer(cpf->sigtype,cp,&j); /* key type (signature type) */
	cp+=j; i+=j;
	ASN1_set_integer(cpf->keylong,cp,&j); /* key long */
	cp+=j; i+=j;

	/* subject filter in the listview... */
	ASN1_set_integer(0,cp,&j); /* dummy */
	cp+=j; i+=j;

	/* Profile working policy (v1.2) */
	ASN1_set_octetstring(8,cpf->pol_flag,cp,&j);
	cp+=j; i+=j;

	/* CA crl number (v0.88) */
	ASN1_set_integer(0,cp,&j); /* dummy */
	cp+=j; i+=j;

	/* cert version */
	ASN1_set_integer(cpf->cert_ver,cp,&j);
	cp+=j; i+=j;

	/* CRL version */
	ASN1_set_integer(0,cp,&j); /* dummy */
	cp+=j; i+=j;

	/* closs cert flag */
	ASN1_set_integer(cpf->isCross,cp,&j);
	cp+=j; i+=j;

	/* template subject info -- (v1.2) */
	if(cpf->sbj_tmpl.num>0){
		if(Cert_DER_subject(&cpf->sbj_tmpl,cp,&j)) goto error;
		*cp=0xa0; i+=j; cp+=j;
	}

	ASN1_set_explicit(i,1,ret,ret_len);
	return 0;
error:
	return -1;
}

int Prof_DER_ctctstat(CertProf *cpf,unsigned char *ret,int *ret_len){
	return prof_der_ctctstat(cpf->stat,ret,ret_len);
}

int prof_der_ctctstat(CertStat *top,unsigned char *ret,int *ret_len){
	unsigned char *cp,*sq;
	CertStat *st;
	int	i,j,k;

	sq = ret;
	*ret_len=0;
	if(top==NULL) return 0;

	for(i=0,st=top; st ;st=st->next){
		ASN1_set_integer(st->state,sq,&j);
		cp = sq+j;
		ASN1_set_integer(st->serialNum,cp,&k);
		cp+=k; j+=k;
		if(ASN1_set_t61(st->subject,cp,&k)) goto error;
		cp+=k; j+=k;
		if(Cert_DER_time(&st->notBefore,cp,&k)) goto error;
		cp+=k; j+=k;
		if(Cert_DER_time(&st->notAfter,cp,&k)) goto error;
		cp+=k; j+=k;

		if(st->revokedDate.tm_year){
			if(Cert_DER_time(&st->revokedDate,cp,&k)) goto error;
			cp+=k; j+=k;
		}
		if(st->acceptID>0){
			ASN1_set_integer(st->acceptID,cp,&k);
			*cp=0x80; cp+=k; j+=k;
		}
		if(st->createdDate.tm_year){
			if(stm2UTC(&st->createdDate,cp,ASN1_GENERALIZEDTIME)==NULL) goto error;
			k=ASN1_tlen(cp)+2; *cp=0x81; cp+=k; j+=k;
		}
		if(st->keyID){
			ASN1_set_octetstring(20,st->keyID,cp,&k);
			*cp=0x82; cp+=k; j+=k;
		}
		ASN1_set_sequence(j,sq,&j);
		sq+=j; i+=j;
	}
	ASN1_set_explicit(i,2,ret,ret_len);
	return 0;
error:
	return -1;
}

int Prof_DER_ext(CA *ca,char exp,unsigned char *ret,int *ret_len){
	unsigned char *cp,*sq;
	CertExt *et;
	int	i,j,k;
  
	sq=ret;
	*ret_len=0;

	et = (exp==3)?(ca->cprof->ext):(ca->lprof->ext);
	if(et==NULL) return 0;

	for(i=0; et ;et=et->next){
		if((et->extnID<=0)&&(et->objid==NULL)) continue;

		cp=sq;
		if(et->extnID){
			ASN1_set_integer(et->extnID,cp,&j);
		}else{
			j = ASN1_tlen(et->objid) + 2; /* should be less than 128 byte */
			memcpy(cp,et->objid,j);
		}
		cp+=j;

		if(et->critical){
			ASN1_set_boolean(et->critical,cp,&k);
			cp+=k; j+=k;
		}

		ASN1_set_octetstring(et->dlen,et->der,cp,&k);
		j+=k;
		ASN1_set_sequence(j,sq,&j);
		sq+=j; i+=j;
	}
	ASN1_set_explicit(i,exp,ret,ret_len);
	return 0;
}

/*-----------------------------------------
  CRL Profile structure to DER
-----------------------------------------*/
int Prof_estimate_cldersize(CRLProf *cpf){
	CertExt *et;
	int	ret=120,i;

	for(i=0,et=cpf->ext; et ;et=et->next)
		ret += et->dlen + 16;

	return ret;
}

unsigned char *Prof_crl_toDER(CA *ca,unsigned char *buf,int *ret_len){
	CRLProf *cpf = ca->lprof;
	unsigned char *cp,*ret;
	int	i,j;

	if(buf==NULL){
		if((i=Prof_estimate_cldersize(cpf))<=0)
			return NULL;

		if((ret=(unsigned char*)malloc(i))==NULL){
			OK_set_error(ERR_ST_MEMALLOC,ERR_LC_PROF,ERR_PT_PFASN1+11,NULL);
			return NULL;
		}
		memset(ret,0,i);
	}else{
		ret=buf;
	}
	
	/*
	 * AiCA 0.81 or older ... version 1
	 * AiCA 0.9 or later  ... version 2
	 */
	ASN1_set_integer(2,ret,&i);
	ASN1_set_explicit(i,0,ret,&i);

	cp = ret+i;
	if(Prof_DER_clstat(cpf,cp,&j)) goto error;
	cp+=j; i+=j;

	if(Prof_DER_clext(ca,cp,&j)) goto error;
	i+=j;

	ASN1_set_sequence(i,ret,ret_len);
	return ret;
error:
	if(ret!=buf){free(ret);}
	return NULL;
}

int Prof_DER_clstat(CRLProf *cpf,unsigned char *ret,int *ret_len){
	unsigned char *cp,*t,buf[8];
	int	i,j,k;

	ASN1_set_integer(0,ret,&i); /* dummy */
	cp = ret+i;
	ASN1_set_integer(0,cp,&j); /* dummy */
	t=cp=cp+j; i+=j;

	/* !!caution!!
	 * old version (crlprof DER ver 1) set crl_days here.
	 * but crlprof DER ver 2 set crl_sec.
	 */
	if(cpf->bgtype==0){ 
		ASN1_set_null(t);k=2;
	}else{
		if(Cert_DER_time(&cpf->cl_begin,t,&k)) goto error;
	}
	t+=k;
	ASN1_set_integer(cpf->crl_sec,t,&j); /* ver 2 */
	k+=j;
	ASN1_set_explicit(k,0,cp,&j);
	cp+=j; i+=j;

	if(ASN1_set_t61("dummy",cp,&j)) goto error; /* dummy */
	cp+=j; i+=j;
	if(ASN1_set_t61("dummy",cp,&j)) goto error; /* dummy */
	cp+=j; i+=j;

	/* default tag point */
	if(ASN1_set_ia5("C",cp,&j)) goto error; /* dummy */
	cp+=j; i+=j;

	/* default subject list */
	if(ASN1_set_t61("$$",cp,&j)) goto error; /* dummy */
	cp+=j; i+=j;

	/* default key type and key long */
	ASN1_set_integer(cpf->sigtype,cp,&j); /* key type (signature type) */
	cp+=j; i+=j;
	ASN1_set_integer(0,cp,&j); /* dummy */
	cp+=j; i+=j;

	/* subject filter in the listview... */
	ASN1_set_integer(0,cp,&j); /* dummy */
	cp+=j; i+=j;

	/* CA policy (v0.84) */
	ASN1_set_octetstring(4,buf,cp,&j); /* dummy */
	cp+=j; i+=j;

	/* CA crl number (v0.88) */
	ASN1_set_integer(cpf->crlNum,cp,&j);
	cp+=j; i+=j;

	/* cert version */
	ASN1_set_integer(0,cp,&j); /* dummy */
	cp+=j; i+=j;

	/* crl version */
	ASN1_set_integer(cpf->crl_ver,cp,&j);
	i+=j;

	ASN1_set_explicit(i,1,ret,ret_len);
	return 0;
error:
	return -1;
}

/*-----------------------------------------
  Cert Profile structure to DER
-----------------------------------------*/
int Prof_estimate_csrdersize(CSRProf *rpf){
	CertStat *st;
	int	ret=64,i;

	for(i=0,st=rpf->stat; st ;st=st->next){
		ret += strlen(st->subject) + 80;
		if(st->keyID) ret += 24;
	}
	return ret;
}

unsigned char *Prof_csr_toDER(CA *ca,unsigned char *buf,int *ret_len){
	CSRProf *rpf = ca->csrProf;
	unsigned char *cp,*ret;
	int	i,j;

	if(buf==NULL){
		if((i=Prof_estimate_csrdersize(rpf))<=0)
			return NULL;

		if((ret=(unsigned char*)malloc(i))==NULL){
			OK_set_error(ERR_ST_MEMALLOC,ERR_LC_PROF,ERR_PT_PFASN1+10,NULL);
			return NULL;
		}
		memset(ret,0,i);
	}else{
		ret=buf;
	}
	
	/*
	 * AiCA 0.9 or later  ... version 1
	 */
	ASN1_set_integer(1,ret,&i);
	ASN1_set_explicit(i,0,ret,&i);

	cp = ret+i;
	if(Prof_DER_csrstat(rpf,cp,&j)) goto error;
	cp+=j; i+=j;

	if(Prof_DER_csrctstat(rpf,cp,&j)) goto error;
	cp+=j; i+=j;

	ASN1_set_sequence(i,ret,ret_len);
	return ret;
error:
	if(ret!=buf){free(ret);}
	return NULL;
}

int Prof_DER_csrstat(CSRProf *rpf,unsigned char *ret,int *ret_len){
	unsigned char *cp;
	int	i;

	ASN1_set_integer(rpf->csrNum,ret,&i); /* current AcceptID */
	cp = ret+i;

	ASN1_set_explicit(i,1,ret,ret_len);
	return 0;
}

int Prof_DER_csrctstat(CSRProf *rpf,unsigned char *ret,int *ret_len){
	int ok = 0;

	OK_set_time_dertag(ASN1_GENERALIZEDTIME);
	ok = prof_der_ctctstat(rpf->stat,ret,ret_len);
	OK_set_time_dertag(0); /* reset DER tag info */

	return ok;
}
