/* lcmptestfc.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_pem.h>

#include "ok_lcmp.h"

#define TSREQ	"testcsr.pem"
#define TSCERT	"testcert.pem"
#define TSCRL	"testcrl.pem"
#define TSKEY	"testkey.pem"

#ifndef PATH
#define PATH 	"."
#endif

/* common/getfpath.c */
char *get_fpath(char *path, char *filename);


/*--------------------------------------
	set dummy content
--------------------------------------*/
LCMPOp *set_test_bindreq0(){
	LO_BindReq *ret;

	if((ret=(LO_BindReq*)LCMP_op_new(LCMP_OP_BINDREQ))==NULL) return NULL;
	ret->authMode = LCMP_OPBRQ_ATSIMPLE;
	if ((ret->caname = strdup("/usr/local/aica/testca")) == NULL) return NULL;
	if ((ret->username = strdup("administrator")) == NULL) return NULL;
	if ((ret->passwd = strdup("abcdef")) == NULL) return NULL;
	
	return (LCMPOp*)ret;
}

LCMPCtrl *set_test_ctrl(){
	LCMPCtrl *ret=NULL,*hd,*ct;
	unsigned char buf[8]={0x1,0x2,0x3,0x4,0x5,0x6,0x7,0x8};
	int i;

	for(i=0; i<2; i++){
		if((ct=LCMP_ctrl_new())==NULL) return NULL;
		if(ret){
			hd->next=ct; hd=ct;
		}else{
			ret=hd=ct;
		}
		if ((hd->ctrlType = strdup("1.2.33.444")) == NULL) return NULL;
		if((hd->value=malloc(10))==NULL) return NULL;
		memcpy(hd->value,buf,8);
		hd->vlen = 8;
		hd->crit = 1;
	}
	/* remove critical flag from last ctrl */
	hd->crit = 0;

	return ret;
}

LCMPOp *set_test_bindreq1(){
	LO_BindReq *ret;

	if((ret=(LO_BindReq*)LCMP_op_new(LCMP_OP_BINDREQ))==NULL) return NULL;
	ret->authMode = LCMP_OPBRQ_ATSSLCL;
	if ((ret->caname = strdup("/usr/local/aica/testca")) == NULL) return NULL;
	if((ret->ctrl=set_test_ctrl())==NULL) return NULL;
	
	return (LCMPOp*)ret;
}

LCMPOp *set_test_bindrsp0(){
	LO_BindRsp *ret;
	CertList *cl,*hd;
	Cert *ct;
	int i;
	char *fp_TSCERT = get_fpath(PATH, TSCERT);

	if((ret=(LO_BindRsp*)LCMP_op_new(LCMP_OP_BINDRSP))==NULL) return NULL;

	ret->resultCode = LCMP_SUCCESS;
	if ((ret->resultMsg = strdup("bind ok.")) == NULL) return NULL;

	for(i=0; i<2; i++){
		if((ct=Cert_read_file(fp_TSCERT))==NULL) return NULL;
		if((cl=Cert_2Certlist(ct))==NULL) return NULL;
		if(ret->list){
			hd->next=cl; hd=cl;
		}else{
			ret->list=hd=cl;
		}
		Cert_free(ct);
	}
	free(fp_TSCERT);
	return (LCMPOp*)ret;
}

LCMPOp *set_test_bindrsp1(){
	LO_BindRsp *ret;

	if((ret=(LO_BindRsp*)LCMP_op_new(LCMP_OP_BINDRSP))==NULL) return NULL;

	ret->resultCode = LCMP_AUTH_ERR;
	if ((ret->resultMsg = strdup("bad bind request.")) == NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_unbindreq0(){
	LCMPOp *ret;

	if((ret=(LCMPOp*)LCMP_op_new(LCMP_OP_UNBIND))==NULL) return NULL;
	if((ret->ctrl=set_test_ctrl())==NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_signreq0(){
	LO_SignReq *ret;
	char *fp_TSREQ = get_fpath(PATH, TSREQ);

	if((ret=(LO_SignReq*)LCMP_op_new(LCMP_OP_SIGNREQ))==NULL) return NULL;

	if((ret->p10=Req_read_file(fp_TSREQ))==NULL) return NULL;
	if ((ret->profName = strdup("SMIME user")) == NULL) return NULL;
	free(fp_TSREQ);

	return (LCMPOp*)ret;
}

LCMPOp *set_test_signreq1(){
	LO_SignReq *ret;
	char *fp_TSREQ = get_fpath(PATH, TSREQ);

	if((ret=(LO_SignReq*)LCMP_op_new(LCMP_OP_SIGNREQ))==NULL) return NULL;

	if((ret->p10=Req_read_file(fp_TSREQ))==NULL) return NULL;
	if ((ret->profName = strdup("SMIME user")) == NULL) return NULL;
	ret->serialNum = 1000;
	free(fp_TSREQ);

	return (LCMPOp*)ret;
}

LCMPOp *set_test_signreq2(){
	LO_SignReq *ret;
	Req *req = NULL;
	char *fp_TSREQ = get_fpath(PATH, TSREQ);

	if((ret=(LO_SignReq*)LCMP_op_new(LCMP_OP_SIGNREQ))==NULL) return NULL;

	if((ret->tmpl=CMP_certtmpl_new())==NULL) return NULL;
	if((req=Req_read_file(fp_TSREQ))==NULL) return NULL;

	if ((ret->profName = strdup("SMIME user")) == NULL) return NULL;
	if(Cert_dncopy(&req->subject_dn,&ret->tmpl->subject)) return NULL;
	ret->tmpl->publicKey = req->pubkey; req->pubkey = NULL;
	Req_free(req);
	free(fp_TSREQ);

	return (LCMPOp*)ret;
}

LCMPOp *set_test_signreq3(){
	LO_SignReq *ret;
	Cert *ct = NULL;
	CertExt *et,*hd;
	char *fp_TSCERT = get_fpath(PATH, TSCERT);

	if((ret=(LO_SignReq*)LCMP_op_new(LCMP_OP_SIGNREQ))==NULL) return NULL;

	if((ret->tmpl=CMP_certtmpl_new())==NULL) return NULL;
	if((ct=Cert_read_file(fp_TSCERT))==NULL) return NULL;

	if ((ret->profName = strdup("SMIME user")) == NULL) return NULL;
	if(Cert_dncopy(&ct->subject_dn,&ret->tmpl->subject)) return NULL;
	ret->tmpl->publicKey = ct->pubkey; ct->pubkey = NULL;
	ret->tmpl->serialNumber = ct->serialNumber;
	memcpy(&ret->tmpl->validity.notBefore,&ct->time.notBefore,sizeof(struct tm));
	memcpy(&ret->tmpl->validity.notAfter,&ct->time.notAfter,sizeof(struct tm));
	if((et=Extnew_basic_cons(1,10))==NULL) return NULL;
	ret->tmpl->ext=hd=et;
	if((et=Extnew_keyusage(0x7f))==NULL) return NULL;
	hd->next=et; hd=et;

	Cert_free(ct);
	free(fp_TSCERT);

	return (LCMPOp*)ret;
}

LCMPOp *set_test_signrsp0(){
	LO_SignRsp *ret;

	if((ret=(LO_SignRsp*)LCMP_op_new(LCMP_OP_SIGNRSP))==NULL) return NULL;

	ret->resultCode = LCMP_OPERATION_ERR;
	if ((ret->resultMsg = strdup("bad pkcs#10 format.")) == NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_signrsp1(){
	LO_SignRsp *ret;
	char *fp_TSCERT = get_fpath(PATH, TSCERT);

	if((ret=(LO_SignRsp*)LCMP_op_new(LCMP_OP_SIGNRSP))==NULL) return NULL;
	if((ret->cert=Cert_read_file(fp_TSCERT))==NULL) return NULL;

	ret->resultCode = LCMP_SUCCESS;
	if ((ret->resultMsg = strdup("certificate returned")) == NULL) return NULL;
	free(fp_TSCERT);

	return (LCMPOp*)ret;
}

LCMPOp *set_test_listreq0(){
	LO_ListReq *ret;

	if((ret=(LO_ListReq*)LCMP_op_new(LCMP_OP_LISTREQ))==NULL) return NULL;

	if ((ret->profName = strdup("SMIME user")) == NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_listreq1(){
	LO_ListReq *ret;

	if((ret=(LO_ListReq*)LCMP_op_new(LCMP_OP_LISTREQ))==NULL) return NULL;

	if ((ret->profName = strdup("SMIME user")) == NULL) return NULL;
	ret->serialNum = 1000;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_listreq2(){
	LO_ListReq *ret;

	if((ret=(LO_ListReq*)LCMP_op_new(LCMP_OP_LISTREQ))==NULL) return NULL;

	if ((ret->profName = strdup("SMIME user")) == NULL) return NULL;
	if ((ret->sbjQuery = strdup("CN=test taro")) == NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_listrsp0(){
	LO_ListRsp *ret;

	if((ret=(LO_ListRsp*)LCMP_op_new(LCMP_OP_LISTRSP))==NULL) return NULL;

	ret->resultCode = LCMP_PERMISSION_DENY;
	if ((ret->resultMsg = strdup("cannot get certlist")) == NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_listrsp1(){
	LO_ListRsp *ret;
	CertStat *st,*hd;
	struct tm *stm;
	time_t t;
	int i;

	if((ret=(LO_ListRsp*)LCMP_op_new(LCMP_OP_LISTRSP))==NULL) return NULL;

	ret->resultCode = LCMP_SUCCESS;
	if ((ret->resultMsg = strdup("return certlist")) == NULL) return NULL;

	for(i=0; i<10; i++){
		if((st=CertStat_new())==NULL) return NULL;
		if(ret->stat){
			hd->next=st; st->prev=hd; hd=st;
		}else{
			ret->stat=hd=st;
		}
		st->state = 1 << (i%5);
		st->serialNum = i*10 + 1;
		if ((st->subject = strdup("C=JP,o=test,cn=testuser")) == NULL) return NULL;
		time(&t); stm=gmtime(&t);
		memcpy(&st->notBefore,stm,sizeof(struct tm));
		memcpy(&st->notAfter,stm,sizeof(struct tm));
		if(st->state&STAT_REVOKED)
			memcpy(&st->revokedDate,stm,sizeof(struct tm));
		if((st->keyID=(unsigned char*)malloc(22))==NULL) return NULL;
		strcpy(st->keyID,"dummy id");
	}

	return (LCMPOp*)ret;
}

LCMPOp *set_test_profreq0(){ /* view prof content */
	LO_ProfReq *ret;

	if((ret=(LO_ProfReq*)LCMP_op_new(LCMP_OP_PROFREQ))==NULL) return NULL;

	ret->profOp=0;
	if ((ret->profName = strdup("SMIME user")) == NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_profreq1_0(){ /* add prof content */
	LO_ProfReq *ret;

	if((ret=(LO_ProfReq*)LCMP_op_new(LCMP_OP_PROFREQ))==NULL) return NULL;

	ret->profOp=1;
	if ((ret->profName = strdup("SMIME user")) == NULL) return NULL;

	if((ret->prof=ProfTmpl_new())==NULL) return NULL;
	if ((ret->prof->tmplName = strdup("SMIME profile template")) == NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_profreq1_1(){ /* add prof content */
	LO_ProfReq *ret;
	ProfTmpl *pf;

	if((ret=(LO_ProfReq*)LCMP_op_new(LCMP_OP_PROFREQ))==NULL) return NULL;

	ret->profOp=1;
	if ((ret->profName = strdup("SMIME user")) == NULL) return NULL;

	if((ret->prof=pf=ProfTmpl_new())==NULL) return NULL;
	if ((pf->tmplName = strdup("SMIME profile template")) == NULL) return NULL;
	pf->baseNum  = 10000;
	pf->certSec  = 1000*(24*3600);
	pf->edtype   = 0;
	pf->certVer  = 2;
	pf->hashType = OBJ_HASH_SHA1;
	pf->keyType  = OBJ_CRYPT_RSA;
	pf->keyLength= 1024;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_profreq2(){ /* delete prof content */
	LO_ProfReq *ret;

	if((ret=(LO_ProfReq*)LCMP_op_new(LCMP_OP_PROFREQ))==NULL) return NULL;

	ret->profOp=2;
	if ((ret->profName = strdup("SMIME user2")) == NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_profreq3_0(){ /* modify prof content */
	LO_ProfReq *ret;

	if((ret=(LO_ProfReq*)LCMP_op_new(LCMP_OP_PROFREQ))==NULL) return NULL;

	ret->profOp=3;
	if ((ret->profName = strdup("SMIME user3")) == NULL) return NULL;

	if((ret->prof=ProfTmpl_new())==NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_profreq3_1(){ /* modify prof content */
	LO_ProfReq *ret;
	ProfTmpl *pf;
	Cert *ct = NULL;
	struct tm *stm;
	time_t t;
	char *fp_TSCERT = get_fpath(PATH, TSCERT);

	if((ret=(LO_ProfReq*)LCMP_op_new(LCMP_OP_PROFREQ))==NULL) return NULL;
	ret->profOp=3;
	if ((ret->profName = strdup("SMIME user3")) == NULL) return NULL;

	if((ret->prof=pf=ProfTmpl_new())==NULL) return NULL;
	pf->baseNum  = 10000;
	pf->certSec  = 1000*(24*3600);
	pf->bgtype   = 1;
	pf->edtype   = 0;
	pf->certVer  = 2;
	pf->hashType = OBJ_HASH_SHA1;
	pf->keyType  = OBJ_CRYPT_RSA;
	pf->keyLength= 1024;
	time(&t); stm=gmtime(&t);
	memcpy(&pf->start,stm,sizeof(struct tm));

	memset(pf->policyFlag,0,8);
	pf->policyFlag[0] |= 0x80;

	if((ct=Cert_read_file(fp_TSCERT))==NULL) return NULL;
	if(Cert_dncopy(&ct->subject_dn,&pf->sbjTmpl)) return NULL;
	Cert_free(ct);
	free(fp_TSCERT);

	return (LCMPOp*)ret;
}

LCMPOp *set_test_profreq3_2(){ /* modify prof content */
	LO_ProfReq *ret;
	ProfTmpl *pf;
	struct tm *stm;
	time_t t;

	if((ret=(LO_ProfReq*)LCMP_op_new(LCMP_OP_PROFREQ))==NULL) return NULL;

	ret->profOp=3;
	if ((ret->profName = strdup("SMIME user3")) == NULL) return NULL;

	if((ret->prof=pf=ProfTmpl_new())==NULL) return NULL;
	time(&t); stm=gmtime(&t);
	memcpy(&pf->start,stm,sizeof(struct tm));
	memcpy(&pf->end,stm,sizeof(struct tm));
	pf->bgtype   = 1;
	pf->edtype   = 1;
	pf->updtype  = 0;
	pf->updSec   = 30 * 24 * 2600;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_profreq3_3(){ /* modify prof content */
	LO_ProfReq *ret;
	ProfTmpl *pf;

	if((ret=(LO_ProfReq*)LCMP_op_new(LCMP_OP_PROFREQ))==NULL) return NULL;

	ret->profOp=3;
	if ((ret->profName = strdup("SMIME user3")) == NULL) return NULL;

	if((ret->prof=pf=ProfTmpl_new())==NULL) return NULL;
	pf->bgtype   = 2;
	pf->edtype   = 2;
	pf->updtype  = 0;
	pf->updSec   = 30 * 24 * 2600;
	pf->start.tm_year  = 9999;
	pf->start.tm_mon   = 99;
	pf->start.tm_mday  = 10;
	pf->end.tm_year    = 3;
	pf->end.tm_mday    = 14;
	return (LCMPOp*)ret;
}

LCMPOp *set_test_profreq4(){ /* view profile extension */
	LO_ProfReq *ret;

	if((ret=(LO_ProfReq*)LCMP_op_new(LCMP_OP_PROFREQ))==NULL) return NULL;

	ret->profOp=4;
	if ((ret->profName = strdup("SMIME user3")) == NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_profreq5(){ /* add profile extension */
	LO_ProfReq *ret;
	CertExt *et,*hd;

	if((ret=(LO_ProfReq*)LCMP_op_new(LCMP_OP_PROFREQ))==NULL) return NULL;

	ret->profOp=5;
	if ((ret->profName = strdup("SMIME user3")) == NULL) return NULL;

	if((et=Extnew_basic_cons(1,10))==NULL) return NULL;
	ret->ext=hd=et;
	if((et=Extnew_keyusage(0x7f))==NULL) return NULL;
	hd->next=et; hd=et;
	if((et=Extnew_ns_flag(0x7f))==NULL) return NULL;
	hd->next=et;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_profreq6(){ /* delete profile extension */
	LO_ProfReq *ret;
	CertExt *et,*hd;

	if((ret=(LO_ProfReq*)LCMP_op_new(LCMP_OP_PROFREQ))==NULL) return NULL;

	ret->profOp=6;
	if ((ret->profName = strdup("SMIME user3")) == NULL) return NULL;

	if((et=CertExt_new(OBJ_X509v3_SbjKeyIdt))==NULL) return NULL;
	ret->ext=hd=et;
	if((et=CertExt_new(OBJ_X509v3_KEY_Usage))==NULL) return NULL;
	hd->next=et; hd=et;
	if((et=CertExt_new(OBJ_NS_CERT_TYPE))==NULL) return NULL;
	hd->next=et; hd=et;
	if((et=CertExt_new(OBJ_PKIX_IDPE_AIA))==NULL) return NULL;
	hd->next=et;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_profreq7(){ /* modify profile extension */
	LO_ProfReq *ret;
	CertExt *et,*hd;

	if((ret=(LO_ProfReq*)LCMP_op_new(LCMP_OP_PROFREQ))==NULL) return NULL;

	ret->profOp=7;
	if ((ret->profName = strdup("SMIME user3")) == NULL) return NULL;

	if((et=Extnew_basic_cons(1,10))==NULL) return NULL;
	ret->ext=hd=et;
	if((et=Extnew_keyusage(0x7f))==NULL) return NULL;
	hd->next=et; hd=et;
	if((et=Extnew_ns_flag(0x7f))==NULL) return NULL;
	hd->next=et;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_profrsp0(){
	LO_ProfRsp *ret;

	if((ret=(LO_ProfRsp*)LCMP_op_new(LCMP_OP_PROFRSP))==NULL) return NULL;

	ret->resultCode = LCMP_OPERATION_ERR;
	if ((ret->resultMsg = strdup("bad operation code.")) == NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_profrsp1(){
	LO_ProfRsp *ret;

	if((ret=(LO_ProfRsp*)LCMP_op_new(LCMP_OP_PROFRSP))==NULL) return NULL;

	ret->resultCode = LCMP_SUCCESS;
	if ((ret->resultMsg = strdup("return profile info")) == NULL) return NULL;

	if((ret->prof=ProfTmpl_new())==NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_profrsp2(){
	LO_ProfRsp *ret;
	ProfTmpl *pf;
	Cert *ct = NULL;
	char *fp_TSCERT = get_fpath(PATH, TSCERT);

	if((ret=(LO_ProfRsp*)LCMP_op_new(LCMP_OP_PROFRSP))==NULL) return NULL;

	ret->resultCode = LCMP_SUCCESS;
	if ((ret->resultMsg = strdup("return profile info")) == NULL) return NULL;

	if((ret->prof=pf=ProfTmpl_new())==NULL) return NULL;
	pf->baseNum   = 10000;
	pf->edtype      = 2;
	pf->end.tm_year = 3;
	pf->end.tm_mon  = 1;
	pf->updSec    = 986*(24*3600);
	pf->updtype   = 0;
	pf->certVer   = 2;
	pf->hashType  = OBJ_HASH_SHA1;
	pf->keyType   = OBJ_CRYPT_RSA;
	pf->keyLength = 1024;

	memset(pf->policyFlag,0,8);
	pf->policyFlag[0] |= 0x80;

	if((ct=Cert_read_file(fp_TSCERT))==NULL) return NULL;
	if(Cert_dncopy(&ct->subject_dn,&pf->sbjTmpl)) return NULL;
	Cert_free(ct);
	free(fp_TSCERT);

	return (LCMPOp*)ret;
}

LCMPOp *set_test_profrsp3(){
	LO_ProfRsp *ret;
	CertExt *et,*hd;

	if((ret=(LO_ProfRsp*)LCMP_op_new(LCMP_OP_PROFRSP))==NULL) return NULL;

	ret->resultCode = LCMP_SUCCESS;
	if ((ret->resultMsg = strdup("return profile info")) == NULL) return NULL;

	if((et=Extnew_basic_cons(1,10))==NULL) return NULL;
	ret->ext=hd=et;
	if((et=Extnew_keyusage(0x7f))==NULL) return NULL;
	hd->next=et; hd=et;
	if((et=Extnew_ns_flag(0x7f))==NULL) return NULL;
	hd->next=et;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_profrsp4(){
	LO_ProfRsp *ret;

	if((ret=(LO_ProfRsp*)LCMP_op_new(LCMP_OP_PROFRSP))==NULL) return NULL;

	ret->resultCode = LCMP_SUCCESS;
	if ((ret->resultMsg = strdup("return profile info")) == NULL) return NULL;

	ret->listnum = 5;
	if((ret->proflist=(char**)malloc(sizeof(char*)*5))==NULL) return NULL;

	if ((ret->proflist[0] = strdup("SMIME user")) == NULL) return NULL;
	if ((ret->proflist[1] = strdup("SMIME user2")) == NULL) return NULL;
	if ((ret->proflist[2] = strdup("student@science")) == NULL) return NULL;
	if ((ret->proflist[3] = strdup("teacher@science")) == NULL) return NULL;
	if ((ret->proflist[4] = strdup("Cross Cert")) == NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_certreq0(){
	LO_CertReq *ret;

	if((ret=(LO_CertReq*)LCMP_op_new(LCMP_OP_CERTREQ))==NULL) return NULL;
	ret->certOp = 0;
	ret->serialNum = 10;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_certreq1(){
	LO_CertReq *ret;

	if((ret=(LO_CertReq*)LCMP_op_new(LCMP_OP_CERTREQ))==NULL) return NULL;
	ret->certOp = 1;
	ret->serialNum = 10;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_certreq2(){
	LO_CertReq *ret;

	if((ret=(LO_CertReq*)LCMP_op_new(LCMP_OP_CERTREQ))==NULL) return NULL;
	ret->certOp = 2;
	ret->serialNum = 10;
	if ((ret->passwd = strdup("abcdef")) == NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_certreq3(){
	LO_CertReq *ret;
	char *fp_TSKEY = get_fpath(PATH, TSKEY);

	if((ret=(LO_CertReq*)LCMP_op_new(LCMP_OP_CERTREQ))==NULL) return NULL;
	ret->certOp = 4;
	ret->serialNum = 10;
	OK_set_passwd("abcde");
	if ((ret->passwd = strdup("abcdef")) == NULL) return NULL;
	if((ret->key=(Key*)PEM_read_rsaprv(fp_TSKEY))==NULL) return NULL;
	free(fp_TSKEY);

	return (LCMPOp*)ret;
}

LCMPOp *set_test_certrsp0(){
	LO_CertRsp *ret;

	if((ret=(LO_CertRsp*)LCMP_op_new(LCMP_OP_CERTRSP))==NULL) return NULL;

	ret->resultCode = LCMP_OPERATION_ERR;
	if ((ret->resultMsg = strdup("bad operation code.")) == NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_certrsp1(){
	LO_CertRsp *ret;
	char *fp_TSCERT = get_fpath(PATH, TSCERT);

	if((ret=(LO_CertRsp*)LCMP_op_new(LCMP_OP_CERTRSP))==NULL) return NULL;

	ret->resultCode = LCMP_SUCCESS;
	if ((ret->resultMsg = strdup("return certificate info")) == NULL) return NULL;
	if((ret->cert=Cert_read_file(fp_TSCERT))==NULL) return NULL;
	free(fp_TSCERT);

	return (LCMPOp*)ret;
}

LCMPOp *set_test_certrsp2(){
	LO_CertRsp *ret;
	char *fp_TSKEY = get_fpath(PATH, TSKEY);

	if((ret=(LO_CertRsp*)LCMP_op_new(LCMP_OP_CERTRSP))==NULL) return NULL;

	ret->resultCode = LCMP_SUCCESS;
	if ((ret->resultMsg = strdup("return certificate info")) == NULL) return NULL;

	OK_set_passwd("abcde");
	if((ret->key=(Key*)PEM_read_rsaprv(fp_TSKEY))==NULL) return NULL;
	free(fp_TSKEY);

	return (LCMPOp*)ret;
}

LCMPOp *set_test_csrreq0(){
	LO_CSRReq *ret;
	char *fp_TSREQ = get_fpath(PATH, TSREQ);

	if((ret=(LO_CSRReq*)LCMP_op_new(LCMP_OP_CSRREQ))==NULL) return NULL;
	ret->csrOp = 0; /* post */
	ret->acceptID  = 0;
	ret->serialNum = 10;
	if ((ret->profName = strdup("SMIME user3")) == NULL) return NULL;
	if((ret->csr=Req_read_file(fp_TSREQ))==NULL) return NULL;
	free(fp_TSREQ);

	return (LCMPOp*)ret;
}

LCMPOp *set_test_csrreq0_1(){
	LO_CSRReq *ret;
	Cert *ct = NULL;
	char *fp_TSCERT = get_fpath(PATH, TSCERT);

	if((ret=(LO_CSRReq*)LCMP_op_new(LCMP_OP_CSRREQ))==NULL) return NULL;
	ret->csrOp = 0; /* post */
	if((ret->tmpl=CMP_certtmpl_new())==NULL) return NULL;
	if((ct=Cert_read_file(fp_TSCERT))==NULL) return NULL;

	if ((ret->profName = strdup("SMIME user")) == NULL) return NULL;
	if(Cert_dncopy(&ct->subject_dn,&ret->tmpl->subject)) return NULL;
	ret->tmpl->publicKey = ct->pubkey; ct->pubkey = NULL;
	ret->tmpl->serialNumber = ct->serialNumber;

	Cert_free(ct);
	free(fp_TSCERT);

	return (LCMPOp*)ret;
}

LCMPOp *set_test_csrreq1(){
	LO_CSRReq *ret;

	if((ret=(LO_CSRReq*)LCMP_op_new(LCMP_OP_CSRREQ))==NULL) return NULL;
	ret->csrOp = 1; /* export CSR */
	ret->acceptID  = 10053;
	ret->serialNum = 0;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_csrreq2(){
	LO_CSRReq *ret;

	if((ret=(LO_CSRReq*)LCMP_op_new(LCMP_OP_CSRREQ))==NULL) return NULL;
	ret->csrOp = 2; /* delete CSR */
	ret->acceptID  = 10053;
	ret->serialNum = 0;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_csrreq3(){
	LO_CSRReq *ret;

	if((ret=(LO_CSRReq*)LCMP_op_new(LCMP_OP_CSRREQ))==NULL) return NULL;
	ret->csrOp = 4; /* reject CSR */
	ret->acceptID     = 10053;
	ret->serialNum    = 0;
	ret->rejectReason = 3;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_csrrsp0(){
	LO_CSRRsp *ret;

	if((ret=(LO_CSRRsp*)LCMP_op_new(LCMP_OP_CSRRSP))==NULL) return NULL;

	ret->resultCode = LCMP_REQEST_REJECTED;
	if ((ret->resultMsg = strdup("CSR accepted")) == NULL) return NULL;
	ret->serialNum = 0;
	ret->acceptID  = 1000534;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_csrrsp1(){
	LO_CSRRsp *ret;
	char *fp_TSREQ = get_fpath(PATH, TSREQ);

	if((ret=(LO_CSRRsp*)LCMP_op_new(LCMP_OP_CSRRSP))==NULL) return NULL;

	ret->resultCode = LCMP_SUCCESS;
	if ((ret->resultMsg = strdup("export CSR")) == NULL) return NULL;
	if((ret->csr=Req_read_file(fp_TSREQ))==NULL) return NULL;
	ret->serialNum = 0;
	ret->acceptID  = 1000534;
	free(fp_TSREQ);

	return (LCMPOp*)ret;
}

LCMPOp *set_test_csrrsp1_1(){
	LO_CSRRsp *ret;
	Cert *ct = NULL;
	char *fp_TSCERT = get_fpath(PATH, TSCERT);

	if((ret=(LO_CSRRsp*)LCMP_op_new(LCMP_OP_CSRRSP))==NULL) return NULL;

	ret->resultCode = LCMP_SUCCESS;
	if ((ret->resultMsg = strdup("export CSR")) == NULL) return NULL;

	if((ret->tmpl=CMP_certtmpl_new())==NULL) return NULL;
	if((ct=Cert_read_file(fp_TSCERT))==NULL) return NULL;

	if(Cert_dncopy(&ct->subject_dn,&ret->tmpl->subject)) return NULL;
	ret->tmpl->publicKey = ct->pubkey; ct->pubkey = NULL;
	ret->tmpl->serialNumber = ct->serialNumber;

	Cert_free(ct);
	free(fp_TSCERT);

	return (LCMPOp*)ret;
}

LCMPOp *set_test_csrrsp2(){
	LO_CSRRsp *ret;

	if((ret=(LO_CSRRsp*)LCMP_op_new(LCMP_OP_CSRRSP))==NULL) return NULL;

	ret->resultCode = LCMP_SUCCESS;
	if ((ret->resultMsg = strdup("issue new certificate")) == NULL) return NULL;
	ret->serialNum = 10532;
	ret->acceptID  = 1000534;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_crlreq0(){
	LO_CRLReq *ret;

	if((ret=(LO_CRLReq*)LCMP_op_new(LCMP_OP_CRLREQ))==NULL) return NULL;

	if ((ret->profName = strdup("CRL-All")) == NULL) return NULL;
	ret->crlOp = 0;
	ret->location = 0;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_crlreq1(){
	LO_CRLReq *ret;

	if((ret=(LO_CRLReq*)LCMP_op_new(LCMP_OP_CRLREQ))==NULL) return NULL;

	if ((ret->profName = strdup("ARL")) == NULL) return NULL;
	ret->crlOp = 1;
	ret->location = 1;
	ret->crlNum = 20;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_crlreq2(){
	LO_CRLReq *ret;

	if((ret=(LO_CRLReq*)LCMP_op_new(LCMP_OP_CRLREQ))==NULL) return NULL;

	if ((ret->profName = strdup("CRL")) == NULL) return NULL;
	ret->crlOp = 2;
	ret->location = 2;
	if ((ret->path = strdup("/usr/local/aica")) == NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_crlrsp0(){
	LO_CRLRsp *ret;

	if((ret=(LO_CRLRsp*)LCMP_op_new(LCMP_OP_CRLRSP))==NULL) return NULL;

	ret->resultCode = LCMP_OPERATION_ERR;
	if ((ret->resultMsg = strdup("bad operation code.")) == NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_crlrsp1(){
	LO_CRLRsp *ret;
	CRLList *cl,*hd;
	CRL *crl;
	int i;
	char *fp_TSCRL = get_fpath(PATH, TSCRL);

	if((ret=(LO_CRLRsp*)LCMP_op_new(LCMP_OP_CRLRSP))==NULL) return NULL;

	ret->resultCode = LCMP_SUCCESS;

	for(i=0; i<2; i++){
		if((crl=CRL_read_file(fp_TSCRL))==NULL) return NULL;
		if((cl=CRL_2CRLlist(crl))==NULL) return NULL;
		if(ret->list){
			hd->next=cl; hd=cl;
		}else{
			ret->list=hd=cl;
		}
		CRL_free(crl);
	}
	free(fp_TSCRL);

	return (LCMPOp*)ret;
}

LCMPOp *set_test_svopreq0(){
	LO_SVOpReq *ret;

	if((ret=(LO_SVOpReq*)LCMP_op_new(LCMP_OP_SVOPREQ))==NULL) return NULL;
	ret->svOp = 0;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_svopreq1(){
	LO_SVOpReq *ret;

	/* add a user */
	if((ret=(LO_SVOpReq*)LCMP_op_new(LCMP_OP_SVOPREQ))==NULL) return NULL;
	ret->svOp = 0;
	if((ret->uinfo=CA_lcmp_authinfo("raadmin","abcde",NULL,100,0,0x7fffffff,NULL))==NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_svopreq2(){
	LO_SVOpReq *ret;

	/* delete a user */
	if((ret=(LO_SVOpReq*)LCMP_op_new(LCMP_OP_SVOPREQ))==NULL) return NULL;
	ret->svOp = 1; 
	if((ret->uinfo=CA_lcmp_authinfo("raadmin",NULL,NULL,-1,-1,-1,NULL))==NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_svopreq3(){
	LO_SVOpReq *ret;

	/* modify a user */
	if((ret=(LO_SVOpReq*)LCMP_op_new(LCMP_OP_SVOPREQ))==NULL) return NULL;
	ret->svOp = 1; 
	if((ret->uinfo=CA_lcmp_authinfo("raadmin",NULL,NULL,-1,-1,0x7f000000,NULL))==NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_svopreq4(){
	LO_SVOpReq *ret;

	/* change password */
	if((ret=(LO_SVOpReq*)LCMP_op_new(LCMP_OP_SVOPREQ))==NULL) return NULL;
	ret->svOp = 1; 
	if((ret->uinfo=CA_lcmp_authinfo("raadmin","newpass","oldpass",-1,-1,-1,NULL))==NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_svoprsp0(){
	LO_SVOpRsp *ret;

	if((ret=(LO_SVOpRsp*)LCMP_op_new(LCMP_OP_SVOPRSP))==NULL) return NULL;

	ret->resultCode = LCMP_OPERATION_ERR;
	if ((ret->resultMsg = strdup("bad operation code.")) == NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_svoprsp1(){
	LO_SVOpRsp *ret;

	if((ret=(LO_SVOpRsp*)LCMP_op_new(LCMP_OP_SVOPRSP))==NULL) return NULL;

	ret->resultCode = LCMP_SUCCESS;
	if ((ret->opRsp = strdup("dummy code")) == NULL) return NULL;
	ret->rspLen = strlen("dummy code") + 1;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_extreq0(){
	LO_ExtReq *ret;

	if((ret=(LO_ExtReq*)LCMP_op_new(LCMP_OP_EXTREQ))==NULL) return NULL;
	if ((ret->opOID = strdup("1.2.333.444")) == NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_extreq1(){
	LO_ExtReq *ret;

	if((ret=(LO_ExtReq*)LCMP_op_new(LCMP_OP_EXTREQ))==NULL) return NULL;
	if ((ret->opOID = strdup("1.2.333.444")) == NULL) return NULL;
	if ((ret->opInfo = strdup("dummy code")) == NULL) return NULL;
	ret->ifLen = strlen("dummy code") + 1;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_extrsp0(){
	LO_ExtRsp *ret;

	if((ret=(LO_ExtRsp*)LCMP_op_new(LCMP_OP_EXTRSP))==NULL) return NULL;

	ret->resultCode = LCMP_OPERATION_ERR;
	if ((ret->resultMsg = strdup("bad operation code.")) == NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_extrsp1(){
	LO_ExtRsp *ret;

	if((ret=(LO_ExtRsp*)LCMP_op_new(LCMP_OP_EXTRSP))==NULL) return NULL;

	ret->resultCode = LCMP_SUCCESS;
	if ((ret->opOID = strdup("1.2.333.444")) == NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_extrsp2(){
	LO_ExtRsp *ret;

	if((ret=(LO_ExtRsp*)LCMP_op_new(LCMP_OP_EXTRSP))==NULL) return NULL;

	ret->resultCode = LCMP_SUCCESS;
	if ((ret->opOID = strdup("1.2.333.444")) == NULL) return NULL;
	if ((ret->opRsp = strdup("dummy code")) == NULL) return NULL;
	ret->rspLen = strlen("dummy code") + 1;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_renewalreq0(){
	LO_RenewalReq *ret;
	char *fp_TSREQ = get_fpath(PATH, TSREQ);

	if((ret=(LO_RenewalReq*)LCMP_op_new(LCMP_OP_RENEWALREQ))==NULL) return NULL;

	ret->serialNum = 5;
	if((ret->p10=Req_read_file(fp_TSREQ))==NULL) return NULL;
	free(fp_TSREQ);

	return (LCMPOp*)ret;
}

LCMPOp *set_test_renewalreq1(){
	LO_RenewalReq *ret;
	char *fp_TSREQ = get_fpath(PATH, TSREQ);

	if((ret=(LO_RenewalReq*)LCMP_op_new(LCMP_OP_RENEWALREQ))==NULL) return NULL;

	ret->serialNum = 10;
	if((ret->p10=Req_read_file(fp_TSREQ))==NULL) return NULL;
	ret->newSerialNum = 100;
	free(fp_TSREQ);

	return (LCMPOp*)ret;
}

LCMPOp *set_test_renewalreq2(){
	LO_RenewalReq *ret;
	Req *req = NULL;
	char *fp_TSREQ = get_fpath(PATH, TSREQ);

	if((ret=(LO_RenewalReq*)LCMP_op_new(LCMP_OP_RENEWALREQ))==NULL) return NULL;

	ret->serialNum = 15;
	if((ret->tmpl=CMP_certtmpl_new())==NULL) return NULL;
	if((req=Req_read_file(fp_TSREQ))==NULL) return NULL;
	if(Cert_dncopy(&req->subject_dn,&ret->tmpl->subject)) return NULL;
	ret->tmpl->publicKey = req->pubkey; req->pubkey = NULL;
	Req_free(req);
	ret->newSerialNum = 1000;
	free(fp_TSREQ);

	return (LCMPOp*)ret;
}

LCMPOp *set_test_renewalreq3(){
	LO_RenewalReq *ret;
	Cert *ct = NULL;
	CertExt *et,*hd;
	char *fp_TSCERT = get_fpath(PATH, TSCERT);

	if((ret=(LO_RenewalReq*)LCMP_op_new(LCMP_OP_RENEWALREQ))==NULL) return NULL;

	ret->serialNum = 20;
	if((ret->tmpl=CMP_certtmpl_new())==NULL) return NULL;
	if((ct=Cert_read_file(fp_TSCERT))==NULL) return NULL;
	if(Cert_dncopy(&ct->subject_dn,&ret->tmpl->subject)) return NULL;
	ret->tmpl->publicKey = ct->pubkey; ct->pubkey = NULL;
	ret->tmpl->serialNumber = ct->serialNumber;
	memcpy(&ret->tmpl->validity.notBefore,&ct->time.notBefore,sizeof(struct tm));
	memcpy(&ret->tmpl->validity.notAfter,&ct->time.notAfter,sizeof(struct tm));
	if((et=Extnew_basic_cons(1,10))==NULL) return NULL;
	ret->tmpl->ext=hd=et;
	if((et=Extnew_keyusage(0x7f))==NULL) return NULL;
	hd->next=et; hd=et;

	Cert_free(ct);
	free(fp_TSCERT);

	return (LCMPOp*)ret;
}

LCMPOp *set_test_renewalrsp0(){
	LO_RenewalRsp *ret;

	if((ret=(LO_RenewalRsp*)LCMP_op_new(LCMP_OP_RENEWALRSP))==NULL) return NULL;

	ret->resultCode = LCMP_OPERATION_ERR;
	if ((ret->resultMsg = strdup("bad pkcs#10 format.")) == NULL) return NULL;

	return (LCMPOp*)ret;
}

LCMPOp *set_test_renewalrsp1(){
	LO_RenewalRsp *ret;
	char *fp_TSCERT = get_fpath(PATH, TSCERT);

	if((ret=(LO_RenewalRsp*)LCMP_op_new(LCMP_OP_RENEWALRSP))==NULL) return NULL;
	if((ret->cert=Cert_read_file(fp_TSCERT))==NULL) return NULL;

	ret->resultCode = LCMP_SUCCESS;
	if ((ret->resultMsg = strdup("certificate returned")) == NULL) return NULL;
	free(fp_TSCERT);

	return (LCMPOp*)ret;
}

/*--------------------------------------
	check test LCMPOp content
--------------------------------------*/
int cmp_lcmp_ctrl(LCMPCtrl *ct1,LCMPCtrl *ct2){
	int i;
	
	while(ct1 && ct2){
		if((ct1->ctrlType==0)^(ct2->ctrlType==0)) goto error;
		if(ct1->ctrlType && ct2->ctrlType)
			if(strcmp(ct1->ctrlType,ct2->ctrlType)) goto error;
		
		if(ct1->crit != ct2->crit) goto error;
		if(ct1->vlen != ct2->vlen) goto error;
		if((ct1->value==0)^(ct2->value==0)) goto error;
		if(ct1->value && ct2->value){
			for(i=0; i<ct1->vlen; i++)
				if(ct1->value[i] != ct2->value[i]) goto error;
		}
		ct1 = ct1->next;
		ct2 = ct2->next;
	}
	if(ct1 || ct2) goto error;
	
	return 0;
error:
	return -1;
}

int cmp_lcmp_common(LCMPOp *op1, LCMPOp *op2){
	
	if(op1->version != op2->version) goto error;
	if(op1->messageId != op2->messageId) goto error;
	if(op1->opId != op2->opId) goto error;
	if(op1->resultCode != op2->resultCode) goto error;
	if((op1->resultMsg==NULL)||(op1->resultMsg[0]==0))
		if(!((op2->resultMsg==NULL)||(op2->resultMsg[0]==0))) goto error;
	if(op1->resultMsg && op2->resultMsg)
		if(strcmp(op1->resultMsg,op2->resultMsg)) goto error;
	if((op1->ctrl==0)^(op2->ctrl==0)) goto error;
	if(op1->ctrl && op2->ctrl)
		if(cmp_lcmp_ctrl(op1->ctrl,op2->ctrl)) goto error;
							
	return 0;
error:
	return -1;
}


int cmp_lcmp_prof(ProfTmpl *pf1, ProfTmpl *pf2){

	if((pf1->tmplName==0)^(pf2->tmplName==0)) goto error;
	if(pf1->tmplName && pf2->tmplName)
		if(strcmp(pf1->tmplName,pf2->tmplName)) goto error;

	if(pf1->certSec != pf2->certSec) goto error;
	if(pf1->certVer != pf2->certVer) goto error;
	if(pf1->baseNum != pf2->baseNum) goto error;
	if(pf1->keyType != pf2->keyType) goto error;
	if(pf1->keyLength != pf2->keyLength) goto error;
	if(pf1->hashType != pf2->hashType) goto error;
	if(stmcmp(&pf1->start,&pf2->start)) goto error;
	if(stmcmp(&pf1->end,&pf2->end)) goto error;
	if(pf1->updSec != pf2->updSec) goto error;
	if(pf1->bgtype != pf2->bgtype) goto error;
	if(pf1->edtype != pf2->edtype) goto error;
	if(pf1->updtype != pf2->updtype) goto error;
	if(memcmp(&pf1->policyFlag,&pf2->policyFlag,8)) goto error;
	if(Cert_dncmp(&pf1->sbjTmpl,&pf2->sbjTmpl)) goto error;
	
	return 0;
error:
	return -1;
}

int cmp_ext(CertExt *ex1,CertExt *ex2){
	/* int i,len; */
	
	while(ex1 && ex2){
		if(ex1->extnID != ex2->extnID) goto error;
		if(ex1->dlen != ex2->dlen) goto error;
		/*
		if((ex1->objid==0)^(ex2->objid==0)) goto error;
		if(ex1->objid && ex2->objid){
			ASN1_skip_(ex1->objid,&len);
			for(i=0; i<len+2; i++)
				if(ex1->objid[i] != ex2->objid[i]) goto error;
		}
		*/
		ex1 = ex1->next;
		ex2 = ex2->next;
	}
	if(ex1 || ex2) goto error;

	return 0;
error:
	return -1;
}

int cmp_lcmp_auth(AuthInfo *ai1, AuthInfo *ai2){

	if((ai1->name==0)^(ai2->name==0)) goto error;
	if(ai1->name && ai2->name)
		if(strcmp(ai1->name,ai2->name)) goto error;

	if((ai1->passwd==0)^(ai2->passwd==0)) goto error;
	if(ai1->passwd && ai2->passwd)
		if(strcmp(ai1->passwd,ai2->passwd)) goto error;

	if((ai1->oldpasswd==0)^(ai2->oldpasswd==0)) goto error;
	if(ai1->oldpasswd && ai2->oldpasswd)
		if(strcmp(ai1->oldpasswd,ai2->oldpasswd)) goto error;

	if(ai1->uid != ai2->uid) goto error;
	if(ai1->gid != ai2->gid) goto error;
	if(ai1->grant != ai2->grant) goto error;
	
	if((ai1->opt==0)^(ai2->opt==0)) goto error;
	if(ai1->opt && ai2->opt)
		if(strcmp(ai1->opt,ai2->opt)) goto error;

	return 0;
error:
	return -1;
}

/*------------------------------------------------------------*/
int cmp_test_bindreq(LCMPOp *op1, LCMPOp *op2){			
	LO_BindReq *lo1=(LO_BindReq*)op1,*lo2=(LO_BindReq*)op2;
	
	if(cmp_lcmp_common(op1,op2)) goto error;
	
	if(lo1->authMode != lo2->authMode) goto error;
	if((lo1->caname==0)^(lo2->caname==0)) goto error;
	if((lo1->username==0)^(lo2->username==0)) goto error;
	if((lo1->passwd==0)^(lo2->passwd==0)) goto error;
	
	if(lo1->caname && lo2->caname)
		if(strcmp(lo1->caname,lo2->caname)) goto error;
	if(lo1->username && lo2->username)
		if(strcmp(lo1->username,lo2->username)) goto error;	
	if(lo1->passwd && lo2->passwd)
		if(strcmp(lo1->passwd,lo2->passwd)) goto error;
	
	return 0;
error:
	return -1;
}

int cmp_test_bindrsp(LCMPOp *op1, LCMPOp *op2){
	LO_BindRsp *lo1=(LO_BindRsp*)op1,*lo2=(LO_BindRsp*)op2;
	CertList *ls1,*ls2;
	
	if(cmp_lcmp_common(op1,op2)) goto error;

	if((lo1->list==0)^(lo2->list==0)) goto error;
	ls1=lo1->list;
	ls2=lo2->list;
	while(ls1 && ls2){
		if(Cert_cmp(ls1->cert,ls2->cert)) goto error;
		ls1 = ls1->next;
		ls2 = ls2->next;
	}
	if(ls1 || ls2) goto error;
	
	return 0;
error:
	return -1;
}

int cmp_test_unbindreq(LCMPOp *op1, LCMPOp *op2){
	
	if(cmp_lcmp_common(op1,op2)) goto error;
		
	return 0;
error:
	return -1;
}

int cmp_test_signreq(LCMPOp *op1, LCMPOp *op2){
	LO_SignReq *lo1=(LO_SignReq*)op1,*lo2=(LO_SignReq*)op2;
	
	if(cmp_lcmp_common(op1,op2)) goto error;
		
	if(lo1->serialNum != lo2->serialNum) goto error;
	if((lo1->profName==0)^(lo2->profName==0)) goto error;
	if((lo1->p10==0)^(lo2->p10==0)) goto error;
	if((lo1->tmpl==0)^(lo2->tmpl==0)) goto error;

	if(lo1->profName && lo2->profName)
		if(strcmp(lo1->profName,lo2->profName)) goto error;
	if(lo1->p10 && lo2->p10)
		if(Req_cmp(lo1->p10,lo2->p10)) goto error;
	if(lo1->tmpl && lo2->tmpl){
		CertExt *e1,*e2;
		time_t t1,t2;

		if(lo1->tmpl->version != lo2->tmpl->version) return -1;
		if(lo1->tmpl->serialNumber != lo2->tmpl->serialNumber) return -1;
		if(lo1->tmpl->signingAlg != lo2->tmpl->signingAlg) return -1;
		if(Cert_dncmp(&lo1->tmpl->issuer,&lo2->tmpl->issuer)) return -1;
		if(Cert_dncmp(&lo1->tmpl->subject,&lo2->tmpl->subject)) return -1;
		t1 = mktime(&lo1->tmpl->validity.notBefore);
		t2 = mktime(&lo2->tmpl->validity.notBefore);
		if(t1 != t2) return -1;
		t1 = mktime(&lo1->tmpl->validity.notAfter);
		t2 = mktime(&lo2->tmpl->validity.notAfter);
		if(t1 != t2) return -1;
		if(Key_cmp(lo1->tmpl->publicKey,lo2->tmpl->publicKey)) return -1;

		e1 = lo1->tmpl->ext; e2 = lo2->tmpl->ext;
		while(e1 && e2){
			if(e1->extnID != e2->extnID) return -1;
			if(e1->critical != e2->critical) return -1;
			e1=e1->next; e2=e2->next;
		}
		if(e1 || e2) return -1;
	}

	return 0;
error:
	return -1;
}

int cmp_test_signrsp(LCMPOp *op1, LCMPOp *op2){
	LO_SignRsp *lo1=(LO_SignRsp*)op1,*lo2=(LO_SignRsp*)op2;
	
	if(cmp_lcmp_common(op1,op2)) goto error;

	if((lo1->cert==0)^(lo2->cert==0)) goto error;
	if(lo1->cert && lo2->cert)
		if(Cert_cmp(lo1->cert,lo2->cert)) goto error;
	
	return 0;
error:
	return -1;
}

int cmp_test_listreq(LCMPOp *op1, LCMPOp *op2){
	LO_ListReq *lo1=(LO_ListReq*)op1,*lo2=(LO_ListReq*)op2;
	
	if(cmp_lcmp_common(op1,op2)) goto error;

	if(lo1->serialNum != lo2->serialNum) goto error;
	
	if((lo1->profName==0)^(lo2->profName==0)) goto error;	
	if(lo1->profName && lo2->profName)
		if(strcmp(lo1->profName,lo2->profName)) goto error;
	
	return 0;
error:
	return -1;
}

int cmp_test_listrsp(LCMPOp *op1, LCMPOp *op2){
	LO_ListRsp *lo1=(LO_ListRsp*)op1,*lo2=(LO_ListRsp*)op2;
	CertStat *st1,*st2;
	
	if(cmp_lcmp_common(op1,op2)) goto error;
		
	if((lo1->stat==0)^(lo2->stat==0)) goto error;
	st1 = lo1->stat;
	st2 = lo2->stat;
	
	while(st1 && st2){
		if(st1->state != st2->state) goto error;
		if(st1->serialNum != st2->serialNum) goto error;
		if(stmcmp(&st1->notBefore,&st2->notBefore)) goto error;
		if(stmcmp(&st1->notAfter,&st2->notAfter)) goto error;
		if(stmcmp(&st1->revokedDate,&st2->revokedDate)) goto error;
		if((st1->subject==0)^(st2->subject==0)) goto error;
		if(st1->subject && st2->subject)
			if(strcmp(st1->subject,st2->subject)) goto error;
		if((st1->keyID==0)^(st2->keyID==0)) goto error;
		if(st1->keyID && st2->keyID)
			if(memcmp(st1->keyID,st2->keyID,20)) goto error;

		st1 = st1->next;
		st2 = st2->next;
	}
	if(st1 || st2) goto error;
	
	return 0;
error:
	return -1;
}

int cmp_test_profreq(LCMPOp *op1, LCMPOp *op2){
	LO_ProfReq *lo1=(LO_ProfReq*)op1,*lo2=(LO_ProfReq*)op2;
	
	if(cmp_lcmp_common(op1,op2)) goto error;

	if(lo1->profOp != lo2->profOp) goto error;
	if((lo1->profName==0)^(lo2->profName==0)) goto error;
	if((lo1->prof==0)^(lo2->prof==0)) goto error;
	if((lo1->ext==0)^(lo2->ext==0)) goto error;
	
	if(lo1->profName && lo2->profName)
		if(strcmp(lo1->profName,lo2->profName)) goto error;
	if(lo1->prof && lo2->prof)
		if(cmp_lcmp_prof(lo1->prof,lo2->prof)) goto error;
	if(lo1->ext && lo2->ext)
		if(cmp_ext(lo1->ext,lo2->ext)) goto error;
	
	return 0;
error:
	return -1;
}

int cmp_test_profrsp(LCMPOp *op1, LCMPOp *op2){
	LO_ProfRsp *lo1=(LO_ProfRsp*)op1,*lo2=(LO_ProfRsp*)op2;
	int i;

	if(cmp_lcmp_common(op1,op2)) goto error;
		
	if((lo1->prof==0)^(lo2->prof==0)) goto error;
	if((lo1->ext==0)^(lo2->ext==0)) goto error;

	if(lo1->prof && lo2->prof)
		if(cmp_lcmp_prof(lo1->prof,lo2->prof)) goto error;
	if(lo1->ext && lo2->ext)
		if(cmp_ext(lo1->ext,lo2->ext)) goto error;

	if(lo1->listnum != lo2->listnum) goto error;
	if((lo1->proflist==0)^(lo2->proflist==0)) goto error;
	if(lo1->proflist && lo2->proflist){
		for(i=0; i<lo1->listnum; i++){
			if((lo1->proflist[i]==0)^(lo2->proflist[i]==0)) goto error;
			if(lo1->proflist[i] && lo2->proflist[i])
				if(strcmp(lo1->proflist[i],lo2->proflist[i])) goto error;
		}
	}


	return 0;
error:
	return -1;
}

int cmp_test_certreq(LCMPOp *op1, LCMPOp *op2){
	LO_CertReq *lo1=(LO_CertReq*)op1,*lo2=(LO_CertReq*)op2;
	
	if(cmp_lcmp_common(op1,op2)) goto error;

	if(lo1->certOp != lo2->certOp) goto error;
	if(lo1->serialNum != lo2->serialNum) goto error;
	if(lo1->revokeReason != lo2->revokeReason) goto error;

	if((lo1->passwd==0)^(lo2->passwd==0)) goto error;
	if(lo1->passwd && lo2->passwd)
		if(strcmp(lo1->passwd,lo2->passwd)) goto error;

	if((lo1->key==0)^(lo2->key==0)) goto error;
	if(lo1->key && lo2->key)
		if(Key_cmp(lo1->key,lo2->key)) goto error;

	return 0;
error:
	return -1;
}

int cmp_test_certrsp(LCMPOp *op1, LCMPOp *op2){
	LO_CertRsp *lo1=(LO_CertRsp*)op1,*lo2=(LO_CertRsp*)op2;
	
	if(cmp_lcmp_common(op1,op2)) goto error;
		
	if((lo1->cert==0)^(lo2->cert==0)) goto error;
	if(lo1->cert && lo2->cert)
		if(Cert_cmp(lo1->cert,lo2->cert)) goto error;

	if((lo1->key==0)^(lo2->key==0)) goto error;
	if(lo1->key && lo2->key)
		if(Key_cmp(lo1->key,lo2->key)) goto error;

	return 0;
error:
	return -1;
}

int cmp_test_csrreq(LCMPOp *op1, LCMPOp *op2){
	LO_CSRReq *lo1=(LO_CSRReq*)op1,*lo2=(LO_CSRReq*)op2;
	
	if(cmp_lcmp_common(op1,op2)) goto error;

	if(lo1->csrOp != lo2->csrOp) goto error;
	if(lo1->acceptID != lo2->acceptID) goto error;
	if(lo1->serialNum != lo2->serialNum) goto error;
	if(lo1->rejectReason != lo2->rejectReason) goto error;
	
	if((lo1->profName==0)^(lo2->profName==0)) goto error;	
	if(lo1->profName && lo2->profName)
		if(strcmp(lo1->profName,lo2->profName)) goto error;

	if((lo1->csr==0)^(lo2->csr==0)) goto error;
	if(lo1->csr && lo2->csr)
		if(Req_cmp(lo1->csr,lo2->csr)) goto error;

	if(lo1->tmpl && lo2->tmpl){
		CertExt *e1,*e2;
		time_t t1,t2;

		if(lo1->tmpl->version != lo2->tmpl->version) return -1;
		if(lo1->tmpl->serialNumber != lo2->tmpl->serialNumber) return -1;
		if(lo1->tmpl->signingAlg != lo2->tmpl->signingAlg) return -1;
		if(Cert_dncmp(&lo1->tmpl->issuer,&lo2->tmpl->issuer)) return -1;
		if(Cert_dncmp(&lo1->tmpl->subject,&lo2->tmpl->subject)) return -1;
		t1 = mktime(&lo1->tmpl->validity.notBefore);
		t2 = mktime(&lo2->tmpl->validity.notBefore);
		if(t1 != t2) return -1;
		t1 = mktime(&lo1->tmpl->validity.notAfter);
		t2 = mktime(&lo2->tmpl->validity.notAfter);
		if(t1 != t2) return -1;
		if(Key_cmp(lo1->tmpl->publicKey,lo2->tmpl->publicKey)) return -1;

		e1 = lo1->tmpl->ext; e2 = lo2->tmpl->ext;
		while(e1 && e2){
			if(e1->extnID != e2->extnID) return -1;
			if(e1->critical != e2->critical) return -1;
			e1=e1->next; e2=e2->next;
		}
		if(e1 || e2) return -1;
	}

	return 0;
error:
	return -1;
}

int cmp_test_csrrsp(LCMPOp *op1, LCMPOp *op2){
	LO_CSRRsp *lo1=(LO_CSRRsp*)op1,*lo2=(LO_CSRRsp*)op2;
	
	if(cmp_lcmp_common(op1,op2)) goto error;
		
	if(lo1->acceptID != lo2->acceptID) goto error;
	if(lo1->serialNum != lo2->serialNum) goto error;
	
	if((lo1->csr==0)^(lo2->csr==0)) goto error;
	if(lo1->csr && lo2->csr)
		if(Req_cmp(lo1->csr,lo2->csr)) goto error;

	if(lo1->tmpl && lo2->tmpl){
		CertExt *e1,*e2;
		time_t t1,t2;

		if(lo1->tmpl->version != lo2->tmpl->version) return -1;
		if(lo1->tmpl->serialNumber != lo2->tmpl->serialNumber) return -1;
		if(lo1->tmpl->signingAlg != lo2->tmpl->signingAlg) return -1;
		if(Cert_dncmp(&lo1->tmpl->issuer,&lo2->tmpl->issuer)) return -1;
		if(Cert_dncmp(&lo1->tmpl->subject,&lo2->tmpl->subject)) return -1;
		t1 = mktime(&lo1->tmpl->validity.notBefore);
		t2 = mktime(&lo2->tmpl->validity.notBefore);
		if(t1 != t2) return -1;
		t1 = mktime(&lo1->tmpl->validity.notAfter);
		t2 = mktime(&lo2->tmpl->validity.notAfter);
		if(t1 != t2) return -1;
		if(Key_cmp(lo1->tmpl->publicKey,lo2->tmpl->publicKey)) return -1;

		e1 = lo1->tmpl->ext; e2 = lo2->tmpl->ext;
		while(e1 && e2){
			if(e1->extnID != e2->extnID) return -1;
			if(e1->critical != e2->critical) return -1;
			e1=e1->next; e2=e2->next;
		}
		if(e1 || e2) return -1;
	}
	return 0;
error:
	return -1;
}

int cmp_test_crlreq(LCMPOp *op1, LCMPOp *op2){
	LO_CRLReq *lo1=(LO_CRLReq*)op1,*lo2=(LO_CRLReq*)op2;
	
	if(cmp_lcmp_common(op1,op2)) goto error;
		
	if((lo1->profName==0)^(lo2->profName==0)) goto error;
	if((lo1->path==0)^(lo2->path==0)) goto error;
	if(lo1->crlOp != lo2->crlOp) goto error;
	if(lo1->location != lo2->location) goto error;
	if(lo1->crlNum != lo2->crlNum) goto error;
	
	if(lo1->profName && lo2->profName)
		if(strcmp(lo1->profName,lo2->profName)) goto error;
	if(lo1->path && lo2->path)
		if(strcmp(lo1->path,lo2->path)) goto error;
	
	return 0;
error:
	return -1;
}

int cmp_test_crlrsp(LCMPOp *op1, LCMPOp *op2){
	LO_CRLRsp *lo1=(LO_CRLRsp*)op1,*lo2=(LO_CRLRsp*)op2;
	CRLList *ls1,*ls2;
	
	if(cmp_lcmp_common(op1,op2)) goto error;
		
	if((lo1->list==0)^(lo2->list==0)) goto error;
	ls1=lo1->list;
	ls2=lo2->list;
	while(ls1 && ls2){
		if(CRL_cmp(ls1->crl,ls2->crl)) goto error;
		ls1 = ls1->next;
		ls2 = ls2->next;
	}
	if(ls1 || ls2) goto error;

	return 0;
error:
	return -1;
}

int cmp_test_svopreq(LCMPOp *op1, LCMPOp *op2){
	LO_SVOpReq *lo1=(LO_SVOpReq*)op1,*lo2=(LO_SVOpReq*)op2;
	
	if(cmp_lcmp_common(op1,op2)) goto error;

	if(lo1->svOp != lo2->svOp) goto error;
	if((lo1->uinfo==0)^(lo2->uinfo==0)) goto error;
	if(lo1->uinfo && lo2->uinfo)
		if(cmp_lcmp_auth(lo1->uinfo,lo2->uinfo)) goto error;

	return 0;
error:
	return -1;
}

int cmp_test_svoprsp(LCMPOp *op1, LCMPOp *op2){
	LO_SVOpRsp *lo1=(LO_SVOpRsp*)op1,*lo2=(LO_SVOpRsp*)op2;
	int i;
	
	if(cmp_lcmp_common(op1,op2)) goto error;
	
	if(lo1->rspLen != lo2->rspLen) goto error;
	if((lo1->opRsp==0)^(lo2->opRsp==0)) goto error;
	
	if(lo1->opRsp && lo2->opRsp){
		for(i=0; i < lo1->rspLen; i++)
			if(lo1->opRsp[i] != lo2->opRsp[i]) goto error;
	}
		
	return 0;
error:
	return -1;
}

int cmp_test_extreq(LCMPOp *op1, LCMPOp *op2){
	LO_ExtReq *lo1=(LO_ExtReq*)op1,*lo2=(LO_ExtReq*)op2;
	int i;
	
	if(cmp_lcmp_common(op1,op2)) goto error;
		
	if(lo1->ifLen != lo2->ifLen) goto error;
	if((lo1->opInfo==0)^(lo2->opInfo==0)) goto error;
	if((lo1->opOID==0)^(lo2->opOID==0)) goto error;
	if(lo1->opOID && lo2->opOID)
		if(strcmp(lo1->opOID,lo2->opOID)) goto error;
	
	if(lo1->opInfo && lo2->opInfo){
		for(i=0; i < lo1->ifLen; i++)
			if(lo1->opInfo[i] != lo2->opInfo[i]) goto error;
	}
	return 0;
error:
	return -1;
}

int cmp_test_extrsp(LCMPOp *op1, LCMPOp *op2){
	LO_ExtRsp *lo1=(LO_ExtRsp*)op1,*lo2=(LO_ExtRsp*)op2;
	int i;
	
	if(cmp_lcmp_common(op1,op2)) goto error;
	
	if(lo1->rspLen != lo2->rspLen) goto error;
	if((lo1->opRsp==0)^(lo2->opRsp==0)) goto error;
	if((lo1->opOID==0)^(lo2->opOID==0)) goto error;
	if(lo1->opOID && lo2->opOID)
		if(strcmp(lo1->opOID,lo2->opOID)) goto error;
	
	if(lo1->opRsp && lo2->opRsp){
		for(i=0; i < lo1->rspLen; i++)
			if(lo1->opRsp[i] != lo2->opRsp[i]) goto error;
	}		
		
	return 0;
error:
	return -1;
}

int cmp_test_renewalreq(LCMPOp *op1, LCMPOp *op2){
	LO_RenewalReq *lo1=(LO_RenewalReq*)op1,*lo2=(LO_RenewalReq*)op2;
	
	if(cmp_lcmp_common(op1,op2)) goto error;
		
	if(lo1->serialNum != lo2->serialNum) goto error;
	if((lo1->p10==0)^(lo2->p10==0)) goto error;
	if((lo1->tmpl==0)^(lo2->tmpl==0)) goto error;
	if(lo1->newSerialNum != lo2->newSerialNum) goto error;

	if(lo1->p10 && lo2->p10)
		if(Req_cmp(lo1->p10,lo2->p10)) goto error;
	if(lo1->tmpl && lo2->tmpl){
		CertExt *e1,*e2;
		time_t t1,t2;

		if(lo1->tmpl->version != lo2->tmpl->version) return -1;
		if(lo1->tmpl->serialNumber != lo2->tmpl->serialNumber) return -1;
		if(lo1->tmpl->signingAlg != lo2->tmpl->signingAlg) return -1;
		if(Cert_dncmp(&lo1->tmpl->issuer,&lo2->tmpl->issuer)) return -1;
		if(Cert_dncmp(&lo1->tmpl->subject,&lo2->tmpl->subject)) return -1;
		t1 = mktime(&lo1->tmpl->validity.notBefore);
		t2 = mktime(&lo2->tmpl->validity.notBefore);
		if(t1 != t2) return -1;
		t1 = mktime(&lo1->tmpl->validity.notAfter);
		t2 = mktime(&lo2->tmpl->validity.notAfter);
		if(t1 != t2) return -1;
		if(Key_cmp(lo1->tmpl->publicKey,lo2->tmpl->publicKey)) return -1;

		e1 = lo1->tmpl->ext; e2 = lo2->tmpl->ext;
		while(e1 && e2){
			if(e1->extnID != e2->extnID) return -1;
			if(e1->critical != e2->critical) return -1;
			e1=e1->next; e2=e2->next;
		}
		if(e1 || e2) return -1;
	}

	return 0;
error:
	return -1;
}

int cmp_test_renewalrsp(LCMPOp *op1, LCMPOp *op2){
	LO_RenewalRsp *lo1=(LO_RenewalRsp*)op1,*lo2=(LO_RenewalRsp*)op2;
	
	if(cmp_lcmp_common(op1,op2)) goto error;

	if((lo1->cert==0)^(lo2->cert==0)) goto error;
	if(lo1->cert && lo2->cert)
		if(Cert_cmp(lo1->cert,lo2->cert)) goto error;
	
	return 0;
error:
	return -1;
}

/*--------------------------------------
	test main
--------------------------------------*/
int test_lcmp(){
	struct test_type{
		char *testname;
		LCMPOp *(*get_cb)();
		int (*check_cb)(LCMPOp *op1,LCMPOp *op2);
	} tsset[] = {
		{"BindReq Test 0",(LCMPOp*(*)())set_test_bindreq0,(int(*)(LCMPOp*,LCMPOp*))cmp_test_bindreq},
		{"BindReq Test 1",(LCMPOp*(*)())set_test_bindreq1,(int(*)(LCMPOp*,LCMPOp*))cmp_test_bindreq},
		{"BindRsp Test 0",(LCMPOp*(*)())set_test_bindrsp0,(int(*)(LCMPOp*,LCMPOp*))cmp_test_bindrsp},
		{"BindRsp Test 1",(LCMPOp*(*)())set_test_bindrsp1,(int(*)(LCMPOp*,LCMPOp*))cmp_test_bindrsp},
		{"UnBindReq Test 0",(LCMPOp*(*)())set_test_unbindreq0,(int(*)(LCMPOp*,LCMPOp*))cmp_test_unbindreq},
		{"SignReq Test 0",(LCMPOp*(*)())set_test_signreq0,(int(*)(LCMPOp*,LCMPOp*))cmp_test_signreq},
		{"SignReq Test 1",(LCMPOp*(*)())set_test_signreq1,(int(*)(LCMPOp*,LCMPOp*))cmp_test_signreq},
		{"SignReq Test 2",(LCMPOp*(*)())set_test_signreq2,(int(*)(LCMPOp*,LCMPOp*))cmp_test_signreq},
		{"SignReq Test 3",(LCMPOp*(*)())set_test_signreq3,(int(*)(LCMPOp*,LCMPOp*))cmp_test_signreq},
		{"SignRsp Test 0",(LCMPOp*(*)())set_test_signrsp0,(int(*)(LCMPOp*,LCMPOp*))cmp_test_signrsp},
		{"SignRsp Test 1",(LCMPOp*(*)())set_test_signrsp1,(int(*)(LCMPOp*,LCMPOp*))cmp_test_signrsp},
		{"ListReq Test 0",(LCMPOp*(*)())set_test_listreq0,(int(*)(LCMPOp*,LCMPOp*))cmp_test_listreq},
		{"ListReq Test 1",(LCMPOp*(*)())set_test_listreq1,(int(*)(LCMPOp*,LCMPOp*))cmp_test_listreq},
		{"ListReq Test 2",(LCMPOp*(*)())set_test_listreq2,(int(*)(LCMPOp*,LCMPOp*))cmp_test_listreq},
		{"ListRsp Test 0",(LCMPOp*(*)())set_test_listrsp0,(int(*)(LCMPOp*,LCMPOp*))cmp_test_listrsp},
		{"ListRsp Test 1",(LCMPOp*(*)())set_test_listrsp1,(int(*)(LCMPOp*,LCMPOp*))cmp_test_listrsp},
		{"ProfReq Test 0",(LCMPOp*(*)())set_test_profreq0,(int(*)(LCMPOp*,LCMPOp*))cmp_test_profreq},
		{"ProfReq Test 1_0",(LCMPOp*(*)())set_test_profreq1_0,(int(*)(LCMPOp*,LCMPOp*))cmp_test_profreq},
		{"ProfReq Test 1_1",(LCMPOp*(*)())set_test_profreq1_1,(int(*)(LCMPOp*,LCMPOp*))cmp_test_profreq},
		{"ProfReq Test 2",(LCMPOp*(*)())set_test_profreq2,(int(*)(LCMPOp*,LCMPOp*))cmp_test_profreq},
		{"ProfReq Test 3_0",(LCMPOp*(*)())set_test_profreq3_0,(int(*)(LCMPOp*,LCMPOp*))cmp_test_profreq},
		{"ProfReq Test 3_1",(LCMPOp*(*)())set_test_profreq3_1,(int(*)(LCMPOp*,LCMPOp*))cmp_test_profreq},
		{"ProfReq Test 3_2",(LCMPOp*(*)())set_test_profreq3_2,(int(*)(LCMPOp*,LCMPOp*))cmp_test_profreq},
		{"ProfReq Test 3_3",(LCMPOp*(*)())set_test_profreq3_3,(int(*)(LCMPOp*,LCMPOp*))cmp_test_profreq},
		{"ProfReq Test 4",(LCMPOp*(*)())set_test_profreq4,(int(*)(LCMPOp*,LCMPOp*))cmp_test_profreq},
		{"ProfReq Test 5",(LCMPOp*(*)())set_test_profreq5,(int(*)(LCMPOp*,LCMPOp*))cmp_test_profreq},
		{"ProfReq Test 6",(LCMPOp*(*)())set_test_profreq6,(int(*)(LCMPOp*,LCMPOp*))cmp_test_profreq},
		{"ProfReq Test 7",(LCMPOp*(*)())set_test_profreq7,(int(*)(LCMPOp*,LCMPOp*))cmp_test_profreq},
		{"ProfRsp Test 0",(LCMPOp*(*)())set_test_profrsp0,(int(*)(LCMPOp*,LCMPOp*))cmp_test_profrsp},
		{"ProfRsp Test 1",(LCMPOp*(*)())set_test_profrsp1,(int(*)(LCMPOp*,LCMPOp*))cmp_test_profrsp},
		{"ProfRsp Test 2",(LCMPOp*(*)())set_test_profrsp2,(int(*)(LCMPOp*,LCMPOp*))cmp_test_profrsp},
		{"ProfRsp Test 3",(LCMPOp*(*)())set_test_profrsp3,(int(*)(LCMPOp*,LCMPOp*))cmp_test_profrsp},
		{"ProfRsp Test 4",(LCMPOp*(*)())set_test_profrsp4,(int(*)(LCMPOp*,LCMPOp*))cmp_test_profrsp},
		{"CertReq Test 0",(LCMPOp*(*)())set_test_certreq0,(int(*)(LCMPOp*,LCMPOp*))cmp_test_certreq},
		{"CertReq Test 1",(LCMPOp*(*)())set_test_certreq1,(int(*)(LCMPOp*,LCMPOp*))cmp_test_certreq},
		{"CertReq Test 2",(LCMPOp*(*)())set_test_certreq2,(int(*)(LCMPOp*,LCMPOp*))cmp_test_certreq},
		{"CertReq Test 3",(LCMPOp*(*)())set_test_certreq3,(int(*)(LCMPOp*,LCMPOp*))cmp_test_certreq},
		{"CertRsp Test 0",(LCMPOp*(*)())set_test_certrsp0,(int(*)(LCMPOp*,LCMPOp*))cmp_test_certrsp},
		{"CertRsp Test 1",(LCMPOp*(*)())set_test_certrsp1,(int(*)(LCMPOp*,LCMPOp*))cmp_test_certrsp},
		{"CertRsp Test 2",(LCMPOp*(*)())set_test_certrsp2,(int(*)(LCMPOp*,LCMPOp*))cmp_test_certrsp},
		{"CSRReq Test 0",(LCMPOp*(*)())set_test_csrreq0,(int(*)(LCMPOp*,LCMPOp*))cmp_test_csrreq},
		{"CSRReq Test 0_1",(LCMPOp*(*)())set_test_csrreq0_1,(int(*)(LCMPOp*,LCMPOp*))cmp_test_csrreq},
		{"CSRReq Test 1",(LCMPOp*(*)())set_test_csrreq1,(int(*)(LCMPOp*,LCMPOp*))cmp_test_csrreq},
		{"CSRReq Test 2",(LCMPOp*(*)())set_test_csrreq2,(int(*)(LCMPOp*,LCMPOp*))cmp_test_csrreq},
		{"CSRReq Test 3",(LCMPOp*(*)())set_test_csrreq3,(int(*)(LCMPOp*,LCMPOp*))cmp_test_csrreq},
		{"CSRRsp Test 0",(LCMPOp*(*)())set_test_csrrsp0,(int(*)(LCMPOp*,LCMPOp*))cmp_test_csrrsp},
		{"CSRRsp Test 1",(LCMPOp*(*)())set_test_csrrsp1,(int(*)(LCMPOp*,LCMPOp*))cmp_test_csrrsp},
		{"CSRRsp Test 1_1",(LCMPOp*(*)())set_test_csrrsp1_1,(int(*)(LCMPOp*,LCMPOp*))cmp_test_csrrsp},
		{"CSRRsp Test 2",(LCMPOp*(*)())set_test_csrrsp2,(int(*)(LCMPOp*,LCMPOp*))cmp_test_csrrsp},
		{"CRLReq Test 0",(LCMPOp*(*)())set_test_crlreq0,(int(*)(LCMPOp*,LCMPOp*))cmp_test_crlreq},
		{"CRLReq Test 1",(LCMPOp*(*)())set_test_crlreq1,(int(*)(LCMPOp*,LCMPOp*))cmp_test_crlreq},
		{"CRLReq Test 2",(LCMPOp*(*)())set_test_crlreq2,(int(*)(LCMPOp*,LCMPOp*))cmp_test_crlreq},
		{"CRLRsp Test 0",(LCMPOp*(*)())set_test_crlrsp0,(int(*)(LCMPOp*,LCMPOp*))cmp_test_crlrsp},
		{"CRLRsp Test 1",(LCMPOp*(*)())set_test_crlrsp1,(int(*)(LCMPOp*,LCMPOp*))cmp_test_crlrsp},
		{"SVOpReq Test 0",(LCMPOp*(*)())set_test_svopreq0,(int(*)(LCMPOp*,LCMPOp*))cmp_test_svopreq},
		{"SVOpReq Test 1",(LCMPOp*(*)())set_test_svopreq1,(int(*)(LCMPOp*,LCMPOp*))cmp_test_svopreq},
		{"SVOpReq Test 2",(LCMPOp*(*)())set_test_svopreq2,(int(*)(LCMPOp*,LCMPOp*))cmp_test_svopreq},
		{"SVOpReq Test 3",(LCMPOp*(*)())set_test_svopreq3,(int(*)(LCMPOp*,LCMPOp*))cmp_test_svopreq},
		{"SVOpReq Test 4",(LCMPOp*(*)())set_test_svopreq4,(int(*)(LCMPOp*,LCMPOp*))cmp_test_svopreq},
		{"SVOpRsp Test 0",(LCMPOp*(*)())set_test_svoprsp0,(int(*)(LCMPOp*,LCMPOp*))cmp_test_svoprsp},
		{"SVOpRsp Test 1",(LCMPOp*(*)())set_test_svoprsp1,(int(*)(LCMPOp*,LCMPOp*))cmp_test_svoprsp},
		{"ExtReq Test 0",(LCMPOp*(*)())set_test_extreq0,(int(*)(LCMPOp*,LCMPOp*))cmp_test_extreq},
		{"ExtReq Test 1",(LCMPOp*(*)())set_test_extreq1,(int(*)(LCMPOp*,LCMPOp*))cmp_test_extreq},
		{"ExtRsp Test 0",(LCMPOp*(*)())set_test_extrsp0,(int(*)(LCMPOp*,LCMPOp*))cmp_test_extrsp},
		{"ExtRsp Test 1",(LCMPOp*(*)())set_test_extrsp1,(int(*)(LCMPOp*,LCMPOp*))cmp_test_extrsp},
		{"ExtRsp Test 2",(LCMPOp*(*)())set_test_extrsp2,(int(*)(LCMPOp*,LCMPOp*))cmp_test_extrsp},
		{"RenewalReq Test 0",(LCMPOp*(*)())set_test_renewalreq0,(int(*)(LCMPOp*,LCMPOp*))cmp_test_renewalreq},
		{"RenewalReq Test 1",(LCMPOp*(*)())set_test_renewalreq1,(int(*)(LCMPOp*,LCMPOp*))cmp_test_renewalreq},
		{"RenewalReq Test 2",(LCMPOp*(*)())set_test_renewalreq2,(int(*)(LCMPOp*,LCMPOp*))cmp_test_renewalreq},
		{"RenewalReq Test 3",(LCMPOp*(*)())set_test_renewalreq3,(int(*)(LCMPOp*,LCMPOp*))cmp_test_renewalreq},
		{"RenewalRsp Test 0",(LCMPOp*(*)())set_test_renewalrsp0,(int(*)(LCMPOp*,LCMPOp*))cmp_test_renewalrsp},
		{"RenewalRsp Test 1",(LCMPOp*(*)())set_test_renewalrsp1,(int(*)(LCMPOp*,LCMPOp*))cmp_test_renewalrsp},
		{"",0,0}
	};
	unsigned char *der;
	LCMPOp *op1,*op2;
	int i,j;
	
	for(i=0;tsset[i].get_cb!=0;i++){
		printf("LCMP %s : read & write DER ",tsset[i].testname);
		
		if(tsset[i].get_cb){
			if((op1=((LCMPOp*(*)())tsset[i].get_cb)())==NULL){
				printf("\nsystem error! : cannot continue test.\n%s\n",CA_get_errstr());
				goto error;
			}
			
			if((der=LCMP_op_toDER(op1,NULL,&j))==NULL){
				printf("\nDER encoding error! : cannot continue test.\n%s\n",
					CA_get_errstr());
				goto error;
			}
			
			if((op2=ASN1_LCMP_msg(der))==NULL){
				printf("\nDER decoding error! : cannot continue test.\n%s\n",
					CA_get_errstr());
				goto error;
			}

			if(((int(*)(LCMPOp*,LCMPOp*))tsset[i].check_cb)(op1,op2)){
				printf("\ncompare test error! (op1 != op2) : stop.\n");
				goto error;
			}
			
			if(der){ free(der); der=NULL; }
			LCMP_op_free(op1); op1=NULL;
			LCMP_op_free(op2); op2=NULL;
		}
		printf("-- ok\n");
	}

	return 0;
error:
	if(der){ free(der); der=NULL; }
	LCMP_op_free(op1); op1=NULL;
	LCMP_op_free(op2); op2=NULL;
	return -1;
}

