/* lcmp_tool.c */
/*
 * Copyright (c) 2004-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>

#include "ok_caerr.h"
#include "ok_ca.h"
#include "ok_lcmp.h"

/*-------------------------------------------------
  get allocated LCMP response
-------------------------------------------------*/
LCMPOp *LCMP_get_response(int opid, int rsp, char *msg){
	LCMPOp *ret=NULL;
	
	if((ret=LCMP_op_new(opid))==NULL) goto error;
	
	if(msg==NULL) msg = " "; /* set dummy */
	if ((ret->resultMsg = strdup(msg)) == NULL) goto error;
	ret->resultCode = rsp;
	
	return ret;
error:
	if(ret) OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMPTOOL,NULL);
	LCMP_op_free((LCMPOp*)ret);
	return NULL;
}

/*-------------------------------------------------
  get allocated LCMP operation
-------------------------------------------------*/
LCMPOp *LCMP_get_bindreq(char *ca,char *user,char *pwd,int auth){
	LO_BindReq *ret=NULL;

	if((ret=(LO_BindReq*)LCMP_op_new(LCMP_OP_BINDREQ))==NULL) goto error;
	if(ca)
	  if ((ret->caname = strdup(ca)) == NULL) goto error;
	if(user)
	  if ((ret->username = strdup(user)) == NULL) goto error;
	if(pwd)
	  if ((ret->passwd = strdup(pwd)) == NULL) goto error;
	ret->authMode = auth;
	
	return (LCMPOp*)ret;
error:
	if(ret) OK_set_error(ERR_ST_STRDUP,ERR_LC_LCMP,ERR_PT_LCMPTOOL+1,NULL);
	LCMP_op_free((LCMPOp*)ret);
	return NULL;
}

LCMPOp *LCMP_get_bindrsp(PKCS12 *p12){
	LO_BindRsp *ret=NULL;
	P12_Baggage *bg;
	CertList *cl,*hd=NULL;

	if((ret=(LO_BindRsp*)LCMP_get_response(LCMP_OP_BINDRSP,LCMP_SUCCESS,"bind ok"))==NULL)
	  goto error;
	
	for(bg=p12->bag; bg ;bg=bg->next){
	  if(bg->type==OBJ_P12v1Bag_CERT){
	    if((cl=Cert_2Certlist(((P12_CertBag*)bg)->cert))==NULL) goto error;
	    if(ret->list){
	      hd->next=cl; hd=cl;
	    }else{
	      ret->list=hd=cl;
	    }
	  }
	}
	
	return (LCMPOp*)ret;
error:
	LCMP_op_free((LCMPOp*)ret);
	return NULL;
}

/*-------------------------------------------------*/
LCMPOp *LCMP_get_signreq(char *profname,enum csrform csrFormat,void *req,int serialNum){
	LO_SignReq *ret=NULL;

	if((ret=(LO_SignReq*)LCMP_op_new(LCMP_OP_SIGNREQ))==NULL) goto error;

	if ((ret->profName = strdup(profname)) == NULL){
	  OK_set_error(ERR_ST_STRDUP,ERR_LC_LCMP,ERR_PT_LCMPTOOL+3,NULL);
	  goto error;
	}
	switch(csrFormat){
	case PKCS10CSR: /* pkcs 10 */
		if((ret->p10=Req_dup((Req*)req))==NULL) goto error;
		break;
	case CMPREQUEST: /* cert template */
		if((ret->tmpl=CMP_certtmpl_dup((CertTemplate*)req))==NULL) goto error;
		break;
	}
	
	ret->serialNum = (serialNum>0)?(serialNum):(0);

	return (LCMPOp*)ret;
error:
	LCMP_op_free((LCMPOp*)ret);
	return NULL;
}

LCMPOp *LCMP_get_signrsp(Cert *ct){
	LO_SignRsp *ret=NULL;

	if((ret=(LO_SignRsp*)LCMP_get_response(LCMP_OP_SIGNRSP,LCMP_SUCCESS,"sign ok"))==NULL)
	  goto error;

	if((ret->cert=Cert_dup(ct))==NULL) goto error;
	
	return (LCMPOp*)ret;
error:
	LCMP_op_free((LCMPOp*)ret);
	return NULL;
}

/*-------------------------------------------------*/
LCMPOp *LCMP_get_listreq_(char *profname,int serialNum,char *sbjQuery){
	LO_ListReq *ret=NULL;

	if((ret=(LO_ListReq*)LCMP_op_new(LCMP_OP_LISTREQ))==NULL) goto error;

	if ((ret->profName = strdup(profname)) == NULL){
		OK_set_error(ERR_ST_STRDUP,ERR_LC_LCMP,ERR_PT_LCMPTOOL+3,NULL);
		goto error;
	}
	ret->serialNum = (serialNum>0)?(serialNum):(0);
	if(sbjQuery){
		if ((ret->sbjQuery = strdup(sbjQuery)) == NULL){
			OK_set_error(ERR_ST_STRDUP,ERR_LC_LCMP,ERR_PT_LCMPTOOL+3,NULL);
			goto error;
		}
	}
	
	return (LCMPOp*)ret;
error:
	LCMP_op_free((LCMPOp*)ret);
	return NULL;
}
	
LCMPOp *LCMP_get_listreq(char *profname,int serialNum){
	return LCMP_get_listreq_(profname,serialNum,NULL);
}

LCMPOp *LCMP_get_listrsp(CertStat *cs,int serialNum){
	LO_ListRsp *ret=NULL;

	if((ret=(LO_ListRsp*)LCMP_get_response(LCMP_OP_LISTRSP,LCMP_SUCCESS,"list ok"))==NULL)
	  goto error;

	if(cs){
	  if(serialNum>0){
	    if((ret->stat=CertStat_dup(cs))==NULL) goto error;
	  }else{
	    if((ret->stat=CertStat_dup_all(cs))==NULL) goto error;
	  }
	}
	return (LCMPOp*)ret;
error:
	LCMP_op_free((LCMPOp*)ret);
	return NULL;
}

/*-------------------------------------------------*/
LCMPOp *LCMP_get_profreq_(int pfOp, char *profname, ProfTmpl *tmpl, CertExt *ext, char *rename){
	LO_ProfReq *ret=NULL;

	if((ret=(LO_ProfReq*)LCMP_op_new(LCMP_OP_PROFREQ))==NULL) goto error;
	ret->profOp = pfOp;

	if ((ret->profName = strdup(profname)) == NULL){
		OK_set_error(ERR_ST_STRDUP,ERR_LC_LCMP,ERR_PT_LCMPTOOL+4,NULL);
		goto error;
	}
	if(tmpl)
		if((ret->prof=ProfTmpl_dup(tmpl))==NULL) goto error;
	if(ext)
		if((ret->ext=CertExt_dup_all(ext))==NULL) goto error;
	if(rename)
		if ((ret->profRename = strdup(rename)) == NULL){
			OK_set_error(ERR_ST_STRDUP,ERR_LC_LCMP,ERR_PT_LCMPTOOL+4,NULL);
			goto error;
		}
		
	return (LCMPOp*)ret;
error:
	LCMP_op_free((LCMPOp*)ret);
	return NULL;
}

LCMPOp *LCMP_get_profreq(int pfOp, char *profname, ProfTmpl *tmpl, CertExt *ext){
	return LCMP_get_profreq_(pfOp,profname,tmpl,ext,NULL);
}

LCMPOp *LCMP_get_profrsp(LCMP *lc,ProfTmpl *tmpl, CertExt *ext, CertProf *cpf){
	LO_ProfRsp *ret=NULL;
	CertProf *top=cpf;
	int i,num=0;

	if((ret=(LO_ProfRsp*)LCMP_get_response(LCMP_OP_PROFRSP,LCMP_SUCCESS,"prof ok"))==NULL)
	  goto error;

	if(tmpl)
		if((ret->prof=ProfTmpl_dup(tmpl))==NULL) goto error;
	if(ext)
		if((ret->ext=CertExt_dup_all(ext))==NULL) goto error;

	while(cpf){ cpf=cpf->next; num++; }
	if((cpf=top) != 0){
		if((ret->proflist=(char**)malloc(sizeof(char*)*num))==NULL) goto error;
		memset(ret->proflist,0,sizeof(char*)*num);

		for(i=0; cpf ; cpf=cpf->next,i++){
			if(lc->gid &&(cpf->gid!=1)&&(cpf->gid!=lc->gid)){ i--; continue; }
			if ((ret->proflist[i] = strdup(cpf->name)) == NULL) goto error;
		}
		ret->listnum = i;
	}
	return (LCMPOp*)ret;
error:
	LCMP_op_free((LCMPOp*)ret);
	return NULL;
}

/*-------------------------------------------------*/
LCMPOp *LCMP_get_certreq(int certOp, int snum, int reason, Key *key, char *kpwd){
	LO_CertReq *ret=NULL;

	if((ret=(LO_CertReq*)LCMP_op_new(LCMP_OP_CERTREQ))==NULL) goto error;
	ret->certOp       = certOp;
	ret->serialNum    = snum;
	ret->revokeReason = reason;
	if(key)
		if((ret->key=Key_dup(key))==NULL) goto error;
	if(kpwd){
		if ((ret->passwd = strdup(kpwd)) == NULL){
			OK_set_error(ERR_ST_STRDUP,ERR_LC_LCMP,ERR_PT_LCMPTOOL+5,NULL);
			goto error;
	}}
	
	return (LCMPOp*)ret;
error:
	LCMP_op_free((LCMPOp*)ret);
	return NULL;
}

LCMPOp *LCMP_get_certrsp(Cert *ct, Key *key){
	LO_CertRsp *ret=NULL;

	if((ret=(LO_CertRsp*)LCMP_get_response(LCMP_OP_CERTRSP,LCMP_SUCCESS,"cert ok"))==NULL)
	  goto error;

	if(ct)
		if((ret->cert=Cert_dup(ct))==NULL) goto error;
	if(key)
		if((ret->key=Key_dup(key))==NULL) goto error;
	
	return (LCMPOp*)ret;
error:
	LCMP_op_free((LCMPOp*)ret);
	return NULL;
}

/*-------------------------------------------------*/
LCMPOp *LCMP_get_csrreq(int csrOp, int accID, int snum, char *pfname, Req *csr, int reason, CertTemplate *tmpl){
	LO_CSRReq *ret=NULL;

	if((ret=(LO_CSRReq*)LCMP_op_new(LCMP_OP_CSRREQ))==NULL) goto error;
	ret->csrOp        = csrOp;
	ret->acceptID     = accID;
	ret->serialNum    = snum;
	ret->rejectReason = reason;
	if(pfname)
		if ((ret->profName = strdup(pfname)) == NULL){
			OK_set_error(ERR_ST_STRDUP,ERR_LC_LCMP,ERR_PT_LCMPTOOL+5,NULL);
			goto error;
		}
	if(csr)
		if((ret->csr=Req_dup(csr))==NULL) goto error;
	if(tmpl)
		if((ret->tmpl=CMP_certtmpl_dup(tmpl))==NULL) goto error;
	
	return (LCMPOp*)ret;
error:
	LCMP_op_free((LCMPOp*)ret);
	return NULL;
}

LCMPOp *LCMP_get_csrrsp(int accID, int snum, Req *csr, CertTemplate *tmpl){
	LO_CSRRsp *ret=NULL;

	if((ret=(LO_CSRRsp*)LCMP_get_response(LCMP_OP_CSRRSP,LCMP_SUCCESS,"csr ok"))==NULL)
		goto error;

	ret->acceptID     = accID;
	ret->serialNum    = snum;
	if(csr)
		if((ret->csr=Req_dup(csr))==NULL) goto error;
	if(tmpl)
		if((ret->tmpl=CMP_certtmpl_dup(tmpl))==NULL) goto error;
	
	return (LCMPOp*)ret;
error:
	LCMP_op_free((LCMPOp*)ret);
	return NULL;
}

/*-------------------------------------------------*/
LCMPOp *LCMP_get_crlreq(int crlOp,char *profname,int location,char *path){
	LO_CRLReq *ret=NULL;

	if((ret=(LO_CRLReq*)LCMP_op_new(LCMP_OP_CRLREQ))==NULL) goto error;

	if(profname==NULL) profname = "CRL-All";

	if ((ret->profName = strdup(profname)) == NULL){
	  OK_set_error(ERR_ST_STRDUP,ERR_LC_LCMP,ERR_PT_LCMPTOOL+6,NULL);
	  goto error;
	}
	if(path){
	  if ((ret->path = strdup(path)) == NULL){
	    OK_set_error(ERR_ST_STRDUP,ERR_LC_LCMP,ERR_PT_LCMPTOOL+6,NULL);
	    goto error;
	  }
	}
	ret->crlOp = crlOp;
	ret->location = location;

	return (LCMPOp*)ret;
error:
	LCMP_op_free((LCMPOp*)ret);
	return NULL;
}

LCMPOp *LCMP_get_crlrsp(CRLList *top){
	LO_CRLRsp *ret=NULL;

	if((ret=(LO_CRLRsp*)LCMP_get_response(LCMP_OP_CRLRSP,LCMP_SUCCESS,"crl ok"))==NULL)
	  goto error;

	if(top){
	  if((ret->list=CRLlist_dup_all(top))==NULL) goto error;
	}
	
	return (LCMPOp*)ret;
error:
	LCMP_op_free((LCMPOp*)ret);
	return NULL;
}

/*-------------------------------------------------*/
LCMPOp *LCMP_get_svopreq(int svOp,AuthInfo *ai){
	LO_SVOpReq *ret=NULL;

	if((ret=(LO_SVOpReq*)LCMP_op_new(LCMP_OP_SVOPREQ))==NULL) goto error;

	ret->svOp = svOp;
	if((ret->uinfo=CA_ai_dup_all(ai))==NULL) goto error;

	return (LCMPOp*)ret;
error:
	LCMP_op_free((LCMPOp*)ret);
	return NULL;
}

LCMPOp *LCMP_get_svoprsp(unsigned char *opRsp,int rspLen){
	LO_SVOpRsp *ret=NULL;

	if((ret=(LO_SVOpRsp*)LCMP_get_response(LCMP_OP_SVOPRSP,LCMP_SUCCESS,"svop ok"))==NULL)
	  goto error;

	if(opRsp){
		if((ret->opRsp=(unsigned char*)malloc(rspLen))==NULL){
		    OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMPTOOL+7,NULL);
		    goto error;
		}
		memcpy(ret->opRsp,opRsp,rspLen);
		ret->rspLen = rspLen;
	}

	return (LCMPOp*)ret;
error:
	LCMP_op_free((LCMPOp*)ret);
	return NULL;
}

/*-------------------------------------------------*/
LCMPOp *LCMP_get_extreq(char *oid, unsigned char *bin, int blen){
	LO_ExtReq *ret=NULL;

	if((ret=(LO_ExtReq*)LCMP_op_new(LCMP_OP_EXTREQ))==NULL) goto error;
	
	if ((ret->opOID = strdup(oid)) == NULL){
	    OK_set_error(ERR_ST_STRDUP,ERR_LC_LCMP,ERR_PT_LCMPTOOL+8,NULL);
	    goto error;
	}
	if((ret->opInfo=(unsigned char*)malloc(blen))==NULL){
	    OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMPTOOL+8,NULL);
	    goto error;
	}
	ret->ifLen = blen;

	return (LCMPOp*)ret;
error:
	LCMP_op_free((LCMPOp*)ret);
	return NULL;
}

LCMPOp *LCMP_get_extrsp(char *oid, unsigned char *bin, int blen){
	LO_ExtRsp *ret=NULL;

	if((ret=(LO_ExtRsp*)LCMP_get_response(LCMP_OP_EXTRSP,LCMP_SUCCESS,"ext ok"))==NULL)
	  goto error;

	if ((ret->opOID = strdup(oid)) == NULL){
	    OK_set_error(ERR_ST_STRDUP,ERR_LC_LCMP,ERR_PT_LCMPTOOL+8,NULL);
	    goto error;
	}
	if((ret->opRsp=(unsigned char*)malloc(blen))==NULL){
	    OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMPTOOL+8,NULL);
	    goto error;
	}
	ret->rspLen = blen;
	
	return (LCMPOp*)ret;
error:
	LCMP_op_free((LCMPOp*)ret);
	return NULL;
}

/*-------------------------------------------------*/
LCMPOp *LCMP_get_renewalreq(int snum,enum csrform csrFormat,void *req,int newSerialNum){
	LO_RenewalReq *ret=NULL;

	if((ret=(LO_RenewalReq*)LCMP_op_new(LCMP_OP_RENEWALREQ))==NULL) goto error;

	ret->serialNum = snum;

	switch(csrFormat){
	case PKCS10CSR: /* pkcs 10 */
		if((ret->p10=Req_dup((Req*)req))==NULL) goto error;
		break;
	case CMPREQUEST: /* cert template */
		if((ret->tmpl=CMP_certtmpl_dup((CertTemplate*)req))==NULL) goto error;
		break;
	}
	
	ret->newSerialNum = (newSerialNum>0)?(newSerialNum):(0);

	return (LCMPOp*)ret;
error:
	LCMP_op_free((LCMPOp*)ret);
	return NULL;
}

LCMPOp *LCMP_get_renewalrsp(Cert *ct){
	LO_RenewalRsp *ret=NULL;

	if((ret=(LO_RenewalRsp*)LCMP_get_response(LCMP_OP_RENEWALRSP,LCMP_SUCCESS,"renewal ok"))==NULL)
	  goto error;

	if(ct)
		if((ret->cert=Cert_dup(ct))==NULL) goto error;

	return (LCMPOp*)ret;
error:
	LCMP_op_free((LCMPOp*)ret);
	return NULL;
}



