/* lcmp_asn1e.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *		this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *		this list of conditions and the following disclaimer in the documentation
 *		and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *		display the following acknowledgment:
 *		"This product includes software developed by Akira Iwata Laboratory,
 *		Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *		acknowledgment:
 *		"This product includes software developed by Akira Iwata Laboratory,
 *		 Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *	 THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *	 AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *	 SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *	 IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *	 INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *	 FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *	 NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *	 WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_uconv.h>

#include "ok_caerr.h"
#include "ok_lcmp.h"

/*-----------------------------------------------
	LCMP ASN1 message encoding
-----------------------------------------------*/
unsigned char *LCMP_op_toDER(LCMPOp *op, unsigned char *buf, int *ret_len){
	unsigned char *cp,*ret;
	int i,j;
	
	if(buf==NULL){
		if((i=LCMPop_estimate_der_size(op))<=0)
			return NULL;
		
		if((ret=(unsigned char*)malloc(i))==NULL){
			OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMPASE,NULL);
			return NULL;
		}
		memset(ret,0,i);
	}else{
		ret=buf;
	}
	
	ASN1_set_integer(op->messageId,ret,&i);
	cp = ret+i;

	switch(op->opId){
	case LCMP_OP_BINDREQ: if(lcmp_DER_bindreq(op,cp,&j)) goto error; break;
	case LCMP_OP_BINDRSP: if(lcmp_DER_bindrsp(op,cp,&j)) goto error; break;
	case LCMP_OP_UNBIND:  cp[0]=LCMP_OP_UNBIND|0x80; cp[1]=0; j=2; break;
	case LCMP_OP_SIGNREQ: if(lcmp_DER_signreq(op,cp,&j)) goto error; break;
	case LCMP_OP_SIGNRSP: if(lcmp_DER_signrsp(op,cp,&j)) goto error; break;
	case LCMP_OP_LISTREQ: if(lcmp_DER_listreq(op,cp,&j)) goto error; break;
	case LCMP_OP_LISTRSP: if(lcmp_DER_listrsp(op,cp,&j)) goto error; break;
	case LCMP_OP_PROFREQ: if(lcmp_DER_profreq(op,cp,&j)) goto error; break;
	case LCMP_OP_PROFRSP: if(lcmp_DER_profrsp(op,cp,&j)) goto error; break;
	case LCMP_OP_CERTREQ: if(lcmp_DER_certreq(op,cp,&j)) goto error; break;
	case LCMP_OP_CERTRSP: if(lcmp_DER_certrsp(op,cp,&j)) goto error; break;
	case LCMP_OP_CSRREQ:  if(lcmp_DER_csrreq(op,cp,&j)) goto error; break;
	case LCMP_OP_CSRRSP:  if(lcmp_DER_csrrsp(op,cp,&j)) goto error; break;
	case LCMP_OP_CRLREQ:  if(lcmp_DER_crlreq(op,cp,&j)) goto error; break;
	case LCMP_OP_CRLRSP:  if(lcmp_DER_crlrsp(op,cp,&j)) goto error; break;
	case LCMP_OP_SVOPREQ: if(lcmp_DER_svopreq(op,cp,&j)) goto error; break;
	case LCMP_OP_SVOPRSP: if(lcmp_DER_svoprsp(op,cp,&j)) goto error; break;
	case LCMP_OP_EXTREQ:  if(lcmp_DER_extreq(op,cp,&j)) goto error; break;
	case LCMP_OP_EXTRSP:  if(lcmp_DER_extrsp(op,cp,&j)) goto error; break;
	case LCMP_OP_RENEWALREQ: if(lcmp_DER_renewalreq(op,cp,&j)) goto error; break;
	case LCMP_OP_RENEWALRSP: if(lcmp_DER_renewalrsp(op,cp,&j)) goto error; break;
	}
	cp+=j; i+=j;

	if(op->ctrl){
		if(lcmp_DER_ctrl(op->ctrl,cp,&j)) goto error;
		i+=j;
	}
	ASN1_set_sequence(i,ret,ret_len);
	
	return ret;
error:
	if(ret!=buf) free(ret);
	return NULL;
}

/*-----------------------------------------------
		LCMP ASN1 control encoding
-----------------------------------------------*/
int lcmp_DER_ctrl(LCMPCtrl *ct, unsigned char *buf, int *ret_len){
	unsigned char *cp,*sq=buf;
	int i,j,k;

	i=j=0;
	while(ct){
		ASN1_set_octetstring(strlen(ct->ctrlType),(unsigned char*)ct->ctrlType,sq,&j);
		cp=sq+j;
		
		ASN1_set_boolean(ct->crit,cp,&k);
		cp+=k; j+=k;

		if(ct->value){
			ASN1_set_octetstring(ct->vlen,ct->value,cp,&k);
			j+=k;
		}
		ASN1_set_sequence(j,sq,&j);
		sq+=j; i+=j;

		ct=ct->next;
	}
	ASN1_set_explicit(i,0,buf,ret_len);
	
	return 0;
}

/*-----------------------------------------------
		LCMP ASN1 response message encoding
-----------------------------------------------*/
int lcmp_DER_result(LCMPOp *op, unsigned char *buf, int *ret_len){
	unsigned char np[4];
	int i,j;

	/* resultCode */
	ASN1_set_enumerated(op->resultCode,buf,&i);
	buf+=i;

	/* resultMessage */
	if(op->resultMsg)
		ASN1_set_octetstring(strlen(op->resultMsg),(unsigned char*)op->resultMsg,buf,&j);
	else
		ASN1_set_octetstring(0,np,buf,&j);

	*ret_len = i+j;
	return 0;
}

/*-----------------------------------------------
	LCMP ASN1 operations encoding
-----------------------------------------------*/
int lcmp_DER_bindreq(LCMPOp *op, unsigned char *buf, int *ret_len){
	LO_BindReq *lo = (LO_BindReq*)op;
	unsigned char *cp;
	int i,j;

	/* version */
	ASN1_set_integer(lo->version,buf,&i);
	cp=buf+i;

	/* caName */
	if(lo->caname==NULL){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_LCMP,ERR_PT_LCMPASE+2,NULL);
		goto error;
	}
	ASN1_set_octetstring(strlen(lo->caname),(unsigned char*)lo->caname,cp,&j);
	cp+=j; i+=j;

	/* userName OPTIONAL */
	if(lo->username){
		ASN1_set_octetstring(strlen(lo->username),(unsigned char*)lo->username,cp,&j);
		cp+=j; i+=j;
	}

	/* AuthenticationChoice */
	switch(lo->authMode){
	case LCMP_OPBRQ_ATSIMPLE: /* Simple */
		if(lo->passwd==NULL){
			OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_LCMP,ERR_PT_LCMPASE+2,NULL);
			goto error;
		}
		ASN1_set_octetstring(strlen(lo->passwd),(unsigned char*)lo->passwd,cp,&j);
		cp[0]=0x80;
		break;
	case LCMP_OPBRQ_ATSSLCL: /* SSL Client */
		cp[0]=0x81; cp[1]=0; j=1;
		break;
	default:
		OK_set_error(ERR_ST_BADPARAM,ERR_LC_LCMP,ERR_PT_LCMPASE+2,NULL);
		goto error;
	}
	i+=j;
	
	ASN1_set_explicit(i,LCMP_OP_BINDREQ,buf,ret_len);

	return 0;
error:
	return -1;
}

int lcmp_DER_bindrsp(LCMPOp *op, unsigned char *buf, int *ret_len){
	LO_BindRsp *lo = (LO_BindRsp*)op;
	unsigned char *cp,*tp;
	int i,j,k;

	/* LCMP Result */
	if(lcmp_DER_result(op,buf,&i)) goto error;
	cp=buf+i;

	/* [0] SET OF Certificate OPTIONAL */
	if(lo->list){
		CertList *cl=lo->list;

		tp=cp; j=0;
		while(cl){
			if(cl->cert->der==NULL){
				OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_LCMP,ERR_PT_LCMPASE+2,NULL);
				goto error;
			}
			ASN1_skip_(cl->cert->der,&k);
			memcpy(cp,cl->cert->der,k);
			cp+=k; j+=k;

			cl=cl->next;
		}
		ASN1_set_explicit(j,0,tp,&j);
		i+=j;
	}
	ASN1_set_explicit(i,LCMP_OP_BINDRSP,buf,ret_len);
	
	return 0;
error:
	return -1;
}

int lcmp_DER_signreq(LCMPOp *op, unsigned char *buf, int *ret_len){
	LO_SignReq *lo = (LO_SignReq*)op;
	unsigned char *cp;
	int i,j;
	
	/* profileName */
	if(lo->profName==NULL){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_LCMP,ERR_PT_LCMPASE+3,NULL);
		goto error;
	}
	ASN1_set_octetstring(strlen(lo->profName),(unsigned char*)lo->profName,buf,&i);
	cp=buf+i;
	
	/* requestBody */
	if(lo->p10 && lo->p10->der){
		/* send pkcs#10 format */
		ASN1_set_enumerated(PKCS10CSR,cp,&j);
		cp+=j; i+=j;

		ASN1_skip_(lo->p10->der,&j);
		ASN1_set_octetstring(j,lo->p10->der,cp,&j);
		cp+=j; i+=j;
	}else if(lo->tmpl){
		/* send certTemplate format */
		ASN1_set_enumerated(CMPREQUEST,cp,&j);
		cp+=j; i+=j;

		if(CMP_DER_certtmpl(lo->tmpl,cp,&j)) goto error;
		ASN1_set_sequence(j,cp,&j);
		*cp = ASN1_OCTETSTRING; cp+=j; i+=j;

	}else{
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_LCMP,ERR_PT_LCMPASE+3,NULL);
		goto error;
	}

	/* requestSNumber [0] OPTIONAL */
	if(lo->serialNum){
		ASN1_set_integer(lo->serialNum,cp,&j);
		cp[0] = 0x80; /* implicit */
		i+=j;
	}
	
	ASN1_set_explicit(i,LCMP_OP_SIGNREQ,buf,ret_len);
	
	return 0;
error:
	return -1;
}

int lcmp_DER_signrsp(LCMPOp *op, unsigned char *buf, int *ret_len){
	LO_SignRsp *lo = (LO_SignRsp*)op;
	unsigned char *cp;
	int i,j;
	
	/* LCMP Result */
	if(lcmp_DER_result(op,buf,&i)) goto error;
	cp=buf+i;

	/* certificate [0] explicit OPTIONAL */
	if(lo->cert){
		if(lo->cert->der==NULL){
			OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_LCMP,ERR_PT_LCMPASE+3,NULL);
			goto error;
		}
		ASN1_skip_(lo->cert->der,&j);
		memcpy(cp,lo->cert->der,j);
		ASN1_set_explicit(j,0,cp,&j);
		i+=j;
	}
	ASN1_set_explicit(i,LCMP_OP_SIGNRSP,buf,ret_len);

	return 0;
error:
	return -1;
}

int lcmp_DER_listreq(LCMPOp *op, unsigned char *buf, int *ret_len){
	LO_ListReq *lo = (LO_ListReq*)op;
	unsigned char *cp;
	char tmp[256];
	int i,j;

	/* profileName */
	if(lo->profName==NULL){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_LCMP,ERR_PT_LCMPASE+4,NULL);
		goto error;
	}
	ASN1_set_octetstring(strlen(lo->profName),(unsigned char*)lo->profName,buf,&i);
	cp=buf+i;

	/* query CHOICE { */
	if(lo->serialNum>0){ /* serialNumber */
		ASN1_set_integer(lo->serialNum,cp,&j);
		i+=j;
	}else if(lo->sbjQuery){ /* subjectQuery */
		if((j=UC_conv(UC_LOCAL_CODESET,UC_CODE_UTF8,lo->sbjQuery,strlen(lo->sbjQuery),tmp,256))<0) goto error;
		if(ASN1_set_utf8(tmp,cp,&j)) goto error;
		i+=j;
	}
	ASN1_set_explicit(i,LCMP_OP_LISTREQ,buf,ret_len);

	return 0;
error:
	return -1;
}

int lcmp_DER_listrsp(LCMPOp *op, unsigned char *buf, int *ret_len){
	LO_ListRsp *lo = (LO_ListRsp*)op;
	unsigned char *cp;
	int i,j;
	
	/* LCMP Result */
	if(lcmp_DER_result(op,buf,&i)) goto error;
	cp=buf+i;

	/* certList [0] StatusList OPTIONAL */
	if(lo->stat){
		OK_set_time_dertag(ASN1_GENERALIZEDTIME);
		if(lcmp_DER_statlist(lo->stat,cp,&j)) goto error;
		OK_set_time_dertag(0);

		cp[0] = 0xa0; /* implicit 0 */
		i+=j;
	}
	
	ASN1_set_explicit(i,LCMP_OP_LISTRSP,buf,ret_len);

	return 0;
error:
	return -1;
}

int lcmp_DER_profreq(LCMPOp *op, unsigned char *buf, int *ret_len){
	LO_ProfReq *lo = (LO_ProfReq*)op;
	unsigned char *cp;
	int i,j;

	/* profileName */
	if(lo->profName==NULL){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_LCMP,ERR_PT_LCMPASE+5,NULL);
		goto error;
	}
	ASN1_set_octetstring(strlen(lo->profName),(unsigned char*)lo->profName,buf,&i);
	cp=buf+i;

	/* profOperation CHOICE */
	switch(lo->profOp){
	case LCMP_OPPF_VIEWPF: /* [0] NULL */
	case LCMP_OPPF_DELPF: /* [2] DelProfContent ::= NULL */
	case LCMP_OPPF_VIEWEXT: /* [4] NULL */
	case LCMP_OPPF_PFLIST: /* [8] listNames */
		cp[0]=0x80 | lo->profOp; cp[1]=0; j=2;
		break;

	case LCMP_OPPF_RENAMEPF: /* [9] LCMPString */
		ASN1_set_octetstring(strlen(lo->profRename),(unsigned char*)lo->profRename,cp,&j);
		cp[0]=0x80 | lo->profOp;
		break;

	case LCMP_OPPF_ADDPF: /* [1] AddProfContent */
	case LCMP_OPPF_MODPF: /* [3] ModProfContent */
		if(lo->prof==NULL){
			OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_LCMP,ERR_PT_LCMPASE+5,NULL);
			goto error;
		}
		if(lcmp_DER_tmpl(lo->prof,cp,&j)) goto error;
		cp[0]=0xa0 | lo->profOp;
		break;
		
	case LCMP_OPPF_ADDEXT: /* [5] AddExtContent */
	case LCMP_OPPF_DELEXT: /* [6] DelExtContent */
	case LCMP_OPPF_UPDEXT: /* [7] UpdExtContent */
		if(lo->ext==NULL){
			OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_LCMP,ERR_PT_LCMPASE+5,NULL);
			goto error;
		}
		if(lo->profOp==6){
			if(lcmp_DER_delext(lo->ext,cp,&j)) goto error;
		}else{
			if(x509_DER_exts(lo->ext,cp,&j)) goto error;
		}
		cp[0]=0xa0 | lo->profOp;
		break;

	default:
		OK_set_error(ERR_ST_BADPARAM,ERR_LC_LCMP,ERR_PT_LCMPASE+5,NULL);
		goto error;
	}
	i+=j;

	ASN1_set_explicit(i,LCMP_OP_PROFREQ,buf,ret_len);

	return 0;
error:
	return -1;
}

int lcmp_DER_profrsp(LCMPOp *op, unsigned char *buf, int *ret_len){
	LO_ProfRsp *lo = (LO_ProfRsp*)op;
	unsigned char *cp,*t;
	int i,j=0,k,l;
	
	/* LCMP Result */
	if(lcmp_DER_result(op,buf,&i)) goto error;
	cp=buf+i;

	/* viewProfile [0] ModProfContent OPTIONAL */
	if(lo->prof){
		if(lcmp_DER_tmpl(lo->prof,cp,&j)) goto error;
		cp[0]=0xa0; cp+=j; i+=j;
	}
	/* viewExtensions [1] Extensions OPTIONAL */
	if(lo->ext){
		if(x509_DER_exts(lo->ext,cp,&j)) goto error;
		cp[0]=0xa1; cp+=j; i+=j;
	}
	/* listedNames [2] SET OF LCMPString OPTIONAL */
	if(lo->proflist){
		for(k=l=0,t=cp; k<lo->listnum; k++)
			if(lo->proflist[k]){
				ASN1_set_octetstring(strlen(lo->proflist[k]),
					(unsigned char*)lo->proflist[k],t,&j);
				t+=j; l+=j;
			}
		ASN1_set_explicit(l,2,cp,&j);
		i+=j;
	}
	
	ASN1_set_explicit(i,LCMP_OP_PROFRSP,buf,ret_len);

	return 0;
error:
	return -1;
}

int lcmp_DER_certreq(LCMPOp *op, unsigned char *buf, int *ret_len){
	LO_CertReq *lo = (LO_CertReq*)op;
	unsigned char *cp;
	int i,j;

	/* serialNumber */
	ASN1_set_integer(lo->serialNum,buf,&i);
	cp=buf+i;
			
	/* certOperation */
	ASN1_set_enumerated(lo->certOp,cp,&j);
	cp+=j; i+=j;
	
	/* revokeReason [0] CRLReason OPTIONAL */
	if(lo->revokeReason>0){
		ASN1_set_integer(lo->revokeReason,cp,&j);
		cp[0] = 0x80; /* implicit 0 */
		cp+=j; i+=j;
	}
	
	/* keyPassword [1] LCMPString OPTIONAL */
	if(lo->passwd){
		ASN1_set_octetstring(strlen(lo->passwd),(unsigned char*)lo->passwd,cp,&j);
		cp[0] = 0x81; /* implicit 1 */
		cp+=j; i+=j;
	}
		
	/* key [3] EXPLICIT KeyType OPTIONAL */
	if(lo->key){
		if(lcmp_DER_keytype(lo->key,cp,&j)) goto error;
		ASN1_set_explicit(j,3,cp,&j);
		cp+=j; i+=j;
	}
	
	ASN1_set_explicit(i,LCMP_OP_CERTREQ,buf,ret_len);
	
	return 0;
error:
	return -1;
}

int lcmp_DER_certrsp(LCMPOp *op, unsigned char *buf, int *ret_len){
	LO_CertRsp *lo = (LO_CertRsp*)op;
	unsigned char *cp;
	int i,j;
	
	/* LCMP Result */
	if(lcmp_DER_result(op,buf,&i)) goto error;
	cp=buf+i;

	/* certificate [0] EXPLICIT Certificate OPTIONAL */
	if(lo->cert){
		if(lo->cert->der==NULL){
			OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_LCMP,ERR_PT_LCMPASE+6,NULL);
			goto error;
		}
		ASN1_skip_(lo->cert->der,&j);
		memcpy(cp,lo->cert->der,j);

		ASN1_set_explicit(j,0,cp,&j);
		cp+=j; i+=j;
	}
	
	/* key [1] EXPLICIT KeyType OPTIONAL */
	if(lo->key){
		if(lcmp_DER_keytype(lo->key,cp,&j)) goto error;
		ASN1_set_explicit(j,1,cp,&j);
		cp+=j; i+=j;
	}	

	ASN1_set_explicit(i,LCMP_OP_CERTRSP,buf,ret_len);

	return 0;
error:
	return -1;
}

int lcmp_DER_csrreq(LCMPOp *op, unsigned char *buf, int *ret_len){
	LO_CSRReq *lo = (LO_CSRReq*)op;
	unsigned char *cp;
	int i,j;

	/* acceptID */
	ASN1_set_integer(lo->acceptID,buf,&i);
	cp=buf+i;

	/* serialNumber OPTIONAL 
	 * -- but always specified because of version compatible.
	 */
	ASN1_set_integer(lo->serialNum,cp,&j);
	cp+=j; i+=j;

	/* csrOperation */
	ASN1_set_enumerated(lo->csrOp,cp,&j);
	cp+=j; i+=j;
	
	/* csr [0] EXPLICIT CertificateReqest OPTIONAL */
	if(lo->csr){
		if(lo->csr->der==NULL){
			OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_LCMP,ERR_PT_LCMPASE+7,NULL);
			goto error;
		}
		ASN1_skip_(lo->csr->der,&j);
		memcpy(cp,lo->csr->der,j);

		ASN1_set_explicit(j,0,cp,&j);
		cp+=j; i+=j;
	}
	/* profileName [1] LCMPSTRING OPTIONAL */
	if(lo->profName){
		ASN1_set_octetstring(strlen(lo->profName),(unsigned char*)lo->profName,cp,&j);
		cp[0] = 0x81; /* implicit 1 */
		cp+=j; i+=j;
	}
	/* revokeReason [2] INTEGER OPTIONAL */
	if(lo->rejectReason>0){
		ASN1_set_integer(lo->rejectReason,cp,&j);
		cp[0] = 0x82; /* implicit 2 */
		cp+=j; i+=j;
	}
	/* certtmpl	[3] EXPLICIT CertTemplate OPTIONAL */
	if(lo->tmpl){
		/* send certTemplate format */
		if(CMP_DER_certtmpl(lo->tmpl,cp,&j)) goto error;

		ASN1_set_explicit(j,3,cp,&j);
		cp+=j; i+=j;
	}

	ASN1_set_explicit(i,LCMP_OP_CSRREQ,buf,ret_len);
	
	return 0;
error:
	return -1;
}

int lcmp_DER_csrrsp(LCMPOp *op, unsigned char *buf, int *ret_len){
	LO_CSRRsp *lo = (LO_CSRRsp*)op;
	unsigned char *cp;
	int i,j;
	
	/* LCMP Result */
	if(lcmp_DER_result(op,buf,&i)) goto error;
	cp=buf+i;

	/* acceptID */
	ASN1_set_integer(lo->acceptID,cp,&j);
	cp+=j; i+=j;

	/* serialNumber OPTIONAL 
	 * -- but always specified because of version compatible.
	 */
	ASN1_set_integer(lo->serialNum,cp,&j);
	cp+=j; i+=j;

	/* csr [0] EXPLICIT CertificateReqest OPTIONAL */
	if(lo->csr){
		if(lo->csr->der==NULL){
			OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_LCMP,ERR_PT_LCMPASE+7,NULL);
			goto error;
		}
		ASN1_skip_(lo->csr->der,&j);
		memcpy(cp,lo->csr->der,j);

		ASN1_set_explicit(j,0,cp,&j);
		cp+=j; i+=j;
	}
	/* certtmpl	[1] EXPLICIT CertTemplate OPTIONAL */
	if(lo->tmpl){
		/* send certTemplate format */
		if(CMP_DER_certtmpl(lo->tmpl,cp,&j)) goto error;

		ASN1_set_explicit(j,1,cp,&j);
		cp+=j; i+=j;
	}

	ASN1_set_explicit(i,LCMP_OP_CSRRSP,buf,ret_len);

	return 0;
error:
	return -1;
}

int lcmp_DER_crlreq(LCMPOp *op, unsigned char *buf, int *ret_len){
	LO_CRLReq *lo = (LO_CRLReq*)op;
	unsigned char *cp;
	int i,j;

	/* profileName */
	if(lo->profName==NULL){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_LCMP,ERR_PT_LCMPASE+8,NULL);
		goto error;
	}
	ASN1_set_octetstring(strlen(lo->profName),(unsigned char*)lo->profName,buf,&i);
	cp=buf+i;

	/* crlOperation */
	ASN1_set_enumerated(lo->crlOp,cp,&j);
	cp+=j; i+=j;

	/* location CHOICE */
	switch(lo->location){
	case 0:
	case 1:
		cp[0]=0x80 | lo->location; cp[1]=0; j=2;
		break;

	case 2:
	case 3:
		if(lo->path==NULL){
			OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_LCMP,ERR_PT_LCMPASE+8,NULL);
			goto error;
		}
		ASN1_set_octetstring(strlen(lo->path),(unsigned char*)lo->path,cp,&j);
		cp[0]=0x80 | lo->location;
		break;
		
	default:
		OK_set_error(ERR_ST_BADPARAM,ERR_LC_LCMP,ERR_PT_LCMPASE+8,NULL);
		goto error;
	}
	cp+=j; i+=j;

	/* crlNumber INTEGER OPTIONAL */
	if(lo->crlNum>0){
		ASN1_set_integer(lo->crlNum,cp,&j);
		i+=j;
	}
	
	ASN1_set_explicit(i,LCMP_OP_CRLREQ,buf,ret_len);

	return 0;
error:
	return -1;
}

int lcmp_DER_crlrsp(LCMPOp *op, unsigned char *buf, int *ret_len){
	LO_CRLRsp *lo = (LO_CRLRsp*)op;
	unsigned char *cp,*tp=NULL;
	int i,j,k;
	
	/* LCMP Result */
	if(lcmp_DER_result(op,buf,&i)) goto error;
	cp=buf+i;

	/* crls [0] SET OF CertificateList OPTIONAL */
	if(lo->list){
		CRLList *cl=lo->list;

		tp=cp; j=0;
		while(cl){
			if(cl->crl->der==NULL){
				OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_LCMP,ERR_PT_LCMPASE+8,NULL);
				goto error;
			}
			ASN1_skip_(cl->crl->der,&k);
			memcpy(cp,cl->crl->der,k);
			cp+=k; j+=k;

			cl=cl->next;
		}
		ASN1_set_explicit(j,0,tp,&j);
		i+=j;
	}
	ASN1_set_explicit(i,LCMP_OP_CRLRSP,buf,ret_len);

	return 0;
error:
	return -1;
}

int lcmp_DER_svopreq(LCMPOp *op, unsigned char *buf, int *ret_len){
	LO_SVOpReq *lo = (LO_SVOpReq*)op;
	unsigned char *cp;
	int i,j;

	/* serverOperation */
	ASN1_set_enumerated(lo->svOp,buf,&i);
	cp=buf+i;

	/* userInfo [0] UserInfo OPTIONAL */
	if(lo->uinfo){
		if(lcmp_DER_userinfo(lo->uinfo,cp,&j)) goto error;
		cp[0]=0xa0; i+=j;
	}
	
	ASN1_set_explicit(i,LCMP_OP_SVOPREQ,buf,ret_len);
	
	return 0;
error:
	return -1;
}

int lcmp_DER_svoprsp(LCMPOp *op, unsigned char *buf, int *ret_len){
	LO_SVOpRsp *lo = (LO_SVOpRsp*)op;
	unsigned char *cp;
	int i,j;
	
	/* LCMP Result */
	if(lcmp_DER_result(op,buf,&i)) goto error;
	cp=buf+i;

	/* returnContent [0] OCTET STRING OPTIONAL */
	if(lo->opRsp){
		ASN1_set_octetstring(lo->rspLen,lo->opRsp,cp,&j);
		cp[0]=0x80; i+=j;
	}
	
	ASN1_set_explicit(i,LCMP_OP_SVOPRSP,buf,ret_len);

	return 0;
error:
	return -1;
}

int lcmp_DER_extreq(LCMPOp *op, unsigned char *buf, int *ret_len){
	LO_ExtReq *lo = (LO_ExtReq*)op;
	unsigned char *cp;
	int i,j;
	
	cp=buf; i=0;
	/* requestName [0] OBJECT IDENTIFIER OPTIONAL */
	if(lo->opOID){
		if(str2objid(lo->opOID,cp,strlen(lo->opOID))<0) goto error;
		ASN1_skip_(cp,&j); cp[0]=0x80;
		cp+=j; i+=j;
	}
	/* requestValue [1] OCTET STRING OPTIONAL */
	if(lo->opInfo){
		ASN1_set_octetstring(lo->ifLen,lo->opInfo,cp,&j);
		cp[0]=0x81; i+=j;
	}

	ASN1_set_explicit(i,LCMP_OP_EXTREQ,buf,ret_len);

	return 0;
error:
	return -1;
}

int lcmp_DER_extrsp(LCMPOp *op, unsigned char *buf, int *ret_len){
	LO_ExtRsp *lo = (LO_ExtRsp*)op;
	unsigned char *cp;
	int i,j;
	
	/* LCMP Result */
	if(lcmp_DER_result(op,buf,&i)) goto error;
	cp=buf+i;

	/* responseName [0] OBJECT IDENTIFIER OPTIONAL */
	if(lo->opOID){
		if(str2objid(lo->opOID,cp,strlen(lo->opOID))<0) goto error;
		ASN1_skip_(cp,&j); cp[0]=0x80;
		cp+=j; i+=j;
	}
	/* response [1] OCTET STRING OPTIONAL */
	if(lo->opRsp){
		ASN1_set_octetstring(lo->rspLen,lo->opRsp,cp,&j);
		cp[0]=0x81; i+=j;
	}

	ASN1_set_explicit(i,LCMP_OP_EXTRSP,buf,ret_len);

	return 0;
error:
	return -1;
}

int lcmp_DER_renewalreq(LCMPOp *op, unsigned char *buf, int *ret_len){
	LO_RenewalReq *lo = (LO_RenewalReq*)op;
	unsigned char *cp;
	int i,j;
	
	/* serialNumber */
	ASN1_set_integer(lo->serialNum,buf,&i);
	cp=buf+i;
			
	/* requestBody */
	if(lo->p10 && lo->p10->der){
		/* send pkcs#10 format */
		ASN1_set_enumerated(PKCS10CSR,cp,&j);
		cp+=j; i+=j;

		ASN1_skip_(lo->p10->der,&j);
		ASN1_set_octetstring(j,lo->p10->der,cp,&j);
		cp+=j; i+=j;
	}else if(lo->tmpl){
		/* send certTemplate format */
		ASN1_set_enumerated(CMPREQUEST,cp,&j);
		cp+=j; i+=j;

		if(CMP_DER_certtmpl(lo->tmpl,cp,&j)) goto error;
		ASN1_set_sequence(j,cp,&j);
		*cp = ASN1_OCTETSTRING; cp+=j; i+=j;

	}else{
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_LCMP,ERR_PT_LCMPASE+9,NULL);
		goto error;
	}

	/* requestSNumber [0] OPTIONAL */
	if(lo->newSerialNum){
		ASN1_set_integer(lo->newSerialNum,cp,&j);
		cp[0] = 0x80; /* implicit */
		i+=j;
	}
	
	ASN1_set_explicit(i,LCMP_OP_RENEWALREQ,buf,ret_len);
	
	return 0;
error:
	return -1;
}

int lcmp_DER_renewalrsp(LCMPOp *op, unsigned char *buf, int *ret_len){
	LO_RenewalRsp *lo = (LO_RenewalRsp*)op;
	unsigned char *cp;
	int i,j;
	
	/* LCMP Result */
	if(lcmp_DER_result(op,buf,&i)) goto error;
	cp=buf+i;

	/* certificate [0] explicit OPTIONAL */
	if(lo->cert){
		if(lo->cert->der==NULL){
			OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_LCMP,ERR_PT_LCMPASE+3,NULL);
			goto error;
		}
		ASN1_skip_(lo->cert->der,&j);
		memcpy(cp,lo->cert->der,j);
		ASN1_set_explicit(j,0,cp,&j);
		i+=j;
	}
	ASN1_set_explicit(i,LCMP_OP_RENEWALRSP,buf,ret_len);

	return 0;
error:
	return -1;
}

/*--------------------------------------*/
int flag_state2bit(int state, unsigned char *ret){
	if(state&STAT_CA)       ret[0] |= 0x80;
	if(state&STAT_EXPIRED)  ret[0] |= 0x40;
	if(state&STAT_REVOKED)  ret[0] |= 0x20;
	if(state&0x08)  ret[0] |= 0x10;
	if(state&STAT_XCERTFWD) ret[0] |= 0x08;
	if(state&STAT_XCERTREV) ret[0] |= 0x04;
	if(state&0x40)  ret[0] |= 0x02;
	if(state&0x80)  ret[0] |= 0x01;
	if(state&STAT_HAVEKEY)  ret[1] |= 0x80;
	return 0;
}

int lcmp_DER_statlist(CertStat *top,unsigned char *ret,int *ret_len){
	unsigned char *cp,*sq,flags[4],tmp[320];
	CertStat *st;
	int	i,j,k;

	sq = ret;
	*ret_len=0;
	if(top==NULL) return 0;

	for(i=0,st=top; st ;st=st->next){
		memset(flags,0,4);
		flag_state2bit(st->state,flags);
		ASN1_set_bitstring(7,2,flags,sq,&j);
		cp = sq+j;
		ASN1_set_integer(st->serialNum,cp,&k);
		cp+=k; j+=k;
		if((k=UC_conv(UC_LOCAL_CODESET,UC_CODE_UTF8,st->subject,strlen(st->subject),tmp,256))<0) goto error;
		if(ASN1_set_utf8(tmp,cp,&k)) goto error;
		cp+=k; j+=k;
		if(Cert_DER_time(&st->notBefore,cp,&k)) goto error;
		cp+=k; j+=k;
		if(Cert_DER_time(&st->notAfter,cp,&k)) goto error;
		cp+=k; j+=k;

		if(st->revokedDate.tm_year){
			if(Cert_DER_time(&st->revokedDate,cp,&k)) goto error;
			cp+=k; j+=k;
		}
		if(st->acceptID>0){
			ASN1_set_integer(st->acceptID,cp,&k);
			*cp=0x80; cp+=k; j+=k;
		}
		if(st->state&0xff0000){
			ASN1_set_integer((st->state>>16)&0xff,cp,&k);
			*cp=0x81; cp+=k; j+=k;
		}
		if(st->keyID){
			ASN1_set_octetstring(20,st->keyID,cp,&k);
			*cp=0x82; cp+=k; j+=k;
		}
		ASN1_set_sequence(j,sq,&j);
		sq+=j; i+=j;
	}
	ASN1_set_explicit(i,2,ret,ret_len);
	return 0;
error:
	return -1;
}

int lcmp_DER_tmpl(ProfTmpl *pf,unsigned char *buf,int *ret_len){
	unsigned char *cp,*sq;
	int i,j,k;

	cp=buf; i=0;
	/* templateName LCMPString
	 * -- not optional, but AddProfContent and ModProfContent share
	 * -- this structure. */
	if(pf->tmplName){
		ASN1_set_octetstring(strlen(pf->tmplName),(unsigned char*)pf->tmplName,cp,&j);
		cp+=j; i+=j;
	}
	
	/* version [0] INTEGER OPTIONAL */
	if(pf->certVer>=0){
		ASN1_set_integer(pf->certVer,cp,&j);
		cp[0]=0x80; cp+=j; i+=j;
	}
	
	/* baseNumber [1] INTEGER OPTIONAL */
	if(pf->baseNum>=0){
		ASN1_set_integer(pf->baseNum,cp,&j);
		cp[0]=0x81; cp+=j; i+=j;
	}

	/* validity [2] StartAndEndTime OPTIONAL */
	if((pf->bgtype>=0)||(pf->edtype>=0)||(pf->updtype>=0)){
		sq = cp; j=k=0;
		if(pf->bgtype>=0){
			switch(pf->bgtype){
			case 0:
			  ASN1_set_null(cp); j=2;
			  break;
			case 1:
			  if(stm2UTC(&pf->start,cp,ASN1_GENERALIZEDTIME)==NULL) goto error;
			  ASN1_skip_(cp,&j);
			  break;
			case 2:
			  if(stm2UTC(&pf->start,cp,ASN1_PRINTABLE_STRING)==NULL) goto error;
			  ASN1_skip_(cp,&j);
			  break;
			}
			ASN1_set_explicit(j,0,cp,&j); cp+=j;
		}
		if(pf->edtype>=0){
			switch(pf->edtype){
			case 0:
			  ASN1_set_integer(pf->certSec,cp,&k);
			  break;
			case 1:
			  if(stm2UTC(&pf->end,cp,ASN1_GENERALIZEDTIME)==NULL) goto error;
			  ASN1_skip_(cp,&k);
			  break;
			case 2:
			  if(stm2UTC(&pf->end,cp,ASN1_PRINTABLE_STRING)==NULL) goto error;
			  ASN1_skip_(cp,&k);
			  break;
			}
			ASN1_set_explicit(k,1,cp,&k); cp+=k; j+=k;
		}
		if(pf->updtype==0){
			ASN1_set_integer(pf->updSec,cp,&k);
			ASN1_set_explicit(k,2,cp,&k); cp+=k; j+=k;
		}
		ASN1_set_explicit(j,2,sq,&j);
		cp=sq+j; i+=j;
	}
	
	/* keyType [3] KeyType OPTIONAL */
	if(pf->keyType>0){
		sq = cp;
		if(ASN1_int_2object(pf->keyType,cp,&j)) goto error;
		cp+=j;
		ASN1_set_integer(pf->keyLength,cp,&k);
		ASN1_set_explicit(j+k,3,sq,&j);
		cp=sq+j; i+=j;
	}
	
	/* sigHashType [4] OBJECT IDENTIFIER OPTIONAL */
	if(pf->hashType>0){
		if(ASN1_int_2object(pf->hashType,cp,&j)) goto error;
		cp[0]=0x84; i+=j; cp+=j;
	}

	/* subjectTemplate [5] Name OPTIONAL */
	if(pf->sbjTmpl.num>0){
		if(Cert_DER_subject(&pf->sbjTmpl,cp,&j)) goto error;
		cp[0]=0xa5; i+=j; cp+=j;
	}

	/* profilePolicyFlag [6] BIT STRING OPTIONAL */
	for(k=0xff,j=0; j<8; j++) k &= pf->policyFlag[j];
	if(k != 0xff){
		/* currently 8 byte string is located */
		ASN1_set_bitstring(0,8,pf->policyFlag,cp,&j);
		cp[0]=0x86; i+=j; cp+=j;
	}
	ASN1_set_sequence(i,buf,ret_len);
	return 0;
error:
	return -1;
}

int lcmp_DER_delext(CertExt *ext,unsigned char *buf,int *ret_len){
	unsigned char *cp;
	int i,j;

	for(cp=buf,i=0; ext ;ext=ext->next){
		j=0;
		if((ext->extnID<=0)&&(ext->objid==NULL)) continue;

		if(ext->objid){
			ASN1_skip_(ext->objid,&j);
			memcpy(cp,ext->objid,j);
		}else{
			if(ASN1_int_2object(ext->extnID,cp,&j)) goto error;
		}
		cp+=j; i+=j;
	}
	ASN1_set_set(i,buf,ret_len);
	
	return 0;
error:
	return -1;
}

int lcmp_DER_keytype(Key *key,unsigned char *buf,int *ret_len){
	unsigned char *cp;
	int i,j,kobj;

	/* type OBJECT IDENTIFIER */
	switch(key->key_type){
	case KEY_RSA_PRV:   kobj=OBJ_CRYPT_RSA; break;
	case KEY_DSA_PRV:   kobj=OBJ_CRYPT_DSA; break;
	case KEY_ECDSA_PRV: kobj=OBJ_CRYPT_ECDSA; break;
	default:
		OK_set_error(ERR_ST_BADPARAM,ERR_LC_LCMP,ERR_PT_LCMPASE+11,NULL);
		goto error;
	}
	if(ASN1_int_2object(kobj,buf,&i)) goto error;
	cp=buf+i;
	
	/* integer */
	ASN1_set_integer(key->size<<3,cp,&j);
	cp+=j; i+=j;

	/* key low DER OPTIONAL */
	switch(key->key_type){
	case KEY_RSA_PRV:   if(RSAprv_toDER((Prvkey_RSA*)key,cp,&j)==NULL) goto error; break;
	case KEY_DSA_PRV:   if(DSAprv_toDER((Prvkey_DSA*)key,cp,&j)==NULL) goto error; break;
	case KEY_ECDSA_PRV: if(ECDSAprv_toDER((Prvkey_ECDSA*)key,cp,&j)==NULL) goto error; break;
	}
	i+=j;
	
	ASN1_set_sequence(i,buf,ret_len);
	
	return 0;
error:
	return -1;
}

int lcmp_DER_userinfo(AuthInfo *ai,unsigned char *buf,int *ret_len){
	unsigned char *cp;
	int i,j;

	cp=buf; i=0;
	/* userName LCMPString, */
	if(ai->name==NULL){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_LCMP,ERR_PT_LCMPASE+13,NULL);
		goto error;
	}
	ASN1_set_octetstring(strlen(ai->name),(unsigned char*)ai->name,cp,&j);
	cp+=j; i+=j;

	/* userPassword [0] LCMPString OPTIONAL */
	if(ai->passwd){
		ASN1_set_octetstring(strlen(ai->passwd),(unsigned char*)ai->passwd,cp,&j);
		cp[0]=0x80; cp+=j; i+=j;
	}
	/* userOldPassword [1] LCMPString OPTIONAL */
	if(ai->oldpasswd){
		ASN1_set_octetstring(strlen(ai->oldpasswd),(unsigned char*)ai->oldpasswd,cp,&j);
		cp[0]=0x81; cp+=j; i+=j;
	}
	/* userID [2] INTEGER OPTIONAL */
	if(ai->uid>=0){
		ASN1_set_integer(ai->uid,cp,&j);
		cp[0]=0x82; cp+=j; i+=j;
	}
	/* userGroupID [3] INTEGER OPTIONAL */
	if(ai->gid>=0){
		ASN1_set_integer(ai->gid,cp,&j);
		cp[0]=0x83; cp+=j; i+=j;
	}
	/* userGrant [4] INTEGER OPTIONAL */
	if(!(ai->grant&0x80000000)){
		ASN1_set_integer(ai->grant,cp,&j);
		cp[0]=0x84; cp+=j; i+=j;
	}
	/* optionalInfo [5] OCTET STRING OPTIONAL */
	if(ai->opt){
		ASN1_set_octetstring(strlen(ai->opt),(unsigned char*)ai->opt,cp,&j);
		cp[0]=0x85; cp+=j; i+=j;
	}
	ASN1_set_sequence(i,buf,ret_len);
	
	return 0;
error:
	return -1;
}

/*-----------------------------------------------
	estimate der size
-----------------------------------------------*/
int LCMPop_estimate_der_size(LCMPOp *op){
	int i,j,k,ret = 32;

	if(op==NULL){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_LCMP,ERR_PT_LCMPASE+14,NULL);
		return -1;
	}

	i = 4;
	switch(op->opId){
	case LCMP_OP_BINDREQ:
		if(((LO_BindReq*)op)->caname) i += strlen(((LO_BindReq*)op)->caname) + 4;
		if(((LO_BindReq*)op)->passwd) i += strlen(((LO_BindReq*)op)->passwd) + 4;
		if(((LO_BindReq*)op)->username) i += strlen(((LO_BindReq*)op)->username) + 4;
		break;
		
	case LCMP_OP_BINDRSP:
		if((i=lcmp_dsz_certlist(((LO_BindRsp*)op)->list))<0) goto error;
		break;
		
	case LCMP_OP_UNBIND:
		break;
		
	case LCMP_OP_SIGNREQ:
		i+=8; /* serialNum */
		if(((LO_SignReq*)op)->profName) i += strlen(((LO_SignReq*)op)->profName) + 4;
		if(((LO_SignReq*)op)->p10){
			if(((LO_SignReq*)op)->p10->der == NULL) goto error;
			ASN1_skip_(((LO_SignReq*)op)->p10->der,&j);
			i+=j;
		}
		if(((LO_SignReq*)op)->tmpl){
			if((j=der_size_certtmpl(((LO_SignReq*)op)->tmpl))<0) goto error;
			i+=j;
		}
		break;
		
	case LCMP_OP_SIGNRSP:
		if(((LO_SignRsp*)op)->cert){
			if(((LO_SignRsp*)op)->cert->der == NULL) goto error;
			ASN1_skip_(((LO_SignRsp*)op)->cert->der,&i);
		}
		break;
		
	case LCMP_OP_LISTREQ:
		i+=8; /* serialNum */
		if(((LO_ListReq*)op)->profName) i += strlen(((LO_ListReq*)op)->profName) + 4;
		if(((LO_ListReq*)op)->sbjQuery) i += strlen(((LO_ListReq*)op)->sbjQuery) + 4;
		break;
		
	case LCMP_OP_LISTRSP:
		if((i=lcmp_dsz_certstat(((LO_ListRsp*)op)->stat))<0) goto error;
		break;
		
	case LCMP_OP_PROFREQ:
		if(((LO_ProfReq*)op)->profName) i += strlen(((LO_ProfReq*)op)->profName) + 4;
		if((j=lcmp_dsz_prof(((LO_ProfReq*)op)->prof))<0) goto error;
		i+=j;
		if((j=lcmp_dsz_ext(((LO_ProfReq*)op)->ext))<0) goto error;
		i+=j;
		break;
		
	case LCMP_OP_PROFRSP:
		if((j=lcmp_dsz_prof(((LO_ProfRsp*)op)->prof))<0) goto error;
		i+=j;
		if((j=lcmp_dsz_ext(((LO_ProfRsp*)op)->ext))<0) goto error;
		i+=j;
		if(((LO_ProfRsp*)op)->proflist){
			for(k=0; k<((LO_ProfRsp*)op)->listnum; k++)
				if(((LO_ProfRsp*)op)->proflist[k])
					i+=strlen(((LO_ProfRsp*)op)->proflist[k])+4;
		}
		break;
		
	case LCMP_OP_CERTREQ:
		if(((LO_CertReq*)op)->passwd) i += strlen(((LO_CertReq*)op)->passwd) + 4;
		if((j=lcmp_dsz_key(((LO_CertReq*)op)->key))<0) goto error;
		i += j + 32;
		break;
		
	case LCMP_OP_CERTRSP:
		if(((LO_CertRsp*)op)->cert){
			if(((LO_CertRsp*)op)->cert->der == NULL) goto error;
			ASN1_skip_(((LO_CertRsp*)op)->cert->der,&i);
		}
		if((j=lcmp_dsz_key(((LO_CertRsp*)op)->key))<0) goto error;
		i += j;
		break;
		
	case LCMP_OP_CSRREQ:
		if(((LO_CSRReq*)op)->csr){
			if(((LO_CSRReq*)op)->csr->der == NULL) goto error;
			ASN1_skip_(((LO_CSRReq*)op)->csr->der,&j);
			i += j;
		}
		if(((LO_CSRReq*)op)->tmpl){
			if((j=der_size_certtmpl(((LO_CSRReq*)op)->tmpl))<0) goto error;
			i+=j;
		}
		i += 48;
		break;
		
	case LCMP_OP_CSRRSP:
		if(((LO_CSRRsp*)op)->csr){
			if(((LO_CSRRsp*)op)->csr->der == NULL) goto error;
			ASN1_skip_(((LO_CSRRsp*)op)->csr->der,&j);
			i += j;
		}
		if(((LO_CSRRsp*)op)->tmpl){
			if((j=der_size_certtmpl(((LO_CSRRsp*)op)->tmpl))<0) goto error;
			i+=j;
		}
		i += 32;
		break;

	case LCMP_OP_CRLREQ:
		if(((LO_CRLReq*)op)->profName) i += strlen(((LO_CRLReq*)op)->profName) + 8;
		if(((LO_CRLReq*)op)->path) i += strlen(((LO_CRLReq*)op)->path) + 4;
		break;
		
	case LCMP_OP_CRLRSP:
		if((i=lcmp_dsz_crllist(((LO_CRLRsp*)op)->list))<0) goto error;
		break;
		
	case LCMP_OP_SVOPREQ:
		if((j=lcmp_dsz_userinfo(((LO_SVOpReq*)op)->uinfo))<0) goto error;
		i += j;
		break;
		
	case LCMP_OP_SVOPRSP:
		if(((LO_SVOpRsp*)op)->opRsp) i += ((LO_SVOpRsp*)op)->rspLen + 8;
		break;

	case LCMP_OP_EXTREQ:
		if(((LO_ExtReq*)op)->opOID) i += strlen(((LO_ExtReq*)op)->opOID) + 4;
		if(((LO_ExtReq*)op)->opInfo) i += ((LO_ExtReq*)op)->ifLen + 8;
		break;
		
	case LCMP_OP_EXTRSP:
		if(((LO_ExtRsp*)op)->opOID) i += strlen(((LO_ExtRsp*)op)->opOID) + 4;
		if(((LO_ExtRsp*)op)->opRsp) i += ((LO_ExtRsp*)op)->rspLen + 8;
		break;

	case LCMP_OP_RENEWALREQ:
		i+=8; /* serialNum */
		if(((LO_RenewalReq*)op)->p10){
			if(((LO_RenewalReq*)op)->p10->der == NULL) goto error;
			ASN1_skip_(((LO_RenewalReq*)op)->p10->der,&j);
			i+=j;
		}
		if(((LO_RenewalReq*)op)->tmpl){
			if((j=der_size_certtmpl(((LO_RenewalReq*)op)->tmpl))<0) goto error;
			i+=j;
		}
		break;
		
	case LCMP_OP_RENEWALRSP:
		if(((LO_RenewalRsp*)op)->cert){
			if(((LO_RenewalRsp*)op)->cert->der == NULL) goto error;
			ASN1_skip_(((LO_RenewalRsp*)op)->cert->der,&i);
		}
		break;
		
	}
	ret+=i;

	if(op->resultMsg) ret += strlen(op->resultMsg) + 4;
	if((i=lcmp_dsz_ctrl(op->ctrl))<0) goto error;
	ret+=i;
	
	return ret;
error:
	return -1;
}

int lcmp_dsz_ctrl(LCMPCtrl *ct){
	int ret=8;

	while(ct){
		if(ct->ctrlType) ret += strlen(ct->ctrlType) + 4;
		if(ct->crit)	ret += 4;
		if(ct->value) ret += ct->vlen + 4;
		ct = ct->next;
	}
	return ret;
}

int lcmp_dsz_certlist(CertList *cl){
	int i,ret=8;

	while(cl){
		if(cl->cert->der == NULL) return -1;
		
		ASN1_skip_(cl->cert->der,&i); ret+=i;
		cl = cl->next;
	}
	return ret;
}

int lcmp_dsz_crllist(CRLList *cl){
	int i,ret=8;

	while(cl){
		if(cl->crl->der == NULL) return -1;
		
		ASN1_skip_(cl->crl->der,&i); ret+=i;		
		cl = cl->next;
	}
	return ret;
}

int lcmp_dsz_ext(CertExt *ext){
	int ret=8;
	while(ext){
		ret += (ext->critical)?(4):(0);
		ret += ext->dlen + 16;
		ext = ext->next;
	}
	return ret;
}

int lcmp_dsz_prof(ProfTmpl *pf){
	int i,ret=84;

	if(pf==NULL) return 0;
	if(pf->tmplName) ret += strlen(pf->tmplName);
	if(pf->policyFlag[0] != 0xff) ret+= 10;
	if(pf->sbjTmpl.num > 0){
		for(i=0;i<pf->sbjTmpl.num;i++)
			if(pf->sbjTmpl.rdn[i].tag)
				ret+=strlen(pf->sbjTmpl.rdn[i].tag)+20;
	}
	return ret;
}

int lcmp_dsz_certstat(CertStat *st){
	int ret=8;

	while(st){
		ret += 80;
		if(st->subject) ret += strlen(st->subject);
		if(st->keyID) ret += 22;
		st = st->next;
	}
	return ret;
}

int lcmp_dsz_key(Key *key){
	int i,ret=8;
	
	if(key==NULL) return 0;
	
	switch(key->key_type){
	case KEY_RSA_PRV:
		ret+=((Prvkey_RSA*)key)->p->top*4*9+48;
		break;
	case KEY_DSA_PRV:
		if((i=DSAprv_estimate_der_size((Prvkey_DSA*)key))<0) goto error;
		ret+=i;
		break;	
	case KEY_ECDSA_PRV:
		if((i=ECDSAprv_estimate_der_size((Prvkey_ECDSA*)key))<0) goto error;
		ret+=i;
		break;
	default:
		goto error;
	}
	return ret;
error:
	return -1;
}

int lcmp_dsz_userinfo(AuthInfo *ai){
	int ret=32;

	if(ai==NULL) return 0;
	if(ai->name) ret += strlen(ai->name);
	if(ai->passwd) ret += strlen(ai->passwd);
	if(ai->oldpasswd) ret += strlen(ai->oldpasswd);
	if(ai->opt) ret += strlen(ai->opt);
	return ret;
}


