/* lcmp_asn1d.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *	this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *	this list of conditions and the following disclaimer in the documentation
 *	and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *	display the following acknowledgment:
 *	"This product includes software developed by Akira Iwata Laboratory,
 *	Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *	acknowledgment:
 *	"This product includes software developed by Akira Iwata Laboratory,
 *	 Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>

#include "ok_caerr.h"
#include "ok_lcmp.h"

/*-----------------------------------------------
  LCMP ASN1 message decoding
-----------------------------------------------*/
LCMPOp *ASN1_LCMP_msg(unsigned char *msg){
	unsigned char *cp;
	LCMPOp *ret=NULL;
	int i,op,msgid=0;

	if(msg[0]!=0x30){
		OK_set_error(ERR_ST_BADFORMAT,ERR_LC_LCMP,ERR_PT_LCMPASD,NULL);
		goto error;
	}
  
	if((cp=ASN1_next(msg))==NULL) goto error;
	if((msgid = ASN1_integer(cp,&i))<0) goto error;

	if((cp=ASN1_next(cp))==NULL) goto error;
	if((*cp & 0xc0)!=0x80){
		OK_set_error(ERR_ST_BADFORMAT,ERR_LC_LCMP,ERR_PT_LCMPASD,NULL);
		goto error; /* check context-specific */
	}
	op = *cp & 0x1f;
	if((ret=LCMP_op_new(op))==NULL) goto error;

	switch(op){
	case LCMP_OP_BINDREQ: if(asn1_lcmp_bindreq(ret,cp)) goto error; break;
	case LCMP_OP_BINDRSP: if(asn1_lcmp_bindrsp(ret,cp)) goto error; break;
	case LCMP_OP_UNBIND:  break;
	case LCMP_OP_SIGNREQ: if(asn1_lcmp_signreq(ret,cp)) goto error; break;
	case LCMP_OP_SIGNRSP: if(asn1_lcmp_signrsp(ret,cp)) goto error; break;
	case LCMP_OP_LISTREQ: if(asn1_lcmp_listreq(ret,cp)) goto error; break;
	case LCMP_OP_LISTRSP: if(asn1_lcmp_listrsp(ret,cp)) goto error; break;
	case LCMP_OP_PROFREQ: if(asn1_lcmp_profreq(ret,cp)) goto error; break;
	case LCMP_OP_PROFRSP: if(asn1_lcmp_profrsp(ret,cp)) goto error; break;
	case LCMP_OP_CERTREQ: if(asn1_lcmp_certreq(ret,cp)) goto error; break;
	case LCMP_OP_CERTRSP: if(asn1_lcmp_certrsp(ret,cp)) goto error; break;
	case LCMP_OP_CSRREQ:  if(asn1_lcmp_csrreq(ret,cp)) goto error; break;
	case LCMP_OP_CSRRSP:  if(asn1_lcmp_csrrsp(ret,cp)) goto error; break;
	case LCMP_OP_CRLREQ:  if(asn1_lcmp_crlreq(ret,cp)) goto error; break;
	case LCMP_OP_CRLRSP:  if(asn1_lcmp_crlrsp(ret,cp)) goto error; break;
	case LCMP_OP_SVOPREQ: if(asn1_lcmp_svopreq(ret,cp)) goto error; break;
	case LCMP_OP_SVOPRSP: if(asn1_lcmp_svoprsp(ret,cp)) goto error; break;
	case LCMP_OP_EXTREQ:  if(asn1_lcmp_extreq(ret,cp)) goto error; break;
	case LCMP_OP_EXTRSP:  if(asn1_lcmp_extrsp(ret,cp)) goto error; break;
	case LCMP_OP_RENEWALREQ: if(asn1_lcmp_renewalreq(ret,cp)) goto error; break;
	case LCMP_OP_RENEWALRSP: if(asn1_lcmp_renewalrsp(ret,cp)) goto error; break;
	}

	ret->messageId = msgid;
	if((ret->der=ASN1_dup(cp))==NULL) goto error;

	if((cp=ASN1_skip(cp))==NULL) goto error;
	if(*cp == 0xa0)
		if(asn1_lcmp_ctrl(ret,cp)) goto error;

	return ret;
error:
	LCMP_op_free(ret);
	return NULL;
}

/*-----------------------------------------------
  LCMP ASN1 control decoding
-----------------------------------------------*/
int asn1_lcmp_ctrl(LCMPOp *op, unsigned char *in){
	unsigned char *sq,*cp;
	LCMPCtrl *hd=NULL,*ct=NULL;
	int len,i,j,k;

	if(*in != 0xa0) return 0;

	if((len=ASN1_tlen(in))<0) goto error;
	if((sq =ASN1_next(in))==NULL) goto error;

	for(i=0; i<len;){
		if((ct=LCMP_ctrl_new())==NULL) goto error;
		if(hd){
			hd->next=ct; hd=ct;
		}else{
			op->ctrl=hd=ct;
		}

		if((cp=ASN1_next(sq))==NULL) goto error;

		/* controlType LCMPOID */
		if(ASN1_octetstring(cp,&j,(unsigned char**)&ct->ctrlType,&k)) goto error;
		if((cp=ASN1_next(cp))==NULL) goto error;

		/* criticality BOOLEAN DEFAULT FALSE */
		if((ct->crit=ASN1_boolean(cp,&j))<0) goto error;
		if((cp=ASN1_next(cp))==NULL) goto error;

		if(*cp == ASN1_OCTETSTRING) /* OPTIONAL */
			if(ASN1_octetstring(cp,&j,&ct->value,&ct->vlen)) goto error;

		if((sq=ASN1_skip_(sq,&j))==NULL) goto error;
		i+=j;
	}

	return 0;
error:
	LCMP_ctrl_free_all(op->ctrl); op->ctrl=NULL;
	return -1;
}

/*-----------------------------------------------
  LCMP ASN1 response message decoding
-----------------------------------------------*/
int asn1_lcmp_result(LCMPOp *op, unsigned char *cp){
	int i,j;

	if((cp=ASN1_next(cp))==NULL) goto error;
	if((op->resultCode=ASN1_enumerated(cp,&i))<0) goto error;

	if((cp=ASN1_next(cp))==NULL) goto error;
	if(ASN1_octetstring(cp,&i,(unsigned char**)&op->resultMsg,&j)) goto error;

	return 0;
error:
	return -1;
}

/*-----------------------------------------------
  LCMP ASN1 operations decoding
-----------------------------------------------*/
int asn1_lcmp_bindreq(LCMPOp *op, unsigned char *req){
	LO_BindReq *lo = (LO_BindReq*)op;
	unsigned char *cp;
	int i,j;

	if((cp=ASN1_next(req))==NULL) goto error;

	if((lo->version=ASN1_integer(cp,&i))<0) goto error;
	if((cp=ASN1_next(cp))==NULL) goto error;

	if(ASN1_octetstring(cp,&i,(unsigned char**)&lo->caname,&j)) goto error;
	if((cp=ASN1_next(cp))==NULL) goto error;

	if(*cp==ASN1_OCTETSTRING){
		if(ASN1_octetstring(cp,&i,(unsigned char**)&lo->username,&j)) goto error;
		if((cp=ASN1_next(cp))==NULL) goto error;
	}

	lo->authMode = *cp & 0x1f;
	switch(lo->authMode){
	case LCMP_OPBRQ_ATSIMPLE: /* Simple Password Authentication */
		if(ASN1_octetstring_(cp,&i,(unsigned char**)&lo->passwd,&j,1)) goto error;
		break;
	case LCMP_OPBRQ_ATSSLCL: /* SSL Client Authentication */
		break;
	default:
		OK_set_error(ERR_ST_BADFORMAT,ERR_LC_LCMP,ERR_PT_LCMPASD+2,NULL);
		goto error;
	}

	return 0;
error:
	return -1;
}

int asn1_lcmp_bindrsp(LCMPOp *op, unsigned char *rsp){
	LO_BindRsp *lo = (LO_BindRsp*)op;
	unsigned char *cp,*sq;
	CertList *cl,*hd=NULL;
	Cert *ct=NULL;
	int len,i,j;

	if(asn1_lcmp_result(op,rsp)) goto error;
	if((cp=ASN1_step(rsp,3))==NULL) goto error;
	
	if(*cp == 0xa0){ /* SET OF Certificate OPTIONAL */
		if((len=ASN1_tlen(cp))<0) goto error;
		if((sq =ASN1_next(cp))==NULL) goto error;

		for(i=0; i<len;){
			if((ct=ASN1_read_cert(sq))==NULL) goto error;
			if((ct->der=ASN1_dup(sq))==NULL) goto error;
			if((cl=Cert_2Certlist(ct))==NULL) goto error;
			if(hd){
				hd->next=cl; hd=cl;
			}else{
				lo->list=hd=cl;
			}
			Cert_free(ct); ct=NULL;

			if((sq=ASN1_skip_(sq,&j))==NULL) goto error;
			i+=j;
		}
	}
	return 0;
error:
	Certlist_free_all(lo->list); lo->list=NULL;
	Cert_free(ct);
	return -1;
}

int asn1_lcmp_signreq(LCMPOp *op, unsigned char *req){
	LO_SignReq *lo = (LO_SignReq*)op;
	unsigned char *cp,*der=NULL;
	int type,i,j;

	if((cp=ASN1_next(req))==NULL) goto error;

	if(ASN1_octetstring(cp,&i,(unsigned char**)&lo->profName,&j)) goto error;
	if((cp=ASN1_next(cp))==NULL) goto error;

	if((type=ASN1_enumerated(cp,&i))<0) goto error;
	if((cp=ASN1_next(cp))==NULL) goto error;

	if(ASN1_octetstring(cp,&i,&der,&j)) goto error;
	if((cp=ASN1_next(cp))==NULL) goto error;
	
	switch(type){
	case PKCS10CSR:
		if((lo->p10=ASN1_read_req(der))==NULL) goto error;
		der = NULL;
		break;
	case CMPREQUEST:
		if((lo->tmpl=ASN1_cmp_certtmpl(der,&i))==NULL) goto error;
		der = NULL;
		break;
	default:
		OK_set_error(ERR_ST_UNSUPPORTED_PARAM,ERR_LC_LCMP,ERR_PT_LCMPASD+3,NULL);
		goto error;
	}

	if(*cp==0x80){ /* [0] INTEGER OPTIONAL */
		if((lo->serialNum=ASN1_integer_(cp,&i,1))<0) goto error;
	}
	return 0;
error:
	if(der) free(der);
	return -1;
}

int asn1_lcmp_signrsp(LCMPOp *op, unsigned char *rsp){
	LO_SignRsp *lo = (LO_SignRsp*)op;
	unsigned char *cp;

	if(asn1_lcmp_result(op,rsp)) goto error;
	if((cp=ASN1_step(rsp,3))==NULL) goto error;

	if(*cp == 0xa0){ /* OPTIONAL */
		if((cp=ASN1_next(cp))==NULL) goto error;
		if((lo->cert=ASN1_read_cert(cp))==NULL) goto error;
		if((lo->cert->der=ASN1_dup(cp))==NULL) goto error;
	}
	return 0;
error:
	Cert_free(lo->cert); lo->cert=NULL;
	return -1;
}

int asn1_lcmp_listreq(LCMPOp *op, unsigned char *req){
	LO_ListReq *lo = (LO_ListReq*)op;
	unsigned char *cp;
	int i,j;

	if((cp=ASN1_next(req))==NULL) goto error;

	if(ASN1_octetstring(cp,&i,(unsigned char**)&lo->profName,&j)) goto error;
	if((cp=ASN1_next(cp))==NULL) goto error;

	/* query CHOICE { */
	if(*cp == ASN1_INTEGER){ /* OPTIONAL */
		if((lo->serialNum=ASN1_integer(cp,&i))<0) goto error;
	}
	else if(*cp == ASN1_UTF8STRING){ /* OPTIONAL */
		if((lo->sbjQuery=ASN1_utf8(cp,&i))==NULL) goto error;
	}
	return 0;
error:
	return -1;
}

int asn1_lcmp_listrsp(LCMPOp *op, unsigned char *rsp){
	LO_ListRsp *lo = (LO_ListRsp*)op;
	unsigned char *cp,*der=NULL;
	int ok=-1;

	if(asn1_lcmp_result(op,rsp)) goto done;
	if((cp=ASN1_step(rsp,3))==NULL) goto done;

	if(*cp == 0xa0){ /* [0] StatusList OPTIONAL */
		if((lo->stat=asn1_lcmp_statlist(cp))==NULL) goto done;
	}
	ok=0;
done:
	if(der) free(der);
	return ok;
}

int asn1_lcmp_profreq(LCMPOp *op, unsigned char *req){
	LO_ProfReq *lo = (LO_ProfReq*)op;
	unsigned char *cp;
	int i,j;

	if((cp=ASN1_next(req))==NULL) goto error;

	if(ASN1_octetstring(cp,&i,(unsigned char**)&lo->profName,&j)) goto error;
	if((cp=ASN1_next(cp))==NULL) goto error;

	lo->profOp = *cp & 0x1f;
	switch(lo->profOp){
	case LCMP_OPPF_VIEWPF: break; /* [0] NULL */
	case LCMP_OPPF_ADDPF:  if((lo->prof=asn1_lcmp_prof(cp))==NULL) goto error; break;
	case LCMP_OPPF_DELPF:  break; /* DelProfContent ::= NULL */
	case LCMP_OPPF_MODPF:  if((lo->prof=asn1_lcmp_prof(cp))==NULL) goto error; break;
	case LCMP_OPPF_VIEWEXT: break; /* [4] NULL */
	case LCMP_OPPF_ADDEXT: if((lo->ext=asn1_get_exts(cp,&i))==NULL) goto error; break;
	case LCMP_OPPF_DELEXT: if((lo->ext=asn1_lcmp_delext(cp))==NULL) goto error; break;
	case LCMP_OPPF_UPDEXT: if((lo->ext=asn1_get_exts(cp,&i))==NULL) goto error; break;
	case LCMP_OPPF_PFLIST: break; /* [8] NULL */
	case LCMP_OPPF_RENAMEPF: if(ASN1_octetstring_(cp,&i,(unsigned char**)&lo->profRename,&j,1)) goto error; break;
	default:
		OK_set_error(ERR_ST_UNSUPPORTED_PARAM,ERR_LC_LCMP,ERR_PT_LCMPASD+5,NULL);
		goto error;
	}
	return 0;
error:
	return -1;
}

int asn1_lcmp_profrsp(LCMPOp *op, unsigned char *rsp){
	LO_ProfRsp *lo = (LO_ProfRsp*)op;
	unsigned char *cp,*t;
	int i,j,k,len;

	if(asn1_lcmp_result(op,rsp)) goto error;
	if((cp=ASN1_step(rsp,3))==NULL) goto error;

	/* viewProfile [0] ModProfContent OPTIONAL */
	if(*cp == 0xa0){
		if((lo->prof=asn1_lcmp_prof(cp))==NULL) goto error;
		if((cp=ASN1_skip(cp))==NULL) goto error;
	}
	/* viewExtensions [1] Extensions OPTIONAL */
	if(*cp == 0xa1){
		if((lo->ext=asn1_get_exts(cp,&i))==NULL) goto error;
		if((cp=ASN1_skip(cp))==NULL) goto error;
	}
	/* listedNames [2] SET OF LCMPString OPTIONAL */
	if(*cp == 0xa2){
		/* check count */
		if((len = ASN1_tlen(cp))<0) goto error;
		if((t   = ASN1_next(cp))==NULL) goto error;
		for(i=j=0; j<len; j+=k,i++)
			if((t=ASN1_skip_(t,&k))==NULL) goto error;

		/* get name list */
		lo->listnum = len = i;
		if((lo->proflist=(char**)malloc(sizeof(char*)*i))==NULL){
			OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMPASD+6,NULL);
			goto error;
		}

		if((t = ASN1_next(cp))==NULL) goto error;
		for(i=0; i<len; i++){
			if(ASN1_octetstring(t,&j,(unsigned char**)&lo->proflist[i],&k)) goto error;
			if((t = ASN1_next(t))==NULL) goto error;
		}
	}
	return 0;
error:
	return -1;
}

int asn1_lcmp_certreq(LCMPOp *op, unsigned char *req){
	LO_CertReq *lo = (LO_CertReq*)op;
	unsigned char *cp,*sq;
	int i,j;

	if((cp=ASN1_next(req))==NULL) goto error;

	if((lo->serialNum=ASN1_integer(cp,&i))<0) goto error;
	if((cp=ASN1_next(cp))==NULL) goto error;

	if((lo->certOp=ASN1_enumerated(cp,&i))<0) goto error;
	if((cp=ASN1_next(cp))==NULL) goto error;

	if(*cp == 0x80){ /* [0] CRLReason OPTIONAL */
		if((lo->revokeReason=ASN1_integer_(cp,&i,1))<0) goto error;
		if((cp=ASN1_next(cp))==NULL) goto error;
	}
	if(*cp == 0x81){ /* [1] LCMPString OPTIONAL */
		if(ASN1_octetstring_(cp,&i,(unsigned char**)&lo->passwd,&j,1)) goto error;
		if((cp=ASN1_next(cp))==NULL) goto error;
	}
	if(*cp == 0xa3){ /* [3] EXPLICIT KeyType OPTIONAL */
		if((sq=ASN1_next(cp))==NULL) goto error;
		if((lo->key=asn1_lcmp_keytype(sq))==NULL) goto error;
		if((cp=ASN1_skip(cp))==NULL) goto error;
	}
	return 0;
error:
	return -1;
}

int asn1_lcmp_certrsp(LCMPOp *op, unsigned char *rsp){
	LO_CertRsp *lo = (LO_CertRsp*)op;
	unsigned char *cp,*sq;

	if(asn1_lcmp_result(op,rsp)) goto error;
	if((cp=ASN1_step(rsp,3))==NULL) goto error;

	if(*cp == 0xa0){ /* [0] EXPLICIT Certificate OPTIONAL */
		if((sq=ASN1_next(cp))==NULL) goto error;
		if((lo->cert=ASN1_read_cert(sq))==NULL) goto error;
		if((lo->cert->der=ASN1_dup(sq))==NULL) goto error;
		if((cp=ASN1_skip(cp))==NULL) goto error;
	}
	if(*cp == 0xa1){ /* [1] EXPLICIT KeyType OPTIONAL */
		if((sq=ASN1_next(cp))==NULL) goto error;
		if((lo->key=asn1_lcmp_keytype(sq))==NULL) goto error;
		if((cp=ASN1_skip(cp))==NULL) goto error;
	}
	return 0;
error:
	return -1;
}

int asn1_lcmp_csrreq(LCMPOp *op, unsigned char *req){
	LO_CSRReq *lo = (LO_CSRReq*)op;
	unsigned char *cp,*sq;
	int i,j;

	if((cp=ASN1_next(req))==NULL) goto error;

	if((lo->acceptID=ASN1_integer(cp,&i))<0) goto error;
	if((cp=ASN1_next(cp))==NULL) goto error;

	/* serial number OPTIONAL */
	if(*cp == ASN1_INTEGER){
		if((lo->serialNum=ASN1_integer(cp,&i))<0) goto error;
		if((cp=ASN1_next(cp))==NULL) goto error;
	}

	if((lo->csrOp=ASN1_enumerated(cp,&i))<0) goto error;
	if((cp=ASN1_next(cp))==NULL) goto error;

	if(*cp == 0xa0){ /* [0] EXPLICIT CertificationRequest OPTIONAL */
		if((sq=ASN1_next(cp))==NULL) goto error;
		if((lo->csr=ASN1_read_req(sq))==NULL) goto error;
		if((lo->csr->der=ASN1_dup(sq))==NULL) goto error;
		if((cp=ASN1_skip(cp))==NULL) goto error;
	}
	if(*cp == 0x81){ /* [1] LCMPSTRING OPTIONAL */
		if(ASN1_octetstring_(cp,&i,(unsigned char**)&lo->profName,&j,1)) goto error;
		if((cp=ASN1_next(cp))==NULL) goto error;
	}
	if(*cp == 0x82){ /* [2] INTEGER OPTIONAL */
		if((lo->rejectReason=ASN1_integer_(cp,&i,1))<0) goto error;
		if((cp=ASN1_next(cp))==NULL) goto error;
	}
	if(*cp == 0xa3){ /* [3] EXPLICIT CertTemplate OPTIONAL */
		if((sq=ASN1_next(cp))==NULL) goto error;
		if((lo->tmpl=ASN1_cmp_certtmpl(sq,&i))==NULL) goto error;
		if((lo->tmpl->der=ASN1_dup(sq))==NULL) goto error;
	}
	return 0;
error:
	return -1;
}

int asn1_lcmp_csrrsp(LCMPOp *op, unsigned char *rsp){
	LO_CSRRsp *lo = (LO_CSRRsp*)op;
	unsigned char *cp,*sq;
	int i;

	if(asn1_lcmp_result(op,rsp)) goto error;
	if((cp=ASN1_step(rsp,3))==NULL) goto error;

	if((lo->acceptID=ASN1_integer(cp,&i))<0) goto error;
	if((cp=ASN1_next(cp))==NULL) goto error;

	/* serial number OPTIONAL */
	if(*cp == ASN1_INTEGER){
		if((lo->serialNum=ASN1_integer(cp,&i))<0) goto error;
		if((cp=ASN1_next(cp))==NULL) goto error;
	}

	if(*cp == 0xa0){ /* [0] EXPLICIT CertificateRequest OPTIONAL */
		if((sq=ASN1_next(cp))==NULL) goto error;
		if((lo->csr=ASN1_read_req(sq))==NULL) goto error;
		if((lo->csr->der=ASN1_dup(sq))==NULL) goto error;
		if((cp=ASN1_skip(cp))==NULL) goto error;
	}
	if(*cp == 0xa1){ /* [1] EXPLICIT CertTemplate OPTIONAL */
		if((sq=ASN1_next(cp))==NULL) goto error;
		if((lo->tmpl=ASN1_cmp_certtmpl(sq,&i))==NULL) goto error;
		if((lo->tmpl->der=ASN1_dup(sq))==NULL) goto error;
	}
	return 0;
error:
	return -1;
}

int asn1_lcmp_crlreq(LCMPOp *op, unsigned char *req){
	LO_CRLReq *lo = (LO_CRLReq*)op;
	unsigned char *cp;
	int i,j;

	if((cp=ASN1_next(req))==NULL) goto error;

	if(ASN1_octetstring(cp,&i,(unsigned char**)&lo->profName,&j)) goto error;
	if((cp=ASN1_next(cp))==NULL) goto error;

	if((lo->crlOp=ASN1_enumerated(cp,&i))<0) goto error;
	if((cp=ASN1_next(cp))==NULL) goto error;

	lo->location = *cp & 0x1f;
	switch(lo->location){
	case 0: break; /* [0] NULL */
	case 1: break; /* [1] NULL */
	case 2:
	case 3:
		if(ASN1_octetstring_(cp,&i,(unsigned char**)&lo->path,&j,1)) goto error;
		break;
	default:
		OK_set_error(ERR_ST_UNSUPPORTED_PARAM,ERR_LC_LCMP,ERR_PT_LCMPASD+7,NULL);
		goto error;
	}
	if((cp=ASN1_next(cp))==NULL) goto error;

	/* crlNumber INTEGER OPTIONAL */
	if(*cp == ASN1_INTEGER){
		if((lo->crlNum=ASN1_integer_(cp,&i,1))<0) goto error;
	}
	return 0;
error:
	return -1;
}

int asn1_lcmp_crlrsp(LCMPOp *op, unsigned char *rsp){
	LO_CRLRsp *lo = (LO_CRLRsp*)op;
	unsigned char *cp,*sq;
	CRLList *cl,*hd=NULL;
	CRL *crl=NULL;
	int len,i,j;

	if(asn1_lcmp_result(op,rsp)) goto error;
	if((cp=ASN1_step(rsp,3))==NULL) goto error;

	if(*cp == 0xa0){ /* SET OF Certificate OPTIONAL */
		if((len=ASN1_tlen(cp))<0) goto error;
		if((sq =ASN1_next(cp))==NULL) goto error;

		for(i=0; i<len;){
			if((crl=ASN1_read_crl(sq))==NULL) goto error;
			if((crl->der=ASN1_dup(sq))==NULL) goto error;
			if((cl=CRL_2CRLlist(crl))==NULL) goto error;
			if(hd){
				hd->next=cl; hd=cl;
			}else{
				lo->list=hd=cl;
			}
			CRL_free(crl); crl=NULL;

			if((sq=ASN1_skip_(sq,&j))==NULL) goto error;
			i+=j;
		}
	}
	return 0;
error:
	CRLlist_free_all(lo->list); lo->list=NULL;
	CRL_free(crl);
	return -1;
}

int asn1_lcmp_svopreq(LCMPOp *op, unsigned char *req){
	LO_SVOpReq *lo = (LO_SVOpReq*)op;
	unsigned char *cp;
	int i;

	if((cp=ASN1_next(req))==NULL) goto error;

	if((lo->svOp=ASN1_enumerated(cp,&i))<0) goto error;
	if((cp=ASN1_next(cp))==NULL) goto error;

	if(*cp == 0xa0){ /* [0] OCTET STRING OPTIONAL */
		if((lo->uinfo=asn1_lcmp_authinfo(cp))==NULL) goto error;
	}
	return 0;
error:
	return -1;
}

int asn1_lcmp_svoprsp(LCMPOp *op, unsigned char *rsp){
	LO_SVOpRsp *lo = (LO_SVOpRsp*)op;
	unsigned char *cp;
	int i;

	if(asn1_lcmp_result(op,rsp)) goto error;
	if((cp=ASN1_step(rsp,3))==NULL) goto error;

	if(*cp == 0x80){ /* [0] OCTET STRING OPTIONAL */
		if(ASN1_octetstring_(cp,&i,(unsigned char**)&lo->opRsp,&lo->rspLen,1)) goto error;
	}
	return 0;
error:
	return -1;
}

int asn1_lcmp_extreq(LCMPOp *op, unsigned char *req){
	LO_ExtReq *lo = (LO_ExtReq*)op;
	unsigned char *cp;
	int i;

	if((cp=ASN1_next(req))==NULL) goto error;

	if(*cp == 0x80){ /* [0] OBJECT IDENTIFIER */
		if((lo->opOID=(char*)malloc(128))==NULL) goto error;
		*cp = ASN1_OBJECT_IDENTIFIER;
		i = objid2str(cp,lo->opOID,126);
		*cp = 0x80;
		
		if(i<0) goto error;
		if((cp=ASN1_next(cp))==NULL) goto error;
	}
	if(*cp == 0x81){ /* [1] OCTET STRING OPTIONAL */
		if(ASN1_octetstring_(cp,&i,(unsigned char**)&lo->opInfo,&lo->ifLen,1)) goto error;
	}
	return 0;
error:
	return -1;
}

int asn1_lcmp_extrsp(LCMPOp *op, unsigned char *rsp){
	LO_ExtRsp *lo = (LO_ExtRsp*)op;
	unsigned char *cp;
	int i;

	if(asn1_lcmp_result(op,rsp)) goto error;
	if((cp=ASN1_step(rsp,3))==NULL) goto error;

	if(*cp == 0x80){ /* [0] OBJECT IDENTIFIER OPTIONAL */
		if((lo->opOID=(char*)malloc(128))==NULL) goto error;
		*cp = ASN1_OBJECT_IDENTIFIER;
		i = objid2str(cp,lo->opOID,126);
		*cp = 0x80;
		
		if(i<0) goto error;
		if((cp=ASN1_next(cp))==NULL) goto error;
	}
	if(*cp == 0x81){ /* [1] OCTET STRING OPTIONAL */
		if(ASN1_octetstring_(cp,&i,(unsigned char**)&lo->opRsp,&lo->rspLen,1)) goto error;
	}
	return 0;
error:
	return -1;
}

int asn1_lcmp_renewalreq(LCMPOp *op, unsigned char *req){
	LO_RenewalReq *lo = (LO_RenewalReq*)op;
	unsigned char *cp,*der=NULL;
	int type,i,j;

	if((cp=ASN1_next(req))==NULL) goto error;

	if((lo->serialNum=ASN1_integer(cp,&i))<0) goto error;
	if((cp=ASN1_next(cp))==NULL) goto error;

	if((type=ASN1_enumerated(cp,&i))<0) goto error;
	if((cp=ASN1_next(cp))==NULL) goto error;

	if(ASN1_octetstring(cp,&i,&der,&j)) goto error;
	if((cp=ASN1_next(cp))==NULL) goto error;
	
	switch(type){
	case PKCS10CSR:
		if((lo->p10=ASN1_read_req(der))==NULL) goto error;
		der = NULL;
		break;
	case CMPREQUEST:
		if((lo->tmpl=ASN1_cmp_certtmpl(der,&i))==NULL) goto error;
		der = NULL;
		break;
	default:
		OK_set_error(ERR_ST_UNSUPPORTED_PARAM,ERR_LC_LCMP,ERR_PT_LCMPASD+9,NULL);
		goto error;
	}

	if(*cp==0x80){ /* [0] INTEGER OPTIONAL */
		if((lo->newSerialNum=ASN1_integer_(cp,&i,1))<0) goto error;
	}
	return 0;
error:
	if(der) free(der);
	return -1;
}

int asn1_lcmp_renewalrsp(LCMPOp *op, unsigned char *rsp){
	LO_RenewalRsp *lo = (LO_RenewalRsp*)op;
	unsigned char *cp;

	if(asn1_lcmp_result(op,rsp)) goto error;
	if((cp=ASN1_step(rsp,3))==NULL) goto error;

	if(*cp == 0xa0){ /* OPTIONAL */
		if((cp=ASN1_next(cp))==NULL) goto error;
		if((lo->cert=ASN1_read_cert(cp))==NULL) goto error;
		if((lo->cert->der=ASN1_dup(cp))==NULL) goto error;
	}
	return 0;
error:
	Cert_free(lo->cert); lo->cert=NULL;
	return -1;
}

/*-------------------------*/
int flag_bit2state(unsigned char *flags, int k, int *state){
	if((flags==NULL)||(k<=0)) return 0;
	if(flags[0]&0x80) *state |= STAT_CA;
	if(flags[0]&0x40) *state |= STAT_EXPIRED;
	if(flags[0]&0x20) *state |= STAT_REVOKED;
	if(flags[0]&0x10) *state |= 0x08;
	if(flags[0]&0x08) *state |= STAT_XCERTFWD;
	if(flags[0]&0x04) *state |= STAT_XCERTREV;
	if(flags[0]&0x02) *state |= 0x40;
	if(flags[0]&0x01) *state |= 0x80;
	if(k<=1) return 0;
	if(flags[1]&0x80) *state |= STAT_HAVEKEY;
	return 0;
}

CertStat *asn1_lcmp_statlist(unsigned char *in){
	unsigned char *cp,*sq,*flags = NULL;
	CertStat *st,*hd=NULL,*ret=NULL;
	int	len,i,j,k,l;

	len=ASN1_tlen(in);
	sq =ASN1_next(in);

	for(i=0;i<len;){
		if((st=CertStat_new())==NULL) goto error;
		if(ret){
			hd->next=st; st->prev=hd; hd=st;
		}else{
			ret=hd=st; 
		}
		cp = ASN1_next(sq);

		if(ASN1_bitstring(cp,&j,&flags,&k,&l)) goto error;
		if(flag_bit2state(flags,k,&st->state)) goto error;
		free(flags); flags=NULL;
		cp=ASN1_next(cp);

		st->serialNum = ASN1_integer(cp,&k);
		cp=ASN1_next(cp);

		if((st->subject = asn1_get_str(cp,&k))==NULL) goto error;
		cp=ASN1_next(cp);

		if(UTC2stm(cp,&st->notBefore)) goto error;
		cp=ASN1_next(cp);

		if(UTC2stm(cp,&st->notAfter)) goto error;
		cp=ASN1_next(cp);

		if((*cp==ASN1_UTCTIME)||(*cp==ASN1_GENERALIZEDTIME)){
			if(UTC2stm(cp,&st->revokedDate)) goto error;
			cp=ASN1_next(cp);
		}
		if(*cp==0x80){
			if((st->acceptID = ASN1_integer_(cp,&k,1))<0) goto error;
			cp=ASN1_next(cp);
		}
		if(*cp==0x81){
			if((l = ASN1_integer_(cp,&k,1))<0) goto error;
			st->state |= (l<<16);
			cp=ASN1_next(cp);
		}
		if(*cp==0x82){ /* keyID OPTIONAL */
			if(ASN1_octetstring_(cp,&k,&st->keyID,&l,1)) goto error;
			cp=ASN1_next(cp);
		}

		if((sq=ASN1_skip_(sq,&j))==NULL) goto error;
		i+=j;
	}
	return ret;
error:
	CertStat_free_all(ret);
	return NULL;
}

Key *asn1_lcmp_keytype(unsigned char *in){
	unsigned char *cp;
	Key *ret=NULL;
	int i,type,len;

	if((cp=ASN1_next(in))==NULL) goto error;

	if((type=ASN1_object_2int(cp))<0) goto error;
	if((cp=ASN1_next(cp))==NULL) goto error;

	if((len=ASN1_integer(cp,&i))<0) goto error;
	if((cp=ASN1_next(cp))==NULL) goto error;

	switch(type){
	case OBJ_CRYPT_RSA:
		if((ret=(Key*)ASN1_read_rsaprv(cp))==NULL) goto error;
		if((((Prvkey_RSA*)ret)->der=ASN1_dup(cp))==NULL) goto error;
		break;
	case OBJ_CRYPT_DSA:
		if((ret=(Key*)ASN1_read_dsaprv(cp))==NULL) goto error;
		if((((Prvkey_DSA*)ret)->der=ASN1_dup(cp))==NULL) goto error;
		break;
	case OBJ_CRYPT_ECDSA:
		if((ret=(Key*)ASN1_read_ecdsaprv(cp))==NULL) goto error;
		if((((Prvkey_ECDSA*)ret)->der=ASN1_dup(cp))==NULL) goto error;
		break;
	}

	return ret;
error:
	Key_free(ret);
	return NULL;
}

CertExt *asn1_lcmp_delext(unsigned char *in){
	unsigned char *cp;
	CertExt *hd,*ex,*ret=NULL;
	int i,j,len,id;

	/* [5] SET OF OBJECT IDENTIFIER */
	if((len= ASN1_tlen(in))<0) goto error;
	if((cp = ASN1_next(in))==NULL) goto error;

	for(i=0; i<len;){
		if((id = ASN1_object_2int(cp))<0) goto error;
		if((ex = CertExt_new(id))==NULL) goto error;
		if(ret){
			hd->next=ex; hd=ex;
		}else{
			ret=hd=ex;
		}
	  
		if((cp=ASN1_skip_(cp,&j))==NULL) goto error;
		i+=j;
	}
	
	return ret;
error:
	CertExt_free_all(ret);
	return NULL;
}

ProfTmpl *asn1_lcmp_prof(unsigned char *in){
	unsigned char *cp,*tmp = NULL;
	ProfTmpl *ret=NULL;
	int i,j;

	if((ret=ProfTmpl_new())==NULL) goto error;
	
	if((in=ASN1_next(in))==NULL) goto error;
	
	/* LCMPString ... not optional, but ModProfContent doesn't have this one */	
	if(*in == ASN1_OCTETSTRING){
		if(ASN1_octetstring(in,&i,(unsigned char**)&ret->tmplName,&j)) goto error;
		if((in=ASN1_next(in))==NULL) goto error;
	}
	if(*in == 0x80){ /* [0] INTEGER OPTIONAL */
		if((ret->certVer=ASN1_integer_(in,&i,1))<0) goto error;
		if((in=ASN1_next(in))==NULL) goto error;
	}
	if(*in == 0x81){ /* [1] INTEGER OPTIONAL */
		if((ret->baseNum=ASN1_integer_(in,&i,1))<0) goto error;
		if((in=ASN1_next(in))==NULL) goto error;
	}
	if(*in == 0xa2){ /* [2] StartAndEndTime OPTIONAL */
		if((cp=ASN1_next(in))==NULL) goto error;
		if(*cp == 0xa0){ /* StartTime */
			if((cp=ASN1_next(cp))==NULL) goto error;
			if(*cp == ASN1_NULL){ /* currentTime NULL */
				ret->bgtype = 0;
			}else if(*cp == ASN1_GENERALIZEDTIME){ /* startTime GeneralizedTime */
				if(UTC2stm(cp,&ret->start)) goto error;
				ret->bgtype = 1;
			}else if(*cp == ASN1_PRINTABLE_STRING){ /* startTimeMix PrintableString */
				if(UTC2stm(cp,&ret->start)) goto error;
				ret->bgtype = 2;
			}
			if((cp=ASN1_next(cp))==NULL) goto error;
		}
		if(*cp == 0xa1){ /* EndTime */
			if((cp=ASN1_next(cp))==NULL) goto error;
			if(*cp == ASN1_INTEGER){ /* seconds */
				if((ret->certSec=ASN1_integer(cp,&i))<0) goto error;
				ret->edtype = 0;
			}else if(*cp == ASN1_GENERALIZEDTIME){ /* endTime */
				if(UTC2stm(cp,&ret->end)) goto error;
				ret->edtype = 1;
			}else if(*cp == ASN1_PRINTABLE_STRING){ /* addInTime */
				if(UTC2stm(cp,&ret->end)) goto error;
				ret->edtype = 2;
			}
			if((cp=ASN1_next(cp))==NULL) goto error;
		}
		if(*cp == 0xa2){ /* UpdateTime */
			if((cp=ASN1_next(cp))==NULL) goto error;
			if(*cp == ASN1_INTEGER){ /* seconds */
				ret->updtype = 0;
				if((ret->updSec=ASN1_integer(cp,&i))<0) goto error;
			}
		}
		if((in=ASN1_skip(in))==NULL) goto error;
	}
	if(*in == 0xa3){ /* [3] KeyType OPTIONAL */
		if((cp=ASN1_next(in))==NULL) goto error;
		if((ret->keyType=ASN1_object_2int(cp))<0) goto error;

		if((cp=ASN1_next(cp))==NULL) goto error;
		if((ret->keyLength=ASN1_integer(cp,&i))<0) goto error;

		if((in=ASN1_skip(in))==NULL) goto error;
	}
	if(*in == 0x84){ /* [4] OBJECT IDENTIFIER OPTIONAL */
		*in = ASN1_OBJECT_IDENTIFIER;
		ret->hashType = ASN1_object_2int(in);
		*in = 0x84;
		if(ret->hashType<0) goto error;
		if((in=ASN1_next(in))==NULL) goto error;
	}
	if(*in == 0xa5){ /* [5] Name OPTIONAL */
		if((tmp=ASN1_get_subject(in,&ret->sbjTmpl))==NULL) goto error;
		free(tmp); tmp=NULL;
		if((in=ASN1_skip(in))==NULL) goto error;
	}
	if(*in == 0x86){ /* [6] BIT STRING OPTIONAL */
		if(ASN1_bitstring_(in,&j,&tmp,&i,&j,1)) goto error;
		memset(ret->policyFlag,0,8);
		memcpy(ret->policyFlag,tmp,(i>8)?(8):(i));
		free(tmp); tmp=NULL;
		if((in=ASN1_next(in))==NULL) goto error;
	}
	return ret;
error:
	if(tmp) free(tmp);
	ProfTmpl_free(ret);
	return NULL;
}

AuthInfo *asn1_lcmp_authinfo(unsigned char *in){
	AuthInfo *ret=NULL;
	int i,j;

	if((ret=CA_ai_new())==NULL) goto error;
	ret->uid = ret->gid = ret->grant=-1;

	if((in=ASN1_next(in))==NULL) goto error;

	/* userName LCMPString, */
	if(*in == ASN1_OCTETSTRING){
		if(ASN1_octetstring(in,&i,(unsigned char**)&ret->name,&j)) goto error;
		if((in=ASN1_next(in))==NULL) goto error;
	}
	/* userPassword [0] LCMPString OPTIONAL */
	if(*in == 0x80){
		if(ASN1_octetstring_(in,&i,(unsigned char**)&ret->passwd,&j,1)) goto error;
		if((in=ASN1_next(in))==NULL) goto error;
	}
	/* userOldPassword [1] LCMPString OPTIONAL */
	if(*in == 0x81){
		if(ASN1_octetstring_(in,&i,(unsigned char**)&ret->oldpasswd,&j,1)) goto error;
		if((in=ASN1_next(in))==NULL) goto error;
	}
	/* userID [2] INTEGER OPTIONAL */
	if(*in == 0x82){
		if((ret->uid=ASN1_integer_(in,&i,1))<0) goto error;
		if((in=ASN1_next(in))==NULL) goto error;
	}
	/* userGroupID [3] INTEGER OPTIONAL */
	if(*in == 0x83){
		if((ret->gid=ASN1_integer_(in,&i,1))<0) goto error;
		if((in=ASN1_next(in))==NULL) goto error;
	}
	/* userGrant [4] INTEGER OPTIONAL */
	if(*in == 0x84){
		/*
		 * ret->grant : unsigned long
		 * (see struct 'authinfo' defined by ok_ca.h)
		 */
		if(ASN1_integer_(in,&i,1) < 0) {
			goto error;
		} else {
			ret->grant = ASN1_integer_(in,&i,1);
		}
		if((in=ASN1_next(in))==NULL) goto error;
	}
	/* optionalInfo [5] OCTET STRING OPTIONAL */
	if(*in == 0x85){
		if(ASN1_octetstring_(in,&i,(unsigned char**)&ret->opt,&j,1)) goto error;
	}
	
	return ret;
error:
	CA_ai_free_all(ret);
	return NULL;
}
