/* lcmp_api.c */
/*
 * Copyright (c) 2004-2015 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *	this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *	this list of conditions and the following disclaimer in the documentation
 *	and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *	display the following acknowledgment:
 *	"This product includes software developed by Akira Iwata Laboratory,
 *	Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *	acknowledgment:
 *	"This product includes software developed by Akira Iwata Laboratory,
 *	 Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <netdb.h>
#include <aicrypto/ok_err.h>

#include "ok_caerr.h"
#include "ok_ca.h"
#include "ok_lcmp.h"

/*-----------------------------------------
  LCMP init  
-----------------------------------------*/
LCMP *LCMP_init(char *svname, int port, char *caname){
	LCMP *ret=NULL;

	if((ret=LCMP_new())==NULL) goto error;
	if ((ret->svname = strdup(svname)) == NULL){
		OK_set_error(ERR_ST_STRDUP,ERR_LC_LCMP,ERR_PT_LCMPAPI,NULL);
		goto error;
	}
	if ((ret->caname = strdup(caname)) == NULL){
		OK_set_error(ERR_ST_STRDUP,ERR_LC_LCMP,ERR_PT_LCMPAPI,NULL);
		goto error;
	}
	ret->port = port;

	return ret;
error:
	LCMP_free(ret);
	return NULL;
}

/*-----------------------------------------
  LCMP bind
-----------------------------------------*/
int LCMP_bind_s(LCMP *lc, char *userid, char *passwd, int auth){
	CertList *cl=NULL;
	PKCS12 *p12=NULL;
	int ret = 0;

	LCMP_op_free(lc->op); lc->op=NULL;

	if((lc==NULL)||(lc->svname==NULL)||(lc->caname==NULL)){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_LCMP,ERR_PT_LCMPAPI+1,NULL);
		goto error;
	}
	if(userid==NULL) userid="";
	if(passwd==NULL) passwd="";

	/* get raw or SSL socket & connect */
	if (lc->sock == NULL) {
		if((ret = LCMP_getsock(lc))) goto error;
	}

	/* get bindreq & write */
	if((lc->op=LCMP_get_bindreq(lc->caname,userid,passwd,auth))==NULL) goto error;

	if(LCMP_write(lc)<0) goto error;

	/* wait a response from CA */
	LCMP_op_free(lc->op); lc->op=NULL;
	if(LCMP_read(lc)<0) goto error;

	if(lc->op==NULL) goto error;
	if((lc->op->resultCode!=LCMP_SUCCESS)||(((LO_BindRsp*)lc->op)->list==NULL)){
		return lc->op->resultCode;
	}
	if(lc->op->opId != LCMP_OP_BINDRSP){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+1,NULL);
		goto error;
	}

	/* set CA certificate as bind information */
	if(lc->ca) CA_free(lc->ca); /* not HSM case */
	if((lc->ca=CA_new())==NULL) goto error;
	if((lc->ca->p12=p12=P12_new())==NULL) goto error;
	for(cl=((LO_BindRsp*)lc->op)->list; cl ; cl=cl->next){
		if(P12_add_cert(p12,cl->cert,NULL,0xff)) goto error;
		cl->cert=NULL;
	}
	if(P12_check_chain(p12,0)) goto error;
	if((lc->ca->cert=P12_get_usercert(p12))==NULL) goto error;

	/* reserve user name */
	if(lc->username) free(lc->username);
	if ((lc->username = strdup(userid)) == NULL){
		OK_set_error(ERR_ST_STRDUP,ERR_LC_LCMP,ERR_PT_LCMPAPI+1,NULL);
		goto error;
	}

	return 0;
error:
	return ret ? ret : -1;
}

int LCMP_getsock(LCMP *lc){
	struct addrinfo hints;
	struct addrinfo *aihead = NULL, *ai;
	SSL *s = NULL;
	PKCS12 *p12 = NULL;
	Cert *ct = NULL;
	Key *key = NULL;
	char *str_port = NULL;
	int ret = 0;

	/* 1. get client socket */
	if(asprintf(&str_port, "%d", lc->port) < 0)
		goto error;
	memset(&hints, 0, sizeof(hints));
	hints.ai_family = AF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;
	if(getaddrinfo(lc->svname, str_port, &hints, &aihead) != 0)
		goto error;
	if(lc->sock) SSL_free(lc->sock);
	for (ai = aihead; ai != NULL; ai = ai->ai_next) {
		if((lc->sock=s=SSL_socket(ai->ai_family, ai->ai_socktype, ai->ai_protocol))==NULL)
			goto loop_warn;

		/* setup SSL if necessary */
		if(lc->usessl){
			if(SSL_set_store(lc->sock,lc->store)) goto loop_warn;

			if(lc->vfycert) SSL_set_vfytype(lc->sock,lc->vfycert);

			if(lc->certid && *(lc->certid)){
				/* store id */
				if(lc->clctpw) OK_set_passwd(lc->clctpw);
				if(SSL_set_clientkey_id(lc->sock,lc->certid)) goto loop_warn;

			}else if(lc->clkeyf && *(lc->clkeyf)){
				/* certificate & private key file */
				if(lc->clctpw) OK_set_passwd(lc->clctpw);

				if((p12=P12_new())==NULL) goto loop_warn;
				if((ct=Cert_read_file(lc->clcert))==NULL) goto loop_warn;
				if((key=Key_read_file(lc->clkeyf))==NULL) goto error;

				if(P12_add_cert(p12,ct,NULL,0xff)) goto loop_warn;
				ct = NULL;
				if(P12_add_key(p12,key,NULL,0xff)) goto loop_warn;
				key = NULL;
				if(P12_check_chain(p12,0)) goto loop_warn;

				if(SSL_set_clientkey_p12(lc->sock,p12)) goto loop_warn;
				p12 = NULL;

			}else if(lc->clcert && *(lc->clcert)){
				/* pkcs12 file */
				if(SSL_set_client_p12(lc->sock,lc->clcert,lc->clctpw))
					goto loop_warn;
			}
		}

		/* connect to the server */
		if(SSL_connect(s,ai->ai_addr,ai->ai_addrlen))
			goto loop_warn;

		break;

loop_warn:
		P12_free(p12);
		Cert_free(ct);
		Key_free(key);
		if(lc->sock != NULL){
			SSL_close(lc->sock); SSL_free(lc->sock); lc->sock=NULL;
		}
	}

	if (lc->sock == NULL) {
		printf("error : no socket connection to %s\n", lc->svname);
		goto error;
	}

	/* do SSL handshake if necessary */
	if(lc->usessl){
		if(SSL_handshake(s)) {
			if (SSL_get_error(s) == SSL_ERR_PROTOCOL_VERSION) {
				ret = -2;
			}
			goto error;
		}
	}

	free(str_port);
	freeaddrinfo(aihead);
	return 0;

error:
	P12_free(p12);
	Cert_free(ct);
	Key_free(key);
	if(lc->sock){ SSL_close(lc->sock); SSL_free(lc->sock); lc->sock=NULL; }
	if(str_port != NULL) {
		free(str_port);
	}
	if(aihead != NULL) {
		freeaddrinfo(aihead);
	}
	return ret ? ret : -1;
}

int LCMP_set_ssl(LCMP *lc, char *store_path, char *certid, char *pass, int vfymode){
	if(store_path && *store_path){
		if ((lc->store = strdup(store_path)) == NULL){
			OK_set_error(ERR_ST_STRDUP,ERR_LC_LCMP,ERR_PT_LCMPAPI+3,NULL);
			goto error;
	}}
	if(certid && *certid){
		if ((lc->certid = strdup(certid)) == NULL){
			OK_set_error(ERR_ST_STRDUP,ERR_LC_LCMP,ERR_PT_LCMPAPI+3,NULL);
			goto error;
	}}
	if(pass && *pass){
		if ((lc->clctpw = strdup(pass)) == NULL){
			OK_set_error(ERR_ST_STRDUP,ERR_LC_LCMP,ERR_PT_LCMPAPI+3,NULL);
			goto error;
	}}
	lc->usessl = 1;
	lc->vfycert = vfymode;
	return 0;
error:
	return -1;
}

/*-----------------------------------------
  LCMP unbind
-----------------------------------------*/
int LCMP_unbind_s(LCMP *lc){
	int ok = -1;
	
	LCMP_op_free(lc->op);
	if((lc->op=LCMP_op_new(LCMP_OP_UNBIND))==NULL) goto done;

	if(LCMP_write(lc)<0) goto done;

	ok = 0;
done:
	LCMP_close(lc);
	return ok;
}

void LCMP_close(LCMP *lc){
	if(lc && lc->sock){
		SSL_close(lc->sock);
		SSL_free(lc->sock); lc->sock=NULL;
	}
}

/*-----------------------------------------
  LCMP sign
-----------------------------------------*/
int LCMP_sign_s(LCMP *lc, char *profile, int sn, void *req, enum csrform csrFormat){
	if(profile==NULL) profile="SMIME user";

	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_signreq(profile,csrFormat,req,sn))==NULL) goto error;
	
	if(LCMP_write(lc)<0) goto error;
	LCMP_op_free(lc->op); lc->op=NULL;

	/* wait a response from CA */
	if(LCMP_read(lc)<0) goto error;

	if(lc->op==NULL) goto error;
	if(lc->op->resultCode != LCMP_SUCCESS){
		return lc->op->resultCode;
	}
	if(lc->op->opId != LCMP_OP_SIGNRSP){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+4,NULL);
		goto error;
	}
	
	return 0;
error:
	return -1;
}

Cert *LCMP_get_signcert(LCMP *lc){
	Cert *ret;
	if((lc->op==NULL)||(lc->op->opId != LCMP_OP_SIGNRSP)){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+4,NULL);
		goto error;
	}
	ret = ((LO_SignRsp*)lc->op)->cert;
	((LO_SignRsp*)lc->op)->cert = NULL;
	return ret;
error:
	return NULL;
}

/*-----------------------------------------
  LCMP list
-----------------------------------------*/
int LCMP_list_s_(LCMP *lc, char *profile, int sn, char *query){
	if(profile==NULL) profile="SMIME user";

	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_listreq_(profile,sn,query))==NULL) goto error;
	
	if(LCMP_write(lc)<0) goto error;
	LCMP_op_free(lc->op); lc->op=NULL;

	/* wait a response from CA */
	if(LCMP_read(lc)<0) goto error;
	if(lc->op==NULL) goto error;
	if(lc->op->resultCode != LCMP_SUCCESS){
		return lc->op->resultCode;
	}
	if(lc->op->opId != LCMP_OP_LISTRSP){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+5,NULL);
		goto error;
	}
	return 0;
error:
	return -1;
}

int LCMP_list_s(LCMP *lc, char *profile, int sn){
	return LCMP_list_s_(lc,profile,sn,NULL);
}

int LCMP_listsbj_s(LCMP *lc, char *profile, char *query){
	return LCMP_list_s_(lc,profile,0,query);
}

CertStat *LCMP_get_statptr(LCMP *lc){
	CertStat *ret;
	if((lc->op==NULL)||(lc->op->opId != LCMP_OP_LISTRSP)){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+5,NULL);
		goto error;
	}
	ret = ((LO_ListRsp*)lc->op)->stat;
	((LO_ListRsp*)lc->op)->stat = NULL;
	return ret;
error:
	return NULL;
}

/*-----------------------------------------
  LCMP prof
-----------------------------------------*/
int LCMP_prof_s_(LCMP *lc,int pfOp, char *profile, ProfTmpl *tmpl, CertExt *ext, char *rename){
	if(profile==NULL) profile="SMIME user";

	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_profreq_(pfOp,profile,tmpl,ext,rename))==NULL) goto error;
	
	if(LCMP_write(lc)<0) goto error;
	LCMP_op_free(lc->op); lc->op=NULL;

	/* wait a response from CA */
	if(LCMP_read(lc)<0) goto error;
	if(lc->op==NULL) goto error;
	if(lc->op->resultCode != LCMP_SUCCESS){
		return lc->op->resultCode;
	}
	if(lc->op->opId != LCMP_OP_PROFRSP){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+6,NULL);
		goto error;
	}
	return 0;
error:
	return -1;
}

int LCMP_prof_s(LCMP *lc,int pfOp, char *profile, ProfTmpl *tmpl, CertExt *ext){
	return LCMP_prof_s_(lc,pfOp,profile,tmpl,ext,NULL);
}

char **LCMP_get_proflist(LCMP *lc,int *listnum){
	char **ret;
	if((lc->op==NULL)||(lc->op->opId != LCMP_OP_PROFRSP)){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+6,NULL);
		goto error;
	}
	ret = ((LO_ProfRsp*)lc->op)->proflist;
	((LO_ProfRsp*)lc->op)->proflist = NULL;
	*listnum = ((LO_ProfRsp*)lc->op)->listnum;
	return ret;
error:
	return NULL;
}

void LCMP_proflist_free(char **list,int num){
	int i;
	if(list==NULL) return;
	for(i=0; i<num; i++)
		if(list[i]){ free(list[i]); list[i]=NULL; }
	free(list);
}

ProfTmpl *LCMP_get_profinfo(LCMP *lc){
	ProfTmpl *ret;
	if((lc->op==NULL)||(lc->op->opId != LCMP_OP_PROFRSP)){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+6,NULL);
		goto error;
	}
	ret = ((LO_ProfRsp*)lc->op)->prof;
	((LO_ProfRsp*)lc->op)->prof = NULL;
	return ret;
error:
	return NULL;
}

CertExt *LCMP_get_profext(LCMP *lc){
	CertExt *ret;
	if((lc->op==NULL)||(lc->op->opId != LCMP_OP_PROFRSP)){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+6,NULL);
		goto error;
	}
	ret = ((LO_ProfRsp*)lc->op)->ext;
	((LO_ProfRsp*)lc->op)->ext = NULL;
	return ret;
error:
	return NULL;
}

/*-----------------------------------------
  LCMP cert
-----------------------------------------*/
int LCMP_cert_s(LCMP *lc,int certOp, int snum, int reason, Key *key, char *pwd){

	/* require to export a certificate */
	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_certreq(certOp,snum,reason,key,pwd))==NULL)
		goto error;

	if(LCMP_write(lc)<0) goto error;
	LCMP_op_free(lc->op); lc->op=NULL;

	/* wait a response from CA */
	if(LCMP_read(lc)<0) goto error;
	if(lc->op==NULL) goto error;
	if(lc->op->resultCode != LCMP_SUCCESS){
		return lc->op->resultCode;
	}
	if(lc->op->opId != LCMP_OP_CERTRSP){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+7,NULL);
		goto error;
	}
		
	return 0;
error:
	return -1;
}

Cert *LCMP_get_expcert(LCMP *lc){
	Cert *ret;
	if((lc->op==NULL)||(lc->op->opId != LCMP_OP_CERTRSP)){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+7,NULL);
		goto error;
	}
	ret = ((LO_CertRsp*)lc->op)->cert;
	((LO_CertRsp*)lc->op)->cert = NULL;
	return ret;
error:
	return NULL;
}

Key *LCMP_get_expkey(LCMP *lc){
	Key *ret;
	if((lc->op==NULL)||(lc->op->opId != LCMP_OP_CERTRSP)){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+7,NULL);
		goto error;
	}
	ret = ((LO_CertRsp*)lc->op)->key;
	((LO_CertRsp*)lc->op)->key = NULL;
	return ret;
error:
	return NULL;
}

/*-----------------------------------------
  LCMP csr
-----------------------------------------*/
int LCMP_csr_s(LCMP *lc,int csrOp, int accID, int snum, char *pfname, Req *csr, int reason, CertTemplate *tmpl){

	/* require to CSR operation */
	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_csrreq(csrOp,accID,snum,pfname,csr,reason,tmpl))==NULL)
		goto error;

	if(LCMP_write(lc)<0) goto error;
	LCMP_op_free(lc->op); lc->op=NULL;

	/* wait a response from CA */
	if(LCMP_read(lc)<0) goto error;
	if(lc->op==NULL) goto error;
	if(lc->op->resultCode != LCMP_SUCCESS){
		return lc->op->resultCode;
	}
	if(lc->op->opId != LCMP_OP_CSRRSP){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+8,NULL);
		goto error;
	}
		
	return 0;
error:
	return -1;
}

Req *LCMP_get_expcsr(LCMP *lc){
	Req *ret;
	if((lc->op==NULL)||(lc->op->opId != LCMP_OP_CSRRSP)){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+8,NULL);
		goto error;
	}
	ret = ((LO_CSRRsp*)lc->op)->csr;
	((LO_CSRRsp*)lc->op)->csr = NULL;
	return ret;
error:
	return NULL;
}

CertTemplate *LCMP_get_exptmpl(LCMP *lc){
	CertTemplate *ret;
	if((lc->op==NULL)||(lc->op->opId != LCMP_OP_CSRRSP)){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+8,NULL);
		goto error;
	}
	ret = ((LO_CSRRsp*)lc->op)->tmpl;
	((LO_CSRRsp*)lc->op)->tmpl = NULL;
	return ret;
error:
	return NULL;
}

int LCMP_get_acceptid(LCMP *lc){
	if((lc->op==NULL)||(lc->op->opId != LCMP_OP_CSRRSP)){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+8,NULL);
		return -1;
	}
	return ((LO_CSRRsp*)lc->op)->acceptID;
}

int LCMP_get_serialnum(LCMP *lc){
	if((lc->op==NULL)||(lc->op->opId != LCMP_OP_CSRRSP)){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+8,NULL);
		return -1;
	}
	return ((LO_CSRRsp*)lc->op)->serialNum;
}

/*-----------------------------------------
  LCMP crl
-----------------------------------------*/
int LCMP_crl_s(LCMP *lc,int crlOp, char *profile){

	/* require to export a certificate */
	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_crlreq(crlOp,profile,LCMP_LCCRL_TOCLIENT,NULL))==NULL)
		goto error;

	if(LCMP_write(lc)<0) goto error;
	LCMP_op_free(lc->op); lc->op=NULL;

	/* wait a response from CA */
	if(LCMP_read(lc)<0) goto error;
	if(lc->op==NULL) goto error;
	if(lc->op->resultCode != LCMP_SUCCESS){
		return lc->op->resultCode;
	}
	if(lc->op->opId != LCMP_OP_CRLRSP){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+9,NULL);
		goto error;
	}
		
	return 0;
error:
	return -1;
}

CRL *LCMP_get_signcrl(LCMP *lc){
	CRL *ret = NULL;
	if((lc->op==NULL)||(lc->op->opId != LCMP_OP_CRLRSP)){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+9,NULL);
		goto error;
	}
	if(((LO_CRLRsp*)lc->op)->list){
		ret = ((LO_CRLRsp*)lc->op)->list->crl;
		((LO_CRLRsp*)lc->op)->list->crl = NULL;
	}
	return ret;
error:
	return NULL;
}

/*-----------------------------------------
  LCMP svop
-----------------------------------------*/
int LCMP_svop_s(LCMP *lc,int svOp, void *opInfo){

	/* require to export a certificate */
	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_svopreq(svOp,(AuthInfo*)opInfo))==NULL)
		goto error;

	if(LCMP_write(lc)<0) goto error;
	LCMP_op_free(lc->op); lc->op=NULL;

	/* wait a response from CA */
	if(LCMP_read(lc)<0) goto error;
	if(lc->op==NULL) goto error;
	if(lc->op->resultCode != LCMP_SUCCESS){
		return lc->op->resultCode;
	}
	if(lc->op->opId != LCMP_OP_SVOPRSP){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+10,NULL);
		goto error;
	}
		
	return 0;
error:
	return -1;
}

/*-----------------------------------------
  LCMP extended operation
-----------------------------------------*/
int LCMP_extop_s(LCMP *lc,char *oid, unsigned char *bin, int blen){

	/* require to export a certificate */
	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_extreq(oid,bin,blen))==NULL)
		goto error;

	if(LCMP_write(lc)<0) goto error;
	LCMP_op_free(lc->op); lc->op=NULL;

	/* wait a response from CA */
	if(LCMP_read(lc)<0) goto error;
	if(lc->op==NULL) goto error;
	if(lc->op->resultCode != LCMP_SUCCESS){
		return lc->op->resultCode;
	}
	if(lc->op->opId != LCMP_OP_EXTRSP){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+11,NULL);
		goto error;
	}		
	return 0;
error:
	return -1;
}

char *LCMP_get_extoid(LCMP *lc){
	char *ret;

	if((lc->op==NULL)||(lc->op->opId != LCMP_OP_EXTRSP)){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+11,NULL);
		goto error;
	}
	ret = ((LO_ExtRsp*)lc->op)->opOID;
	((LO_ExtRsp*)lc->op)->opOID = NULL;

	return ret;
error:
	return NULL;
}

unsigned char *LCMP_get_extresp(LCMP *lc, int *rsplen){
	unsigned char *ret;

	if((lc->op==NULL)||(lc->op->opId != LCMP_OP_EXTRSP)){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+11,NULL);
		goto error;
	}
	ret = ((LO_ExtRsp*)lc->op)->opRsp;
	((LO_ExtRsp*)lc->op)->opRsp = NULL;

	*rsplen = ((LO_ExtRsp*)lc->op)->rspLen;

	return ret;
error:
	return NULL;
}

/*-----------------------------------------
  LCMP renewal
-----------------------------------------*/
int LCMP_renewal_s(LCMP *lc, int sn, void *req, enum csrform csrFormat){
	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_renewalreq(sn,csrFormat,req,0))==NULL) goto error;
	
	if(LCMP_write(lc)<0) goto error;
	LCMP_op_free(lc->op); lc->op=NULL;

	/* wait a response from CA */
	if(LCMP_read(lc)<0) goto error;

	if(lc->op==NULL) goto error;
	if(lc->op->resultCode != LCMP_SUCCESS){
		return lc->op->resultCode;
	}
	if(lc->op->opId != LCMP_OP_RENEWALRSP){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+12,NULL);
		goto error;
	}
	
	return 0;
error:
	return -1;
}

Cert *LCMP_get_renewalcert(LCMP *lc){
	Cert *ret;
	if((lc->op==NULL)||(lc->op->opId != LCMP_OP_RENEWALRSP)){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_LCMP,ERR_PT_LCMPAPI+12,NULL);
		goto error;
	}
	ret = ((LO_RenewalRsp*)lc->op)->cert;
	((LO_RenewalRsp*)lc->op)->cert = NULL;
	return ret;
error:
	return NULL;
}
