/* lcmp.c */
/*
 * Copyright (c) 2004-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *	this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *	this list of conditions and the following disclaimer in the documentation
 *	and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *	display the following acknowledgment:
 *	"This product includes software developed by Akira Iwata Laboratory,
 *	Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *	acknowledgment:
 *	"This product includes software developed by Akira Iwata Laboratory,
 *	 Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>

#include "ok_caerr.h"
#include "ok_ca.h"
#include "ok_lcmp.h"

/*-----------------------------------------
  struct LCMP alloc & free
-----------------------------------------*/
LCMP *LCMP_new(void){
	LCMP *ret;

	if((ret=(LCMP*)malloc(sizeof(LCMP)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(LCMP));
	return ret;
}

void LCMP_free(LCMP *lc){
	if(lc==NULL) return;

	CA_free(lc->ca);
	SSL_free(lc->sock); /* should be closed */
	LCMP_op_free(lc->op);

	if(lc->svname) free(lc->svname);
	if(lc->caname) free(lc->caname);
	if(lc->store) free(lc->store);
	if(lc->certid) free(lc->certid);
	if(lc->clcert) free(lc->clcert);
	if(lc->clctpw) free(lc->clctpw);
	if(lc->clkeyf) free(lc->clkeyf);
	if(lc->username) free(lc->username);

	free(lc);
}

/*-----------------------------------------
  struct LCMP op alloc & free
-----------------------------------------*/
LCMPOp *LCMP_op_new(int op){
	LCMPOp *ret=NULL;

	switch(op){
	case LCMP_OP_BINDREQ: ret=(LCMPOp*)LCMP_bindreq_new(); break;
	case LCMP_OP_BINDRSP: ret=(LCMPOp*)LCMP_bindrsp_new(); break;
	case LCMP_OP_UNBIND:  ret=(LCMPOp*)LCMP_unbind_new(); break;
	case LCMP_OP_SIGNREQ: ret=(LCMPOp*)LCMP_signreq_new(); break;
	case LCMP_OP_SIGNRSP: ret=(LCMPOp*)LCMP_signrsp_new(); break;
	case LCMP_OP_LISTREQ: ret=(LCMPOp*)LCMP_listreq_new(); break;
	case LCMP_OP_LISTRSP: ret=(LCMPOp*)LCMP_listrsp_new(); break;
	case LCMP_OP_PROFREQ: ret=(LCMPOp*)LCMP_profreq_new(); break;
	case LCMP_OP_PROFRSP: ret=(LCMPOp*)LCMP_profrsp_new(); break;
	case LCMP_OP_CERTREQ: ret=(LCMPOp*)LCMP_certreq_new(); break;
	case LCMP_OP_CERTRSP: ret=(LCMPOp*)LCMP_certrsp_new(); break;
	case LCMP_OP_CSRREQ:  ret=(LCMPOp*)LCMP_csrreq_new(); break;
	case LCMP_OP_CSRRSP:  ret=(LCMPOp*)LCMP_csrrsp_new(); break;
	case LCMP_OP_CRLREQ:  ret=(LCMPOp*)LCMP_crlreq_new(); break;
	case LCMP_OP_CRLRSP:  ret=(LCMPOp*)LCMP_crlrsp_new(); break;
	case LCMP_OP_SVOPREQ: ret=(LCMPOp*)LCMP_svopreq_new(); break;
	case LCMP_OP_SVOPRSP: ret=(LCMPOp*)LCMP_svoprsp_new(); break;
	case LCMP_OP_EXTREQ:  ret=(LCMPOp*)LCMP_extreq_new(); break;
	case LCMP_OP_EXTRSP:  ret=(LCMPOp*)LCMP_extrsp_new(); break;
	case LCMP_OP_RENEWALREQ: ret=(LCMPOp*)LCMP_renewalreq_new(); break;
	case LCMP_OP_RENEWALRSP: ret=(LCMPOp*)LCMP_renewalrsp_new(); break;
	}

	ret->opId = op;

	return ret;
}

void LCMP_op_free(LCMPOp *op){
	if(op==NULL) return;

	if(op->resultMsg) free(op->resultMsg);
	if(op->der) free(op->der);
	LCMP_ctrl_free_all(op->ctrl);

	switch(op->opId){
	  case LCMP_OP_BINDREQ: LCMP_bindreq_free((LO_BindReq*)op); break;
	  case LCMP_OP_BINDRSP: LCMP_bindrsp_free((LO_BindRsp*)op); break;
	  case LCMP_OP_UNBIND:  break;
	  case LCMP_OP_SIGNREQ: LCMP_signreq_free((LO_SignReq*)op); break;
	  case LCMP_OP_SIGNRSP: LCMP_signrsp_free((LO_SignRsp*)op); break;
	  case LCMP_OP_LISTREQ: LCMP_listreq_free((LO_ListReq*)op); break;
	  case LCMP_OP_LISTRSP: LCMP_listrsp_free((LO_ListRsp*)op); break;
	  case LCMP_OP_PROFREQ: LCMP_profreq_free((LO_ProfReq*)op); break;
	  case LCMP_OP_PROFRSP: LCMP_profrsp_free((LO_ProfRsp*)op); break;
	  case LCMP_OP_CERTREQ: LCMP_certreq_free((LO_CertReq*)op); break;
	  case LCMP_OP_CERTRSP: LCMP_certrsp_free((LO_CertRsp*)op); break;
	  case LCMP_OP_CSRREQ:  LCMP_csrreq_free((LO_CSRReq*)op); break;
	  case LCMP_OP_CSRRSP:  LCMP_csrrsp_free((LO_CSRRsp*)op); break;
	  case LCMP_OP_CRLREQ:  LCMP_crlreq_free((LO_CRLReq*)op); break;
	  case LCMP_OP_CRLRSP:  LCMP_crlrsp_free((LO_CRLRsp*)op); break;
	  case LCMP_OP_SVOPREQ: LCMP_svopreq_free((LO_SVOpReq*)op); break;
	  case LCMP_OP_SVOPRSP: LCMP_svoprsp_free((LO_SVOpRsp*)op); break;
	  case LCMP_OP_EXTREQ:  LCMP_extreq_free((LO_ExtReq*)op); break;
	  case LCMP_OP_EXTRSP:  LCMP_extrsp_free((LO_ExtRsp*)op); break;
	  case LCMP_OP_RENEWALREQ: LCMP_renewalreq_free((LO_RenewalReq*)op); break;
	  case LCMP_OP_RENEWALRSP: LCMP_renewalrsp_free((LO_RenewalRsp*)op); break;
	}

	free(op);
}

/*-----------------------------------------
  LCMP ctrl alloc & free
-----------------------------------------*/
LCMPCtrl *LCMP_ctrl_new(void){
	LCMPCtrl *ret;

	if((ret=(LCMPCtrl*)malloc(sizeof(LCMPCtrl)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP+1,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(LCMPCtrl));
	return ret;
}

void LCMP_ctrl_free(LCMPCtrl *ctrl){
	if(ctrl==NULL) return;
	if(ctrl->ctrlType) free(ctrl->ctrlType);
	if(ctrl->value) free(ctrl->value);
	free(ctrl);
}

void LCMP_ctrl_free_all(LCMPCtrl *top){
	LCMPCtrl *next;
	while(top){
		next = top->next;
		LCMP_ctrl_free(top);
		top = next;
	}
}


/*-----------------------------------------
  LCMP op
-----------------------------------------*/
LO_BindReq *LCMP_bindreq_new(void){
	LO_BindReq *ret;

	if((ret=(LO_BindReq*)malloc(sizeof(LO_BindReq)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP+2,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(LO_BindReq));
	return ret;
}

void LCMP_bindreq_free(LO_BindReq *op){
	if(op->caname) free(op->caname);
	if(op->username) free(op->username);
	if(op->passwd) free(op->passwd);
}

LO_BindRsp *LCMP_bindrsp_new(void){
	LO_BindRsp *ret;

	if((ret=(LO_BindRsp*)malloc(sizeof(LO_BindRsp)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP+2,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(LO_BindRsp));
	return ret;
}

void LCMP_bindrsp_free(LO_BindRsp *op){
	Certlist_free_all(op->list);
}

/*-----------------*/
LCMPOp *LCMP_unbind_new(void){
	LCMPOp *ret;

	if((ret=(LCMPOp*)malloc(sizeof(LCMPOp)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP+3,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(LCMPOp));
	return ret;
}

/*-----------------*/
LO_SignReq *LCMP_signreq_new(void){
	LO_SignReq *ret;

	if((ret=(LO_SignReq*)malloc(sizeof(LO_SignReq)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP+4,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(LO_SignReq));
	return ret;
}

void LCMP_signreq_free(LO_SignReq *op){
	if(op->profName) free(op->profName);
	Req_free(op->p10);
	CMP_certtmpl_free(op->tmpl);
}

LO_SignRsp *LCMP_signrsp_new(void){
	LO_SignRsp *ret;

	if((ret=(LO_SignRsp*)malloc(sizeof(LO_SignRsp)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP+4,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(LO_SignRsp));
	return ret;
}

void LCMP_signrsp_free(LO_SignRsp *op){
	Cert_free(op->cert);
}

/*-----------------*/
LO_ListReq *LCMP_listreq_new(void){
	LO_ListReq *ret;

	if((ret=(LO_ListReq*)malloc(sizeof(LO_ListReq)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP+5,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(LO_ListReq));
	return ret;
}

void LCMP_listreq_free(LO_ListReq *op){
	if(op->profName) free(op->profName);
	if(op->sbjQuery) free(op->sbjQuery);
}

LO_ListRsp *LCMP_listrsp_new(void){
	LO_ListRsp *ret;

	if((ret=(LO_ListRsp*)malloc(sizeof(LO_ListRsp)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP+5,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(LO_ListRsp));
	return ret;
}

void LCMP_listrsp_free(LO_ListRsp *op){
	CertStat_free_all(op->stat);
}

/*-----------------*/
LO_ProfReq *LCMP_profreq_new(void){
	LO_ProfReq *ret;

	if((ret=(LO_ProfReq*)malloc(sizeof(LO_ProfReq)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP+6,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(LO_ProfReq));
	return ret;
}

void LCMP_profreq_free(LO_ProfReq *op){
	if(op->profName) free(op->profName);
	if(op->profRename) free(op->profRename);
	ProfTmpl_free(op->prof);
	CertExt_free_all(op->ext);
}

LO_ProfRsp *LCMP_profrsp_new(void){
	LO_ProfRsp *ret;

	if((ret=(LO_ProfRsp*)malloc(sizeof(LO_ProfRsp)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP+6,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(LO_ProfRsp));
	return ret;
}

void LCMP_profrsp_free(LO_ProfRsp *op){
	int i;
	ProfTmpl_free(op->prof);
	CertExt_free_all(op->ext);
	if(op->proflist){
		for(i=0; i<op->listnum; i++)
			if(op->proflist[i]) free(op->proflist[i]);
		free(op->proflist);
	}
}

/*-----------------*/
ProfTmpl *ProfTmpl_new(void){
	ProfTmpl *ret;

	if((ret=(ProfTmpl*)malloc(sizeof(ProfTmpl)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP+7,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(ProfTmpl));
	memset(ret->policyFlag,0xff,16);
	ret->certSec   = -1;
	ret->certVer   = -1;
	ret->baseNum   = -1;
	ret->keyType   = -1;
	ret->keyLength = -1;
	ret->hashType  = -1;
	ret->updSec    = -1;
	ret->bgtype    = -1;
	ret->edtype    = -1;
	ret->updtype   = -1;

	return ret;
}

void ProfTmpl_free(ProfTmpl *pt){
	if(pt==NULL) return;
	if(pt->tmplName) free(pt->tmplName);
	cert_dn_free(&pt->sbjTmpl);
	free(pt);
}

ProfTmpl *ProfTmpl_dup(ProfTmpl *org){
	ProfTmpl *ret=NULL;

	if(org==NULL) goto error;

	if((ret=ProfTmpl_new())==NULL) goto error;
	memcpy(ret,org,sizeof(ProfTmpl));

	ret->tmplName = NULL;
	if(org->tmplName){
		if ((ret->tmplName = strdup(org->tmplName)) == NULL){
			OK_set_error(ERR_ST_STRDUP,ERR_LC_LCMP,ERR_PT_LCMP+7,NULL);
			goto error;
	}}
	if(org->sbjTmpl.num > 0){
		if(Cert_dncopy(&org->sbjTmpl,&ret->sbjTmpl)) goto error;	
	}
	return ret;
error:
	ProfTmpl_free(ret);
	return NULL;
}

/*-----------------*/
LO_CertReq *LCMP_certreq_new(void){
	LO_CertReq *ret;

	if((ret=(LO_CertReq*)malloc(sizeof(LO_CertReq)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP+8,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(LO_CertReq));
	return ret;
}

void LCMP_certreq_free(LO_CertReq *op){
	if(op->passwd) free(op->passwd);
	Key_free(op->key);
}

LO_CertRsp *LCMP_certrsp_new(void){
	LO_CertRsp *ret;

	if((ret=(LO_CertRsp*)malloc(sizeof(LO_CertRsp)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP+8,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(LO_CertRsp));
	return ret;
}

void LCMP_certrsp_free(LO_CertRsp *op){
	Cert_free(op->cert);
	Key_free(op->key);
}

/*-----------------*/
LO_CSRReq *LCMP_csrreq_new(void){
	LO_CSRReq *ret;

	if((ret=(LO_CSRReq*)malloc(sizeof(LO_CSRReq)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP+9,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(LO_CSRReq));
	return ret;
}

void LCMP_csrreq_free(LO_CSRReq *op){
	if(op->profName) free(op->profName);
	Req_free(op->csr);
	CMP_certtmpl_free(op->tmpl);
}

LO_CSRRsp *LCMP_csrrsp_new(void){
	LO_CSRRsp *ret;

	if((ret=(LO_CSRRsp*)malloc(sizeof(LO_CSRRsp)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP+9,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(LO_CSRRsp));
	return ret;
}

void LCMP_csrrsp_free(LO_CSRRsp *op){
	Req_free(op->csr);
	CMP_certtmpl_free(op->tmpl);
}

/*-----------------*/
LO_CRLReq *LCMP_crlreq_new(void){
	LO_CRLReq *ret;

	if((ret=(LO_CRLReq*)malloc(sizeof(LO_CRLReq)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP+10,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(LO_CRLReq));
	return ret;
}

void LCMP_crlreq_free(LO_CRLReq *op){
	if(op->profName) free(op->profName);
	if(op->path) free(op->path);
}

LO_CRLRsp *LCMP_crlrsp_new(void){
	LO_CRLRsp *ret;

	if((ret=(LO_CRLRsp*)malloc(sizeof(LO_CRLRsp)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP+10,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(LO_CRLRsp));
	return ret;
}

void LCMP_crlrsp_free(LO_CRLRsp *op){
	CRLlist_free_all(op->list);
}

/*-----------------*/
LO_SVOpReq *LCMP_svopreq_new(void){
	LO_SVOpReq *ret;

	if((ret=(LO_SVOpReq*)malloc(sizeof(LO_SVOpReq)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP+11,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(LO_SVOpReq));
	return ret;
}

void LCMP_svopreq_free(LO_SVOpReq *op){
	CA_ai_free_all(op->uinfo);
}

LO_SVOpRsp *LCMP_svoprsp_new(void){
	LO_SVOpRsp *ret;

	if((ret=(LO_SVOpRsp*)malloc(sizeof(LO_SVOpRsp)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP+11,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(LO_SVOpRsp));
	return ret;
}

void LCMP_svoprsp_free(LO_SVOpRsp *op){
	if(op->opRsp) free(op->opRsp);
}

/*-----------------*/
LO_ExtReq *LCMP_extreq_new(void){
	LO_ExtReq *ret;

	if((ret=(LO_ExtReq*)malloc(sizeof(LO_ExtReq)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP+12,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(LO_ExtReq));
	return ret;
}

void LCMP_extreq_free(LO_ExtReq *op){
	if(op->opOID) free(op->opOID);
	if(op->opInfo) free(op->opInfo);
}

LO_ExtRsp *LCMP_extrsp_new(void){
	LO_ExtRsp *ret;

	if((ret=(LO_ExtRsp*)malloc(sizeof(LO_ExtRsp)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP+12,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(LO_ExtRsp));
	return ret;
}

void LCMP_extrsp_free(LO_ExtRsp *op){
	if(op->opOID) free(op->opOID);
	if(op->opRsp) free(op->opRsp);
}

/*-----------------*/
LO_RenewalReq *LCMP_renewalreq_new(void){
	LO_RenewalReq *ret;

	if((ret=(LO_RenewalReq*)malloc(sizeof(LO_RenewalReq)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP+13,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(LO_RenewalReq));
	return ret;
}

void LCMP_renewalreq_free(LO_RenewalReq *op){
	Req_free(op->p10);
	CMP_certtmpl_free(op->tmpl);
}

LO_RenewalRsp *LCMP_renewalrsp_new(void){
	LO_RenewalRsp *ret;

	if((ret=(LO_RenewalRsp*)malloc(sizeof(LO_RenewalRsp)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_LCMP,ERR_PT_LCMP+13,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(LO_RenewalRsp));
	return ret;
}

void LCMP_renewalrsp_free(LO_RenewalRsp *op){
	Cert_free(op->cert);
}

