/* ca_tool.c */
/*
 * Copyright (c) 2004-2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>

#include "ok_caerr.h"
#include "ok_ca.h"

extern char path[];
extern char caf[];
extern char info[];

/*-----------------------------------------
  count issued certificates
-----------------------------------------*/
int CA_now_certnum(CA *ca){
	CertProf *cpf;
	int ret=0;

	for(cpf=ca->profList; cpf; cpf=cpf->next)
		ret+=Prof_now_certnum(cpf);

	return ret;
}

/*-----------------------------------------
  is the serial num usable ?
  output : 0 ... no problem to use
         : 1 ... it's already used
         : -1... system error
-----------------------------------------*/
int CA_can_serialNum_(CA *ca,int sn){
	CertStat *st;

	if(ca->cert->serialNumber == sn){
		OK_set_error(ERR_ST_CA_BADSN,ERR_LC_CA,ERR_PT_CATOOL+2,NULL);
		return -1;
	}

	if((st=CA_find_stat(ca,sn)) != NULL){
		if(!(st->state&STAT_EXPIRED)){
			OK_set_error(ERR_ST_CA_BADSN,ERR_LC_CA,ERR_PT_CATOOL+2,NULL);
			return 1;
		}
		CA_delete_stat(st);
	}
	return 0;
}

int CA_can_serialNum(CA *ca,Cert *req,int sn){
	int ret = CA_can_serialNum_(ca,sn);
	if(ret == 0) req->serialNumber = sn;
	return ret;
}

/*-----------------------------------------
  is certificate possible to resign ?
  output : 0 ... no problem
         : 1 ... not possible
         : -1... system error;
-----------------------------------------*/
int CA_can_resign(CA *ca,Cert *ct){
	CertStat *st;
	int ret = -1;

	/* check subject */
	if(strcmp(ca->subject,ct->issuer)){
		OK_set_error(ERR_ST_CA_BADSUBJECT,ERR_LC_CA,ERR_PT_CATOOL+3,NULL);
		return 1;
	}

	/* check revoked */
	if((st=CA_find_stat(ca,ct->serialNumber))==NULL) return 1;
	if((ca->cprof=CA_find_prof_sn(ca,ct->serialNumber))==NULL) return 1;

	if(st->state&STAT_REVOKED){
		OK_set_error(ERR_ST_CA_CTREVOKED,ERR_LC_CA,ERR_PT_CATOOL+3,NULL);
		return 1;
	}

	/* check in renewal period */
	if((ret=CA_inrenewal_check(st,ca->cprof->upd_sec))) return ret;

	return 0;
}

/* 0..in renewal period, 1..not in renewal period */
int CA_inrenewal_check(CertStat *st, int upd_sec){
	time_t t1,t2;

	time(&t1);

	if((t2 = timegm(&st->notAfter))<0) return -1; /* return utc */
	t2 -= upd_sec;

	if(t2 > t1){
		OK_set_error(ERR_ST_CA_NOTINUPD,ERR_LC_CA,ERR_PT_CATOOL+4,NULL);
		return 1;
	}
	return 0;
}

int CA_replace_subject(CertDN *tmpl,Req *req){
	return CA_replace_subject_(tmpl,req,NULL);
}

int CA_replace_subject_(CertDN *tmpl,Req *req,unsigned char *pol_flag){
	CertDN tmp;
	int i,j,k,ok = -1;
	int tag[4]={OBJ_DIR_OU,OBJ_DIR_CN,OBJ_DIR_UID,OBJ_DIR_EMAIL};

	cert_dn_init(&tmp);

	if(Cert_dncopy(tmpl,&tmp)) goto done;
	j = tmp.num;

	for(k=0; k<4; k++){
		/* check policy flag */
		/* normally, CSR's OU should be replaced (ignored).
		 * but, in this flag, CSR's first OU will be added
		 * after subject template DN.
		 */
		if(k==0 && pol_flag==NULL) continue;
		if(k==0 && !(pol_flag[1]&CPF_WP1_replaceDNWithoutOU)) continue;

		if(Cert_find_dn(&req->subject_dn,tag[k],&i,0)){
			/* copy RDN */
			tmp.rdn[j].derform = req->subject_dn.rdn[i].derform;
			tmp.rdn[j].tagoid  = req->subject_dn.rdn[i].tagoid;
			if ((tmp.rdn[j].tag = strdup(req->subject_dn.rdn[i].tag)) == NULL){
				OK_set_error(ERR_ST_STRDUP,ERR_LC_CA,ERR_PT_CATOOL+5,NULL);
				goto done;
			}
			tmp.num++; j++;
		}
	}

	cert_dn_free(&req->subject_dn);
	free(req->subject); req->subject=NULL;
	if(Cert_dncopy(&tmp,&req->subject_dn)) goto done;
	if((req->subject=Cert_subject_str(&tmp))==NULL) goto done;

	ok = 0;
done:
	cert_dn_free(&tmp);
	return ok;
}

/*-----------------------------------------
  CA policy check with CSR
  output : 0 ... no error
         : -1... policy mismatch
-----------------------------------------*/
int CA_dnexist_check(CA *ca, CertProf *cpf, Cert *ct){
	/* tkind is 0(C),1(ST),2(L),3(O),4(OU),5(CN),6(UID),7(EMAIL) */
	int i,j,k,flag;
	int kd[] ={OBJ_DIR_C, OBJ_DIR_ST, OBJ_DIR_L, OBJ_DIR_O, 
		       OBJ_DIR_OU, OBJ_DIR_CN, OBJ_DIR_UID, OBJ_DIR_EMAIL};
	int num  = ct->subject_dn.num;
	int cnum = cpf->sbj_tmpl.num;

	if(cpf == NULL){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_CA,ERR_PT_CATOOL+6,NULL);
		return -1;
	}

	for(i=0;i<8;i++){
		k = (cpf->pol_flag[4]<<8)|(cpf->pol_flag[5]);
		k = (k >> (7-i)*2) & 0x03;

		switch(k){
		case 2:/* match */
			for(flag=k=j=0;j<cnum;j++){
				if(cpf->sbj_tmpl.rdn[j].tagoid==kd[i]){
					for(flag=1;k<num;k++){
						if(ct->subject_dn.rdn[k].tagoid==kd[i]) {
							if(strcmp(cpf->sbj_tmpl.rdn[j].tag,ct->subject_dn.rdn[k].tag)) {
								goto error;
							}else{
								flag=0;
								break;
							}
						}
					}
				}
			}
			if(flag) goto error;
			break;
		case 1:/* supplied */
			for(j=0;j<num;j++)
				if(ct->subject_dn.rdn[j].tagoid==kd[i])
					break;
			if(j==num) goto error;
			break;
		case 0:/* optional */
		default:
			break;
		}
	}

	return 0;
error:
	OK_set_error(ERR_ST_CA_POLMIS,ERR_LC_CA,ERR_PT_CATOOL+6,NULL);
	return -1;
}

/*-----------------------------------------
  check expiration of certificates
  output : 0 ... no expired cert
         : 1 ... some certs was expired
-----------------------------------------*/
int CA_expire_check(CA *ca){
	CertProf *cpf;
	int ret=0;

	for(cpf=ca->profList; cpf; cpf=cpf->next)
		ret+=Prof_expire_check(cpf);

	return ret;
}

/*-----------------------------------------
  CA add new certificate info
-----------------------------------------*/
int CA_add_cert(CA *ca,Cert *ct){
	return Prof_add_cert(ca->cprof,ct);
}

int CA_add_certfile(CA *ca,Cert *ct){
	return Prof_add_certfile(ca->cprof,ct);
}

/*-----------------------------------------
  CA delete certificate info
-----------------------------------------*/
void CA_delete_stat(CertStat *st){
	/* we don't need "(ca->stat=st->next);" */
	Prof_delete_stat(st);
}

/*-----------------------------------------
  CA update certificate info
-----------------------------------------*/
int CA_update_cert(CA *ca,Cert *ct){
	CertStat *st;
	int hk;

	if((st=CA_find_stat(ca,ct->serialNumber))==NULL) return -1;
	hk = st->state & STAT_HAVEKEY;
	st->state = STAT_OK | hk|((Cert_is_CA(ct)>0)?(STAT_CA):(0));
	memcpy(&st->notBefore,&ct->time.notBefore,sizeof(struct tm));
	memcpy(&st->notAfter,&ct->time.notAfter,sizeof(struct tm));
	memset(&st->revokedDate,0,sizeof(struct tm));
	return 0;
}

/*-----------------------------------------
  CA utility : signature algo from prvkey
  Deprecated. use select_sigalgo()[aicrypto/tool/sigalgo.c] instead.
-----------------------------------------*/
int CA_key2sigalgo(int type){
	switch(type){
	case KEY_RSA_PRV:
	case KEY_RSA_PUB:
		return OBJ_SIG_SHA1RSA;
	case KEY_DSA_PRV:
	case KEY_DSA_PUB:
		return OBJ_SIG_SHA1DSA;
	case KEY_ECDSA_PRV:
	case KEY_ECDSA_PUB:
		return OBJ_SIG_SHA1ECDSA;
	}
	return -1;
}

/*-------------------------------------------
  CA utility : find CertStat by serial num
-------------------------------------------*/
CertStat *CA_find_stat(CA *ca,int serial){
	CertProf *cpf;
	CertStat *st;

	for(cpf=ca->profList; cpf; cpf=cpf->next)
		if((st=Prof_find_stat(cpf->stat,serial)) != NULL)
			return st;

	return NULL;
}

/*-----------------------------------------
  CA utility : find CertStat by subject
-----------------------------------------*/
CertStat *CA_find_statstr(CA *ca,char *subject){
	CertProf *cpf;
	CertStat *st;

	for(cpf=ca->profList; cpf; cpf=cpf->next)
		if((st=Prof_find_statstr(cpf->stat,subject)) != NULL)
			return st;

	return NULL;
}

void CA_find_statstr_(CertProf **cur_pf,CertStat **cur_st,char *subject){
	while(*cur_pf){
		if((*cur_st=Prof_find_statstr(*cur_st,subject)) != NULL) break;
		*cur_pf=(*cur_pf)->next;
		if(*cur_pf) *cur_st=(*cur_pf)->stat;
	}
}

/*-----------------------------------------
  CA utility : find CertStat by keyID
-----------------------------------------*/
CertStat *CA_find_statkeyID(CA *ca,unsigned char *kID){
	CertProf *cpf;
	CertStat *st;

	for(cpf=ca->profList; cpf; cpf=cpf->next)
		if((st=Prof_find_statkeyID(cpf->stat,kID)) != NULL)
			return st;

	return NULL;
}

void CA_find_statkeyID_(CertProf **cur_pf,CertStat **cur_st,unsigned char *kID){
	while(*cur_pf){
		if((*cur_st=Prof_find_statkeyID(*cur_st,kID)) != NULL) break;
		*cur_pf=(*cur_pf)->next;
		if(*cur_pf) *cur_st=(*cur_pf)->stat;
	}
}

/*--------------------------------------------------
  CA utility : find CertStat by part of subject
---------------------------------------------------*/
CertStat *CA_find_sbjquery(CA *ca, char *query){
	CertProf *cpf;
	CertStat *top=NULL,*st,*hd;

	for(cpf=ca->profList; cpf; cpf=cpf->next){
		if((st = Prof_find_sbjquery(cpf->stat,query)) != NULL){ /* not equal */
			if(top==NULL){
				top = hd = st;
			}else{
				hd->next = st;
			}
			while(hd && hd->next) hd = hd->next;
		}
	}
	return top;
}

/*-----------------------------------------
  CA utility : find Profile functions
-----------------------------------------*/
CertProf *CA_find_prof_sn(CA *ca,int serial){
	CertProf *cpf;
	CertStat *st;

	for(cpf=ca->profList; cpf; cpf=cpf->next)
		if((st=Prof_find_stat(cpf->stat,serial)) != NULL)
			return cpf;
	return NULL;
}

CertProf *CA_find_prof_sbj(CA *ca,char *subject){
	CertProf *cpf;
	CertStat *st;

	for(cpf=ca->profList; cpf; cpf=cpf->next)
		if((st=Prof_find_statstr(cpf->stat,subject)) != NULL)
			return cpf;
	return NULL;
}


CertProf *CA_find_prof_keyid(CA *ca,unsigned char *kID){
	CertProf *cpf;
	CertStat *st;

	for(cpf=ca->profList; cpf; cpf=cpf->next)
		if((st=Prof_find_statkeyID(cpf->stat,kID)) != NULL)
			return cpf;
	return NULL;
}

/*-----------------------------------------
  CA utility : Hash utilities

  sighash2keyobj(int sig): get keyobj from sigtype.
    sig    : OBJ_CRYPT_XXX or OBJ_SIG_XXX
    return : OBJ_CRYPT_XXX (keyobj)
             -1 (error)

  keyhash2sigobj(int ktype, int hash, int oldsig):
    create sigtype from keyobj and hashobj
    ktype  : OBJ_CRYPT_XXX or -1.
             get ktype from oldsig when ktype is -1.
    hash   : OBJ_HASH_XXX or -1
             get hash from oldsig when hash is -1.
    oldsig : OBJ_SIG_XXX
    return : OBJ_SIG_XXX (sigtype)
             -1 (invalid arguments)
-----------------------------------------*/
int sighash2keyobj(int sig){
	int ret=-1;
	switch(sig){
	case OBJ_CRYPT_RSA:
	case OBJ_SIG_MD2RSA:
	case OBJ_SIGOIW_MD2RSA:
	case OBJ_SIG_MD5RSA:
	case OBJ_SIGOIW_MD5RSA:
	case OBJ_SIG_SHA1RSA:
	case OBJ_SIGOIW_SHA1RSA:
	case OBJ_SIG_SHA224RSA:
	case OBJ_SIG_SHA256RSA:
	case OBJ_SIG_SHA384RSA:
	case OBJ_SIG_SHA512RSA:
	case OBJ_SIG_SHA512224RSA:
	case OBJ_SIG_SHA512256RSA:
	case OBJ_SIG_SHA3_224RSA:
	case OBJ_SIG_SHA3_256RSA:
	case OBJ_SIG_SHA3_384RSA:
	case OBJ_SIG_SHA3_512RSA:
		ret = OBJ_CRYPT_RSA;
		break;
	case OBJ_CRYPT_DSA:
	case OBJ_SIG_SHA1DSA:
	case OBJ_SIG_SHA224DSA:
	case OBJ_SIG_SHA256DSA:
	case OBJ_SIG_SHA3_224DSA:
	case OBJ_SIG_SHA3_256DSA:
		ret = OBJ_CRYPT_DSA;
		break;
	case OBJ_CRYPT_ECDSA:
	case OBJ_SIG_SHA1ECDSA:
	case OBJ_SIG_SHA224ECDSA:
	case OBJ_SIG_SHA256ECDSA:
	case OBJ_SIG_SHA384ECDSA:
	case OBJ_SIG_SHA512ECDSA:
	case OBJ_SIG_SHA3_224ECDSA:
	case OBJ_SIG_SHA3_256ECDSA:
	case OBJ_SIG_SHA3_384ECDSA:
	case OBJ_SIG_SHA3_512ECDSA:
		ret = OBJ_CRYPT_ECDSA;
		break;
	}
	return ret;
}

int keyhash2sigobj(int ktype, int hash, int oldsig){
	int ret = -1;

	if(ktype<0) ktype = sighash2keyobj(oldsig);
	if(hash <0) hash  = obj_sig2hash(oldsig);
	
	switch(ktype){
	case OBJ_CRYPT_DSA:
		switch(hash){
		case OBJ_HASH_SHA1: ret = OBJ_SIG_SHA1DSA; break;
		case OBJ_HASH_SHA224: ret = OBJ_SIG_SHA224DSA; break;
		case OBJ_HASH_SHA256: ret = OBJ_SIG_SHA256DSA; break;
		case OBJ_HASH_SHA3_224: ret = OBJ_SIG_SHA3_224DSA; break;
		case OBJ_HASH_SHA3_256: ret = OBJ_SIG_SHA3_256DSA; break;
		}
		break;
	case OBJ_CRYPT_ECDSA:
		switch(hash){
		case OBJ_HASH_SHA1: ret = OBJ_SIG_SHA1ECDSA; break;
		case OBJ_HASH_SHA224: ret = OBJ_SIG_SHA224ECDSA; break;
		case OBJ_HASH_SHA256: ret = OBJ_SIG_SHA256ECDSA; break;
		case OBJ_HASH_SHA384: ret = OBJ_SIG_SHA384ECDSA; break;
		case OBJ_HASH_SHA512: ret = OBJ_SIG_SHA512ECDSA; break;
		case OBJ_HASH_SHA3_224: ret = OBJ_SIG_SHA3_224ECDSA; break;
		case OBJ_HASH_SHA3_256: ret = OBJ_SIG_SHA3_256ECDSA; break;
		case OBJ_HASH_SHA3_384: ret = OBJ_SIG_SHA3_384ECDSA; break;
		case OBJ_HASH_SHA3_512: ret = OBJ_SIG_SHA3_512ECDSA; break;
		}
		break;
	case OBJ_CRYPT_RSA:
		switch(hash){
		case OBJ_HASH_SHA1: ret = OBJ_SIG_SHA1RSA; break;
		case OBJ_HASH_SHA224: ret = OBJ_SIG_SHA224RSA; break;
		case OBJ_HASH_SHA256: ret = OBJ_SIG_SHA256RSA; break;
		case OBJ_HASH_SHA384: ret = OBJ_SIG_SHA384RSA; break;
		case OBJ_HASH_SHA512: ret = OBJ_SIG_SHA512RSA; break;
		case OBJ_HASH_SHA512224: ret = OBJ_SIG_SHA512224RSA; break;
		case OBJ_HASH_SHA512256: ret = OBJ_SIG_SHA512256RSA; break;
		case OBJ_HASH_SHA3_224: ret = OBJ_SIG_SHA3_224RSA; break;
		case OBJ_HASH_SHA3_256: ret = OBJ_SIG_SHA3_256RSA; break;
		case OBJ_HASH_SHA3_384: ret = OBJ_SIG_SHA3_384RSA; break;
		case OBJ_HASH_SHA3_512: ret = OBJ_SIG_SHA3_512RSA; break;
		case OBJ_HASH_MD2: ret = OBJ_SIG_MD2RSA; break;
		case OBJ_HASH_MD5: ret = OBJ_SIG_MD5RSA; break;
		}
	}
	return ret;
}

/*-----------------------------------------
  CA init, lock, unlock
-----------------------------------------*/
AILock CA_init_lock(char *ca,char *prof){
	char lcn[128];
	strcpy (lcn,LOCKDIR);
	strcat (lcn,"/");
	strncat(lcn,ca,32);
	strcat (lcn,"-");
	strncat(lcn,prof,32);
	return OK_init_lock(lcn);
}

int CA_release_lock(AILock al){
	return OK_release_lock(al);
}

int CA_lock(AILock al,int msec){
	return OK_lock(al,msec);
}

int CA_unlock(AILock *al){
	int ret;

	if(!(ret = OK_unlock(*al))) *al = NULL;
	return ret;
}
