/* ca_sign.c */
/*
 * Copyright (c) 2004-2015 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_store.h>

#include "ok_caerr.h"
#include "ok_ca.h"

/*----------------------------------------------
  Signing on the certificate
  ca .. certificate authority info [in]
  req .. csr or cert [in] and return cert [out]
----------------------------------------------*/
int CA_sign(CA *ca,Req *req){
	int i,resig=0,err=-1,sig;

	if(req->issuer) resig=1;
	
	req->version=ca->cprof->cert_ver;
	
	if(!resig){
		if(!req->serialNumber){
			for(i = ca->cprof->serialNum;
				CA_find_stat(ca,i)||(ca->cert->serialNumber==i)||(i<=0);i++);
			req->serialNumber=i;
			ca->cprof->serialNum=i+1;
		}
		if((req!=ca->cert)&& !strcmp(req->subject,ca->cert->subject)){
			OK_set_error(ERR_ST_CA_BADSUBJECT,ERR_LC_CA,ERR_PT_CASIGN,NULL);
			goto done;
		}
		if(Cert_dncopy(&(ca->cert->subject_dn),&(req->issuer_dn))) goto done;
		if((req->issuer=Cert_subject_str(&(req->issuer_dn)))==NULL) goto done;
	}
	sig = select_sigalgo(ca->cert->pubkey_algo, obj_sig2hash(ca->cprof->sigtype));
	if (sig==-1) {
		OK_set_error(ERR_ST_CA_CANCELED,ERR_LC_CA,ERR_PT_CASIGN+1,NULL);
		printf("\nalgorithm mismatch...\n");
		goto done;
	}

	if(CA_sign_validity(ca,req)) goto done;
	if(CA_sign_ext(ca,req)) goto done;
	
	/* update CA information */
	if(resig){
		if(CA_update_cert(ca,req)) goto done;
	}else{
		if(CA_add_cert(ca,req)) goto done;
	}

	/* free useless memory first */
	if(req->signature){
		free(req->signature);
		req->signature = NULL;
	}
	if(req->der) free(req->der);

	/* get signed certificate */
	OK_set_cert_sig_algo(sig);

	if((req->der = Cert_toDER(req,ca->prvkey,NULL,&i))==NULL) goto done;
	
	err=0;
done:
	return err;
}	

int CA_sign_validity(CA *ca,Cert *ct){
	time_t t;
	struct tm *stm = NULL, tmp;
	Cert *cat = ca->cert;
	int tmy = cat->time.notAfter.tm_year;

	/* set notBefore */
	/* XXX: all the values of 'bgtype' are 0, 1, 2? */
	if(ca->cprof->bgtype==0){
		time(&t);
		stm=(struct tm*)gmtime(&t);
	}else if(ca->cprof->bgtype==1){
		stm=&ca->cprof->ct_begin;
		if((t = timegm(stm))<0) return -1; /* return utc */
	}else if(ca->cprof->bgtype==2){
		time(&t);
		stm=(struct tm*)gmtime(&t);

		memcpy(&tmp,&ca->cprof->ct_begin,sizeof(struct tm));
		if(tmp.tm_year==9999) tmp.tm_year = stm->tm_year;
		if(tmp.tm_mon ==99)   tmp.tm_mon  = stm->tm_mon;
		if(tmp.tm_mday==99)   tmp.tm_mday = stm->tm_mday;
		if(tmp.tm_hour==99)   tmp.tm_hour = stm->tm_hour;
		if(tmp.tm_min ==99)   tmp.tm_min = stm->tm_min;
		if(tmp.tm_sec ==99)   tmp.tm_sec = stm->tm_sec;

		stm=&tmp;
		if((t = timegm(stm))<0) return -1; /* return utc */
	}
	/* XXX: should add an block for bgtype != 0, 1, 2 if needed */
	/* else {
	 * }
	 */
	memcpy(&ct->time.notBefore,stm,sizeof(struct tm));

	/* set notAfter */
	if(ca->cprof->edtype==0){
		t += ca->cprof->cert_sec;

		if(tmy && (timegm(&cat->time.notAfter) <t))
			stm = &cat->time.notAfter;
		else
			stm = (struct tm*)gmtime(&t); /* this is not allocated memory */
	}else if(ca->cprof->edtype==1){
		if(tmy && (stmcmp(&cat->time.notAfter,&ca->cprof->ct_end)>0))
			stm = &cat->time.notAfter;
		else
			stm = &ca->cprof->ct_end;
	}else if(ca->cprof->edtype==2){
		memcpy(&tmp,&ct->time.notBefore,sizeof(struct tm));
		tmp.tm_year += ca->cprof->ct_end.tm_year;
		tmp.tm_mon  += ca->cprof->ct_end.tm_mon + 1;
		tmp.tm_mday += ca->cprof->ct_end.tm_mday;
		tmp.tm_hour += ca->cprof->ct_end.tm_hour;
		tmp.tm_min  += ca->cprof->ct_end.tm_min;
		tmp.tm_sec  += ca->cprof->ct_end.tm_sec;
	
		stm = &tmp;
		mktime(stm); /* normalize */

		if(tmy && (stmcmp(&cat->time.notAfter,stm)>0))
			stm = &cat->time.notAfter;
	}
	memcpy(&ct->time.notAfter,stm,sizeof(struct tm));
	return 0;
}

int CA_sign_ext(CA *ca,Cert *ct){
	CertExt *et,*er,*hd=NULL,*san=NULL;
	int flg = 0;
	
	/* check original extension and clean them */
	for(et=ct->ext; et ;et=et->next){
		switch(et->extnID){
		case OBJ_P9_EXT_REQ:
			for(er=((CE_ExtReq*)et)->ext; er ;er=er->next){
				switch(er->extnID){
				case OBJ_X509v3_SbjKeyIdt:
					break;
				case OBJ_X509v3_SbjAltName:
					CertExt_free(san);
					san = CertExt_dup(er); /* ignore error */
					break;
				}
			}
			break;
		case OBJ_X509v3_SbjAltName:
			CertExt_free(san);
			san = CertExt_dup(et); /* ignore error */
			break;
		}
	}
	CertExt_free_all(ct->ext);
	
	/* get new one */
	if((ct->ext=hd=CertExt_new(-1))==NULL) goto error; /* get dummy */
	
	for(et=ca->cprof->ext; et ;et=et->next){
		if((et->extnID<=0)&&(et->objid==NULL)) continue;
		
		switch(et->extnID){
		case OBJ_X509v3_AuthKeyIdt:
			flg |= (((CE_AuthKID*)et)->keyid)?(0x4):(0);
//			flg |= (((CE_AuthKID*)et)->slen) ?(0x2):(0);
			flg |= (((CE_AuthKID*)et)->slen) ?(0x1):(0);
//			flg |= (((CE_AuthKID*)et)->authorityCertIssuer)?(0x1):(0);
			flg |= (((CE_AuthKID*)et)->authorityCertIssuer)?(0x2):(0);
			if((hd->next=Extnew_authkey_id(ca->cert,flg))==NULL) goto error;
			break;
		case OBJ_X509v3_SbjKeyIdt:
			if((hd->next=Extnew_sbjkey_id(ct))==NULL) goto error;
			break;

		case OBJ_X509v3_SbjAltName:
			if((hd->next=san) == NULL) continue;
			san = NULL;
			break;
			
		case OBJ_MOJ_RegCoInfo:
			/* pass this one */

		case OBJ_X509v3_BASIC:
		case OBJ_X509v3_NameConst:
		case OBJ_X509v3_PolicyConst:
		case OBJ_X509v3_KEY_Usage:
		case OBJ_X509v3_ExtKeyUsage:
		case OBJ_X509v3_CERT_Pol:
		case OBJ_X509v3_CertPolMap:

		case OBJ_X509v3_IssAltName:
		case OBJ_X509v3_CRL_Point:
		case OBJ_X509v3_SubDirAtt:

		case OBJ_PKIX_IDPE_AIA:
		case OBJ_PKIX_OCSP_NOCHECK:

		case OBJ_NS_CERT_CRLURL:
		case OBJ_NS_CERT_COMMENT:
		case OBJ_NS_CERT_TYPE:

		case OBJ_MOJ_JCertPol:
		case OBJ_MOJ_Registrar:
		default:
			if((hd->next=CertExt_dup(et))==NULL) goto error;
			break;
		}
		hd = hd->next;
		hd->critical = et->critical;
	}

	CertExt_free(san);
	return 0;
error:
	CertExt_free(san);
	CertExt_free_all(ct->ext); ct->ext=NULL;
	return -1;
}

int CA_check_profpol(CA *ca, CertProf *cpf, Req *req, unsigned int flags){
	unsigned char kid[32];
	CertProf *tmp,*pf = NULL;
	CertStat *st = NULL;
	Cert *ct = NULL;
	int i, err = -1;

	if(CA_dnexist_check(ca,cpf,req)){
		OK_set_error(ERR_ST_CA_BADSUBJECT,ERR_LC_CA,ERR_PT_CASIGN+3,NULL);
		err = 1; goto done;
	}
	if(cpf->pol_flag[1] & CPF_WP1_replaceWithTmplDN){ /* replace subject mode */
		if(CA_replace_subject_(&cpf->sbj_tmpl,req,cpf->pol_flag)) goto done;
	}

	pf = ca->profList;
	st = pf->stat;
	CA_find_statstr_(&pf,&st,req->subject);

	while(st){ /* find stat */
		if((flags & CHKPOL_REUSE_SAMESBJDN)&&
		   (cpf->pol_flag[0] & CPF_WP0_allowSameSbjDN)){
			/* same subject is available */
		}else if((flags & CHKPOL_REUSE_EXPIREDSBJDN)&&
			 (cpf->pol_flag[0] & CPF_WP0_allowExpdSbjDN)&&(prof_stat_expire(st,0))){
			/* expired subject is available */
		}else if((flags & CHKPOL_REUSE_REVOKEDSBJDN)&&
			 (cpf->pol_flag[0] & CPF_WP0_allowRvkdSbjDN)&&(st->state&STAT_REVOKED)){
			/* revoked subject is available */
		}else if((flags & CHKPOL_REUSE_SBJDNINRENEWAL)&&
			 (cpf->pol_flag[0] & CPF_WP0_allowInUpdSbjDN)&&(!CA_inrenewal_check(st,cpf->upd_sec))){
			/* subject in renewal period is available */
		}else{
			OK_set_error(ERR_ST_CA_BADSUBJECT,ERR_LC_CA,ERR_PT_CASIGN+3,NULL);
			err = 2; goto done;
		}
		st = st->next;
		CA_find_statstr_(&pf,&st,req->subject);
	}

	if(cs_get_keyhash(req->pubkey,kid,&i)){
		OK_set_error(ERR_ST_UNSUPPORTED_ALGO,ERR_LC_CA,ERR_PT_CASIGN+3,NULL);
		err = 3; goto done;
	}

	pf = ca->profList;
	st = pf->stat;
	CA_find_statkeyID_(&pf,&st,kid);

	while(st){ /* find stat */
		if((ct=Prof_get_cert(st))==NULL){
			tmp = ca->cprof;
			if((ca->cprof=CA_find_prof_sn(ca,st->serialNum))==NULL) goto done;
			i = Prof_open_ctfile(ca);
			ca->cprof = tmp;
			if(i) goto done; /* error */

			if((ct=Prof_get_cert(st))==NULL) goto done;
		}

		i = Key_cmp(req->pubkey,ct->pubkey);
		Cert_free(ct);

		if(i==0){ /* same key exists !! */
			if((flags & CHKPOL_REUSE_SAMEPUBKEY)&&
			   cpf->pol_flag[0] & CPF_WP0_allowSameKey){
				/* same public key is available */
			}else if((flags & CHKPOL_REUSE_EXPIREDPUBKEY)&&
				 (cpf->pol_flag[0] & CPF_WP0_allowExpdKey)&&(prof_stat_expire(st,0))){
				/* expired public key is available */
			}else if((flags & CHKPOL_REUSE_REVOKEDPUBKEY)&&
				 (cpf->pol_flag[0] & CPF_WP0_allowRvkdKey)&&(st->state&STAT_REVOKED)){
				/* revoked public key is available */
			}else if((flags & CHKPOL_REUSE_PUBKEYINRENEWAL)&&
				 (cpf->pol_flag[0] & CPF_WP0_allowInUpdKey)&&(!CA_inrenewal_check(st,cpf->upd_sec))){
				/* public key in renewal period is available */
			}else{
				OK_set_error(ERR_ST_BADKEY,ERR_LC_CA,ERR_PT_CASIGN+3,NULL);
				err = 4; goto done;
			}
		}
		st = st->next;
		CA_find_statkeyID_(&pf,&st,kid);
	}

	err = 0;
done:
	return err;
}



