/* ca_file.c */
/*
 * Copyright (c) 2004-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_pem.h>

#include "ok_caerr.h"
#include "ok_ca.h"

/*-----------------------------------------
  read CA information
-----------------------------------------*/
CA *CA_info_read(char *path){
	unsigned char *der=NULL;
	char buf[256],tmp[256];
	int i,sz;
	CA *ret=NULL;

	if (set_path(buf, 256, path, "/ca.cai", NULL) == -1) {
		goto error;
	}

	if((der = ASN1_read_der(buf))==NULL) goto error;

	if((ret=ASN1_CA_info(der))==NULL) goto error;
	ret->der=der; der=NULL;

	/* get current CA path */
	if(!strcmp(path,".")){
		getcwd(tmp,256);
		path = tmp;
	}

	/* set CA path & name */
	if ((ret->ca_path = strdup(path)) == NULL){
		OK_set_error(ERR_ST_STRDUP,ERR_LC_CA,ERR_PT_CAFILE,NULL);
		goto error;
	}

	sz = strlen(path);
	if(path[sz-1] == '/')
	{
		path[sz-1] = 0;
	}

	for(i=sz-1; i>0; i--){
		if(path[i] == '/')
		{
			i++; break;
		}
	}

	if ((ret->ca_name = strdup(&path[i])) == NULL){
		OK_set_error(ERR_ST_STRDUP,ERR_LC_CA,ERR_PT_CAFILE,NULL);
		goto error;
	}

	if(prof_get_curstat(buf,&ret->ca_mtime,&sz)) goto error;

	return ret;
error:
	if(ret && ret->p11) ret->p11_nofinal=1; /* HSM should not stop with finalize */
	CA_free(ret);
	if(der){free(der);}
	return NULL;
}

/*-----------------------------------------
  read CA information
-----------------------------------------*/
int CA_info_reload(CA *ca){
	unsigned char *der=NULL;
	CertProf *cp,*cpf = NULL;
	CRLProf *lp,*lpf = NULL;
	CA *tmp = NULL;
	char buf[256];
	time_t cur;
	int sz,ok=-1;

	if (set_path(buf, 256,
		     ca->ca_path, "/ca.cai", NULL) == -1) {
		goto done;
	}

	if(prof_get_curstat(buf,&cur,&sz)) goto done;

	if(cur != ca->ca_mtime){
		if((der = ASN1_read_der(buf))==NULL) goto done;
		if((tmp = ASN1_CA_info(der))==NULL) goto done;
		for(cpf=tmp->profList;cpf;cpf=cpf->next){
			cpf->pCA = ca;
			cpf->lock = (cp=Prof_find(ca,cpf->name))?(cp->lock):(NULL);
		}
		for(lpf=tmp->crlpList;lpf;lpf=lpf->next){
			lpf->pCA = ca;
			lpf->lock = (lp=Prof_crl_find(ca,lpf->name))?(lp->lock):(NULL);
		}
		Prof_cert_free_all(ca->profList);
		Prof_crl_free_all(ca->crlpList);
		ca->cprof=ca->profList=tmp->profList; tmp->profList=NULL;
		ca->lprof=ca->crlpList=tmp->crlpList; tmp->crlpList=NULL;
		ca->ca_mtime=cur;
	}
	ok = 0;
done:
	if(tmp && tmp->p11) tmp->p11_nofinal=1; /* HSM should not stop with finalize */
	CA_free(tmp);
	if(der){free(der);}
	return ok;
}

/*-----------------------------------------
  write CA information
-----------------------------------------*/
int CA_info_write(CA *ca){
	char buf[256];

	if(ca->ca_path==NULL){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_CA,ERR_PT_CAFILE+1,NULL);
		return -1;
	}
	if (set_path(buf, 256,
		     ca->ca_path, "/ca.cai", NULL) == -1) {
		return -1;
	}

	if(ASN1_write_der(ca->der,buf)) return -1;
	printf("Update CA information.\n");
	return 0;
}

/*-----------------------------------------
  read CA pkcs#12 file
-----------------------------------------*/
int CA_p12_read(CA *ca){
	char buf[256];

	if(ca==NULL) return -1;
	if(ca->ca_path==NULL){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_CA,ERR_PT_CAFILE+2,NULL);
		return -1;
	}
	if (set_path(buf, 256,
		     ca->ca_path, "/ca.p12", NULL) == -1) {
		return -1;
	}

	printf("CA PKCS#12 file open \n");

	if((ca->p12=P12_read_file(buf))==NULL) goto error;
	
	if((ca->cert=P12_get_usercert(ca->p12))==NULL) goto error;

	if((ca->prvkey=P12_get_privatekey(ca->p12))==NULL) goto error;

	if(strcmp(ca->subject,ca->cert->subject)||
		strcmp(ca->issuer,ca->cert->issuer)){
		/* when creating a new CA, ca structure doesn't have
		 * any subject and issuer information. In this case,
		 * specially we return "1".
		 */
		return 1;
	}

	return 0;
error:
	return -1;
}

/*-----------------------------------------
  read CA pkcs#11 & pkcs#7 file
-----------------------------------------*/
int CA_p11_read(CA *ca){
	CK_SLOT_ID slot_id[8];
	CK_ULONG slot_count = 8;
	AILock lock = NULL;
	P11Key *tmp = NULL;
	char buf[256],pwd[PWD_BUFLEN];
	unsigned int i;

	if(ca==NULL) return -1;
	if(ca->ca_path==NULL){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_CA,ERR_PT_CAFILE+3,NULL);
		return -1;
	}
	strncpy(buf,ca->ca_path,240);
	strcat (buf,"/");
	strncat(buf,"ca.p7b",16);

	printf("CA HW Key (PKCS#11) open \n");

	if((ca->p12=(PKCS12*)P7b_read_file(buf))==NULL) goto error;	
	if((ca->cert=P12_get_usercert(ca->p12))==NULL) goto error;

	if(P11_get_slotlist(ca->p11, TRUE, slot_id, &slot_count)) goto error;
	if(slot_count <= 0){
		printf("cannot find active Criptoki slot.\n"); goto error;
	}
	if((lock=CA_init_lock(ca->ca_name,"pkcs11"))==NULL) goto error;

	/***** find appropriate pkcs11 slot *****/
	for(i=0; i< slot_count; i++){
		if(lock){ if(OK_lock(lock,30000)){lock=NULL; goto error;}}
		tmp = p11_open_key_(ca->p11,NULL,KEY_P11RSA_PUB,slot_id[i],
				(CKF_SERIAL_SESSION | CKF_RW_SESSION),CKU_USER,NULL,ca->label,0,1,ca->p11_mofn);
		if(lock) OK_unlock(lock);

		if(tmp){ OK_clear_error(); P11key_free(tmp); tmp=NULL; break; }
	}
	if(i >= slot_count) goto error;
	/***** end slot check *****/

	OK_get_passwd("Token Login PIN : ",(unsigned char*)pwd,0);
	/***** start private key lock *****/
	if(lock){ if(OK_lock(lock,30000)){lock=NULL; goto error;}}
	if((ca->prvkey=(Key*)p11_open_key_(ca->p11,NULL,KEY_P11RSA_PRV,slot_id[i],
		(CKF_SERIAL_SESSION | CKF_RW_SESSION),CKU_USER,pwd,ca->label,1,1,ca->p11_mofn))==NULL) goto error;
	if(lock) OK_unlock(lock);
	/***** end private key lock *****/

	memset(pwd,0,PWD_BUFLEN);
	ca->p11s = ((P11Key*)ca->prvkey)->p11s;
	((P11Key*)ca->prvkey)->lock = lock;
	if(P12_add_key(ca->p12,ca->prvkey,NULL,0xff)) goto error;

	if(strcmp(ca->subject,ca->cert->subject)||
		strcmp(ca->issuer,ca->cert->issuer)){
		/* when creating a new CA, ca structure doesn't have
		 * any subject and issuer information. In this case,
		 * specially we return "1".
		 */
		return 1;
	}

	return 0;
error:
	if(lock) OK_unlock(lock);
	return -1;
}

/*-----------------------------------------
  output CRL & certificate file
-----------------------------------------*/
int CA_write_cert(CA *ca,Cert *cert,char *fname){
	char buf[256],tmp[32];

	printf("output certificate ..");
	fflush(stdout);

	if(PEM_write_cert(cert,fname)) return -1;

	sprintf(tmp,"%d.cer",cert->serialNumber);
	if (set_path(buf, 256,
		     ca->ca_path, "/cert/", tmp, NULL) == -1) {
		return -1;
	}

	if(PEM_write_cert(cert,buf)) return -1;
	printf(" done.\n");
	return 0;
}

int CA_write_crl(CA *ca,char *fname){
	printf("output %s ..",ca->lprof->name);
	fflush(stdout);

	if(PEM_write_crl(ca->crl,fname)) return -1;

	printf(" done.\n");
	return 0;
}

