/* ca_crl.c */
/*
 * Copyrightc 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>

#include "ok_caerr.h"
#include "ok_ca.h"

/*---------------------------------------------------
  revoke or unrevoke a certificate with serial num.
  0  .. no error
  -1 .. system error
  1  .. status error (revoked)
---------------------------------------------------*/
int CA_cert_revoke(CA *ca,int revoke,int reason){
	CertStat *st;
	time_t t;
	struct tm *stm;

	if((st=CA_find_stat(ca,revoke))==NULL){
		OK_set_error(ERR_ST_CA_BADSN,ERR_LC_CA,ERR_PT_CACRL,NULL);
		return -1;
	}
	if(st->state&STAT_REVOKED){
		OK_set_error(ERR_ST_CA_CTREVOKED,ERR_LC_CA,ERR_PT_CACRL,NULL);
		return 1;
	}

	st->state |= STAT_REVOKED | (reason&0x00ff0000);

	time(&t);

	stm=(struct tm*)gmtime(&t);
	memcpy(&st->revokedDate,stm,sizeof(struct tm));

	return 0;
}

int CA_cert_unrevoke(CA *ca,int unrev){
	CertStat *st;

	if((st=CA_find_stat(ca,unrev))==NULL){
		OK_set_error(ERR_ST_CA_BADSN,ERR_LC_CA,ERR_PT_CACRL+1,NULL);
		return -1;
	}
	st->state &= ~STAT_REVOKED;

	memset(&st->revokedDate,0,sizeof(struct tm));

	return 0;  
}

/*-----------------------------------------
  create new CRL
-----------------------------------------*/
int CA_set_CRL(CA *ca){
	CRL *crl;
	int i,sig;

	if(ca->crl) CRL_free(ca->crl);
	if((crl=ca->crl=CRL_new())==NULL) goto error;

	/* set CRL signature algorithm */
	sig = select_sigalgo(ca->cert->pubkey_algo, obj_sig2hash(ca->lprof->sigtype));
	if (sig == -1) {
		OK_set_error(ERR_ST_CA_CANCELED,ERR_LC_CA,ERR_PT_CACRL+2,NULL);
		printf("\nalgorithm mismatch...\n");
		goto error;
	}
	OK_set_crl_sig_algo(sig);

	crl->version = ca->lprof->crl_ver;
	if(Cert_dncopy(&(ca->cert->subject_dn),&(crl->issuer_dn))) goto error;
	if((crl->issuer= Cert_subject_str(&(crl->issuer_dn)))==NULL) goto error;

	if(CA_crl_last_next(ca,crl)) goto error;
	if(CA_crl_set_revoked(ca,crl,&i)) goto error;

	if(crl->version)
		if(CA_crl_set_ext(ca,crl)) goto error;

	if((crl->der=CRL_toDER(crl,ca->prvkey,NULL,&i))==NULL) goto error;

	return 0;
error:
	CRL_free(crl);
	ca->crl=NULL;
	return -1;
}


int CA_crl_last_next(CA *ca,CRL *crl){
	time_t s,t;
	struct tm *stm;
	int i;

	time(&t);
	if(ca->lprof->bgtype==0){
	    stm=(struct tm*)gmtime(&t);
	}else{
	    /* specify start time -- calcurate appropriate start time */
	    if((s = timegm(&ca->lprof->cl_begin))<0) return -1; /* return utc */
	    
	    if(s > t){ /* start time is future */
		t = s;
	    }else{ /* start time is past */
		i = (int)((t - s)/(ca->lprof->crl_sec));
		t = s + ca->lprof->crl_sec * i;
	    }
	    stm = (struct tm*)gmtime(&t);
	}
	memcpy(&crl->lastUpdate,stm,sizeof(struct tm));

	t += ca->lprof->crl_sec;
	stm = (struct tm*)gmtime(&t); /* this is not allocated memory */

	memcpy(&crl->nextUpdate,stm,sizeof(struct tm));

	return 0;
}

int CA_crl_set_revoked(CA *ca,CRL *crl,int *rev_num){
	CertProf *cpf;
	CertStat *st;
	Revoked  *rv,*tmp;
	int type=0,rsn=0;

	if(!strcmp(ca->lprof->name,"CRL")) type = 1; /* only contain User cert */
	if(!strcmp(ca->lprof->name,"ARL")) type = 2; /* only contain CA cert */

	rsn = (CertExt_find(ca->lprof->ext,OBJ_X509v3_CRLReason)!=NULL);

	*rev_num=0;
	for(cpf=ca->profList; cpf ;cpf=cpf->next){
		for(st=cpf->stat; st ;st=st->next){
			if(st->state&STAT_REVOKED){
				if((type==1)&&(st->state&(STAT_CA|STAT_XCERTFWD))) continue;
				if((type==2)&&(!(st->state&(STAT_CA|STAT_XCERTFWD)))) continue;

				if(stmcmp(&st->notAfter,&crl->lastUpdate)>0) continue;

				if((rv=Revoked_new())==NULL) goto error;
				rv->serialNumber = st->serialNum;
				memcpy(&rv->revocationDate,&st->revokedDate,sizeof(struct tm));

				if(rsn){
					if((rv->entExt=Extnew_reason_code((st->state>>16)&0xff)) == NULL)
						goto error;
				}

				tmp=crl->next;
				crl->next=rv;
				rv->next =tmp;
				(*rev_num)++;
			}
		}
	}
	return 0;
error:
	return -1;
}

int CA_crl_set_ext(CA *ca,CRL *crl){
	CertExt *et,*hd=NULL;
	int i,flg = 0;

    CertExt_free_all(crl->ext);
    crl->ext=hd=CertExt_new(-1); /* get dummy */

	for(et=ca->lprof->ext; et ;et=et->next){
		if(et->extnID < 0) continue;

		switch(et->extnID){
		case OBJ_X509v3_AuthKeyIdt:
			flg |= (((CE_AuthKID*)et)->keyid)?(0x4):(0);
//			flg |= (((CE_AuthKID*)et)->slen) ?(0x2):(0);
			flg |= (((CE_AuthKID*)et)->slen) ?(0x1):(0);
//			flg |= (((CE_AuthKID*)et)->authorityCertIssuer)?(0x1):(0);
			flg |= (((CE_AuthKID*)et)->authorityCertIssuer)?(0x2):(0);
			if((hd->next=Extnew_authkey_id(ca->cert,flg))==NULL) goto error;
			break;
		case OBJ_X509v3_CRLNumber:
			if((hd->next=Extnew_crl_number(ca->lprof->crlNum))==NULL) goto error;
			i = (ca->lprof->crlNum+1)%10000000;
			ca->lprof->crlNum = (i>0)?(i):(1);
			break;
		case OBJ_X509v3_CRLReason:
			/* just flag for entry extension */
			continue;
		case OBJ_X509v3_IssDistPoint:
		case OBJ_X509v3_IssAltName:
		default:
			if((hd->next=CertExt_dup(et))==NULL) goto error;
			break;
		}
		hd = hd->next;
		hd->critical = et->critical;
	}

	return 0;
error:
	return -1;
}
