/* ca_auth.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_io.h>
#include <aicrypto/ok_rand.h>
#include <aicrypto/ok_base64.h>

#include "ok_caerr.h"
#include "ok_ca.h"

/*-----------------------------------------
  allocate & free AuthInfo information
-----------------------------------------*/
AuthInfo *CA_ai_new(){
	AuthInfo *ret;

	if((ret=(AuthInfo*)malloc(sizeof(AuthInfo)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_AUTH,ERR_PT_AUTH,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(AuthInfo));
	return ret;
}

void CA_ai_free(AuthInfo *ai){
	if(ai==NULL) return;
	if(ai->name) free(ai->name);
	if(ai->group) free(ai->group);
	if(ai->passwd) free(ai->passwd);
	if(ai->oldpasswd) free(ai->oldpasswd);
	if(ai->opt) free(ai->opt);
	free(ai);
}

void CA_ai_free_all(AuthInfo *top){
	AuthInfo *next;

	while(top){
		next=top->next;
		CA_ai_free(top);
		top=next;
	}
}

/*-----------------------------------------
  duplicate AuthInfo information
-----------------------------------------*/
AuthInfo *CA_ai_dup(AuthInfo *org){
	AuthInfo *ret=NULL;

	if((ret=CA_ai_new())==NULL) goto error;
	ret->uid   = org->uid;
	ret->gid   = org->gid;
	ret->grant = org->grant;
	if(org->name){
		if ((ret->name = strdup(org->name)) == NULL){
			OK_set_error(ERR_ST_MEMALLOC,ERR_LC_AUTH,ERR_PT_AUTH+2,NULL);
			goto error;
		}
	}
	if(org->group){
		if ((ret->group = strdup(org->group)) == NULL){
			OK_set_error(ERR_ST_MEMALLOC,ERR_LC_AUTH,ERR_PT_AUTH+2,NULL);
			goto error;
		}
	}
	if(org->passwd){
		if ((ret->passwd = strdup(org->passwd)) == NULL){
			OK_set_error(ERR_ST_MEMALLOC,ERR_LC_AUTH,ERR_PT_AUTH+2,NULL);
			goto error;
		}
	}
	if(org->oldpasswd){
		if ((ret->oldpasswd = strdup(org->oldpasswd)) == NULL){
			OK_set_error(ERR_ST_MEMALLOC,ERR_LC_AUTH,ERR_PT_AUTH+2,NULL);
			goto error;
		}
	}
	if(org->opt){
		if ((ret->opt = strdup(org->opt)) == NULL){
			OK_set_error(ERR_ST_MEMALLOC,ERR_LC_AUTH,ERR_PT_AUTH+2,NULL);
			goto error;
		}
	}

	return ret;
error:
	CA_ai_free(ret);
	return NULL;
}

AuthInfo *CA_ai_dup_all(AuthInfo *top){
	AuthInfo *hd,*ai,*ret=NULL;

	while(top){
		if((ai=CA_ai_dup(top))==NULL) goto error;
		if(ret){
			hd->next=ai; hd=ai;
		}else{
			ret=hd=ai;
		}
		top=top->next;
	}

	return ret;
error:
	CA_ai_free_all(ret);
	return NULL;
}

/*-----------------------------------------
  read AuthInfo information
-----------------------------------------*/
AuthInfo *CA_read_authinfo(char *fname){
	AuthInfo *ret=NULL,*hd,*ai;
	char *p,*c,*line,*it,*buf=NULL;
	off_t sz;
	int i,j,k;

	if((buf=(char*)get_file2buf(fname,&sz))==NULL) goto error;

	i=strlen(buf);
	line=buf; j=0;
	while(j<i){
		if((p=strchr(line,'\n')) != NULL) *p=0;
		k=strlen(line);
		c=strchr(line,':');

		if((line[0]!='#')&&(k>5)&&(c)){
			if((ai=CA_ai_new())==NULL) goto error;
			if(ret){
				hd->next=ai; hd=ai;
			}else{
				ret=hd=ai;
			}

			/* user name */
			*c=0; c++;
			if ((ai->name = strdup(line)) == NULL){
				OK_set_error(ERR_ST_STRDUP,ERR_LC_AUTH,ERR_PT_AUTH+3,NULL);
				goto error;
			}

			/* user password (maybe hashed or null) */
			it=c;
			if((c=strchr(it,':'))==NULL) goto error;
			*c=0; c++;
			if(*it){
				if ((ai->passwd = strdup(it)) == NULL){
					OK_set_error(ERR_ST_STRDUP,ERR_LC_AUTH,ERR_PT_AUTH+3,NULL);
					goto error;
				}
			}
		
			/* user id */
			it=c;
			if((c=strchr(it,':'))==NULL) goto error;
			*c=0; c++;
			ai->uid = atoi(it);
	
			/* group id */
			it=c;
			if((c=strchr(it,':'))==NULL) goto error;
			*c=0; c++;
			ai->gid = atoi(it);
	
			/* user grant */
			it=c;
			if((c=strchr(it,':'))==NULL) goto error;
			*c=0; c++;
			ai->grant = strtol(it,(char**)NULL,16);
	
			/* optional grant constraints */
			if((*c !='\r')&&(*c !='\n')){
				if ((ai->opt = strdup(c)) == NULL){
					OK_set_error(ERR_ST_STRDUP,ERR_LC_AUTH,ERR_PT_AUTH+3,NULL);
					goto error;
				}
			}
		}
		if(p) k++;

		j+=k; line+=k;
	}

	if(buf) free(buf);
	return ret;
error:
	CA_ai_free_all(ret);
	if(buf) free(buf);
	if(OK_get_error()==0) OK_set_error(ERR_ST_BADFORMAT,ERR_LC_AUTH,ERR_PT_AUTH+3,NULL);
	return NULL;
}

/*-----------------------------------------
  write AuthInfo information
-----------------------------------------*/
int CA_write_authinfo(AuthInfo *ai,char *fname){
	FILE *fp=NULL;
	char buf[512],num[32];
	int ok=-1;

	if((fp=fopen(fname,"w"))==NULL){
		OK_set_error(ERR_ST_FILEOPEN,ERR_LC_AUTH,ERR_PT_AUTH+4,NULL);
		goto done;
	}
	while(ai){
		memset(buf,0,512);
		strncpy(buf,ai->name,256);
		strcat (buf,":");
		if(ai->passwd) strncat(buf,ai->passwd,64);
		else		   strcat (buf,"");

		sprintf(num,":%d:",ai->uid);
		strcat (buf,num);
		sprintf(num,"%d:",ai->gid);
		strcat (buf,num);
		sprintf(num,"0x%.8lx:",ai->grant); /* unsigned long grant; */
		strcat (buf,num);

		if(ai->opt) strncat(buf,ai->opt,160);
		strcat (buf,"\n");

		if(fputs(buf,fp)<0){
			OK_set_error(ERR_ST_FILEWRITE,ERR_LC_AUTH,ERR_PT_AUTH+4,NULL);
			goto done;
		}
		ai=ai->next;
	}

	ok = 0;
done:
	if(fp) fclose(fp);
	return ok;
}

/*-----------------------------------------
  AuthInfo tools
-----------------------------------------*/
AuthInfo *CA_get_authinfo(char *name, char *raw_pwd, int uid, int gid,
			  int grant, char *opt){
	AuthInfo *ret=NULL;
	char buf[128];
	
	if(name==NULL){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_AUTH,ERR_PT_AUTH+5,NULL);
		goto error;
	}

	if((ret=CA_ai_new())==NULL) goto error;
	
	if ((ret->name = strdup(name)) == NULL){
		OK_set_error(ERR_ST_STRDUP,ERR_LC_AUTH,ERR_PT_AUTH+5,NULL);
		goto error;
	}
	
	if(raw_pwd){
		memset(buf,0,128);
		if(CA_get_md5pwd(raw_pwd,buf,128)) goto error;
		if ((ret->passwd = strdup(buf)) == NULL){
			OK_set_error(ERR_ST_STRDUP,ERR_LC_AUTH,ERR_PT_AUTH+5,NULL);
			goto error;
		}
	}
	if(opt){
		if ((ret->opt = strdup(opt)) == NULL){
			OK_set_error(ERR_ST_STRDUP,ERR_LC_AUTH,ERR_PT_AUTH+5,NULL);
			goto error;
		}
	}
	ret->uid   = uid;
	ret->gid   = gid;
	ret->grant = grant;

	return ret;
error:
	CA_ai_free(ret);
	return NULL;
}

/* max should be big enough */
int CA_get_md5pwd(char *raw_pwd, char *ret, int max){
	char buf[PWD_BUFLEN+4];

	memset(buf,0,PWD_BUFLEN+4);
	if(RAND_bytes((unsigned char*)buf,2)) goto error;
	buf[0]='A'+((buf[0]&0x7f)%26);
	buf[1]='A'+((buf[1]&0x7f)%26);
	strncpy(&buf[2],raw_pwd,PWD_BUFLEN-2);

	if(md5_hash_pwd(buf,ret,max)) goto error;

	return 0;
error:
	return -1;
}

AuthInfo *CA_lcmp_authinfo(char *name, char *pwd, char *oldpwd,
						   int uid, int gid, int grant, char *opt){
	AuthInfo *ret=NULL;
	
	if(name==NULL){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_AUTH,ERR_PT_AUTH+6,NULL);
		goto error;
	}

	if((ret=CA_ai_new())==NULL) goto error;
	
	if ((ret->name = strdup(name)) == NULL){
		OK_set_error(ERR_ST_STRDUP,ERR_LC_AUTH,ERR_PT_AUTH+6,NULL);
		goto error;
	}
	if(pwd){
		if ((ret->passwd = strdup(pwd)) == NULL){
			OK_set_error(ERR_ST_STRDUP,ERR_LC_AUTH,ERR_PT_AUTH+6,NULL);
			goto error;
		}
	}
	if(oldpwd){
		if ((ret->oldpasswd = strdup(oldpwd)) == NULL){
			OK_set_error(ERR_ST_STRDUP,ERR_LC_AUTH,ERR_PT_AUTH+6,NULL);
			goto error;
		}
	}
	if(opt){
		if ((ret->opt = strdup(opt)) == NULL){
			OK_set_error(ERR_ST_STRDUP,ERR_LC_AUTH,ERR_PT_AUTH+6,NULL);
			goto error;
		}
	}
	ret->uid   = uid;
	ret->gid   = gid;
	ret->grant = grant;

	return ret;
error:
	CA_ai_free(ret);
	return NULL;
}

int md5_hash_pwd(char *in, char *ret, int max){
	unsigned char hash[32];
	char *bs=NULL;
	int ok=-1;

	OK_MD5(strlen(in),(unsigned char*)in,hash);
	if((bs=Base64_encode(16,hash,16))==NULL) goto done;

	strncpy(ret,in,2);
	strncpy(&ret[2],bs,max-2);

	ok = 0;
done:
	if(bs) free(bs);
	return ok;
}

/* compare raw password and hashed password
 * if two password is good pair, return 0.
 * otherwise if they are not pair, return 1;
 * if system error occurs, return -1;
 */
int CA_cmp_md5pwd(char *raw_pwd, char *md5_pwd){
	char in[PWD_BUFLEN+4],buf[128];

	if((raw_pwd==NULL)||(md5_pwd==NULL)){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_AUTH,ERR_PT_AUTH+7,NULL);
		goto error;
	}
	memset(in,0,PWD_BUFLEN+4);
	memset(buf,0,128);
	strncpy(in,md5_pwd,2);
	strncpy(&in[2],raw_pwd,PWD_BUFLEN-2);

	if(md5_hash_pwd(in,buf,128)) goto error;

	return (!strcmp(buf,md5_pwd))?(0):(1);
error:
	return -1;
}

/* find authentication info */
AuthInfo *CA_ai_findbyname(AuthInfo *top, char *name){
	char buf[256];
	int i=strlen(name);
	
	memset(buf,0,256);
	strncpy(buf,name,254);

	if((i<256)&&(buf[i-2]==',')&&(buf[i-1]==' ')) buf[i-2]=0; /* clear */
	
	while(top){
		if(!strcmp(top->name,buf)) return top;
		top=top->next;
	}
	return NULL;
}

AuthInfo *CA_ai_findbyuid(AuthInfo *top, int uid){
	while(top){
		if(top->uid == uid) return top;
		top=top->next;
	}
	return NULL;
}

/*-----------------------------------------
  read Profile group information
-----------------------------------------*/
int CA_read_groupinfo(CA *ca){
	CertProf *cpf;
	CRLProf *lpf;
	char *p,*c,*line,*it,*buf=NULL,path[256];
	off_t sz;
	int i,j,k,ok=-1;

	if (set_path(path, 256,
		     ca->ca_path, "/ca.group", NULL) == -1) {
		return -1;
	}

	if((buf=(char*)get_file2buf(path,&sz))==NULL) goto done;

	i=strlen(buf);
	line=buf; j=0;
	while(j<i){
		if((p=strchr(line,'\n')) != NULL) *p=0;
		k=strlen(line);
		c=strchr(line,':');

		if((line[0]!='#')&&(k>5)&&(c)){
			/* profile name */
			*c=0; c++;
			cpf = Prof_find(ca,line);
			lpf = Prof_crl_find(ca,line);

			/* password -- ignore */
			it=c;
			if((c=strchr(it,':'))==NULL) goto done;
			*c=0; c++;
	
			/* group id */
			it=c;
			if((c=strchr(it,':'))==NULL) goto done;
			*c=0; c++;
			if(cpf) cpf->gid = atoi(it);
			if(lpf) lpf->gid = atoi(it);
	
			/* group user list */
			if((*c !='\r')&&(*c !='\n')){
				/* nothing to do */
			}
		}
		if(p) k++;

		j+=k; line+=k;
	}

	ok=0;
done:
	if(ok ) OK_set_error(ERR_ST_BADFORMAT,ERR_LC_AUTH,ERR_PT_AUTH+8,NULL);
	if(buf) free(buf);
	return ok;
}

/*-----------------------------------------
  write Profile group information
-----------------------------------------*/
int CA_write_groupinfo(CA *ca){
	CertProf *cpf=NULL;
	CRLProf *lpf=NULL;
	FILE *fp=NULL;
	char buf[512],num[32];
	int ok=-1;

	if (set_path(buf, 256,
		     ca->ca_path, "/ca.group", NULL) == -1) {
		goto done;
	}

	if((fp=fopen(buf,"w"))==NULL){
		OK_set_error(ERR_ST_FILEOPEN,ERR_LC_AUTH,ERR_PT_AUTH+9,NULL);
		goto done;
	}
	/* output default values */
	strcpy(buf,"caadmin::0:\n");
	strcat(buf,"guest::1:\n");
	if((lpf=Prof_crl_find(ca,"ARL")) != NULL){
		snprintf(num,32,"ARL::%d:\n",lpf->gid);
		strcat(buf,num);
	}
	if((lpf=Prof_crl_find(ca,"CRL")) != NULL){
		snprintf(num,32,"CRL::%d:\n",lpf->gid);
		strcat(buf,num);
	}
	if((lpf=Prof_crl_find(ca,"CRL-All")) != NULL){
		snprintf(num,32,"CRL-All::%d:\n",lpf->gid);
		strcat(buf,num);
	}
	if((cpf=Prof_find(ca,"Cross Cert")) != NULL){
		snprintf(num,32,"Cross Cert::%d:\n",cpf->gid);
		strcat(buf,num);
	}
	if(fputs(buf,fp)<0){
		OK_set_error(ERR_ST_FILEWRITE,ERR_LC_AUTH,ERR_PT_AUTH+9,NULL);
		goto done;
	}

	/* output Certificate Profiles */
	for(cpf=ca->profList; cpf ; cpf=cpf->next){
		if(!strcmp(cpf->name,"Cross Cert")) continue;

		memset(buf,0,512);
		strncpy(buf,cpf->name,256);

		sprintf(num,"::%d:\n",cpf->gid);
		strcat (buf,num);

		if(fputs(buf,fp)<0){
			OK_set_error(ERR_ST_FILEWRITE,ERR_LC_AUTH,ERR_PT_AUTH+9,NULL);
			goto done;
		}
	}

	ok = 0;
done:
	if(fp) fclose(fp);
	return ok;
}
