/* ca_asn1.c */
/*
 * Copyright (c) 2004-2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_x509.h>

#include "ok_caerr.h"
#include "ok_ca.h"

/*-----------------------------------------
  CA structure to DER
-----------------------------------------*/
unsigned char *CA_toDER(CA *ca,unsigned char *buf,int *ret_len){
	unsigned char *cp,*ret;
	int	i,j;

	if(buf==NULL){
		if((i=CA_estimate_der_size(ca))<=0)
			return NULL;

		if((ret=(unsigned char*)malloc(i))==NULL){
			OK_set_error(ERR_ST_MEMALLOC,ERR_LC_CA,ERR_PT_CAASN1,NULL);
			return NULL;
		}
		memset(ret,0,i);
	}else{
		ret=buf;
	}

	ASN1_set_integer(2,ret,&i);	/* set version */
	ASN1_set_explicit(i,0,ret,&i);
	cp = ret+i;

	if(CA_DER_castat(ca,cp,&j)) goto error;
	i+=j;

	ASN1_set_sequence(i,ret,ret_len);
	return ret;
error:
	if(ret!=buf){free(ret);}
	return NULL;
}

int CA_DER_castat(CA *ca,unsigned char *ret,int *ret_len){
	unsigned char *cp,*t;
	CertProf *ctp;
	CRLProf *clp;
	char *st,*buf;
	int	i,j,k;

	ASN1_set_integer(ca->serialNum,ret,&i);
	cp = ret+i;
	ASN1_set_integer(ca->cert_days,cp,&j);
	cp+=j; i+=j;
	ASN1_set_integer(ca->crl_days,cp,&j);
	cp+=j; i+=j;
	if(ASN1_set_t61(ca->issuer,cp,&j)) goto error;
	cp+=j; i+=j;
	if(ASN1_set_t61(ca->subject,cp,&j)) goto error;
	cp+=j; i+=j;

	/* default tag point */
	st = (ca->m_tag)?(ca->m_tag):("CN");
	if(ASN1_set_ia5(st,cp,&j)) goto error;
	cp+=j; i+=j;

	/* default subject list */
	if((buf=CA_get_defsubstr(ca))==NULL){
		if(ASN1_set_t61("C=JP/$",cp,&j)) goto error;
	}else{
		if(ASN1_set_t61(buf,cp,&j)) goto error;
		free(buf);
	}
	cp+=j; i+=j;

	/* default key type and key long */
	ASN1_set_integer(ca->sigtype,cp,&j);
	cp+=j; i+=j;

	ASN1_set_integer(ca->keylong,cp,&j); /* key long */
	cp+=j; i+=j;

	/* subject filter in the listview... */
	ASN1_set_integer(ca->sbj_ft,cp,&j);
	cp+=j; i+=j;

	/* CA policy (EasyCert v0.84) */
	if(ca->policy){
		ASN1_set_octetstring(16,ca->policy,cp,&j);
		cp+=j; i+=j;
	}

	/* CA crl number (v0.88) */
	ASN1_set_integer(ca->crl_num,cp,&j);
	cp+=j; i+=j;

	/* cert version */
	ASN1_set_integer(ca->cert_ver,cp,&j);
	cp+=j; i+=j;
	/* CRL version */
	ASN1_set_integer(ca->crl_ver,cp,&j);
	cp+=j; i+=j;

	/* CertProfile Name list */
	t = cp;
	for(ctp=ca->profList,j=0; ctp ;ctp=ctp->next){
		if(ASN1_set_t61(ctp->name,t,&k)) goto error;
		t+=k; j+=k;
	}
	ASN1_set_explicit(j,6,cp,&j);
	cp+=j; i+=j;

	/* CRLProfile Name list */
	t = cp;
	for(clp=ca->crlpList,j=0; clp ;clp=clp->next){
		if(ASN1_set_t61(clp->name,t,&k)) goto error;
		t+=k; j+=k;
	}
	ASN1_set_explicit(j,7,cp,&j);
	cp+=j; i+=j;

	/* Set PKCS11 Info (if exist) */
	/*
	 * XXX:the following information should be kept in aica.cnf.
	 */
	if(ca->p11){
		t = cp; j = 0;
		if(ASN1_set_t61(ca->p11->libname,t,&k)) goto error;
		t+=k; j+=k;

		if(ASN1_set_t61(ca->label,t,&k)) goto error;
		t+=k; j+=k;

		/* use m of n (OPTIONAL) */
		ASN1_set_integer(ca->p11_mofn,t,&k);
		t+=k; j+=k;

		ASN1_set_explicit(j,8,cp,&j);
		cp+=j; i+=j;
	}

	ASN1_set_explicit(i,1,ret,ret_len);
	return 0;
error:
	return -1;
}

char *CA_get_defsubstr(CA *ca){
	char *ret=NULL;
	int i,j;

	for(i=j=0; i<ca->sbj_num; i++)
		j += strlen(ca->sbjList[i])+2;

	if((ret=malloc(j+2))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_CA,ERR_PT_CAASN1+3,NULL);
		goto done;
	}
	memset(ret,0,j+2);

	for(i=0; i<ca->sbj_num; i++){
		strcat(ret,ca->sbjList[i]);
		strcat(ret,"/$");
	}
done:
	return ret;
}

/*-----------------------------------------
  Estimate CA DER size
-----------------------------------------*/
int CA_estimate_der_size(CA *ca){
	CertProf *ctp;
	CRLProf *clp;
	int len=120;

	if(ca==NULL){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_CA,ERR_PT_CAASN1+4,NULL);
		return -1;
	}

	/* CA information */
	if(ca->subject) len+=(strlen(ca->subject) << 1);
	if(ca->issuer)  len+= strlen(ca->issuer);

	for(ctp=ca->profList; ctp ;ctp=ctp->next)
		len += strlen(ctp->name) + 4;

	for(clp=ca->crlpList; clp ;clp=clp->next)
		len += strlen(clp->name) + 4;

	if(ca->p11){
		len += strlen(ca->p11->libname);
		len += strlen(ca->label);
		len += 8;
	}

	return len;
}

/*-----------------------------------------
  read DER Info to CA structure
-----------------------------------------*/
CA *ASN1_CA_info(unsigned char *der){
	unsigned char *cp,*t;
	CA *ret=NULL;
	int i;

	cp=ASN1_next(der);
	if((cp[0]!=0xa0)||(cp[1]!=0x03)){
		OK_set_error(ERR_ST_BADFORMAT,ERR_LC_CA,ERR_PT_CAASN1+5,NULL);
		goto error;
	}
	cp=ASN1_next(cp);

	if((ret = CA_new())==NULL) goto error;

	/* version should be 0 or more */
	if((ret->ver = ASN1_integer(cp,&i)) < 0){
		OK_set_error(ERR_ST_BADFORMAT,ERR_LC_CA,ERR_PT_CAASN1+5,NULL);
		goto error;
	}
	cp = t = ASN1_next(cp);

	if(ret->ver < 2){
		if((ret->profList=Prof_cert_new(ret,"cert"))==NULL) goto error;
		ret->cprof = ret->profList;

		/* read ca stat */
		if(ASN1_ctprof_stat(ret,cp)) goto error;
		cp = ASN1_skip(cp);

		/* read cert stat */
		if(ASN1_ctprof_ctstat(ret,cp)) goto error;

		if(*cp==0xa2)
			if((cp = ASN1_skip(cp))==NULL) goto error;

		/* read cert extension info */
		if(ASN1_ctprof_ext(ret,cp)) goto error;

		/*** create cross cert profile ***/
		if((ret->cprof->next=Prof_cert_new(ret,"Cross Cert"))==NULL) goto error;
		ret->cprof = ret->cprof->next;
		if(Prof_cert_settmpl(ret,"Cross Cert Profile template")) goto error;
		ret->cprof->isCross = 1;
		ret->cprof->gid = 20;

		/*** create CRL profile for version compatibility ***/
		/* ARL */
		if((ret->crlpList=Prof_crl_new(ret,"ARL"))==NULL) goto error;
		ret->lprof = ret->crlpList;
		ret->lprof->gid = 10;
		if(Prof_crl_settmpl(ret,"ARL Profile template")) goto error;

		/* CRL */
		if((ret->lprof->next=Prof_crl_new(ret,"CRL"))==NULL) goto error;
		ret->lprof = ret->lprof->next;
		ret->lprof->gid = 11;
		if(Prof_crl_settmpl(ret,"CRL Profile template")) goto error;

		/* CRL-All */
		if((ret->lprof->next=Prof_crl_new(ret,"CRL-All"))==NULL) goto error;
		ret->lprof = ret->lprof->next;
		ret->lprof->gid = 12;

		if(ASN1_clprof_stat(ret,t)) goto error;
		if(ret->ver){ /* version 0 doesn't have CRL extension */
			if((cp = ASN1_skip(cp))==NULL) goto error;

			/* read crl extension info */
			if(ASN1_clprof_ext(ret,cp)) goto error;
		}
	}

	/* version 2 Info file */
	if(ASN1_CA_castat(ret,cp)) goto error;

	return ret;
error:
	if(ret && ret->p11) ret->p11_nofinal=1; /* HSM should not stop with finalize */
	CA_free(ret);
	return NULL;
}

int ASN1_CA_castat(CA *ca,unsigned char *in){
	CertProf *ctp;
	CRLProf *clp;
	unsigned char *cp,*t;
	char *buf,*hp,*mp;
	int i,j,k,len,err=-1;

	if(*in!=0xa1){
		OK_set_error(ERR_ST_BADFORMAT,ERR_LC_CA,ERR_PT_CAASN1+6,NULL);
		goto done;
	}

	cp = ASN1_next(in);
	if((ca->serialNum = ASN1_integer(cp,&i))<0) goto done;
	cp+=i;
	if((ca->cert_days = ASN1_integer(cp,&i))<0) goto done;
	cp+=i;
	if((ca->crl_days  = ASN1_integer(cp,&i))<0) goto done;
	cp+=i;

	if((ca->issuer = asn1_get_str(cp,&i))==NULL) goto done;
	cp+=i;

	if((ca->subject = asn1_get_str(cp,&i))==NULL) goto done;
	cp+=i;

	/* different version */
	if(*cp != ASN1_IA5STRING) return 0; /* if file is old type, return... */

	/* get default tag point */
	if((ca->m_tag = ASN1_ia5((char*)cp,&i))==NULL) goto done;
	cp+=i;

	/* get default directory information (for EasyCert) */
	if((buf = asn1_get_str(cp,&i))==NULL) goto done;

	for(k=0,hp=buf; *hp ;hp=&mp[2],k++){
		if((mp=strstr(hp,"/$"))==NULL) goto done;
		mp[0]=mp[1] = 0;
		if ((ca->sbjList[k] = strdup(hp)) == NULL) goto done;
	}
	if(buf) free(buf);
	cp+=i; ca->sbj_num=k;

	/* get default signature algorithm */
	if(*cp != ASN1_INTEGER) return 0;	/* if file is old type, return... */

	if((ca->sigtype=ASN1_integer(cp,&i))<0) goto done;

	if(ca->sigtype<=1000) ca->sigtype = DEFAULT_SIG_TYPE;
	cp+=i;

	/* get default public key long */
	if((ca->keylong = ASN1_integer(cp,&i))<0) goto done;
	cp+=i;

	/* get subject filter */
	if((ca->sbj_ft = ASN1_integer(cp,&i))<0) goto done;
	cp+=i;

	/* if EasyCert version is 0.84 or later, CA policy is supported. */
	if(*cp == ASN1_OCTETSTRING){
		if(ASN1_octetstring(cp,&i,&ca->policy,&i))
			goto done;
		cp = ASN1_next(cp);
	}else{
		if((ca->policy=(unsigned char*)malloc(16))==NULL){
			OK_set_error(ERR_ST_MEMALLOC,ERR_LC_CA,ERR_PT_CAASN1+6,NULL);
			goto done;
		}
		memset(ca->policy,2,16);
	}

	/* if version is 0.88 or later, CA has crl number */
	if(*cp == ASN1_INTEGER){
		if((ca->crl_num = ASN1_integer(cp,&i))<0) goto done;
		cp = ASN1_next(cp);
	}
	/* if version is 0.88 or later, CA has cert version */
	if(*cp == ASN1_INTEGER){
		if((ca->cert_ver = ASN1_integer(cp,&i))<0) goto done;
		cp = ASN1_next(cp);
	}
	/* if version is 0.88 or later, CA has CRL version */
	if(*cp == ASN1_INTEGER){
		if((ca->crl_ver = ASN1_integer(cp,&i))<0) goto done;
		cp = ASN1_next(cp);
	}

	/* get Cert Profile list Names !! */
	if(*cp == 0xa6){
		len=ASN1_tlen(cp);
		t  =ASN1_next(cp);

		for(i=0,k=100;i<len;){
			if((buf=asn1_get_str(t,&j))==NULL) goto done;
			t+=j; i+=j; 

			if((ctp = Prof_cert_new(ca,buf))==NULL) goto done;
			if(ca->profList==NULL){
				ca->profList = ca->cprof = ctp;
			}else{
				ca->cprof->next = ctp;
				ca->cprof = ctp;
			}
			free(buf); buf=NULL;
			ctp->gid = k++;
		}
		cp = ASN1_skip(cp);
	}
	/* get CRL Profile list Names !! */
	if(*cp == 0xa7){
		len=ASN1_tlen(cp);
		t  =ASN1_next(cp);

		for(i=0;i<len;){
			if((buf=asn1_get_str(t,&j))==NULL) goto done;
			t+=j; i+=j; 

			if((clp = Prof_crl_new(ca,buf))==NULL) goto done;
			if(ca->crlpList==NULL){
				ca->crlpList = ca->lprof = clp;
			}else{
				ca->lprof->next = clp;
				ca->lprof = clp;
			}
			free(buf); buf=NULL;
			clp->gid = k++;
		}
		cp = ASN1_skip(cp);
	}
	/* Get PKCS11 Info (if exist) */
	/*
	 * XXX:the following information should be kept in aica.cnf.
	 */
	if(*cp == 0xa8){
		t  =ASN1_next(cp);
		if((buf=asn1_get_str(t,&j))==NULL) goto done;
		if((ca->p11=P11_init(buf))==NULL) goto done;
		free(buf); buf=NULL;

		t  =ASN1_next(t);
		if((ca->label=asn1_get_str(t,&j))==NULL) goto done;
		ca->use_p11 = 1;

		/* mofn INTEGER OPTIONAL */
		t  =ASN1_next(t);
		if(*t==ASN1_INTEGER){
			ca->p11_mofn = ASN1_integer(t,&j);
		}
		cp = ASN1_skip(cp);
	}

	err=0;
done:
	return err;
}

int ASN1_CA_certstat(CA *ca,unsigned char *in){
	unsigned char *cp,*sq;
	CertStat *st;
	int len,i,j,k;

	if(*in!=0xa2){
		OK_set_error(ERR_ST_BADFORMAT,ERR_LC_CA,ERR_PT_CAASN1+7,NULL);
		goto error;
	}

	len=ASN1_length(in+1,&i);
	sq=in+i+1;

	for(i=0;i<len;i+=j,sq+=j){
		j =ASN1_length(sq+1,&k);
		j+=k+1;

		if(i){
			if((st->next=CertStat_new())==NULL) goto error;
			st->next->prev=st;
			st=st->next;
		}else{
			if((ca->stat=CertStat_new())==NULL) goto error;
			st=ca->stat;
		}

		cp = ASN1_next(sq);
		if((st->state = ASN1_integer(cp,&k))<0) goto error;
		cp+=k;
		if((st->serialNum = ASN1_integer(cp,&k))<0) goto error;
		cp+=k;
		if((st->subject = asn1_get_str(cp,&k))==NULL) goto error;

		cp=ASN1_next(cp);
		if(UTC2stm(cp,&st->notBefore)) goto error;

		cp=ASN1_next(cp);
		if(UTC2stm(cp,&st->notAfter)) goto error;

		cp=ASN1_next(cp);
		if((*cp==ASN1_UTCTIME)||(*cp==ASN1_GENERALIZEDTIME)){
			UTC2stm(cp,&st->revokedDate);
		}
	}
	return 0;
error:
	CertStat_free_all(ca->stat);
	ca->stat=NULL;
	return -1;
}


int ASN1_CA_extns(CA *ca,unsigned char *in){
	unsigned char *cp,*ecp;
	int i,j,id,cr,len;
	CertExt *et,*now;

	if(*in!=0xa3) return -1;

	len = ASN1_tlen(in);
	in  = ASN1_next(in);

	CertExt_free_all(ca->ext);
	ca->ext = NULL;

	for(et=now=NULL,i=0;i<len;){
		cp = ASN1_next(in);
		cr = 0;
	    
		if(*cp==ASN1_INTEGER){
		  if((id = ASN1_integer(cp,&j))<0) goto error;
		  
		}else if(*cp==ASN1_OBJECT_IDENTIFIER){
		  if((id = ASN1_object_2int(cp))<0) goto error;
		}
		cp = ASN1_next(cp);

		/* get critical */
		if(*cp==ASN1_BOOLEAN){
		  cr = cp[2];
		  cp = ASN1_next(cp);
		}
		
		/* get octetstring */
		if(ASN1_octetstring(cp,&j,&ecp,&j)) goto error;
	  
		if((now=ASN1_get_ext(id,ecp))==NULL) goto error;
		now->critical = cr;
		free(ecp);

		if(ca->ext){
		  et->next = now;
		  et = now;
		}else{
		  ca->ext=et=now;
		}

		if((in = ASN1_skip_(in,&j))==NULL) goto error;
		i+= j;
	}

	return 0;
error:
	CertExt_free_all(ca->ext);
	ca->ext = NULL;
	return -1;
}

