/* ca.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_x509.h>
#include <aicrypto/ok_x509ext.h>

#include "ok_caerr.h"
#include "ok_ca.h"

/*-----------------------------------------
  struct CA alloc & free
-----------------------------------------*/
CA *CA_new(void){
	CA *ret;

	if((ret=(CA*)malloc(sizeof(CA)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_CA,ERR_PT_CA,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(CA));
	return ret;
}

void CA_free(CA *ca){
	int i;
	if(ca==NULL) return;

	if(ca->p11){
		P7_free((PKCS7*)ca->p12);
		if(ca->p11_nofinal) ca->p11->pFunc=NULL;
		P11_free(ca->p11);
	}else{
		P12_free(ca->p12);
	}

	if(ca->subject) free(ca->subject);
	if(ca->issuer)  free(ca->issuer);
	if(ca->der)		free(ca->der);
	if(ca->policy)	free(ca->policy);
	if(ca->ca_name)	free(ca->ca_name);
	if(ca->ca_path) free(ca->ca_path);
	if(ca->lib_path)free(ca->lib_path);
	if(ca->m_tag)	free(ca->m_tag);
	if(ca->label)	free(ca->label);

	for(i=0; i<ca->sbj_num; i++)
		if(ca->sbjList[i]) free(ca->sbjList[i]);

	/* ca->cert & ca->prvkey is just pointer
	 * to PKCS#12 bags. Don't need to free them.
	 */

	CRL_free(ca->crl);

	CertExt_free_all(ca->ext);
	CertStat_free_all(ca->stat);

	Prof_cert_free_all(ca->profList);
	Prof_crl_free_all(ca->crlpList);
	Prof_csr_free_all(ca->csrProf);

	free(ca);
}

/*-----------------------------------------
  struct CertStat alloc & free
-----------------------------------------*/
CertStat *CertStat_new(void){
	CertStat *ret;

	if((ret=(CertStat*)malloc(sizeof(CertStat)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_CA,ERR_PT_CA+2,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(CertStat));
	return ret;
}

void CertStat_free(CertStat *cs){
	if(cs==NULL) return;
	if(cs->subject) free(cs->subject);
	if(cs->keyID) free(cs->keyID);
	free(cs);
}

void CertStat_free_all(CertStat *cs){
	CertStat *next;
	while(cs){
		next=cs->next;
		CertStat_free(cs);
		cs=next;
	}
}

/*-----------------------------------------
  struct CertStat dup
-----------------------------------------*/
CertStat *CertStat_dup(CertStat *org){
	CertStat *ret=NULL;

	if(org==NULL) goto error;
	
	if((ret=CertStat_new())==NULL) goto error;
	
	memcpy(ret,org,sizeof(CertStat));
	if ((ret->subject = strdup(org->subject)) == NULL){
		OK_set_error(ERR_ST_STRDUP,ERR_LC_CA,ERR_PT_CA+3,NULL);
		goto error;
	}
	if(org->keyID){
		if((ret->keyID=(unsigned char*)malloc(22))==NULL){
			OK_set_error(ERR_ST_MEMALLOC,ERR_LC_CA,ERR_PT_CA+3,NULL);
			goto error;
		}
		memcpy(ret->keyID,org->keyID,20);
	}
	ret->next=ret->prev=NULL;
	ret->fp  =NULL;
	memset(&ret->pos,0,sizeof(fpos_t));

	return ret;
error:
	CertStat_free(ret);
	return NULL;
}

CertStat *CertStat_dup_all(CertStat *top){
	CertStat *ret=NULL,*hd,*cs;
	while(top){
		if((cs=CertStat_dup(top))==NULL) goto error;
		if(ret){
		  hd->next=cs; hd=cs;
		}else{
		  ret=hd=cs;
		}
		top=top->next;
	}
	return ret;
error:
	CertStat_free_all(ret);
	return NULL;
}

/*-----------------------------------------
    set CA path
-----------------------------------------*/
int CA_set_ca_path(CA *ca,char *path){

    if ((ca->ca_path = strdup(path)) == NULL){
    OK_set_error(ERR_ST_STRDUP,ERR_LC_CA,ERR_PT_CA+4,NULL);
    return -1;
  }
  return 0;
}

