/* req.c */
/*
 * Copyright (c) 2004-2015 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_tool.h>

#include "ok_conf.h"

static char *dirtag[10]={
	"C","ST","L","O","OU","CN","UID","DC","EMAIL",""};

char defdir[10][64]={
	"","","","","","","","","",""};

char title[10][32]={
	"Country","STate or province","Locality",
	"Organization","Organization Unit","Common Name",
	"UserID","Domain Component","Email Address"};

/*-----------------------------------------
  read CSR config
-----------------------------------------*/
void config_set_defdir(char *str){
	int i,rlen;
	char *cp,*d,*buf;
  
	if((cp=strstr(str,"\r\n")) != NULL) cp+=2;
	else if((cp=strstr(str,"\n")) != NULL) cp++;

	for(;;cp=d+rlen){
		if((buf=strchr(cp,'='))==NULL) break;

		*buf=0; buf++;
		if((d=strstr(buf,"\r\n")) != NULL) rlen=2;
		else if((d=strstr(buf,"\n")) != NULL) rlen=1;
		else break;

		*d=0; i=8;
		if(*cp=='#') continue;
    
		switch(*cp){
		case 'C': i=(cp[1]=='N')?(5):(0); break;
		case 'S': i=1; break;
		case 'L': i=2; break;
		case 'O': i=(cp[1]=='U')?(4):(3); break;
		case 'U': i=6; break;
		case 'D': i=7; break;
		case 'E': i=8; break;
		}
		strcpy(defdir[i],buf);
	}
}

int req_config(char *fname, int num){
	char begin[64],end[64];
	char *buf;

	sprintf(begin,"[subject DN %d]",num);
	sprintf(end  ,"[subject DN %d end]",num);

	/* need compatibility with older version */
	if((buf=conf_get_section(fname,
				(num)?(begin):("[subject DN]"),
				(num)?(end):("[subject DN end]")))==NULL)
		return -1;

	config_set_defdir(buf);

	free(buf);
	return 0;
}


  
/*-----------------------------------------
  get both Private & Public Keys
-----------------------------------------*/
int get_key_pairs(Key **pub,Key **prv,int size,int type){
	/* use only KEY_DSA_XXX */
	DSAParam *pm=NULL;
	int hash_algo;

	/* use only KEY_ECDSA_XXX */
	ECParam *E=NULL;
	int curve_type;

	printf("generate private key (size %d bits)\n",size);

	*prv=*pub=NULL;

	switch(type){
	case KEY_DSA_PUB:
	case KEY_DSA_PRV:
		if((size!=1024)&&(size!=2048)&&(size!=3072)){
			printf("bad key size ...\n");
			goto error;
		}
		if((*prv=(Key*)DSAprvkey_new())==NULL) goto error;
		if((*pub=(Key*)DSApubkey_new())==NULL) goto error;

		hash_algo = OK_get_sign_digest_algo();
		if((pm = DSAPm_gen_parameter(size,hash_algo))==NULL) goto error;
		if(DSAprv_generate(pm,(Prvkey_DSA*)*prv)) goto error;
		DSAPm_free(pm); pm=NULL;

		DSAprv_2pub((Prvkey_DSA*)*prv,(Pubkey_DSA*)*pub);
		break;
	case KEY_ECDSA_PUB:
	case KEY_ECDSA_PRV:
		if(size<160){
			printf("bad key size ...\n");
			goto error;
		}
		if((*prv=(Key*)ECDSAprvkey_new())==NULL) goto error;
		if((*pub=(Key*)ECDSApubkey_new())==NULL) goto error;

		curve_type=ECPm_get_recommended_elliptic_curve(size);
		if((E = ECPm_get_std_parameter(curve_type))==NULL) goto error;
		if(ECDSAprv_generate(E,(Prvkey_ECDSA*)*prv)) goto error;
		ECPm_free(E); E=NULL;

		ECDSAprv_2pub((Prvkey_ECDSA*)*prv,(Pubkey_ECDSA*)*pub);
		break;
	default:
		if((size<RSA_KEY_BITLENGTH_MIN)||(size>RSA_KEY_BITLENGTH_MAX)){
			printf("bad key size ...\n");
			goto error;
		}

		if((*prv=(Key*)RSAprvkey_new())==NULL) goto error;
		if((*pub=(Key*)RSApubkey_new())==NULL) goto error;

		if(RSAprv_generate((Prvkey_RSA*)*prv,size>>4)) goto error;

		RSAprv_2pub((Prvkey_RSA*)*prv,(Pubkey_RSA*)*pub);
		break;
	}
	return 0;

error:
	if(*prv){Key_free(*prv);*prv=NULL;}
	if(*pub){Key_free(*pub);*pub=NULL;}
	DSAPm_free(pm);
	ECPm_free(E);
	return -1;
}

/*-----------------------------------------
  input interfaces (subject & altname)
-----------------------------------------*/
char *input_subject(CertDN *dn,char *prompt){
	char buf[128],str[512]="",*ret=NULL;
	int i,j,k,kind;

	if(prompt==NULL) printf("input Distinguished Name (DN).\n");
	else printf("%s", prompt);

	for(*str=i=j=0;j<RDN_MAX;i++){
		/* get current default rdn */
		for(;i<9;i++)
			if(strcmp(defdir[i],"")) break;

		printf("select directory tag (input number)\n");
		printf("(1.C, 2.ST, 3.L, 4.O, 5.OU, 6.CN, 7.UID, 8.DC, 9.Email, 10.Quit)[%d]:",i+1);
		fgets(buf,64,stdin);
		k=atoi(buf);
	
		if((0<k)&&(k<11)) i=k-1;
		if(i>=9) break;
		switch(i){
		case 8:  kind = OBJ_DIR_EMAIL; break;
		case 7:  kind = OBJ_DIR_DC; break;
		case 6:  kind = OBJ_DIR_UID; break;
		default: kind = OBJ_DIR_C + i; break;
		}

		printf("%s [%s]: ",title[i],defdir[i]);
		fgets(buf,64,stdin);
		k=strlen(buf);
		buf[k-1]=0;		/* clean return code */

		if(k==1){
			if(strcmp(defdir[i],"")){
				if ((dn->rdn[j].tag = strdup(defdir[i])) == NULL) goto done;
			}else{
				continue;
			}
		}else{
			if ((dn->rdn[j].tag = strdup(buf)) == NULL) goto done;
		}

		dn->rdn[j].tagoid=kind;
		switch(kind){
		case OBJ_DIR_C:     k = ASN1_PRINTABLE_STRING; break;
		case OBJ_DIR_EMAIL: k = ASN1_IA5STRING; break;
		default: k = asn1_str_type(buf); break;
		}
		dn->rdn[j].derform = k;

		sprintf(buf,"%s=%s, ",dirtag[i],dn->rdn[j].tag);
		strcat(str,buf);
		j++;
	}
  
	dn->num=j;
	ret = strdup(str);
done:
	return ret;
}

/*
 * input_ipaddress() doesn't yet support the Name Constraints
 * [RFC5280 4.2.1.10]. The Name Constraints MUST hold both IP address
 * and prefix length in the iPAddress field.
 */
ExtGenNames *input_ipaddress(){
	char buf[128];
	unsigned char *bin;
	struct addrinfo hints, *res = NULL;
	int len, error;
	ExtGenNames *ret = NULL;

	printf("IP Address : ");
	fgets(buf, sizeof(buf), stdin);

	if (strchr(buf, '\n') != NULL) {
		buf[strlen(buf) - 1] = '\0';
	} else {
		do {
			fgets(buf, sizeof(buf), stdin);			
		} while (strchr(buf, '\n') == NULL);
		printf("INPUT ERROR: Too long\n");
		goto error;
	}

	/* get IP address */
	memset(&hints, 0, sizeof(hints));
	hints.ai_family = AF_UNSPEC;
	hints.ai_socktype = SOCK_RAW;
	hints.ai_flags = AI_NUMERICHOST;
	error = getaddrinfo(buf, NULL, &hints, &res);
	if (error != 0) {
		printf("INPUT ERROR: Invalid IP Address(%s)\n",
		       gai_strerror(error));
		goto error;
	}
	
	switch (res->ai_family) {
	case AF_INET:
		bin = (unsigned char *)&((struct sockaddr_in *)res->ai_addr)->sin_addr.s_addr;
		len = sizeof(struct in_addr);
		break;
	case AF_INET6:
		bin = (unsigned char *)&((struct sockaddr_in6 *)res->ai_addr)->sin6_addr.s6_addr;
		len = sizeof(struct in6_addr);
		break;
	default:
		printf("INPUT ERROR: Don't know how to encode Address Family(%d)\n",
		       res->ai_family);
		goto error;
	}

	ret = ExtGN_set_ip(bin,len);

error:
	if (res != NULL) {
		freeaddrinfo(res);
	}
	return ret;
}

OtherName *input_othername(){
	OtherName *ret;
	char buf[128];

	if((ret=ExtGN_on_new())==NULL) goto error;

	/* now only support UPN */
	switch(0){
	case 0:
		printf("input UPN : ");
		fgets(buf,126,stdin);
		buf[strlen(buf)-1]=0; /* clean return code */

		ret->oid = OBJ_MS_GN_UPN;
		if ((ret->oidc = strdup("1.3.6.1.4.1.311.20.2.3")) == NULL) goto error;
		if((ret->name=(unsigned char*)malloc(strlen(buf)+4))==NULL) goto error;

		/* should be English ?? */
		if(ASN1_set_utf8(buf,ret->name,&ret->nlen)) goto error;
		break;
	}
	return ret;
error:
	ExtGN_on_free(ret);
	return NULL;
}

ExtGenNames *input_gennames(char *prompt){
	char *title[]={"Email Address","DNS Name","URL","","IP Address"};
	char buf[128],*tmp;
	ExtGenNames *ret=NULL,*hd,*now;
	OtherName *nm;
	CertDN dn;
	int k;

	if(prompt==NULL) printf("input GeneralNames.\n");
	else printf("%s", prompt);
	cert_dn_init(&dn);

	for(;;){
		printf("select a GeneralName (input number)\n");
		printf("(1.EMail, 2.DNS Name, 3.URL, 4.DN, 5.IP Address, 6.Other, 7.Quit)[%d]:",7);
		fgets(buf,64,stdin);
		k=atoi(buf);
		if((k==7)||(k==0)) break;

		if(k<4){
			printf("%s : ",title[k-1]);
			fgets(buf,64,stdin);
			buf[strlen(buf)-1]=0; /* clean return code */
		}
		switch(k){
		case 1:
			if((now=ExtGN_set_email(buf))==NULL) continue;
			break;
		case 2:
			if((now=ExtGN_set_dns(buf))==NULL) continue;
			break;
		case 3:
			if((now=ExtGN_set_url(buf))==NULL) continue;
			break;
		case 4:
			if((tmp=input_subject(&dn,NULL))==NULL) continue;
			free(tmp);
			if((now=ExtGN_set_dn(&dn))==NULL) continue;
			cert_dn_free(&dn);
			break;
		case 5:
			if((now=input_ipaddress())==NULL) continue;
			break;
		case 6:
			if((nm=input_othername())==NULL) continue;
			if((now=ExtGN_set_oth(nm,nm->nlen))==NULL) continue;
			break;
		default:
			continue;
		}

		if(ret==NULL){
			ret=hd=now;
		}else{
			hd->next=now;
			hd=now;
		}
	}
	return ret;
}

/*-------------------------------------------------
  set default rdn information for subject input
-------------------------------------------------*/
int ca_set_defdir(CertDN *dn){
	int i,j;

	for(i=0; i<dn->num; i++){
		j = dn->rdn[i].tagoid;
		if(j < OBJ_DIR_CN)
			strncpy(defdir[j-OBJ_DIR_C],dn->rdn[i].tag,62);
	}
	return 0;
}

