/* mdgst.c */
/*
 * Copyright (c) 2014-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <limits.h>

#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_tool.h>

#define DEFAULT_HASH	"sha512"
#define BUFSIZE		(8 * 1024 * 1024)

struct hash {
	struct hash		*next;
	HASHAlgorithm		*ha;
	union {
		MD2_CTX		md2;
		MD4_CTX		md4;
		MD5_CTX		md5;
		SHA1_CTX	sha1;
		SHA256_CTX	sha256;
		SHA512_CTX	sha512;
		SHA3_CTX	sha3;
	} hash_ctx;
	int			digest_size;
	unsigned char		digest[];
};

enum output_format {
	NORMAL = 0,
	COREUTILS = 1
} format = NORMAL;

char *opt_hash = NULL;
int variablesize = DEFAULT_SHAKE_DIGESTSIZE;

struct hash *get_hashlist(char *str_hashs, enum output_format fmt);
void output_digest(const struct hash *hash, const char *filename,
		   const unsigned char *digest, enum output_format format);
int to_hexstr(unsigned char *dst, size_t dst_sz,
	      const unsigned char *src, size_t src_sz);

void options(int argc, char *argv[]);
void usage(void);

/* common/version.c */
void print_version(char *argv[]);

#ifndef HAVE_GETPROGNAME
const char *getprogname(void);
void setprogname(const char *progname);
#endif

/**
 * compute message digets
 *
 */
int main(int argc, char *argv[])
{
	struct hash *hash_list;
	unsigned char *hexdigest;
	size_t hexdigestsize;

	FILE *fp;
	char *filename;
	unsigned char *buf;

	int exitcode = EXIT_SUCCESS;

	setprogname(argv[0]);

	/* use DEFAULT_HASH or progname as the default hash algorithm */
	if (gethashalgobyname(getprogname()) != NULL) {
		opt_hash = strdup(getprogname());
	} else {
		opt_hash = strdup(DEFAULT_HASH);
	}
	if (opt_hash == NULL) {
		perror("opt_hash");
		exit(EXIT_FAILURE);
	}

	/* parse options */
	options(argc, argv);
	argc -= optind;
	argv += optind;

	/* get the list of hash algorithms */
	if ((hash_list = get_hashlist(opt_hash, format)) == NULL) {
		exit(EXIT_FAILURE);
	}

	/* allocate buf */
	if ((buf = malloc(BUFSIZE)) == NULL) {
		perror("buf");
		exit(EXIT_FAILURE);
	}

	/* allocate hexdigest */
	if (variablesize > AC_MAX_DIGESTSIZE) {
		hexdigestsize = variablesize * 2 + 1;
	} else {
		hexdigestsize = AC_MAX_DIGESTSIZE * 2 + 1;
	}
	if ((hexdigest = malloc(hexdigestsize)) == NULL) {
		perror("hexdigest");
		exit(EXIT_FAILURE);
	}

	/* use stdin as default input */
	fp = stdin;
	filename = "stdin";

	/* compute message digest */
	do {
		size_t read_sz;
		struct hash *h;

		/* use argv as inpput message if argv exists */
		if (*argv != NULL) {
			if ((fp = fopen(*argv, "r")) == NULL) {
				perror(*argv++);
				exitcode = EXIT_FAILURE;
				continue;
			}
			filename = *argv++;
		}

		/* initialize each hash algorithm */
		for (h = hash_list; h != NULL; h = h->next) {
			(*h->ha->hash_init)(&h->hash_ctx);
		}

		/* update each hash algorithm */
		while ((read_sz = fread(buf, sizeof(unsigned char),
					BUFSIZE, fp)) != 0) {
			for (h = hash_list; h != NULL; h = h->next) {
				(*h->ha->hash_update)(&h->hash_ctx,
						      buf, read_sz);
			}
		}

		/* finalize each hash algorithm */
		for (h = hash_list; h != NULL; h = h->next) {
			switch (h->ha->aioid) {
			case OBJ_HASH_SHAKE128:
			case OBJ_HASH_SHAKE256:
				(*h->ha->hash_final2)(h->digest,
						      h->digest_size,
						      &h->hash_ctx);
				break;
			default:
				(*h->ha->hash_final)(h->digest, &h->hash_ctx);
				break;
			}
		}

		/* output each message digest */
		for (h = hash_list; h != NULL; h = h->next) {
			memset(hexdigest, 0, hexdigestsize);
			to_hexstr(hexdigest, hexdigestsize,
				  h->digest, (size_t)h->digest_size);
			output_digest(h, filename, hexdigest, format);
		}

		if (fp != stdin) {
			fclose(fp);
		}
	} while (*argv != NULL);

	exit(exitcode);
}

/**
 * Get the list of hash algorithms
 *
 * This function return the list of hash algorithms
 *
 * @param[in] str_hash the comma-separetaed hash string.
 * @param[in] fmt output format.
 * @returns pointer to the list of hash algorithms.
 *
 */
struct hash *get_hashlist(char *str_hashs, enum output_format fmt)
{
	struct hash *hash_list = NULL;
	struct hash *hash_list_tail = NULL;

	char *hash_name;
	char *last = NULL;

	/* use only first hash algotirhm when coreutils option is enbaled */
	if (fmt == COREUTILS) {
		char *commaptr;

		if ((commaptr = strchr(str_hashs, ',')) != NULL) {
			fprintf(stderr,
				"THE coreutils OPTION SUPPORTS ONLY"
				" ONE HASH ALGORITHM\n");
			*commaptr = '\0';
		}
	}

	/* create the list of hash algorithms */
	for (hash_name = strtok_r(str_hashs, ",", &last); hash_name != NULL;
	     hash_name = strtok_r(NULL, ",", &last)) {
		HASHAlgorithm *ha;
		struct hash *h;
		int digestsize;
		int found = 0;

		if ((ha = gethashalgobyname(hash_name)) == NULL) {
			fprintf(stderr,
				"UNKNOWN HASH ALGORITHM: %s\n", hash_name);
			continue;
		}

		for (h = hash_list; h != NULL; h = h->next) {
			if (h->ha->aioid == ha->aioid) {
				found = 1;
			}
		}
		if (found == 1) continue;

		if ((h = malloc((sizeof(struct hash) + ha->hash_size)))
		    == NULL) {
		    perror("malloc");
		    exit(EXIT_FAILURE);
		}

		switch (ha->aioid) {
		case OBJ_HASH_SHAKE128:
		case OBJ_HASH_SHAKE256:
			digestsize = variablesize;
			break;
		default:
			digestsize = ha->hash_size;
			break;
		}

		h = malloc((sizeof(struct hash) + digestsize));
		if (h == NULL) {
		    perror(ha->name);
		    exit(EXIT_FAILURE);
		}
		memset(h, 0, sizeof(struct hash) + digestsize);

		h->ha = ha;
		h->digest_size = digestsize;
		h->next = NULL;
		if (hash_list == NULL) {
			hash_list = hash_list_tail = h;
		} else {
			hash_list_tail->next = h;
			hash_list_tail = h;
		}
	}

	return hash_list;
}

/**
 * Output a message digest.
 *
 * This function outputs a message digest.
 *
 * @param[in] hash hash object that computed digest.
 * @param[in] filename filename of message.
 * @param[in] digest message digest.
 * @param[in] fmt output format.
 *
 */
void output_digest(const struct hash *hash, const char *filename,
		   const unsigned char *digest, enum output_format fmt)
{
	if (fmt == COREUTILS) {
		printf("%s  %s\n", digest, filename);
	} else {
		const char *hash_name = hash->ha->name;

		switch (hash->ha->aioid) {
		case OBJ_HASH_SHAKE128:
		case OBJ_HASH_SHAKE256:
			printf("%s[%d](%s)= %s\n", hash_name,
			       hash->digest_size * 8, filename, digest);
			break;
		default:
			printf("%s(%s)= %s\n", hash_name, filename, digest);
			break;
		}
	}
}

/**
 * Convert from an array of bytes to a hexadecimal string.
 *
 * This function converts from an array of bytes to a hexadecimal string.
 *
 * @param[in, out] dst array for hexadecimal string.
 * @param[in] dst_sz the size of dst in bytes.
 * @param[in] src array of bytes.
 * @param[in] src_sz the size of src in bytes.
 * @returns the size of the array of bytes can be converted, -1 if error.
 *
 */
int to_hexstr(unsigned char *dst, size_t dst_sz,
	      const unsigned char *src, size_t src_sz)
{
	unsigned char *_dst = dst;
	const unsigned char *_src = src;

	static const unsigned char hex[] = "0123456789abcdef";

	if (dst == NULL || src == NULL || dst_sz < 3) {
		return -1;
	}

	while (src_sz > 0 && dst_sz > 2) {
		*(_dst++) = hex[(*_src & 0xf0) >> 4];
		*(_dst++) = hex[*_src & 0x0f];
		_src++;
		src_sz--;
		dst_sz -= 2;
	};
	*_dst = '\0';

	return _src - src;
}

/**
 * Check options
 *
 * This function check options
 *
 * @param[in] argc the size of argv.
 * @param[in] argv the array of arguments.
 *
 */
void options(int argc, char *argv[])
{
	int i;

	for (i = 1, optind = 1; i < argc; i++) {
		if (strcmp("-hash", argv[i]) == 0) {
			if (i + 1 >= argc) {
				printf("requries an argument: %s\n", argv[i]);
				printf("Try `%s -help' for more information.\n",
				       getprogname());
				exit(EXIT_FAILURE);
			}
			free(opt_hash);
			opt_hash = argv[i + 1];
			i++;
			optind += 2;
		} else if (strcmp("-variablesize", argv[i]) == 0) {
			char *ep;
			char *strvsize;
			long vsize;

			if (i + 1 >= argc) {
				printf("requries an argument: %s\n", argv[i]);
				printf("Try `%s -help' for more information.\n",
				       getprogname());
				exit(EXIT_FAILURE);
			}

			strvsize = argv[i + 1];
			vsize = strtol(strvsize, &ep, 10);
			if (strvsize[0] == '\0' || *ep != '\0') {
				printf("not decimal number: %s\n", strvsize);
				printf("Try `%s -help' for more information.\n",
				       getprogname());
				exit(EXIT_FAILURE);

			}
			if (vsize <= 0 || vsize > INT_MAX) {
				printf("out of range: %s\n", strvsize);
				printf("Try `%s -help' for more information.\n",
				       getprogname());
				exit(EXIT_FAILURE);
			}
			if ((vsize % 8) != 0) {
				printf("not multiples of 8: %s\n", strvsize);
				printf("Try `%s -help' for more information.\n",
				       getprogname());
				exit(EXIT_FAILURE);
			}
			variablesize = vsize / 8;
			i++;
			optind += 2;
		} else if (strcmp("-coreutils", argv[i]) == 0) {
			format = COREUTILS;
			optind += 1;
		} else if ((strcmp("-h", argv[i]) == 0) ||
		    (strcmp("-help", argv[i]) == 0)) {
			usage();
			exit(EXIT_SUCCESS);
		} else if (strcmp("-version", argv[i]) == 0) {
			print_version(argv);
			exit(EXIT_SUCCESS);
		} else if (argv[i][0] == '-') {
			printf("unknown option: %s\n", argv[i]);
			printf("Try `%s -help' for more information.\n",
			       getprogname());
			exit(EXIT_FAILURE);
		}
	}
}

/**
 * Display usage
 *
 * This function displays the usage of mdgst.
 *
 */
void usage(void)
{
	const char *default_hash;

	if (gethashalgobyname(getprogname()) != NULL) {
		default_hash = getprogname();
	} else {
		default_hash = DEFAULT_HASH;
	}

	printf(
"Usage: %s [OPTION...] [FILE...]\n"
"\n"
"Opionts:\n"
"  -hash ALG[,ALG]     set a hash algorithm list used to compute the digest\n"
"                      [default:%s]\n"
"  -variablesize SIZE  set the digest size, in bits, of variable hash\n"
"                      algorithm(i.e., SHAKE128, SHAKE256) [default:%d]\n"
"  -coreutils          print the digest in coreutils format\n"
"  -version            print version information and exit\n"
"  -h                  display this help and exit\n"
"  -help               display this help and exit\n"
"\n"
"This supported hash algorithm is as follows:\n"
"   md2                the MD2 algorithm will be used\n"
"   md4                the MD4 algorithm will be used\n"
"   md5                the MD5 algorithm will be used\n"
"   sha1               the SHA-1 algorithm will be used\n"
"   sha224             the SHA-224 algorithm will be used\n"
"   sha256             the SHA-256 algorithm will be used\n"
"   sha384             the SHA-384 algorithm will be used\n"
"   sha512             the SHA-512 algorithm will be used\n"
"   sha512-224         the SHA-512/224 algorithm will be used\n"
"   sha512-256         the SHA-512/256 algorithm will be used\n"
"   sha3-224           the SHA3-224 algorithm will be used\n"
"   sha3-256           the SHA3-256 algorithm will be used\n"
"   sha3-384           the SHA3-384 algorithm will be used\n"
"   sha3-512           the SHA3-512 algorithm will be used\n"
"   shake128           the SHAKE128 algorithm will be used\n"
"   shake256           the SHAKE256 algorithm will be used\n"
		, getprogname(), default_hash, DEFAULT_SHAKE_DIGESTSIZE * 8);
}
