/* certview.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_x509.h>
#include <aicrypto/ok_pem.h>
#include <aicrypto/ok_pkcs.h>
#include <aicrypto/ok_spkac.h>
#include <aicrypto/ok_tool.h>

char *pass = NULL;
char *in[8];
int fn;

/* functions */
void usage(void);
void options(int argc, char **argv);

/* key.c */
void key_print_dsaparam(DSAParam *pm);
void key_print_ecparam(ECParam *pm);

/* common/version.c */
void print_version(char *argv[]);

/* main */
int main(int argc, char **argv)
{
	unsigned char *der, i;
	CertPair *cp;
	Cert *ct;
	CRL *crl;
	Key *key;
	SPKAC *spk;
	PKCS7 *p7;
	PKCS12 *p12;
	DSAParam *pm;
	ECParam *E;

	if (argc == 1) {
		usage();
		exit(1);
	}

	options(argc, argv);

	OK_set_prompt("Input Password: ");
	if (pass)
		OK_set_passwd(pass);

	for (i = 0; i < fn; i++) {
		if (!fopen(in[i], "r"))
			continue;

		if ((ct = Cert_read_file(in[i])) != NULL) {
			Cert_print(ct);
			Cert_free(ct);
			continue;
		}
		if ((ct = Req_read_file(in[i])) != NULL) {
			Cert_print(ct);
			Cert_free(ct);
			continue;
		}
		if ((crl = CRL_read_file(in[i])) != NULL) {
			CRL_print(crl);
			CRL_free(crl);
			continue;
		}
		if ((cp = CertPair_read_file(in[i])) != NULL) {
			CertPair_print(cp);
			CertPair_free(cp);
			continue;
		}
		if ((spk = SPKAC_read_file(in[i])) != NULL) {
			SPKAC_print(spk);
			SPKAC_free(spk);
			continue;
		}
		if ((key = (Key *) PEM_read_rsaprv(in[i])) != NULL) {
			Key_print(key);
			Key_free(key);
			continue;
		}
		if ((key = (Key *) PEM_read_dsaprv(in[i])) != NULL) {
			Key_print(key);
			Key_free(key);
			continue;
		}
		if ((key = (Key *) PEM_read_ecdsaprv(in[i])) != NULL) {
			Key_print(key);
			Key_free(key);
			continue;
		}
		if ((pm = PEM_read_dsaparam(in[i])) != NULL) {
			key_print_dsaparam(pm);
			DSAPm_free(pm);
			continue;
		}
		if ((E = PEM_read_ecparam(in[i])) != NULL) {
			key_print_ecparam(E);
			ECPm_free(E);
			continue;
		}
		if ((p7 = P7b_read_file(in[i])) != NULL) {
			P7_print(p7);
			P7_free(p7);
			continue;
		}
		if ((p12 = P12_read_file(in[i])) != NULL) {
			P12_print(p12);
			P12_free(p12);
			continue;
		}
		if ((key = (Key *) P8_read_file(in[i])) != NULL) {
			Key_print(key);
			Key_free(key);
			continue;
		}
		if ((key = (Key *) P8enc_read_file(in[i])) != NULL) {
			Key_print(key);
			Key_free(key);
			continue;
		}
		if ((der = ASN1_read_der(in[i])) != NULL) {
			if ((key = (Key *) ASN1_read_rsaprv(der)) != NULL) {
				Key_print(key);
				Key_free(key);
				continue;
			}
			if ((key = (Key *) ASN1_read_dsaprv(der)) != NULL) {
				Key_print(key);
				Key_free(key);
				continue;
			}
			if ((key = (Key *) ASN1_read_ecdsaprv(der)) != NULL) {
				Key_print(key);
				Key_free(key);
				continue;
			}
			if ((pm = ASN1_read_dsaparam(der, 0)) != NULL) {
				key_print_dsaparam(pm);
				DSAPm_free(pm);
				continue;
			}
			if ((E = ASN1_read_ecparam(der)) != NULL) {
				key_print_ecparam(E);
				ECPm_free(E);
				continue;
			}
		}
	}
	return 0;
}

void usage(void)
{
	printf("\
Usage: certview [OPTION...] FILE...\n\
\n\
Options:\n\
  -p PASSWD	set password for key or PKCS#12 file\n\
  -help		print this message\n\
  -version	print version information and exit\n\
\n\
The available format of FILE is as follows:\n\
  Certificate	*.cer *.p7b *.p12 *.pem [x509(DER),PKCS#7,PKCS#12,PEM]\n\
  CRL		*.crl *.p7b *.p12 *.pem [x509(DER),PKCS#7,PKCS#12,PEM]\n\
  Private Key	*.p12 *.crtx *.pem [PKCS#12,PKCS#8,PEM]\n\
  Key Parameter	*.pem [dsa,ecdsa]\n\
  Cert Request	*.p10 *.pem *.spkac [PKCS#10,PEM,SPKAC]\n\
  XCertPair	*.ccp *.pem [x509(DER),PEM]\n\
");
}

void options(int argc, char **argv)
{
	int i;

	for (i = 1, fn = 0; i < argc; i++) {
		if (argv[i][0] == '-') {
			if (!strcmp(argv[i], "-p")) {
				i++;
				if (i < argc)
					pass = argv[i];
			} else if (!strcmp(argv[i], "-help")) {
				usage();
				exit(EXIT_SUCCESS);
			} else if (!strcmp(argv[i], "-version")) {
				print_version(argv);
				exit(EXIT_SUCCESS);
			} else {
				printf("option error!\n");
				printf("unknown option: `%s'\n", argv[i]);
				usage();
				exit(EXIT_FAILURE);
			}
		} else {
			in[fn] = strdup(argv[i]);
			fn++;
		}
	}
}
