/* certvfy.c */
/*
 * Copyright (c) 2004-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_x509.h>
#include <aicrypto/ok_store.h>

#include "ok_conf.h"

#ifndef AICONFIG
#define AICONFIG "aica.cnf"
#endif

/* global value */
char conf[CONF_PATH_MAX] = AICONFIG;
char ctstore[CONF_PATH_MAX] = "/";
char cert[256] = "";
int depth = 10;
int type = 0;

/* functions */
void usage(void);
void options(int argc, char **argv);

/* store_conf.c */
int store_config(char *fname);

/* common/version.c */
void print_version(char *argv[]);

int main(int argc, char **argv)
{
	STManager *stm = NULL;
	CertList *hd, *cl = NULL;
	CRLList *rl = NULL;
	Cert *ct = NULL;
	int i, sv = 0, err = -1;

	OK_clear_error();
	options(argc, argv);

	if (store_config(conf)) {
		printf("cannot find \"store_dir\" in the file:\n%s\n", conf);
		goto done;
	}

	/* read certificate file */
	if ((ct = Cert_read_file(cert)) == NULL) {
		printf("cannot open a certificate file : %s\n%s\n", cert,
		       OK_get_errstr());
		goto done;
	}

	/* open store manager */
	if ((stm = STM_open(ctstore)) == NULL) {
		printf("cannot open certificate store : %s\n%s\n", ctstore,
		       OK_get_errstr());
		goto done;
	}

	if ((cl = STM_get_pathcert(stm, ct)) == NULL) {
		printf("cannot find trusted CA certificate for %s\n", cert);
		printf
		    ("please import a CA certificate with 'aistore' command.\n");
		goto done;
	}
	rl = STM_get_pathcrl(stm, cl);	/* might be null */

	/* find last of the list (should be ca cert) */
	for (hd = cl; hd; hd = hd->next)
		if (hd->next == NULL)
			break;

	/* list certificate */
	i = 0;
	while (hd) {
		/* verify certificate */
		if ((err =
		     Cert_verify(cl, rl, hd->cert, depth,
				 type | ALLOW_SELF_SIGN)) < 0) {
			printf("cannot verify a certificate file : %s\n%s\n",
			       cert, OK_get_errstr());
			goto done;
		}

		if (err >> 8)
			printf("[%d:err] ", i);
		else
			printf("[%d:ok ] ", i);
		printf("%s", hd->cert->subject);
		printf("\n");

		if ((sv == 0) && (err >> 8))
			sv = err | i;

		hd = hd->prev;
		i++;
	}

	/* print result */
	printf("%s : %s\n", cert, Cert_get_vfyerrstr(sv));

done:
	Cert_free(ct);
	Certlist_free_all(cl);
	CRLlist_free_all(rl);
	if (stm)
		STM_close(stm);
	return sv;
}

void usage(void)
{
	printf("\
Usage: certvfy [OPTION...] FILE\n\
\n\
Options:\n\
  -conf PATH	set the path for an aica configuration file\n\
  -depth NUM	ca-chain max number\n\
  -help		print this message\n\
  -ifcrl	if there is not CRL, don't do revocation check\n\
  -nocrl	do not use CRL to verify certificate\n\
  -novfycrl	do not verify CRL's signature and expiry date\n\
  -version	print version information and exit\n\
");
}

void options(int argc, char **argv)
{
	int i;

	if (argc == 1) {
		usage();
		exit(EXIT_FAILURE);
	}

	for (i = 1; i < argc; i++) {
		if (!strcmp(argv[i], "-conf")) {
			i++;
			if (i < argc)
				strncpy(conf, argv[i], 254);
		} else if (!strcmp(argv[i], "-depth")) {
			i++;
			if (i < argc)
				depth = atoi(argv[i]);
		} else if (!strcmp(argv[i], "-ifcrl")) {
			type |= IF_NO_CRL_DONT_CHECK_REVOKED;
		} else if (!strcmp(argv[i], "-nocrl")) {
			type |= DONT_CHECK_REVOKED;
		} else if (!strcmp(argv[i], "-novfycrl")) {
			type |= DONT_VERIFY_CRL;
		} else if (strstr(argv[i], "-h")) {
			usage();
			exit(EXIT_SUCCESS);
		} else if (!strcmp(argv[i], "-version")) {
			print_version(argv);
			exit(EXIT_SUCCESS);
		} else {
			strncpy(cert, argv[i], 254);
		}
	}
}
