/*
 * Copyright (c) 2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_tool.h>	/* PWD_BUFLEN, OK_set_pem_cry_algo(), ... */

#include "certreq.h"

enum req_op op;

/* objects defined in certreq.c */
extern int type;
extern int keytype;
extern int keysize;
extern int hash;
extern int f_login;	/* introduced from v2.5. see "certreq.h". */
extern int f_simple;
extern int f_simple_ou;
extern int f_sbjalt;
extern int f_sbjalt_dns;
extern int f_sbjalt_dns_fqdn;
extern int f_noint;
extern int accid;
extern int anon;
#ifdef	KEKGRIDCA
extern int f_ucert;
extern int f_hcert;
extern int f_lcert;
extern int f_withhostslash;
#endif

extern char conf[];
extern char reqf[];
extern char keyf[];
extern char pass[];
extern char alt_dns_fqdn[];
extern char inreq[];
extern char crtf[];
extern char caf[];
extern char p12f[];
extern char group[];
extern char ou[];
extern char cn[];
extern char email[];
extern char sgl[];
extern char clcert[];
extern char clkey[];

/* certreq_conf.c */
extern int caport;
extern int usessl;

extern char clp12f[];
extern char certid[];
extern char userid[];
extern char userpw[];
extern char svpath[];

/* common/version.c */
void print_version(char *argv[]);

/* aicrypto/sigalgo.c */
char *get_algoname(int keytype);

/* certreq_v24usage.c */
void usage(void);

/**
 * enable old-style usage
 */
enum req_op options(int argc, char *argv[])
{
	int i;
	enum req_op op = REQ_OP_CSR;

	for (i = 1; i < argc; i++) {
		if (*argv[i] == 0) {
			/* ignore */
		} else if (strcmp("-version", argv[i]) == 0) {
			print_version(argv);
			printf("enabled the OLD-style command-line syntax ");
			printf("(v2.4 or earlier)\n");
			printf("\ndefault algorithm for public-key cryptography: ");
			printf("%s, %d [bit]\n", get_algoname(DEFAULT_KEY_TYPE), DEFAULT_KEY_SIZE);
			exit(EXIT_SUCCESS);
		} else if (strcmp("-conf", argv[i]) == 0) {
			i++;
			if (i < argc) strncpy(conf, argv[i], 254);
		} else if (strcmp("-der", argv[i]) == 0) {
			type = 1;
		} else if (strcmp("-req", argv[i]) == 0) {
			i++;
			if (i < argc) strncpy(reqf, argv[i], 254);
		} else if (strcmp("-key", argv[i]) == 0) {
			i++;
			if (i < argc) strncpy(keyf, argv[i], 254);
		} else if (strcmp("-kp", argv[i]) == 0) {
			i++;
			if (i < argc) strncpy(pass, argv[i], PWD_BUFLEN);
		} else if (strcmp("-noenc", argv[i]) == 0) {
			OK_set_pem_cry_algo(0);	 /* don't encrypt PEM */
		} else if (strcmp("-algo", argv[i]) == 0) {
			i++;
			if (i >= argc) {
				printf("option error!\n");
				printf("-algo: missing argument.\n");
				exit(EXIT_FAILURE);
			}
			keytype = get_keytype(argv[i]);
			if (keytype < 0){
				printf("option error!\n");
				printf("unknown algorithm: `%s'\n", argv[i]);
				exit(EXIT_FAILURE);
			}
		} else if (strcmp("-hash", argv[i]) == 0) {
			i++;
			if (i >= argc) {
				printf("option error!\n");
				printf("-hash: missing argument.\n");
				exit(EXIT_FAILURE);
			}
			hash = get_hash_algo(argv[i]);
			if (hash < 0){
				printf("option error!\n");
				printf("unknown algorithm: `%s'\n", argv[i]);
				exit(EXIT_FAILURE);
			}
		} else if (strcmp("-size", argv[i]) == 0) {
			i++;
			if (i < argc) keysize = atoi(argv[i]);
		} else if (strcmp("-s", argv[i]) == 0) {
			f_simple = 1;
		} else if (strcmp("-sou", argv[i]) == 0) {
			f_simple_ou = 1;
		} else if (strcmp("-alt", argv[i]) == 0) {
			f_sbjalt = 1;
		} else if (strcmp("-alt-dns", argv[i]) == 0) {
			f_sbjalt_dns = 1;
		} else if (strcmp("-alt-dns-fqdn", argv[i]) == 0) {
			f_sbjalt_dns_fqdn = 1;
			i++;
			if (i < argc) strncpy(alt_dns_fqdn, argv[i], 126);
		/*
		 * options for RA communication
		 */
		} else if (strcmp("-new", argv[i]) == 0) {
			op = REQ_OP_ISSUE;	/* old 0; */
			f_login |= CERTREQ_LOGIN_LICE;
			i++;
			if (i < argc) strncpy(userid, argv[i], 30);
		} else if (strcmp("-rvk", argv[i]) == 0) {
			op = REQ_OP_REVOKE;	/* old 1; */
		} else if (strcmp("-upd", argv[i]) == 0) {
			op = REQ_OP_RENEW;	/* old 2; */
		} else if (strcmp("-kupd", argv[i]) == 0) {
			/* op = 3; */
			/* this option is disabled so far. actually,
			 * it seemed that the function req_op_kupdate() was
			 * not completely implemented.
			 */
		} else if (strcmp("-recv", argv[i]) == 0) {
			op = REQ_OP_RECEIVE;	/* old 4; */
			i++;
			if (i < argc) accid = atoi(argv[i]);
		} else if (strcmp("-renew", argv[i]) == 0) {
			op = REQ_OP_REKEY;	/* old 5; CONFUSED... */
			i++;
			if (i < argc) strncpy(userid, argv[i], 30);
		} else if (strcmp("-in", argv[i]) == 0) {
			i++;
			if (i < argc) strncpy(inreq, argv[i], 254);
		} else if (strcmp("-cer", argv[i]) == 0) {
			i++;
			if (i < argc) strncpy(crtf, argv[i], 254);
		} else if (strcmp("-cacer", argv[i]) == 0) {
			i++;
			if (i < argc) strncpy(caf, argv[i], 254);
		} else if (strcmp("-p12", argv[i]) == 0) {
			i++;
			if (i < argc) strncpy(p12f, argv[i], 254);
		} else if (strcmp("-pf", argv[i]) == 0) {
			i++;
			if (i < argc) strncpy(group,argv[i],30);
		/*
		 * options for RA connection
		 */
		} else if (strcmp("-sv", argv[i]) == 0) {
			/*
			 * we need to set op = REQ_OP_ISSUE because v2.4 
			 * does not distinguish between creating CSR only
			 * and issuing a certificate.
			 */
			if (op == REQ_OP_CSR) op = REQ_OP_ISSUE;
			i++;
			if (i < argc) strncpy(svpath, argv[i], 254);
		} else if (strcmp("-pt", argv[i]) == 0) {
			i++;
			if (i < argc) caport = atoi(argv[i]);
		} else if (strcmp("-ssl", argv[i]) == 0) {
			usessl = 1;
		} else if (strcmp("-idpw", argv[i]) == 0) {
			f_login |= CERTREQ_LOGIN_IDPW;	/* old: f_idpw = 1; */
		} else if (strcmp("-ch", argv[i]) == 0) {
			f_login |= CERTREQ_LOGIN_CHPN;	/* old: f_chpin = 1; */
		} else if (strcmp("-anon", argv[i]) == 0) {
			anon = 1;	/* used in req_op_revoke() but... */
		} else if (strcmp("-u", argv[i]) == 0) {
			i++;
			if (i < argc) strncpy(userid, argv[i], 30);
		} else if (strcmp("-p", argv[i]) == 0) {
			i++;
			if (i < argc) strncpy(userpw, argv[i], PWD_BUFLEN);
		} else if (strcmp("-g", argv[i]) == 0) {
			i++;
			if (i < argc) strncpy(group, argv[i], 30);
		} else if (strcmp("-ou", argv[i]) == 0) {
			f_simple = f_simple_ou = 1;
			f_noint = 1;
			i++;
			if (i < argc) strncpy(ou, argv[i], 126);
		} else if (strcmp("-cn", argv[i]) == 0) {
			f_simple = 1;
			f_noint = 1;
			i++;
			if (i < argc) strncpy(cn,argv[i],126);
		} else if (strcmp("-em", argv[i]) == 0) {
			f_simple = 1;
			f_noint = 1;
			i++;
			if (i < argc) strncpy(email, argv[i], 126);
		} else if (strcmp("-sgl", argv[i]) == 0) {
			f_simple = 1;
			i++;
			if (i < argc) strncpy(sgl, argv[i], 30);
		} else if (strcmp("-cl", argv[i]) == 0) {
			usessl = 1;
			i++;
			if (i < argc) strncpy(clp12f, argv[i], 254);
		} else if (strcmp("-clid", argv[i]) == 0) {
			usessl = 1;
			i++;
			if (i < argc) strncpy(certid, argv[i], 30);
		} else if (strcmp("-clcer", argv[i]) == 0) {
			usessl = 1;
			i++;
			if (i < argc) strncpy(clcert, argv[i], 254);
		} else if (strcmp("-clkey", argv[i]) == 0) {
			usessl = 1;
			i++;
			if (i < argc) strncpy(clkey, argv[i], 254);
#ifdef KEKGRIDCA
		} else if (!strcmp("-uid",argv[i])) {
			i++;
			if (i < argc) strncpy(userid,argv[i],30);
		} else if (!strcmp("-lic",argv[i])) {
			i++;
			if (i < argc) strncpy(userid,argv[i],46);
		} else if (!strcmp("-fqdn",argv[i])) {
			i++;
			if (i < argc) strncpy(fqdn,argv[i],254);
		} else if (!strcmp("-ucert", argv[i])) {
			f_login |= CERTREQ_LOGIN_CHPN;
			f_ucert = 1;
			strcpy(svpath,"rra01.kek.jp:globus_client");
			usessl = 1;
		} else if (!strcmp("-hcert", argv[i])) {
			f_hcert = 1;
			strcpy(svpath,"rra01.kek.jp:globus_server");
			usessl = 1;
			OK_set_pem_cry_algo(0);	 /* don't encrypto PEM */
		} else if (!strcmp("-with-host-slash", argv[i])){
                        f_withhostslash=1;
                        i++;

		} else if (!strcmp("-lcert", argv[i])) {
			f_lcert = 1;
			strcpy(svpath,"rra01.kek.jp:globus_server");
			usessl = 1;
			OK_set_pem_cry_algo(0);	 /* don't encrypto PEM */
		} else if (!strcmp("-mou", argv[i])) {
			i++;
			if (i < argc) strncpy(ou,argv[i],126);
#endif
		} else if (strcmp("-help", argv[i]) == 0) {
		  	usage();
			exit(EXIT_SUCCESS);
		} else {
			printf("option error!\n");
			printf("unknown option: `%s'\n", argv[i]);
			printf("Try `certreq -help' for more information.\n");
			exit(EXIT_FAILURE);
		}
	}
	return op;
}

/**
 * print old-style usage
 */
void usage(void)
{
#ifdef KEKGRIDCA
	printf("\
usage: certreq [options]\n\
options (general)\n\
  -ucert       : with -uid option, send a cert-request and get a new user certificate.\n\
  -hcert       : with -fqdn option, send a cert-request and get a new host certificate.\n\
  -lcert       : with -fqdn option, send a cert-request and get a new ldap certificate.\n\
  -rcert       : with -name option and -alt option(Email Address), send a cert-request and get a new robot certificate.\n\
  -uid         : set user id for SSL client authentication'.\n\
  -fqdn hname  : set subject cn for a host/ladp certificate.\n\
  -rvk         : revoke a 'SSL client certificate'.\n\
  -lic license : set one-time license to get a new certificate.\n\
  -mou orgunit : set additional subject ou.\n\
  -with-host-slash : create a CSR using a traditional "host/FQDN" format for the Common Name (CN) field.
  -clcer file  : set a certificate file for SSL client authentication.\n\
  -clkey file  : set a private key file for SSL client authentication.\n\
  -key file    : output a private key named \"file\".\n\
  -cer file    : output a certificate named \"file\".\n\
  -cacer file  : output a CA certificate named \"file\".\n\
  -alt         : request subject alt name.\n\
  -conf path   : set aica config path.\n\
  -version     : print version information and exit.\n\
");
#else
	printf("\
usage: certreq [options]\n\
options (general)\n\
  -algo alg    : prvkey algorithm ... rsa,dsa,ecdsa\n\
  -hash alg    : set CSR signature hash algorithm ...\n\
               : sha1,sha224,sha256,sha384,sha512,md5,md2\n\
  -size num    : prvkey size\n\
  -req file    : output a CSR named \"file\".\n\
  -key file    : output a private key named \"file\".\n\
  -kp passwd   : password for output private key.\n\
  -noenc       : save private key without encryption\n\
  -der         : set output as DER format [default:PEM]\n\
  -s           : simple subject input mode.\n\
  -sou         : input OU at simple subject input mode.\n\
  -sgl title   : globus cn title (host/ldap) for simple subject input.\n\
  -alt         : request subject alt name.\n\
  -conf path   : set aica config path.\n\
  -version     : print version information and exit.\n\
options (remote op)\n\
  -new license : send a cert-request and get a new certificate.\n\
  -rvk         : revoke a 'SSL client certificate'.\n\
  -upd         : update a 'SSL client certificate' (use same public key).\n\
  -renew license : revoke old certificate and get a new certificate (rekey).\n\
  -recv accID  : receive a certificate (CA operator confirmation mode).\n\
  -in  file    : input a CSR named \"file\".\n\
  -cer file    : output a certificate named \"file\".\n\
  -cacer file  : output a CA certificate named \"file\".\n\
  -p12 file    : output a PKCS#12 named \"file\".\n\
options (client info)\n\
  -sv path     : set RA server path.\n\
  -pt port     : set RA server port number.\n\
  -ssl         : use SSL connection.\n\
  -cl    file  : set a PKCS#12 file for SSL client authentication.\n\
  -clid  id    : set store cert-id with SSL client authentication.\n\
  -clcer file  : set a certificate file for SSL client authentication.\n\
  -clkey file  : set a private key file for SSL client authentication.\n\
  -p passwd    : password for client private key and id/pw login mode.\n\
  -g group     : set group name (for non-interactive mode).\n\
  -ou orgunit  : set subject ou (set non-interactive mode).\n\
  -cn name     : set subject cn (set non-interactive mode).\n\
  -em mail     : set subject email (set non-interactive mode).\n\
  -alt-dns     : use CN as dNSName in subjectAltName (non-interactive mode).\n\
  -alt-dns-fqdn FQDN : set dNSName in subjectAltName to FQDN (ditto).\n\
  -idpw        : use id/pw authentication mode instead of licenseID\n\
  -ch          : use licenseID & challenge PIN mode\n\
");
#endif
}
