/* certreq.c */
/*
 * Copyright (c) 2004-2018 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_rand.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_pem.h>
#include <aicrypto/ok_uconv.h>

#include "ok_lcmp.h"
#include "ok_conf.h"
#include "certreq.h"

#ifndef	AICONFIG
#define	AICONFIG "aica.cnf"
#endif

/* usage messages */
#define HEAD_OPTIONS    "\nOptions:\n"
#define HEAD_FILEOPTS   "\nFile Options:\n"

#define USAGE_OPT_REQ   "  -req FILE	output a CSR into FILE\n"
#define USAGE_OPT_DER   "  -der		output in DER format [default:PEM]\n"
#define USAGE_OPT_KEY   "  -key FILE	output a private key into FILE\n"
#define USAGE_OPT_KP    "  -kp PASSWD	set password for a private key\n"
#define USAGE_OPT_NOENC "  -noenc	save a private key without encryption\n"
#define USAGE_OPT_CER   "  -cer FILE	output a certificate into FILE\n"
#define USAGE_OPT_P12   "  -p12 FILE	save a private key and certificates as FILE in PKCS#12 format\n"
#define USAGE_OPT_CACER "  -cacer FILE	save the CA certificate obatained from an RA as FILE\n"

#define USAGE_OPT_IN "  -in FILE	read a CSR from FILE\n"
/* a private key should be specified if the options `-in' & `-p12' were used. */
#define USAGE_OPT_INKEY "  -inkey FILE	read a private key from FILE\n"

#define HELP_OPERATIONS "\nSee 'certreq help' for a list of commands and general options.\n"

#define	usage_keyhash() \
	printf("  -algo ALG	set an algorithm for public-key cryptography [rsa,dsa,ecdsa]\n");\
	printf("  -size NUM	set key size or key length [default:%d]\n", DEFAULT_KEY_SIZE); \
	printf("  -hash ALG	set a hash algorithm used in CSR signature\n");\
	printf("		[sha256,sha384,sha512,sha512-224,sha512-256,sha1,sha224,\n"); \
	printf("		 sha3-224,sha3-256,sha3-384,sha3-512]\n");

#ifdef KEKGRIDCA
#define	usage_authmode() \
	printf("\nAuthentication Options (REQUIRED, except anonymous auth-mode):\n");\
	printf("  -lic LICENSE	set LicenseID for licenceID or license/PIN auth-mode\n");
#else
#define	usage_authmode() \
	printf("\nAuthentication Options (REQUIRED, except anonymous auth-mode):\n");\
	printf("  -user ID	set user ID for id/password auth-mode\n");\
	printf("  -pw PASSWD	set password for id/password auth-mode\n");\
	printf("  -lic LICENSE	set LicenseID for licenceID or license/PIN auth-mode\n");\
	printf("  -pin PIN	set PIN for licenseID/PIN auth-mode\n");\
	printf("  -ch		enable to input PIN in the licenseID/PIN auth-mode (interactive)\n");\
	printf("According to the auth-mode configuration of your RA, specify one\n");\
	printf("combination from below:\n");\
	printf("  (none)			anonymous auth-mode\n");\
	printf("  -user ID -pw PASSWD		id/password auth-mode\n");\
	printf("  -lic LICENSE			licenseID auth-mode\n");\
	printf("  -lic LICENSE -pin PIN		licenseID/PIN auth-mode\n");\
	printf("  -lic LICENSE -ch		ditto (interactive)\n");
#endif

#define	usage_ssl_client_auth() \
	printf("\nEnd-Entity Certificate Options (REQUIRED AT LEAST ONE OPTION):\n"); \
	printf("  -cl FILE	set a PKCS#12 file for SSL client authentication\n"); \
	printf("  -clcer FILE	set a certificate file for SSL client authentication\n"); \
	printf("  -clkey FILE	set a private key file for SSL client authentication\n"); \
	printf("  -clpw PASSWD	set password for the private key used in the authentication\n"); \
	printf("Specify one combination from below:\n"); \
	printf("  -cl FILE\n"); \
	printf("  -clcer FILE -clkey FILE [-clpw passwd]\n");

#ifdef KEKGRIDCA
#define	usage_sbj_input() \
	printf("\nSubject Options:\n"); \
	printf("  -mou orgunit          set additional subject ou.\n");\
	printf("  -with-host-slash      create a CSR using a traditional 'host/FQDN'\n                        format for the Common Name (CN) field.\n");
#else
#define	usage_sbj_input() \
	printf("\nSubject Options:\n"); \
	printf("  -s		enter simple input mode (interactive)\n"); \
	printf("  -sou		enable to input orgUnit in the simple input mode\n"); \
	printf("  -ou orgUnit	set organizational unit in subject\n"); \
	printf("  -cn NAME	set common name in subject\n"); \
	printf("  -em MAIL	set email address in subject (for legacy profile)\n");
#endif
	/*
	printf("  -sgl TITLE	: globus cn title (host/ldap) for simple subject input.\n"); \
	 */

#define	usage_sbjalt_input() \
	printf("\nsubjectAltName Options:\n"); \
	printf("  -alt			add a subjectAltName to a CSR (interactive)\n"); \
	printf("  -alt-dns		use CN as dNSName in subjectAltName\n"); \
	printf("  -alt-dns-fqdn FQDN	set dNSName in subjectAltName to FQDN\n");

#ifdef	KEKGRIDCA
#define	usage_kek() \
	printf("  -ucert	user certificate operation\n");\
	printf("  -hcert	host certificate operation\n");\
	printf("  -lcert	ldap certificate operation\n");\
        printf("  -rcert        robot certificate operation\n");\
	printf("  -uid uname	set user name for SSL client authentication\n");\
	printf("  -fqdn hname	set subject cn for a host/ladp certificate\n");\
        printf("  -name fullname   set your fullname registered in NAREGI CA for a robot certificate\n");
#endif

/* XXX: need comment */
#define AUTH_ISSET(var, flag)  ( ( (~(var)) ^ (~(flag)) ) ==0)

/* req.c */
int get_key_pairs(Key **pub,Key **prv,int size,int type);
char *input_subject(CertDN *dn,char *prompt);
ExtGenNames *input_gennames(char *prompt);

/* certreq.c */
#ifndef OLD_USAGE
void options(int argc,char **argv);
void usage(enum req_op op);
#else
/* certreq_v24usage.c */
enum req_op options(int argc, char *argv[]);
void usage(void);
#endif
static enum req_op get_operation(int argc, char **argv, int *idx);
static int check_file_permission(char *fname);
static Req *create_csr(Key **prv, CertDN *reuse_dn, CertExt *reuse_ex);
int req_op_csr();
int req_op_issue();
int req_op_revoke();
int req_op_renew();
int req_op_getcert();
int req_op_rekey();

void print_protocol_version_mismatch(void);

/* certreq_conf.c */
int req_read_config(char *fname);

/* certreq_tool.c */
char *input_simple_subject(CertDN *dn, int idpw, int ch);
int input_simple_subject2(Req *req);
int req_ask_onoff(char *txt,int on);
int req_ask_yesno(char *txt,int yes);
int req_ask_num(char *txt,int num);
int req_ask_comment(char *txt,char *buf, int max);
#ifdef KEKGRIDCA
char *input_simple_subjectk(CertDN *dn);
char *input_simple_subjecth(CertDN *dn);
#endif

/* common/version.c */
void print_version(char *argv[]);

/* aicrypto/sigalgo.c */
char *get_algoname(int keytype);

enum req_op op;     /* operation */
int  type     = 0;  /* 1: DER, 0: PEM */
int  f_simple = 0;  /* simple subject input mode (maybe for grid environment) */
int  f_simple_ou = 0;  /* request OU flag at simple input mode */

int  f_sbjalt = 0;  /* request subject alt name flag (interactive mode) */
int  f_sbjalt_dns = 0;	/* dNSName in subjectAltName (non-interactive mode) */
int  f_sbjalt_dns_fqdn = 0;	/* ditto, but a FQDN required as an argument */
bool f_dry_run = false;
int  f_noint  = 0;  /* non interactive mode */
int  f_login  = CERTREQ_LOGIN_NONE;  /* login mode */
int  keysize  = DEFAULT_KEY_SIZE;
int  f_keysize= 0; /* 1: -size flag is specified */
int  keytype  = DEFAULT_KEY_TYPE;
int  f_keytype= 0; /* 1: -algo flag is specified */
int  hash     = DEFAULT_HASH_TYPE;
int  accid    = 0;
int  anon     = 0;
#ifdef KEKGRIDCA
int  f_ucert  = 0;
int  f_hcert  = 0;
int  f_lcert  = 0;
int  f_rcert  = 0;
int  f_mou    = 0;
int  f_withhostslash = 0;
char fqdn[256]= "";
char name[256]="";
char service[256]="";
#endif

char pass[PWD_BUFLEN] = "";
char inreq[256] = "";
char inkey[256] = ""; /* an existing private key */
char keyf[256]  = "newkey.key";
char reqf[256]  = "newreq.p10";
char crtf[256]  = "newcert.cer";
char p12f[256]  = "";
char caf[256]   = "cacert.cer";
char conf[256]  = AICONFIG;
char group[32]  = "";
char sgl[32]    = "";	/* Kerberos-style "service" name in CN (GFD.125,3.2) */

char ou[128]    = "";
char cn[128]    = "";
char email[128] = "";
//char alt_dns_fqdn[128] = "";
char alt_dns_fqdn[FQDN_LEN_MAX * 2] = "";

char clcert[256]= "";
char clkey[256] = "";
char clctpw[PWD_BUFLEN*2] = "";

char licid[64]  = "";
char pin[PWD_BUFLEN*2] = "";

char *authid = NULL;
char *authpw = NULL;

CertDN tmpdn;
CertExt *tmpex = NULL;

extern char svpath[];
extern char caname[];
extern char ctstore[];

extern char clp12f[];
extern char certid[];
extern char userid[];
extern char userpw[];

extern int usessl;
extern int caport;
extern int vfycert;


/*-------------------------------------------------
	begin certreq main
-------------------------------------------------*/
int main(int argc,char **argv){
	int ok = -1;

	RAND_init();
	OK_clear_error();

#ifndef OLD_USAGE
	options(argc,argv);
#else
	op = options(argc, argv);
#ifdef KEKGRIDCA
	if (f_ucert || f_hcert || f_lcert || f_rcert){
		if (op == REQ_OP_CSR) op = REQ_OP_ISSUE;
	}
#endif
#endif

#ifdef KEKGRIDCA
	if((f_ucert+f_hcert+f_lcert+f_rcert)==0){
		printf("necessary option missing!\n");
		usage(op);
		exit(1);
	}else if((f_ucert+f_hcert+f_lcert+f_rcert)>1) {
		printf("option combination error!\n");
		usage(op);
		exit(1);
	}
	if(op == REQ_OP_CSR || op == REQ_OP_ISSUE || op == REQ_OP_REKEY){
		if((f_hcert || f_lcert) && ((!*fqdn) || (!strcmp(userid, "dummy")))){
			printf("option combination error!\n");
			usage(op);
			exit(1);
                }
                if((f_rcert) && ((!*name) || (!*service))){
                        printf("option combination error!\n");
                        usage(op);
                        exit(1);

		}
		if(f_ucert && (!strcmp(userid, "dummy"))){
			printf("option combination error!\n");
			usage(op);
			exit(1);
		}
                if(f_ucert && f_mou ){
                        printf("option combination error!\n");
                        usage(op);
                        exit(1);
                }
	}
	if(f_ucert){
		keysize = 4096;
		strcpy(keyf,"userkey.pem");
		strcpy(reqf,"usercert.p10");
		strcpy(crtf,"usercert.pem");
		strcpy(caf, "e7a6d5cc.0");
		strcpy(group,"Globus Client");
	}else if(f_hcert){
		keysize = 4096;
		strcpy(keyf,"hostkey.pem");
		strcpy(reqf,"hostcert.p10");
		strcpy(crtf,"hostcert.pem");
		strcpy(caf, "e7a6d5cc.0");
		strcpy(group,"Globus Server");
	}else if(f_lcert){
		keysize = 4096;
		strcpy(keyf,"ldapkey.pem");
		strcpy(reqf,"ldapcert.p10");
		strcpy(crtf,"ldapcert.pem");
		strcpy(caf, "e7a6d5cc.0");
		strcpy(group,"Globus Server");
	}else if(f_rcert){
                keysize = 4096;
                strcpy(keyf,"robotkey.pem");
                strcpy(reqf,"robotcert.p10");
                strcpy(crtf,"robotcert.pem");
                strcpy(caf, "e7a6d5cc.0");
                strcpy(group,"Globus Robot");
	}
#endif
	/* Note that the following function does NOT overwrite the settings
	 * from command-line options. [#20]
	 */
	/* XXX: REQ_OP_CSR doesn't need any information defined in aica.cnf. */
	if (REQ_OP_CSR != op) {
		if(req_read_config(conf)){
			printf("cannot find a config file : %s\n",conf);
			goto done;
		}
	}
	
	/* do operation */
	switch(op){
	case REQ_OP_CSR: /* create a request */
		if(req_op_csr()) goto done;
		break;
	case REQ_OP_ISSUE: /* create a request and get a certificate */
		if(req_op_issue()) goto done;
		break;
	case REQ_OP_REVOKE: /* revoke a certificate */
		if(req_op_revoke()) goto done;
		break;
	case REQ_OP_RENEW: /* renew a certificate */
		if(req_op_renew()) goto done;
		break;
	case REQ_OP_RECEIVE: /* receive a certificate */
		if(req_op_getcert()) goto done;
		break;
	case REQ_OP_REKEY:  /* update key & certificate */
		if(req_op_rekey()) goto done;
		break;
	default:
		break;
	}

	ok = 0;
done:
	if(ok && OK_get_error()) printf("%s\n",CA_get_errstr());
	memset(pass,0,PWD_BUFLEN);
	memset(userpw,0,PWD_BUFLEN);
	memset(pin,0,PWD_BUFLEN);
	memset(clctpw,0,PWD_BUFLEN);
	RAND_cleanup();
	free_u2j_table();
	return ok;
}

#ifndef OLD_USAGE
/*-----------------------------------------
	usage and option check
-----------------------------------------*/
void options(int argc,char **argv){
	int	i;

	op = get_operation(argc, argv, &i);

	if (op==REQ_OP_HELP||op==REQ_OP_UNKNOWN){
	  	usage(op);
		exit((op==REQ_OP_HELP) ? 0 : 1);
	}

#ifdef KEKGRIDCA
	if(i==1){
		op=REQ_OP_UNKNOWN;
	}
#endif

	for(;i<argc;i++){
		if(*argv[i]==0){
			/* ignore */
		 
		}else if(!strcmp("-conf",argv[i])){
			i++;
			if(i<argc) strncpy(conf,argv[i],254);
		}else if(!strcmp("-der",argv[i])){
			type = 1;

		}else if(!strcmp("-req",argv[i])){
			i++;
			if(i<argc) strncpy(reqf,argv[i],254);
		}else if(!strcmp("-key",argv[i])){
			i++;
			if(i<argc) strncpy(keyf,argv[i],254);
		}else if(!strcmp("-kp",argv[i])){
			i++;
			if(i<argc) strncpy(pass,argv[i],PWD_BUFLEN);
		}else if(!strcmp("-noenc",argv[i])){
			OK_set_pem_cry_algo(0);	 /* don't encrypt PEM */

		}else if(!strcmp("-algo",argv[i])){
			i++;
			if(i>=argc){
			  printf("option error!\n");
			  printf("-algo: missing argument.\n");
			  exit(1); /* FIXME: what is the appropriate number? */
			}
			keytype=get_keytype(argv[i]);
			if(keytype<0){
			  printf("option error!\n");
			  printf("unknown algorithm: `%s'\n", argv[i]);
			  exit(1); /* FIXME: what is the appropriate number? */
			}
			f_keytype=1;

		}else if(!strcmp("-hash",argv[i])){
			i++;
			if(i>=argc){
			  printf("option error!\n");
			  printf("-hash: missing argument.\n");
			  exit(1); /* FIXME: what is the appropriate number? */
			}
			hash=get_hash_algo(argv[i]);
			if(hash<0){
			  printf("option error!\n");
			  printf("unknown algorithm: `%s'\n", argv[i]);
			  exit(1); /* FIXME: what is the appropriate number? */
			}

		}else if(!strcmp("-size",argv[i])){
			i++;
			if(i<argc){
				keysize=atoi(argv[i]);
				f_keysize=1;
			}
		}else if(!strcmp("-s",argv[i])){
			f_simple = 1;

		}else if(!strcmp("-sou",argv[i])){
			f_simple_ou = 1;

		}else if(!strcmp("-alt",argv[i])){
			f_sbjalt = 1;
		}else if(!strcmp("-alt-dns",argv[i])){
			f_sbjalt_dns = 1;
		} else if (strcmp("-alt-dns-fqdn", argv[i]) == 0) {
			f_sbjalt_dns_fqdn = 1;
			i++;
			if (i < argc)
				strncpy(alt_dns_fqdn, argv[i],
					sizeof(alt_dns_fqdn) - 1);
		} else if (strcmp("-dry-run", argv[i]) == 0) {
			f_dry_run = true;
		}else if(!strcmp("-acpt",argv[i])){
			i++;
			if(i<argc) accid = atoi(argv[i]);
		}else if(!strcmp("-in",argv[i])){
			i++;
			if(i<argc) strncpy(inreq,argv[i],254);
		}else if (strcmp("-inkey", argv[i]) == 0) {
			i++;
			if (i < argc) strncpy(inkey, argv[i], 254);
		}else if(!strcmp("-cer",argv[i])){
			i++;
			if(i<argc) strncpy(crtf,argv[i],254);
		}else if(!strcmp("-cacer",argv[i])){
			i++;
			if(i<argc) strncpy(caf,argv[i],254);
		}else if(!strcmp("-p12",argv[i])){
			i++;
			if(i<argc) strncpy(p12f,argv[i],254);
		}else if(!strcmp("-pf",argv[i])){
			i++;
			if(i<argc) strncpy(group,argv[i],30);

		/* connection options */
		}else if(!strcmp("-sv",argv[i])){
			i++;
			if(i<argc) strncpy(svpath,argv[i],254);
		}else if(!strcmp("-pt",argv[i])){
			i++;
			if(i<argc) caport = atoi(argv[i]);
		}else if(!strcmp("-ssl",argv[i])){
			usessl=1;

		}else if(!strcmp("-user",argv[i])){
			f_login |= CERTREQ_LOGIN_IDPW;
			/* f_idpw=1; */
			i++;
			if(i<argc) strncpy(userid,argv[i],30);
		}else if(!strcmp("-pw",argv[i])){
			i++;
			if(i<argc) strncpy(userpw,argv[i],PWD_BUFLEN);
#ifndef KEKGRIDCA
		}else if(!strcmp("-lic",argv[i])){
			f_login |= CERTREQ_LOGIN_LICE;
			i++;
			if(i<argc) strncpy(licid,argv[i],30);
#endif
		}else if(!strcmp("-pin",argv[i])){
			i++;
			if(i<argc) strncpy(pin,argv[i],PWD_BUFLEN);
		}else if(!strcmp("-ch",argv[i])){
			f_login |= CERTREQ_LOGIN_CHPN;
		}else if(!strcmp("-g",argv[i])){
			i++;
			if(i<argc) strncpy(group,argv[i],30);
		}else if(!strcmp("-ou",argv[i])){
			i++; f_simple = f_simple_ou = 1; f_noint = 1;
			if(i<argc) strncpy(ou,argv[i],126);
		}else if(!strcmp("-cn",argv[i])){
			i++; f_simple = 1; f_noint = 1;
			if(i<argc) strncpy(cn,argv[i],126);
		}else if(!strcmp("-em",argv[i])){
			i++;
                        // f_simple = 1; f_noint = 1;
			if(i<argc) strncpy(email,argv[i],126);
		}else if(!strcmp("-sgl",argv[i])){
			i++; f_simple = 1;
			if(i<argc) strncpy(sgl,argv[i],30);
		}else if(!strcmp("-cl",argv[i])){
			usessl=1; i++;
			if(i<argc) strncpy(clp12f,argv[i],254);
		}else if(!strcmp("-clid",argv[i])){
			usessl=1; i++;
			if(i<argc) strncpy(certid,argv[i],30);
		}else if(!strcmp("-clcer",argv[i])){
			usessl=1; i++;
			if(i<argc) strncpy(clcert,argv[i],254);
		}else if(!strcmp("-clkey",argv[i])){
			usessl=1; i++;
			if(i<argc) strncpy(clkey,argv[i],254);
#ifdef KEKGRIDCA
		}else if(!strcmp("-uid",argv[i])){
			i++;
			if(i<argc) strncpy(userid,argv[i],30);
                }else if(!strcmp("-name",argv[i])){
                        i++;
                        if(i<argc) strncpy(name,argv[i],254);
                }else if(!strcmp("-service",argv[i])){
                        i++;
                        if(i<argc) strncpy(service,argv[i],254);
		}else if(!strcmp("-lic",argv[i])){
			i++;
			if(i<argc) strncpy(userid,argv[i],46);
		}else if(!strcmp("-fqdn",argv[i])){
			i++;
			if(i<argc) strncpy(fqdn,argv[i],254);
		}else if(!strcmp("-ucert", argv[i])){
			f_login |= CERTREQ_LOGIN_CHPN;
			f_ucert=1;
			strcpy(svpath,"rra01.kek.jp:globus_client");
			usessl=1;		
		}else if(!strcmp("-hcert", argv[i])){
			f_hcert=1;
			strcpy(svpath,"rra01.kek.jp:globus_server");
			usessl=1;		
			OK_set_pem_cry_algo(0);	 /* don't encrypto PEM */
		}else if(!strcmp("-rcert", argv[i])){
			f_rcert=1;
                        f_sbjalt = 1;
			strcpy(svpath,"rra01.kek.jp:globus_robot");
			usessl=1;
			OK_set_pem_cry_algo(0);	 /* don't encrypto PEM */
                }else if(!strcmp("-lcert", argv[i])){
                        f_lcert=1;
                        strcpy(svpath,"rra01.kek.jp:globus_server");
                        usessl=1;
                        OK_set_pem_cry_algo(0);  /* don't encrypto PEM */
		}else if(!strcmp("-mou", argv[i])){
                        f_mou=1;
			i++;
			if(i<argc) strncpy(ou,argv[i],126);
		}else if(!strcmp("-with-host-slash", argv[i])){
                        f_withhostslash=1;
#endif
		}else if(!strcmp("-clpw",argv[i])){
			i++;
			if(i<argc) strncpy(clctpw,argv[i],PWD_BUFLEN);
		}else if(!strcmp("-help",argv[i])){
		  	usage(op);
			exit(0);
		}else if(!strcmp("-version",argv[i])){
		  	print_version(argv);
			printf("\ndefault algorithm for public-key cryptography: ");
			printf("%s, %d [bit]\n", get_algoname(DEFAULT_KEY_TYPE), DEFAULT_KEY_SIZE);
			exit(EXIT_SUCCESS);
		}else{
			printf("unknown option: `%s'\n", argv[i]);
			printf("Try `certreq COMMAND -help' for more information.\n");
			/* usage(op); */
			exit(EXIT_FAILURE);
		}
	}
#ifdef KEKGRIDCA
	if (op==REQ_OP_UNKNOWN){
	  	usage(op);
		exit(1);
	}
#endif
}

void usage(enum req_op op){
	switch (op) {
	case REQ_OP_CSR:
		printf("Usage: certreq csr [OPTION...]\n");
		printf(HEAD_OPTIONS);
		printf(USAGE_OPT_REQ);
		printf(USAGE_OPT_DER);
#ifndef KEKGRIDCA
		printf(USAGE_OPT_KEY);
		printf(USAGE_OPT_KP);
		printf(USAGE_OPT_NOENC);
#endif
#ifdef KEKGRIDCA
		usage_kek();
#endif
		usage_keyhash();
		usage_sbj_input();
		usage_sbjalt_input();
		printf(HELP_OPERATIONS);
		break;
	case REQ_OP_ISSUE:
		printf("Usage: certreq issue [OPTION...]\n");
		usage_authmode();
		printf(HEAD_FILEOPTS);
#ifndef KEKGRIDCA
		printf(USAGE_OPT_KEY);
		printf(USAGE_OPT_KP);
		printf(USAGE_OPT_NOENC);
#endif
#ifdef KEKGRIDCA
		usage_kek();
#endif
		usage_keyhash();
		printf(USAGE_OPT_IN);
		printf(USAGE_OPT_INKEY);
#ifndef KEKGRIDCA
		printf(USAGE_OPT_CER);
		printf(USAGE_OPT_P12);
		printf(USAGE_OPT_CACER);
#endif
		printf(USAGE_OPT_REQ);
		printf(USAGE_OPT_DER);
		usage_sbj_input();
		usage_sbjalt_input();
		printf("\nRA Options:\n");
		printf("  -g GROUP	set a group name used in a request to an RA.\n");
		printf(HELP_OPERATIONS);
		break;
	case REQ_OP_REVOKE:
		printf("Usage: certreq revoke OPTION...\n");
#ifdef KEKGRIDCA
		usage_kek();
#endif
		usage_ssl_client_auth();
		printf(HELP_OPERATIONS);
		break;
	case REQ_OP_RENEW:
		printf("Usage: certreq renew OPTION...\n");
		printf(HEAD_FILEOPTS);
#ifndef KEKGRIDCA
		printf(USAGE_OPT_KP);
		printf(USAGE_OPT_CER);
		printf(USAGE_OPT_P12);
		printf(USAGE_OPT_CACER);
#endif
		printf(USAGE_OPT_DER);
		usage_ssl_client_auth();
		printf(HELP_OPERATIONS);
		break;
	case REQ_OP_RECEIVE:
		printf("Usage: certreq receive -acpt acptID [OPTION...]\n");
		printf(HEAD_OPTIONS);
		printf("  -acpt acptID	set CSR acceptID (required)\n");
		printf(USAGE_OPT_DER);
#ifndef KEKGRIDCA
		printf(USAGE_OPT_CER);
		printf(USAGE_OPT_CACER);
#endif
		printf(HELP_OPERATIONS);
		break;
	case REQ_OP_REKEY:
		printf("Usage: certreq rekey OPTION...\n");
		usage_ssl_client_auth();
		usage_authmode();
		printf(HEAD_FILEOPTS);
#ifndef KEKGRIDCA
		printf(USAGE_OPT_KEY);
		printf(USAGE_OPT_NOENC);
#endif
#ifdef KEKGRIDCA
		usage_kek();
#endif
		usage_keyhash();
#ifndef KEKGRIDCA
		printf(USAGE_OPT_KP);
#endif
		printf(USAGE_OPT_IN);
#ifndef KEKGRIDCA
		printf(USAGE_OPT_CER);
		printf(USAGE_OPT_P12);
		printf(USAGE_OPT_CACER);
#endif
		printf(USAGE_OPT_REQ);
		printf(USAGE_OPT_DER);
		usage_sbjalt_input();
		printf(HELP_OPERATIONS);
		break;
	default:
		printf("Usage: certreq [COMMAND] [OPTION...]\n");
		printf("\nCommands:\n");
		printf("  csr		create a CSR and save it (default)\n");
		printf("  issue		send a CSR and get a new end-entity certificate\n");
		printf("  revoke	revoke a certificate\n");
		printf("  receive	receive a certificate (CA operator confirmation mode)\n");
		printf("  rekey		re-key a certificate (create a new key-pair)\n");
		printf("  renew		renew a certificate (use same public key)\n");
		printf("  help		print this message\n");
		printf("\nGeneral Options (except `csr' command):\n");
		printf("  -sv RAPATH	set an RA server [RAPTH=hostname:ra_name]\n");
		printf("  -pt PORT	set a port number that the RA daemon listens on\n");
		printf("  -clid ID	specify a stored certificate used in SSL client authentication\n");
		printf("\nSee `certreq COMMAND -help' for more information on a specific command.\n");
	}
}
#endif /* !OLD_USAGE */

static enum req_op get_operation(int argc, char **argv, int *idx){
	enum req_op op = REQ_OP_CSR;
	int i = 1;
	char *arg;

	if(argc > 1){
		arg=argv[i];
		if(*arg != '-'){
			i++;
			if(!strcmp("csr",arg)){
				op = REQ_OP_CSR;
			}else if(!strcmp("issue",arg)){
				op = REQ_OP_ISSUE;
			}else if(!strcmp("revoke",arg)){
				op = REQ_OP_REVOKE;
			}else if(!strcmp("renew",arg)){
				op = REQ_OP_RENEW;
			}else if(!strcmp("receive",arg)){
				op = REQ_OP_RECEIVE;
			}else if(!strcmp("rekey",arg)){
				op = REQ_OP_REKEY;
			}else if(!strcmp("help",arg)){
				op = REQ_OP_HELP;
			}else{
				fprintf(stderr,"unknown command: %s\n",arg);
				op = REQ_OP_UNKNOWN;
			}
		}
#ifdef KEKGRIDCA
	}else{
		op = REQ_OP_HELP;
#endif
	}

	*idx=i;
	return op;
} 

/* save private key */
static int
save_private_key(Key *prv)
{
	if (prv == NULL) {
		return 1; /* ignore */
	}
	if (*pass) {
		OK_set_passwd(pass);
	}
	switch (prv->key_type) {
	case KEY_RSA_PRV:
		if (PEM_write_rsaprv((Prvkey_RSA *)prv, keyf)) {
			goto error;
		}
		break;
	case KEY_DSA_PRV:
		if (PEM_write_dsaprv((Prvkey_DSA *)prv, keyf)) {
			goto error;
		}
		break;
	case KEY_ECDSA_PRV:
		if (PEM_write_ecdsaprv((Prvkey_ECDSA *)prv, keyf)) {
			goto error;
		}
		break;
	}
	printf("save a private key file : %s\n", keyf);
	return 1;
error:
	printf("cannot save a private key file : %s\n", keyf);
	return 0;
}

static int
save_csr(Req *req)
{
	if (req == NULL) {
		return 1; /* ignore */
	}
	if (type) {
		if (ASN1_write_der(req->der, reqf)) {
			goto error;
		}
	} else {
		if (PEM_write_req(req, reqf)) {
			goto error;
		}
	}
	printf("save a pkcs#10 (CSR) file : %s\n", reqf);
	return 1;
error:
	printf("cannot save a CSR file : %s\n", reqf);
	return 0;
}

static int check_file_permission(char *fname){
	FILE *fp; 
	if((fp=fopen(fname,"ab+"))==NULL){
		printf("permission error. cannot open file : %s\n",fname);
		return 0;
	}
	fclose(fp);
	return 1;
}

static Req *create_csr(Key **prv, CertDN *reuse_dn, CertExt *reuse_ex){
	Req *req=NULL;
	Key *pub=NULL;
	ExtGenNames *egn = NULL;
	CertExt *et=NULL;
	char *cp;
	int i;
	int sigalgo;
	int cn_set_userid = 0;
	int cn_set_dummy = 0;
	int ok = 0;

	if (f_dry_run)
		goto subjectAltName;

	sigalgo = select_sigalgo(keytype, hash);
	if (sigalgo < 0) {
		printf("invalid combination of key type and hash type.\n");
		goto done;
	}
	OK_set_sign_digest_algo(sigalgo);
	if(get_key_pairs(&pub,prv,keysize,keytype)) goto done;

	if ((req = Req_new()) == NULL) {
		goto done;
	}

	req->pubkey_algo = keytype;
	req->pubkey = pub; pub = NULL;

	printf("-------------------------------------------\n");
	printf("  creating a certificate signing request\n");
	printf("-------------------------------------------\n");

	if(reuse_dn!=NULL){ /* for rekey operation */
		/* set subject dn */
		if(Cert_dncopy(reuse_dn,&req->subject_dn)) goto done;
		cp = Cert_find_dn(reuse_dn,OBJ_DIR_CN,&i,0);
		if(cp) strncpy(cn,cp,126);
		cp = Cert_find_dn(reuse_dn,OBJ_DIR_EMAIL,&i,0);
		if(cp) strncpy(email,cp,126);

		req->subject=Cert_subject_str(&req->subject_dn);
#if 0
		printf("--- create_csr() ---\n");
		printf("    reuse_dn=%s\n", Cert_subject_str(reuse_dn));
		printf("req->subject=%s\n", Cert_subject_str(&req->subject_dn));
		printf("----------------------\n");
#endif
	}else{
#ifdef KEKGRIDCA
		if(AUTH_ISSET(f_login, CERTREQ_LOGIN_CHPN) && (*userpw==0)){
			printf("\nplease input your challenge pin or password to get a certificate.\n");
			OK_get_passwd("Input Challenge PIN or Password : ",userpw,0);
		}
#endif
		if(f_simple){
			if((*licid==0)&& AUTH_ISSET(f_login, CERTREQ_LOGIN_IDPW))
				cn_set_userid=1;

			if((*licid)&&(*pin)) cn_set_dummy=1;

#if 0
			printf("--- create_csr() ---\n");
			printf("cn_set_userid=%d\n", cn_set_userid);
			printf("cn_set_dummy=%d\n", cn_set_dummy);
			printf("----------------------\n");
#endif

			if((req->subject=input_simple_subject(&req->subject_dn,
							cn_set_userid, cn_set_dummy))==NULL)
				goto done;
#ifdef KEKGRIDCA
		} else if(f_ucert){
			if((req->subject=input_simple_subjectk(&req->subject_dn))==NULL) goto done;
		} else if(f_hcert || f_lcert || f_rcert ) {
			if((req->subject=input_simple_subjecth(&req->subject_dn))==NULL) goto done;
#endif
		}else{
			if((req->subject=input_subject(&req->subject_dn,NULL))==NULL) goto done;
		}
	}

subjectAltName:
	if(f_sbjalt_dns_fqdn){
		char expd_fqdn[FQDN_MAX][FQDN_LEN_MAX];
		int fqdn_num;
		ExtGenNames *head, *tmp;

		fqdn_num = expand_str(expd_fqdn, alt_dns_fqdn);
		if (fqdn_num > 0) {
			int k;
			for (k = 0; k < fqdn_num; k++) {
				if (f_dry_run) {
					printf("fqdn[%d]: %s\n", k, expd_fqdn[k]);
				} else {
					tmp = ExtGN_set_dns(expd_fqdn[k]);
					if (egn == NULL) {
						egn = head = tmp;
					} else {
						head->next = tmp;
						head = tmp;
					}
				}
			}
		} else if (fqdn_num == 0) {
			if (f_dry_run) {
				printf("%s\n", alt_dns_fqdn);
			} else {
				egn = ExtGN_set_dns(alt_dns_fqdn);
			}
		} else {
			if (f_dry_run) {
				printf("no expanded string.(err=%d)\n", fqdn_num);
			} else {
				/* XXX: how should it be handled? */
			}
		}
		if (f_dry_run)
			goto done;
	}else if(f_sbjalt_dns){
		egn=ExtGN_set_dns(cn);
	}else if(f_sbjalt){
		egn=input_gennames("input SubjectAltName.\n"); /* not compare */
#ifdef KEKGRIDCA
	}else if(f_hcert || f_lcert){
		if((egn=ExtGN_set_dns(fqdn))==NULL) goto done;
#endif
	}
	if(egn != NULL){
		if((et=Extnew_altname(OBJ_X509v3_SbjAltName,egn))==NULL) goto done;
		if((req->ext=Extnew_extreq(et))==NULL) goto done;
	}else if(reuse_ex){
		/* set subject alt name */
		if((req->ext=Extnew_extreq(reuse_ex))==NULL) goto done;
	}

	OK_set_cert_sig_algo(sigalgo);
	if((req->der = Req_toDER(req,*prv,NULL,&i))==NULL) goto done;
	ok = 1;
done:
	Key_free(pub);
	if (ok) {
		return req;
	} else {
		Req_free(req);
		return NULL;
	}
}

/*---------------------------------------------------------------*/
int req_op_csr(){
	Req *req=NULL;
	Key *prv=NULL;
	int ok=-1;
	/* No group information is needed for only creating a CSR. */
	strcpy(group, "NONE_NEEDED");

	/* check file permission. */
	if(check_file_permission(reqf)==0) goto done;
	if(check_file_permission(keyf)==0) goto done;

	/* create a new certificate request */
	if((req=create_csr(&prv, NULL, NULL))==NULL) goto done;

	/* save CSR */
	Req_print(req);
	if (save_csr(req) == 0) {
		goto done;
	}

	/* save private key */
	if (save_private_key(prv) == 0) {
		goto done;
	}

	ok = 0;
done:
	Req_free(req);
	Key_free(prv);
	return ok;
}

/*---------------------------------------------------------------*/
int req_op_issue(){
	PKCS12 *p12=NULL;
	Req *req=NULL;
	Cert *ct=NULL,*cact;
	Key *prv=NULL;
	LCMP *lc=NULL;
	int i,ok=-1;
	int ret = 0;

	if(*svpath == 0){
		printf("unknown server name. operation canceled.\n");
		goto done;
	}

	if(conf_parse_svpath(svpath,svpath,256,caname,32)!=CONF_PARSE_SVPATH_PARSE_SUCCESS) goto done;

	signal(SIGINT, SIG_IGN);
	signal(SIGHUP, SIG_IGN);
	signal(SIGTERM, SIG_IGN);

#if 0
	printf("--- req_op_issue() ---\n");
	printf("userid=\"%s\"\n", userid);
	printf("userpw=\"%s\"\n", userpw);
	printf(" licid=\"%s\"\n", licid);
	printf("   pin=\"%s\"\n", pin);
	printf("clctpw=\"%s\"\n", clctpw);
	printf("----------------------\n");
#endif

	/* check file permission, before connecting RA server. */
	if (*p12f) {
		if (check_file_permission(p12f) == 0) {
			goto done;
		}
	}

	/* create a new certification request or read a CSR file */
	if(*inreq){ /* read the CSR file */
		if (*p12f != 0) {
			if (*inkey == 0) {
				printf("a private key should be specified\n");
				goto done;
			} else if ((prv = Key_read_file(inkey)) == NULL) {
				printf("cannot read the private key : %s\n", inkey);
				goto done;
			}
		}
		if((req=Req_read_file(inreq))==NULL){
			printf("cannot read a CSR file : %s\n",inreq);
			goto done;
		}
		if(input_simple_subject2(req)) goto done;

#ifdef KEKGRIDCA
		if(AUTH_ISSET(f_login, CERTREQ_LOGIN_CHPN) && (*userpw==0)){
			printf("\nplease input your challenge pin or password to get a certificate\n");
			OK_get_passwd("Input Challenge PIN or Password : ",userpw,0);
		}
#endif
	}else{ /* create a CSR file */
		/* check file permission, before connecting RA server. */
#ifndef KEKGRIDCA
		if (check_file_permission(reqf) == 0) {
			goto done;
		}
#endif
		if (check_file_permission(keyf) == 0) {
			goto done;
		}

#ifndef KEKGRIDCA
		if(AUTH_ISSET(f_login, (CERTREQ_LOGIN_LICE|CERTREQ_LOGIN_CHPN)) &&
		   *pin==0){
			printf("\nplease input your challenge pin to get a certificate\n");
			OK_get_passwd("Input Challenge PIN : ",pin,0);
		}
#endif
		if((req=create_csr(&prv, NULL, NULL))==NULL) goto done;

#ifndef KEKGRIDCA
		/* save a CSR, before connecting RA server. */
		if (save_csr(req) == 0) {
			goto done;
		}
#endif
		/* save a private key, before connecting RA server. */
		if (save_private_key(prv) == 0) {
			goto done;
		}
	}

	/* get a new certificate from CA server */

	/* connect to the server and get a certificate */
	if((lc=LCMP_init(svpath,caport,caname))==NULL) goto done;
	if(usessl){
		if(LCMP_set_ssl(lc,ctstore,certid,clctpw,vfycert)) goto done;
		usessl = (*certid)?(LCMP_OPBRQ_ATSSLCL):(LCMP_OPBRQ_ATSIMPLE);
	}

	if(*licid){
		authid = licid;
		authpw = (*pin!=0) ? pin:NULL;
	}else if(*userid){
#ifndef KEKGRIDCA
		/* create "userid@group" for RAd_auth_pwd() [aica/airad_user.c] */
		if(*group){ strcat(userid,"@"); strncat(userid,group,30); }
#endif
		authid = userid;
		authpw = userpw;
	}

#if 0
	printf("--- req_op_issue() ---\n");
	printf("authid=\"%s\"\n", authid);
	printf("authpw=\"%s\"\n", authpw);
	printf("----------------------\n");
#endif
	printf("trying to connect RA server : %s (%d) ... ",svpath,caport); fflush(stdout);

	if((ret=LCMP_bind_s(lc,authid,authpw,usessl))){
		if (ret == -2) {
			print_protocol_version_mismatch();
		} else
		{
			printf("error!\n");
		}
		goto done;
	}

	printf("ok.\n");
	printf("request for issuing a new certificate ... "); 

	i = LCMP_sign_s(lc,group,0,req,PKCS10CSR);
	switch(i){
	case LCMP_SUCCESS:
		if((ct=LCMP_get_signcert(lc))==NULL){ printf("cannot get issued certificate!\n"); goto done; }
		printf("ok.\n");
		break;
	case LCMP_REQEST_ACCEPTED:
		printf("ok.\n\nyour request is accepted. (AcceptID=%.7d)\n",((LO_CSRRsp*)lc->op)->acceptID);
		printf("CA operator will send an email to tell a result.\n");
		i = 0;
		break;
	case LCMP_NOSUCH_PROFILE:
		printf("\n\ngroup name (%s) is not valid! please input again.\n",group);
		goto done;
	case LCMP_BAD_SUBJECTNAME:
		printf("\n\nuser subject name (%s) is already used! please input again.\n",req->subject);
		goto done;
	default:
		printf("LCMP error.\n");
		printf("Your new certificate may exist on the CA server.\n");
		goto done;
	}
	LCMP_unbind_s(lc);

	p12 = lc->ca->p12; lc->ca->p12 = NULL;
	cact= lc->ca->cert; lc->ca->cert = NULL; /* cact is not allocated memory */
	LCMP_free(lc); lc = NULL;

	/* save CA Cert */
	if(cact){
		printf("save a CA certificate file : %s\n",caf);
		if(type){
			if(ASN1_write_der(cact->der,caf)) goto done;
		}else{
			if(PEM_write_cert(cact,caf)) goto done;
		}
	}

	/* save Cert */
	if(ct &&(*p12f==0)){
		printf("save a certificate file : %s\n",crtf);
		if(type){
			if(ASN1_write_der(ct->der,crtf)) goto done;
		}else{
			if(PEM_write_cert(ct,crtf)) goto done;
		}
	}

	/* save p12 or private key */
	if(p12 && *p12f){
		/* save pkcs#12 file */
		printf("save a PKCS#12 file : %s\n",p12f);
		if(*pass) OK_set_passwd(pass);
		if(ct){
			if(P12_add_cert(p12,ct,NULL,0xff)) goto done;
			ct = NULL;
			if(P12_add_key(p12,prv,NULL,0xff)) goto done;
			prv = NULL;
			if(P12_check_chain(p12,0)) goto done;
			if(P12_write_file(p12,p12f)) goto done;

			if (*inreq == 0) { /* key from create_csr() */
				printf("remove the private key "
				       "(unnecessary) : %s\n", keyf);
				unlink(keyf);
			}
		}else{
			printf("Certificate is not found, save only the private key\n");
			goto done;
		}
	}

	ok = 0;
#ifndef KEKGRIDCA
	if (*inreq == 0) { /* CSR from create_csr() */
		printf("remove the CSR (unnecessary) : %s\n", reqf);
		unlink(reqf);
	}
#endif
done:
	if(lc && lc->op && lc->op->resultCode != LCMP_SUCCESS){
		printf("%s (%d) => %s\n",LCMP_msg2str(lc->op->resultCode),lc->op->resultCode,
		 lc->op->resultMsg);
	}
	if(lc){ if(lc->sock){ LCMP_unbind_s(lc); } LCMP_free(lc); }
	P12_free(p12);
	Cert_free(ct);
	Req_free(req);
	Key_free(prv);
	return ok;
}

/*---------------------------------------------------------------*/
int req_op_revoke(){
	LCMP *lc=NULL;
	Cert *ct=NULL;
	PKCS12 *p12=NULL;
	int sn=0,rs=0,ok = -1;
	char *sbj="";
	int ret = 0;

	if(*svpath == 0){
		printf("unknown server name. operation canceled.\n");
		goto done;
	}
	if(usessl && !( *clp12f || *certid || (*clcert && *clkey))){
		printf("SSL client certificate & key is not specified. operation canceled.\n");
		goto done;
	}

	/* read client certificate */
	if(*clp12f){
		OK_get_passwd("Input PKCS#12 Passwd:",clctpw,0);
		OK_set_passwd(clctpw);
		if((p12=P12_read_file(clp12f))==NULL) goto done;
		if((ct=P12_get_usercert(p12))==NULL) goto done;
		sn = ct->serialNumber; sbj = ct->subject;

	}else if(*clcert){
		if((ct=Cert_read_file(clcert))==NULL) goto done;
		sn = ct->serialNumber; sbj = ct->subject;
	}

#if 0
	if(op==5){ /* keep subject & subjectAltName for renew mode */
		if(Cert_dncopy(&ct->subject_dn,&tmpdn)) goto done;
		tmpex = CertExt_find(ct->ext,OBJ_X509v3_SbjAltName);
		if(tmpex) tmpex = CertExt_dup(tmpex); /* allocate new memory */
	}
#endif
	if(*clp12f) ct = NULL;

	/* get revocation info */
	if(*sbj){
		printf("-------------------------------------------\n");
		printf("  revoke a current user certificate\n");
		printf("-------------------------------------------\n");
		printf("Certificate DATA:\n");
		printf("    serial number : %d\n",sn);
		printf("    subject:\n    %s\n",sbj);
	}
	if(!f_noint)
		if(!req_ask_yesno("do you revoke this certificate ?",1)) goto done;

	if(!f_noint){
		printf("------\n");
		printf("Set revocation reason >>\n");
		printf("  unspecified(0), keyCompromise(1), cACompromise(2),\n");
		printf("  affiliationChanged(3), superseded(4), cessationOfOperation(5),\n");
		printf("  certificateHold(6), removeFromCRL(8), privilegeWithdrawn(9),\n");
		printf("  aaCompromise(10)\n");

		rs = req_ask_num("select reason code (-1 means 'cancel')",0);
		if(!((rs>=0)&&(rs<=10)&&(rs!=7))) goto done;
	}

	/* exec certificate revocation */
	if(conf_parse_svpath(svpath,svpath,256,caname,32)!=CONF_PARSE_SVPATH_PARSE_SUCCESS) goto done;

	if((lc=LCMP_init(svpath,caport,caname))==NULL) goto done;

	if(LCMP_set_ssl(lc,ctstore,NULL,clctpw,vfycert)) goto done;
	if(*certid){
		if ((lc->certid = strdup(certid)) == NULL) goto done;
	}else if(*clp12f){
		if ((lc->clcert = strdup(clp12f)) == NULL) goto done;
	}else if(*clcert && !anon){
		if ((lc->clcert = strdup(clcert)) == NULL) goto done;
		if ((lc->clkeyf = strdup(clkey)) == NULL) goto done;
	}

	printf("trying to connect RA server : %s (%d) \n",svpath,caport); fflush(stdout);

	if((ret=LCMP_bind_s(lc,NULL,NULL,LCMP_OPBRQ_ATSSLCL))) {
		if (ret == -2) {
			print_protocol_version_mismatch();
		}
		goto done;
	}

	printf("request for certificate revocation ... "); 

	if(LCMP_cert_s(lc,LCMP_OPCT_RVKCERT,sn,rs,NULL,NULL)){
		printf("cannot revoke a certificate on RA server.\n");
		goto done;
	}
	printf("ok\n");

	LCMP_unbind_s(lc);
	LCMP_free(lc); lc = NULL;

	printf("success to revoke a certificate (sn:%d)\n",sn);

	ok = 0;
done:
	if(lc && lc->op && lc->op->resultCode != LCMP_SUCCESS){
		printf("%s (%d) => %s\n",LCMP_msg2str(lc->op->resultCode),lc->op->resultCode,
		 lc->op->resultMsg);
	}
	if(lc){ if(lc->sock){ LCMP_unbind_s(lc); } LCMP_free(lc); }
	P12_free(p12);
	Cert_free(ct);
	OK_clear_passwd();
	return ok;
}

/*---------------------------------------------------------------*/
int req_op_renew(){
	LCMP *lc=NULL;
	Cert *ct=NULL;
	PKCS12 *p12=NULL;
	P12_Baggage *bg=NULL;
	int sn=0,i,ok = -1;
	struct tm *bf,*af;
	char *cp,*sbj="";
	int ret = 0;

	if(*svpath == 0){
		printf("unknown server name. operation canceled.\n");
		goto done;
	}
	if(usessl && !( *clp12f || *certid || (*clcert && *clkey))){
		printf("SSL client certificate & key is not specified. operation canceled.\n");
		goto done;
	}

	/* read client certificate */
	if(*clp12f){
		OK_get_passwd("Input PKCS#12 Passwd:",clctpw,0);
		OK_set_passwd(clctpw);
		if((p12=P12_read_file(clp12f))==NULL) goto done;
		if((ct=P12_get_usercert(p12))==NULL) goto done;
		sn = ct->serialNumber; sbj = ct->subject;
		bf = &ct->time.notBefore; af = &ct->time.notAfter; ct = NULL;

	}else if(*clcert){
		if((ct=Cert_read_file(clcert))==NULL) goto done;
		sn = ct->serialNumber; sbj = ct->subject;
		bf = &ct->time.notBefore; af = &ct->time.notAfter;
	}

	/* get update info */
	if(*sbj){
		printf("------\n");
		printf("Certificate DATA:\n");
		printf("    serial number : %d\n",sn);
		printf("    subject:\n    %s\n",sbj);
		printf("    Validity\n");
		if((cp=stm2str(bf,0))==NULL) goto done;
		printf("      Not Before: %s\n",cp);
		if((cp=stm2str(af,0))==NULL) goto done;
		printf("      Not After : %s\n",cp);
	}
	if(!f_noint)
		if(!req_ask_yesno("do you renew this certificate ?",1)) goto done;

	/* exec certificate renew */
	if(conf_parse_svpath(svpath,svpath,256,caname,32)!=CONF_PARSE_SVPATH_PARSE_SUCCESS) goto done;

	if((lc=LCMP_init(svpath,caport,caname))==NULL) goto done;

	if(LCMP_set_ssl(lc,ctstore,NULL,clctpw,vfycert)) goto done;
	if(*certid){
		if ((lc->certid = strdup(certid)) == NULL) goto done;
	}else if(*clp12f){
		if ((lc->clcert = strdup(clp12f)) == NULL) goto done;
	}else if(*clcert){
		if ((lc->clcert = strdup(clcert)) == NULL) goto done;
		if ((lc->clkeyf = strdup(clkey)) == NULL) goto done;
	}

	printf("trying to connect RA server : %s (%d) \n",svpath,caport); fflush(stdout);

	if((ret=LCMP_bind_s(lc,NULL,NULL,LCMP_OPBRQ_ATSSLCL))) {
		if (ret == -2) {
			print_protocol_version_mismatch();
		}
		goto done;
	}

	printf("request for renewing a certificate ... ");

	if(LCMP_cert_s(lc,LCMP_OPCT_UPDCERT,sn,0,NULL,NULL)){
		printf("cannot renew a certificate on RA server.\n");
		goto done;
	}

	Cert_free(ct);
	if((ct=LCMP_get_updcert(lc))==NULL){
		printf("cannot get a renewd certificate.\n");
		goto done;
	}
	printf("ok\n");

	LCMP_unbind_s(lc);
	LCMP_free(lc); lc = NULL;

	/* save pkcs#12 file */
	if(p12 && *p12f){
		printf("save a PKCS#12 file : %s\n",p12f);
		if(*pass) OK_set_passwd(pass);

		i = P12_max_depth(p12,OBJ_P12v1Bag_CERT);
		if((bg=P12_find_bag(p12,OBJ_P12v1Bag_CERT,(char)i))==NULL) goto done;
		Cert_free(((P12_CertBag*)bg)->cert);
		((P12_CertBag*)bg)->cert = ct; ct = NULL;

		if(P12_write_file(p12,p12f)) goto done;
	}
	/* save Cert */
	else if(ct){
		printf("save a certificate file : %s\n",crtf);
		if(type){
			if(ASN1_write_der(ct->der,crtf)) goto done;
		}else{
			if(PEM_write_cert(ct,crtf)) goto done;
		}
	}
	printf("success to renew a certificate (sn:%d)\n",sn);

	ok = 0;
done:
	if(lc && lc->op && lc->op->resultCode != LCMP_SUCCESS){
		printf("%s (%d) => %s\n",LCMP_msg2str(lc->op->resultCode),lc->op->resultCode,
		 lc->op->resultMsg);
	}
	if(lc){ if(lc->sock){ LCMP_unbind_s(lc); } LCMP_free(lc); }
	P12_free(p12);
	Cert_free(ct);
	return ok;
}

/*---------------------------------------------------------------*/
int req_op_getcert(){
	LCMP *lc=NULL;
	Cert *ct=NULL;
	int ok = -1;
	int ret = 0;

	if(*svpath == 0){
		printf("unknown server name. operation canceled.\n");
		goto done;
	}

	/* exec certificate update */
	if(conf_parse_svpath(svpath,svpath,256,caname,32)!=CONF_PARSE_SVPATH_PARSE_SUCCESS) goto done;

	if((lc=LCMP_init(svpath,caport,caname))==NULL) goto done;

	if(usessl){
		if(LCMP_set_ssl(lc,ctstore,certid,clctpw,vfycert)) goto done;
		usessl = (*certid)?(LCMP_OPBRQ_ATSSLCL):(LCMP_OPBRQ_ATSIMPLE);
	}
	printf("trying to connect RA server : %s (%d) \n",svpath,caport); fflush(stdout);

	if((ret=LCMP_bind_s(lc,"anonymous","non",usessl))) {
		if (ret == -2) {
			print_protocol_version_mismatch();
		}
		goto done;
	}

	printf("request for exporting a certificate ... "); 

	if(LCMP_cert_s(lc,LCMP_OPCT_EXPCERT,accid,0,NULL,NULL)){
		printf("cannot get a certificate from RA server.\n");
		goto done;
	}
	if((ct=LCMP_get_expcert(lc))==NULL){
		printf("cannot get a exported certificate.\n");
		goto done;
	}
	printf("ok\n");

	LCMP_unbind_s(lc);

	/* save CA Cert */
	if(lc->ca->cert){
		printf("save a CA certificate file : %s\n",caf);
		if(type){
			if(ASN1_write_der(lc->ca->cert->der,caf)) goto done;
		}else{
			if(PEM_write_cert(lc->ca->cert,caf)) goto done;
		}
	}
	LCMP_free(lc); lc = NULL;

	/* save Cert */
	if(ct){
		printf("save a certificate file : %s\n",crtf);
		if(type){
			if(ASN1_write_der(ct->der,crtf)) goto done;
		}else{
			if(PEM_write_cert(ct,crtf)) goto done;
		}
	}
	ok = 0;
done:
	if(lc && lc->op && lc->op->resultCode != LCMP_SUCCESS){
		printf("%s (%d) => %s\n",LCMP_msg2str(lc->op->resultCode),lc->op->resultCode,
		 lc->op->resultMsg);
	}
	if(lc){ if(lc->sock){ LCMP_unbind_s(lc); } LCMP_free(lc); }
	Cert_free(ct);
	return ok;
}

/*---------------------------------------------------------------*/
int req_op_rekey(){
	LCMP *lc=NULL;
	Cert *ct=NULL;
	Cert *cact;
	Req *req=NULL;
	Key *prv=NULL;
	PKCS12 *p12=NULL;
	int sn=0,i,ok = -1;
	struct tm *bf,*af;
	char *cp,*sbj="";
	CertDN reuse_dn;
	CertExt *reuse_ex = NULL;
	int ret = 0;

	cert_dn_init(&reuse_dn);

	if(*svpath == 0){
		printf("unknown server name. operation canceled.\n");
		goto done;
	}

	if(!( *clp12f || (*clcert && *clkey))){
		printf("SSL client certificate & key is not specified. operation canceled.\n");
		goto done;
	}

	if(*clp12f){
		OK_get_passwd("Input PKCS#12 Passwd:",clctpw,0);
		OK_set_passwd(clctpw);
		if((p12=P12_read_file(clp12f))==NULL) goto done;
		if((ct=P12_get_usercert(p12))==NULL) goto done;

	}else if(*clcert){
		if((ct=Cert_read_file(clcert))==NULL) goto done;
	}

	/* keep subject & subjectAltName before freeing ct */
	if(Cert_dncopy(&ct->subject_dn,&reuse_dn)) goto done;
	reuse_ex = CertExt_find(ct->ext,OBJ_X509v3_SbjAltName);
	if(reuse_ex){
		reuse_ex = CertExt_dup(reuse_ex); /* allocate new memory */
		if (reuse_ex==NULL) goto done;
	}
	sn = ct->serialNumber; sbj = ct->subject;
	bf = &ct->time.notBefore; af = &ct->time.notAfter;

	/* get re-key info */
	if(*sbj){
		printf("------\n");
		printf("Certificate DATA:\n");
		printf("    serial number : %d\n",sn);
		printf("    subject:\n    %s\n",sbj);
		printf("    Validity\n");
		if((cp=stm2str(bf,0))==NULL) goto done;
		printf("      Not Before: %s\n",cp);
		if((cp=stm2str(af,0))==NULL) goto done;
		printf("      Not After : %s\n",cp);
	}

	/*
	 * Get keytype and keysize from the client certificate
	 * if does not specified on the command line.
	 */
	if(!f_keysize){
		keysize=ct->pubkey->size*8;
	}
	if(!f_keytype){
		keytype=ct->pubkey->key_type;
	}
#if 0
	printf("Public Key:\n");
	printf("    keysize=%d\n",keysize);
	printf("    key_type=%d\n",keytype);
#endif
	if(*clp12f){
		ct = NULL;
	} else {
		Cert_free(ct);
		ct = NULL;
	}

	if(!f_noint)
		if(!req_ask_yesno("do you re-key this certificate ?",1)) goto done;

	if(conf_parse_svpath(svpath,svpath,256,caname,32)!=CONF_PARSE_SVPATH_PARSE_SUCCESS) goto done;

#if 0
	printf("--- req_op_rekey() ---\n");
	printf("userid=\"%s\"\n", userid);
	printf("userpw=\"%s\"\n", userpw);
	printf(" licid=\"%s\"\n", licid);
	printf("   pin=\"%s\"\n", pin);
	printf("clctpw=\"%s\"\n", clctpw);
	printf("----------------------\n");
#endif

	/* check file permission, before connecting RA server. */
	if (*p12f) {
		if (check_file_permission(p12f) == 0) {
			goto done;
		}
	}

	if(*inreq){ /* read csr file */
		if((req=Req_read_file(inreq))==NULL){
			printf("cannot read a CSR file : %s\n",inreq);
			goto done;
		}
		if(input_simple_subject2(req)) goto done;

	}else{ /* create csr file */
		/* check file permission, before connecting RA server. */
#ifndef KEKGRIDCA
		if (check_file_permission(reqf) == 0) {
			goto done;
		}
#endif
		if (check_file_permission(keyf) == 0) {
			goto done;
		}

#ifdef KEKGRIDCA
		if(AUTH_ISSET(f_login, CERTREQ_LOGIN_CHPN) && (*userpw==0)){
			printf("\nplease input your challenge pin or password to get a certificate\n");
			OK_get_passwd("Input Challenge PIN or Password : ",userpw,0);
#else
		if(AUTH_ISSET(f_login, (CERTREQ_LOGIN_LICE|CERTREQ_LOGIN_CHPN)) &&
		   *pin==0){
			printf("\nplease input your challenge pin to get a certificate\n");
			OK_get_passwd("Input Challenge PIN : ",pin,0);
#endif
		}

		if((req=create_csr(&prv, &reuse_dn, reuse_ex))==NULL) goto done;

#ifndef KEKGRIDCA
		/* save a CSR, before connecting RA server. */
		if (save_csr(req) == 0) {
			goto done;
		}
#endif
		/* save a private key, before connecting RA server. */
		if (save_private_key(prv) == 0) {
			goto done;
		}
	}

	/* get a new certificate from CA server */

	/* connect to the server and get a certificate */
	if((lc=LCMP_init(svpath,caport,caname))==NULL) goto done;

	if(LCMP_set_ssl(lc,ctstore,NULL,clctpw,vfycert)) goto done;
	if(*clp12f){
		if ((lc->clcert = strdup(clp12f)) == NULL) goto done;
	}else if(*clcert){
		if ((lc->clcert = strdup(clcert)) == NULL) goto done;
		if ((lc->clkeyf = strdup(clkey)) == NULL) goto done;
	}

	/*
	 * authmode priority:
	 * 1. LicenseID
	 * 2. ID/Password
	 * 3. LicenseID/Challenge PIN
	 */
	if((*licid) && (*pin==0)){
		authid = licid;
		/* authpw is null. */
	}else if(*userid){
#ifndef KEKGRIDCA
		/* create "userid@group" for RAd_auth_pwd() [aica/airad_user.c] */
		if(*group){ strcat(userid,"@"); strncat(userid,group,30); }
#endif
		authid = userid;
		authpw = userpw;
	}else if((*licid) && (*pin)){
		authid = licid;
		authpw = pin;
	}
	/* else anonymous mode. */

	printf("trying to connect RA server : %s (%d) \n",svpath,caport); fflush(stdout);

	if((ret=LCMP_bind_s(lc,authid,authpw,LCMP_OPBRQ_ATSIMPLE))){
	    printf("Simple bind failed.\n");
	    if (ret == -2) {
		print_protocol_version_mismatch();
	    }
	    fflush(stdout);
	    goto done;
	}

	if((ret=LCMP_bind_s(lc,userid,userpw,LCMP_OPBRQ_ATSSLCL))){
	    printf("SSL client authentication failed.\n");
	    if (ret == -2) {
		print_protocol_version_mismatch();
	    }
	    fflush(stdout);
	    goto done;
	}

	printf("request for issuing a re-keyed certificate ... "); fflush(stdout);

	i = LCMP_renewal_s(lc,sn,req,PKCS10CSR);

	switch(i){
	case LCMP_SUCCESS:
		if((ct=LCMP_get_renewalcert(lc))==NULL){
			printf("cannot get re-keyed certificate!\n");
			goto done;
		}
		printf("ok.\n");
		break;
	default:
		printf("LCMP error.\n");
		printf("Your new certificate may exist on the CA server.\n");
		printf("Your old certificate may be revoked.\n");
		goto done;
	}

	LCMP_unbind_s(lc);

	p12 = lc->ca->p12; lc->ca->p12 = NULL;
	cact= lc->ca->cert; lc->ca->cert = NULL; /* cact is not allocated memory */
	LCMP_free(lc); lc = NULL;

	/* save CA Cert */
	if(cact){
		printf("save a CA certificate file : %s\n",caf);
		if(type){
			if(ASN1_write_der(cact->der,caf)) goto done;
		}else{
			if(PEM_write_cert(cact,caf)) goto done;
		}
	}

	/* save Cert */
	if(ct &&(*p12f==0)){
		printf("save a certificate file : %s\n",crtf);
		if(type){
			if(ASN1_write_der(ct->der,crtf)) goto done;
		}else{
			if(PEM_write_cert(ct,crtf)) goto done;
		}
	}

	/* save p12 or private key */
	if(p12 && *p12f){
		/* save pkcs#12 file */
		printf("save a PKCS#12 file : %s\n",p12f);
		if(*pass) OK_set_passwd(pass);
		if(ct){
			if(P12_add_cert(p12,ct,NULL,0xff)) goto done;
			ct = NULL;
			if(P12_add_key(p12,prv,NULL,0xff)) goto done;
			prv = NULL;
			if(P12_check_chain(p12,0)) goto done;
			if(P12_write_file(p12,p12f)) goto done;

			if (*inreq == 0) { /* key from create_csr() */
				printf("remove the private key "
				       "(unnecessary) : %s\n", keyf);
				unlink(keyf); /* unnecessary anymore */
			}
		}else{
			/* unexpected */
			printf("Certificate is not found, save only the private key\n");
			goto done;
		}
	}

	ok = 0;
	if (*inreq == 0) { /* CSR from create_csr() */
		printf("remove the CSR (unnecessary) : %s\n", reqf);
		unlink(reqf); /* unnecessary anymore */
	}
done:
	if(lc && lc->op && lc->op->resultCode != LCMP_SUCCESS){
		printf("%s (%d) => %s\n",LCMP_msg2str(lc->op->resultCode),
			   lc->op->resultCode,lc->op->resultMsg);
	}
	if(lc){ if(lc->sock){ LCMP_unbind_s(lc); } LCMP_free(lc); }
	cert_dn_free(&reuse_dn);
	CertExt_free(reuse_ex);
	return ok;
}

/*---------------------------------------------------------------*/
void print_protocol_version_mismatch(void){
#if defined(HAVE_LIBNRGTLS) || defined(HAVE_LIBAISSL)
	printf(
"\n"
#ifdef HAVE_LIBAISSL
"Airad you tried to connect doesn't supports SSL 3.0,\n"
"but certreq supports only SSL 3.0.\n"
"\n"
"PLEASE REBUILD NAREGI-CA USING `configure --without-aissl'.\n"
#else
"Airad you tried to connect supports only SSL 3.0,\n"
"but certreq doesn't support SSL 3.0.\n"
"\n"
"PLEASE USE certreq-aissl INSTEAD.\n"
#endif /* HAVE_LIBAISSL */
		);
#endif /* HAVE_LIBNRGTLS || HAVE_LIBAISSL */
}
