/* certconv.c */
/*
 * Copyright (c) 2004-2018 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 *  Akira Iwata Laboratory. 
 *  Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usagi@mars.elcom.nitech.ac.jp)
 * And if you want to contact us, send e-mail to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * This library is free for commercial and non-commercial use as long as
 * the following conditions are aheared to.
 * If you want to use aicrypto library and CA applications code in product,
 * should be e-mail to Akira Iwata Laboratory (wakayama@elcom...).
 * 
 * Please note that MD2 and MD5 include RSA Data Security, Inc. LICENSE.
 * Those are besed on RFC1319 and RFC1321 document. And copyright distribution
 * is following in ok_md2.h ok_md5.h .
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <string.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_pem.h>
#include <aicrypto/ok_pkcs.h>
#include <aicrypto/ok_tool.h>

/* common/version.c */
void print_version(char *argv[]);

static void usage(void);
static void options(int argc, char **argv);
PKCS12 *read_files(void);
int write_files(PKCS12 *p12);

char pwdout[PWD_BUFLEN] = "";
char pwdin[PWD_BUFLEN] = "";
char *in[8];

char *cert;
char *key;
char *crl;
char *crtp;
char *p7b;
char *p12f;
char *p10;
char *p8;

int type;			/* 0.. pem, 1.. x509(DER) */
int noenc;
bool force_rfc5915 = false;
unsigned char depth;

int main(int argc, char **argv)
{
	PKCS12 *p12 = NULL;
	int err = -1;

	cert = key = crl = p7b = p12f = p10 = p8 = NULL;
	noenc = type = 0;
	depth = 0xff;

	if (argc == 1) {
		usage();
		exit(EXIT_FAILURE);
	}

	options(argc, argv);

	if (*pwdin)
		OK_set_passwd(pwdin);
	if ((p12 = read_files()) == NULL)
		goto done;

	if (P12_check_chain(p12, 1))
		goto done;

	if (depth == 0xff)
		depth = P12_max_depth(p12, OBJ_P12v1Bag_CERT);

	if (*pwdout)
		OK_set_passwd(pwdout);
	if (write_files(p12))
		goto done;
	err = 0;

done:
	P12_free(p12);
	return err;
}

static void usage(void)
{
	printf("\
Usage: certconv [OPTION...] FILE...\n\
\n\
Options:\n\
  -p12 FILE	convert inputs to FILE in PKCS#12\n\
  -p10 FILE	convert input to FILE in PKCS#10\n\
  -p8 FILE	convert input to FILE in PKCS#8\n\
  -p7b FILE	convert inputs to FILE in PKCS#7\n\
  -cert FILE	output certificate to FILE\n\
  -crl FILE	output CRL to FILE\n\
  -key FILE	output private key to FILE\n\
  -crtp FILE	convert input certificates to FILE in crossCertificatePair\n\
\n\
  -depth NUM	set chain depth for output (0..top)\n\
  -pem		set output in PEM format [default]\n\
  -der		set output in DER format\n\
  -noenc	save private key with no encryption\n\
  -pwin PASSWD	set password for input file\n\
  -pwout PASSWD	set password for output file\n\
  -rfc5915	convert ECDSA private key into one compatible with RFC5915\n\
  -help		print this message\n\
  -version	print version information and exit\n\
");
}

static void options(int argc, char **argv)
{
	int i, j;

	for (i = 1, j = 0; i < argc; i++) {
		if (!strcmp(argv[i], "-pem")) {
			type = 0;
		} else if (!strcmp(argv[i], "-der")) {
			type = 1;
		} else if (!strcmp(argv[i], "-p12")) {
			i++;
			if (i < argc)
				p12f = strdup(argv[i]);
		} else if (!strcmp(argv[i], "-p7b")) {
			i++;
			if (i < argc)
				p7b = strdup(argv[i]);
		} else if (!strcmp(argv[i], "-key")) {
			i++;
			if (i < argc)
				key = strdup(argv[i]);
		} else if (!strcmp(argv[i], "-cert")) {
			i++;
			if (i < argc)
				cert = strdup(argv[i]);
		} else if (!strcmp(argv[i], "-crl")) {
			i++;
			if (i < argc)
				crl = strdup(argv[i]);
		} else if (!strcmp(argv[i], "-p10")) {
			i++;
			if (i < argc)
				p10 = strdup(argv[i]);
		} else if (!strcmp(argv[i], "-crtp")) {
			i++;
			if (i < argc)
				crtp = strdup(argv[i]);
		} else if (!strcmp(argv[i], "-p8")) {
			i++;
			if (i < argc)
				p8 = strdup(argv[i]);
		} else if (!strcmp(argv[i], "-depth")) {
			i++;
			if (i < argc)
				depth = atoi(argv[i]);
		} else if (!strcmp(argv[i], "-pwin")) {
			i++;
			if (i < argc)
				strncpy(pwdin, argv[i], PWD_BUFLEN);
		} else if (!strcmp(argv[i], "-pwout")) {
			i++;
			if (i < argc)
				strncpy(pwdout, argv[i], PWD_BUFLEN);
		} else if (!strcmp(argv[i], "-noenc")) {
			noenc++;
			OK_set_pem_cry_algo(0);	/* don't encrypto PEM */
		} else if (!strcmp(argv[i], "-version")) {
			print_version(argv);
			exit(EXIT_SUCCESS);
		} else if (strcmp(argv[i], "-rfc5915") == 0) {
			force_rfc5915 = true;
		} else if (!strcmp(argv[i], "-help")) {
			usage();
			exit(EXIT_SUCCESS);

		} else if (argv[i][0] == '-') {
			printf("option error!\n");
			printf("unknown option: `%s'\n", argv[i]);
			usage();
			exit(EXIT_FAILURE);
		} else {
			in[j] = strdup(argv[i]);
			j++;
		}
	}

	if (!j) {
		usage();
		exit(EXIT_FAILURE);
	}
}

PKCS12 *read_files(void)
{
	unsigned char *der;	/* XXX: should initialize? */
	Cert *ct;
	CRL *crl;
	Key *key;
	CertPair *ctp;
	PKCS12 *p12, *ret;
	PKCS7 *p7;
	int i;

	if ((ret = P12_new()) == NULL)
		return NULL;

	for (i = 0; in[i] != NULL; i++) {
		if (fopen(in[i], "r") == NULL)
			continue;

		if ((ct = Cert_read_file(in[i])) != NULL) {
			P12_add_cert(ret, ct, NULL, 0xff);
		}
		if ((ct = Req_read_file(in[i])) != NULL) {
			P12_add_cert(ret, ct, NULL, 0xff);
		}
		if ((crl = CRL_read_file(in[i])) != NULL) {
			P12_add_crl(ret, crl, NULL, 0xff);
		}
		if ((ctp = CertPair_read_file(in[i])) != NULL) {
			if (ctp->issuedToThisCA)
				P12_add_cert(ret, ctp->issuedToThisCA, NULL,
					     0xff);
			if (ctp->issuedByThisCA)
				P12_add_cert(ret, ctp->issuedByThisCA, NULL,
					     0xff);

			ctp->issuedToThisCA = ctp->issuedByThisCA = NULL;
			CertPair_free(ctp);
		}
		if ((key = (Key *) PEM_read_rsaprv(in[i])) != NULL) {
			P12_add_key(ret, key, NULL, 0xff);
		}
		if ((key = (Key *) PEM_read_dsaprv(in[i])) != NULL) {
			P12_add_key(ret, key, NULL, 0xff);
		}
		if ((key = (Key *) PEM_read_ecdsaprv(in[i])) != NULL) {
			P12_add_key(ret, key, NULL, 0xff);
		}
		if ((key = (Key *) P8_read_file(in[i])) != NULL) {
			P12_add_key(ret, key, NULL, 0xff);
		}
		if ((key = (Key *) P8enc_read_file(in[i])) != NULL) {
			P12_add_key(ret, key, NULL, 0xff);
		}
		if ((p7 = P7b_read_file(in[i])) != NULL) {
			P12_mov_p12bags(ret, (PKCS12 *) p7);
			P7_free(p7);
		}
		if ((p12 = P12_read_file(in[i])) != NULL) {
			P12_mov_p12bags(ret, p12);
			P12_free(p12);
		}

		if ((der = ASN1_read_der(in[i])) != NULL) {
			if ((key = (Key *) ASN1_read_rsaprv(in[i])) != NULL) {
				P12_add_key(ret, key, NULL, 0xff);
			}
			if ((key = (Key *) ASN1_read_dsaprv(in[i])) != NULL) {
				P12_add_key(ret, key, NULL, 0xff);
			}
			if ((key = (Key *) ASN1_read_ecdsaprv(in[i])) != NULL) {
				P12_add_key(ret, key, NULL, 0xff);
			}
		}
	}
	return (ret);
}

int write_files(PKCS12 *p12)
{
	P12_Baggage *bg, *bg2;
	int out = 0;

	if (cert) {
		if ((bg = P12_find_bag(p12, OBJ_P12v1Bag_CERT, depth)) == NULL)
			goto error;

		if (type)
			ASN1_write_der(((P12_CertBag *) bg)->cert->der, cert);
		else
			PEM_write_cert(((P12_CertBag *) bg)->cert, cert);
		out++;
	}
	if (key) {
#define p12_kybg ((P12_KeyBag *)bg)

		if ((bg = P12_find_bag(p12, OBJ_P12v1Bag_PKCS8, depth)) == NULL)
			goto error;

		if (type || noenc)
			OK_set_pem_cry_algo(0);
		else
			printf("please input password for output key.\n");

		switch (p12_kybg->key->key_type) {
		case KEY_RSA_PRV:
#define rsa_prv ((Prvkey_RSA *)(p12_kybg->key))

			if (type)
				ASN1_write_der(rsa_prv->der, key);
			else
				PEM_write_rsaprv(rsa_prv, key);
			break;

		case KEY_DSA_PRV:
#define dsa_prv ((Prvkey_DSA *)(p12_kybg->key))

			if (type)
				ASN1_write_der(dsa_prv->der, key);
			else
				PEM_write_dsaprv(dsa_prv, key);
			break;

		case KEY_ECDSA_PRV:
#define ecdsa_prv ((Prvkey_ECDSA *)(p12_kybg->key))

			if (force_rfc5915) {
				int dummy;
				if (ecdsa_prv->der != NULL)
					free(ecdsa_prv->der);
				ecdsa_prv->der = ECDSAprv_toDER(ecdsa_prv, NULL, &dummy);
			}
			if (type)
				ASN1_write_der(ecdsa_prv->der, key);
			else
				PEM_write_ecdsaprv(ecdsa_prv, key);
			break;
		}
		out++;
	}

	if (crl) {
		if ((bg = P12_find_bag(p12, OBJ_P12v1Bag_CRL, depth)) == NULL)
			goto error;

		if (type)
			ASN1_write_der(((P12_CRLBag *) bg)->crl->der, crl);
		else
			PEM_write_crl(((P12_CRLBag *) bg)->crl, crl);
		out++;
	}
	if (p10) {
		if ((bg = P12_find_bag(p12, OBJ_P12v1Bag_CERT, depth)) == NULL)
			goto error;

		Cert_print(((P12_CertBag *) bg)->cert);
		if (type)
			ASN1_write_der(((P12_CertBag *) bg)->cert->der, p10);
		else
			PEM_write_req(((P12_CertBag *) bg)->cert, p10);
		out++;
	}
	if (crtp) {
		bg = P12_find_bag(p12, OBJ_P12v1Bag_CERT, 0);
		bg2 = P12_find_bag(p12, OBJ_P12v1Bag_CERT, 1);

		if ((bg) && (bg2)) {
			int i;
			CertPair *tmp;

			if ((tmp = CertPair_new()) == NULL)
				goto error;

			tmp->issuedToThisCA = ((P12_CertBag *) bg)->cert;
			tmp->issuedByThisCA = ((P12_CertBag *) bg2)->cert;

			if ((tmp->der = CertPair_toDER(tmp, NULL, &i)) == NULL)
				goto error;

			if (type)
				ASN1_write_der(tmp->der, crtp);
			else
				PEM_write_crtp(tmp, crtp);

			tmp->issuedToThisCA = tmp->issuedByThisCA = NULL;
			CertPair_free(tmp);
			out++;
		}
	}
	if (p7b) {
		if (type)
			P7b_write_file((PKCS7 *) p12, p7b);
		else
			PEM_write_p7((PKCS7 *) p12, p7b);
		out++;
	}
	if (p8) {
		if ((bg = P12_find_bag(p12, OBJ_P12v1Bag_PKCS8, depth)) == NULL)
			goto error;

		if (noenc) {
			if (type)
				P8_write_file(((P12_KeyBag *) bg)->key, p8);
			else
				PEM_write_p8(((P12_KeyBag *) bg)->key, p8);
		} else {
			printf("please input password for output key.\n");
			if (type)
				P8enc_write_file(((P12_KeyBag *) bg)->key, p8);
			else
				PEM_write_p8enc(((P12_KeyBag *) bg)->key, p8);
		}
		out++;
	}
	if (p12f) {
		if ((P12_find_bag(p12, OBJ_P12v1Bag_PKCS8, depth) == NULL)
		    || (P12_find_bag(p12, OBJ_P12v1Bag_CERT, depth) == NULL))
			goto error;

		/* if only client certificate & key are included, local key id in both
		 * bag is set "0". but OpenSSL recognizes that "0" key id certificate
		 * is CA certificate but not client certificate. this might be a problem.
		 * set "1" local key id for compatibility.
		 */
		if (P12_max_depth(p12, OBJ_P12v1Bag_CERT) == 0) {
			P12_Baggage *bg;

			for (bg = p12->bag; bg; bg = bg->next)
				bg->localKeyID[0] = 1;
		}

		P12_write_file(p12, p12f);
		out++;
	}

	if (out)
		printf("output a file ..\n");
	return 0;
error:
	printf("bad depth number ..\n");
	return -1;
}
