/* aistore.c */
/*
 * Copyright (c) 2004-2015 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_pem.h>
#include <aicrypto/ok_store.h>
#include <aicrypto/ok_tool.h>

#include "ok_conf.h"

#ifndef AICONFIG
#define AICONFIG "aica.cnf"
#endif

/* global value */
char conf[CONF_PATH_MAX] = AICONFIG;
char ctstore[CONF_PATH_MAX] = "/";
char pass[PWD_BUFLEN] = "";
int new = 0;
int del = 0;
int print = 0;
int expfl = 0;
int so = -1;
int type = CSTORE_CTX_CERT;
char *sto = STORE_MY;
char *id = NULL;
char *imp = NULL;
char *expt = NULL;

static void options(int argc, char **argv);
static void usage(void);

int ai_list_proc(STManager *stm, char *store, int type);
int ai_print_proc(STManager *stm, char *store, int type, char *id);
int ai_delete_proc(STManager *stm, char *store, int type, char *id);
int del_pkey_bag(STManager *stm, CSBag *cbg);
int ai_import_proc(STManager *stm, char *fname);
int update_store_p12(STManager *stm, PKCS12 *p12, char *uid);
int ai_export_proc(STManager *stm, char *store, int type, char *id,
		   char *fname, int format);
Key *get_pkey_bag(STManager *stm, CSBag *cbg);

/* store_conf.c */
int store_config(char *fname);

/* common/version.c */
void print_version(char *argv[]);

/*------------------------------------
  main
------------------------------------*/
int main(int argc, char **argv)
{
	STManager *stm = NULL;
	int ret = EXIT_FAILURE;

	OK_clear_error();
	options(argc, argv);

	if (store_config(conf)) {
		printf("cannot find \"store_dir\" in the file:\n%s\n", conf);
		goto done;
	}

	/* initialize */
	if (new) {
		if ((stm = STM_system_new(ctstore)) == NULL) {
			printf("cannot initialize certificate store : %s\n%s\n",
			       ctstore, OK_get_errstr());
			goto done;
		} else {
			ret = EXIT_SUCCESS;
			goto done;
		}
	}

	/* open */
	if ((stm = STM_open(ctstore)) == NULL) {
		printf("cannot open certificate store : %s\n%s\n", ctstore,
		       OK_get_errstr());
		goto done;
	}

	/* do operation */
	if (print) {
		if (ai_print_proc(stm, sto, type, id)) {
			printf
			    ("cannot print a store data : store=%s,type=%s,id=%s\n",
			     sto,
			     (type == CSTORE_CTX_CERT) ? ("CERT") : ("other"),
			     (id) ? (id) : ("NULL"));
			goto done;
		}
		printf("print a store data successfully.\n");
	} else if (imp) {
		if (ai_import_proc(stm, imp)) {
			printf("cannot import a file to the store : %s\n%s\n",
			       imp, OK_get_errstr());
			goto done;
		}
		printf("import a file to the store successfully.\n");
	} else if (expt) {
		if (ai_export_proc(stm, sto, type, id, expt, expfl)) {
			printf
			    ("cannot export a store data : store=%s,type=%s,id=%s : %s\n%s\n",
			     sto,
			     (type == CSTORE_CTX_CERT) ? ("CERT") : ("other"),
			     (id) ? (id) : ("NULL"), expt, OK_get_errstr());
			goto done;
		}
		printf("export a store data successfully.\n");
	} else if (del) {
		if (ai_delete_proc(stm, sto, type, id)) {
			printf
			    ("cannot delete a store data : store=%s,type=%s,id=%s\n%s\n",
			     sto,
			     (type == CSTORE_CTX_CERT) ? ("CERT") : ("other"),
			     (id) ? (id) : ("NULL"), OK_get_errstr());
			goto done;
		}
		printf("delete a store data successfully.\n");
	} else {		/* list store */
		if (ai_list_proc(stm, sto, type)) {
			printf
			    ("cannot list a certificate store : store=%s,type=%s\n%s\n",
			     sto,
			     (type == CSTORE_CTX_CERT) ? ("CERT") : ("other"),
			     OK_get_errstr());
			goto done;
		}
		if (so < 0)
			printf("list a certificate store successfully.\n");
	}
	ret = EXIT_SUCCESS;
done:
	if (stm)
		STM_close(stm);
	if (imp)
		free(imp);
	if (expt)
		free(expt);
	if (id)
		free(id);
	return ret;
}

/*-----------------------------------------
  usage and option check
-----------------------------------------*/
static void options(int argc, char **argv)
{
	int i;

	for (i = 1; i < argc; i++) {
		if (!strcmp("-conf", argv[i])) {
			i++;
			if (i < argc)
				strncpy(conf, argv[i], 254);
		} else if (!strcmp("-new", argv[i])) {
			new = 1;
		} else if (!strcmp("-l", argv[i])) {
		} else if (!strcmp("-p", argv[i])) {
			print = 1;
		} else if (!strcmp("-d", argv[i])) {
			del = 1;
		} else if (!strcmp("-so", argv[i])) {
			/* simple ID name output -- internal mode */
			i++;
			if (i < argc)
				so = atoi(argv[i]);
		} else if (!strcmp("-i", argv[i])) {
			i++;
			if (i < argc)
				imp = strdup(argv[i]);
		} else if (!strcmp("-e", argv[i])) {
			i++;
			if (i < argc)
				expt = strdup(argv[i]);
		} else if (!strcmp("-st", argv[i])) {
			i++;
			if (i < argc) {
				if (!strcmp("root", argv[i]))
					sto = STORE_ROOT;
				else if (!strcmp("sub", argv[i]))
					sto = STORE_MIDCA;
				else if (!strcmp("other", argv[i]))
					sto = STORE_OTHER;
			}
		} else if (!strcmp("-tp", argv[i])) {
			i++;
			if (i < argc) {
				if (!strcmp("crl", argv[i]))
					type = CSTORE_CTX_CRL;
				else if (!strcmp("key", argv[i]))
					type = CSTORE_CTX_KEY;
				else if (!strcmp("req", argv[i]))
					type = CSTORE_CTX_CSR;
			}
		} else if (!strcmp("-id", argv[i])) {
			i++;
			if (i < argc)
				id = strdup(argv[i]);
		} else if (!strcmp("-ef", argv[i])) {
			i++;
			if (i < argc) {
				if (!strcmp("pem", argv[i]))
					expfl = 1;
				else if (!strcmp("pk7", argv[i]))
					expfl = 2;
				else if (!strcmp("pk12", argv[i]))
					expfl = 3;
			}
		} else if (!strcmp("-pw", argv[i])) {
			i++;
			if (i < argc)
				strcpy(pass, argv[i]); /* XXX: check size! */
		} else if (!strcmp("-version", argv[i])) {
			print_version(argv);
			exit(EXIT_SUCCESS);
		} else if (!strcmp("-help", argv[i])) {
			usage();
			exit(EXIT_SUCCESS);
		} else {
			printf("option error!\n");
			printf("unknown option: `%s'\n", argv[i]);
			usage();
			exit(EXIT_FAILURE);
		}
	}
}

static void usage(void)
{
	printf("\
Usage: aistore [COMMAND] [OPTION...]\n\
\n\
Commands:\n\
  -new		initialize certificate store\n\
  -l		list a store data [default]\n\
  -i FILE	import a file to the store\n\
  -e FILE	export a certificate or a CRL from the store\n\
  -p		print a store data specified with store name & id\n\
  -d		delete a store data specified with store name & id\n\
\n\
Options:\n\
  -st STORE	specify store name [my(default),other,sub,root]\n\
  -tp TYPE	specify store type [cert(default),crl]\n\
  -id ID	specify store data id\n\
  -ef FORMAT	export with FORMAT [der(default),pem,pk7,pk12]\n\
  -pw PASSWORD	set password for key store data\n\
  -conf PATH	set the path for an aica configuration file\n\
  -help		print this message\n\
  -version	print version information and exit\n\
");
}

/*-----------------------------------------
  list proc
-----------------------------------------*/
int ai_list_proc(STManager *stm, char *store, int type)
{
	CStore *cs;
	CSBag *bg;
	Cert *ct;
	CRL *crl;
	int i;

	if ((cs = STM_find_byName(stm, store, CSTORE_ON_STORAGE, type)) == NULL)
		return -1;

	/* simple ID output mode */
	if (so >= 0) {
		for (i = 0, bg = cs->bags; bg; i++, bg = bg->next)
			if (i == so)
				printf("%s", bg->unique_id);
		return 0;
	}

	/* normal list mode */
	switch (type) {
	case CSTORE_CTX_CERT:
		printf("[unique-id]\t\tsubject\t\t\tserialNumber\n");
		break;
	case CSTORE_CTX_CRL:
		printf("[unique-id]\t\tissuer\n");
		break;
	}
	printf
	    ("----------------------------------------------------------------------\n");

	/* set items into list control */
	for (i = 0, bg = cs->bags; bg; i++, bg = bg->next) {
		if (bg->ctx_type == CSTORE_CTX_CERT) {
			if ((ct = (Cert *) bg->cache) == NULL)
				continue;	/* should be cached */

			printf("[");
			printf("%s", bg->unique_id);
			printf("]  ");
			printf("%s", ct->subject);
			printf("  %d\n", ct->serialNumber);
		} else if (bg->ctx_type == CSTORE_CTX_CRL) {
			if ((crl = (CRL *) bg->cache) == NULL)
				continue;	/* should be cached */
			printf("[");
			printf("%s", bg->unique_id);
			printf("]  ");
			printf("%s", crl->issuer);
			printf("\n");
		} else {
			printf("[");
			printf("%s", bg->unique_id);
			printf("]\n");
		}
	}
	return 0;
}

/*-----------------------------------------
  print proc
-----------------------------------------*/
int ai_print_proc(STManager *stm, char *store, int type, char *id)
{
	CSBag *bg;

	if (id == NULL)
		goto error;
	if ((bg =
	     STM_find_byID(stm, store, CSTORE_ON_STORAGE, type, id)) == NULL)
		goto error;

	switch (type) {
	case CSTORE_CTX_CERT:
		Cert_print((Cert *) bg->cache);
		break;
	case CSTORE_CTX_CRL:
		CRL_print((CRL *) bg->cache);
		break;
	}
	return 0;
error:
	return 1;
}

/*-----------------------------------------
  import proc
-----------------------------------------*/
int ai_import_proc(STManager *stm, char *fname)
{
	Cert *ct = NULL;
	CRL *crl = NULL;
	Key *key = NULL;
	PKCS7 *p7b = NULL;
	PKCS12 *p12 = NULL;
	int ok = -1;

	OK_clear_passwd();

	/* check input file */
	if ((ct = Cert_read_file(fname)) != NULL) {
	} else if ((crl = CRL_read_file(fname)) != NULL) {
	} else if ((p7b = P7b_read_file(fname)) != NULL) {
	} else if ((p12 = P12_read_file(fname)) != NULL) {
	} else {
		OK_clear_error();
		goto done;
	}

	OK_clear_error();
	if (STM_reload(stm))
		goto done;

	/* update cert store */
	if (ct) {
		if (STM_import_cert(stm, ct, id))
			goto done;

	} else if (crl) {
		if (STM_import_crl(stm, crl, id))
			goto done;

	} else if (p7b) {
		if (update_store_p12(stm, (PKCS12 *) p7b, NULL))
			goto done;

	} else if (p12) {
		if (P12_check_chain(p12, 0))
			goto done;
		if ((ct = P12_get_usercert(p12)) == NULL)
			goto done;
		if ((key = P12_get_privatekey(p12)) == NULL)
			goto done;

		if (pass[0] == '\0') {
			OK_get_passwd("Save Access Password : ", pass, M_VRFY);
		}
		OK_set_passwd(pass);
		if (STM_import_certkey(stm, ct, key, id))
			goto done;

		if (update_store_p12(stm, p12, id))
			goto done;
	}
	if (STM_update(stm))
		goto done;

	ok = 0;
done:
	OK_clear_passwd();
	return ok;
}

int update_store_p12(STManager *stm, PKCS12 *p12, char *uid)
{
	P12_Baggage *bg;

	/* if same certificate or CRL is already in the Store
	 * ignore to install such pkcs12 bag.
	 */
	for (bg = p12->bag; bg; bg = bg->next) {
		if (bg->type == OBJ_P12v1Bag_CERT) {
			if (STM_find_byCert(stm, ((P12_CertBag *) bg)->cert) ==
			    NULL) {
				/* ignore it's error */
				STM_import_cert(stm, ((P12_CertBag *) bg)->cert,
						uid);
			}
		} else if (bg->type == OBJ_P12v1Bag_CRL) {
			if (STM_find_byCRL(stm, ((P12_CRLBag *) bg)->crl) ==
			    NULL) {
				/* ignore it's error */
				STM_import_crl(stm, ((P12_CRLBag *) bg)->crl,
					       uid);
			}
		}
	}
	return 0;
}

/*-----------------------------------------
  export proc
-----------------------------------------*/
int ai_export_proc(STManager *stm, char *store, int type, char *id,
		   char *fname, int format)
{
	CertList *cl, *top = NULL;
	CStore *cs;
	CSBag *bg = NULL;
	PKCS12 *p12 = NULL;
	Key *key = NULL;

	if (id == NULL)
		goto error;
	if (STM_reload(stm))
		goto error;

	/* get store */
	if ((cs = STM_find_byName(stm, store, CSTORE_ON_STORAGE, type)) == NULL)
		goto error;

	if ((bg = CStore_find_byID(cs->bags, id)) == NULL)
		return 1;

	if (format == 3) {	/* output PKCS#12 */
		if (bg->cache == NULL)
			goto error;
		if (strcmp(store, STORE_MY))
			goto error;

		if ((p12 = P12_new()) == NULL)
			goto error;
		if ((top = STM_get_pathcert(stm, (Cert *) bg->cache)) == NULL)
			goto error;

		for (cl = top; cl; cl = cl->next) {
			if (P12_add_cert(p12, cl->cert, NULL, 0xff))
				goto error;
			cl->cert = NULL;
		}

		/* get key */
		if ((key = get_pkey_bag(stm, bg)) == NULL)
			goto error;
		if (P12_add_key(p12, key, NULL, 0xff))
			goto error;

		if (P12_check_chain(p12, 0))
			goto error;
		if (P12_write_file(p12, fname))
			goto error;
		OK_clear_passwd();

		P12_free(p12);
		Certlist_free_all(top);

	} else if (format == 2) {	/* output PKCS#7 */
		if (bg->cache == NULL)
			goto error;

		if ((p12 = P12_new()) == NULL)
			goto error;
		if ((top = STM_get_pathcert(stm, (Cert *) bg->cache)) == NULL)
			goto error;

		for (cl = top; cl; cl = cl->next) {
			if (P12_add_cert(p12, cl->cert, NULL, 0xff))
				goto error;
			cl->cert = NULL;
		}

		if (P12_check_chain(p12, 0))
			goto error;
		if (P7b_write_file((PKCS7 *) p12, fname))
			goto error;

		P12_free(p12);
		Certlist_free_all(top);

	} else if (format == 1) {	/* output a certificate or a CRL (PEM) */
		if (bg->cache == NULL)
			goto error;
		if (type == CSTORE_CTX_CERT) {
			if (PEM_write_cert((Cert *) bg->cache, fname))
				goto error;
		} else {
			if (PEM_write_crl((CRL *) bg->cache, fname))
				goto error;
		}

	} else {		/* output a certificate or a CRL (DER) */
		if (ASN1_write_der(bg->der, fname))
			goto error;
	}

	return 0;
error:
	return -1;
}

Key *get_pkey_bag(STManager *stm, CSBag *cbg)
{
	CStore *ks;
	CSBag *kbg;
	Key *ret = NULL;
	char buf[PWD_BUFLEN];

	if ((ks =
	     STM_find_byName(stm, STORE_MY, CSTORE_ON_STORAGE,
			     CSTORE_CTX_KEY)) == NULL)
		goto done;

	/* check private key password */
	OK_set_passwd(pass);

	if ((kbg = CStore_find_byKeyHash(ks->bags, cbg->key_hash)) == NULL)
		goto done;

	/* first time */
	if ((ret = CStore_get_key(ks, kbg)) != NULL)
		goto done;

	/* second time */
	OK_get_passwd("Access Private Key:", (unsigned char *)buf, 0);
	OK_set_passwd(buf);
	if ((ret = CStore_get_key(ks, kbg)) != NULL)
		goto done;

done:
	OK_clear_passwd();
	memset(buf, 0, PWD_BUFLEN);
	return ret;
}

/*-----------------------------------------
  delete proc
-----------------------------------------*/
int ai_delete_proc(STManager *stm, char *store, int type, char *id)
{
	CStore *cs;
	CSBag *bg;

	if (id == NULL)
		goto error;
	if (STM_reload(stm))
		goto error;

	/* get key hash */
	if ((cs = STM_find_byName(stm, store, CSTORE_ON_STORAGE, type)) == NULL)
		goto error;

	if ((bg = CStore_find_byID(cs->bags, id)) == NULL)
		return 1;

	if (!strcmp(store, STORE_MY)) {
		if (del_pkey_bag(stm, bg))
			goto error;
	}

	/* clean certificate bag */
	if (CStore_del_byID(cs, id))
		goto error;

	/* update store */
	if (STM_update(stm))
		goto error;

	return 0;
error:
	return -1;
}

int del_pkey_bag(STManager *stm, CSBag *cbg)
{
	CStore *ks;
	CSBag *kbg;
	Key *tmp;
	char buf[PWD_BUFLEN];

	if ((ks =
	     STM_find_byName(stm, STORE_MY, CSTORE_ON_STORAGE,
			     CSTORE_CTX_KEY)) == NULL)
		goto error;

	/* check private key password */
	OK_set_passwd(pass);

	if ((kbg = CStore_find_byKeyHash(ks->bags, cbg->key_hash)) == NULL)
		goto error;
	if ((tmp = CStore_get_key(ks, kbg)) == NULL) {
		OK_get_passwd("Access Private Key:", (unsigned char *)buf, 0);
		OK_set_passwd(buf);
		if ((tmp = CStore_get_key(ks, kbg)) == NULL)
			goto error;
	}
	Key_free(tmp);
	OK_clear_passwd();
	memset(buf, 0, PWD_BUFLEN);

	if (CStore_del_bag(ks, kbg))
		goto error;

	return 0;
error:
	return -1;
}
