/* x509testfc.c */
/*
 * Modified by National Institute of Informatics in Japan, 2013-2014.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_pem.h>
#include <aicrypto/ok_rsa.h>
#include <aicrypto/ok_tool.h>
#include <aicrypto/ok_x509.h>

#define CACERT		"ca.cer"
#define CAKEY		"ca.key"
#define CACRL		"test.crl"
#define USERCERT	"test.cer"
#define USERREQ		"test.p10"
#define USERREQ2	"test2.p10"
#define USERKEY		"test.key"
#define PKIXCERT1	"pkix-ex1.ber"
#define PKIXCERT2	"pkix-ex2.ber"
#define CROSSPAIR	"cross.der"
#define DSACERT		"dsacert.der"
#define DSAKEY		"dsakey.der"
#define ECCACERT	"ecca.cer"
#define ECCAKEY		"ecca.key"
#define ECUSERCERT	"ecusr.cer"
#define ECUSERKEY	"ecusr.key"
#define LISTNAME	"verify.idx"
#ifndef PATH
# define PATH		"."
#endif

/* test/getfpath.c */
char *get_fpath(char *path, char *filename);


int test_x509_cert(){
	unsigned char *der,*cp;
	Cert *ct,*dmy;
	Key *key;
	int i,j;
	char *fp_USERCERT = get_fpath(PATH, USERCERT);
	char *fp_CAKEY = get_fpath(PATH, CAKEY);
	char *fp_CACERT = get_fpath(PATH, CACERT);
	char *fp_DSACERT = get_fpath(PATH, DSACERT);
	char *fp_DSAKEY = get_fpath(PATH, DSAKEY);
	char *fp_PKIXCERT1 = get_fpath(PATH, PKIXCERT1);
	char *fp_PKIXCERT2 = get_fpath(PATH, PKIXCERT2);
	char *fp_ECCACERT = get_fpath(PATH, ECCACERT);
	char *fp_ECCAKEY = get_fpath(PATH, ECCAKEY);
	char *fp_ECUSERCERT = get_fpath(PATH, ECUSERCERT);

	/* read DER file */
	for(i=0;i<11;i++){
		if ((ct = Cert_read_file(fp_USERCERT)) == NULL){
			printf("test x509 cert read file (1) -- error!\n");
			return -1;
		}

		/* Oops X(, actually I need to get CA private key to generate
		 * certificate DER. so here, read non-encrypted PEM private key file
		 * before PEM test X(
		 */
		OK_set_passwd("abcde");
		if ((key = (Key*)PEM_read_rsaprv(fp_CAKEY)) == NULL){
			printf("test pem read RSA key file -- error!\n");
			return -1;
		}

		/* get old one :-) */
		der=ct->der;

		/* get new one :-) */
		OK_set_cert_sig_algo(OBJ_SIG_SHA1RSA);
		free(ct->signature);
		ct->signature=NULL;

		if((ct->der=Cert_toDER(ct,key,NULL,&j))==NULL){
			printf("cannot create DER encoding with cert -- error!\n");
			return -1;
		}
		if(memcmp(der,ct->der,j)){
			printf("test x509 cert gen DER -- error!\n");
			ASN1_write_der(der,"#errcert1.der");
			ASN1_write_der(ct->der,"#errcert2.der");
			return -1;
		}

		free(der); der=NULL;
		Key_free(key); key=NULL;
		Cert_free(ct); ct=NULL;
		if((i%5)==0)printf("test Cert decode & encode -- ok : %d\n",i);
	}

	/* verify cert file */
	for(i=0;i<11;i++){
		if ((der = ASN1_read_der(fp_CACERT)) == NULL){
			printf("test x509 cert read file (2) -- error!\n");
			return -1;
		}

		if((ct=ASN1_read_cert(der))==NULL){
			printf("test x509 cert read asn.1 -- error!\n");
			return -1;
		}
		dmy=Cert_dup(ct);
		Cert_free(ct);

		if ((ct = Cert_read_file(fp_USERCERT)) == NULL){
			printf("test x509 cert read file (3) -- error!\n");
			return -1;
		}

		if(Cert_signature_verify(dmy,ct)){
			printf("test x509 cert signature verify -- error!\n");
			return -1;
		}

		der=NULL;
		Cert_free(ct); ct =NULL;
		Cert_free(dmy);dmy=NULL;
		if((i%5)==0)printf("test Cert signature verify -- ok : %d\n",i);
	}

	/* PKIX DSA certificate */
	for(i=0;i<11;i++){
		if ((ct = Cert_read_file(fp_DSACERT)) == NULL){
			printf("test x509 cert read file (1) -- error!\n");
			return -1;
		}
		if ((der = ASN1_read_der(fp_DSAKEY)) == NULL){
			printf("test read DSA key file -- error!\n");
			return -1;
		}
		/* key->der will have allocated memory pointer */
		if((key=(Key*)ASN1_read_dsaprv(der))==NULL){
			printf("test decode DSA key -- error!\n");
			return -1;
		}

		/* set old der */
		der = ct->der;
		/* get new one :-) */
		OK_set_cert_sig_algo(OBJ_SIG_SHA1DSA);
		free(ct->signature);
		ct->signature=NULL;

		if((ct->der=Cert_toDER(ct,key,NULL,&j))==NULL){
			printf("cannot create DER encoding with DSA cert -- error!\n");
			return -1;
		}

		/* cannot compare old der and new one, because DSA uses one time
		 * private key...and every signature should be different, even
		 * though we sign with same DSA private key.
		 */
		cp =ASN1_next(der);
		ASN1_skip_(cp,&j);

		if(memcmp(cp,ASN1_next(ct->der),j)){
			printf("test x509 cert gen DER -- error!\n");
			ASN1_write_der(ct->der,"#errcert.der");
			return -1;
		}

		free(der); der=NULL;
		Key_free(key); key=NULL;
		Cert_free(ct); ct=NULL;
		if((i%5)==0)printf("test DSA Cert decode & encode -- ok : %d\n",i);
	}

	/* PKIX DSA certificate -- verify test */
	for(i=0;i<11;i++){
		if ((der = ASN1_read_der(fp_PKIXCERT1)) == NULL){
			printf("test x509 cert read file (4) -- error!\n");
			return -1;
		}
		if((ct=ASN1_read_cert(der))==NULL){
			printf("test x509 cert read asn.1 (4) -- error!\n");
			return -1;
		}
		dmy=Cert_dup(ct);
		Cert_free(ct);

		if ((ct = Cert_read_file(fp_PKIXCERT2)) == NULL){
			printf("test x509 cert read file (4) -- error!\n");
			return -1;
		}
		if(Cert_signature_verify(dmy,ct)){
			printf("test x509 cert signature verify -- error!\n");
			return -1;
		}

		der=NULL;
		Cert_free(ct); ct=NULL;
		Cert_free(dmy);dmy=NULL;
		if((i%5)==0)printf("test DSA Cert signature verify -- ok : %d\n",i);
	}

	/* PKIX ECDSA certificate */
	for(i=0;i<11;i++){
		if ((ct = Cert_read_file(fp_ECCACERT)) == NULL){
			printf("test x509 cert read file (1) -- error!\n");
			return -1;
		}
		if ((der = ASN1_read_der(fp_ECCAKEY)) == NULL){
			printf("test read ECDSA key file -- error!\n");
			return -1;
		}
		/* key->der will have allocated memory pointer */
		if((key=(Key*)ASN1_read_ecdsaprv(der))==NULL){
			printf("test decode ECDSA key -- error!\n");
			return -1;
		}

		/* set old der */
		der = ct->der;
		/* get new one :-) */
		OK_set_cert_sig_algo(OBJ_SIG_SHA1ECDSA);
		free(ct->signature);
		ct->signature=NULL;

		if((ct->der=Cert_toDER(ct,key,NULL,&j))==NULL){
			printf("cannot create DER encoding with ECDSA cert -- error!\n");
			return -1;
		}

		/* cannot compare old der and new one, because DSA uses one time
		 * private key...and every signature should be different, even
		 * though we sign with same DSA private key.
		 */
		cp =ASN1_next(der);
		ASN1_skip_(cp,&j);

		if(memcmp(cp,ASN1_next(ct->der),j)){
			printf("test x509 cert gen DER -- error!\n");
			ASN1_write_der(ct->der,"#errcert.der");
			return -1;
		}

		free(der); der=NULL;
		Key_free(key); key=NULL;
		Cert_free(ct); ct=NULL;
		if((i%5)==0)printf("test ECDSA Cert decode & encode -- ok : %d\n",i);
	}

	/* PKIX ECDSA certificate -- verify test */
	for(i=0;i<11;i++){
		if ((der = ASN1_read_der(fp_ECCACERT)) == NULL){
			printf("test x509 cert read file (4) -- error!\n");
			return -1;
		}
		if((ct=ASN1_read_cert(der))==NULL){
			printf("test x509 cert read asn.1 (4) -- error!\n");
			return -1;
		}
		dmy=Cert_dup(ct);
		Cert_free(ct);

		if ((ct = Cert_read_file(fp_ECUSERCERT)) == NULL){
			printf("test x509 cert read file (4) -- error!\n");
			return -1;
		}
		if(Cert_signature_verify(dmy,ct)){
			printf("test x509 cert signature verify -- error!\n");
			return -1;
		}

		der=NULL;
		Cert_free(ct); ct=NULL;
		Cert_free(dmy);dmy=NULL;
		if((i%5)==0)printf("test ECDSA Cert signature verify -- ok : %d\n",i);
	}

	/* compare two certificate */
	for(i=0;i<11;i++){
		if ((ct = Cert_read_file(fp_DSACERT)) == NULL){
			printf("test x509 cert read file (5) -- error!\n");
			return -1;
		}
		if ((dmy = Cert_read_file(fp_CACERT)) == NULL){
			printf("test x509 cert read file (5) -- error!\n");
			return -1;
		}
		if(Cert_cmp(ct,ct)){
			printf("test compare two Cert structure -- ok : %d\n",i);
			return -1;
		}
		if(!Cert_cmp(ct,dmy)){
			printf("test compare two Cert structure -- ok : %d\n",i);
			return -1;
		}

		Cert_free(ct); ct=NULL;
		Cert_free(dmy);dmy=NULL;
		if((i%5)==0)printf("test compare two Cert structure -- ok : %d\n",i);
	}
	free(fp_USERCERT);
	free(fp_CAKEY); free(fp_CACERT);
	free(fp_DSACERT); free(fp_DSAKEY);
	free(fp_PKIXCERT1); free(fp_PKIXCERT2);
	free(fp_ECCACERT); free(fp_ECCAKEY);
	free(fp_ECUSERCERT);
	return 0;
}

int test_x509_crl(){
	unsigned char *der;
	Cert *ca;
	CRL  *crl,*dmy;
	Prvkey_RSA  *key;
	int i,j;
	char *fp_CACRL = get_fpath(PATH, CACRL);
	char *fp_CAKEY = get_fpath(PATH, CAKEY);
	char *fp_CACERT = get_fpath(PATH, CACERT);

	/* read DER file */
	for(i=0;i<11;i++){
		if ((crl = CRL_read_file(fp_CACRL)) == NULL){
			printf("test x509 CRL read file (1) -- error!\n");
			return -1;
		}

		/* Oops X(, actually I need to get CA private key to generate
		 * certificate DER. so here, read non-encrypted PEM private key file
		 * before PEM test X(
		 */
		if ((key = PEM_read_rsaprv(fp_CAKEY)) == NULL){
			printf("test pem read RSA key file -- error!\n");
			return -1;
		}

		/* get old one :-) */
		der=crl->der;
		/* get new one :-) */
		OK_set_crl_sig_algo(OBJ_SIG_SHA1RSA);
		free(crl->signature);
		crl->signature=NULL;

		if((crl->der=CRL_toDER(crl,(Key*)key,NULL,&j))==NULL){
			printf("cannot create DER encoding with crl -- error!\n");
			return -1;
		}

		if(memcmp(der,crl->der,j)){
			printf("test x509 CRL gen DER -- error!\n");
			return -1;
		}
		
		free(der);
		Key_free((Key*)key);
		CRL_free(crl);
		if((i%5)==0)printf("test CRL decode & encode -- ok : %d\n",i);
	}

	/* verify CRL file */
	for(i=0;i<11;i++){
		if ((der = ASN1_read_der(fp_CACRL)) == NULL){
			printf("test x509 CRL read file (2) -- error!\n");
			return -1;
		}
		if((crl=ASN1_read_crl(der))==NULL){
			printf("test x509 CRL read asn.1 -- error!\n");
			return -1;
		}
		dmy=CRL_dup(crl);
		CRL_free(crl);
		if ((ca = Cert_read_file(fp_CACERT)) == NULL){
			printf("test x509 cert read file in CRL check -- error!\n");
			return -1;
		}
		if(CRL_signature_verify(ca,dmy)){
			printf("test x509 CRL signature verify -- error!\n");
			return -1;
		}

		Cert_free(ca);
		CRL_free(dmy);
		if((i%5)==0)printf("test CRL signature verify -- ok : %d\n",i);
	}

	/* compare two CRLs */
	for(i=0;i<11;i++){
		if ((crl = CRL_read_file(fp_CACRL)) == NULL){
			printf("test x509 CRL read file (3) -- error!\n");
			return -1;
		}
		if(CRL_cmp(crl,crl)){
			printf("test compare two CRL structure -- ok : %d\n",i);
			return -1;
		}

		CRL_free(crl); crl=NULL;
		if((i%5)==0)printf("test compare two CRL structure -- ok : %d\n",i);
	}
	free(fp_CACRL); free(fp_CAKEY); free(fp_CACERT);
	return 0;
}

int test_x509_certpair(){
	unsigned char *der,*cp;
	CertPair *ctp,*dmy;
	int i,j;
	char *fp_CROSSPAIR = get_fpath(PATH, CROSSPAIR);

	/* read DER file */
	for(i=0;i<11;i++){
		if ((der = ASN1_read_der(fp_CROSSPAIR)) == NULL){
			printf("test x509 certpair read file (1) -- error!\n");
			return -1;
		}

		if((ctp=ASN1_read_crtp(der))==NULL){
			printf("test x509 decode certpair -- error!\n");
			return -1;
		}
		if((dmy=CertPair_dup(ctp))==NULL){
			printf("test x509 certpair duplication -- error!\n");
			return -1;
		}

		if((cp=CertPair_toDER(dmy,NULL,&j))==NULL){
			printf("cannot create DER encoding with certpair -- error!\n");
			return -1;
		}
		if(memcmp(der,cp,j)){
			printf("test x509 certpair gen DER -- error!\n");
			ASN1_write_der(cp,"#errcert.der");
			return -1;
		}
		
		free(cp);  cp =NULL;
		CertPair_free(ctp); ctp=NULL;
		CertPair_free(dmy); dmy=NULL;
		if((i%5)==0)printf("test CertPair decode & encode -- ok : %d\n",i);
	}
	free(fp_CROSSPAIR);

	return 0;
}

/* X( actually, request is not defined in the X.509.
 * it's PKCS#10...
 */
int test_x509_req(){
	unsigned char *der;
	Req  *req,*dmy;
	Prvkey_RSA  *key;
	CertExt *ex;
	int i,j;
	char *fp_USERREQ = get_fpath(PATH, USERREQ);
	char *fp_USERKEY = get_fpath(PATH, USERKEY);
	char *fp_USERREQ2 = get_fpath(PATH, USERREQ2);

	/* read DER file */
	for(i=0;i<11;i++){
		if ((req = Req_read_file(fp_USERREQ)) == NULL){
			printf("test req read file (1) -- error!\n");
			goto error;
		}

		/* Oops X(, actually I need to get user's private key to generate
		 * certificate DER. so here, read non-encrypted PEM private key file
		 * before PEM test X(
		 */
		OK_set_passwd("abcde");
		if ((key = PEM_read_rsaprv(fp_USERKEY)) == NULL){
			printf("test pem read RSA key file -- error!\n");
			goto error;
		}
		OK_clear_passwd();

		/* get old one :-) */
		der=req->der;
		/* get new one :-) */
		OK_set_cert_sig_algo(OBJ_SIG_SHA1RSA);
		free(req->signature);
		req->signature=NULL;
		req->der=Req_toDER(req,(Key*)key,NULL,&j);

		if(memcmp(der,req->der,j)){
			printf("test req gen DER -- error!\n");
			goto error;
		}
		
		free(der);
		Key_free((Key*)key);
		Req_free(req);
		if((i%5)==0)printf("test Req decode & encode -- ok : %d\n",i);
	}

	/* read DER file (2) */
	for(i=0;i<11;i++){
		if ((req = Req_read_file(fp_USERREQ2)) == NULL){
			printf("test req read file (2) -- error!\n");
			goto error;
		}

		OK_set_passwd("abcde");
		if ((key = PEM_read_rsaprv(fp_USERKEY)) == NULL){
			printf("test pem read RSA key file (2) -- error!\n");
			goto error;
		}
		OK_clear_passwd();

		/* add cert ext */
		CertExt_free_all(req->ext);
		if((ex =req->ext=Extnew_cha_passwd("mysoft"))==NULL) goto error;
		if((ex->next=Extnew_unst_name("Test/AiCrypto/0.0"))==NULL) goto error;

		/* get old one :-) */
		der=req->der;
		/* get new one :-) */
		OK_set_cert_sig_algo(OBJ_SIG_MD5RSA);
		free(req->signature);
		req->signature=NULL;
		if((req->der=Req_toDER(req,(Key*)key,NULL,&j))==NULL) goto error;

		if(memcmp(der,req->der,j)){
			printf("test req gen DER -- error!\n");
			return -1;
		}
		
		free(der);
		Key_free((Key*)key);
		Req_free(req);
		if((i%5)==0)printf("test Req decode & encode (2) -- ok : %d\n",i);
	}

	/* verify Req file */
	for(i=0;i<11;i++){
		if ((der = ASN1_read_der(fp_USERREQ)) == NULL){
			printf("test req read file (2) -- error!\n");
			goto error;
		}
		if((req=ASN1_read_req(der))==NULL){
			printf("test req read asn.1 -- error!\n");
			goto error;
		}
		dmy=Req_dup(req);
		Req_free(req);

		if(Req_signature_verify(dmy)){
			printf("test req signature verify -- error!\n");
			goto error;
		}

		Req_free(dmy);
		if((i%5)==0)printf("test Req signature verify -- ok : %d\n",i);
	}
	free(fp_USERREQ); free(fp_USERKEY); free(fp_USERREQ2);

	return 0;
error:
	if(fp_USERREQ) free(fp_USERREQ);
	if(fp_USERKEY) free(fp_USERKEY);
	if(fp_USERREQ2) free(fp_USERREQ2);
	return -1;
}

int test_x509_certlist(){
	CertList *crtl,*tmp,*ctdp;
	CRLList *crll,*crdp;
	Cert *ct,*ca;
	CRL *crl;
	int i;
	char *t1="+-;1234abcdefghijklmnopqrstuvwxyz";
	char *t2="+-;1234ABCDEFGHIJKLMNOPQRSTUVWXYZ";
	char *t3="+-;2345";
	char *fp_CACERT = get_fpath(PATH, CACERT);
	char *fp_USERCERT = get_fpath(PATH, USERCERT);
	char *fp_CACRL = get_fpath(PATH, CACRL);

	/* check igcase_strcmp */
	if(igcase_strcmp(t1,t2)){
		printf("test case ignore strcmp (1) -- error!\n");
		return -1;
	}
	printf("test case ignore strcmp (1) -- ok\n");
	if(igcase_strcmp(t2,t3)>=0){
		printf("test case ignore strcmp (2) -- error!\n");
		return -1;
	}
	printf("test case ignore strcmp (2) -- ok\n");

	/* verification test */
	for(i=0;i<11;i++){
		if ((ca = Cert_read_file(fp_CACERT)) == NULL){
			printf("test x509 cert read file (2) -- error!\n");
			return -1;
		}
		if ((ct = Cert_read_file(fp_USERCERT)) == NULL){
			printf("test x509 cert read file (2) -- error!\n");
			return -1;
		}
		if ((crl = CRL_read_file(fp_CACRL)) == NULL){
			printf("test x509 CRL read file (2) -- error!\n");
			return -1;
		}
		/* get cert & CRL list */
		if((crtl=Cert_2Certlist(ct))==NULL){
			printf("test x509 certlist new (2) -- error!\n");
			return -1;
		}
		if((tmp=Cert_2Certlist(ca))==NULL){
			printf("test x509 certlist new (2) -- error!\n");
			return -1;
		}
		if(Certlist_insert(crtl,tmp)){
			printf("test x509 certlist new (2) -- error!\n");
			return -1;
		}
		if((crll=CRL_2CRLlist(crl))==NULL){
			printf("test x509 CRLlist new (2) -- error!\n");
			return -1;
		}
		if((ctdp=Certlist_dup_all(crtl))==NULL){
			printf("test x509 certlist dup -- error!\n");
			return -1;
		}
		if((crdp=CRLlist_dup_all(crll))==NULL){
			printf("test x509 CRLlist dup -- error!\n");
			return -1;
		}
		/* verify cert test */
		if(Cert_verify(ctdp,crdp,ct,9,0)){
			printf("test x509 verify (1) -- error!\n");
			return -1;
		}
		if(Cert_verify(ctdp,NULL,ct,9,DONT_CHECK_REVOKED)){
			printf("test x509 verify (2) -- error!\n");
			return -1;
		}

		Cert_free(ca); Cert_free(ct); CRL_free(crl);
		Certlist_free_all(crtl);
		Certlist_free_all(ctdp);
		CRLlist_free_all(crll);
		CRLlist_free_all(crdp);
		if((i%5)==0)printf("test x509 certlist & crllist verify -- ok : %d\n",i);
	}
	free(fp_CACERT); free(fp_USERCERT); free(fp_CACRL);

	return 0;
}

ExtGenNames *test_get_full_extgn(CertDN *dn){
	ExtGenNames	*egn,*gtp;
	unsigned char ip[16];
	OtherName *on;

	if((gtp=ExtGN_set_email("sample@aaa.dot.com"))==NULL) goto error;
	if((gtp->next=ExtGN_set_dns("aaa.dot.com"))==NULL) goto error;
	egn = gtp->next;
	if((egn->next=ExtGN_set_url("http://aaa.dot.com/"))==NULL) goto error;
	egn = egn->next;
	if((egn->next=ExtGN_set_dn(dn))==NULL) goto error;
	egn = egn->next;
	if((egn->next=ExtGN_set_ip(ip,4))==NULL) goto error;
	egn = egn->next;

	if((on=ExtGN_on_new())==NULL) goto error;
	on->oid = OBJ_MS_GN_UPN;
	if ((on->oidc = strdup("1.3.6.1.4.1.311.20.2.3")) == NULL) goto error;
	if((on->name = malloc(16))==NULL) goto error;
	if(ASN1_set_utf8("test@test.com",on->name,&on->nlen)) goto error;

	if((egn->next=ExtGN_set_oth(on,on->nlen+strlen(on->oidc)))==NULL) goto error;

	ExtGN_on_free(on);
	return gtp;
error:
	ExtGN_free(gtp);
	return NULL;
}

IPAddrFamily *test_get_full_ipaddr(){
	IPAddrFamily *ret,*fam;
	AddrOrRange *aor;

	/**** set new family ****/
	if((fam=ret=ExtAddrFam_new())==NULL) goto error;
	/* IPv4 Unicast */
	fam->afi = 1;
	fam->safi = 1;
	/* set addresses */
	if((fam->addr=aor=ExtIP_get_addrrange(EXT_ADRG_IPADDR,"10.0.32/20",NULL))==NULL) goto error;
	if((aor->next=ExtIP_get_addrrange(EXT_ADRG_IPADDR,"10.0.64/24",NULL))==NULL) goto error;
	aor = aor->next;
	if((aor->next=ExtIP_get_addrrange(EXT_ADRG_IPADDR,"10.1/16",NULL))==NULL) goto error;
	aor = aor->next;
	if((aor->next=ExtIP_get_addrrange(EXT_ADRG_RANGE,"10.2.48/20","10.2.64/24"))==NULL) goto error;
	aor = aor->next;
	if((aor->next=ExtIP_get_addrrange(EXT_ADRG_IPADDR,"10/8",NULL))==NULL) goto error;
	aor = aor->next;
	if((aor->next=ExtIP_get_addrrange(EXT_ADRG_IPADDR,"10.3.4.5",NULL))==NULL) goto error;

	/**** set new family ****/
	if((fam->next=ExtAddrFam_new())==NULL) goto error;
	fam = fam->next;
	/* IPv6 */
	fam->afi = 2;
	/* inherit -- no ip addresses */

	/**** set new family ****/
	if((fam->next=ExtAddrFam_new())==NULL) goto error;
	fam = fam->next;
	/* IPv4 Multicast */
	fam->afi = 1;
	fam->safi = 2;
	/* inherit -- no ip addresses */

	/**** set new family ****/
	if((fam->next=ExtAddrFam_new())==NULL) goto error;
	fam = fam->next;
	/* IPv6 */
	fam->afi = 2;
	/* set addresses */
	if((fam->addr=aor=ExtIP_get_addrrange(EXT_ADRG_IPADDR,"2001:0:2/47",NULL))==NULL) goto error;
	if((aor->next=ExtIP_get_addrrange(EXT_ADRG_RANGE,"2001:0:2/48","2001:0:200:3:0:0:0:1"))==NULL) goto error;
	aor = aor->next;
	if((aor->next=ExtIP_get_addrrange(EXT_ADRG_IPADDR,"2001::200:3:5:/118",NULL))==NULL) goto error;

	return ret;
error:
	ExtAddrFam_free_all(ret);
	return NULL;
}

int test_cmp_ipaddr(CertExt *ce1, CertExt *ce2){
	CE_IPAddrDlg *c1,*c2;
	IPAddrFamily *f1,*f2;
	AddrOrRange *a1,*a2;

	c1 = (CE_IPAddrDlg*)ce1;
	c2 = (CE_IPAddrDlg*)ce2;
	f1 = c1->family;
	f2 = c2->family;
	while(f1 || f2){
		if(!(f1 && f2)) return -1;
		if(f1->afi != f2->afi) return -1;
		if(f1->safi != f2->safi) return -1;

		a1 = f1->addr;
		a2 = f2->addr;
		while(a1 || a2){
			if(!(a1 && a2)) return -1;
			if(a1->type != a2->type) return -1;
			if(a1->min_prefix != a2->min_prefix) return -1;
			if(memcmp(a1->min_addr,a2->min_addr,16)) return -1;
			if(a1->max_prefix != a2->max_prefix) return -1;
			if(memcmp(a1->max_addr,a2->max_addr,16)) return -1;

			a1 = a1->next;
			a2 = a2->next;
		}

		f1 = f1->next;
		f2 = f2->next;
	}
	return 0;
}

ASIdOrRange *test_get_full_asid(){
	ASIdOrRange *ret,*as;

	/* set ids */
	if((ret=as=ExtASId_get_addrrange(EXT_ASRG_ASID,135,0))==NULL) goto error;
	if((as->next=ExtASId_get_addrrange(EXT_ASRG_RANGE,3000,3999))==NULL) goto error;
	as = as->next;
	if((as->next=ExtASId_get_addrrange(EXT_ASRG_ASID,5001,0))==NULL) goto error;
	as = as->next;

	return ret;
error:
	ExtASIdRang_free_all(ret);
	return NULL;
}

int test_cmp_asid(CertExt *ce1, CertExt *ce2){
	CE_ASId *c1,*c2;
	ASIdOrRange *a1,*a2;

	c1 = (CE_ASId*)ce1;
	c2 = (CE_ASId*)ce2;

	if(c1->asnum || c2->asnum){
		if(!(c1->asnum && c2->asnum)) return -1;
		a1 = c1->asnum;
		a2 = c2->asnum;
		while(a1 || a2){
			if(!(a1 && a2)) return -1;
			if(a1->type != a2->type) return -1;
			if(a1->min != a2->min) return -1;
			if(a1->max != a2->max) return -1;
			a1=a1->next; a2=a2->next;
		}
	}
	if(c1->rdi || c2->rdi){
		if(!(c1->rdi && c2->rdi)) return -1;
		a1 = c1->rdi;
		a2 = c2->rdi;
		while(a1 || a2){
			if(!(a1 && a2)) return -1;
			if(a1->type != a2->type) return -1;
			if(a1->min != a2->min) return -1;
			if(a1->max != a2->max) return -1;
			a1=a1->next; a2=a2->next;
		}
	}
	return 0;
}

int test_x509_certext(){
	Cert *ct;
	CertExt *ce,*ce2,*ce3;
	ExtCertPol *ecp;
	ExtSubTrees *tr1,*tr2;
	ExtGenNames	*gtp,*gtp2;
	IPAddrFamily *fam;
	ASIdOrRange *as1,*as2;
	char *cbuf = NULL,*testoids[16],buf1[1024],buf2[1024];
	unsigned char flag[4],tp[32]={
		0x02,0x07,0x07,0x1c,0x17,0x03,0x2b,0xc4,0x67};
	int i,err=-1,ep=0;
	char *fp_CACERT = get_fpath(PATH, CACERT);

	if ((ct = Cert_read_file(fp_CACERT)) == NULL){
		printf("test x509 cert read file in certext -- error!\n");
		return -1;
	}
	free(fp_CACERT);

	/* authkey_id */
	cbuf = "authority_key_id";
	if((ce=Extnew_authkey_id(ct,0xf))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_authkey(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(Ext_authkey_str((CE_AuthKID*)ce ,buf1,1022)<0) goto done;
	if(Ext_authkey_str((CE_AuthKID*)ce3,buf2,1022)<0) goto done;
	if(strcmp(buf1,buf2)){ ep = 3; goto done;}

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

	/* sbjkey_id */
	cbuf = "sbject_key_id";
	if((ce=Extnew_sbjkey_id(ct))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_sbjkey(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(Ext_sbjkey_str((CE_SbjKID*)ce ,buf1,1022)<0) goto done;
	if(Ext_sbjkey_str((CE_SbjKID*)ce3,buf2,1022)<0) goto done;
	if(strcmp(buf1,buf2)){ ep = 3; goto done;}

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

	/* key_usage */
	cbuf = "key_usage";
	if((ce=Extnew_keyusage(0xfe))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_keyusage(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(Ext_keyusage_str((CE_KUsage*)ce ,buf1,1022)<0) goto done;
	if(Ext_keyusage_str((CE_KUsage*)ce3,buf2,1022)<0) goto done;
	if(strcmp(buf1,buf2)){ ep = 3; goto done;}

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

	/* extended_key_usage */
	for(i=0;i<16;i++) testoids[i]=NULL;
	if ((testoids[0] = strdup("1.3.6.1.4.1.311.10.3.4")) == NULL) goto done;
	if ((testoids[1] = strdup("1.3.6.1.5.5.7.3.4")) == NULL) goto done;
	if ((testoids[2] = strdup("1.3.6.1.5.5.7.3.2")) == NULL) goto done;
	cbuf = "ext_key_usage";
	if((ce=Extnew_extkeyusage(testoids))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_extkeyusage(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(Ext_extkeyusage_str((CE_ExtKUsage*)ce ,buf1,1022)<0) goto done;
	if(Ext_extkeyusage_str((CE_ExtKUsage*)ce3,buf2,1022)<0) goto done;
	if(strcmp(buf1,buf2)){ ep = 3; goto done;}

	for(i=0;i<16;i++) if(testoids[i]) free(testoids[i]);
	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

	/* cert_policy */
	cbuf = "cert_policy";
	if((ecp=ExtCP_new())==NULL) goto done;
	if ((ecp->policyID = strdup("1.2.392.100300.1.3.1")) == NULL) goto done;

	if((ecp->info=ExtPI_get_cps("1.3.6.1.5.5.7.2.1","this is sample qualifier"))==NULL)
		goto done;
	if((ecp->info->next=ExtPI_get_unotice("1.3.6.1.5.5.7.2.2","sample org",1,"sample expText"))==NULL)
		goto done;
	if((ecp->next=ExtCP_dup(ecp))==NULL) goto done;

	if((ce=Extnew_cert_policy(OBJ_X509v3_CERT_Pol,ecp))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_certpol(OBJ_X509v3_CERT_Pol,ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(Ext_certpol_str((CE_CertPol*)ce ,buf1,1022)<0) goto done;
	if(Ext_certpol_str((CE_CertPol*)ce3,buf2,1022)<0) goto done;
	if(strcmp(buf1,buf2)){ ep = 3; goto done;}

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	/* ecp will be free in CertExt_free() */
	printf("test extention : %s -- ok!\n",cbuf);
	
	/* policy_map */
	cbuf = "policy_mapping";
	if((ce=Extnew_policy_map("1.2.33.444","1.2.444.5555"))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_certpolmap(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(Ext_certpolmap_str((CE_PolMap*)ce ,buf1,1022)<0) goto done;
	if(Ext_certpolmap_str((CE_PolMap*)ce3,buf2,1022)<0) goto done;
	if(strcmp(buf1,buf2)){ ep = 3; goto done;}

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

	/* sbject_alt_name */
	cbuf = "sbject,issuer alt_name";
	if((gtp=test_get_full_extgn(&ct->subject_dn))==NULL) goto done;

	if((ce=Extnew_altname(OBJ_X509v3_SbjAltName,gtp))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_altname(OBJ_X509v3_SbjAltName,ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(Ext_altname_str((CE_SbjAltName*)ce ,buf1,1022)<0) goto done;
	if(Ext_altname_str((CE_SbjAltName*)ce3,buf2,1022)<0) goto done;
	if(strcmp(buf1,buf2)){ ep = 3; goto done;}

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	/* egn will be free in CertExt_free() */
	printf("test extention : %s -- ok!\n",cbuf);

	/* basic constraints */
	cbuf = "basic constraints";
	if((ce=Extnew_basic_cons(1,8))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_basiccons(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(Ext_basiccons_str((CE_BasicCons*)ce ,buf1,1022)<0) goto done;
	if(Ext_basiccons_str((CE_BasicCons*)ce3,buf2,1022)<0) goto done;
	if(strcmp(buf1,buf2)){ ep = 3; goto done;}

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

	/* name constraints */
	cbuf = "name constraints";
	if((tr1=ExtSubT_get_tree(ExtGN_set_url("http://aaa.dot.com/"),2,-1))==NULL) goto done;
	if((tr1->next=ExtSubT_get_tree(ExtGN_set_dn(&ct->issuer_dn),2,3))==NULL) goto done;
	if((tr2=ExtSubT_dup_all(tr1))==NULL) goto done;

	if((ce=Extnew_name_cons(tr1,tr2))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_namecons(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(Ext_namecons_str((CE_NameCons*)ce ,buf1,1022)<0) goto done;
	if(Ext_namecons_str((CE_NameCons*)ce3,buf2,1022)<0) goto done;
	if(strcmp(buf1,buf2)){ ep = 3; goto done;}

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);


	/* policy constraints */
	cbuf = "policy constraints";
	if((ce=Extnew_policy_cons(3,2))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_policons(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(Ext_polcons_str((CE_PolCons*)ce ,buf1,1022)<0) goto done;
	if(Ext_polcons_str((CE_PolCons*)ce3,buf2,1022)<0) goto done;
	if(strcmp(buf1,buf2)){ ep = 3; goto done;}

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);


	/* crl distribution point */
	cbuf = "crl distribution point";
	if((gtp=test_get_full_extgn(&ct->subject_dn))==NULL) goto done;
	memset(flag,0,4); flag[0]=0xfe; flag[1]=0x80;
	if((gtp2=test_get_full_extgn(&ct->issuer_dn))==NULL) goto done;

	if((ce=Extnew_crl_distpoint(gtp,flag,gtp2))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_crlpoint(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(Ext_crlpoint_str((CE_CRLDistPt*)ce ,buf1,1022)<0) goto done;
	if(Ext_crlpoint_str((CE_CRLDistPt*)ce3,buf2,1022)<0) goto done;
	if(strcmp(buf1,buf2)){ ep = 3; goto done;}

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

	/* PKIX AIA */
	cbuf = "pkix-aia";
	if((gtp=test_get_full_extgn(&ct->subject_dn))==NULL) goto done;

	if((ce=Extnew_pkix_aia("1.3.6.1.5.5.7.48.2",gtp))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_pkixaia(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(Ext_pkixaia_str((CE_AIA*)ce ,buf1,1022)<0) goto done;
	if(Ext_pkixaia_str((CE_AIA*)ce3,buf2,1022)<0) goto done;
	if(strcmp(buf1,buf2)){ ep = 3; goto done;}

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

	/* OCSP no check */
	cbuf = "ocsp-nocheck";
	if((ce=Extnew_ocsp_nocheck())==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_ocspnochk(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(Ext_ocspnochk_str(ce ,buf1,1022)<0) goto done;
	if(Ext_ocspnochk_str(ce3,buf2,1022)<0) goto done;
	if(strcmp(buf1,buf2)){ ep = 3; goto done;}

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

	/* ns crlurl */
	cbuf = "ns_crl_url";
	if((ce=Extnew_ns_crlurl("http://aaa.dot.com/a.crl"))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_nscrlurl(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(Ext_comment_str((CE_Com*)ce ,buf1,1022)<0) goto done;
	if(Ext_comment_str((CE_Com*)ce3,buf2,1022)<0) goto done;
	if(strcmp(buf1,buf2)){ ep = 3; goto done;}

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

	/* ns comment */
	cbuf = "ns_comment";
	if((ce=Extnew_ns_comment("this is sample comment for ns extension."))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_nscomment(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(Ext_comment_str((CE_Com*)ce ,buf1,1022)<0) goto done;
	if(Ext_comment_str((CE_Com*)ce3,buf2,1022)<0) goto done;
	if(strcmp(buf1,buf2)){ ep = 3; goto done;}

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

	/* ns cert type */
	cbuf = "ns_cert_type";
	if((ce=Extnew_ns_flag(0xff))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_nscerttype(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(Ext_nscerttype_str((CE_NSType*)ce ,buf1,1022)<0) goto done;
	if(Ext_nscerttype_str((CE_NSType*)ce3,buf2,1022)<0) goto done;
	if(strcmp(buf1,buf2)){ ep = 3; goto done;}

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

	/* MOJ RegCorpInfo */
	cbuf = "moj_reg_corpInfo";
	if((ce=Extnew_moj_corpinfo("mycorporation","104000101010","tokyo-to,abcde-shi,hogehoge-cho 1-2-3",
				"hoge taro","C.E.O",NULL,"tokyo-houmsyo",0))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_mojcorpinfo(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(Ext_mojcorpinfo_str((CE_MOJCoInfo*)ce ,buf1,1022)<0) goto done;
	if(Ext_mojcorpinfo_str((CE_MOJCoInfo*)ce3,buf2,1022)<0) goto done;
	if(strcmp(buf1,buf2)){ ep = 3; goto done;}

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

	/* MOJ TimeLimit */
	cbuf = "moj_timelimit";
	if((ce=Extnew_moj_timelimit(3))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_timelimit(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(strcmp(((CE_Com*)ce)->comment,((CE_Com*)ce3)->comment)){ ep = 3; goto done;}

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

	/* MOJ suspendSecretCode */
	cbuf = "moj_suspendCode";
	if((ce=Extnew_moj_suspcode(OBJ_HASH_SHA1,"abcdefgh",8))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_suspcode(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(((CE_MOJSuspCode*)ce)->hash_algo != ((CE_MOJSuspCode*)ce3)->hash_algo){ ep = 3; goto done;}
	if(((CE_MOJSuspCode*)ce)->hlen != ((CE_MOJSuspCode*)ce3)->hlen){ ep = 3; goto done;}
	if(memcmp(((CE_MOJSuspCode*)ce)->hash,((CE_MOJSuspCode*)ce3)->hash,20)){ ep = 3; goto done;}

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

	/* MOJ GenmInfoReq */
	cbuf = "moj_GenmInfoReq";
	if((ce=Extnew_moj_genmreq(OBJ_CRYALGO_3DESCBC,OBJ_CRYPT_RSA,OBJ_HASH_SHA1))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_mojgenmreq(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(((CE_MOJGenmReq*)ce)->nego_num != ((CE_MOJGenmReq*)ce3)->nego_num){ ep = 3; goto done;}
	if(((CE_MOJGenmReq*)ce)->nego[0].symm_algo != ((CE_MOJGenmReq*)ce3)->nego[0].symm_algo){
		ep = 3; goto done;}
	if(((CE_MOJGenmReq*)ce)->nego[0].pub_algo != ((CE_MOJGenmReq*)ce3)->nego[0].pub_algo){
		ep = 3; goto done;}
	if(((CE_MOJGenmReq*)ce)->nego[0].hash_algo != ((CE_MOJGenmReq*)ce3)->nego[0].hash_algo){
		ep = 3; goto done;}

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

	/* MOJ GenpInfoRes */
	cbuf = "moj_GenpInfoRes";
	if((ce=Extnew_moj_genpres(2,OBJ_CRYALGO_3DESCBC,OBJ_CRYPT_RSA,OBJ_HASH_SHA1))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_mojgenpres(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(((CE_MOJGenpRes*)ce)->nego_num != ((CE_MOJGenpRes*)ce3)->nego_num){ ep = 3; goto done;}
	if(((CE_MOJGenpRes*)ce)->pki_status != ((CE_MOJGenpRes*)ce3)->pki_status){ ep = 3; goto done;}
	if(((CE_MOJGenpRes*)ce)->nego[0].symm_algo != ((CE_MOJGenpRes*)ce3)->nego[0].symm_algo){
		ep = 3; goto done;}
	if(((CE_MOJGenpRes*)ce)->nego[0].pub_algo != ((CE_MOJGenpRes*)ce3)->nego[0].pub_algo){
		ep = 3; goto done;}
	if(((CE_MOJGenpRes*)ce)->nego[0].hash_algo != ((CE_MOJGenpRes*)ce3)->nego[0].hash_algo){
		ep = 3; goto done;}

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

	/* MOJ GenpSuspendReq */
	cbuf = "moj_GenmSuspReq";

	if((ce=Extnew_moj_genspreq(ASN1_dup(tp),&ct->issuer_dn,0x06,1,ct->pubkey,cbuf,strlen(cbuf)))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_mojgenspreq(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(memcmp(((CE_MOJGenSpReq*)ce)->snum_der,((CE_MOJGenSpReq*)ce3)->snum_der,9)){ ep = 3; goto done;}
	if(((CE_MOJGenSpReq*)ce)->keyAlg != ((CE_MOJGenSpReq*)ce3)->keyAlg){ ep = 3; goto done;}
	if(((CE_MOJGenSpReq*)ce)->suspReason != ((CE_MOJGenSpReq*)ce3)->suspReason){ ep = 3; goto done;}
	if(((CE_MOJGenSpReq*)ce)->revReason[0] != ((CE_MOJGenSpReq*)ce3)->revReason[0]){ ep = 3; goto done;}
	if(Cert_dncmp(&((CE_MOJGenSpReq*)ce)->issuer_dn,&((CE_MOJGenSpReq*)ce3)->issuer_dn)){ ep = 3; goto done;}
	if(((CE_MOJGenSpReq*)ce)->enc_len != ((CE_MOJGenSpReq*)ce3)->enc_len){ ep = 3; goto done;}
	if(memcmp(((CE_MOJGenSpReq*)ce)->encValue,((CE_MOJGenSpReq*)ce3)->encValue,ct->pubkey->size)){ ep = 3; goto done;}

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

	/* MOJ GenpSuspResContent */
	cbuf = "moj_GenpSuspRes";
	if((ce=Extnew_moj_genspres(0,&ct->issuer_dn,ASN1_dup(tp)))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_mojgenspres(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(memcmp(((CE_MOJGenSpRes*)ce)->snum_der,((CE_MOJGenSpRes*)ce3)->snum_der,9)){ ep = 3; goto done;}
	if(((CE_MOJGenSpRes*)ce)->status != ((CE_MOJGenSpRes*)ce3)->status){ ep = 3; goto done;}
	if(Cert_dncmp(&((CE_MOJGenSpRes*)ce)->issuer_dn,&((CE_MOJGenSpRes*)ce3)->issuer_dn)){ ep = 3; goto done;}

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

	/* ns pkcs9_extreq */
	cbuf = "pkcs9_extreq";
	if((gtp=test_get_full_extgn(&ct->subject_dn))==NULL) goto done;

	if((ce=Extnew_altname(OBJ_X509v3_SbjAltName,gtp))==NULL){
		ep = 1; goto done;} /* test extension */
	if((ce->next=Extnew_sbjkey_id(ct))==NULL){
		ep = 1; goto done;}
	if((ce=Extnew_extreq(ce))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_extreq(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(Ext_altname_str((CE_SbjAltName*)((CE_ExtReq*)ce)->ext ,buf1,1022)<0) goto done;
	if(Ext_altname_str((CE_SbjAltName*)((CE_ExtReq*)ce3)->ext,buf2,1022)<0) goto done;
	if(strcmp(buf1,buf2)){ ep = 3; goto done;}

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	/* egn will be free in CertExt_free() */
	printf("test extention : %s -- ok!\n",cbuf);

	/* crl entry reason code */
	cbuf = "[crl entry] reason code";
	if((ce=Extnew_reason_code(3))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_reasoncode(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

	/* crl number */
	cbuf = "[crl] crl_number";
	if((ce=Extnew_crl_number(0x3fea8733))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_crlnumber(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

	/* CRL Issuing Dist Point */
	cbuf = "[crl] issuing_dist_point";
	if((gtp=test_get_full_extgn(&ct->subject_dn))==NULL) goto done;
	memset(flag,0,4); flag[0]=0xfe; flag[1]=0x80;
	if((ce=Extnew_crl_issdistpt(gtp,flag,EXT_IDP_UCert|EXT_IDP_indCRL))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_issdistpt(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);
	
	/* IP Address Delegation */
	cbuf = "ip_address_delegation";
	if((fam=test_get_full_ipaddr())==NULL) goto done;
	if((ce=Extnew_ipaddr(fam))==NULL){
		ep = 1; goto done;}
	/* ASN1_write_der(ce->der,"ipout.der"); */
	if((ce2=ASN1_ext_ipaddr(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(test_cmp_ipaddr(ce,ce2)) goto done;

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

	/* AS Identifier */
	cbuf = "as_identifier";
	if((as1=test_get_full_asid())==NULL) goto done;
	if((as2=ExtASId_get_addrrange(EXT_ASRG_INHERIT,0,0))==NULL) goto done;
	if((ce=Extnew_asid(as1,as2))==NULL){
		ep = 1; goto done;}
	ASN1_write_der(ce->der,"asidout.der");
	if((ce2=ASN1_ext_asid(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(test_cmp_asid(ce,ce2)) goto done;

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

	/* AS Identifier */
	cbuf = "as_identifier (NULL)";
	if((ce=Extnew_asid(NULL,NULL))==NULL){
		ep = 1; goto done;}
	if((ce2=ASN1_ext_asid(ce->der))==NULL){
		ep = 2; goto done;}
	if((ce3=CertExt_dup(ce))==NULL) goto done;

	if(test_cmp_asid(ce,ce2)) goto done;

	CertExt_free(ce); CertExt_free(ce2);
	CertExt_free(ce3);
	printf("test extention : %s -- ok!\n",cbuf);

#if 0
	/* read and dup and free */
	for(i=0;i<11;i++){
		if((ct2=Cert_read_file("00015.cer"))==NULL){
			printf("test x509 cert read file (1) -- error!\n");
			return -1;
		}
		ct3=Cert_dup(ct2);
		Cert_free(ct2);
		Cert_free(ct3);
	}
#endif

	err=0;
done:
	if(err){
		if(ep==1)      printf("test ext : cannot get %s -- error!\n",cbuf);
		else if(ep==2) printf("test ext : cannot decode %s -- error!\n",cbuf);
		else           printf("test ext : systematic error ! memory allocation orelse... : %s\n",cbuf);
	}
	Cert_free(ct);
	return err;
}

int test_tool_sign(){
	unsigned char *sig,*der,dig[32];
	char *txt="abcdefghijklmnopqrstuvwxyz123456789";
	Cert *ct;
	Key *key;
	int i,j,slen;
	char *fp_USERCERT = get_fpath(PATH, USERCERT);
	char *fp_USERKEY = get_fpath(PATH, USERKEY);
	char *fp_DSACERT = get_fpath(PATH, DSACERT);
	char *fp_DSAKEY = get_fpath(PATH, DSAKEY);
	char *fp_ECCACERT = get_fpath(PATH, ECCACERT);
	char *fp_ECCAKEY = get_fpath(PATH, ECCAKEY);

	/* test RSA signature & verify */
	for(i=0;i<11;i++){
		if ((ct = Cert_read_file(fp_USERCERT)) == NULL){
			printf("test x509 cert read file (1) -- error!\n");
			goto error;
		}
		OK_set_passwd("abcde");
		if ((key = (Key*)PEM_read_rsaprv(fp_USERKEY)) == NULL){
			printf("test pem read RSA key file -- error!\n");
			goto error;
		}
		OK_clear_passwd();

		/* get signature */
		if(OK_do_signature(key,txt,strlen(txt),&sig,&slen,OBJ_SIG_SHA1RSA)){
			printf("test OK_do_signature -- error!\n");
			goto error;
		}
		/* verify signature */
		/* 1. get do digest */
		if(OK_do_digest(OBJ_SIG_SHA1RSA,txt,strlen(txt),dig,&j)==NULL){
			printf("test OK_do_digest -- error!\n");
			goto error;
		}
		/* 2. check signature */
		if(OK_do_verify(ct->pubkey,dig,sig,OBJ_SIG_SHA1RSA)){
			printf("test OK_do_verify -- error!\n");
			goto error;
		}

		free(sig); sig=NULL;
		Key_free(key); key=NULL;
		Cert_free(ct); ct=NULL;
		if((i%5)==0)printf("test tool RSA signature & verify -- ok : %d\n",i);
	}

	/* test DSA signature & verify */
	for(i=0;i<11;i++){
		if ((ct = Cert_read_file(fp_DSACERT)) == NULL){
			printf("test x509 cert read file (1) -- error!\n");
			return -1;
		}
		if ((der = ASN1_read_der(fp_DSAKEY)) == NULL){
			printf("test read DSA key file -- error!\n");
			return -1;
		}
		/* key->der = this pointer */
		if((key=(Key*)ASN1_read_dsaprv(der))==NULL){
			printf("test decode DSA key -- error!\n");
			return -1;
		}

		/* get signature */
		if(OK_do_signature(key,txt,strlen(txt),&sig,&slen,OBJ_SIG_SHA1DSA)){
			printf("test OK_do_signature -- error!\n");
			goto error;
		}
		/* verify signature */
		/* 1. get do digest */
		if(OK_do_digest(OBJ_SIG_SHA1DSA,txt,strlen(txt),dig,&j)==NULL){
			printf("test OK_do_digest -- error!\n");
			goto error;
		}
		/* 2. check signature */
		if(OK_do_verify(ct->pubkey,dig,sig,OBJ_SIG_SHA1DSA)){
			printf("test OK_do_verify -- error!\n");
			goto error;
		}

		free(sig); sig=NULL;
		Key_free(key); key=NULL;
		Cert_free(ct); ct=NULL;
		if((i%5)==0)printf("test tool DSA signature & verify -- ok : %d\n",i);
	}

	/* test ECDSA signature & verify */
	for(i=0;i<11;i++){
		if ((ct = Cert_read_file(fp_ECCACERT)) == NULL){
			printf("test x509 cert read file (1) -- error!\n");
			return -1;
		}
		if ((der = ASN1_read_der(fp_ECCAKEY)) == NULL){
			printf("test read ECDSA key file -- error!\n");
			return -1;
		}
		/* key->der = this pointer */
		if((key=(Key*)ASN1_read_ecdsaprv(der))==NULL){
			printf("test decode ECDSA key -- error!\n");
			return -1;
		}

		/* get signature */
		if(OK_do_signature(key,txt,strlen(txt),&sig,&slen,OBJ_SIG_SHA1ECDSA)){
			printf("test OK_do_signature -- error!\n");
			goto error;
		}
		/* verify signature */
		/* 1. get do digest */
		if(OK_do_digest(OBJ_SIG_SHA1RSA,txt,strlen(txt),dig,&j)==NULL){
			printf("test OK_do_digest -- error!\n");
			goto error;
		}
		/* 2. check signature */
		if(OK_do_verify(ct->pubkey,dig,sig,OBJ_SIG_SHA1ECDSA)){
			printf("test OK_do_verify -- error!\n");
			goto error;
		}

		free(sig); sig=NULL;
		Key_free(key); key=NULL;
		Cert_free(ct); ct=NULL;
		if((i%5)==0)printf("test tool ECDSA signature & verify -- ok : %d\n",i);
	}
	free(fp_USERCERT); free(fp_USERKEY);
	free(fp_DSACERT); free(fp_DSAKEY);
	free(fp_ECCACERT); free(fp_ECCAKEY);
	
	return 0;
error:
	if (fp_USERCERT) free(fp_USERCERT);
	if (fp_USERKEY) free(fp_USERKEY);
	if (fp_DSACERT) free(fp_DSACERT);
	if (fp_DSAKEY) free(fp_DSAKEY);
	if (fp_ECCACERT) free(fp_ECCACERT);
	if (fp_ECCAKEY) free(fp_ECCAKEY);

	return -1;
}
