/* x509_time.c */
/*
 * Modified by National Institute of Informatics in Japan, 2013-2016.
 *
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "aiconfig.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_rsa.h>
#include <aicrypto/ok_tool.h>
#include <aicrypto/ok_x509.h>

/*-----------------------------------------
  ASN.1 UTC or GTIME to struct tm
-----------------------------------------*/
int UTC2stm(unsigned char *utc, struct tm *ctm){
	unsigned char buf[16];
	int i,j,pr=0;

	memset(ctm,0,sizeof(struct tm));
	memset(buf,0,16);
	if(*utc==ASN1_GENERALIZEDTIME){
		buf[0]=utc[2];buf[1]=utc[3];
		buf[2]=utc[4];buf[3]=utc[5]; i=6;
		ctm->tm_year = atoi(buf) - 1900;
	}else if(*utc==ASN1_UTCTIME){
		buf[0]=utc[2];buf[1]=utc[3]; i=4;
		ctm->tm_year = atoi(buf);
		if(ctm->tm_year<50) ctm->tm_year+=100;	/* 2000 year problem */
	}else if(*utc==ASN1_PRINTABLE_STRING){
		buf[0]=utc[2];buf[1]=utc[3];
		buf[2]=utc[4];buf[3]=utc[5]; i=6;
		ctm->tm_year = atoi(buf); pr = 1;
	}else{
		OK_set_error(ERR_ST_BADFORMAT,ERR_LC_X509,ERR_PT_X509TIME,NULL);
		return -1;
	}

	buf[2]=0;
	if(utc[i]=='Z') return 0;
	memcpy(buf,&utc[i],2);
	j = atoi(buf);
	ctm->tm_mon = (pr)?((j==99)?(99):(j-1)):((j-1) % 12);

	if(utc[i+2]=='Z') return 0;
	memcpy(buf,&utc[i+2],2);
	ctm->tm_mday = (pr)?(atoi(buf)):((atoi(buf)-1) % 31 + 1);

	if(utc[i+4]=='Z') return 0;
	memcpy(buf,&utc[i+4],2);
	ctm->tm_hour = (pr)?(atoi(buf)):(atoi(buf) % 24);

	if(utc[i+6]=='Z') return 0;
	memcpy(buf,&utc[i+6],2);
	ctm->tm_min = (pr)?(atoi(buf)):(atoi(buf) % 60);
  
	if((utc[i+8]=='Z')||(utc[i+9]=='+')||(utc[i+10]=='-')){
		ctm->tm_sec = 0;
	}else{
		memcpy(buf,&utc[i+8],2);
		ctm->tm_sec = (pr)?(atoi(buf)):(atoi(buf) % 60);
	}
	return 0;
}

/*-----------------------------------------
  ASN.1 UTC or GTIME to time_t
-----------------------------------------*/
time_t UTC2time_t(unsigned char *utc){
	struct tm ctm;
	time_t ret;

	if(UTC2stm(utc,&ctm)) return -1;
	ret = timegm(&ctm);

	return ret;
} 

/*-----------------------------------------
  ASN.1 struct tm to UTC
-----------------------------------------*/
unsigned char *stm2UTC(struct tm *stm,unsigned char *buf,unsigned char tag){
	unsigned char *ret;

	if(buf==NULL){
		if((ret=(unsigned char*)malloc(20))==NULL){
			OK_set_error(ERR_ST_MEMALLOC,ERR_LC_X509,ERR_PT_X509TIME+1,NULL);
			return NULL;
		}
	}else{
		ret=buf;
	}

	if((stm->tm_year<150)&&(tag==ASN1_UTCTIME)){
		ret[0] = ASN1_UTCTIME;
		sprintf(&ret[2],"%.2d%.2d%.2d%.2d%.2d%.2dZ",
			stm->tm_year%100,stm->tm_mon+1,stm->tm_mday,
			stm->tm_hour,stm->tm_min,stm->tm_sec);
	}else if(tag==ASN1_GENERALIZEDTIME){
		ret[0] = ASN1_GENERALIZEDTIME;
		sprintf(&ret[2],"%.4d%.2d%.2d%.2d%.2d%.2dZ",
			stm->tm_year+1900,stm->tm_mon+1,stm->tm_mday,
			stm->tm_hour,stm->tm_min,stm->tm_sec);
	}else if(tag==ASN1_PRINTABLE_STRING){
		ret[0] = ASN1_PRINTABLE_STRING;
		sprintf(&ret[2],"%.4d%.2d%.2d%.2d%.2d%.2dZ",
			stm->tm_year,
			(stm->tm_mon==99)?(99):(stm->tm_mon+1),
			stm->tm_mday,stm->tm_hour,stm->tm_min,stm->tm_sec);
	}
	ret[1] = strlen(&ret[2]);

	return ret;
}

/*-----------------------------------------
  struct tm to string
-----------------------------------------*/
char *stm2str(struct tm *stm,int type){
	struct tm *ltm;
	time_t t;
	static char buf[64];
	char mon[12][8]={
		"Jan","Feb","Mar","Apr","May","Jun",
		"Jul","Aug","Sep","Oct","Nov","Dec"};

	/* get local time */
	if((t = timegm(stm))<0)
		ltm = stm;
	else
		ltm = localtime(&t); /* get localtime struct tm */

	/* get string */
	if(type==0){ /* default type */
		sprintf(buf,"%s %.2d %.2d:%.2d:%.2d %.4d %s",
			mon[ltm->tm_mon],ltm->tm_mday,ltm->tm_hour,
			ltm->tm_min,ltm->tm_sec,ltm->tm_year+1900,(t<0)?("UTC"):(""));
	}else if(type==1){
		sprintf(buf,"%.2d/%.2d/%.2d %.2d:%.2d %s",
			ltm->tm_year%100,ltm->tm_mon+1,ltm->tm_mday,
			ltm->tm_hour,ltm->tm_min,(t<0)?("UTC"):(""));
	}else if(type==2){
		/* Cert Profile time mode. caution!! "stm" is used */
		sprintf(buf,"%.4d/%.2d/%.2d %.2d:%.2d:%.2d",
			stm->tm_year,
			(stm->tm_mon==99)?(99):(stm->tm_mon+1),
			stm->tm_mday,stm->tm_hour,stm->tm_min,stm->tm_sec);
	}else if(type==3){
		sprintf(buf,"%.4d/%.2d/%.2d %.2d:%.2d:%.2d",
			ltm->tm_year+1900,ltm->tm_mon+1,ltm->tm_mday,
			ltm->tm_hour,ltm->tm_min,ltm->tm_sec);		
	}
	return buf;
}

char *UTC2str(unsigned char *utc,int type){
	struct tm stm;
	char *ret;

	if(UTC2stm(utc,&stm)<0) return NULL;

	ret = stm2str(&stm,type);
	return ret;
}

/*-----------------------------------------
  compare struct tm
  output : b is newer than a ... positive
         : same ... 0
		 : b is older than a ... negative
-----------------------------------------*/
int stmcmp(struct tm *a, struct tm *b){
	int i;

	if((i=b->tm_year-a->tm_year) != 0) goto done;
	if((i=b->tm_mon -a->tm_mon) != 0)  goto done;
	if((i=b->tm_mday-a->tm_mday) != 0) goto done;
	if((i=b->tm_hour-a->tm_hour) != 0) goto done;
	if((i=b->tm_min -a->tm_min) != 0)  goto done;
	if((i=b->tm_sec -a->tm_sec) != 0)  goto done;

	i=0;
done:
	return i;
}

#ifndef HAVE_TIMEGM
/*-----------------------------------------
  return time_t from struct tm (UTC)
-----------------------------------------*/
time_t timegm(struct tm *stm){
	time_t ret;

	if((ret = mktime(stm))<0) goto done;
	ret -= timezone;
done:
	return ret;
}
#endif
