/* ext_ip.c */
/*
 * Modified by National Intitute of Informatics in Japan, 2013-2014.
 *
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *	this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *	this list of conditions and the following disclaimer in the documentation
 *	and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *	display the following acknowledgment:
 *	"This product includes software developed by Akira Iwata Laboratory,
 *	Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *	acknowledgment:
 *	"This product includes software developed by Akira Iwata Laboratory,
 *	 Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_x509.h>
#include <aicrypto/ok_x509ext.h>

/*-----------------------------------------
    alloc & free AddrOrRange
-----------------------------------------*/
AddrOrRange *ExtAddrRang_new(){
	AddrOrRange *ret;

	if((ret=(AddrOrRange*)malloc(sizeof(AddrOrRange)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_X509EXT,ERR_PT_EXTIP,NULL);
		return NULL;
	}

	memset(ret,0,sizeof(AddrOrRange));
	return(ret);	
}

void ExtAddrRang_free(AddrOrRange *aor){
	if(aor==NULL) return;
	free(aor);
}

void ExtAddrRang_free_all(AddrOrRange *top){
	AddrOrRange *next;
	while(top){
		next=top->next;
		ExtAddrRang_free(top);
		top=next;
	}
}

AddrOrRange *ExtAddrRang_dup(AddrOrRange *aor){
	AddrOrRange *ret=NULL;

	if((ret=ExtAddrRang_new())==NULL) goto error;

	memcpy(ret,aor,sizeof(AddrOrRange));
	ret->next = NULL;

	return ret;
error:
	ExtAddrRang_free(ret);
	return NULL;
}

AddrOrRange *ExtAddrRang_dup_all(AddrOrRange *top){
	AddrOrRange *hd,*now,*ret=NULL;
	while(top){
		if((now=ExtAddrRang_dup(top))==NULL) goto error;
		if(ret){
			hd->next = now;
			hd = hd->next;
		}else{
			hd = ret = now;
		}
		top=top->next;
	}
	return ret;
error:
	ExtAddrRang_free_all(ret);
	return NULL;
}


/*-----------------------------------------
    alloc & free IPAddrFamily
-----------------------------------------*/
IPAddrFamily *ExtAddrFam_new(){
	IPAddrFamily *ret;

	if((ret=(IPAddrFamily*)malloc(sizeof(IPAddrFamily)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_X509EXT,ERR_PT_EXTIP+1,NULL);
		return NULL;
	}

	memset(ret,0,sizeof(IPAddrFamily));
	return(ret);	
}

void ExtAddrFam_free(IPAddrFamily *ipaf){
	if(ipaf==NULL) return;
	ExtAddrRang_free_all(ipaf->addr);
	free(ipaf);
}

void ExtAddrFam_free_all(IPAddrFamily *top){
	IPAddrFamily *next;
	while(top){
		next=top->next;
		ExtAddrFam_free(top);
		top=next;
	}
}

IPAddrFamily *ExtAddrFam_dup(IPAddrFamily *ipaf){
	IPAddrFamily *ret=NULL;

	if((ret=ExtAddrFam_new())==NULL) goto error;

	memcpy(ret,ipaf,sizeof(IPAddrFamily));
	ret->next = NULL;

	return ret;
error:
	ExtAddrFam_free(ret);
	return NULL;
}

IPAddrFamily *ExtAddrFam_dup_all(IPAddrFamily *top){
	IPAddrFamily *hd,*now,*ret=NULL;
	while(top){
		if((now=ExtAddrFam_dup(top))==NULL) goto error;
		if(ret){
			hd->next = now;
			hd = hd->next;
		}else{
			hd = ret = now;
		}
		top=top->next;
	}
	return ret;
error:
	ExtAddrFam_free_all(ret);
	return NULL;
}

/*-----------------------------------------
    alloc & free ASIdOrRange
-----------------------------------------*/
ASIdOrRange *ExtASIdRang_new(){
	ASIdOrRange *ret;

	if((ret=(ASIdOrRange*)malloc(sizeof(ASIdOrRange)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_X509EXT,ERR_PT_EXTIP+2,NULL);
		return NULL;
	}

	memset(ret,0,sizeof(ASIdOrRange));
	return(ret);	
}

void ExtASIdRang_free(ASIdOrRange *aor){
	if(aor==NULL) return;
	free(aor);
}

void ExtASIdRang_free_all(ASIdOrRange *top){
	ASIdOrRange *next;
	while(top){
		next=top->next;
		ExtASIdRang_free(top);
		top=next;
	}
}

ASIdOrRange *ExtASIdRang_dup(ASIdOrRange *aor){
	ASIdOrRange *ret=NULL;

	if((ret=ExtASIdRang_new())==NULL) goto error;

	memcpy(ret,aor,sizeof(ASIdOrRange));
	ret->next = NULL;

	return ret;
error:
	ExtASIdRang_free(ret);
	return NULL;
}

ASIdOrRange *ExtASIdRang_dup_all(ASIdOrRange *top){
	ASIdOrRange *hd,*now,*ret=NULL;
	while(top){
		if((now=ExtASIdRang_dup(top))==NULL) goto error;
		if(ret){
			hd->next = now;
			hd = hd->next;
		}else{
			hd = ret = now;
		}
		top=top->next;
	}
	return ret;
error:
	ExtASIdRang_free_all(ret);
	return NULL;
}

/*-----------------------------------------
  Address Family to DER
-----------------------------------------*/
int extip_ip2bitstr(unsigned char *in, int prefix, unsigned char *ret,int *ret_len){
	int len,bit;

	if(prefix <= 0){
		OK_set_error(ERR_ST_BADPARAM,ERR_LC_X509EXT,ERR_PT_EXTIP+3,NULL);
		return -1;
	}
	len = ((prefix-1)>>3) + 1;
	bit = (8 - (prefix % 8)) % 8; 
	ASN1_set_bitstring(bit,len,in,ret,ret_len);

	return 0;
}

int ExtIP_DER_addrrang(AddrOrRange *aor,unsigned char *ret,int *ret_len){
	unsigned char *cp;
	int i,j;

	*ret_len=0; 

	switch(aor->type){
	case EXT_ADRG_IPADDR:
		if(extip_ip2bitstr(aor->min_addr,aor->min_prefix,ret,ret_len)) goto error;
		break;
	case EXT_ADRG_RANGE:
		i = 0; cp=ret;
		if(extip_ip2bitstr(aor->min_addr,aor->min_prefix,cp,&i)) goto error;
		cp+=i;
		if(extip_ip2bitstr(aor->max_addr,aor->max_prefix,cp,&j)) goto error;
		i+=j;
		ASN1_set_sequence(i,ret,ret_len);
		break;
	}
	return 0;
error:
	return -1;
}

int ExtIP_DER_addrfam(IPAddrFamily *fam,unsigned char *ret,int *ret_len){
	AddrOrRange *now;
	unsigned char *cp,*t,tmp[16];
	int i,j,k=0;


	/* octet string */
	tmp[0] = (unsigned char)(fam->afi>>8);
	tmp[1] = (unsigned char)fam->afi;
	k+=2;

	if(fam->safi){
		tmp[2] = (unsigned char)fam->safi;
		k+=1;
	}

	*ret_len=0; cp=ret;

	/* addressFamily */
	ASN1_set_octetstring(k,tmp,ret,&i);
	cp+=i; t=cp;

	/* ip address choice */
	if(fam->addr){
		for(j=0,now=fam->addr; now ;now=now->next){
			if(ExtIP_DER_addrrang(now,cp,&k)) goto error;
			cp+=k; j +=k;
		}
		ASN1_set_sequence(j,t,&j);
		i+=j;
	}else{
		ASN1_set_null(t); 
		i+=2;
	}
	ASN1_set_sequence(i,ret,ret_len);

	return 0;
error:
	return -1;
}

unsigned char *ExtIP_toDER(IPAddrFamily *top,unsigned char *buf,int *ret_len){
	IPAddrFamily *now;
	unsigned char *cp,*ret;
	int i,j=0;

	if(buf==NULL){
		if((i=ExtAddrFam_estimate_der_size(top))<=0)
			return NULL;

		if((ret=(unsigned char*)malloc(i))==NULL){
			OK_set_error(ERR_ST_MEMALLOC,ERR_LC_X509EXT,ERR_PT_EXTIP+4,NULL);
			return NULL;
		}
		memset(ret,0,i);
	}else{
		ret=buf;
	}

	*ret_len=0; cp=ret;
	for(now=top; now ;now=now->next){
		if(ExtIP_DER_addrfam(now,cp,&i)) goto error;
		cp+=i; j +=i;
	}
	ASN1_set_sequence(j,ret,ret_len);
	return ret;
error:
	if(ret!=buf) free(ret);
	return NULL;
}

/*-----------------------------------------
  ASId to DER
-----------------------------------------*/
int ExtASId_DER_asid(ASIdOrRange *as,unsigned char *ret,int *ret_len){
	unsigned char *cp,*t;
	int i,j,k;

	*ret_len = 0;
	for(cp=ret,i=0; as ; as=as->next){
		switch(as->type){
		case EXT_ASRG_INHERIT:
			ASN1_set_null(ret); *ret_len = 2; 
			return 0;
		case EXT_ASRG_ASID:
			ASN1_set_integer(as->min,cp,&j);
			cp+=j; i+=j;
			break;
		case EXT_ASRG_RANGE:
			t = cp;
			ASN1_set_integer(as->min,t,&j); t+=j;
			ASN1_set_integer(as->max,t,&k);
			ASN1_set_sequence(j+k,cp,&j);
			cp+=j; i+=j;
			break;
		}
	}
	ASN1_set_sequence(i,ret,ret_len);

	return 0;
}

/*-----------------------------------------
  estimate IPAddrFamily DER size from Cert
-----------------------------------------*/
int ExtAddrRang_estimate_der_size(AddrOrRange *top){
	int ret = 8;
	while(top){
		switch(top->type){
		case EXT_ADRG_IPADDR: ret += 22;
		case EXT_ADRG_RANGE:  ret += 48;
		}
		top=top->next;
	}
	return ret;
}

int ExtAddrFam_estimate_der_size(IPAddrFamily *top){
	IPAddrFamily *ipaf;
	int ret = 0;

	if(top==NULL){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_X509EXT,ERR_PT_EXTIP+6,NULL);
		return -1;
	}

	for(ret=0,ipaf=top;ipaf;ipaf=ipaf->next){
		ret += 16;
		ret += ExtAddrRang_estimate_der_size(ipaf->addr);
	}
	return ret;
}

/*-----------------------------------------
  estimate ASIdOrRange DER size from Cert
-----------------------------------------*/
int ExtASIdRang_estimate_der_size(ASIdOrRange *top){
	ASIdOrRange *as;
	int ret = 0;

	if(top==NULL){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_X509EXT,ERR_PT_EXTIP+7,NULL);
		return -1;
	}

	for(ret=0,as=top;as;as=as->next){
		ret += 8;
		switch(as->type){
		case EXT_ASRG_INHERIT: ret += 0;
		case EXT_ASRG_ASID:    ret += 8;
		case EXT_ASRG_RANGE:   ret += 16;
		}
	}
	return ret;
}

/*-----------------------------------------
  CertExt IP Addres Delegation
-----------------------------------------*/
/* support */
CertExt *Extnew_ipaddr(IPAddrFamily *fam){
	CE_IPAddrDlg *ret = NULL;
	int i,k=16;

	/* estimate size & malloc DER */
	if((i=ExtAddrFam_estimate_der_size(fam))<0) goto error;
	k+=i;

	if((ret=(CE_IPAddrDlg*)CertExt_new(OBJ_PKIX_IDPE_IPADDR))==NULL) goto error;

	if((ret->der=(unsigned char*)malloc(k))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_X509EXT,ERR_PT_EXTIP+8,NULL);
		goto error;
	}
	memset(ret->der,0,k);

	/* set data */
	ret->family = fam;
	/* get DER */
	if(ExtIP_toDER(fam,ret->der,&ret->dlen)==NULL) goto error;	

	return (CertExt*)ret;
error:
	CertExt_free((CertExt*)ret);
	return NULL;
}

/*-----------------------------------------
  CertExt AS Id
-----------------------------------------*/
/* support */
CertExt *Extnew_asid(ASIdOrRange *asnum,ASIdOrRange *rdi){
	CE_ASId *ret;
	unsigned char *cp;
	int i=0,j=0;

	/* estimate size & malloc DER */
	if(asnum)
		if((i=ExtASIdRang_estimate_der_size(asnum))<=0) return NULL;
	if(rdi)
		if((j=ExtASIdRang_estimate_der_size(rdi))<=0) return NULL;

	/* get extension */
	if((ret=(CE_ASId*)CertExt_new(OBJ_PKIX_IDPE_ASID))==NULL) goto error;

	if((ret->der=(unsigned char*)malloc(i+j+16))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_X509EXT,ERR_PT_EXTIP+9,NULL);
		goto error;
	}
	memset(ret->der,0,i+j+16);

	/* set data */
	ret->asnum = asnum;
	ret->rdi = rdi;
	/* get DER */
	i=0; cp=ret->der;
	/* asnum */
	if(asnum){
		if(ExtASId_DER_asid(asnum,cp,&j)) goto error;
		ASN1_set_explicit(j,0,cp,&j);
		cp+=j; i+=j;
	}
	/* rdi */
	if(rdi){
		if(ExtASId_DER_asid(rdi,cp,&j)) goto error;
		ASN1_set_explicit(j,1,cp,&j);
		cp+=j; i+=j;
	}
	ASN1_set_sequence(i,ret->der,&ret->dlen);

	return (CertExt*)ret;
error:
	CertExt_free((CertExt*)ret);
	return NULL;
}

/*-----------------------------------------
  utility functions
-----------------------------------------*/
int ip2text(int type,unsigned char *addr,int prefix,char *text){
	char buf[32];
	int i,j,len;

	*text = 0;
	len = ((prefix-1)/8)+1;

	switch(type){
	case 1: /* IPv4 */
		if(prefix > 32) return -1;

		for(i=0; i<len; i++){
			if(i) strcat(text,".");
			sprintf(buf,"%d",addr[i]);
			strcat(text,buf);
		}
		if(prefix < 32){
			sprintf(buf,"/%d",prefix);
			strcat(text,buf);
		}
		break;
	case 2: /* IPv6 */
		if(prefix > 128) return -1;

		for(i=0; i<len; i+=2){
			if(i) strcat(text,":");
			j = (addr[i]<<8)|addr[i+1];
			sprintf(buf,"%x",j);
			strcat(text,buf);
		}
		if(prefix < 128){
			sprintf(buf,"/%d",prefix);
			strcat(text,buf);
		}
		break;
	default: /* other */
		for(i=0; i<len; i++){
			if(i) strcat(text,":");
			sprintf(buf,"%.2x",addr[i]);
			strcat(text,buf);
		}
		break;
	}
	return 0;
}

int text2ip(char *txt, unsigned char *addr, int *prefix){
	int i=0,slen,ver=0;
	long j=0;
	char *tp;

	/* check ip version */
	if(strchr(txt,'.')){
		ver = 4; *prefix=32;
	}else if(strchr(txt,':')){
		ver = 6; *prefix=128;
	}else if(strchr(txt,'/')){
		ver = 4; *prefix=32;
	}else{ /* error */
		OK_set_error(ERR_ST_BADFORMAT,ERR_LC_X509EXT,ERR_PT_EXTIP+10,NULL);
		return -1;
	}

	slen = strlen(txt); tp=txt;
	while(ver && (i<16)){
		switch(*txt){
		case '.': /* address segmentation */
			addr[i] = atoi(tp);
			i++; tp=&txt[1];
			break;
		case ':': /* address segmentation */
			j = strtol(tp,&txt,16);
			addr[i] = (unsigned char)(j>>8);
			addr[i+1] = (unsigned char)j;
			i+=2; tp=&txt[1];
			break;
		case '/': /* prefix segmentation */
			if(ver==4){
				addr[i] = atoi(tp); i++;
			}else{
				j = strtol(tp,&txt,16);
				addr[i] = (unsigned char)(j>>8);
				addr[i+1] = (unsigned char)j;
				i+=2;
			}
			txt++; ver=0; /* finish */
			*prefix = atoi(txt);
			break;
		case 0: /* NULL -- end of string */
			if(ver==4){
				addr[i] = atoi(tp); i++;
			}else{
				j = strtol(tp,&txt,16);
				addr[i] = (unsigned char)(j>>8);
				addr[i+1] = (unsigned char)j;
				i+=2;
			}
			ver=0; /* finish */
			break;
		}
		txt++;
	}
	return 0;
}

AddrOrRange *ExtIP_get_addrrange(int type, char *ip_min, char *ip_max){
	AddrOrRange *ret;

	if((ret=ExtAddrRang_new())==NULL) return NULL;

	ret->type = type;
	switch(type){
	case EXT_ADRG_IPADDR:
		if(text2ip(ip_min,ret->min_addr,&ret->min_prefix)) goto error;
		break;
	case EXT_ADRG_RANGE:
		if(text2ip(ip_min,ret->min_addr,&ret->min_prefix)) goto error;
		if(text2ip(ip_max,ret->max_addr,&ret->max_prefix)) goto error;
		break;
	}

	return ret;
error:
	ExtAddrRang_free(ret);
	return NULL;
}

ASIdOrRange *ExtASId_get_addrrange(int type, int min, int max){
	ASIdOrRange *ret;

	if((ret=ExtASIdRang_new())==NULL) return NULL;

	ret->type = type;
	switch(type){
	case EXT_ASRG_ASID:
		ret->min = min;
		break;
	case EXT_ASRG_RANGE:
		ret->min = min;
		ret->max = max;
		break;
	}

	return ret;
}



